/*

  sshglob.h

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1999-2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Globbing. Primarily for use with scp and sftp, but can be used
  elsewhere.
  
 */

#ifndef SSHGLOB_H
#define SSHGLOB_H

#include "sshincludes.h"

#define SSH_WILDCARDS "*?[](|)#"

/* Note: if you have to debug these functions, set the debugger
   breakpoint to the appropriate _ssh_* function (i.e. for those
   functions, whose ssh_* pair is a macro). This should be
   self-evident, but I know someone will try wring my neck about this
   eventually, so it is documented here. */

/* Returns TRUE if one of SSH_WILDCARDS is found from string
   unescaped. */
Boolean _ssh_glob_check_for_wildcards(const char *str, int echar);

/* Matches a string with a pattern. 'glob_string' may or may not
   contain wildcards, which are matched approriately.'glob_string' may
   contain escapes. Returns TRUE if strings match, FALSE otherwise. */
Boolean _ssh_glob_match_pattern(const char *glob_string,
                                const char *string, int echar);

/* Return the pointer in string 'string' where next un-escaped
   character 'ch' is. If character is escaped with 'echar', it is
   skipped. Note that you get incorrect results if you specify
   identical 'ch'and 'echar' values. Return NULL if approriate 'ch' is
   not found or 'string' is NULL. */
char *_ssh_glob_next_unescaped_char(const char *string, int ch, int echar);

/* Return pointer in string 'string' to the next un-escaped
   wildcard-character (which are in SSH_WILDCARDS). Return NULL if
   there are none. */
char *_ssh_glob_next_unescaped_wildchar(const char *string, int echar);

/* Check if a character in a string is escaped. 'ch' is a pointer to
   the character being checked, and it must belong to
   'string'. Returns TRUE if escaped, FALSE otherwise. Escape character
   is 'echar'.

   e.g. if a string is '\\\\\*' (5*'\'), then the '*' is escaped
   (returns TRUE), assuming '\' is the escape character. */
Boolean _ssh_glob_isescaped(const char *ch, const char *string, int echar);

/* This function is used to strip possible escapes from a string. The
   original string is not touched, but a new one is allocated which is
   then stripped. Escapes which are escaped are left to the
   string. The caller must free the resultant string. If 'string' is
   NULL, NULL is returned. */
char *_ssh_glob_strip_escapes(const char *string, int echar);

/* Macro to ease use of ssh_isescaped in most systems. You can still
   use _ssh_isescaped directly. */
#define ssh_glob_isescaped(ch, string) \
        _ssh_glob_isescaped((ch), (string), '\\')

/* Similar macro for _ssh_next_unescaped_char(). */
#define ssh_glob_next_unescaped_char(string, ch) \
        _ssh_glob_next_unescaped_char((string), (ch), '\\')

/* Similar macro for _ssh_next_unescaped_wildchar(). */
#define ssh_glob_next_unescaped_wildchar(string) \
        _ssh_glob_next_unescaped_wildchar((string), '\\')

/* Similar macro for _ssh_glob_strip_escapes(). */
#define ssh_glob_strip_escapes(string) \
        _ssh_glob_strip_escapes((string), '\\')

/* Similar macro for _ssh_glob_check_for_wildcards(). */
#define ssh_glob_check_for_wildcards(string) \
        _ssh_glob_check_for_wildcards((string), '\\')

/* Similar macro for _ssh_glob_match_pattern(). */
#define ssh_glob_match_pattern(glob_string, string) \
        _ssh_glob_match_pattern((glob_string), (string), '\\')

#endif /* SSHGLOB_H */
