/*
 * Copyright (C) 1999, 2000, 2001, 2002  Lorenzo Bettini <bettini@gnu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

// htmlgeneratorfactory.cc

/**
   Decorate text with terminal control sequences (escape sequences).

   To set the attribute/fg_color/bg_color we need to print
   "\033[AA;FF;BBm"
   where AA is one of these:
   00=none 01=bold 04=underscore 05=blink 07=reverse 08=concealed,
   FF is one of these:
   30=black 31=red 32=green 33=yellow 34=blue 35=magenta 36=cyan 37=white
   BB is one of these:
   40=black 41=red 42=green 43=yellow 44=blue 45=magenta 46=cyan 47=white

   Any of AA,FF,BB may be omited.

   To reset do default color/attribute we need to print
   "\033[m"

   originally written by "Konstantine 'KCC' Serebriany" <kcc@mcst.ru>
*/


#include "escgeneratorfactory.h"

#include "chartranslator.h"
#include "colors.h"
#include "colormap.h"

EscGeneratorFactory::EscGeneratorFactory()
{
}

CharTranslator *
EscGeneratorFactory::createCharTranslator()
{
  return new CharTranslator ();
}

ColorMap *
EscGeneratorFactory::createColorMap()
{
  ColorMap *cm = new ColorMap;

  (*cm)[ GREEN ] = "32";
  (*cm)[ RED ] = "31";
  (*cm)[ DARKRED ] = "31";
  (*cm)[ BLUE ] = "34";
  (*cm)[ BROWN ] = "31";
  (*cm)[ YELLOW ] = "33";
  (*cm)[ CYAN ] = "36";
  (*cm)[ PURPLE ] = "35";
  (*cm)[ PINK ] = "35";
  (*cm)[ ORANGE ] = "31";
  (*cm)[ BRIGHTORANGE ] = "31";
  (*cm)[ BRIGHTGREEN ] = "32";
  (*cm)[ DARKGREEN ] = "32";
  (*cm)[ BLACK ] = "30";
  (*cm)[ TEAL ] = "37";
  (*cm)[ GRAY ] = "37";
  (*cm)[ DARKBLUE ] = "34";

  cm->setDefault ("30;");

  return cm;
}

const string
LeftEsc(const string &face) 
{
    return "\033[" + face + "m";
}

// restore default color and shape
const string
RightEsc() 
{
    return "\033[m";
}

const string
EscGeneratorFactory::esc_tag(const string &t)
{
  string esc_string = t;
  
  if (startTagLen())
    esc_string += ";";
  
  return esc_string;
}

const string
EscGeneratorFactory::getPreTag()
{
  return LeftEsc(GeneratorFactory::getPreTag());
}

const string
EscGeneratorFactory::getPostTag()
{
  return RightEsc();
}

void
EscGeneratorFactory::addBold()
{
  addStartTag(esc_tag("01"));
}

void
EscGeneratorFactory::addItalic()
{
  // seems to be no code for italics
  // TODO: generate a warning?
}

void
EscGeneratorFactory::addUnderline()
{
  addStartTag(esc_tag("04"));
}

void
EscGeneratorFactory::addColor(const string &color)
{
  addStartTag(esc_tag(color));
}

