/*
Copyright 1994 Silicon Graphics, Inc. -- All Rights Reserved

If the Software is acquired by or on behalf of an entity of government
of  the  United States of America, the following provision applies: U.
S.  GOVERNMENT  RESTRICTED  RIGHTS  LEGEND:    Use,   duplication   or
disclosure of Software by the Government is subject to restrictions as
set forth in FAR 52.227-19(c)(2) or  subparagraph  (c)(1)(ii)  of  the
Rights  in  Technical  Data  and  Computer  Software  clause  at DFARS
252.227-7013 and/or in similar or successor clauses in the FAR, or the
DOD  or  NASA  FAR Supplement. Unpub-lished- rights reserved under the
Copyright  Laws  of  the  United  States.  Contractor/manufacturer  is
SILICON  GRAPHICS,  INC.,  2011  N. Shoreline Blvd., Mountain View, CA
94039- 7311.

Silicon Graphics, Inc. hereby grants  to  you  a  non-exclusive,  non-
transferable,  personal, paid-up license to use, modify and distribute
the Software solely with SGI computer products.  You must include,  in
all  copies  of  the  Software  and  any associated documentation, the
copyright notice and restricted rights legend set forth above.

THE SOFTWARE IS PROVIDED  TO  YOU  "AS-IS"  AND  WITHOUT  ANY  SUPPORT
OBLIGATION  OR  WARRANTY  OF  ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
INCLUDING WITHOUT  LIMITATION,  ANY  WARRANTY  OF  MERCHANTABILITY  OR
FITNESS  FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL SGI BE LIABLE FOR
SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF  LIABILITY,
ARISING  OUT  OF  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

You agree that you will not export or re-export the Software, directly
or  indirectly,  unless  (a)  the  Export  Administration of the U. S.
Department of Commerce explicitly permits the export or  re-export  of
the  Software  or  (b)  the  Office  of  Export Licensing of the U. S.
Department of Commerce has granted au-thorization to  you  in  writing
for the  export or re- export the Software.

If you fail to fulfill any  of  the  foregoing  obligations,  SGI  may
pursue  all  available  legal  remedies  to  enforce  these  terms and
conditions, and SGI may,  at  any  time  after  your  default  hereof,
terminate  the  license  and  rights  granted  to  you hereunder.  You
further agree that, if SGI terminates this license for  your  default,
you  will, within ten (10) days after any such termination, deliver to
SGI or  render  unusable  all  Software  originally  provided  to  you
hereunder and any copies thereof embodied in any medium.
*/


#include "cppArgs.h"

#include "genSetup.H"
#include "hciUpdateScreen.h"
#include "misCube.h"
#include "misTransferFunc.h"
#include "volInit.h"
#include "volBrick.h"
#include "volClipPlanes.H"
#include "volGeomUtils.h"


static void LoadTexMat(VRVolumeData *vd, Brick *br)
{
  matrix inv;

  matrix_copy(IdentityMatrix, 4, inv);
  matrix_translate(1.0, 1.0, 1.0, inv);
  matrix_scale(0.5, 0.5, 0.5, inv);
  matrix_scale(vd->nxBricks, vd->nyBricks, vd->nzBricks, inv);
  matrix_translate(-br->xOff, -br->yOff, -br->zOff, inv);
  matrix_translate(br->txOff, br->tyOff, 0.0, inv);
  matrix_scale(br->txScl, br->tyScl, 1.0, inv);

#ifdef VR_IRISGL
  mmode(MTEXTURE);
  loadmatrix(inv);
  mmode(MVIEWING);
#endif

#ifdef VR_OPENGL
  glMatrixMode(GL_TEXTURE);
  glLoadMatrixf((GLfloat*) inv);
  glMatrixMode(GL_MODELVIEW);
#endif
}


static void LoadClipPlanes(VRState *state, VRVolumeData *vd, Brick *br)
{
  int i, plane;

#ifdef VR_OPENGL
  int j;
  GLdouble glPlane[4];
#endif

  for (i=0; i<br->nPlanes; i++) {

    plane = (MAX_PLANES-1) - i;

    if (state->view->debug == 2)
      printf("Defining plane #%d as [%8.4f,%8.4f,%8.4f,%8.4f]\n", plane,
	     br->plane[i][0], br->plane[i][1], br->plane[i][2], br->plane[i][3]);

#ifdef VR_IRISGL
    clipplane(plane, CP_DEFINE, br->plane[i]);
    clipplane(plane, CP_ON, NULL);
#endif

#ifdef VR_OPENGL
    for (j=0; j<4; j++)
      glPlane[j] = br->plane[i][j];

    glClipPlane(GL_CLIP_PLANE0 + plane, glPlane);
    glEnable(GL_CLIP_PLANE0 + plane);
#endif
  }
}


static void UnloadClipPlanes(VRVolumeData *vd, Brick *b)
{
  int i, plane;

  for (i=0; i<b->nPlanes; i++) {

    plane = (MAX_PLANES-1) - i;

#ifdef VR_IRISGL
    clipplane(plane, CP_OFF, NULL);
#endif

#ifdef VR_OPENGL
    glDisable(GL_CLIP_PLANE0 + plane);
#endif
  }
}


/*
 * The classic quick sort, right out of Wirth... (page 79)
 */
static void QuickSort( int l, int r, Brick **blist )
{
  int   i,  j;
  Brick *x, *w;

  i = l;
  j = r;
  x = blist[(l+r) / 2];
 
  do {
    while (blist[i]->center.z < x->center.z) i++;
    while (blist[j]->center.z > x->center.z) j--;

    if ( i <= j ) {

      w        = blist[i];
      blist[i] = blist[j];
      blist[j] = w;

      i++;
      j--;
    }
  } while ( i <= j );

  if ( l < j ) QuickSort( l, j, blist );
  if ( i < r ) QuickSort( i, r, blist );
}


static void ReorderVolumeBricks( VRState *state, int Direction, int volNumber )
{
  int   i;
  float w;
  coord center;
  VRVolumeData *vd = state->volumeData[volNumber];

  /* Copy the current time's bricks to the sorted brick list */
  memcpy(vd->sbrick[state->mode->brickRes],
	 vd->brick[state->mode->brickRes][vd->curTime],
	 vd->nBricks * sizeof(Brick*));

  /* Rotate the brick centers into the eye coordinate system */
  for ( i = 0; i < vd->nBricks; i++ )
  {
    Brick *b = vd->sbrick[state->mode->brickRes][i];

    center.x = (b->xOff + 0.5) * 2.0 / vd->nxBricks - 1.0;
    center.y = (b->yOff + 0.5) * 2.0 / vd->nyBricks - 1.0;
    center.z = (b->zOff + 0.5) * 2.0 / vd->nzBricks - 1.0;

    coord_transform( center, vd->VTWMat, center );
    coord_transz( center, state->view->WTSMat, b->center );
    coord_transw( center, state->view->WTSMat, w );
    b->center.z /= -w * Direction;
  }

  /* Sort the bricks, smaller z to bigger z in the eye coordinate system */
  QuickSort( 0, vd->nBricks-1, vd->sbrick[state->mode->brickRes] );
}


long SurfaceColor(VRState *state, int s)
{
  /* Get the appropriate color for this surface */
  if (state->world->enumSurfColorString[s])
    return( state->world->enumSurfColor[s] );
  else
    if (state->surfaceData[s]->surfaceType == XY_SURFACE)
      return( state->world->xySurfColor );
    else
      return( state->world->zSurfColor );
}


static void DrawVector( VRState *state, coord *start, coord *end, long color )
{
  double theta, phi;
  coord dir;

#ifdef VR_IRISGL
  float update[10];
  float verts[8][3], norms[2][3];
#endif

#ifdef VR_OPENGL
  GLfloat update[10];
#endif

  /* Compute the vector direction */
  coord_sub( *end, *start, dir );

  /* Get the rotations around X and Y for this vector */
  VectorRotationAngles(&dir, &theta, &phi);

#ifdef VR_IRISGL
  /* Twist the world around */
  pushmatrix();
  translate( start->x, start->y, start->z );
  rot( -theta/M_PI*180, 'y' );
  rot(   -phi/M_PI*180, 'x' );
#endif

#ifdef VR_OPENGL
  /* Twist the world around */
  glPushMatrix();
  glTranslatef( start->x, start->y, start->z );
  glRotatef( -theta/M_PI*180, 0.0, 1.0, 0.0 );
  glRotatef(   -phi/M_PI*180, 1.0, 0.0, 0.0 );
#endif

#ifdef VR_IRISGL
  /* Update the front material color and alpha */
  update[0] = ALPHA;
  update[1] = ((color >> 24) & 0xff) / 255.0;
  update[2] = DIFFUSE;
  update[3] = ((color >>  0) & 0xff) / 255.0;
  update[4] = ((color >>  8) & 0xff) / 255.0;
  update[5] = ((color >> 16) & 0xff) / 255.0;
  update[6] = LMNULL;
  lmdef(DEFMATERIAL, 3, 3, update);
  lmbind(MATERIAL, 3);
  lmbind(BACKMATERIAL, 0);
#endif

#ifdef VR_OPENGL
  /* Update the front material color and alpha */
  update[0] = ((color >>  0) & 0xff) / 255.0;
  update[1] = ((color >>  8) & 0xff) / 255.0;
  update[2] = ((color >> 16) & 0xff) / 255.0;
  update[3] = ((color >> 24) & 0xff) / 255.0;
  glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, update);
#endif

#ifdef VR_IRISGL
  /* Active the materials (front and back) and turn on lighting */
  lmbind(LMODEL, 1);
  cpack( OPAQUE_WHITE );
#endif

#ifdef VR_OPENGL
  /* Active the materials (front and back) and turn on lighting */
  glEnable(GL_LIGHTING);
  glColor4f(COLOR_TO_OGL(OPAQUE_WHITE));
#endif

  {
    int i, angles = 18;
    float ah = 0.2;				/* Arrowhead size */
    float aw = 0.04;				/* Arrowhead width */
    float sw = 0.01;				/* Arrowstem width */

    float lf = 2.0 * M_PI * (angles-1) / angles;
    float lx = sin(lf);
    float ly = cos(lf);

    for (i=0; i<angles; i++) {

      float f = 2.0 * M_PI * i / angles;
      float x = sin(f);
      float y = cos(f);

#ifdef VR_IRISGL
      norms[0][0] =    lx;  norms[0][1] =    ly;  norms[0][2] = 0.0;
      norms[1][0] =     x;  norms[1][1] =     y;  norms[1][2] = 0.0;
      verts[0][0] =   0.0;  verts[0][1] =   0.0;  verts[0][2] = 1.0;
      verts[1][0] = aw*lx;  verts[1][1] = aw*ly;  verts[1][2] = 1.0-ah;
      verts[2][0] =  aw*x;  verts[2][1] =  aw*y;  verts[2][2] = 1.0-ah;
      verts[3][0] = sw*lx;  verts[3][1] = sw*ly;  verts[3][2] = 1.0-ah;
      verts[4][0] =  sw*x;  verts[4][1] =  sw*y;  verts[4][2] = 1.0-ah;
      verts[5][0] = sw*lx;  verts[5][1] = sw*ly;  verts[5][2] = 0.0;
      verts[6][0] =  sw*x;  verts[6][1] =  sw*y;  verts[6][2] = 0.0;
      verts[7][0] =   0.0;  verts[7][1] =   0.0;  verts[7][2] = 0.0;

      bgnqstrip();
      n3f(norms[0]);  v3f(verts[0]);
      n3f(norms[1]);  v3f(verts[0]);
      n3f(norms[0]);  v3f(verts[1]);
      n3f(norms[1]);  v3f(verts[2]);
      n3f(norms[0]);  v3f(verts[3]);
      n3f(norms[1]);  v3f(verts[4]);
      n3f(norms[0]);  v3f(verts[5]);
      n3f(norms[1]);  v3f(verts[6]);
      n3f(norms[0]);  v3f(verts[7]);
      n3f(norms[1]);  v3f(verts[7]);
      endqstrip();
#endif

#ifdef VR_OPENGL
      glBegin(GL_QUAD_STRIP);
      glNormal3f(lx, ly, 0.0);  glVertex3f(0.0, 0.0, 1.0);
      glNormal3f(x, y, 0.0);    glVertex3f(0.0, 0.0, 1.0);
      glNormal3f(lx, ly, 0.0);  glVertex3f(aw*lx, aw*ly, 1.0-ah);
      glNormal3f(x, y, 0.0);    glVertex3f(aw*x, aw*y, 1.0-ah);
      glNormal3f(lx, ly, 0.0);  glVertex3f(sw*lx, sw*ly, 1.0-ah);
      glNormal3f(x, y, 0.0);    glVertex3f(sw*x, sw*y, 1.0-ah);
      glNormal3f(lx, ly, 0.0);  glVertex3f(sw*lx, sw*ly, 0.0);
      glNormal3f(x, y, 0.0);    glVertex3f(sw*x, sw*y, 0.0);
      glNormal3f(lx, ly, 0.0);  glVertex3f(0.0, 0.0, 0.0);
      glNormal3f(x, y, 0.0);    glVertex3f(0.0, 0.0, 0.0);
      glEnd();
#endif

      lf = f;
      lx = x;
      ly = y;
    }
  }

#ifdef VR_IRISGL
  /* Return the stack to its previous state */
  popmatrix();
#endif

#ifdef VR_OPENGL
  glPopMatrix();
#endif

#ifdef VR_IRISGL
  /* Turn off lighting */
  lmbind(LMODEL, 0);
#endif

#ifdef VR_OPENGL
  /* Turn off lighting */
  glDisable(GL_LIGHTING);
#endif
}


static void DrawSurfVectors(VRState *state, int s)
{
  VRSurfaceData *sd = state->surfaceData[s];
  coord start;

  start.x = start.y = start.z = 0.0;

  DrawVector(state, &start, &sd->normal, SurfaceColor(state, s));
  DrawVector(state, &start, &sd->up, SurfaceColor(state, s));
}


static void DrawDirVectors(VRState *state)
{
  VRVolumeData *vd = state->volumeData[0];
  float start[3], end[3];
  float extents[3];
  int xdir, ydir, zdir;
  float xPlaneIntersect, yPlaneIntersect;
  float sinNorth, cosNorth;

  /* Compute the relative directions */
  xdir = (state->world->surfaceNormal+1) % 3;
  ydir = (state->world->surfaceNormal+2) % 3;
  zdir = (state->world->surfaceNormal+0) % 3;


  if (state->world->drawAxes) {

    start[xdir] = start[ydir] = start[zdir] = 0.0;

    /* Draw the X direction vector */
    end[ydir] = end[zdir] = 0.0;
    end[xdir] = 1.0;
    DrawVector(state, (coord*) start, (coord*) end, state->world->xVectorColor );
    
    /* Draw the Y direction vector */
    end[xdir] = end[zdir] = 0.0;
    end[ydir] = 1.0;
    DrawVector(state, (coord*) start, (coord*) end, state->world->yVectorColor );
    
    /* Draw the Z direction vector */
    end[xdir] = end[ydir] = 0.0;
    end[zdir] = 1.0;
    DrawVector(state, (coord*) start, (coord*) end, state->world->zVectorColor );
    
  } else {

    /* Compute the start and end of the up vector */
    if (state->world->nVols == 1) {

      /* Compute the box extents */
      extents[0] = 1.0 * vd->xScl;
      extents[1] = 1.0 * vd->yScl;
      extents[2] = 1.0 * vd->zScl;
      
      /* For one volume vectors jut from side of volume */
      start[xdir] = start[ydir] = 0.0;
      start[zdir] = extents[zdir];
      end[xdir] = end[ydir] = 0.0;
      end[zdir] = extents[zdir] + 0.4;
      
    } else {
      
      /* For multiple volumes just draw vectors from the origin */
      start[xdir] = start[ydir] = start[zdir] = 0.0;
      end[xdir] = 0.0;
      end[ydir] = 0.0;
      end[zdir] = 3.0;
    }
    
    /* Draw the up direction vector */
    DrawVector( state, (coord*) start, (coord*) end, state->world->upVectorColor );
    
    /* Compute the start and end of the north vector */
    sinNorth = sin(-state->world->northAngle);
    cosNorth = cos(-state->world->northAngle);
    
    if (state->world->nVols == 1) {

      /* For one volume vectors jut from side of volume */
      xPlaneIntersect = fabs(extents[xdir] / sinNorth);
      yPlaneIntersect = fabs(extents[ydir] / cosNorth);

      if (yPlaneIntersect < xPlaneIntersect) {

	/* Vector sticks out of Y plane */
	start[xdir] = yPlaneIntersect * sinNorth;
	start[ydir] = yPlaneIntersect * cosNorth;
	start[zdir] = 0.0;

	end[xdir] = (yPlaneIntersect * 1.4) * sinNorth; 
	end[ydir] = (yPlaneIntersect * 1.4) * cosNorth;
	end[zdir] = 0.0;

      } else {

	/* Vector sticks out of Y plane */
	start[xdir] = xPlaneIntersect * sinNorth;
	start[ydir] = xPlaneIntersect * cosNorth;
	start[zdir] = 0.0;

	end[xdir] = (xPlaneIntersect * 1.4) * sinNorth;
	end[ydir] = (xPlaneIntersect * 1.4) * cosNorth;
	end[zdir] = 0.0;

      }

    } else {
      
      /* For multiple volumes just draw vectors from the origin */
      start[xdir] = start[ydir] = start[zdir] = 0.0;
      end[xdir] = 3.0 * cosNorth;
      end[ydir] = 3.0 * sinNorth;
      end[zdir] = 0.0;
    }
    
    /* Draw the north direction vector */
    DrawVector(state, (coord*) start, (coord*) end, state->world->northVectorColor );
  }
}

static void DrawPoints( VRState *state, VRPointData *pd )
{
  int j;

#ifdef VR_IRISGL
  float p[3];
#endif

#ifdef VR_OPENGL
  GLfloat p[3];
#endif

#ifdef VR_IRISGL
  pntsize(state->mode->pixelRes ? 5 : 2);
#endif

#ifdef VR_OPENGL
  glPointSize(state->mode->pixelRes ? 5 : 2);
#endif

  for ( j=0; j<pd->nPoints; j++ )
  {
    if ( pd->pointMoving && (j == pd->curPoint) )
      if ( state->mode->sliceMode && state->mode->clipGeomMode )
	DisableAllClipPlanes( state );

    p[0] = pd->point[j].x;
    p[1] = pd->point[j].y;
    p[2] = pd->point[j].z;

#ifdef VR_IRISGL
    bgnpoint();

    if (j == pd->curPoint)
      cpack( state->world->curPointColor );
    else
      cpack( state->world->pointColor );

    v3f(p);
    endpoint();
#endif

#ifdef VR_OPENGL
    glBegin(GL_POINTS);

    if (j == pd->curPoint)
      glColor4f(COLOR_TO_OGL(state->world->curPointColor));
    else
      glColor4f(COLOR_TO_OGL(state->world->pointColor));

    glVertex3fv(p);
    glEnd();
#endif

    if ( pd->pointMoving && (j == pd->curPoint) )
      if ( state->mode->sliceMode && state->mode->clipGeomMode )
	EnableActiveClipPlanes( state, -1 );
  }
}


static void DrawTriMesh(coord *polys, coord *normals, int polyWidth, int polyHeight, int wireframe)
{
  int i, j;
  coord *vptr, *nptr;

  if (!wireframe) {

#ifdef VR_IRISGL
    /* Draw the surface */
    for (i=0; i<polyWidth-1; i++) {
      bgntmesh();
      for (j=0; j<polyHeight; j++) {
	int joff = polyWidth*j;
	t3f((float*) &(polys[  i+0 + joff]));
	n3f((float*) &(normals[i+0 + joff]));
	v3f((float*) &(polys[  i+0 + joff]));
	t3f((float*) &(polys[  i+1 + joff]));
	n3f((float*) &(normals[i+1 + joff]));
	v3f((float*) &(polys[  i+1 + joff]));
      }
      endtmesh();
    }
#endif

#ifdef VR_OPENGL
    /* Draw the surface */
    for (i=0; i<polyWidth-1; i++) {
      glBegin(GL_TRIANGLE_STRIP);
      for (j=0; j<polyHeight; j++) {
	int joff = polyWidth*j;
	glTexCoord3fv((GLfloat*) &(polys[  i+0 + joff]));
	glNormal3fv((GLfloat*) &(normals[i+0 + joff]));
	glVertex3fv((GLfloat*) &(polys[  i+0 + joff]));
	glTexCoord3fv((GLfloat*) &(polys[  i+1 + joff]));
	glNormal3fv((GLfloat*) &(normals[i+1 + joff]));
	glVertex3fv((GLfloat*) &(polys[  i+1 + joff]));
      }
      glEnd();
    }
#endif

  } else {

#ifdef VR_IRISGL
    for (i=0; i<polyWidth; i++) {
      bgnline();
      vptr = &(polys[i]);
      nptr = &(normals[i]);
      for (j=0; j<polyHeight; j++) {
	t3f((float*) vptr);
	n3f((float*) nptr);
	v3f((float*) vptr);
	vptr += polyWidth;
	nptr += polyWidth;
      }
      endline();
    }
    vptr = polys;
    nptr = normals;
    for (j=0; j<polyHeight; j++) {
      bgnline();
      for (i=0; i<polyWidth; i++) {
	t3f((float*) vptr);
	n3f((float*) nptr++);
	v3f((float*) vptr++);
      }
      endline();
    }
#endif

#ifdef VR_OPENGL
    for (i=0; i<polyWidth; i++) {
      glBegin(GL_LINE_STRIP);
      vptr = &(polys[i]);
      nptr = &(normals[i]);
      for (j=0; j<polyHeight; j++) {
	glTexCoord3fv((GLfloat*) vptr);
	glNormal3fv((GLfloat*) nptr);
	glVertex3fv((GLfloat*) vptr);
	vptr += polyWidth;
	nptr += polyWidth;
      }
      glEnd();
    }
    vptr = polys;
    nptr = normals;
    for (j=0; j<polyHeight; j++) {
      glBegin(GL_LINE_STRIP);
      for (i=0; i<polyWidth; i++) {
	glTexCoord3fv((GLfloat*) vptr);
	glNormal3fv((GLfloat*) nptr);
	glVertex3fv((GLfloat*) vptr++);
      }
      glEnd();
    }
#endif
  }
}


static void DrawSurface( VRState *state, VRSurfaceData *sd, int s, int forceMesh )
{
  int i, vol;
  long color;

#ifdef VR_IRISGL
  float update[10];
#endif

#ifdef VR_OPENGL
  GLfloat update[10];
#endif

  color = SurfaceColor(state, s);

#ifdef VR_IRISGL
  cpack( color );
#endif

#ifdef VR_OPENGL
  glColor4f(COLOR_TO_OGL(color));
#endif

#ifdef VR_IRISGL
  /* Turn off lighting */
  lmbind(LMODEL, 0);
#endif

#ifdef VR_OPENGL
  /* Turn off lighting */
  glDisable(GL_LIGHTING);
#endif

  if (sd->drawLighted) {

#ifdef VR_IRISGL
    /* Update the front material color and alpha */
    update[0] = ALPHA;
    update[1] = ((color >> 24) & 0xff) / 255.0;
    update[2] = DIFFUSE;
    update[3] = ((color >>  0) & 0xff) / 255.0;
    update[4] = ((color >>  8) & 0xff) / 255.0;
    update[5] = ((color >> 16) & 0xff) / 255.0;
    update[6] = LMNULL;
    lmdef(DEFMATERIAL, 3, 3, update);
    lmbind(MATERIAL, 3);
    lmbind(BACKMATERIAL, 0);
#endif

#ifdef VR_OPENGL
    /* Update the front material color and alpha */
    update[0] = ((color >>  0) & 0xff) / 255.0;
    update[1] = ((color >>  8) & 0xff) / 255.0;
    update[2] = ((color >> 16) & 0xff) / 255.0;
    update[3] = ((color >> 24) & 0xff) / 255.0;
    glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, update);
#endif

#ifdef VR_IRISGL
    /* Active the materials (front and back) and turn on lighting */
    lmbind(LMODEL, 1);
#endif

#ifdef VR_OPENGL
    /* Active the materials (front and back) and turn on lighting */
    glEnable(GL_LIGHTING);
#endif
  }

  if (sd->drawTextured) {

#ifdef VR_IRISGL
    cpack( OPAQUE_WHITE );
#endif

#ifdef VR_OPENGL
    glColor4f(COLOR_TO_OGL(OPAQUE_WHITE));
#endif
  }

  if (state->view->curVol != -1)
    vol = state->view->curVol;
  else
    if (state->world->nVols == 1)
      vol = 0;
    else
      vol = -1;

  if (sd->drawTextured && (vol != -1)) {

    VRVolumeData *vd = state->volumeData[vol];

    LoadTlut(state, vol);

    for (i=0; i<vd->nBricks; i++) {

      Brick *b = vd->brick[state->mode->brickRes][vd->curTime][i];

      LoadBrick(state, vd, b);
      LoadTexMat(vd, b);
      LoadClipPlanes(state, vd, b);

      DrawTriMesh(sd->polys, sd->normals, sd->polyWidth, sd->polyHeight,
		  forceMesh || sd->drawMeshed);

      UnloadClipPlanes(vd, b);
    }

    UnloadTlut(state);

  } else
    DrawTriMesh(sd->polys, sd->normals, sd->polyWidth, sd->polyHeight,
		forceMesh || sd->drawMeshed);

  if (sd->drawLighted) {

#ifdef VR_IRISGL
    /* Turn off lighting */
    lmbind(LMODEL, 0);
#endif

#ifdef VR_OPENGL
    /* Turn off lighting */
    glDisable(GL_LIGHTING);
#endif
  }
}


static void DrawPolyObjects( VRState *state )
{
  int i, j;

#ifdef VR_IRISGL
  float update[10];
#endif

#ifdef VR_OPENGL
  GLfloat update[10];
#endif

#ifdef VR_IRISGL
  /* Active the materials (front and back) and turn on lighting */
  lmbind(LMODEL, 1);
#endif

#ifdef VR_OPENGL
  /* Active the materials (front and back) and turn on lighting */
  glEnable(GL_LIGHTING);
#endif

#ifdef VR_IRISGL
  /* Update the front material color and alpha */
  update[0] = ALPHA;
  update[1] = ((state->world->frontPolyColor >> 24) & 0xff) / 255.0;
  update[2] = DIFFUSE;
  update[3] = ((state->world->frontPolyColor >> 16) & 0xff) / 255.0;
  update[4] = ((state->world->frontPolyColor >>  8) & 0xff) / 255.0;
  update[5] = ((state->world->frontPolyColor >>  0) & 0xff) / 255.0;
  update[6] = LMNULL;
  lmdef(DEFMATERIAL, 1, 3, update);
  lmbind(MATERIAL, 1);

  /* Update the back material color and alpha */
  update[0] = ALPHA;
  update[1] = ((state->world->backPolyColor >> 24) & 0xff) / 255.0;
  update[2] = DIFFUSE;
  update[3] = ((state->world->backPolyColor >> 16) & 0xff) / 255.0;
  update[4] = ((state->world->backPolyColor >>  8) & 0xff) / 255.0;
  update[5] = ((state->world->backPolyColor >>  0) & 0xff) / 255.0;
  update[6] = LMNULL;
  lmdef(DEFMATERIAL, 2, 3, update);
  lmbind(BACKMATERIAL, 2);
#endif

#ifdef VR_OPENGL
  /* Update the front material color and alpha */
  update[0] = ((state->world->frontPolyColor >> 24) & 0xff) / 255.0;
  update[1] = ((state->world->frontPolyColor >> 16) & 0xff) / 255.0;
  update[2] = ((state->world->frontPolyColor >>  8) & 0xff) / 255.0;
  update[3] = ((state->world->frontPolyColor >>  0) & 0xff) / 255.0;
  glMaterialfv(GL_FRONT, GL_DIFFUSE, update);

  /* Update the back material color and alpha */
  update[0] = ((state->world->backPolyColor >> 24) & 0xff) / 255.0;
  update[1] = ((state->world->backPolyColor >> 16) & 0xff) / 255.0;
  update[2] = ((state->world->backPolyColor >>  8) & 0xff) / 255.0;
  update[3] = ((state->world->backPolyColor >>  0) & 0xff) / 255.0;
  glMaterialfv(GL_BACK, GL_DIFFUSE, update);
#endif

  /* Draw the polygon objects */
  for (i=0; i<state->polygonData->nObjects; i++) {

    PolyObject *obj = &state->polygonData->object[i];

    switch (obj->type) {

    case DISJOINT_TRIANGLES:

      for (j=0; j<obj->nPoints; j++) {
#ifdef VR_IRISGL
	bgnpolygon();
	n3f( (float*) &(obj->normal[j]) );
	v3f( (float*) &(obj->point[j++]) );
	n3f( (float*) &(obj->normal[j]) );
	v3f( (float*) &(obj->point[j++]) );
	n3f( (float*) &(obj->normal[j]) );
	v3f( (float*) &(obj->point[j]) );
	endpolygon();
#endif

#ifdef VR_OPENGL
	glBegin(GL_POLYGON);
	glNormal3fv((GLfloat*) &(obj->normal[j]));
	glVertex3fv((GLfloat*) &(obj->point[j++]));
	glNormal3fv((GLfloat*) &(obj->normal[j]));
	glVertex3fv((GLfloat*) &(obj->point[j++]));
	glNormal3fv((GLfloat*) &(obj->normal[j]));
	glVertex3fv((GLfloat*) &(obj->point[j]));
	glEnd();
#endif
      }
      break;

    case QUADRILATERAL_STRIP:

#ifdef VR_IRISGL
      bgnqstrip();
      for (j=0; j<obj->nPoints; j++) {
	n3f( (float*) &(obj->normal[j]) );
	v3f( (float*) &(obj->point[j]) );
      }
      endqstrip();
#endif

#ifdef VR_OPENGL
      glBegin(GL_QUAD_STRIP);
      for (j=0; j<obj->nPoints; j++) {
	glNormal3fv((GLfloat*) &(obj->normal[j]));
	glVertex3fv((GLfloat*) &(obj->point[j]));
      }
      glEnd();
#endif
      break;
    }
  }

#ifdef VR_IRISGL
  /* Turn off lighting */
  lmbind(LMODEL, 0);
#endif

#ifdef VR_OPENGL
  /* Turn off lighting */
  glEnable(GL_LIGHTING);
#endif
}


void DrawElevation( VRState *state, VRElevData *ed )
{
  int i, j;

#ifdef VR_IRISGL
  float update[10];
  float tCoord[2];
#endif

#ifdef VR_OPENGL
  GLfloat update[10];
  GLfloat tCoord[2];
#endif

#ifdef VR_IRISGL
  pushmatrix();
  translate(ed->xTrn, ed->yTrn, ed->zTrn);
  rot(ed->xyRot, 'z');
  scale(ed->xScl, ed->yScl, ed->zScl);
#endif

#ifdef VR_OPENGL
  glPushMatrix();
  glTranslatef(ed->xTrn, ed->yTrn, ed->zTrn);
  glRotatef(ed->xyRot, 0.0, 0.0, 1.0);
  glScalef(ed->xScl, ed->yScl, ed->zScl);
#endif

  if (ed->haveTexture) {

#ifdef VR_IRISGL
    texbind(TX_TEXTURE_0, ed->texId);
    tevbind(TV_ENV0, ed->tevId);
#endif

#ifdef VR_OPENGL
#ifdef VR_TEXOBJS
    glBindTextureEXT(GL_TEXTURE_2D, ed->texId);
#endif
#ifdef VR_DISPLIST
    glCallList(ed->texId);
#endif

    glDisable(GL_TEXTURE_3D_EXT);
    glEnable(GL_TEXTURE_2D);
#endif

#ifdef VR_IRISGL
     mmode(MTEXTURE);
     loadmatrix(IdentityMatrix);
     mmode(MVIEWING);
#endif

#ifdef VR_OPENGL
     glMatrixMode(GL_TEXTURE);
     glLoadIdentity();
     glMatrixMode(GL_MODELVIEW);
#endif
  }

  if (ed->light) {

#ifdef VR_IRISGL
    /* Update the front material color and alpha */
    update[0] = ALPHA;
    update[1] = ((state->world->frontElevColor >> 24) & 0xff) / 255.0;
    update[2] = DIFFUSE;
    update[3] = ((state->world->frontElevColor >>  0) & 0xff) / 255.0;
    update[4] = ((state->world->frontElevColor >>  8) & 0xff) / 255.0;
    update[5] = ((state->world->frontElevColor >> 16) & 0xff) / 255.0;
    update[6] = LMNULL;
    lmdef(DEFMATERIAL, 4, 3, update);
    lmbind(MATERIAL, 4);

    /* Update the back material color and alpha */
    update[0] = ALPHA;
    update[1] = ((state->world->backElevColor >> 24) & 0xff) / 255.0;
    update[2] = DIFFUSE;
    update[3] = ((state->world->backElevColor >>  0) & 0xff) / 255.0;
    update[4] = ((state->world->backElevColor >>  8) & 0xff) / 255.0;
    update[5] = ((state->world->backElevColor >> 16) & 0xff) / 255.0;
    update[6] = LMNULL;
    lmdef(DEFMATERIAL, 5, 3, update);
    lmbind(BACKMATERIAL, 5);
#endif

#ifdef VR_OPENGL
    /* Update the front material color and alpha */
    update[0] = ((state->world->frontElevColor >>  0) & 0xff) / 255.0;
    update[1] = ((state->world->frontElevColor >>  8) & 0xff) / 255.0;
    update[2] = ((state->world->frontElevColor >> 16) & 0xff) / 255.0;
    update[3] = ((state->world->frontElevColor >> 24) & 0xff) / 255.0;
    glMaterialfv(GL_FRONT, GL_DIFFUSE, update);

    /* Update the front material color and alpha */
    update[0] = ((state->world->backElevColor >>  0) & 0xff) / 255.0;
    update[1] = ((state->world->backElevColor >>  8) & 0xff) / 255.0;
    update[2] = ((state->world->backElevColor >> 16) & 0xff) / 255.0;
    update[3] = ((state->world->backElevColor >> 24) & 0xff) / 255.0;
    glMaterialfv(GL_BACK, GL_DIFFUSE, update);
#endif

#ifdef VR_IRISGL
    /* Active the materials (front and back) and turn on lighting */
    lmbind(LMODEL, 1);
    cpack( OPAQUE_WHITE );
#endif

#ifdef VR_OPENGL
    /* Active the materials (front and back) and turn on lighting */
    glEnable(GL_LIGHTING);
    glColor4f(COLOR_TO_OGL(OPAQUE_WHITE));
#endif
  } else {
#ifdef VR_IRISGL
    cpack(state->world->frontElevColor);
#endif

#ifdef VR_OPENGL
    glColor4f(COLOR_TO_OGL(state->world->frontElevColor));
#endif
  }

#ifdef VR_IRISGL
  for (i=0; i<ed->polyWidth-1; i++) {
    bgntmesh();
    for (j=0; j<ed->polyHeight; j++) {
      int joff = ed->polyWidth*j;
      tCoord[0] = ed->txScl * (i+0) / (ed->polyWidth-1);
      tCoord[1] = ed->tyScl * (j+0) / (ed->polyHeight-1);
      t3f(tCoord);
      n3f((float*) &(ed->normals[i+0 + joff]));
      v3f((float*) &(ed->polys[  i+0 + joff]));
      tCoord[0] = ed->txScl * (i+1) / (ed->polyWidth-1);
      tCoord[1] = ed->tyScl * (j+0) / (ed->polyHeight-1);
      t3f(tCoord);
      n3f((float*) &(ed->normals[i+1 + joff]));
      v3f((float*) &(ed->polys[  i+1 + joff]));
    }
    endtmesh();
  }
#endif

#ifdef VR_OPENGL
  /* Draw the surface */
  for (i=0; i<ed->polyWidth-1; i++) {
    glBegin(GL_TRIANGLE_STRIP);
    for (j=0; j<ed->polyHeight; j++) {
      int joff = ed->polyWidth*j;
      tCoord[0] = ed->txScl * (i+0) / ed->polyWidth;
      tCoord[1] = ed->tyScl * (j+0) / ed->polyHeight;
      glTexCoord2fv(tCoord);
      glNormal3fv((GLfloat*) &(ed->normals[i+0 + joff]));
      glVertex3fv((GLfloat*) &(ed->polys[  i+0 + joff]));
      tCoord[0] = ed->txScl * (i+1) / ed->polyWidth;
      tCoord[1] = ed->tyScl * (j+0) / ed->polyHeight;
      glTexCoord2fv(tCoord);
      glNormal3fv((GLfloat*) &(ed->normals[i+1 + joff]));
      glVertex3fv((GLfloat*) &(ed->polys[  i+1 + joff]));
    }
    glEnd();
  }
#endif

  if (ed->haveTexture) {
#ifdef VR_IRISGL
    texbind(TX_TEXTURE_0, 0);
    tevbind(TV_ENV0, 0);
#endif

#ifdef VR_OPENGL
    glDisable(GL_TEXTURE_2D);
#endif
  }

  if (ed->light) {
#ifdef VR_IRISGL
    lmbind(LMODEL, 0);
#endif

#ifdef VR_OPENGL
    glDisable(GL_LIGHTING);
#endif
  }

#ifdef VR_IRISGL
  popmatrix();
#endif

#ifdef VR_OPENGL
  glPopMatrix();
#endif
}

static void SetFont(VRState *state, int i)
{
  if (i == -1) {
    float avg = (state->view->width+state->view->height) / 2.0;
    i = (avg-500.0) / 250.0;
    if (i < 0)
      i = 0;
    else if (i >= state->view->numFonts)
      i = state->view->numFonts-1;
  }

#ifdef VR_OPENGL
  glListBase(state->view->fontBase[i]);
#endif
}


static void DrawString(VRState *state, char *s, float x, float y, float z)
{
  if (!state->view->noFonts) {
#ifdef VR_OPENGL
    glRasterPos3f(x, y, z);
    glCallLists(strlen(s), GL_UNSIGNED_BYTE, (GLubyte*) s);
#endif
  }
}


static void DrawText(VRState *state)
{
  /* Draw the text */
  char str[512];
  float y = -1.0;

#ifdef VR_OPENGL
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-1.0, 1.0, -1.0, 1.0, -1.0, 1.0);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glScalef(0.95, 0.95, 0.95);

  SetFont(state, 1);
  glColor3f(1.0, 1.0, 1.0);

  /* Draw the selected clip plane information */
  {
    int n = state->view->curPlane;

    DrawString(state, "Selected Clip Plane :", -1.0, y, 0.0);

    if (n != -1) {
      Plane *p = &(state->planeData->plane[n]);
      sprintf(str, "#%d  ( %4.2fx + %4.2fy + %4.2fz + %4.2f = 0 )",
              n+1, p->a, p->b, p->c, p->d);
    } else
      sprintf(str, "None");
    DrawString(state, str, -0.4, y, 0.0); y += 0.07;
  }

  /* Draw the selected surface information */
  {
    int s = state->view->curSurf;

    DrawString(state, "Selected Surface :", -1.0, y, 0.0);

    if (s != -1)
      sprintf(str, "#%d", s+1);
    else
      sprintf(str, "None");
    DrawString(state, str, -0.4, y, 0.0); y += 0.07;
  }

  /* Draw the selected volume information */
  {
    int v = state->view->curVol;

    DrawString(state, "Selected Volume :", -1.0, y, 0.0);

    if (v != -1) {
      VRVolumeData *vd = state->volumeData[v];
      if (vd->tRes > 1)
	sprintf(str, "#%d  ( Time %02d )", v+1, vd->curTime+1);
      else
	sprintf(str, "#%d", v+1);
    } else
      sprintf(str, "None");
    DrawString(state, str, -0.4, y, 0.0); y += 0.07;
  }

  SetFont(state, 3);
  if (state->world->macrofps >= state->world->maxfps)
    sprintf(str, "%4.1f+ fps", state->world->maxfps);
  else
    sprintf(str, "%4.1f fps", state->world->macrofps);
  DrawString(state, str, -1.0, 0.93, 0.0);

  glListBase(0);
#endif
}


void FocusView( VRState *state )
{
#ifdef VR_IRISGL
  GLXwinset(state->view->disp, state->view->wind);

  mmode(MPROJECTION);
  ortho(0.0, 1.0, 0.0, 1.0, 0.0, 1.0);
  mmode(MVIEWING);
  loadmatrix(IdentityMatrix);

  zbuffer(FALSE);
  blendfunction(BF_ONE, BF_ZERO);

  viewport(0, state->view->width-1, 0, state->view->height-1);
#endif

#ifdef VR_OPENGL
  glXMakeCurrent(state->view->disp, state->view->wind, state->view->context);

  glMatrixMode(GL_PROJECTION);
  glOrtho(0.0, 1.0, 0.0, 1.0, 0.0, 1.0);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  glDisable(GL_DEPTH_TEST);
  glDisable(GL_BLEND);

  glViewport(0, 0, state->view->width-1, state->view->height-1);
#endif
}


void StorePerVolumeMatrix( VRState *state, int volNumber )
{
  VRVolumeData *vd = state->volumeData[volNumber];

  /* Build the Volume-To-World matrix */
  matrix_copy( IdentityMatrix, 4, vd->VTWMat );
  matrix_scale( vd->xScl, vd->yScl, vd->zScl, vd->VTWMat );
  matrix_scale( vd->uxScl, vd->uyScl, vd->uzScl, vd->VTWMat );
  matrix_mult_safe( vd->VTWMat, vd->InvRotMat, vd->VTWMat );
  matrix_translate( vd->xTrn, vd->yTrn, vd->zTrn, vd->VTWMat );
  matrix_invert4( vd->VTWMat, vd->WTVMat );

  /*
   * Build the Volume-To-Rotated matrix
   *
   * This transformation takes a volume all the way
   *  into rotated space, the only transforms required
   *  after this are global scales and projections.
   */
  matrix_mult( vd->VTWMat, state->view->InvRotMat, vd->VTRMat );
  matrix_invert4( vd->VTRMat, vd->RTVMat );
}


int RenderDirection = 1;


void UpdateView( VRState *state )
{
  VRVolumeData *vd;
  int            i, j;
  int            activePlanes = 0;
  int            curSurf = state->view->curSurf;
  int            volNumber;
  int            elevNumber;
  int            activeVols;
  matrix         mat;
  struct timeval end;

#ifdef VR_IRISGL
  unsigned long  zmask;
#endif

#ifdef VR_OPENGL
  GLenum zmask;
#endif

  FocusView(state);

  if (!state->view->boundFonts) {

    int i;

#ifdef VR_IRISGL
#endif

#ifdef VR_OPENGL
    /* Define the font(s) */
    state->view->noFonts = FALSE;

    for (i=0; i<state->view->numFonts; i++) {
      state->view->font[i] = XLoadFont(state->view->disp,
				       state->view->fontName[i]);
      if (state->view->font[i] == BadName)
	fprintf(stderr, "Unable to properly allocate font \"%s\"!\n",
		state->view->fontName[i]);
      else {
	XFontStruct *fontinfo = XQueryFont(state->view->disp, state->view->font[i]);
	state->view->fontBase[i] = glGenLists(128);
	glXUseXFont(state->view->font[i], 0, 127, state->view->fontBase[i]);
	XFreeFontInfo(NULL, fontinfo, 0);
      }
    }
#endif

    state->view->boundFonts = TRUE;
  }

#ifdef BIDIRECTIONAL_RENDERING
  RenderDirection *= -1;
#endif

#ifdef VR_IRISGL
  if ( state->mode->frontbufMode )
     frontbuffer( TRUE );
  else
     frontbuffer( FALSE );
#endif

#ifdef VR_OPENGL
  if ( state->world->stereoMode )
    if ( state->mode->stereoEye == RIGHT )
      if ( state->mode->frontbufMode )
	glDrawBuffer(GL_RIGHT);
      else
	glDrawBuffer(GL_BACK_RIGHT);
    else
      if ( state->mode->frontbufMode )
	glDrawBuffer(GL_LEFT);
      else
	glDrawBuffer(GL_BACK_LEFT);
  else
    if ( state->mode->frontbufMode )
      glDrawBuffer(GL_FRONT_AND_BACK);
    else
      glDrawBuffer(GL_BACK);
#endif

  /* Move the slice planes if looping is activated */
  if ( state->mode->loopMode )
    for (i=0; i<state->planeData->nPlanes; i++) {

      Plane *plane = &state->planeData->plane[i];
      if (plane->active) {

	plane->d += state->view->delta * (plane->loopDir ? 1.0 : -1.0);

	if ( plane->d >= plane->maxD || plane->d <= plane->minD ) {
	  plane->loopDir = !plane->loopDir;
	  plane->d = (plane->d >= plane->maxD) ? plane->maxD : plane->minD;
	}
      }
    }

#ifdef VR_IRISGL
  if ( !state->mode->pixelRes )
    viewport(0, state->view->width/2, 0, state->view->height/2);
  else
    viewport(0, state->view->width-1, 0, state->view->height-1);
#endif

#ifdef VR_OPENGL
  if ( !state->mode->pixelRes )
    glViewport(0, 0, state->view->width/2, state->view->height/2);
  else
    glViewport(0, 0, state->view->width-1, state->view->height-1);
#endif

#ifdef VR_IRISGL
  pixmode( PM_SIZE, 24 );
  pixmode( PM_INPUT_FORMAT, PM_RGB );
  pixmode( PM_INPUT_TYPE,   PM_UNSIGNED_BYTE );

  pixmode( PM_OUTPUT_FORMAT, PM_RGB );
  pixmode( PM_OUTPUT_TYPE,   PM_UNSIGNED_BYTE );
#endif

  /*
   * If we are in multi-clip plane non-volume mode, we must
   * opaquify the slices.  They cannot be rendered correctly
   * in a transparent form without splitting and sorting the
   * clip planes (which we do not do!)
   */
  for (i=0; i<state->planeData->nPlanes; i++)
    if (state->planeData->plane[i].active)
      activePlanes++;

  if ( !state->mode->volumeMode && state->mode->sliceMode && (activePlanes > 1) ) {
    if (!state->tableData->opaquify) {
      state->tableData->tableRedef = TRUE;
      state->tableData->opaquify   = TRUE;
    }
  } else {
    if (state->tableData->opaquify) {
      state->tableData->tableRedef = TRUE;
      state->tableData->opaquify   = FALSE;
    }
  }

  InstallTables( state );
      
#ifdef VR_IRISGL
  /* Now set up the global transformation matrix */
  mmode(MPROJECTION);
  loadmatrix( IdentityMatrix );
#endif

#ifdef VR_OPENGL
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
#endif

#ifdef VR_IRISGL
  if ( !state->view->snapping )

    if ( state->mode->perspMode )
      perspective( (Angle) state->mode->perspFov, state->view->aspect,
		  state->mode->perspNear, state->mode->perspFar );
    else
      ortho( -state->view->aspect, state->view->aspect, -1.0, 1.0,
	    state->mode->perspNear, state->mode->perspFar );

  else

    fprintf(stderr, "Can't snap in IrisGL!\n");
#endif

#ifdef VR_OPENGL
  if ( !state->view->snapping )

    if ( state->mode->perspMode )
      gluPerspective(0.1 * ((GLfloat) state->mode->perspFov), state->view->aspect,
		     state->mode->perspNear, state->mode->perspFar );
    else
      glOrtho(-state->view->aspect, state->view->aspect, -1.0, 1.0,
	      state->mode->perspNear, state->mode->perspFar );

  else

    if ( state->mode->perspMode )
      glFrustum( state->view->snapLft, state->view->snapRgt,
		 state->view->snapBot, state->view->snapTop,
		 state->mode->perspNear, state->mode->perspFar );
    else
      glOrtho( state->view->snapLft, state->view->snapRgt,
	       state->view->snapBot, state->view->snapTop,
	       state->mode->perspNear, state->mode->perspFar );
#endif

#ifdef VR_IRISGL
  /* Build the Camera-To-Screen matrix */
  getmatrix( state->view->CTSMat );
  mmode(MVIEWING);
  loadmatrix( IdentityMatrix );
#endif

#ifdef VR_OPENGL
  glGetFloatv(GL_PROJECTION_MATRIX, (GLfloat*) state->view->CTSMat);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
#endif

  /* Build the Rotated-To-Camera matrix */
  matrix_copy( IdentityMatrix, 4, state->view->RTCMat );
  if ( state->mode->perspMode )
    matrix_scale( state->mode->perspScale, state->mode->perspScale,
		  state->mode->perspScale, state->view->RTCMat );
  matrix_scale( state->view->userScale, state->view->userScale,
	        state->view->userScale, state->view->RTCMat);

#ifdef VR_STEREO
  /* Handle stereo separation and rotation */
  if (state->world->stereoMode)
    if (state->mode->stereoEye == RIGHT) {

#ifdef VR_IRISGL
      translate( -state->mode->stereoSep, 0.0, 0.0 );
      rot( -state->mode->stereoRot, 'y' );
      getmatrix( mat );
      rightbuffer( TRUE );
      leftbuffer( FALSE );
#endif

#ifdef VR_OPENGL
      glLoadIdentity();
      glTranslatef(-state->mode->stereoSep, 0.0, 0.0);
      glRotatef(-state->mode->stereoRot, 0.0, 1.0, 0.0);
      glGetFloatv(GL_MODELVIEW_MATRIX, (GLfloat*) mat);
#endif

      matrix_mult_safe( state->view->RTCMat, mat, state->view->RTCMat );

    } else {

#ifdef VR_IRISGL
      loadmatrix( IdentityMatrix );
      translate( state->mode->stereoSep, 0.0, 0.0 );
      rot( state->mode->stereoRot, 'y' );
      getmatrix( mat );
      rightbuffer( FALSE );
      leftbuffer( TRUE );
#endif

#ifdef VR_OPENGL
      glLoadIdentity();
      glTranslatef(state->mode->stereoSep, 0.0, 0.0);
      glRotatef(state->mode->stereoRot, 0.0, 1.0, 0.0);
      glGetFloatv(GL_MODELVIEW_MATRIX, (GLfloat*) mat);
#endif
      
      matrix_mult_safe( state->view->RTCMat, mat, state->view->RTCMat );
    }
#endif

  /* Push the world back away from the eye */
  matrix_translate( 0.0, 0.0, state->view->zTrans, state->view->RTCMat );

  /* Insert the rotation for the phantom eye */
  matrix_mult_safe(state->view->PhanRotMat, state->view->RTCMat, state->view->RTCMat);

  /* Build the World-To-Camera matrix */
  matrix_mult( state->view->InvRotMat, state->view->RTCMat, state->view->WTCMat );
  matrix_invert4( state->view->WTCMat, state->view->CTWMat );

  /* Build the World-To-Screen matrix */
  matrix_mult( state->view->WTCMat, state->view->CTSMat, state->view->WTSMat );

#ifdef VR_IRISGL
  /* Clear the buffer and get ready to draw */
  cpack( state->world->backColor );
  clear();
  zbuffer( TRUE );
  zwritemask(zmask = 0xffffffff);
  zclear();
#endif

#ifdef VR_OPENGL
  /* Clear the buffer and get ready to draw */
  glClearColor(COLOR_TO_OGL(state->world->backColor));
  glClearDepth(1.0);
  glEnable(GL_DEPTH_TEST);
  glDepthMask(zmask = GL_TRUE);
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
#endif
  
#ifdef VR_IRISGL
  loadmatrix(state->view->WTCMat);
#endif

#ifdef VR_OPENGL
  glLoadMatrixf((GLfloat*) state->view->WTCMat);
#endif

  /* Draw direction vectors */
  if (state->mode->vectorMode)
    if ((curSurf >= 0) && (state->surfaceData[curSurf]->haveSurf))
      DrawSurfVectors(state, curSurf);
    else
      DrawDirVectors(state);

  /* Enable the clip planes, if necessary */
  if ( state->mode->sliceMode && state->mode->clipGeomMode ) {
#ifdef VR_IRISGL
    pushmatrix();
    multmatrix( state->view->RotMat );
#endif
    
#ifdef VR_OPENGL
    glPushMatrix();
    glMultMatrixf((GLfloat*) state->view->RotMat);
#endif

    DefineClipPlanes( state, NULL );
    EnableActiveClipPlanes( state, -1 );

#ifdef VR_IRISGL
    popmatrix();
#endif

#ifdef VR_OPENGL
    glPopMatrix();
#endif
  }

  /* Draw the points */
  if (state->mode->pointsMode)
    if (curSurf == -1)
      for (i=0; i<MAX_SURFS; i++)
	DrawPoints( state, state->pointData[i] );
    else
      DrawPoints( state, state->pointData[curSurf] );

  /* Draw the surface */
  if (state->mode->surfaceMode)
    for (i=0; i<MAX_SURFS; i++)
      if (state->surfaceData[i]->haveSurf && state->surfaceData[i]->active)
	if (curSurf == i)
	  DrawSurface(state, state->surfaceData[i], i, FALSE);
	else
	  DrawSurface(state, state->surfaceData[i], i, (curSurf != -1));

  /* Draw the polygon objects */
  if (state->mode->surfaceMode && state->world->polygonFile)
    DrawPolyObjects( state );

  /* Disable the clip planes */
  if ( state->mode->sliceMode && state->mode->clipGeomMode )
    DisableAllClipPlanes( state );

  if ( state->mode->elevMode )
    for (elevNumber=0; elevNumber<state->world->nElevs; elevNumber++)
      DrawElevation( state, state->elevData[elevNumber] );

  activeVols = 0;

  /*
   * Loop through all the volumes and bricks to generate the volume outlines
   */
  for ( volNumber = 0; volNumber < state->world->nVols; volNumber++ ) {

    vd = state->volumeData[volNumber];
    if (vd->active) {

      activeVols++;

      /* Build the per volume matrices for this volume */
      StorePerVolumeMatrix( state, volNumber );

#ifdef VR_IRISGL
      /* Load the global scale matrix (other transformations will be done by hand) */
      loadmatrix(state->view->RTCMat);
#endif

#ifdef VR_OPENGL
      /* Load the global scale matrix (other transformations will be done by hand) */
      glLoadMatrixf((GLfloat*) state->view->RTCMat);
#endif

      /* Sort the bricks back to front */
      ReorderVolumeBricks( state, RenderDirection, volNumber );

#ifdef VR_IRISGL
      linesmooth( SML_OFF );
      zwritemask(zmask = 0xffffffff);
#endif

#ifdef VR_OPENGL
      glDisable(GL_LINE_SMOOTH);
      glDepthMask(zmask = GL_TRUE);
#endif

      /* Render a NULL brick to draw the volume outline/frame */
      RenderBrick(state, vd, &(vd->volBrick), zmask, FALSE, TRUE, TRUE, FALSE,
		  RenderDirection, TRUE, TRUE, volNumber);
    }
  }

#ifdef VR_IRISGL
  zbuffer( TRUE );
  if ( state->mode->volumeMode )
    zwritemask(zmask = 0x00000000);
  else 
    zwritemask(zmask = 0xffffffff);
#endif

#ifdef VR_OPENGL
  glEnable(GL_DEPTH_TEST);
  if ( state->mode->volumeMode )
    glDepthMask(zmask = GL_FALSE);
  else
    glDepthMask(zmask = GL_TRUE);
#endif

  if (activeVols == 1) {

    /* Find the active volume */
    volNumber = 0;
    while ( !state->volumeData[volNumber]->active )
      volNumber++;
    vd = state->volumeData[volNumber];

#ifdef VR_IRISGL
    /* Load the global scale matrix (other transformations will be done by hand) */
    loadmatrix( state->view->RTCMat );
#endif

#ifdef VR_OPENGL
    /* Load the global scale matrix (other transformations will be done by hand) */
    glLoadMatrixf((GLfloat*) state->view->RTCMat);
#endif

    LoadTlut(state, volNumber);
    
    /* Loop over each brick and render it if there's a single volume	*/
    for (j=0; j<vd->nBricks; j++)
      RenderBrick(state, vd, vd->sbrick[state->mode->brickRes][j], zmask,
		  TRUE, (state->view->debug == 8), FALSE, TRUE, 
		  RenderDirection, TRUE, TRUE, volNumber);

  } else {

    RenderSetup( state, RenderDirection );

    if ( state->mode->sliceMode ) {

#ifdef VR_IRISGL
      zwritemask(0xffffffff);
#endif

#ifdef VR_OPENGL
      glDepthMask(GL_TRUE);
#endif

      RenderSlices( state, RenderDirection, TRUE, FALSE );

#ifdef VR_IRISGL
      zwritemask(zmask);
#endif

#ifdef VR_OPENGL
      glDepthMask(zmask);
#endif
    }

    RenderPlanes( state, TRUE, RenderDirection );

    if ( state->mode->sliceMode ) {

#ifdef VR_IRISGL
      zwritemask(0xffffffff);
#endif

#ifdef VR_OPENGL
      glDepthMask(GL_TRUE);
#endif

      RenderSlices( state, RenderDirection, FALSE, TRUE );

#ifdef VR_IRISGL
      zwritemask(zmask);
#endif

#ifdef VR_OPENGL
      glDepthMask(zmask);
#endif
    }
  }

  UnloadTlut(state);

#ifdef VR_IRISGL
  blendfunction(BF_ONE, BF_ZERO);
  zbuffer(FALSE);
#endif

#ifdef VR_OPENGL
  glBlendFunc(GL_ONE, GL_ZERO);
  glDisable(GL_DEPTH_TEST);
#endif

  /*
   * We can get about a factor of 4 speed up
   * if we draw things 4 times as small and
   * then pixel replicate up to "full" screen.
   */
  if (!state->mode->pixelRes) {

    /* Buf points to a 4096 page aligned chunk'o memory */
    unsigned long buffer[1024*1024*4], *buf;

    buf = buffer + 4095;
    buf = (unsigned long*) ((unsigned int) buf & 0xfffff000);

#ifdef VR_IRISGL
    rectzoom( 1.0, 1.0 );
    lrectread(  0, 0, state->view->width/2, state->view->height/2, buf );

    viewport(0, state->view->width-1, 0, state->view->height-1);

    rectzoom( 2.0, 2.0 );
    lrectwrite( 0, 0, state->view->width/2, state->view->height/2, buf );
#endif

#ifdef VR_OPENGL
    glPixelZoom(1.0, 1.0);
    glReadPixels(0, 0, state->view->width/2, state->view->height/2,
		 GL_RGBA, GL_UNSIGNED_BYTE, buf);

    glPixelZoom(2.0, 2.0);
    glViewport(0, 0, state->view->width-1, state->view->height-1);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0.0, state->view->width-1.0, 0.0, state->view->height-1.0, -1.0, 1.0);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    glRasterPos2i(0, 0);
    glDrawPixels(state->view->width/2, state->view->height/2,
		 GL_RGBA, GL_UNSIGNED_BYTE, buf);
#endif
  }

  /* Compute the micro fps timing information */
  if (!state->view->snapping && (state->mode->stereoEye == LEFT)) {

    long mdiff;
    float fps, mratio;

    gettimeofday(&end, NULL);

    mdiff = (end.tv_sec  - state->world->lastFrame.tv_sec)  * 1000 +
            (end.tv_usec - state->world->lastFrame.tv_usec) / 1000;
    fps = 1000.0 / mdiff;

    /* Clamp the micro fps to the maximum fps allowable */
    if (fps > state->world->maxfps)
      fps = state->world->maxfps;

    /* Set the microfps and save the frame time */
    state->world->microfps = fps * 0.6 + state->world->microfps * 0.4;
    memcpy(&state->world->lastFrame, &end, sizeof(struct timeval));

    mdiff = (end.tv_sec  - state->world->lastUpdate.tv_sec)  * 1000 +
            (end.tv_usec - state->world->lastUpdate.tv_usec) / 1000;
    mratio = state->world->microfps / state->world->macrofps;

    /* Update the macro fps every three seconds or if the rate changes by 30% or more */
    if ((mratio < 0.7) || (mratio > 1.3) || (mdiff > 3000)) {

      /* Yes, reset the macrofps */
      state->world->macrofps = state->world->microfps;
      memcpy(&state->world->lastUpdate, &end, sizeof(struct timeval));
    }
  }

  if (!state->view->snapping && state->mode->textMode)
    DrawText(state);

  if (!state->view->snapping) {
#ifdef VR_STEREO
    /* If in stereo, draw the other eye! */
    if (state->world->stereoMode) {
      if (state->mode->stereoEye == LEFT) {

	state->mode->stereoEye = RIGHT;
	UpdateView( state );

      } else {

	state->mode->stereoEye = LEFT;

#ifdef VR_IRISGL
	leftbuffer(TRUE);
	rightbuffer(FALSE);
	swapbuffers();
#endif

#ifdef VR_OPENGL
	glDrawBuffer(GL_BACK_LEFT);
	glXSwapBuffers(state->view->disp, state->view->wind);
#endif

      }
    } else {

#ifdef VR_IRISGL
      swapbuffers();
#endif

#ifdef VR_OPENGL
      glXSwapBuffers(state->view->disp, state->view->wind);
#endif
    }

#else

#ifdef VR_IRISGL
    swapbuffers();
#endif

#ifdef VR_OPENGL
    glXSwapBuffers(state->view->disp, state->view->wind);
#endif
#endif
  }

#ifdef VR_OPENGL
  glFinish();
#endif
}
