/*
Copyright 1994 Silicon Graphics, Inc. -- All Rights Reserved

If the Software is acquired by or on behalf of an entity of government
of  the  United States of America, the following provision applies: U.
S.  GOVERNMENT  RESTRICTED  RIGHTS  LEGEND:    Use,   duplication   or
disclosure of Software by the Government is subject to restrictions as
set forth in FAR 52.227-19(c)(2) or  subparagraph  (c)(1)(ii)  of  the
Rights  in  Technical  Data  and  Computer  Software  clause  at DFARS
252.227-7013 and/or in similar or successor clauses in the FAR, or the
DOD  or  NASA  FAR Supplement. Unpub-lished- rights reserved under the
Copyright  Laws  of  the  United  States.  Contractor/manufacturer  is
SILICON  GRAPHICS,  INC.,  2011  N. Shoreline Blvd., Mountain View, CA
94039- 7311.

Silicon Graphics, Inc. hereby grants  to  you  a  non-exclusive,  non-
transferable,  personal, paid-up license to use, modify and distribute
the Software solely with SGI computer products.  You must include,  in
all  copies  of  the  Software  and  any associated documentation, the
copyright notice and restricted rights legend set forth above.

THE SOFTWARE IS PROVIDED  TO  YOU  "AS-IS"  AND  WITHOUT  ANY  SUPPORT
OBLIGATION  OR  WARRANTY  OF  ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
INCLUDING WITHOUT  LIMITATION,  ANY  WARRANTY  OF  MERCHANTABILITY  OR
FITNESS  FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL SGI BE LIABLE FOR
SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF  LIABILITY,
ARISING  OUT  OF  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

You agree that you will not export or re-export the Software, directly
or  indirectly,  unless  (a)  the  Export  Administration of the U. S.
Department of Commerce explicitly permits the export or  re-export  of
the  Software  or  (b)  the  Office  of  Export Licensing of the U. S.
Department of Commerce has granted au-thorization to  you  in  writing
for the  export or re- export the Software.

If you fail to fulfill any  of  the  foregoing  obligations,  SGI  may
pursue  all  available  legal  remedies  to  enforce  these  terms and
conditions, and SGI may,  at  any  time  after  your  default  hereof,
terminate  the  license  and  rights  granted  to  you hereunder.  You
further agree that, if SGI terminates this license for  your  default,
you  will, within ten (10) days after any such termination, deliver to
SGI or  render  unusable  all  Software  originally  provided  to  you
hereunder and any copies thereof embodied in any medium.
*/


#include <sys/time.h>

#include "hciHandlers.h"
#include "hciUpdateScreen.h"
#include "genVolRen.h"
#include "genWindowMaint.h"
#include "volGeomUtils.h"
#include "volInit.h"


/*
 * MOUSE X/Y filtering weights, these must add to 1
 */
#define W1 0.1
#define W2 0.2
#define W3 0.7


int SpinVolume( VRState *state, int volNumber, VRMotion *motion, matrix *imat )
{
  matrix mat, rotMat;

  if (motion == NULL)
    if (volNumber == -1) {
      motion = &state->view->motion;
      matrix_copy(state->view->RotMat, 4, mat);
    } else {
      motion = &state->volumeData[volNumber]->motion;
      matrix_copy(state->volumeData[volNumber]->RotMat, 4, mat);
    }
  else
    matrix_copy(*imat, 4, mat);

  if ( ComputeVsphereMatrix( state, motion->xLast, motion->yLast,
                             motion->xWin[0]*W1 + motion->xWin[1]*W2 + motion->xWin[2]*W3,
                             motion->yWin[0]*W1 + motion->yWin[1]*W2 + motion->yWin[2]*W3,
                             rotMat, False ) ) {

    if (imat == NULL)
      if (volNumber == -1) {
	matrix_mult_safe(rotMat, mat, mat);
        matrix_copy(mat, 4, state->view->RotMat);
        matrix_transpose(mat, 4, state->view->InvRotMat);
        state->view->stateChanged = True;
      } else {
	matrix_mult_safe(mat, state->view->InvRotMat, mat);
	matrix_mult_safe(mat, rotMat, mat);
	matrix_mult_safe(mat, state->view->RotMat, mat);
        matrix_copy(mat, 4, state->volumeData[volNumber]->RotMat);
        matrix_transpose(mat, 4, state->volumeData[volNumber]->InvRotMat);
        state->volumeData[volNumber]->stateChanged = True;
      }
    else {
      matrix_mult_safe(rotMat, mat, mat);
      matrix_copy(mat, 4, *imat);
    }

    /* Caller should redraw */
    return( TRUE );
  }

  return( FALSE );
}


void MoveVolume(VRState *state, VRVolumeData *vd, float dx, float dy, float dz)
{
  coord v1, v2;

  v1.x = -dx;
  v1.y = -dy;
  v1.z = -dz*3.0;

  /* Transform the vector backwards by the current world rotation */
  coord_transform(v1, state->view->RotMat, v2);

  vd->xTrn += v2.x;
  if (vd->xTrn > 100.0)
    vd->xTrn = 100.0;
  else if (vd->xTrn < -100.0)
    vd->xTrn = -100.0;

  vd->yTrn += v2.y;
  if (vd->yTrn > 100.0)
    vd->yTrn = 100.0;
  else if (vd->yTrn < -100.0)
    vd->yTrn = -100.0;

  vd->zTrn += v2.z;
  if (vd->zTrn > 100.0)
    vd->zTrn = 100.0;
  else if (vd->zTrn < -100.0)
    vd->zTrn = -100.0;
}


void TrackMovingVolume(VRState *state, int xCur, int yCur, int track, int volNumber)
{
  VRMotion *motion;

  if (volNumber == -1)
    motion = &state->view->motion;
  else
    motion = &state->volumeData[volNumber]->motion;

  /* The button was just pressed */
  if (track == TRACK_START) {

    motion->xWin[0] = motion->xWin[1] = motion->xWin[2] = 0;
    motion->yWin[0] = motion->yWin[1] = motion->yWin[2] = 0;

  } else {

    if (track == TRACK) {

      /*
       * Use a simple box filter over the last three inputs
       * to remove high frequency jittering when rotating the
       * global patch.
       */
      motion->xWin[0] = motion->xWin[1];
      motion->xWin[1] = motion->xWin[2];
      motion->xWin[2] = motion->xLast - xCur;

      motion->yWin[0] = motion->yWin[1];
      motion->yWin[1] = motion->yWin[2];
      motion->yWin[2] = motion->yLast - yCur;

      if ( SpinVolume( state, volNumber, NULL, NULL ) ) {

	if (volNumber != -1) {
	  ComputeAllPlaneExtents( state, state->planeData );
	}

	RedrawWindow( state );
      }

    } else {			/* track == TRACK_END */

      /*
       * Check to see how long it has been since the user last moved the
       * pointer.  If it has been longer than 3/4ths of a second then
       * don't auto spin.  This way the user must coordinate their drag
       * with the button release to achieve auto spinning.
       */
      long mdiff;
      int autoSpin;

      gettimeofday(&(state->view->t3), NULL);
      mdiff = (state->view->t3.tv_sec  - state->view->t4.tv_sec)  * 1000 + 
	      (state->view->t3.tv_usec - state->view->t4.tv_usec) / 1000;

      autoSpin = ((mdiff < 750) && (sqrt( SQUARE(motion->xWin[2]) +
					  SQUARE(motion->yWin[2]) ) > 1.8));
      if (autoSpin) {
	motion->xWin[2] *= 0.5;
	motion->yWin[2] *= 0.5;
      }

      if (volNumber == -1)
	state->view->autoSpin = autoSpin;
      else {
	state->volumeData[volNumber]->autoSpin = autoSpin;
	ComputeAllPlaneExtents( state, state->planeData );
      }

      RedrawWindow( state );
    }
  }

  motion->xLast = xCur;
  motion->yLast = yCur;
  gettimeofday(&state->view->t4, NULL);
}


void TrackZoomingVolume(VRState *state, int xCur, int yCur, int track, int volNumber)
{
  VRMotion *motion;
  float diff, ds = 0.01;

  motion = &state->view->motion;

  /* The button was just pressed */
  if (track == TRACK_START) {

    motion->yWin[0] = motion->yWin[1] = motion->yWin[2] = 0;

  } else {

    /*
     * Use a simple box filter over the last three inputs
     * to remove high frequency jittering when rotating the
     * global patch.
     */
    motion->yWin[0] = motion->yWin[1];
    motion->yWin[1] = motion->yWin[2];
    motion->yWin[2] = motion->yLast - yCur;

    diff = ( motion->yWin[0]*W1 +
	     motion->yWin[1]*W2 +
	     motion->yWin[2]*W3 ) * ds;

    if (volNumber == -1) {

      state->view->userScale -= diff;

      /*
       * Clamp the scale so that things don't get too big or
       * mirror because scale < 0
       */
      if (state->view->userScale < 0.01) state->view->userScale = 0.01;
      if (state->view->userScale > 100) state->view->userScale = 100;

    } else {

      VRVolumeData *vd = state->volumeData[volNumber];

      vd->uxScl -= diff;
      if (vd->uxScl < 0.1)
	vd->uxScl = 0.1;
      if (vd->uxScl > 100)
	vd->uxScl = 100;

      vd->uyScl -= diff;
      if (vd->uyScl < 0.1)
	vd->uyScl = 0.1;
      if (vd->uyScl > 100)
	vd->uyScl = 100;

      vd->uzScl -= diff;
      if (vd->uzScl < 0.1)
	vd->uzScl = 0.1;
      if (vd->uzScl > 100)
	vd->uzScl = 100;

      ComputeAllPlaneExtents( state, state->planeData );
    }

    RedrawWindow( state );
  }

  motion->xLast = xCur;
  motion->yLast = yCur;
}


void TrackPickingPoints(VRState *state, int xCur, int yCur, int track)
{
  VRPointData *pd = state->pointData[state->view->curSurf];
  VRSurfaceData *sd = state->surfaceData[state->view->curSurf];
  coord p;
  int plane;

  /* The button was just pressed */
  if (track == TRACK_START) {

    pd->pointMoving = TRUE;

    /* See if the user is selecting a point that already exists */
    pd->curPoint = PointMatches(state, pd, xCur, yCur);

    /* If an old point was selected, move it to the end of the */
    /* point list, just in case it is deleted */
    if (pd->curPoint != -1) {
      coord_assign( pd->point[pd->curPoint], p );
      coord_assign( pd->point[pd->nPoints-1], pd->point[pd->curPoint] );
      coord_assign( p, pd->point[pd->nPoints-1] );
      pd->curPoint = pd->nPoints-1;
    }

  } else if (track == TRACK_END) {

    pd->pointMoving = FALSE;
  }

  /* Find the new point's location and add it to the point list */
  plane = FindPoint( state, xCur, yCur, &p );

  if (plane >= 0) {

    if (pd->curPoint == -1) {
      pd->curPoint = pd->nPoints;
      if ( pd->nPoints < (pd->maxPoints-1) )
	pd->nPoints++;
    }
    coord_assign( p, pd->point[pd->curPoint] );

  } else {

    if (pd->curPoint != -1) {
      pd->nPoints--;
      pd->curPoint = -1;
    }
  }

  if (state->mode->iaSurfMode)
    CreateSurfaceFromPoints(state, pd, sd);

  state->view->curPlane = plane;
  pd->stateChanged = TRUE;
  RedrawWindows();
}


void TrackPickingPlanes(VRState *state, int xCur, int yCur, int track)
{
  coord p;
  int plane;

  /* Find the new point's location and add it to the point list */
  plane = FindPoint( state, xCur, yCur, &p );

  if (plane >= 0)
    state->view->curPlane = plane;
  else
    state->view->curPlane = -1;

  RedrawWindow( state );
}


void TrackMovingSurface(VRState *state, int xCur, int yCur, int point, int track)
{
  static int yLast;
  static coord pointLoc;
  static coord surfNormal;
  float diff;
  coord tmp;
  VRPointData *pd = state->pointData[state->view->curSurf];
  VRSurfaceData *sd = state->surfaceData[state->view->curSurf];

  if (track == TRACK_START) {

    pd->curPoint = point;
    pd->pointMoving = TRUE;
    sd->surfaceMoving = TRUE;

    coord_assign(pd->point[pd->curPoint], pointLoc);
    coord_assign(sd->normal, surfNormal);
    yLast = yCur;

    RedrawWindow(state);
    return;
  }

  diff = (yLast - yCur) / 500.0;

  coord_mult(surfNormal, diff, tmp);
  coord_add(pointLoc, tmp, pd->point[pd->curPoint]);

  if (state->mode->iaSurfMode)
    CreateSurfaceFromPoints(state, pd, sd);

  if (track == TRACK_END) {

    pd->pointMoving = FALSE;
    sd->surfaceMoving = FALSE;

    pd->stateChanged = TRUE;
    sd->stateChanged = TRUE;

    RedrawWindows();
    return;
  }

  pd->stateChanged = TRUE;
  sd->stateChanged = TRUE;
  RedrawWindow(state);
}


void TrackMovingXYVolume(VRState *state, int xCur, int yCur, int track)
{
  static int xLast, yLast;
  float xdiff, ydiff;
  VRVolumeData *vd = state->volumeData[state->view->curVol];

  if (track == TRACK_START) {

    xLast = xCur;
    yLast = yCur;
    vd->volumeMoving = TRUE;
    RedrawWindow( state );

    return;
  }

  xdiff = (xLast - xCur) / 500.0;
  ydiff = (yLast - yCur) / 500.0;

  MoveVolume(state, vd, xdiff, ydiff, 0.0);

  xLast = xCur;
  yLast = yCur;

  if (track == TRACK_END)
    vd->volumeMoving = FALSE;

  vd->stateChanged = TRUE;

  ComputeAllPlaneExtents( state, state->planeData );
  RedrawWindows();
}


void TrackMovingSlicePlane(VRState *state, int xCur, int yCur, int track)
{
  VRMotion *motion;
  matrix mat, rotMat;

  motion = &state->view->motion;

  if (!state->mode->sliceMode)
    return;

  /* The button was just pressed */
  if (track == TRACK_START) {

    motion->xWin[0] = motion->xWin[1] = motion->xWin[2] = 0;
    motion->yWin[0] = motion->yWin[1] = motion->yWin[2] = 0;
    state->planeData->planeMoving = TRUE;

  } else if (track == TRACK_END) {

    state->planeData->planeMoving = FALSE;

  } else {

    /*
     * Use a simple box filter over the last three inputs
     * to remove high frequency jittering when rotating
     */
    motion->xWin[0] = motion->xWin[1];
    motion->xWin[1] = motion->xWin[2];
    motion->xWin[2] = motion->xLast - xCur;

    motion->yWin[0] = motion->yWin[1];
    motion->yWin[1] = motion->yWin[2];
    motion->yWin[2] = motion->yLast - yCur;

    matrix_copy( IdentityMatrix, 4, mat );

    if ( ComputeVsphereMatrix( state, motion->xLast, motion->yLast,
			       motion->xWin[0]*W1 + motion->xWin[1]*W2 + motion->xWin[2]*W3,
			       motion->yWin[0]*W1 + motion->yWin[1]*W2 + motion->yWin[2]*W3,
			       rotMat, TRUE ) ) {
      matrix imat;
      coord p;
      Plane *plane = &(state->planeData->plane[state->view->curPlane]);

      matrix_mult_safe(rotMat, mat, mat);
      matrix_transpose(mat, 4, imat);
      p.x = plane->a;
      p.y = plane->b;
      p.z = plane->c;
      coord_transform( p, imat, p );
      coord_normalize( p, p );
      plane->a = p.x;
      plane->b = p.y;
      plane->c = p.z;

      ComputePlaneExtents( state, plane );
    }
  }

  state->planeData->stateChanged = TRUE;
  RedrawWindows();

  motion->xLast = xCur;
  motion->yLast = yCur;
}


void TrackDrivingSlicePlane(VRState *state, int xCur, int yCur, int track)
{
  static int yLast;
  float diff;

  Plane *p;

  if (track == TRACK_START) {

    yLast = yCur;
    state->planeData->planeMoving = TRUE;
    RedrawWindow( state );

    return;
  }

  p = &(state->planeData->plane[state->view->curPlane]);

  diff = (yLast - yCur) / 500.0 * (p->maxD - p->minD);
  p->d += diff;

  if (p->d > p->maxD)
    p->d = p->maxD;
  else if (p->d < p->minD)
    p->d = p->minD;

  if (state->view->debug == 2)
    printf("Plane %d d - %5.2f\n", state->view->curPlane, p->d);

  yLast = yCur;

  if (track == TRACK_END)
    state->planeData->planeMoving = FALSE;

  state->planeData->stateChanged = TRUE;
  RedrawWindows();
}


void TrackMovingZVolume(VRState *state, int xCur, int yCur, int track)
{
  static int yLast;
  float diff;
  VRVolumeData *vd = state->volumeData[state->view->curVol];

  if (track == TRACK_START) {

    yLast = yCur;
    vd->volumeMoving = TRUE;
    RedrawWindow( state );

    return;
  }

  diff = (yLast - yCur) / 400.0;

  MoveVolume(state, vd, 0.0, 0.0, diff);

  yLast = yCur;

  if (track == TRACK_END)
    vd->volumeMoving = FALSE;

  vd->stateChanged = TRUE;

  ComputeAllPlaneExtents( state, state->planeData );
  RedrawWindows();
}


void TrackMovingPhantomEye(VRState *state, int xCur, int yCur, int track)
{
  VRMotion *motion;
  int redraw;

  motion = &state->view->phantomMotion;

  /* The button was just pressed */
  if (track == TRACK_START) {

    motion->xWin[0] = motion->xWin[1] = motion->xWin[2] = 0;
    motion->yWin[0] = motion->yWin[1] = motion->yWin[2] = 0;

  } else {

    if (track == TRACK) {

      /*
       * Use a simple box filter over the last three inputs
       * to remove high frequency jittering when rotating the
       * global patch.
       */
      motion->xWin[0] = motion->xWin[1];
      motion->xWin[1] = motion->xWin[2];
      motion->xWin[2] = motion->xLast - xCur;

      motion->yWin[0] = motion->yWin[1];
      motion->yWin[1] = motion->yWin[2];
      motion->yWin[2] = motion->yLast - yCur;

      redraw = SpinVolume(state, -1, motion, &(state->view->InvPhanRotMat));
      matrix_transpose(state->view->InvPhanRotMat, 4, state->view->PhanRotMat);
      state->view->stateChanged = TRUE;

      if (redraw)
        RedrawWindow( state );

    }
  }

  motion->xLast = xCur;
  motion->yLast = yCur;
}


void TrackResettingPhantomEye(VRState *state, int xCur, int yCur, int track)
{
  if (track == TRACK_END) {

    matrix_copy(IdentityMatrix, 4, state->view->InvPhanRotMat);
    matrix_transpose(state->view->InvPhanRotMat, 4, state->view->PhanRotMat);
    state->view->stateChanged = TRUE;

    RedrawWindow( state );
  }
}
