/*
Copyright 1994 Silicon Graphics, Inc. -- All Rights Reserved

If the Software is acquired by or on behalf of an entity of government
of  the  United States of America, the following provision applies: U.
S.  GOVERNMENT  RESTRICTED  RIGHTS  LEGEND:    Use,   duplication   or
disclosure of Software by the Government is subject to restrictions as
set forth in FAR 52.227-19(c)(2) or  subparagraph  (c)(1)(ii)  of  the
Rights  in  Technical  Data  and  Computer  Software  clause  at DFARS
252.227-7013 and/or in similar or successor clauses in the FAR, or the
DOD  or  NASA  FAR Supplement. Unpub-lished- rights reserved under the
Copyright  Laws  of  the  United  States.  Contractor/manufacturer  is
SILICON  GRAPHICS,  INC.,  2011  N. Shoreline Blvd., Mountain View, CA
94039- 7311.

Silicon Graphics, Inc. hereby grants  to  you  a  non-exclusive,  non-
transferable,  personal, paid-up license to use, modify and distribute
the Software solely with SGI computer products.  You must include,  in
all  copies  of  the  Software  and  any associated documentation, the
copyright notice and restricted rights legend set forth above.

THE SOFTWARE IS PROVIDED  TO  YOU  "AS-IS"  AND  WITHOUT  ANY  SUPPORT
OBLIGATION  OR  WARRANTY  OF  ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
INCLUDING WITHOUT  LIMITATION,  ANY  WARRANTY  OF  MERCHANTABILITY  OR
FITNESS  FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL SGI BE LIABLE FOR
SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF  LIABILITY,
ARISING  OUT  OF  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

You agree that you will not export or re-export the Software, directly
or  indirectly,  unless  (a)  the  Export  Administration of the U. S.
Department of Commerce explicitly permits the export or  re-export  of
the  Software  or  (b)  the  Office  of  Export Licensing of the U. S.
Department of Commerce has granted au-thorization to  you  in  writing
for the  export or re- export the Software.

If you fail to fulfill any  of  the  foregoing  obligations,  SGI  may
pursue  all  available  legal  remedies  to  enforce  these  terms and
conditions, and SGI may,  at  any  time  after  your  default  hereof,
terminate  the  license  and  rights  granted  to  you hereunder.  You
further agree that, if SGI terminates this license for  your  default,
you  will, within ten (10) days after any such termination, deliver to
SGI or  render  unusable  all  Software  originally  provided  to  you
hereunder and any copies thereof embodied in any medium.
*/



#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "misMath.h"
#include "misTransferFunc.h"
#include "volInit.h"
#include "volGeomUtils.h"


static int GetGraphicsTextureMemory(void)
{
  char  file[32], command[64], *ptr;
  int   fd, textureMem, status;
  void *FileData;

  sprintf( file, "/tmp/VR%x", getpid() );
  sprintf( command, "/usr/gfx/gfxinfo > %s", file );

  status = system( command );
  if (status == -1) {
    fprintf(stderr, "System call to gfxinfo failed!\n");
    fprintf( stderr, "volren: %s\n", strerror(oserror()) );
    exit( EXIT_FAILURE );
  }

  fd = open( file, O_RDONLY );
  if (fd == -1) {
    fprintf(stderr, "Unable to open file '%s' for reading!\n", file);
    fprintf( stderr, "volren: %s\n", strerror(oserror()) );
    exit( EXIT_FAILURE );
  }

  FileData = mmap( 0, 256, PROT_WRITE, MAP_PRIVATE, fd, (off_t)0 );
  if ( FileData == (void*) -1 ) {
    fprintf( stderr, "Could not use gfxinfo to find graphics configuration\n" );
    fprintf( stderr, "volren: %s\n", strerror(oserror()) );
    exit( EXIT_FAILURE );
  }

  /* Ensure NULL termination */
  ptr = (char*) FileData;
  ptr[255] = 0;

  /* Read texture amount the IMPACT way! */
  ptr = strstr((char*) FileData, "is \"IMPACT\" graphics.");
  if (ptr) {

    int nTRAMs;

#ifndef VR_IMPACT
    fprintf(stderr, "Uh-oh!\nThis version of volren was compiled for %s.\n"
	    "This machine appears to be an Impact.\nFailure seems imminent!\n",
	    VR_NAME_MACHINE);
#endif

    ptr = strstr( (char *) FileData, "TRAM" );

    if ( ptr == 0 ) {
      fprintf(stderr, "Uh-oh!\nMachine: Impact(?)\nTexture memory: Unknown\n"
              "Guessing 1Mb...\n");
      nTRAMs = 1;
    } else
      nTRAMs = atoi( ptr - 2 );

    return( nTRAMs * 1024*1024 );
  }

  /* Read texture amount the VENICE way! */
  ptr = strstr((char*) FileData, "is \"REV\" graphics.");
  if (ptr) {

    int boardId;

#ifndef VR_VENICE
    fprintf(stderr, "Uh-oh!\nThis version of volren was compiled for %s.\n"
	    "This machine appears to be a Reality Engine.\nFailure seems imminent!\n",
	    VR_NAME_MACHINE);
#endif

    ptr = strstr( (char *) FileData, "RM" );

    if ( ptr == 0 ) {
      fprintf(stderr, "Uh-oh!\nMachine: Reality Engine(?)\nTexture memory: Unknown\n"
              "Guessing 4Mb...\n");
      boardId = 4;
    } else 
      boardId = atoi( ptr + 2 );

    switch (boardId) {
    case 4:
      textureMem = 4*1024*1024;
      break;
    case 5:
      textureMem = 16*1024*1024;
      break;
    default: 
      textureMem = 4*1024*1024;
      break;
    }

    return( textureMem );
  }

  /* Read texture amount the KONA way! */
  ptr = strstr((char*) FileData, "is \"KONA\" graphics.");
  if (ptr) {

    int i;

#ifndef VR_KONA
    fprintf(stderr, "Uh-oh!\nThis version of volren was compiled for %s.\n"
	    "This machine appears to be an Infinite Reality.\nFailure seems imminent!\n",
	    VR_NAME_MACHINE);
#endif

    ptr = strstr( (char *) FileData, "Texture Memory:" );

    if ( ptr == 0 ) {
      fprintf(stderr, "Uh-oh!\nMachine: Infinite Reality(?)\nTexture memory: Unknown\n"
              "Guessing 16Mb...\n");
      return( 16 * 1024*1024 );
    }

    ptr += 16;
    textureMem = 0;

    for (i=0; i<4; i++) {
      textureMem += atoi(ptr);
      if (i < 3) {
	ptr = strstr( ptr, "/" );
	ptr++;
      }
    }

    textureMem = textureMem * 1024*1024;

    return( textureMem );
  }

  fprintf(stderr, "Uh-oh!\nMachine: Unknown(!)\nTexture memory: Unknown\n"
	  "Guessing 1Mb...\n");
  fprintf(stderr, "Uh-oh!\nThis version of volren was compiled for %s.\n"
	  "This machine appears to contain another graphics type.\nFailure seems imminent!\n",
	  VR_NAME_MACHINE);

  return( 1024*1024 );
}


static void ResetPlane( Plane *p, float a, float b, float c, float d,
		        int active, int loopDir )
{
  coord n;

  n.x = a;
  n.y = b;
  n.z = c;
  coord_normalize( n, n );

  p->a = n.x;
  p->b = n.y;
  p->c = n.z;
  p->d = d;
  p->active = active;
  p->loopDir = loopDir;
}


VRWorld* MakeVRWorld( VRWorld *oworld )
{
  VRWorld *world;

  world = (VRWorld*) malloc(sizeof(VRWorld));
  if (world == NULL) {
    fprintf(stderr, "Unable to allocate memory for internal structures!\n");
    exit( EXIT_FAILURE );
  }

  if (oworld)
    memcpy(world, oworld, sizeof(VRWorld));
  else {
    world->stereoMode = FALSE;
    ResetVRWorld( world );
  }

  world->refCount = 1;
  world->stateChanged = TRUE;

  return( world );
}


VRState* MakeVRState( VRState *ostate, VRWorld *world )
{
  int i;
  VRState *state;

  state = (VRState*) malloc(sizeof(VRState));
  if (state == NULL) {
    fprintf(stderr, "Unable to allocate memory for internal structures!\n");
    exit( EXIT_FAILURE );
  }

  /* If a state is passed in, copy it, otherwise fill in nulls */
  if (ostate) {

    memcpy(state, ostate, sizeof(VRState));

    state->world->refCount++;
    state->view->refCount++;
    state->mode->refCount++;
    if ( state->polygonData ) state->polygonData->refCount++;
    state->tableData->refCount++;
    state->planeData->refCount++;

    for (i=0; i<MAX_SURFS; i++) {
      state->pointData[i]->refCount++;
      state->surfaceData[i]->refCount++;
    }
    
    for (i=0; i<ostate->world->nVols; i++)
       state->volumeData[i]->refCount++;

    for (i=0; i<ostate->world->nElevs; i++)
      state->elevData[i]->refCount++;

  } else {

    /* First set up the state->world! */
    state->world       = world;

    /* Next set up the state->volumeData, other structures need it to reset! */
    for (i=0; i<state->world->nVols; i++)
       state->volumeData[i] = MakeVRVolumeData( state, NULL, i );

    /* Next set up the state->elevData, other structures need it to reset! */
    for (i=0; i<state->world->nElevs; i++)
       state->elevData[i] = MakeVRElevData( state, NULL, i );

    /* Be sure to setup state->view before state->mode! */
    state->view        = MakeVRView( state, NULL );

    state->mode        = MakeVRMode( state, NULL );
    state->polygonData = NULL;
    state->tableData   = MakeVRTableData( state, NULL );
    state->planeData   = MakeVRPlaneData( state, NULL );

    for (i=0; i<MAX_SURFS; i++) {
      state->pointData[i]   = MakeVRPointData( state, NULL );
      state->surfaceData[i] = MakeVRSurfaceData( state, NULL );
    }
  }

  return( state );
}


VRView* MakeVRView( VRState *state, VRView *oview )
{
  VRView *view;

  view = (VRView*) malloc(sizeof(VRView));
  if (view == NULL) {
    fprintf(stderr, "Unable to allocate memory for internal structures!\n");
    exit( EXIT_FAILURE );
  }

  if (oview)
    memcpy(view, oview, sizeof(VRView));
  else {

    view->noFonts = TRUE;

    view->numFonts = 4;
    view->font = malloc(view->numFonts*sizeof(Font));
    view->fontBase = malloc(view->numFonts*sizeof(int));
    view->fontName = malloc(view->numFonts*sizeof(char*));

    view->width  = state->world->width;
    view->height = state->world->height;

    ResetVRView( state, view );
  }

  view->boundLights   = FALSE;
  view->boundFonts    = FALSE;

  view->refCount = 1;
  view->stateChanged = TRUE;

  return( view );
}


VRMode* MakeVRMode( VRState *state, VRMode *omode )
{
  VRMode *mode;

  mode = (VRMode*) malloc(sizeof(VRMode));
  if (mode == NULL) {
    fprintf(stderr, "Unable to allocate memory for internal structures!\n");
    exit( EXIT_FAILURE );
  }

  if (omode)
    memcpy(mode, omode, sizeof(VRMode));
  else
    ResetVRMode( state, mode );

  mode->refCount = 1;
  mode->stateChanged = TRUE;

  return( mode );
}


VRPolygonData* MakeVRPolygonData( VRState *state, VRPolygonData *opolygonData )
{
  VRPolygonData *polygonData;

  polygonData = (VRPolygonData*) malloc(sizeof(VRPolygonData));
  if (polygonData == NULL) {
    fprintf(stderr, "Unable to allocate memory for internal structures!\n");
    exit( EXIT_FAILURE );
  }

  if (opolygonData)
    memcpy(polygonData, opolygonData, sizeof(VRPolygonData));
  else
    ResetVRPolygonData( state, polygonData );

  polygonData->refCount = 1;
  polygonData->stateChanged = TRUE;

  return( polygonData );
}


VRTableData* MakeVRTableData( VRState *state, VRTableData *otableData )
{
  VRTableData *tableData;

  tableData = (VRTableData*) malloc(sizeof(VRTableData));
  if (tableData == NULL) {
    fprintf(stderr, "Unable to allocate memory for internal structures!\n");
    exit( EXIT_FAILURE );
  }

  if (otableData) {

    int i;

    memcpy(tableData, otableData, sizeof(VRTableData));

    /* Deep copy the colormap data */
    for (i=0; i<otableData->nColormaps; i++)
      memcpy(tableData->cmap[i], otableData->cmap[i],
	     256 * 3 * sizeof(float));

    /* Deep copy the alphamap data */
    for (i=0; i<otableData->nColormaps; i++)
      memcpy(tableData->amap[i], otableData->amap[i],
	     256 * 1 * sizeof(float));

  } else
    ResetVRTableData( state, tableData );

  tableData->refCount = 1;
  tableData->stateChanged = TRUE;

  return( tableData );
}


VRPlaneData* MakeVRPlaneData( VRState *state, VRPlaneData *oplaneData )
{
  VRPlaneData *planeData;

  planeData = (VRPlaneData*) malloc(sizeof(VRPlaneData));
  if (planeData == NULL) {
    fprintf(stderr, "Unable to allocate memory for internal structures!\n");
    exit( EXIT_FAILURE );
  }

  if (oplaneData)
    memcpy(planeData, oplaneData, sizeof(VRPlaneData));
  else
    ResetVRPlaneData( state, planeData );

  planeData->refCount = 1;
  planeData->stateChanged = TRUE;

  return( planeData );
}


VRPointData* MakeVRPointData( VRState *state, VRPointData *opointData )
{
  VRPointData *pointData;

  pointData = (VRPointData*) malloc(sizeof(VRPointData));
  if (pointData == NULL) {
    fprintf(stderr, "Unable to allocate memory for internal structures!\n");
    exit( EXIT_FAILURE );
  }

  if (opointData)
    memcpy(pointData, opointData, sizeof(VRPointData));
  else {

    pointData->maxPoints = MAX_POINTS;

    ResetVRPointData( state, pointData );
  }

  pointData->refCount = 1;
  pointData->stateChanged = TRUE;

  return( pointData );
}


VRSurfaceData* MakeVRSurfaceData( VRState *state, VRSurfaceData *osurfaceData )
{
  VRSurfaceData *surfaceData;

  surfaceData = (VRSurfaceData*) malloc(sizeof(VRSurfaceData));
  if (surfaceData == NULL) {
    fprintf(stderr, "Unable to allocate memory for internal structures!\n");
    exit( EXIT_FAILURE );
  }

  if (osurfaceData)
    memcpy(surfaceData, osurfaceData, sizeof(VRSurfaceData));
  else {

    surfaceData->polyWidth  = 50;
    surfaceData->polyHeight = 50;

    ResetVRSurfaceData( state, surfaceData );
  }

  surfaceData->polys = (coord*) malloc(surfaceData->polyWidth *
				       surfaceData->polyHeight *
				       sizeof(coord));
  if (surfaceData->polys == NULL) {
    (void) fprintf( stderr, "Unable to allocate space for surface points!\n" );
    exit( EXIT_FAILURE );
  }

  surfaceData->normals = (coord*) malloc(surfaceData->polyWidth *
					 surfaceData->polyHeight *
					 sizeof(coord));
  if (surfaceData->normals == NULL) {
    (void) fprintf( stderr, "Unable to allocate space for surface niormals!\n" );
    exit( EXIT_FAILURE );
  }

  /* Deep copy the polys and normals structures */
  if (osurfaceData) {
    memcpy(surfaceData->polys, osurfaceData->polys,
	   surfaceData->polyWidth * surfaceData->polyHeight * sizeof(coord));
    memcpy(surfaceData->normals, osurfaceData->normals,
	   surfaceData->polyWidth * surfaceData->polyHeight * sizeof(coord));
  }

  surfaceData->refCount = 1;
  surfaceData->stateChanged = TRUE;

  return( surfaceData );
}


VRVolumeData* MakeVRVolumeData( VRState *state, VRVolumeData *ovolumeData, int volNumber )
{
  VRVolumeData *volumeData;

  volumeData = (VRVolumeData*) malloc(sizeof(VRVolumeData));
  if (volumeData == NULL) {
    fprintf(stderr, "Unable to allocate memory for internal structures!\n");
    exit( EXIT_FAILURE );
  }

  /* This code does not do the right thing if you copy another volumeData
     structure.  It should deep copy the brick structures and it does not.
     Right now, no volumeData copies should occur so this code need not be
     written. */

  if (ovolumeData)
    memcpy(volumeData, ovolumeData, sizeof(VRVolumeData));
  else {

    volumeData->xRes = volumeData->yRes = volumeData->zRes = 0;

    volumeData->brick[BRICK8BITS] = NULL;
    volumeData->brick[BRICK4BITS] = NULL;

    volumeData->nxBricks = volumeData->nyBricks = volumeData->nzBricks = 0;
    volumeData->nBricks = 0;

    volumeData->active = FALSE;

    /*
     * This will be reset when the state is reset
     *
    ResetVRVolumeData( state, volumeData, volNumber );
     */
  }

  volumeData->refCount = 1;
  volumeData->stateChanged = TRUE;
  volumeData->histValid = FALSE;

  return( volumeData );
}


VRElevData* MakeVRElevData( VRState *state, VRElevData *oelevData, int elevNumber )
{
  VRElevData *elevData;

  elevData = (VRElevData*) malloc(sizeof(VRElevData));
  if (elevData == NULL) {
    fprintf(stderr, "Unable to allocate memory for internal structures!\n");
    exit( EXIT_FAILURE );
  }

  if (oelevData)
    memcpy(elevData, oelevData, sizeof(VRElevData));
  else {

    elevData->polyWidth  = 50;
    elevData->polyHeight = 50;

    /*
     * This will be reset when the state is reset
     *
    ResetVRElevData( state, elevData, volNumber );
     */
  }

  elevData->polys = (coord*) malloc(elevData->polyWidth *
				       elevData->polyHeight *
				       sizeof(coord));
  if (elevData->polys == NULL) {
    (void) fprintf( stderr, "Unable to allocate space for elevation points!\n" );
    exit( EXIT_FAILURE );
  }

  elevData->normals = (coord*) malloc(elevData->polyWidth *
					 elevData->polyHeight *
					 sizeof(coord));
  if (elevData->normals == NULL) {
    (void) fprintf( stderr, "Unable to allocate space for elevation normals!\n" );
    exit( EXIT_FAILURE );
  }

  /* Deep copy the polys and normals structures */
  if (oelevData) {
    memcpy(elevData->polys, oelevData->polys,
	   elevData->polyWidth * elevData->polyHeight * sizeof(coord));
    memcpy(elevData->normals, oelevData->normals,
	   elevData->polyWidth * elevData->polyHeight * sizeof(coord));
  }

  elevData->refCount     = 1;
  elevData->stateChanged = TRUE;
  elevData->haveTexture  = FALSE;

  return( elevData );
}


void ResetVRWorld( VRWorld *world )
{
  int volNumber, elevNumber, i;

  world->stateChanged = TRUE;

  world->textureMem         = GetGraphicsTextureMemory();
  world->usableTexMem       = world->textureMem;

  world->microfps           = 0.0;
  world->macrofps           = 0.0;
  world->maxfps             = 60.0;

  memset(&world->lastUpdate, 0, sizeof(struct timeval));
  memset(&world->lastFrame, 0, sizeof(struct timeval));

  world->msg[0]             = NULL;

  world->timeVariantVolumes = FALSE;

  world->width              = 512;
  world->height             = 512;
  world->verbose            = FALSE;
  world->debug              = 0;
  world->slabs              = 0;
  world->downSample         = TRUE;
  world->resampleFilter     = BoxFilter;
  world->define4BitBricks   = FALSE;
  world->doTexturedSurfs    = FALSE;
  world->allowStereoMode    = FALSE;
  world->slices             = 175;
  world->sliceMin           = 3;
  world->sliceMax           = 5000;
  world->modCmap            = TRUE;
  world->modAmap            = TRUE;
  world->nPlanes            = MAX_PLANES;
  world->nVols              = 0;
  world->nElevs             = 0;
  world->multiResMode       = FALSE;
  world->magicExtensions    = FALSE;
  for (i=0; i<MAX_RMODES; i++)
    world->allowRenderMode[i] = TRUE;

  for (volNumber=0; volNumber<MAX_VOLS; volNumber++) {
    world->handedness[volNumber]    = LEFT;
    world->sOff[volNumber]          = 0;
    world->xOff[volNumber]          = 0;
    world->yOff[volNumber]          = 0;
    world->teRes[volNumber]         = 1;
    world->timeInterval[volNumber] = 3000;
    world->brickResGiven[volNumber] = FALSE;
    world->xScl[volNumber]          = 1.0;
    world->yScl[volNumber]          = 1.0;
    world->zScl[volNumber]          = 1.0;
    world->xTrn[volNumber]          = 0;
    world->yTrn[volNumber]          = 0;
    world->zTrn[volNumber]          = 0;
  }

  for (elevNumber=0; elevNumber<MAX_ELEVS; elevNumber++) {
     world->elevXScl[elevNumber]  = 1.0;
     world->elevYScl[elevNumber]  = 1.0;
     world->elevZScl[elevNumber]  = 1.0;
     world->elevLight[elevNumber] = TRUE;
     world->elevRot[elevNumber]   = 0.0;
     world->elevXTrn[elevNumber]  = 0.0;
     world->elevYTrn[elevNumber]  = 0.0;
     world->elevZTrn[elevNumber]  = 0.0;
     world->elevColorImg[elevNumber] = NULL;
     world->elevElevImg[elevNumber]  = NULL;
  }

  world->surfaceNormal      = NORMAL_IS_Z;	/* defaults to z */
  world->drawAxes           = TRUE;

  world->backColor          = OPAQUE_BLACK;
  world->frameColor         = OPAQUE_DARK_GREY;
  world->curFrameColor      = OPAQUE_WHITE;
  world->planeColor         = OPAQUE_DARK_GREY;
  world->curPlaneColor      = OPAQUE_WHITE;
  world->pointColor         = OPAQUE_DARK_GREY;
  world->curPointColor      = OPAQUE_WHITE;
  world->xVectorColor       = OPAQUE_RED;
  world->yVectorColor       = OPAQUE_GREEN;
  world->zVectorColor       = OPAQUE_BLUE;
  world->upVectorColor      = OPAQUE_RED;
  world->northVectorColor   = OPAQUE_GREEN;
  world->frontPolyColor     = OPAQUE_YELLOW;
  world->backPolyColor      = OPAQUE_DARK_YELLOW;
  world->frontElevColor     = OPAQUE_PURPLE;
  world->backElevColor      = OPAQUE_DARK_PURPLE;

  world->xySurfColor        = OPAQUE_YELLOW;
  world->zSurfColor         = OPAQUE_PURPLE;

  world->backColorString          = NULL;
  world->frameColorString         = NULL;
  world->curFrameColorString      = NULL;
  world->planeColorString         = NULL;
  world->curPlaneColorString      = NULL;
  world->pointColorString         = NULL;
  world->curPointColorString      = NULL;
  world->xVectorColorString       = NULL;
  world->yVectorColorString       = NULL;
  world->zVectorColorString       = NULL;
  world->upVectorColorString      = NULL;
  world->northVectorColorString   = NULL;
  world->frontPolyColorString     = NULL;
  world->backPolyColorString      = NULL;
  world->frontElevColorString     = NULL;
  world->backElevColorString      = NULL;

  world->xySurfColorString     = NULL;
  world->zSurfColorString      = NULL;
  for (i=0; i<MAX_SURFS; i++)
    world->enumSurfColorString[i] = NULL;

  world->nMaps              = 0;
  world->nCMaps             = 0;
  world->nAMaps             = 0;
  world->nMonitors          = 0;

  world->byteCompiling = FALSE;
  world->readCompiled  = FALSE;
  world->allowDataReread = FALSE;

  world->doTextures    = TRUE;
  world->polygonFile   = NULL;

  world->snapWidth     = 1000;
  world->snapHeight    = 1000;
  world->snapFile      = "snapfile.tif";
}


void ResetVRState( VRState *state )
{
  int i;
  /* We assume that the state->world is already set up! */

  /* We must reset this first, other resets are based upon these values! */
  for (i=0; i<state->world->nVols; i++)
    ResetVRVolumeData( state, state->volumeData[i], i );

  /* We must reset this first, other resets are based upon these values! */
  for (i=0; i<state->world->nElevs; i++)
    ResetVRElevData( state, state->elevData[i], i );

  /* We must reset state->view next, others depend upon it! */
  ResetVRView( state, state->view );

  ResetVRMode( state, state->mode );
  ResetVRTableData( state, state->tableData );
  ResetVRPlaneData( state, state->planeData );

  for (i=0; i<MAX_SURFS; i++) {
    ResetVRPointData( state, state->pointData[i] );
    ResetVRSurfaceData( state, state->surfaceData[i] );
  }

  state->winList = NULL;
  state->winListCount = 0;
}


void ResetVRView( VRState *state, VRView *view )
{

  view->stateChanged = TRUE;

  view->aspect  = ((float) view->width) / view->height;

  view->fontName[0] = getenv("SCREENFONT0");
  if (!view->fontName[0])
    view->fontName[0] = "-*-helvetica-medium-r-*-8-*";
  view->fontName[1] = getenv("SCREENFONT1");
  if (!view->fontName[1])
    view->fontName[1] = "-*-helvetica-bold-r-*-14-*";
  view->fontName[2] = getenv("SCREENFONT2");
  if (!view->fontName[2])
    view->fontName[2] = "-*-helvetica-bold-r-*-18-*";
  view->fontName[3] = getenv("SCREENFONT3");
  if (!view->fontName[3])
    view->fontName[3] = "-*-helvetica-bold-o-*-24-*";

  view->tlutBase = -1;

  view->verbose = state->world->verbose;
  view->debug   = state->world->debug;

  view->snapping = FALSE;

  view->autoSpin = FALSE;

  view->slices = state->world->slices;

  view->userScale = 0.5;

  view->zTrans = -2.0 * sqrt( SQUARE(state->world->xScl[0]) +
			      SQUARE(state->world->yScl[0]) +
			      SQUARE(state->world->zScl[0]) );

  matrix_copy( IdentityMatrix, 4, view->RotMat );
  matrix_copy( IdentityMatrix, 4, view->InvRotMat );

  matrix_copy( IdentityMatrix, 4, view->PhanRotMat );
  matrix_copy( IdentityMatrix, 4, view->InvPhanRotMat );

  view->curPlane    = 0;
  view->curSurf     = -1;
  view->curVol      = -1;

  view->motion.xWin[0] = view->motion.xWin[1] = view->motion.xWin[2] = 0;
  view->motion.yWin[0] = view->motion.yWin[1] = view->motion.yWin[2] = 0;

  view->deltas[INTERACTIVE]    = view->slices * 0.6;
  view->deltas[NONINTERACTIVE] = view->slices;
  view->delta                  = SLICE_FACTOR / view->slices;
}


void ResetVRMode( VRState *state, VRMode *mode )
{
  mode->stateChanged = TRUE;

  mode->interactive  = FALSE;

  mode->renderMode   = MERGE_RENDER;

  mode->volumeMode   = TRUE;
  mode->elevMode     = TRUE;
  mode->pointsMode   = TRUE;
  mode->surfaceMode  = TRUE;
  mode->sliceMode    = FALSE;
  mode->textMode     = TRUE;

  mode->sliderMode   = 2;

  mode->iaSurfMode   = TRUE;
#ifdef VR_IMPACT
  mode->iaSliderMode = FALSE;
#else
  mode->iaSliderMode = TRUE;
#endif

  mode->mipMode      = FALSE;
  mode->vectorMode   = FALSE;
  mode->followMode   = TRUE;
  mode->frontbufMode = FALSE;
  mode->frameMode    = TRUE;
  mode->stereoEye    = LEFT;
  mode->stereoSep    = 0.01;
  mode->stereoRot    = atan2(state->view->zTrans, mode->stereoSep);
  mode->perspMode    = FALSE;
  mode->perspFov     = 300.0;	/* 30 degrees */
  mode->perspNear    = 0.01;
  mode->perspFar     = -10.0 * state->view->zTrans;
  mode->pixelRes     = TRUE;
  mode->brickRes     = BRICK8BITS;

  mode->loopMode     = FALSE;
  mode->clipGeomMode = FALSE;
}


void ResetVRPolygonData( VRState *state, VRPolygonData *polygonData )
{
  polygonData->stateChanged = TRUE;

  polygonData->nObjects    = 0;
  polygonData->object      = NULL;
}


void ResetVRTableData( VRState *state, VRTableData *tableData )
{
  int f = 0;

  tableData->stateChanged = TRUE;

  tableData->cafunc.title         = "Colormap / Alphamap";
  tableData->cafunc.slider[0]     = "Cmap Rot";
  tableData->cafunc.sensitive[0]  = TRUE;
  tableData->cafunc.slider[1]     = "Cmap Scl";
  tableData->cafunc.sensitive[1]  = TRUE;
  tableData->cafunc.slider[2]     = "Amap Rot";
  tableData->cafunc.sensitive[2]  = TRUE;
  tableData->cafunc.slider[3]     = "Amap Scl";
  tableData->cafunc.sensitive[3]  = TRUE;
  tableData->cafunc.useMap        = TRUE;
  tableData->cafunc.func          = NULL;
  tableData->cafunc.pmin[0]       = 0.0;
  tableData->cafunc.parm[0]       = 0.0;
  tableData->cafunc.pmax[0]       = 1.0;
  tableData->cafunc.pmin[1]       = 0.0;
  tableData->cafunc.parm[1]       = 1.0;
  tableData->cafunc.pmax[1]       = 5.0;
  tableData->cafunc.pmin[2]       = 0.0;
  tableData->cafunc.parm[2]       = 0.0;
  tableData->cafunc.pmax[2]       = 1.0;
  tableData->cafunc.pmin[3]       = 0.0;
  tableData->cafunc.parm[3]       = 1.0;
  tableData->cafunc.pmax[3]       = 5.0;

  /* Medical window level transfer function */
  tableData->func[f].title        = "Window/Level";
  tableData->func[f].slider[0]    = "Window";
  tableData->func[f].sensitive[0] = TRUE;
  tableData->func[f].slider[1]    = "Level";
  tableData->func[f].sensitive[1] = TRUE;
  tableData->func[f].slider[2]    = "Center";
  tableData->func[f].sensitive[2] = TRUE;
  tableData->func[f].slider[3]    = "";
  tableData->func[f].sensitive[3] = FALSE;
  tableData->func[f].useMap       = TRUE;
  tableData->func[f].func         = WinLevFunc;
  tableData->func[f].pmin[0]      = -M_PI/2+0.0001;
  tableData->func[f].parm[0]      = M_PI/4;
  tableData->func[f].pmax[0]      = M_PI/2-0.0001;
  tableData->func[f].pmin[1]      = 0.0;
  tableData->func[f].parm[1]      = 0.5;
  tableData->func[f].pmax[1]      = 1.0;
  tableData->func[f].pmin[2]      = 0.0;
  tableData->func[f].parm[2]      = 0.5;
  tableData->func[f].pmax[2]      = 1.0;
  tableData->func[f].pmin[3]      = 0.0;
  tableData->func[f].parm[3]      = 0.0;
  tableData->func[f].pmax[3]      = 1.0;
  f++;

  /* Triangle transfer function */
  tableData->func[f].title        = "Triangle";
  tableData->func[f].slider[0]    = "Center";
  tableData->func[f].sensitive[0] = TRUE;
  tableData->func[f].slider[1]    = "Width";
  tableData->func[f].sensitive[1] = TRUE;
  tableData->func[f].slider[2]    = "Height";
  tableData->func[f].sensitive[2] = TRUE;
  tableData->func[f].slider[3]    = "";
  tableData->func[f].sensitive[3] = FALSE;
  tableData->func[f].useMap       = TRUE;
  tableData->func[f].func         = TriangFunc;
  tableData->func[f].pmin[0]      = -1.0;
  tableData->func[f].parm[0]      = 0.5;
  tableData->func[f].pmax[0]      = 2.0;
  tableData->func[f].pmin[1]      = 0.0;
  tableData->func[f].parm[1]      = M_PI/4;
  tableData->func[f].pmax[1]      = M_PI/2-0.0001;
  tableData->func[f].pmin[2]      = 0.0;
  tableData->func[f].parm[2]      = 1.0;
  tableData->func[f].pmax[2]      = 8.0;
  tableData->func[f].pmin[3]      = 0.0;
  tableData->func[f].parm[3]      = 0.0;
  tableData->func[f].pmax[3]      = 1.0;
  f++;

  /* Inverse triangle transfer function */
  tableData->func[f].title        = "Inverse Triangle";
  tableData->func[f].slider[0]    = "Center";
  tableData->func[f].sensitive[0] = TRUE;
  tableData->func[f].slider[1]    = "Width";
  tableData->func[f].sensitive[1] = TRUE;
  tableData->func[f].slider[2]    = "Height";
  tableData->func[f].sensitive[2] = TRUE;
  tableData->func[f].slider[3]    = "";
  tableData->func[f].sensitive[3] = FALSE;
  tableData->func[f].useMap       = TRUE;
  tableData->func[f].func         = InvTriFunc;
  tableData->func[f].pmin[0]      = -1.0;
  tableData->func[f].parm[0]      = 0.5;
  tableData->func[f].pmax[0]      = 2.0;
  tableData->func[f].pmin[1]      = 0.0;
  tableData->func[f].parm[1]      = M_PI/4;
  tableData->func[f].pmax[1]      = M_PI/2-0.0001;
  tableData->func[f].pmin[2]      = 0.0;
  tableData->func[f].parm[2]      = 1.0;
  tableData->func[f].pmax[2]      = 8.0;
  tableData->func[f].pmin[3]      = 0.0;
  tableData->func[f].parm[3]      = 0.0;
  tableData->func[f].pmax[3]      = 1.0;
  f++;

  /* Double triangle transfer function */
  tableData->func[f].title        = "Double Triangle";
  tableData->func[f].slider[0]    = "Center";
  tableData->func[f].sensitive[0] = TRUE;
  tableData->func[f].slider[1]    = "Width";
  tableData->func[f].sensitive[1] = TRUE;
  tableData->func[f].slider[2]    = "Height";
  tableData->func[f].sensitive[2] = TRUE;
  tableData->func[f].slider[3]    = "Dist.";
  tableData->func[f].sensitive[3] = TRUE;
  tableData->func[f].useMap       = TRUE;
  tableData->func[f].func         = DblTriFunc;
  tableData->func[f].pmin[0]      = -1.0;
  tableData->func[f].parm[0]      = 0.5;
  tableData->func[f].pmax[0]      = 2.0;
  tableData->func[f].pmin[1]      = 0.0;
  tableData->func[f].parm[1]      = M_PI/3;
  tableData->func[f].pmax[1]      = M_PI/2-0.0001;
  tableData->func[f].pmin[2]      = 0.0;
  tableData->func[f].parm[2]      = 1.0;
  tableData->func[f].pmax[2]      = 8.0;
  tableData->func[f].pmin[3]      = 0.0;
  tableData->func[f].parm[3]      = 0.2;
  tableData->func[f].pmax[3]      = 1.0;
  f++;

  /* Bob drebin's medical transfer function */
  tableData->func[f].title        = "Medical";
  tableData->func[f].slider[0]    = "Air";
  tableData->func[f].sensitive[0] = TRUE;
  tableData->func[f].slider[1]    = "Fat";
  tableData->func[f].sensitive[1] = TRUE;
  tableData->func[f].slider[2]    = "Tissue";
  tableData->func[f].sensitive[2] = TRUE;
  tableData->func[f].slider[3]    = "Bone";
  tableData->func[f].sensitive[3] = TRUE;
  tableData->func[f].useMap       = FALSE;
  tableData->func[f].func         = DrebinFunc;
  tableData->func[f].pmin[0]      = 0.0;
  tableData->func[f].parm[0]      = 0.03;
  tableData->func[f].pmax[0]      = 1.0;
  tableData->func[f].pmin[1]      = 0.0;
  tableData->func[f].parm[1]      = 0.1;
  tableData->func[f].pmax[1]      = 1.0;
  tableData->func[f].pmin[2]      = 0.0;
  tableData->func[f].parm[2]      = 0.3;
  tableData->func[f].pmax[2]      = 1.0;
  tableData->func[f].pmin[3]      = 0.0;
  tableData->func[f].parm[3]      = 0.8;
  tableData->func[f].pmax[3]      = 1.0;
  f++;

  tableData->nFuncs = f;

  tableData->opaquify = FALSE;
  tableData->tableRedef = TRUE;
}


void ResetVRPlaneData( VRState *state, VRPlaneData *planeData )
{
  planeData->stateChanged = TRUE;

  ResetPlane( &(planeData->plane[0]), 0.0, 0.0, 1.0, 0.0001, 1, 1 );
  ResetPlane( &(planeData->plane[1]), 0.0, 1.0, 0.0, 0.0001, 0, 1 );
  ResetPlane( &(planeData->plane[2]), 1.0, 0.0, 0.0, 0.0001, 0, 1 );
  ResetPlane( &(planeData->plane[3]), 1.0, 0.0, 1.0, 0.0001, 0, 1 );
  ResetPlane( &(planeData->plane[4]), 0.0, 1.0, 1.0, 0.0001, 0, 1 );
  ResetPlane( &(planeData->plane[5]), 1.0, 1.0, 0.0, 0.0001, 0, 1 );

  planeData->nPlanes     = state->world->nPlanes;
  planeData->planeMoving = FALSE;

  ComputeAllPlaneExtents( state, planeData );
}


void ResetVRPointData( VRState *state, VRPointData *pointData )
{
  pointData->stateChanged = TRUE;

  pointData->nPoints     = 0;
  pointData->curPoint    = -1;
  pointData->pointMoving = FALSE;
}


void ResetVRSurfaceData( VRState *state, VRSurfaceData *surfaceData )
{
  surfaceData->stateChanged = TRUE;

  surfaceData->haveSurf      = FALSE;
  surfaceData->drawTextured  = FALSE;
  surfaceData->drawLighted   = TRUE;
  surfaceData->drawMeshed    = FALSE;
  surfaceData->surfaceMoving = FALSE;
  surfaceData->active        = TRUE;
}


void ResetVRVolumeData( VRState *state, VRVolumeData *vd, int volNumber )
{
  vd->stateChanged = TRUE;

  if ( state->world->debug > 0 ) {
    printf( "nxBricks = %d, nyBricks = %d nzBricks = %d\n",
	   vd->nxBricks, vd->nyBricks, vd->nzBricks );
  }

  /* Load in volume position and scaling */
  vd->curTime = 0;
  vd->timerId = 0;
  vd->timeInterval = state->world->timeInterval[volNumber];

  vd->xTrn = state->world->xTrn[volNumber];
  vd->yTrn = state->world->yTrn[volNumber];
  vd->zTrn = state->world->zTrn[volNumber];
  vd->uxScl = 1.0;
  vd->uyScl = 1.0;
  vd->uzScl = 1.0;
  matrix_copy(IdentityMatrix, 4, vd->RotMat);
  matrix_copy(IdentityMatrix, 4, vd->InvRotMat);

  vd->curFunc = 0;
  vd->parmValid = False;
  vd->cmapValid = False;
  vd->amapValid = False;

  vd->curCmap  = 0;
  vd->curAmap  = 0;
  vd->autoSpin = False;

  vd->modCmap  = state->world->modCmap;
  vd->modAmap  = state->world->modAmap;

  vd->drawColor = True;
  vd->drawInterp = True;

  vd->motion.xWin[0] =
  vd->motion.xWin[1] =
  vd->motion.xWin[2] = 0;
  vd->motion.yWin[0] =
  vd->motion.yWin[1] =
  vd->motion.yWin[2] = 0;

  vd->volBrick.texId = 0;
  vd->volBrick.tevId = 0;
  vd->volBrick.xOff = 0;
  vd->volBrick.yOff = 0;
  vd->volBrick.zOff = 0;
      
  vd->volBrick.xRes = vd->xRes;
  vd->volBrick.yRes = vd->yRes;
  vd->volBrick.zRes = vd->zRes;

  vd->volBrick.xScl = 1.0;
  vd->volBrick.yScl = 1.0;
  vd->volBrick.zScl = 1.0;

  vd->volBrick.txScl = 0.0;
  vd->volBrick.tyScl = 0.0;
  vd->volBrick.txOff = 0.0;
  vd->volBrick.tyOff = 0.0;

  vd->volBrick.own   = FALSE;
  vd->volBrick.data  = NULL;
  vd->volBrick.iown  = FALSE;
  vd->volBrick.idata = NULL;

  vd->active = TRUE;
}


void ResetVRElevData( VRState *state, VRElevData *elevData, int elevNumber )
{
  elevData->stateChanged = TRUE;

  elevData->xScl   = state->world->elevXScl[elevNumber];
  elevData->yScl   = state->world->elevYScl[elevNumber];
  elevData->zScl   = state->world->elevZScl[elevNumber];
  elevData->light  = state->world->elevLight[elevNumber];
  elevData->xyRot  = state->world->elevRot[elevNumber];
  elevData->xTrn   = state->world->elevXTrn[elevNumber];
  elevData->yTrn   = state->world->elevYTrn[elevNumber];
  elevData->zTrn   = state->world->elevZTrn[elevNumber];
}


void ReleaseVRState( VRState *state )
{
  int i;

  ReleaseVRWorld( state->world );
  ReleaseVRView( state->view );
  ReleaseVRMode( state->mode );
  ReleaseVRTableData( state->tableData );
  ReleaseVRPlaneData( state->planeData );

  for (i=0; i<MAX_SURFS; i++) {
    ReleaseVRPointData( state->pointData[i] );
    ReleaseVRSurfaceData( state->surfaceData[i] );
  }

  for (i=0; i<state->world->nVols; i++)
    ReleaseVRVolumeData( state->volumeData[i] );

  for (i=0; i<state->world->nElevs; i++)
    ReleaseVRElevData( state->elevData[i] );

  free( state );
}


void ReleaseVRWorld( VRWorld *world )
{
  world->refCount--;

  if (world->refCount <= 0)
    free( world );
}


void ReleaseVRView( VRView *view )
{
  view->refCount--;

  if (view->refCount <= 0) {

    XCloseDisplay(view->disp);
    free( view );
  }
}


void ReleaseVRMode( VRMode *mode )
{
  mode->refCount--;

  if (mode->refCount <= 0)
    free( mode );
}


void ReleaseVRPolygonData( VRPolygonData *polygonData )
{
  int i;

  polygonData->refCount--;

  if (polygonData->refCount <= 0) {

    if (polygonData->object) {

      for (i=0; i<polygonData->nObjects; i++)
	if (polygonData->object[i].point)
	  free( polygonData->object[i].point );

      free( polygonData->object );
    }

    free( polygonData );
  }
}


void ReleaseVRTableData( VRTableData *tableData )
{
  tableData->refCount--;

  if (tableData->refCount <= 0)
    free( tableData );
}


void ReleaseVRPlaneData( VRPlaneData *planeData )
{
  planeData->refCount--;

  if (planeData->refCount <= 0)
    free( planeData );
}


void ReleaseVRPointData( VRPointData *pointData )
{
  pointData->refCount--;

  if (pointData->refCount <= 0)
    free( pointData );
}


void ReleaseVRSurfaceData( VRSurfaceData *surfaceData )
{
  surfaceData->refCount--;

  if (surfaceData->refCount <= 0) {

    free( surfaceData->polys );
    free( surfaceData->normals );
    free( surfaceData );
  }
}


void ReleaseVRVolumeData( VRVolumeData *vd )
{
  vd->refCount--;

  if (vd->refCount <= 0) {

    int i, t;

    for (t=0; t<vd->tRes; t++) {

      for (i=0; i<vd->nBricks; i++) {

	Brick *b;

	if (vd->brick[BRICK8BITS]) {
	  b = vd->brick[BRICK8BITS][t][i];
	  if (b) {
	    if (b->own && b->data)
	      free( b->data );
	    if (b->iown && b->idata)
	      free( b->idata );
	    free( b );
	  }
	}

	if (vd->brick[BRICK4BITS]) {
	  b = vd->brick[BRICK4BITS][t][i];
	  if (b) {
	    if (b->own && b->data)
	      free( b->data );
	    if (b->iown && b->idata)
	      free( b->idata );
	    free( b );
	  }
	}
      }
    }

    free( vd );
  }
}


void ReleaseVRElevData( VRElevData *elevData )
{
  elevData->refCount--;

  if (elevData->refCount <= 0) {

    free( elevData->polys );
    free( elevData->normals );
    free( elevData );
  }
}
