/*
Copyright 1994 Silicon Graphics, Inc. -- All Rights Reserved

If the Software is acquired by or on behalf of an entity of government
of  the  United States of America, the following provision applies: U.
S.  GOVERNMENT  RESTRICTED  RIGHTS  LEGEND:    Use,   duplication   or
disclosure of Software by the Government is subject to restrictions as
set forth in FAR 52.227-19(c)(2) or  subparagraph  (c)(1)(ii)  of  the
Rights  in  Technical  Data  and  Computer  Software  clause  at DFARS
252.227-7013 and/or in similar or successor clauses in the FAR, or the
DOD  or  NASA  FAR Supplement. Unpub-lished- rights reserved under the
Copyright  Laws  of  the  United  States.  Contractor/manufacturer  is
SILICON  GRAPHICS,  INC.,  2011  N. Shoreline Blvd., Mountain View, CA
94039- 7311.

Silicon Graphics, Inc. hereby grants  to  you  a  non-exclusive,  non-
transferable,  personal, paid-up license to use, modify and distribute
the Software solely with SGI computer products.  You must include,  in
all  copies  of  the  Software  and  any associated documentation, the
copyright notice and restricted rights legend set forth above.

THE SOFTWARE IS PROVIDED  TO  YOU  "AS-IS"  AND  WITHOUT  ANY  SUPPORT
OBLIGATION  OR  WARRANTY  OF  ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
INCLUDING WITHOUT  LIMITATION,  ANY  WARRANTY  OF  MERCHANTABILITY  OR
FITNESS  FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL SGI BE LIABLE FOR
SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF  LIABILITY,
ARISING  OUT  OF  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

You agree that you will not export or re-export the Software, directly
or  indirectly,  unless  (a)  the  Export  Administration of the U. S.
Department of Commerce explicitly permits the export or  re-export  of
the  Software  or  (b)  the  Office  of  Export Licensing of the U. S.
Department of Commerce has granted au-thorization to  you  in  writing
for the  export or re- export the Software.

If you fail to fulfill any  of  the  foregoing  obligations,  SGI  may
pursue  all  available  legal  remedies  to  enforce  these  terms and
conditions, and SGI may,  at  any  time  after  your  default  hereof,
terminate  the  license  and  rights  granted  to  you hereunder.  You
further agree that, if SGI terminates this license for  your  default,
you  will, within ten (10) days after any such termination, deliver to
SGI or  render  unusable  all  Software  originally  provided  to  you
hereunder and any copies thereof embodied in any medium.
*/


#include "cppArgs.h"

#include "volGeomUtils.h"
#include "misGenerateSurface.H"
#include "misLeastSquaresPlane.H"
#include "misCube.h"


void ComputeDeltas( VRState *state, VRView *view )
{
   state->view->deltas[INTERACTIVE]    = view->slices * 0.6;
   state->view->deltas[NONINTERACTIVE] = view->slices;
   state->view->delta                  = SLICE_FACTOR / view->slices;
}


void ComputePlaneExtents( VRState *state, Plane *plane )
{
  int   i, j;
  float d, minD, maxD;
  coord n, v;

  /*
   * Compute the range that plane->d can travel
   */
  minD =  HUGE;
  maxD = -HUGE;

  n.x = plane->a;
  n.y = plane->b;
  n.z = plane->c;
   
  for (j=0; j<state->world->nVols; j++)
    if (state->volumeData[j]->active) {
	for (i=0; i<8; i++) {

	    VRVolumeData *vd = state->volumeData[j];

	    coord_transform( PlusMinusCube[i], vd->VTWMat, v );
	    coord_dot_product( n, v, d );
	    d = -d;

	    if (state->view->debug == 2) {
	      printf("Point %d - ", i);
	      printf("(%5.2f %5.2f %5.2f) ",
		     PlusMinusCube[i].x, PlusMinusCube[i].y, PlusMinusCube[i].z );
	      printf("(%5.2f %5.2f %5.2f) ", v.x, v.y, v.z);
	      printf("(%5.2f %5.2f %5.2f) ", n.x, n.y, n.z);
	      printf("%5.2f\n", d);
	    }

	    if (d < minD) minD = d + 0.015;
	    if (d > maxD) maxD = d - 0.015;
	  }
      }

  if (state->view->debug == 2)
    printf("\n\n");

  if (state->view->debug == 1)
    printf( "min = %f max = %f\n", minD, maxD );

  plane->minD = minD;
  plane->maxD = maxD;
}


void ComputeAllPlaneExtents( VRState *state, VRPlaneData *planeData )
{
  int i;

  for (i=0; i<planeData->nPlanes; i++)
    ComputePlaneExtents( state, &(planeData->plane[i]) );
}


/*
 * This procedure computes a rotation matrix from the
 * delta X/Y mouse motion.  It assumes that the rotation
 * is being accumalated in the passed matrix paramater.
 * That is it will concatenate a the delta rotation onto
 * the matrix which is passed in as a parameter.
 */
int ComputeVsphereMatrix( VRState *state, float x, float y, float dx, float dy,
			  matrix m, int relative )
{
  coord      p0, p1, v0, v1;
  coord      sv;
  quaternion q;
  double     mag;
  float      HalfAngle, sin_HalfAngle, cos_HalfAngle;
  long       Xcenter, Ycenter;

  /*
   * Get screen size to normalize x/y cursor motion to unity.
   */
  Xcenter = state->view->width  / 2;
  Ycenter = state->view->height / 2;

  /*
   * Compute the old and new cursor position on
   * the virtual sphere
   */
  p0.x = (x - Xcenter) * 2 / state->view->width;
  p0.y = (y - Ycenter) * 2 / state->view->height;
  p0.z = 1 - SQUARE(p0.x) - SQUARE(p0.y);

  if ( p0.z < 0 )
    p0.z = 0;
  else
    p0.z = sqrt( p0.z );

  p1.x = (x - dx - Xcenter) * 2 / state->view->width;
  p1.y = (y - dy - Ycenter) * 2 / state->view->height;
  p1.z = 1 - SQUARE(p1.x) - SQUARE(p1.y);

  if ( p1.z < 0 )
    p1.z = 0;
  else
    p1.z = sqrt( p1.z );

  /*
   * Let vectors v0 and v1 represent the
   * plane of rotation defined by p0 and p1
   */
  coord_normalize( p0, v0 );
  coord_normalize( p1, v1 );

  /*
   * Let sv be the normal to that plane
   */
  coord_cross_product( v0, v1, sv );
  coord_norm( sv, mag );
  coord_div(  sv, mag, sv );

  if ( relative )
    coord_transform( sv, state->view->RotMat, sv );

  if ( mag != 0 )
  {
    /*
     * Create an unit quaternion whose rotational component is a proportional
     * to the magnitude of the vector
     */
    HalfAngle     = asin(mag);
    sin_HalfAngle = sin( HalfAngle );
    cos_HalfAngle = cos( HalfAngle );

    q.x = sin_HalfAngle * sv.x;
    q.y = sin_HalfAngle * sv.y;
    q.z = sin_HalfAngle * sv.z;
    q.w = cos_HalfAngle;

    /*
     * Now create a rotation matrix out of this quaternion.
     */
    quaternion_matrix( &q, m );

    /*
     * Tell the caller whether something really changed
     */
    return TRUE;

  } else {

    return FALSE;
  }
}


void ComputePerspScale( VRState *state )
{
  matrix m;          /* Projection matrix */
  float  x, y, z, w; /* Point in the Z midplane of the voxel volume */
  float  Xp, Wp;     /* Perspective transformed x, w */
  float  Xo, Wo;     /* Orthographic  transformed x, w */

#ifdef VR_IRISGL
  int    currentMMode;
#endif

#ifdef VR_OPENGL
  GLint  currentMMode;
#endif

#ifdef VR_IRISGL
  /* Save the current matrix mode */
  currentMMode = getmmode();
#endif

#ifdef VR_OPENGL
  /* Save the current matrix mode */
  glGetIntegerv(GL_MATRIX_MODE, &currentMMode);
#endif

  /* Midplane 3-D volume coordinate */
  x = 1; y = 1; z = state->view->zTrans; w = 1;

#ifdef VR_IRISGL
  /* Build a projection matrix */
  mmode( MPROJECTION );
  perspective( (Angle) state->mode->perspFov, 1.0,
	       state->mode->perspNear, state->mode->perspFar );
  getmatrix(m);
#endif

#ifdef VR_OPENGL
  /* Build a projection matrix */
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(0.1 * ((GLfloat) state->mode->perspFov), 1.0,
		 state->mode->perspNear, state->mode->perspFar);
  glGetFloatv(GL_PROJECTION_MATRIX, (GLfloat*) m);
#endif

  /* Get an x perspective screen coordinate */
  Xp = m[0][0]*x + m[1][0]*y + m[2][0]*z + m[3][0]*w;
  Wp = m[0][3]*x + m[1][3]*y + m[2][3]*z + m[3][3]*w;
  Xp /= Wp;

#ifdef VR_IRISGL
  /* Do the same for the orthographic projection */
  ortho(-1.0, 1.0, -1.0, 1.0, -1.0, 1.0);
  getmatrix(m);
#endif

#ifdef VR_OPENGL
  /* Do the same for the orthographic projection */
  glLoadIdentity();
  glOrtho(-1.0, 1.0, -1.0, 1.0, -1.0, 1.0);
  glGetFloatv(GL_PROJECTION_MATRIX, (GLfloat*) m);
#endif

  Xo = m[0][0]*x + m[1][0]*y + m[2][0]*z + m[3][0]*w;
  Wo = m[0][3]*x + m[1][3]*y + m[2][3]*z + m[3][3]*w;
  Xo /= Wo;
     
  /* The scale factor is for perspective is */
  state->mode->perspScale = Xo / Xp;

#ifdef VR_IRISGL
  /* Restore the matrix mode to the caller's mode */
  mmode( currentMMode );
#endif

#ifdef VR_OPENGL
  /* Restore the matrix mode to the caller's mode */
  glMatrixMode((GLenum) currentMMode);
#endif
}


int FindRayPlaneIntersection( coord *rayOrigin, coord *rayDir, Plane *plane,
			      float *t)
{
  float denom = plane->a*rayDir->x + plane->b*rayDir->y + plane->c*rayDir->z;

  if (denom == 0.0)
    return( FALSE );

  *t = -( plane->a*rayOrigin->x + plane->b*rayOrigin->y +
	  plane->c*rayOrigin->z + plane->d ) / denom;

  return( TRUE );
}


int PointInPlusMinusCube( coord *p )
{
   int flag;

   flag = (p->x >= -1.0) &&
          (p->x <=  1.0) &&
	  (p->y >= -1.0) &&
	  (p->y <=  1.0) &&
	  (p->z >= -1.0) &&
	  (p->z <=  1.0);

   return( flag );
}


int PlaneClipsPoint( Plane *pl, coord *p )
{
  return( (pl->a*p->x + pl->b*p->y + pl->c*p->z + pl->d) < 0 );
}


int FindPoint( VRState *state, int CurX, int CurY, coord *p)
{
  float xs, ys, s;
  coord p1, p2, p3;
  float mag, tmin;
  int i, j, k, planeIndex, nohits;

  xs = (CurX - (state->view->width/2.0)) / (state->view->width/2.0);
  ys = (CurY - (state->view->height/2.0)) / (state->view->height/2.0);

  if (state->mode->perspMode)
  {
    s = state->mode->perspNear * tan(state->mode->perspFov / 1800.0 * M_PI / 2.0);

    /* The first point is the origin */
    p1.x = p1.y = p1.z = 0.0;

    /* Compute the second point, the one on the screen plane */
    p2.x = state->view->aspect * s * xs;
    p2.y = s * ys;
    p2.z = -state->mode->perspNear;
  }
  else
  {
    /* Compute the first point */
    p1.x = state->view->aspect * xs;
    p1.y = ys;
    p1.z = 0.0;

    /* The second point is nearly the same */
    p2.x = p1.x;
    p2.y = p1.y;
    p2.z = -state->mode->perspNear;
  }

  /* Rotate the vector */
  coord_transform( p1, state->view->CTWMat, p1 );
  coord_transform( p2, state->view->CTWMat, p2 );

  /* Make point two be the vector between one and two */
  p2.x = p2.x - p1.x;
  p2.y = p2.y - p1.y;
  p2.z = p2.z - p1.z;

  /* Unitize the vector */
  mag = sqrt( p2.x*p2.x + p2.y*p2.y + p2.z*p2.z );
  p2.x /= mag;
  p2.y /= mag;
  p2.z /= mag;

  /* Intersect the vector with things */
  tmin = HUGE;
  planeIndex = -1;
  nohits = TRUE;
  for (i=0; i<state->planeData->nPlanes; i++)
    if (state->planeData->plane[i].active) {

      Plane *pl = &(state->planeData->plane[i]);
      int hit;
      float t;

      /* Intersect the ray and plane */
      hit = FindRayPlaneIntersection(&p1, &p2, pl, &t);

      /* If there was an intersection and it is closer than any previous */
      if (hit && (t < tmin))
      {
	int clipped = 0;
	int inside = 0;

	/* Compute the intersection point */
	p3.x = p1.x + t * p2.x;
	p3.y = p1.y + t * p2.y;
	p3.z = p1.z + t * p2.z;

	/* See if the point is clipped by any of the other clip planes */
	for (j=0; j<state->planeData->nPlanes; j++)
	  if ( state->mode->volumeMode && (i != j) &&
	       state->planeData->plane[j].active &&
	       PlaneClipsPoint(&(state->planeData->plane[j]), &p3) ) {
	    clipped = 1;
	    break;
	  }

	/* See if point is inside any of the volumes */
	if (!clipped)
	  for (k=0; k<state->world->nVols; k++) {

	    coord p;

	    coord_transform( p3, state->volumeData[k]->WTVMat, p );

	    if (PointInPlusMinusCube( &p )) {
	      inside = 1;
	      break;
	    }
	  }

	if (!clipped && inside) {

	  /* Record the new hit */
	  tmin = t;
	  planeIndex = i;
	  nohits = FALSE;
	}
      }
    }

  /* If no clipping planes were hit, return negative one */
  if (nohits)
    return( -1 );

  /* Find the actual intersection point */
  p->x = p1.x + tmin * p2.x;
  p->y = p1.y + tmin * p2.y;
  p->z = p1.z + tmin * p2.z;

  /* Return the index of the plane we hit */
  return( planeIndex );
}


static int PointMatchesHelper( VRState *state, VRPointData *pd, int x, int y, float *zReturn )
{
  int i, bestIndex;
  coord p;
  float w, bestZ;

  bestIndex = -1;
  bestZ = HUGE;

  for (i=0; i<pd->nPoints; i++) {

    if (state->view->debug == 9)
      printf("Testing against point #%d...\n", i);

    coord_transx( pd->point[i], state->view->WTSMat, p );
    coord_transw( pd->point[i], state->view->WTSMat, w );
    p.x /= w;
    p.x = p.x * (state->view->width/2.0) + (state->view->width/2.0);

    /* Test the x coordinate */
    if (fabs(p.x - x) < 7.0) {

      if (state->view->debug == 9)
	printf("    Passed X coord test\n");

      coord_transy( pd->point[i], state->view->WTSMat, p );
      p.y /= w;
      p.y = p.y * (state->view->height/2.0) + (state->view->height/2.0);

      /* Test the y coordinate */
      if (fabs(p.y - y) < 7.0) {

	if (state->view->debug == 9)
	  printf("    Passed Y coord test\n");

	coord_transz( pd->point[i], state->view->WTSMat, p );
	p.z /= w;

	if (p.z < bestZ) {

	  if (state->view->debug == 9)
	    printf("    Passed Z coord test\n");

	  bestIndex = i;
	  bestZ = p.z;
	}
      }
    }
  }

  *zReturn = bestZ;

  if (bestIndex >= 0)
    return( bestIndex );
  else
    return( -1 );
}


int PointMatches( VRState *state, VRPointData *pd, int x, int y )
{
  int i, p, bestS, bestP=-1;
  float z, bestZ=HUGE;

  if (pd != NULL)
    bestP = PointMatchesHelper(state, pd, x, y, &bestZ);
  else {
    for (i=0; i<MAX_SURFS; i++) {

      p = PointMatchesHelper(state, state->pointData[i], x, y, &z);

      if ((p != -1) && (z < bestZ)) {
	bestZ = z;
	bestS = i;
	bestP = p;
      }
    }

    if (bestP != -1) {
      state->view->curSurf = bestS;
      state->pointData[bestS]->curPoint = bestP;
    }
  }

  return( bestP );
}


void VectorRotationAngles(coord *vec, double *yrot, double *xrot)
{
  double theta, phi;
  double cosTheta, sinTheta;
  coord tdir;

  /* Compute the angle of rotation around the Y axis */
  theta = - atan2( vec->x, vec->z );

  /* Rotate the vector around the Y axis */
  cosTheta = cos(theta);
  sinTheta = sin(theta);

  tdir.x =  vec->x*cosTheta + vec->z*sinTheta;
  tdir.y =  vec->y;
  tdir.z = -vec->x*sinTheta + vec->z*cosTheta;

  /* Compute the angle of rotation around the X axis */
  phi = atan2( tdir.y, tdir.z );

  *yrot = theta;
  *xrot = phi;
}


void CreateSurfaceFromPoints(VRState *state, VRPointData *pd, VRSurfaceData *sd)
{
  float *normal, *up;
  int xdir, ydir, zdir;

  if (pd->nPoints < 3)
    return;

  LeastSquaresPlanarFit(pd, &sd->normal);

  normal = (float*) &sd->normal;
  up = (float*) &sd->up;
  xdir = (state->world->surfaceNormal+1) % 3;
  ydir = (state->world->surfaceNormal+2) % 3;
  zdir = (state->world->surfaceNormal+0) % 3;

  up[xdir] = up[ydir] = up[zdir] = 0.0;
  if (fabs(normal[zdir]) >= sqrt(0.5)) {
    sd->surfaceType = XY_SURFACE;
    up[ydir] = 1.0;
  } else {
    sd->surfaceType = Z_SURFACE;
    up[zdir] = 1.0;
  }

  if (state->view->debug == 5) {
    printf("Planar fit ::=  %fx + %fy + %fz = 1\n",
	   sd->normal.x, sd->normal.y, sd->normal.z);
    printf("Plane type is %s and up vector is [%f,%f,%f]\n",
	   (sd->surfaceType==XY_SURFACE)?"XY":"Z",
	   sd->up.x, sd->up.y, sd->up.z);
  }

  GenerateSurface(pd->point, pd->nPoints, &sd->normal, &sd->up,
		  sd->polys, sd->normals, sd->polyWidth, sd->polyHeight);
}
