/*
Copyright 1994 Silicon Graphics, Inc. -- All Rights Reserved

If the Software is acquired by or on behalf of an entity of government
of  the  United States of America, the following provision applies: U.
S.  GOVERNMENT  RESTRICTED  RIGHTS  LEGEND:    Use,   duplication   or
disclosure of Software by the Government is subject to restrictions as
set forth in FAR 52.227-19(c)(2) or  subparagraph  (c)(1)(ii)  of  the
Rights  in  Technical  Data  and  Computer  Software  clause  at DFARS
252.227-7013 and/or in similar or successor clauses in the FAR, or the
DOD  or  NASA  FAR Supplement. Unpub-lished- rights reserved under the
Copyright  Laws  of  the  United  States.  Contractor/manufacturer  is
SILICON  GRAPHICS,  INC.,  2011  N. Shoreline Blvd., Mountain View, CA
94039- 7311.

Silicon Graphics, Inc. hereby grants  to  you  a  non-exclusive,  non-
transferable,  personal, paid-up license to use, modify and distribute
the Software solely with SGI computer products.  You must include,  in
all  copies  of  the  Software  and  any associated documentation, the
copyright notice and restricted rights legend set forth above.

THE SOFTWARE IS PROVIDED  TO  YOU  "AS-IS"  AND  WITHOUT  ANY  SUPPORT
OBLIGATION  OR  WARRANTY  OF  ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
INCLUDING WITHOUT  LIMITATION,  ANY  WARRANTY  OF  MERCHANTABILITY  OR
FITNESS  FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL SGI BE LIABLE FOR
SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF  LIABILITY,
ARISING  OUT  OF  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

You agree that you will not export or re-export the Software, directly
or  indirectly,  unless  (a)  the  Export  Administration of the U. S.
Department of Commerce explicitly permits the export or  re-export  of
the  Software  or  (b)  the  Office  of  Export Licensing of the U. S.
Department of Commerce has granted au-thorization to  you  in  writing
for the  export or re- export the Software.

If you fail to fulfill any  of  the  foregoing  obligations,  SGI  may
pursue  all  available  legal  remedies  to  enforce  these  terms and
conditions, and SGI may,  at  any  time  after  your  default  hereof,
terminate  the  license  and  rights  granted  to  you hereunder.  You
further agree that, if SGI terminates this license for  your  default,
you  will, within ten (10) days after any such termination, deliver to
SGI or  render  unusable  all  Software  originally  provided  to  you
hereunder and any copies thereof embodied in any medium.
*/


#include "cppArgs.h"

#include <stdio.h>
#include <math.h>
#include <malloc.h>

#include "volBrick.h"
#include "volClipPlanes.H"
#include "misCube.h"
#include "misTransferFunc.h"
#include "hciUpdateScreen.h"

#define SLICE_SHIFT_FRACTION 0.01


extern int RenderDirection;


static void BlendFunction(long src, long dst, int dir)
{
#ifdef VR_IRISGL
  if (dir == -1) {
    if (src == BF_SA && dst == BF_MSA)
      blendfunction(BF_MIN_SA_MDA, BF_ONE);
    else if (src == BF_MAX && dst == BF_ZERO)
      blendfunction(BF_ZERO, BF_MAX);
  } else
    blendfunction(src, dst);
#endif

#ifdef VR_OPENGL
  if ((src == GL_MAX_EXT) && (dst == GL_ZERO)) {
    glBlendFunc(GL_DST_COLOR, GL_SRC_COLOR);
    glBlendEquationEXT(GL_MAX_EXT);
    glEnable(GL_BLEND);
  } else if (dir == -1) {
    if ((src == GL_SRC_ALPHA) && (dst == GL_ONE_MINUS_SRC_ALPHA)) {
      glBlendFunc(GL_SRC_ALPHA_SATURATE,  GL_ONE);
      glBlendEquationEXT(GL_FUNC_ADD_EXT);
      glEnable(GL_BLEND);
    }
    /* what about other cases!? */
  } else {
    glBlendFunc(src, dst);
    if ((src == GL_ONE) && (dst == GL_ZERO)) {
      glDisable(GL_BLEND);
    } else {
      glBlendEquationEXT(GL_FUNC_ADD_EXT);
      glEnable(GL_BLEND);
    }
  }
#endif
}


static void SetupBlending(VRState *state, int one, int over, int mip, int sub)
{
  if (one) {

#ifdef VR_IRISGL
    BlendFunction(BF_ONE, BF_ZERO, RenderDirection);
#endif

#ifdef VR_OPENGL
    BlendFunction(GL_ONE, GL_ZERO, RenderDirection);
#endif

    return;
  }

  if (mip && !state->tableData->opaquify) {

#ifdef VR_IRISGL
    BlendFunction(BF_MAX, BF_ZERO, RenderDirection);
#endif

#ifdef VR_OPENGL
    BlendFunction(GL_MAX_EXT, GL_ZERO, RenderDirection);
#endif

    return;
  }

  if (over || state->tableData->opaquify) {

#ifdef VR_IRISGL
    BlendFunction(BF_SA, BF_MSA, RenderDirection);
#endif

#ifdef VR_OPENGL
    BlendFunction(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA, RenderDirection);
#endif

    return;
  }

  if (sub) {

#ifdef VR_IRISGL
    /*
     * WARNING!
     *
     * This is not supported in the IrisGL standard specification.
     */
    BlendFunction(BF_MIN, BF_MIN, RenderDirection);
#endif

#ifdef VR_OPENGL
    glBlendFunc(GL_SRC_COLOR, GL_DST_COLOR);
    glBlendEquationEXT(GL_FUNC_SUBTRACT_EXT);
#endif

    return;
  }
}


/* Enable the current brick */
void EnableBrick(VRState *state, Brick *b)
{
  if (!state->world->doTextures)
    return;

#ifdef VR_IRISGL
  texbind(0, b->texId);
  tevbind(0, b->tevId);
#endif

#ifdef VR_OPENGL
  if (b->texId)
    glEnable(GL_TEXTURE_3D_EXT);
#endif
}


/* Disable the current brick */
void DisableBrick(VRState *state)
{
  if (!state->world->doTextures)
    return;

#ifdef VR_IRISGL
  texbind(0, 0);
  tevbind(0, 0);
#endif

#ifdef VR_OPENGL
  glDisable(GL_TEXTURE_3D_EXT);
#endif
}


/*
 * Load a brick into texture memory
 */
void LoadBrick(VRState *state, VRVolumeData *vd, Brick *b)
{
  if (!state->world->doTextures)
    return;

#ifdef VR_IRISGL
  texbind(0, b->texId);
  tevbind(0, b->tevId);
#endif

#ifdef VR_OPENGL
  if (b->texId) {

#ifdef VR_TEXOBJS
    glBindTextureEXT(GL_TEXTURE_3D_EXT, b->texId);
#else
    b->forceDownload = TRUE;
#endif

    if (vd->drawInterp) {
      glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    } else {
      glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
      glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    }

#ifdef VR_IMPACT
    if (state->tableData->opaquify) {
      glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
      glTexParameteri(GL_TEXTURE_3D_EXT, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    }
#endif

    if (b->forceDownload) {
#if (defined(VR_KONA) || defined(VR_VENICE))
#ifdef VR_SUBTEX
      glTexSubImage3DEXT(GL_TEXTURE_3D_EXT, 0, 0, 0, 0, b->xRes, b->yRes, b->zRes,
			 GL_LUMINANCE, GL_UNSIGNED_BYTE, b->data);
#else
      glTexImage3DEXT(GL_TEXTURE_3D_EXT, 0, GL_INTENSITY8_EXT,
		      b->xRes, b->yRes, b->zRes, 0, GL_LUMINANCE,
		      GL_UNSIGNED_BYTE, b->data);
#endif
#endif

#ifdef VR_IMPACT
#ifdef VR_SUBTEX
      if (b->forceRedef) {
	if (vd->drawColor)
	  glTexImage3DEXT(GL_TEXTURE_3D_EXT, 0, GL_RGBA8_EXT,
			  b->xRes, b->yRes, b->zRes, 0, GL_COLOR_INDEX,
			  GL_UNSIGNED_BYTE, NULL);
	else
	  glTexImage3DEXT(GL_TEXTURE_3D_EXT, 0, GL_LUMINANCE8_ALPHA8_EXT,
			  b->xRes, b->yRes, b->zRes, 0, GL_COLOR_INDEX,
			  GL_UNSIGNED_BYTE, NULL);
	b->forceRedef = FALSE;
      }

      glTexSubImage3DEXT(GL_TEXTURE_3D_EXT, 0, 0, 0, 0, b->xRes, b->yRes, b->zRes,
			 GL_COLOR_INDEX, GL_UNSIGNED_BYTE, b->data);
#else
      if (vd->drawColor)
	glTexImage3DEXT(GL_TEXTURE_3D_EXT, 0, GL_RGBA8_EXT,
			b->xRes, b->yRes, b->zRes, 0, GL_COLOR_INDEX,
			GL_UNSIGNED_BYTE, b->data);
      else
	glTexImage3DEXT(GL_TEXTURE_3D_EXT, 0, GL_LUMINANCE8_ALPHA8_EXT,
			b->xRes, b->yRes, b->zRes, 0, GL_COLOR_INDEX,
			GL_UNSIGNED_BYTE, b->data);
#endif
#endif
      b->forceDownload = FALSE;
    }
  }
#endif

  EnableBrick(state, b);
}


void SetWhiteAlpha(VRState *state)
{
  float alphaScale;

  if ( state->mode->volumeMode ) {
    alphaScale = exp(15.0*state->view->delta);
    if (state->view->debug == 8)
      printf("Alpha scale = (%f)\n", alphaScale, state->view->delta);
  } else
    alphaScale = 1.0;
     
#ifdef VR_IRISGL
  {
    long white = 0x00ffffff | (((char) 255.0 * alphaScale) << 24);
    cpack(white);
  }
#endif

#ifdef VR_OPENGL
  glColor4f(1.0, 1.0, 1.0, alphaScale);
#endif
}


matrix BTRMat, RTTMat;

void LookUpBrickVertices(Brick *br, VRState *state, coord cpt[8], int RenderVolume,
			  int volNumber)
{
  VRVolumeData   *vd = state->volumeData[volNumber];
  register int    i;

  /*
   * Place the brick cube at it's proper location and rotate it.
   *   Save the resulting transformation matrix and apply it to
   *   each vertex of the cube.
   */
  matrix_copy(IdentityMatrix, 4, BTRMat);
  if (RenderVolume) {
    matrix_translate(1.0, 1.0, 1.0, BTRMat);
    matrix_scale(0.5, 0.5, 0.5, BTRMat);
    matrix_translate(br->xOff, br->yOff, br->zOff, BTRMat);
    matrix_scale(1.0/vd->nxBricks, 1.0/vd->nyBricks, 1.0/vd->nzBricks, BTRMat);
    matrix_scale(2.0, 2.0, 2.0, BTRMat);
    matrix_translate(-1.0, -1.0, -1.0, BTRMat);
  }
  matrix_mult_safe(BTRMat, vd->VTRMat, BTRMat);

  /*
   * Transform a cube into the Z-eye coordinate system
   */
  for (i=0; i<8; i++)
    coord_transform(PlusMinusCube[i], BTRMat, cpt[i]);

  /*
   * Create a transformation which undoes (inverts) the above transformation 
   * so the texture coordinates live in the range [0.0, 1.0]
   */
  matrix_invert4(BTRMat, RTTMat);
  matrix_translate(1.0, 1.0, 1.0, RTTMat);
  matrix_scale(0.5, 0.5, 0.5, RTTMat);
  matrix_translate(br->txOff, br->tyOff, br->tzOff, RTTMat);
  matrix_scale(br->txScl, br->tyScl, br->tzScl, RTTMat);
}


/*
 * Clip the slice plane against the brick in GL eye coordinate space
 */
static int GetPoly(float a, float b, float c, float d, 
		    float tl[8][3], float cpt[8][3])
{
  register int    i, n, j;
  int             edge[8];
  int             faces[6];
  register int    lastedge, lastface, face;
  float           pleval, plevalnxt;
  register int    span;

  for (i = 0; i < 6; i++)
    faces[i] = 0;

  n = i = 0;

  pleval    = cpt[edgevertices[i][0]][0]*a + 
	      cpt[edgevertices[i][0]][1]*b +
	      cpt[edgevertices[i][0]][2]*c + d;

  plevalnxt = cpt[edgevertices[i][1]][0]*a + 
	      cpt[edgevertices[i][1]][1]*b +
	      cpt[edgevertices[i][1]][2]*c + d;

  span = (((*(int *)&pleval)&0x80000000) ^ ((*(int*)(&plevalnxt)&0x80000000)));


  /* attempt to find an initial intersection of slice with 3D volume */
  /* if sign bits are different, endpoints of edge span slicing plane */
  while (i < 12 && !span) 
  {
    i++;
    pleval    = cpt[edgevertices[i][0]][0]*a + 
		cpt[edgevertices[i][0]][1]*b +
		cpt[edgevertices[i][0]][2]*c + d;

    plevalnxt = cpt[edgevertices[i][1]][0]*a + 
		cpt[edgevertices[i][1]][1]*b +
		cpt[edgevertices[i][1]][2]*c + d;
    span = (((*(int *)&pleval)&0x80000000) ^ 
	    ((*(int*)(&plevalnxt)&0x80000000)));
  }

  if (i < 12) {
    face = edgefaces[i][0];
    lastface = edgefaces[i][1];
    lastedge = -1;

    while (faces[lastface] < 2) 
    {
      j = 0;
      if (faces[face] > 1) 
      {
	if (faces[edgefaces[lastedge][0]] < 2)
	  face = edgefaces[lastedge][0];
	else
	  face = edgefaces[lastedge][1];
      }

      while (faces[face] < 2) {
	i = faceedges[face][j];

	pleval    = cpt[edgevertices[i][0]][0]*a + 
		    cpt[edgevertices[i][0]][1]*b +
		    cpt[edgevertices[i][0]][2]*c + d;

	plevalnxt = cpt[edgevertices[i][1]][0]*a + 
	            cpt[edgevertices[i][1]][1]*b +
		    cpt[edgevertices[i][1]][2]*c + d;

	span = (((*(int *)&pleval)&0x80000000) ^ 
		((*(int*)(&plevalnxt)&0x80000000)));

	if ((lastedge != i) && span) {
	  float t1,t2;

	  if (plevalnxt == pleval)
	    fprintf(stderr,"GetPoly: will get divide by 0\n");

	  t1 = 1./(plevalnxt - pleval);
	  t1 *= plevalnxt;
	  t2 = (1.-t1);

	  tl[n][0] = cpt[edgevertices[i][0]][0]*t1 + cpt[edgevertices[i][1]][0]*t2;
	  tl[n][1] = cpt[edgevertices[i][0]][1]*t1 + cpt[edgevertices[i][1]][1]*t2;
	  tl[n][2] = cpt[edgevertices[i][0]][2]*t1 + cpt[edgevertices[i][1]][2]*t2;

	  edge[n] = i;
	  lastedge = i;
	  faces[edgefaces[i][0]]++;
	  faces[edgefaces[i][1]]++;
	  n++;
	}
	j++;
      }
    }
  }
  return n;
}


static void DrawPolys(Brick *b, VRState *state, float cpt[8][3], int Direction)
{
  int             n, i, j;
  float           d;
  float           minZ, maxZ;
  int             PolyCount;

#ifdef VR_IRISGL
  float p[8][3];
#endif

#ifdef VR_OPENGL
  GLfloat p[8][3];
#endif

  if (!state->mode->loopMode && (state->view->debug != 6))
  {
     float delta = state->view->delta;

#ifdef VR_IRISGL
     mmode(MTEXTURE);
     loadmatrix(RTTMat);
     mmode(MVIEWING);
#endif

#ifdef VR_OPENGL
     glMatrixMode(GL_TEXTURE);
     glLoadMatrixf((GLfloat *) RTTMat);
     glMatrixMode(GL_MODELVIEW);
#endif

     minZ =  HUGE;
     maxZ = -HUGE;

     for (i = 0; i < 8; i++) {
	if (minZ > cpt[i][2]) minZ = cpt[i][2];
	if (maxZ < cpt[i][2]) maxZ = cpt[i][2];
     }

     minZ = floor(minZ/delta) * delta;
     maxZ = floor(maxZ/delta) * delta;

     PolyCount = 0;

     for (d = minZ; (Direction==1) ? (d<=maxZ) : (d>=maxZ); d += delta*Direction) {

	n = GetPoly(0.0, 0.0, 1.0, -d, p, cpt);

	if (n  > 2) {

	   PolyCount++;

	   if (state->view->debug != 5) {

#ifdef VR_IRISGL
	     bgnpolygon();
	     for (j=0; j<n; j++) {
	       t3f(p[j]); v3f(p[j]);
	     }
	     endpolygon();
#endif

#ifdef VR_OPENGL
	     glBegin(GL_POLYGON);
	     for (j=0; j<n; j++) {
	       glTexCoord3fv(p[j]); glVertex3fv(p[j]);
	     }
	     glEnd();
#endif

	   } else {

#ifdef VR_IRISGL
	     bgnline();
	     for (j = 0; j < n; j++) {
	       t3f(p[j]); v3f(p[j]);
	     }
	     t3f(p[0]); v3f(p[0]);
	     endline();
#endif

#ifdef VR_OPENGL
	     glBegin(GL_LINE_STRIP);
	     for (j = 0; j < n; j++) {
	       glTexCoord3fv(p[j]); glVertex3fv(p[j]);
	     }
	     glTexCoord3fv(p[0]); glVertex3fv(p[0]);
	     glEnd();
#endif
	   }
	 }
      }
   }
}


static void DrawSlice(VRState *state, Brick *b,
		      float cpt[8][3], float cp[4],
		      int RenderVolume, int RenderOutline,
		      int RenderPlane, int RenderDirection,
		      unsigned long color)
{
  float           p[8][3];
  int             n, j;

  if (!RenderOutline && !RenderPlane)
    return;

  if (!state->world->doTextures)
    RenderPlane = FALSE;

#ifdef VR_IRISGL
  mmode(MTEXTURE);
  loadmatrix(RTTMat);
  mmode(MVIEWING);
#endif

#ifdef VR_OPENGL
  glMatrixMode(GL_TEXTURE);
  glLoadMatrixf((GLfloat *) RTTMat);
  glMatrixMode(GL_MODELVIEW);
#endif

  n = GetPoly(cp[0], cp[1], cp[2], cp[3], p, cpt);

  if (n > 2) {
    if (state->mode->sliceMode && RenderOutline)
    {
      DisableBrick(state);

      SetupBlending(state, False, True, False, False);
      
#ifdef VR_IRISGL
      cpack(color);
      bgnline();
      for (j = 0; j < n; j++) {
        v3f(p[j]);
      }
      v3f(p[0]);
      endline();
#endif

#ifdef VR_OPENGL
      glColor4f(COLOR_TO_OGL(color));
      glBegin(GL_LINE_STRIP);
      for (j=0; j<n; j++) {
        glVertex3fv(p[j]);
      }
      glVertex3fv(p[0]);
      glEnd();
#endif

      EnableBrick(state, b);
    }

    if (RenderVolume && RenderPlane) {
      if (state->mode->loopMode || (state->view->debug == 6))
	SetupBlending(state, True, False, False, False);
      else
	if (state->mode->mipMode)
	  SetupBlending(state, False, False, True, False);
	else
	  SetupBlending(state, False, True, False, False);

      SetWhiteAlpha(state);

      if (state->view->debug != 4) {

#ifdef VR_IRISGL
	bgnpolygon();
	for (j=0; j<n; j++) {
	  t3f(p[j]); v3f(p[j]);
	}
	endpolygon();
#endif

#ifdef VR_OPENGL
	glBegin(GL_POLYGON);
	for (j=0; j<n; j++) {
	  glTexCoord3fv(p[j]); glVertex3fv(p[j]);
	}
	glEnd();
#endif

      } else {

	DisableBrick(state);

#ifdef VR_IRISGL
	bgnpolygon();
	for (j = 0; j < n; j++) {
	  v3f(p[j]);
	}
	endpolygon();
#endif

#ifdef VR_OPENGL
	glBegin(GL_POLYGON);
	for (j=0; j<n; j++) {
	  glVertex3fv(p[j]);
	}
	glEnd();
#endif

	EnableBrick(state, b);
      }
    }
  }
}


/*
 * Minima of the bound cuboid
 */
static float minZ;

/*
 * Tests whether the point parameter touches the
 * backsize of the volume bounding cuboid.
 */
#define X                 0
#define Y                 1
#define Z                 2
#define TouchesBack(pt) (minZ == (pt)[Z])  

static void DrawFrontFrame(VRState *state, float cpt[8][3],
			    int RenderDirection, int over, int volNumber);


/*
 *  Used to draw the back part of the boarder on this brick
 */
static void DrawBackFrame(VRState *state, float cpt[8][3],
			   int RenderDirection, int over, int volNumber)
{
  int i;

  if (RenderDirection == 1) 
  {
     /*
     * Compute the minimum point of the bounding volume
     */

    minZ = HUGE;

    for (i = 0; i < 8; i++)
      if (minZ > cpt[i][Z]) minZ = cpt[i][Z];
   
    /* 
     * Use the over operator to get anti-aliased line rendering
     *   done correctly.
     */

#ifdef VR_IRISGL
    if (state->view->curVol == volNumber)
      cpack(state->world->curFrameColor);
    else
      cpack(state->world->frameColor);
#endif

#ifdef VR_OPENGL
    if (state->view->curVol == volNumber)
      glColor4f(COLOR_TO_OGL(state->world->curFrameColor));
    else
      glColor4f(COLOR_TO_OGL(state->world->frameColor));
#endif

    SetupBlending(state, False, True, False, False);

    for (i = 0; i < 4; i++)
      if (TouchesBack(cpt[i]) || TouchesBack(cpt[(i+1)%4])) {

#ifdef VR_IRISGL
	bgnline();
	v3f(cpt[i]);
	v3f(cpt[(i+1)%4]);
	endline();
#endif

#ifdef VR_OPENGL
	glBegin(GL_LINE_STRIP);
	glVertex3fv(cpt[i]);
	glVertex3fv(cpt[(i+1)%4]);
	glEnd();
#endif
      }

    for (i = 0; i < 4; i++)
      if (TouchesBack(cpt[i+4]) || TouchesBack(cpt[((i+1)%4)+4])) {

#ifdef VR_IRISGL
	bgnline();
	v3f(cpt[i+4]);
	v3f(cpt[((i+1)%4)+4]);
	endline();
#endif

#ifdef VR_OPENGL
	glBegin(GL_LINE_STRIP);
	glVertex3fv(cpt[i+4]);
	glVertex3fv(cpt[((i+1)%4)+4]);
	glEnd();
#endif
      }

    for (i = 0; i < 4; i++)
      if (TouchesBack(cpt[i]) || TouchesBack(cpt[i+4])) {

#ifdef VR_IRISGL
	bgnline();
	v3f(cpt[i]);
	v3f(cpt[i+4]);
	endline();
#endif

#ifdef VR_OPENGL
	glBegin(GL_LINE_STRIP);
	glVertex3fv(cpt[i]);
	glVertex3fv(cpt[i+4]);
	glEnd();
#endif
      }

  } else

     DrawFrontFrame(state, cpt, 1, RenderDirection, volNumber);
}

/*
 *  Used to draw the front part of the boarder on this brick
 */
static void DrawFrontFrame(VRState *state, float cpt[8][3],
			    int RenderDirection, int over, int volNumber)
{
  int i;

  if (RenderDirection == 1) {

    /* Use the over operator to get anti-aliased line rendering done correctly. */

#ifdef VR_IRISGL
    if (state->view->curVol == volNumber)
      cpack(state->world->curFrameColor);
    else
      cpack(state->world->frameColor);
#endif

#ifdef VR_OPENGL
    if (state->view->curVol == volNumber)
      glColor4f(COLOR_TO_OGL(state->world->curFrameColor));
    else
      glColor4f(COLOR_TO_OGL(state->world->frameColor));
#endif

    SetupBlending(state, False, True, False, False);

    for (i = 0; i < 4; i++)
      if (!TouchesBack(cpt[i]) && !TouchesBack(cpt[(i+1)%4])) {

#ifdef VR_IRISGL
	bgnline();
	v3f(cpt[i]);
	v3f(cpt[(i+1)%4]);
	endline();
#endif

#ifdef VR_OPENGL
	glBegin(GL_LINE_STRIP);
	glVertex3fv(cpt[i]);
	glVertex3fv(cpt[(i+1)%4]);
	glEnd();
#endif
      }

    for (i = 0; i < 4; i++)
      if (!TouchesBack(cpt[i+4]) && !TouchesBack(cpt[((i+1)%4)+4])) {

#ifdef VR_IRISGL
	bgnline();
	v3f(cpt[i+4]);
	v3f(cpt[((i+1)%4)+4]);
	endline();
#endif

#ifdef VR_OPENGL
	glBegin(GL_LINE_STRIP);
	glVertex3fv(cpt[i+4]);
	glVertex3fv(cpt[((i+1)%4)+4]);
	glEnd();
#endif
      }

    for (i = 0; i < 4; i++)
      if (!TouchesBack(cpt[i]) && !TouchesBack(cpt[i+4])) {

#ifdef VR_IRISGL
	bgnline();
	v3f(cpt[i]);
	v3f(cpt[i+4]);
	endline();
#endif

#ifdef VR_OPENGL
	glBegin(GL_LINE_STRIP);
	glVertex3fv(cpt[i]);
	glVertex3fv(cpt[i+4]);
	glEnd();
#endif
      }

  } else

    DrawBackFrame(state, cpt, 1, RenderDirection, volNumber);
}


/*
 * Draw the volume
 */
void RenderBrick(VRState      *state,
		 VRVolumeData *vd,
		 Brick        *b,
#ifdef VR_IRISGL
		  unsigned long zmask,
#endif
#ifdef VR_OPENGL
		  GLenum        zmask,
#endif
		  int           RenderVolume,
		  int           RenderFrame,
		  int           RenderSliceOutline,
		  int           RenderSlicePlane,
		  int           RenderDirection,
                  int           DrawBackPartOfFrame,
		  int           DrawFrontPartOfFrame,
		  int           volNumber)
{
  int             i;
  static float    BrickVertex[8][3];
  float           cp[MAX_PLANES][4];

  /*
   * Set up cube coordinates.
   */
  LookUpBrickVertices(b, state, (coord *)BrickVertex, RenderVolume, volNumber);

  /*
   * Set up the clipping planes
   */
  if (state->mode->sliceMode) {
    DefineClipPlanes(state, (float*) cp);

    /* Shift the planes slightly by volume number */
    for (i=0; i<state->planeData->nPlanes; i++)
      cp[i][3] += SLICE_SHIFT_FRACTION * volNumber;
  }

  LoadBrick(state, vd, b);

  /*
   * Draw the slicing planes that face away
   */
  if (state->mode->sliceMode) {

#ifdef VR_IRISGL
    zwritemask(0xffffffff);

    if (state->tableData->opaquify)
      afunction(0, AF_NOTEQUAL);
#endif

#ifdef VR_OPENGL
    glDepthMask(GL_TRUE);

    if (state->tableData->opaquify) {
      glEnable(GL_ALPHA_TEST);
      glAlphaFunc(GL_NOTEQUAL, 0.0);
    }
#endif

    for (i=0; i<state->planeData->nPlanes; i++) {

      Plane *p = &(state->planeData->plane[i]);

      if (!p->facing && DrawBackPartOfFrame && p->active) 
      {
	/* Turn on all active planes except this one */
	if (state->mode->volumeMode &&
	     !(state->planeData->planeMoving && (i == state->view->curPlane)))
	  EnableActiveClipPlanes(state, i);

	/* Draw this plane */
	DrawSlice(state, b, BrickVertex, cp[i], RenderVolume,
		   RenderSliceOutline, RenderSlicePlane, RenderDirection,
		   (state->view->curPlane == i) ?
		    state->world->curPlaneColor : state->world->planeColor);

	/* Turn all planes off */
	if (state->mode->volumeMode)
	  DisableAllClipPlanes(state);
      }
    }

#ifdef VR_IRISGL
    zwritemask(zmask);

    if (state->tableData->opaquify)
      afunction(0, AF_ALWAYS);
#endif

#ifdef VR_OPENGL
    glDepthMask(zmask);

    if (state->tableData->opaquify)
      glDisable(GL_ALPHA_TEST);
#endif
  }

  /*
   * Draw the back part of the brick outline depending on whether the
   * user and caller have asked for it
   */
  DisableBrick(state);
  if (state->mode->frameMode && RenderFrame && DrawBackPartOfFrame) 
  {
     if (state->mode->sliceMode && state->mode->volumeMode && !state->mode->loopMode)
      EnableActiveClipPlanes(state, -1);
    
    DrawBackFrame(state, BrickVertex, RenderDirection, RenderDirection, volNumber);

    if (state->mode->sliceMode && state->mode->volumeMode && !state->mode->loopMode)
      DisableAllClipPlanes(state);
  }
  EnableBrick(state, b);

  /*
   * Draw the polygons which make up the volume rendering
   */
  if (state->world->doTextures && state->mode->volumeMode && RenderVolume) 
  {
    if (state->mode->sliceMode)
      EnableActiveClipPlanes(state, -1);

     /* Set the blend function to what ever rendering mode we're in */
    if (state->mode->mipMode)
      SetupBlending(state, False, False, True, False);
    else
      SetupBlending(state, False, True, False, False);

    SetWhiteAlpha(state);

    DrawPolys(b, state, BrickVertex, RenderDirection);

    if (state->mode->sliceMode)
      DisableAllClipPlanes(state);
  }

  /*
   * Draw the front part of the brick outline depending on whether the
   * user and caller have asked for it
   */
  DisableBrick(state);
  if (state->mode->frameMode && RenderFrame && DrawFrontPartOfFrame) 
  {
     if (state->mode->sliceMode && state->mode->volumeMode && !state->mode->loopMode)
      EnableActiveClipPlanes(state, -1);

    DrawFrontFrame(state, BrickVertex, RenderDirection, RenderDirection, volNumber);

    if (state->mode->sliceMode && state->mode->volumeMode && !state->mode->loopMode)
      DisableAllClipPlanes(state);
  }
  EnableBrick(state, b);

  /*
   * Draw the slicing plane last if it's in front of the volume
   */
  if (state->mode->sliceMode) {

#ifdef VR_IRISGL
    zwritemask(0xffffffff);

    if (state->tableData->opaquify)
      afunction(0, AF_NOTEQUAL);
#endif

#ifdef VR_OPENGL
    glDepthMask(GL_TRUE);

    if (state->tableData->opaquify) {
      glEnable(GL_ALPHA_TEST);
      glAlphaFunc(GL_NOTEQUAL, 0.0);
    }
#endif

    for (i=0; i<state->planeData->nPlanes; i++) {

      Plane *p = &(state->planeData->plane[i]);

      if (p->facing && DrawFrontPartOfFrame && p->active) 
      {
	/* Turn on all active planes except this one */
	if (state->mode->volumeMode &&
	     !(state->planeData->planeMoving && (i == state->view->curPlane)))
	  EnableActiveClipPlanes(state, i);

	/* Draw this plane (perhaps with outline) */
	DrawSlice(state, b, BrickVertex, cp[i], RenderVolume,
		   RenderSliceOutline, RenderSlicePlane, RenderDirection,
		   (state->view->curPlane == i) ?
		   state->world->curPlaneColor : state->world->planeColor);

	/* Turn all planes off */
	if (state->mode->volumeMode)
	  DisableAllClipPlanes(state);
      }
    }

#ifdef VR_IRISGL
    zwritemask(zmask);

    if (state->tableData->opaquify)
      afunction(0, AF_ALWAYS);
#endif

#ifdef VR_OPENGL
    glDepthMask(zmask);

    if (state->tableData->opaquify)
      glDisable(GL_ALPHA_TEST);
#endif
  }
}


#define MaxBricks     128
static  Polygon  PolyList[MAX_VOLS][MaxBricks];
static  float    CubeList[MAX_VOLS][MaxBricks][8][3];
static  matrix   TextureMat[MAX_VOLS][MaxBricks];

void MergeRenderPolyList(VRState *state, int RenderDirection, int RenderingSlice)
{
#ifdef VR_MERGING
  VRVolumeData *vd;
  int      j, k, l;
  Polygon *p;

  /* Set the blend function to what ever rendering mode we're in */
  if (state->mode->mipMode)
    SetupBlending(state, False, False, True, False);
  else
    SetupBlending(state, False, True, False, False);

  SetWhiteAlpha(state);
  
  for (j=0; j<state->world->nVols; j++) {

    vd = state->volumeData[j];
    if (state->volumeData[j]->active) {

      LoadTlut(state, j);

      for (k=0; k<vd->nBricks; k++) {

	p = &PolyList[j][k];
	if (p->n > 0) {
	  if (RenderingSlice && (state->view->debug == 4)) {

	    DisableBrick(state);

#ifdef VR_IRISGL
	    bgnpolygon();
	    for (l=0; l<p->n; l++) {
	      v3f((float*) &(p->Gvertex[l]));
	    }
	    endpolygon();
#endif

#ifdef VR_OPENGL
	    glBegin(GL_POLYGON);
	    for (l=0; l<p->n; l++) {
	      glVertex3fv((GLfloat*) &(p->Gvertex[l]));
	    }
	    glEnd();
#endif

	  } else {

	    LoadBrick(state, vd, vd->sbrick[state->mode->brickRes][k]);

	    if (!RenderingSlice && (state->view->debug == 5)) {

#ifdef VR_IRISGL
	      bgnline();
	      for (l=0; l<p->n; l++) {
		t3f((float*) &(p->Tvertex[l]));
		v3f((float*) &(p->Gvertex[l]));
	      }
	      t3f((float*) &(p->Tvertex[0]));
	      v3f((float*) &(p->Gvertex[0]));
	      endline();
#endif

#ifdef VR_OPENGL
	      glBegin(GL_LINE_STRIP);
	      for (l=0; l<p->n; l++) {
		glTexCoord3fv((float*) &(p->Tvertex[l]));
		glVertex3fv((float*) &(p->Gvertex[l]));
	      }
	      glTexCoord3fv((float*) &(p->Tvertex[0]));
	      glVertex3fv((float*) &(p->Gvertex[0]));
	      glEnd();
#endif

	    } else {

#ifdef VR_IRISGL
	      bgnpolygon();
	      for (l=0; l<p->n; l++) {
		t3f((float*) &(p->Tvertex[l]));
		v3f((float*) &(p->Gvertex[l]));
	      }
	      endpolygon();
#endif

#ifdef VR_OPENGL
	      glBegin(GL_POLYGON);
	      for (l=0; l<p->n; l++) {
		glTexCoord3fv((float*) &(p->Tvertex[l]));
		glVertex3fv((float*) &(p->Gvertex[l]));
	      }
	      glEnd();
#endif
	    }
	  }
	}
      }
    }
  }
#endif
}


void MatteRenderPolyList(VRState *state, int RenderDirection, int RenderingSlice)
{
#ifdef VR_MATTING
  VRVolumeData *vd;
  int      j, k, l;
  Polygon *p;
  int      FirstPass;
  long     xOrig, yOrig;

#ifdef VR_IRISGL
  static int ilbuff = 0;
#endif
  
#ifdef VR_IRISGL
  /* Get an IL buffer */
  if (ilbuff == 0)
    if (ilbuffer(1) != 1) {
      fprintf(stderr, "Could not allocate aux buffer\n");
      exit(EXIT_FAILURE);
    }
#endif

#ifdef VR_IRISGL
  getorigin(&xOrig, &yOrig);
  readsource(SRC_ILBUFFER_1);
  ildraw(1);
  cpack(0);
  clear();
#endif

  /* Loop through all the polygons rendering them into the aux buffer. */
  FirstPass = TRUE;

  for (j=0; j<state->world->nVols; j++) {

    vd = state->volumeData[j];
    if (vd->active) {

      if (FirstPass)
	SetupBlending(state, True, False, False, False);
      else {

#ifdef VR_IRISGL
	blendfunction(BF_DC, BF_ZERO);
#endif
      }

#ifdef VR_IRISGL
      /* Draw in opaque white so that we can see the texture */
      cpack(OPAQUE_WHITE);
#endif

      /*
       * After the first volume's worth of polygons in the
       * current plane then use all the subsequent volumes
       * to modulate what ever is already in the frame
       * buffer.
       */
#ifdef VR_IRISGL
      tlutbind(0,  vd->curTable+1);
#endif

      for (k=0; k<vd->nBricks; k++) {

	p = &PolyList[j][k];
	if (p->n > 0) {

	  LoadBrick(state, vd, vd->sbrick[state->mode->brickRes][k]);

#ifdef VR_IRISGL
	  bgnpolygon();
	  for (l=0; l<p->n; l++) {
	    t3f((float*) &(p->Tvertex[l]));
	    v3f((float*) &(p->Gvertex[l]));
	  }
	  endpolygon();
#endif
	}
      }

      if (FirstPass) FirstPass = FALSE;
    }
  }
    
  SetupBlending(state, True, False, False, False);

#ifdef VR_IRISGL
  cpack(0);
#endif

  for (j=0; j<state->world->nVols; j++) {

    vd = state->volumeData[j];
    if (vd->active) {

      for (k=0; k<vd->nBricks; k++) {

	p = &PolyList[j][k];
	if (p->n > 0) {
#ifdef VR_IRISGL
	  bgnpolygon();
	  for (l=0; l<p->n; l++) { 
	    v3f((float*) &(p->Gvertex[l]));
	  }
	  endpolygon();
#endif
	}
      }
      break;
    }
  }

  /* Set the blend function to what ever rendering mode we're in */
  if (state->mode->mipMode)
    SetupBlending(state, False, False, True, False);
  else
    SetupBlending(state, False, True, False, False);

#ifdef VR_IRISGL
  /* Move the pixels from the aux buffer blend them into the frame buffer */
  ildraw(0);
  rectcopy(0, 0, state->view->width - 1, state->view->height - 1, 0, 0);
  readsource(SRC_AUTO);
#endif

#endif
}


static coord LightDir = { 0.01, 0.01, 0.01 };
static coord RotatedLightDir;

void ShadeRenderPolyList(VRState *state, int RenderDirection, int RenderingSlice)
{
#ifdef VR_SHADING
  VRVolumeData *vd;
  int      j, k, l;
  int      pass;
  Polygon *p;
  long     xOrig, yOrig;

#ifdef VR_IRISGL
  /* Set things up to use the frame buffer to draw */
  ildraw(1);
  getorigin(&xOrig, &yOrig);
  viewport(xOrig, xOrig + state->view->width - 1, yOrig, yOrig + state->view->height - 1);
  cpack(OPAQUE_WHITE);
  readsource(SRC_ILBUFFER_1);
#endif
  
   /*
    * Loop through all the polygons rendering them
    * into the main buffer and using the aux buffer to
    * compute the shifted image
    */
   for (pass=0; pass<2; pass++) {
     if (pass==0) {
       SetupBlending(state, True, False, False, False);

#ifdef VR_IRISGL
       cpack(OPAQUE_WHITE);
       zbuffer(TRUE);
#endif
     } else {

       SetupBlending(state, False, False, False, True);
#ifdef VR_IRISGL
       cpack(0x00ffffff);
       zbuffer(FALSE);
#endif
     }

     for (j=0; j<state->world->nVols; j++) {

       vd = state->volumeData[j];
       if (vd->active) {

#ifdef VR_IRISGL
	 tlutbind(0, vd->curTable+1);
#endif

	 for (k=0; k<vd->nBricks; k++) {

	   p = &PolyList[j][k];
	   if (p->n > 0) {

	     LoadBrick(state, vd, vd->sbrick[state->mode->brickRes][k]);

	     if (pass == 0) {

#ifdef VR_IRISGL
	       bgnpolygon();
	       for (l=0; l<p->n; l++) {
		 t3f((float*) &(p->Tvertex[l]));
		 v3f((float*) &(p->Gvertex[l]));
	       }
	       endpolygon();
#endif

	     } else {

#ifdef VR_IRISGL
	       bgnpolygon();
	       for (l=0; l<p->n; l++) {
		 p->Tvertex[l].x += RotatedLightDir.x;
		 p->Tvertex[l].y += RotatedLightDir.y;
		 p->Tvertex[l].z += RotatedLightDir.z;

		 t3f((float*) &(p->Tvertex[l]));
		 v3f((float*) &(p->Gvertex[l]));
	       }
	       endpolygon();
#endif
	     }
	   }
	 }
       }
     }
   }
 
#ifdef VR_IRISGL
  /* Move the pixels from the aux buffer blend them into the frame buffer */
   ildraw(0);
  
   /* Make sure the alpha and zbuffer get updates */
   zbuffer(TRUE);
#endif

  /* Set the blend function to what ever rendering mode we're in */
  if (state->mode->mipMode)
    SetupBlending(state, False, False, True, False);
  else
    SetupBlending(state, False, True, False, False);

#ifdef VR_IRISGL
  rectcopy(0, 0, state->view->width - 1, state->view->height - 1, 0, 0);
  readsource(SRC_AUTO);
#endif

#endif
}


static float    minZ, maxZ;

void RenderSetup(VRState *state, int RenderDirection)
{
  VRVolumeData *vd;
  int      i, j, k;
  float    delta = state->view->delta;

  minZ =  HUGE;
  maxZ = -HUGE;

  for (j=0; j<state->world->nVols; j++) {

    vd = state->volumeData[j];
    if (vd->active) {

      StorePerVolumeMatrix(state, j);
       
      for (k=0; k<vd->nBricks; k++) {

	/* Set up cube coordinates */
	LookUpBrickVertices(vd->sbrick[state->mode->brickRes][k], 
			    state, (coord *)CubeList[j][k], TRUE, j);

	matrix_copy(RTTMat, 4, TextureMat[j][k]);
	for (i = 0; i < 8; i++) {

	  if (minZ > CubeList[j][k][i][2]) minZ = CubeList[j][k][i][2];
	  if (maxZ < CubeList[j][k][i][2]) maxZ = CubeList[j][k][i][2];
	}

	minZ = floor(minZ/delta) * delta;
	maxZ = floor(maxZ/delta) * delta;

      }
    }
  }

  /* When we're rendering front to back, then the range of the planes must be swapped. */
  if (RenderDirection == -1) {
    float tmp;
    tmp  = minZ;
    minZ = maxZ;
    maxZ = tmp;
  }
}


void RenderPlanes(VRState *state, int RenderVolume, int RenderDirection)
{
  VRVolumeData *vd;
  int PolyCount;

  if (state->mode->volumeMode && RenderVolume && !state->mode->loopMode) {

    int      i, j, k;
    int      n;
    float    p[8][3];
    float    d;
    float    delta = state->view->delta;
    Polygon *poly;

#ifdef VR_IRISGL
    static int ilbuff = 0;
#endif

#ifdef VR_IRISGL
    /*
     * Make sure the pixmode is in a mode for optimal framebuffer rectcopies
     * That is make sure it does not have to do any thing that requires a pass
     * through the GEs.
     */
    pixmode(PM_SIZE,          32);
    pixmode(PM_INPUT_FORMAT,  PM_ABGR);
    pixmode(PM_OUTPUT_FORMAT, PM_ABGR);
    pixmode(PM_INPUT_TYPE,    PM_UNSIGNED_BYTE);
    pixmode(PM_OUTPUT_TYPE,   PM_UNSIGNED_BYTE);
#endif

    /* Get an IL buffer */
    if (state->mode->renderMode == SHADE_RENDER) {
#ifdef VR_IRISGL
      if (ilbuff == 0) {
	if (ilbuffer(1) != 1) {
	  fprintf(stderr, "Could not allocate aux buffer\n");
	  exit(EXIT_FAILURE);
	}
      }
#endif

#ifdef VR_IRISGL
      /* Clear the aux buffer used to do the shading */
      ildraw(1);
      cpack(0);
      clear();

      /* Go back to the regular frame buffer by default */
      ildraw(0);
#endif
      

      /* Make rotate the light source to counter the rotation of the volume */
      coord_transform(LightDir, state->view->RotMat, RotatedLightDir);
    }

#ifdef VR_IRISGL
    /* Make sure there's a global view matrix */
    loadmatrix(state->view->RTCMat);

    /* Texture coordinates are already transformed */
    mmode(MTEXTURE);
    loadmatrix(IdentityMatrix);
    mmode(MVIEWING);
#endif
    
#ifdef VR_OPENGL
    /* Make sure there's a global view matrix */
    glLoadMatrixf((GLfloat*) state->view->RTCMat);

    /* Texture coordinates are already transformed */
    glMatrixMode(GL_TEXTURE);
    glLoadMatrixf((GLfloat *) RTTMat);
    glLoadIdentity();
    glMatrixMode(GL_MODELVIEW);
#endif
    
    PolyCount = 0;

    for (d=minZ; (RenderDirection==1)?(d<=maxZ):(d>=maxZ); d+=delta*RenderDirection) {
      for (j=0; j<state->world->nVols; j++) {

	vd = state->volumeData[j];
	if (vd->active) {
	  for (k=0; k<vd->nBricks; k++) {

	    float planeEq[4];

	    /* Shift the plane slightly based upon which volume we're in */
	    planeEq[0] = 0.0;
	    planeEq[1] = 0.0;
	    planeEq[2] = 1.0;
	    planeEq[3] = -d + delta * 0.05 * j;

	    /* Set up cube coordinates */
	    n = GetPoly(planeEq[0], planeEq[1], planeEq[2], planeEq[3],
			p, CubeList[j][k]);
	    n = GetPoly(0.0, 0.0, 1.0, -d, p, CubeList[j][k]);
	
	    poly = &PolyList[j][k];

	    if (n > 2) {

	      poly->n = n;
	      for (i=0; i<n; i++) {
		poly->Gvertex[i].x = p[i][0];
		poly->Gvertex[i].y = p[i][1];
		poly->Gvertex[i].z = p[i][2];

		coord_transform(poly->Gvertex[i], TextureMat[j][k],
				poly->Tvertex[i]);
	      }
	      PolyCount++;

	    } else {
	      poly->n = 0;
	    }
	  }
	}
      }

      /* Draw the polygons which make up the volume rendering */
      if (state->mode->sliceMode)
	EnableActiveClipPlanes(state, -1);
	        
      switch (state->mode->renderMode) {
      case MERGE_RENDER : MergeRenderPolyList(state, RenderDirection, FALSE); break;
      case MATTE_RENDER : MatteRenderPolyList(state, RenderDirection, FALSE); break;
      case SHADE_RENDER : ShadeRenderPolyList(state, RenderDirection, FALSE); break;
      default           : MergeRenderPolyList(state, RenderDirection, FALSE); break;
      }

      if (state->mode->sliceMode)
	DisableAllClipPlanes(state);
    }
  }
}


void RenderSlices(VRState *state, int RenderDirection,
		  int DrawBackPartOfFrame, int DrawFrontPartOfFrame)
{
  VRVolumeData *vd;
  int PolyCount;

  if (state->mode->sliceMode && (state->view->debug != 6)) {

    int      i, j, k, l;
    int      n;
    float    p[8][3];
    float    cp[MAX_PLANES][4];
    Polygon *poly;

#ifdef VR_IRISGL
    /*
     * Make sure the pixmode is in a mode for optimal framebuffer rectcopies
     * That is make sure it does not have to do any thing that requires a pass
     * through the GEs.
     */
    pixmode(PM_INPUT_FORMAT,  PM_ABGR);
    pixmode(PM_OUTPUT_FORMAT, PM_ABGR);
    pixmode(PM_INPUT_TYPE,    PM_UNSIGNED_BYTE);
    pixmode(PM_OUTPUT_TYPE,   PM_UNSIGNED_BYTE);
#endif

#ifdef VR_IRISGL
    /* Make sure there's a global view matrix */
    loadmatrix(state->view->RTCMat);

    /* Texture coordinates are already transformed */
    mmode(MTEXTURE);
    loadmatrix(IdentityMatrix);
    mmode(MVIEWING);
#endif

#ifdef VR_OPENGL
    /* Make sure there's a global view matrix */
    glLoadMatrixf((GLfloat*) state->view->RTCMat);

    /* Texture coordinates are already transformed */
    glMatrixMode(GL_TEXTURE);
    glLoadIdentity();
    glMatrixMode(GL_MODELVIEW);
#endif
    
    /* Set up the clipping planes */
    DefineClipPlanes(state, (float*) cp);

    PolyCount = 0;

    for (l=0; l<state->planeData->nPlanes; l++) {

      Plane *plane = &(state->planeData->plane[l]);

      if (plane->active && ((plane->facing && DrawFrontPartOfFrame) ||
			    (!plane->facing && DrawBackPartOfFrame))) {

	for (j=0; j<state->world->nVols; j++) {

	  vd = state->volumeData[j];
	  if (vd->active)
	    for (k=0; k<vd->nBricks; k++) {

	      float planeEq[4];

	      planeEq[0] = cp[l][0];
	      planeEq[1] = cp[l][1];
	      planeEq[2] = cp[l][2];
	      planeEq[3] = cp[l][3] + SLICE_SHIFT_FRACTION * j;

	      /* Set up cube coordinates. */
	      n = GetPoly(planeEq[0], planeEq[1], planeEq[2], planeEq[3],
			  p, CubeList[j][k]);
	
	      poly = &PolyList[j][k];

	      if (n > 2) {

		poly->n = n;
		for (i = 0; i < n; i++) {

		  poly->Gvertex[i].x = p[i][0];
		  poly->Gvertex[i].y = p[i][1];
		  poly->Gvertex[i].z = p[i][2];

		  coord_transform(poly->Gvertex[i], TextureMat[j][k],
				  poly->Tvertex[i]);
		}
		PolyCount++;

	      } else
		poly->n = 0;
	    }
	}

	/* Draw the polygons which make up the slice plane */
	if (state->mode->volumeMode &&
	    !(state->planeData->planeMoving && (state->view->curPlane == l)))
	  EnableActiveClipPlanes(state, l);

	switch (state->mode->renderMode) {
	case MERGE_RENDER : MergeRenderPolyList(state, RenderDirection, TRUE); break;
	case MATTE_RENDER : MatteRenderPolyList(state, RenderDirection, TRUE); break;
	case SHADE_RENDER : ShadeRenderPolyList(state, RenderDirection, TRUE); break;
	default           : MergeRenderPolyList(state, RenderDirection, TRUE); break;
	}
      
	DisableAllClipPlanes(state);
      }
    }
  }
}
