/*
Copyright 1994 Silicon Graphics, Inc. -- All Rights Reserved

If the Software is acquired by or on behalf of an entity of government
of  the  United States of America, the following provision applies: U.
S.  GOVERNMENT  RESTRICTED  RIGHTS  LEGEND:    Use,   duplication   or
disclosure of Software by the Government is subject to restrictions as
set forth in FAR 52.227-19(c)(2) or  subparagraph  (c)(1)(ii)  of  the
Rights  in  Technical  Data  and  Computer  Software  clause  at DFARS
252.227-7013 and/or in similar or successor clauses in the FAR, or the
DOD  or  NASA  FAR Supplement. Unpub-lished- rights reserved under the
Copyright  Laws  of  the  United  States.  Contractor/manufacturer  is
SILICON  GRAPHICS,  INC.,  2011  N. Shoreline Blvd., Mountain View, CA
94039- 7311.

Silicon Graphics, Inc. hereby grants  to  you  a  non-exclusive,  non-
transferable,  personal, paid-up license to use, modify and distribute
the Software solely with SGI computer products.  You must include,  in
all  copies  of  the  Software  and  any associated documentation, the
copyright notice and restricted rights legend set forth above.

THE SOFTWARE IS PROVIDED  TO  YOU  "AS-IS"  AND  WITHOUT  ANY  SUPPORT
OBLIGATION  OR  WARRANTY  OF  ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
INCLUDING WITHOUT  LIMITATION,  ANY  WARRANTY  OF  MERCHANTABILITY  OR
FITNESS  FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL SGI BE LIABLE FOR
SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF  LIABILITY,
ARISING  OUT  OF  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

You agree that you will not export or re-export the Software, directly
or  indirectly,  unless  (a)  the  Export  Administration of the U. S.
Department of Commerce explicitly permits the export or  re-export  of
the  Software  or  (b)  the  Office  of  Export Licensing of the U. S.
Department of Commerce has granted au-thorization to  you  in  writing
for the  export or re- export the Software.

If you fail to fulfill any  of  the  foregoing  obligations,  SGI  may
pursue  all  available  legal  remedies  to  enforce  these  terms and
conditions, and SGI may,  at  any  time  after  your  default  hereof,
terminate  the  license  and  rights  granted  to  you hereunder.  You
further agree that, if SGI terminates this license for  your  default,
you  will, within ten (10) days after any such termination, deliver to
SGI or  render  unusable  all  Software  originally  provided  to  you
hereunder and any copies thereof embodied in any medium.
*/


#ifndef _MIS_COORD_H_
#define _MIS_COORD_H_


#include "misMath.h"
#include "misMatrix.h"


#ifdef __cplusplus
extern "C" {
#endif


/* Cartesian geometry types */

typedef struct
{
   float x,
         y,
         z;
} coord;


/* By convention input parameters are in lower case and output parameters
 * are in upper case.
 */


/* Z = c1.c2 */

#define coord_dot_product(  c1,  c2,  Z )                                     \
{                                                                             \
   Z = (c1).x * (c2).x  +  (c1).y * (c2).y  +  (c1).z * (c2).z;               \
}


/* c3 = c1 x c2 */

#define coord_cross_product(  c1,  c2,  C3 )                                  \
{                                                                             \
   (C3).x = (c1).y * (c2).z  -  (c1).z * (c2).y;                              \
   (C3).y = (c1).z * (c2).x  -  (c1).x * (c2).z;                              \
   (C3).z = (c1).x * (c2).y  -  (c1).y * (c2).x;                              \
}


/* c2 = c1 */

#define coord_assign(  c1,  C2 )                                              \
{                                                                             \
   (C2).x = (c1).x;                                                           \
   (C2).y = (c1).y;                                                           \
   (C2).z = (c1).z;                                                           \
}                                                                             \

#define coord_fill( a, b, c, C1 )                                             \
{                                                                             \
   (C1).x = (a);                                                              \
   (C1).y = (b);                                                              \
   (C1).z = (c);                                                              \
}
/* C3 = c1 + c2 */

#define coord_add(  c1,  c2,  C3 )                                            \
{                                                                             \
   (C3).x = (c1).x + (c2).x;                                                  \
   (C3).y = (c1).y + (c2).y;                                                  \
   (C3).z = (c1).z + (c2).z;                                                  \
}


/* C3 = c1 - c2 */

#define coord_sub(  c1,  c2,  C3 )                                            \
{                                                                             \
   (C3).x = (c1).x - (c2).x;                                                  \
   (C3).y = (c1).y - (c2).y;                                                  \
   (C3).z = (c1).z - (c2).z;                                                  \
}


/* C2 = s * c1 */

#define coord_mult(  c1,  s,  C2 )                                            \
{                                                                             \
   (C2).x = s * (c1).x;                                                       \
   (C2).y = s * (c1).y;                                                       \
   (C2).z = s * (c1).z;                                                       \
}


/* C2 = c1 / s */

#define coord_div( c1,  s,  C2 )                                              \
{                                                                             \
   float __divisor;                                                           \
                                                                              \
                                                                              \
   __divisor = 1.0 / (s);                                                     \
   coord_mult( c1, __divisor, C2 );                                           \
}


/* Z = v*v */

#define coord_sqr( c1,  Z )                                                   \
{                                                                             \
   coord_dot_product( c1, c1, Z );                                            \
}

 

/* Z = |v| */

#define coord_norm( c1, Z )                                                   \
{                                                                             \
   Z = sqrt( SQUARE( (c1).x ) +                                               \
             SQUARE( (c1).y ) +                                               \
             SQUARE( (c1).z ) );                                              \
}


/* C2 = c1 / |c1| */

#define coord_normalize(  c1, C2 )                                            \
{                                                                             \
   float __magnitude;                                                         \
                                                                              \
   coord_norm( c1, __magnitude );                                             \
   coord_div( c1, __magnitude, C2 );                                          \
}


/* C2 = c1 */

#define coord_to_homogeneous( c1, C2 )                                        \
{                                                                             \
	coord_assign( c1, C2 );                                               \
        (C2).w = 1.0;                                                         \
}

#define coord_to_polar( c1, C2 )                                              \
{                                                                             \
   coord_norm( c1, (C2).rho );                                                \
   (C2).theta   = acos( (c1).z / (C2).rho );                                  \
   if ( (c1).x != 0 )                                                         \
   {                                                                          \
      (C2).phi = atan2( (c1).y, (c1).x );                                     \
      if ( (C2).phi < 0 )                                                     \
      {                                                                       \
         (C2).phi += PI_2;                                                    \
      }                                                                       \
   }                                                                          \
   else                                                                       \
   {                                                                          \
      if ( (c1).y > 0 )                                                       \
      {                                                                       \
         (C2).phi =  PI_DIV_2;                                                \
      }                                                                       \
      else                                                                    \
      {                                                                       \
         (C2).phi = -PI_DIV_2;                                                \
      }                                                                       \
   }                                                                          \
}

#define coord_unit_to_polar( c1, C2 )                                         \
{                                                                             \
   (C2).rho     = 1.0;                                                        \
   (C2).theta   = acos( (c1).z );                                             \
   if ( (c1).x != 0 )                                                         \
   {                                                                          \
      (C2).phi = atan2( (c1).y, (c1).x );                                     \
      if ( (C2).phi < 0 )                                                     \
      {                                                                       \
         (C2).phi += PI_2;                                                    \
      }                                                                       \
   }                                                                          \
   else                                                                       \
   {                                                                          \
      if ( (c1).y > 0 )                                                       \
      {                                                                       \
         (C2).phi =  PI_DIV_2;                                                \
      }                                                                       \
      else                                                                    \
      {                                                                       \
         (C2).phi = -PI_DIV_2;                                                \
      }                                                                       \
   }                                                                          \
}

#define coord_transform( c1, a, C2 )         	  	  	              \
{								              \
   coord tmp;						                      \
									      \
   tmp.x = a[0][0] * (c1).x +					              \
	   a[1][0] * (c1).y +					              \
	   a[2][0] * (c1).z +					              \
           a[3][0];						              \
									      \
   tmp.y = a[0][1] * (c1).x +					              \
	   a[1][1] * (c1).y +					              \
           a[2][1] * (c1).z +					              \
           a[3][1];						              \
									      \
   tmp.z = a[0][2] * (c1).x +					              \
	   a[1][2] * (c1).y +					              \
	   a[2][2] * (c1).z +					              \
           a[3][2];						              \
									      \
   coord_assign( tmp, C2 );  				                      \
}

#define coord_transx( c1, a, c2 )         	  	  	              \
{								              \
   c2.x = a[0][0] * (c1).x +					              \
	  a[1][0] * (c1).y +					              \
	  a[2][0] * (c1).z +					              \
          a[3][0];						              \
}

#define coord_transy( c1, a, c2 )         	  	  	              \
{								              \
   c2.y = a[0][1] * (c1).x +					              \
	  a[1][1] * (c1).y +					              \
          a[2][1] * (c1).z +					              \
          a[3][1];						              \
}

#define coord_transz( c1, a, c2 )         	  	  	              \
{								              \
   c2.z = a[0][2] * (c1).x +					              \
	  a[1][2] * (c1).y +					              \
	  a[2][2] * (c1).z +					              \
          a[3][2];						              \
}

#define coord_transw( c1, a, w )         	  	  	              \
{								              \
   w = a[0][3] * (c1).x +					              \
       a[1][3] * (c1).y +					              \
       a[2][3] * (c1).z +					              \
       a[3][3];							              \
}

   
#define coord_point_edge_check( c0, c1, pt, SIDE )  	                      \
{                                                                             \
   float __A , __B , __C ;                                                    \
  								              \
   __A = (c0).y - (c1).y;                		                      \
   __B = (c1).x - (c0).x;			                              \
   __C = ((c1).x - (c0).x)*(c0).y + ((c0).y - (c1).y)*(c0).x;	              \
									      \
   SIDE = SIGN(__A*(pt).x + __B*(pt).y - __C);		                      \
}

/* Reflect c1 about c2 returning c3 */

#define coord_reflect( c1, c2, cos_gamma, C3 )                                \
{                                                                             \
   (C3).x = 2.0*(c2).x*cos_gamma - (c1).x;                                    \
   (C3).y = 2.0*(c2).y*cos_gamma - (c1).y;                                    \
   (C3).z = 2.0*(c2).z*cos_gamma - (c1).z;                                    \
}

/* Project c1 onto plane define by normal c2 returning C3 */

#define coord_project( c1, c2, cos_gamma, C3 )                                \
{                                                                             \
    (C3).x = (c1).x - (c2).x*cos_gamma;                                       \
    (C3).y = (c1).y - (c2).y*cos_gamma;                                       \
    (C3).z = (c1).z - (c2).z*cos_gamma;                                       \
}

#define coord_perspective( fovy, near, far, aspect, M )                       \
{                                                                             \
   float ctan;                                                                \
                                                                              \
   ctan = -1.0 / tan(fovy*0.5);                                               \
                                                                              \
   matrix_assign( ctan/aspect,    0,               0,  0,                     \
                  0,           ctan,               0,  0,                     \
                  0,              0,  1.0/(far-near), -1,                     \
                  0,              0, near/(far-near),  0, M );                \
}


#define coord_print( c ) printf( "%9.4f %9.4f %9.4f", (c).x, (c).y, (c).z );


/* Routines */
extern void coord_orient( coord *c1,  coord *c2, matrix *M );
extern void coord_view(   coord *c1,  coord *c2, float theta, matrix M );
extern void coord_look( coord *c1,  coord *c2, matrix M );


#ifdef __cplusplus
}
#endif


#endif
