/** @file setenv.c
 * Setting or unsetting environment variables
 * @author Marko Mkel <marko.makela@iki.fi>
 */

/* Copyright  2006 Marko Mkel.

   This file is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   The software is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

#if defined WIN32 || defined __WIN32
# include <windows.h>
#else
# define _BSD_SOURCE
# include <stdlib.h>
#endif
#include <stdio.h>

/** Set an environment variable
 * @param name	name of the environment variable
 * @param value	value to assign to the environment variable,
 *		or NULL to unset the variable
 * @return	zero on success, nonzero on error
 */
int
set_environment (const char* name,
		 const char* value)
{
#if defined WIN32 || defined __WIN32
  if (!SetEnvironmentVariable (name, value)) {
    char* msg;
    long err = GetLastError ();
    FormatMessage (FORMAT_MESSAGE_ALLOCATE_BUFFER |
		   FORMAT_MESSAGE_FROM_SYSTEM |
		   FORMAT_MESSAGE_IGNORE_INSERTS,
		   0,
		   err,
		   0, /* default language */
		   (LPTSTR) &msg,
		   0,
		   0);
    AnsiToOem (msg, msg);
    (void) fprintf (stderr,
		    "SetEnvironmentVariable failed with code %ld: %s\n",
		    err, msg);
    LocalFree (msg);
    return -1;
  }
#else
  if (value) {
    if (setenv (name, value, 1)) {
      perror ("setenv");
      return -1;
    }
  }
  else if (unsetenv (name)) {
    perror ("unsetenv");
    return -1;
  }
#endif
  return 0;
}
