/**
 * @file photomolo.c
 * The main program
 * @author Marko Mkel <marko.makela@iki.fi>
 */

/* Copyright  2003,2004,2005,2006 Marko Mkel.

   This file is part of PHOTOMOLO, a program for generating
   thumbnail images and HTML files for browsing digital photographs.

   PHOTOMOLO is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   PHOTOMOLO is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

/* sample command line options with present syntax:
 * photomolo -T _8/s8 -T _4/s4 -T _120/h120 -T _240/h240 -T _480/h480 cd1
 */

#if defined WIN32 || defined __WIN32
# undef HAVE_PROGRESS
#else
# define HAVE_PROGRESS
# ifndef _POSIX_C_SOURCE
#  define _POSIX_C_SOURCE 200112L /* for popen(3) */
# endif
# ifndef _BSD_SOURCE
#  define _BSD_SOURCE /* for strdup(3) */
# endif
#endif
#define HAVE_CONVERTER

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>
#include "scale.h"
#include "tree.h"
#include "info.h"
#include "spec.h"
#include "html.h"
#ifdef HAVE_CONVERTER
# include <sys/stat.h> /* open(2) */
# include <sys/types.h>
# include <fcntl.h>
# include <unistd.h> /* close(2), unlink(2) */
# include "setenv.h"
#endif /* HAVE_CONVERTER */

#define PHOTOMOLO_VERSION "pHoToMoLo 1.2.4"

/** The libjpeg decompression context */
static struct jpeg_decompress_struct cin;
/** The libjpeg compression context */
static struct jpeg_compress_struct cout;

/** Thumbnail specifications */
static struct thumb_spec* thumbs = 0;
/** libjpeg compression quality */
static int quality = 75;

/** HTML options */
static struct html_opts hopts = {
  0, /**< flag: do not display time stamps for files */
  "index.html",
  "index.css",
  0, /* "icon" file name */
  0, /* "shortcut icon" file name */
  "iso-8859-1", /* character set */
  PHOTOMOLO_VERSION ";\n "
  "http://www.iki.fi/~msmakela/software/photomolo/photomolo.html;\n"
  /* contents of the HTML META Generator element */
};

/** Length of hopts.html, plus 1 */
static unsigned indexnamelen;
/** Length of hopts.css, plus 1 */
static unsigned cssnamelen;

/** flag: generate HTML code? */
static int dump_html = 1;
/** flag: remove thumbnail directories */
static int remove_thumbdirs = 0;

/** number of command-line arguments */
static int main_argc;
/** command-line arguments */
static char** main_argv;

#ifdef HAVE_LINK
/** Path to linked image file tree, or NULL */
static const char* linkdir = 0;
/** strlen (linkdir), or 0 */
static unsigned linkdirlen = 0;
/** Flag: nonzero=generate symlinks to linkdir, zero=generate hard links */
static int linkdir_sym = 0;
#else
# define linkdirlen 0
#endif /* HAVE_LINK */

#ifdef HAVE_PROGRESS
/** progress indicator */
static FILE* progress;
/** number of image files processed */
static unsigned progresscnt;
/** number of image files to process */
static unsigned progresstotal;
#endif /* HAVE_PROGRESS */

#ifdef HAVE_CONVERTER
/** name of external JPEG converter program */
static const char* converter;
#endif /* HAVE_CONVERTER */

/** Cleanup function */
static void
cleanup (void)
{
  /* deallocate the thumbnail specifications */
  struct thumb_spec* spec = thumbs;
  while (spec) {
    thumbs = spec->next;
    free (spec);
    spec = thumbs;
  }
  /* deallocate the libjpeg contexts */
  jpeg_destroy_compress (&cout);
  jpeg_destroy_decompress (&cin);
#ifdef HAVE_PROGRESS
  /* close the progress indicator */
  if (progress)
    pclose (progress);
#endif /* HAVE_PROGRESS */
}

/** Create the thumbnail images
 * @param dir	the directory; dir->n_images may be updated
 * @param info	(in) the image information (for pruning non-JPEG files),
 *		(out) image information from converted non-JPEG files
 * @param comm	(out) image comment from converted non-JPEG files
 * @return	zero on success, nonzero on failure
 */
static int
make_thumbnails (struct dirtree* dir,
		 struct imageinfo* info,
		 struct comment** comm)
{
  unsigned i;
  char* fn = 0;
  unsigned fnlen = strlen (dir->name + linkdirlen) + 1;
#ifdef HAVE_CONVERTER
  char* jpgfn = 0;
#endif /* HAVE_CONVERTER */

  if (!(fn = malloc (fnlen + 1))) {
  memory:
    free (fn);
#ifdef HAVE_CONVERTER
    free (jpgfn);
#endif /* HAVE_CONVERTER */
    (void) fputs ("out of memory\n", stderr);
    return 1;
  }

  memcpy (fn, dir->name + linkdirlen, fnlen - 1);
  fn[fnlen - 1] = '/';

  for (i = 0; i < dir->n_files; i++) {
    const struct thumb_spec* spec;
    FILE* input;
    char* s;
    unsigned namelen;
    struct fileinfo* file = dir->files + i;
#ifdef HAVE_PROGRESS
    progresscnt++;
    if (progress && progresstotal) {
      if (0 > fprintf (progress, "%u\n",
		       100 * progresscnt / progresstotal) ||
	  fflush (progress)) {
	perror ("write (progress)");
	free (fn);
# ifdef HAVE_CONVERTER
	free (jpgfn);
# endif /* HAVE_CONVERTER */
	return 4;
      }
    }
#endif /* HAVE_PROGRESS */
    if (file->kind == FILE_OTHER)
      continue;
    namelen = strlen (file->name) + 1;
    if (!(s = realloc (fn, fnlen + namelen)))
      goto memory;
    fn = s;
    memcpy (fn + fnlen, file->name, namelen);
#ifdef HAVE_CONVERTER
    if (file->kind == FILE_IMAGE && converter) {
      /* Convert the file to JPEG.  First convert the file name. */
      unsigned basenamelen;
      int fd;
      const char suffix[] = ".jpg";
# if 0 /* Just append the suffix to better avoid collisions. */
      const char* end = strrchr (file->name, '.');
      if (end) {
	/* Try to replace the last dot-separated component of
	the file name with the .jpg suffix. */

	basenamelen = end - file->name;
	if (!basenamelen /* There is no dot-separated component in the name */
	    || !strcmp (end, suffix) /* The name already ends in .jpg */)
	  goto append_to_name;
      }
      else
      append_to_name:
# endif
	/* Append the suffix to the original file name. */
	basenamelen = namelen - 1; /* account for the terminating NUL */

      /* Generate the file name for the conversion */
      if (!(s = realloc (jpgfn, fnlen + basenamelen + sizeof suffix)))
	goto memory;
      jpgfn = s;
      memcpy (jpgfn, fn, fnlen + basenamelen);
      memcpy (jpgfn + fnlen + basenamelen, suffix, sizeof suffix);

      free (file->converted);
      if (!(file->converted = strdup (jpgfn + fnlen)))
	goto memory;

#if 0 /* This would not produce HTML thumbnails on subsequent runs. */
      /* Determine whether the thumbnail images have
      already been generated for this image, in order to
      avoid invoking the external converter. */
      {
	unsigned jpgnamelen = strlen (file->converted) + 1;
	char* tfn = malloc (fnlen);
	char* t;
	if (!tfn)
	  goto memory;
	memcpy (tfn, fn, fnlen);

	for (spec = thumbs; spec; spec = spec->next) {
	  unsigned speclen = strlen (spec->thumbdir) + 1;
	  if (!(t = realloc (tfn, fnlen + jpgnamelen + speclen))) {
	    free (tfn);
	    goto memory;
	  }
	  tfn = t;
	  memcpy (tfn + fnlen, spec->thumbdir, speclen);
	  tfn[fnlen + speclen - 1] = '/';
	  memcpy (tfn + fnlen + speclen, file->converted, jpgnamelen);
	  if (!exists_mtime (tfn, file->mtime)) {
	    free (t);
	    goto do_convert;
	  }
	}

	free (tfn);
      }
      /* No need to convert */
      continue;

    do_convert:
#endif
      /* Try creating the file exclusively */
      fd = open (jpgfn, O_WRONLY | O_CREAT | O_EXCL, 0666);
      if (fd < 0) {
	(void) fputs (jpgfn, stderr), (void) fflush (stderr);
	perror (": open (creating)");
	free (file->converted);
	file->converted = 0;
	continue;
      }
      if (close (fd)) {
	(void) fputs (jpgfn, stderr), (void) fflush (stderr);
	perror (": close (creating)");
	free (file->converted);
	file->converted = 0;
	continue;
      }
      if (set_environment ("IMG_IN", fn) ||
	  set_environment ("IMG_OUT", jpgfn) ||
	  system (converter)) {
	free (file->converted);
	file->converted = 0;
	file->kind = FILE_OTHER;
	goto remove_converted;
      }
    }
#endif /* HAVE_CONVERTER */
    if (!(input = fopen (s, "rb"))) {
      (void) fputs (s, stderr), (void) fflush (stderr);
      perror (": open");
      free (fn);
#ifdef HAVE_CONVERTER
      free (jpgfn);
#endif /* HAVE_CONVERTER */
      return 2;
    }
#ifdef HAVE_CONVERTER
    if (s == jpgfn) {
      if (set_environment ("IMG_IN", 0) ||
	  set_environment ("IMG_OUT", 0))
	goto close_input;
      s = (char*) readinfo (info + i, comm + i, input);
      if (s) {
	/* error while reading image info */
	(void) fputs (jpgfn + fnlen, stderr);
	(void) fputs (": ", stderr), (void) fputs (s, stderr);
	(void) putc ('\n', stderr);
	freeinfo (comm[i]); comm[i] = 0;
	/* clear the info structure for this file */
	memset (info + i, 0, sizeof *info);
	/* ignore this file */
	(void) fclose (input);
	free (file->converted);
	file->converted = 0;
	file->kind = FILE_OTHER;
	if (unlink (jpgfn))
	  goto remove_failed;
	continue;
      }
      rewind (input);
      namelen = strlen (file->converted) + 1;
      dir->n_images++;
    }
#endif /* HAVE_CONVERTER */
    for (spec = thumbs; spec; spec = spec->next) {
      unsigned speclen = strlen (spec->thumbdir) + 1;
      if (!(s = realloc (fn, fnlen + namelen + speclen))) {
	(void) fclose (input);
	goto memory;
      }
      fn = s;
      memcpy (fn + fnlen, spec->thumbdir, speclen);
      if (make_directory (fn)) {
      errexit:
	(void) fclose (input);
	free (fn);
#ifdef HAVE_CONVERTER
	free (jpgfn);
#endif /* HAVE_CONVERTER */
	return 3;
      }
      fn[fnlen + speclen - 1] = '/';
      memcpy (fn + fnlen + speclen, file->converted
	      ? file->converted
	      : file->name,
	      namelen);
      if (!exists_mtime (fn, file->mtime)) {
	FILE* output = fopen (fn, "wb");
	if (!output) {
	  (void) fputs (fn, stderr), (void) fflush (stderr);
	  perror (": open (writing)");
	  goto errexit;
	}
	rewind (input);
	jpeg_stdio_src (&cin, input);
	jpeg_stdio_dest (&cout, output);
	cout.image_width = spec->max_width;
	cout.image_height = spec->max_height;
	if (scale_image (&cin, &cout, spec->shrink, quality) ||
	    touch_mtime (fn, file->mtime)) {
	  (void) fclose (output);
	  goto errexit;
	}
	if (fclose (output)) {
	  (void) fputs (fn, stderr), (void) fflush (stderr);
	  perror (": close");
	  goto errexit;
	}
      }
    }
#ifdef HAVE_CONVERTER
  close_input:
    s = jpgfn && *jpgfn ? jpgfn : fn;
    if (fclose (input)) {
      (void) fputs (s, stderr), (void) fflush (stderr);
      perror (": close"); /* report but otherwise ignore the error */
    }
  remove_converted:
    if (jpgfn && *jpgfn && unlink (jpgfn)) {
    remove_failed:
      (void) fputs (jpgfn, stderr), (void) fflush (stderr);
      perror (": unlink");
      free (fn);
      free (jpgfn);
      return 3;
    }
    if (jpgfn)
      *jpgfn = 0;
#else /* HAVE_CONVERTER */
    if (fclose (input)) {
      memcpy (fn + fnlen, file->name, namelen);
      (void) fputs (fn, stderr), (void) fflush (stderr);
      perror (": close"); /* report but otherwise ignore the error */
    }
#endif /* HAVE_CONVERTER */
  }

  free (fn);
#ifdef HAVE_CONVERTER
  free (jpgfn);
#endif /* HAVE_CONVERTER */
  return 0;
}

/** Remove the thumbnail directories
 * @param tree	the directory
 * @return	zero on success, nonzero on failure
 */
static int
remove_thumbs (const struct dirtree* tree)
{
  int status = 0;
  const struct thumb_spec* spec;
  unsigned dirnamelen = strlen (tree->name + linkdirlen) + 1;

  char* filename = malloc (dirnamelen + 1);
  if (!filename) {
  memory:
    (void) fputs ("out of memory\n", stderr);
    return 1;
  }

  memcpy (filename, tree->name + linkdirlen, dirnamelen - 1);
  filename[dirnamelen - 1] = '/';

  for (spec = thumbs; spec; spec = spec->next) {
    unsigned speclen = strlen (spec->thumbdir) + 1;
    char* fn = realloc (filename, dirnamelen + speclen + indexnamelen + 1);
    if (!fn)
      goto memory;
    filename = fn;
    memcpy (filename + dirnamelen, spec->thumbdir, speclen);
    if (remove_directory (filename)) {
      status = 2;
      break;
    }
  }
  free (filename);
  return status;
}

/** Dump the HTML code
 * @param tree	the directory
 * @param level	number of '/' characters in the root directory name
 * @param info	image dimensions
 * @param comm	image comments
 * @param top	name of parent directory
 * @param prev	name of previous directory
 * @param next	name of next directory
 * @return	zero on success, nonzero on failure
 */
static int
dumphtmls (const struct dirtree* tree,
	   unsigned level,
	   const struct imageinfo* info,
	   const struct comment*const* comm,
	   const char* top,
	   const char* prev,
	   const char* next)
{
  int status = 0;
  const struct thumb_spec* spec;
  unsigned dirnamelen = strlen (tree->name + linkdirlen) + 1;

  char* filename = malloc (dirnamelen + 1);
  if (!filename) {
  memory:
    (void) fputs ("out of memory\n", stderr);
    return 1;
  }

  memcpy (filename, tree->name + linkdirlen, dirnamelen - 1);
  filename[dirnamelen - 1] = '/';

  for (spec = thumbs; spec; spec = spec->next) {
    unsigned speclen = strlen (spec->thumbdir) + 1;
    FILE* f;
    char* fn = realloc (filename, dirnamelen + speclen + indexnamelen + 1);
    if (!fn)
      goto memory;
    filename = fn;
    memcpy (filename + dirnamelen, spec->thumbdir, speclen);
    if (make_directory (filename))
      goto errexit;
    filename[dirnamelen + speclen - 1] = '/';
    memcpy (filename + dirnamelen + speclen, hopts.html, indexnamelen);
    if (!(f = fopen (filename, "wb"))) {
      (void) fputs (filename, stderr), (void) fflush (stderr);
      perror (": open");
    errexit:
      status = 2;
      break;
    }
    status = dumphtml (f, main_argc, main_argv,
		       tree, linkdirlen,
		       level, info, comm, &hopts, thumbs, spec,
		       top, prev, next);
    if (fclose (f) || status) {
      (void) fputs (filename, stderr), (void) fflush (stderr);
      perror (status ? ": write" : ": close");
      if (!status)
	status = -1;
      break;
    }
  }
  free (filename);
  return status;
}

/** Dump the image file directory listing
 * @param tree	the directory; n_images may be updated
 * @param level	number of '/' characters in the root directory name
 * @param top	name of parent directory
 * @param prev	name of previous directory
 * @param next	name of next directory
 * @return	zero on success, nonzero on failure
 */
static int
dumpfiles (struct dirtree* tree,
	   unsigned level,
	   const char* top,
	   const char* prev,
	   const char* next)
{
  /** exit status */
  int status = 0;
  /** counter */
  unsigned i;
  /** image file information */
  struct imageinfo* info;
  /** image file comments */
  struct comment** comment;
  /** directory name length */
  unsigned dirnamelen;
  /** total length of "../" components in dirnamelen */
  unsigned uplen = 0;
#ifdef HAVE_LINK
  /** path component leading to parent directory */
  static const char up[3] = "../";
#endif /* HAVE_LINK */
  /** path name for reading tree->files and creating links */
  char* filename;

  dirnamelen = strlen (tree->name);

#ifdef HAVE_LINK
  /* creating relative symlinks? */
  if (linkdirlen && linkdir_sym && *tree->name != '/') {
    /* then prefix the path with "../" components */
    const char* s;
    unsigned level = 1;
    for (s = tree->name + linkdirlen; *s; s++)
      if (*s == '/')
	level++;
    dirnamelen += uplen = level * sizeof up;
  }
#endif /* HAVE_LINK */

  filename = malloc (dirnamelen + 2);
  if (!filename) {
  memory:
    (void) fputs ("out of memory\n", stderr);
    return 1;
  }

#ifdef HAVE_LINK
  if (uplen) {
    /* Prefix the file name with "../" components */
    unsigned i;
    for (i = 0; i < uplen; i += sizeof up)
      memcpy (filename + i, up, sizeof up);
  }
#endif /* HAVE_LINK */

  memcpy (filename + uplen, tree->name, dirnamelen - uplen);

#ifdef HAVE_LINK
  if (linkdirlen) {
    /* Create the base directory in the output tree. */
    filename[dirnamelen] = 0;
    if (make_directory (filename + uplen + linkdirlen)) {
      free (filename);
      return 2;
    }
  }
#endif /* HAVE_LINK */

  filename[dirnamelen++] = '/';

  /* gather image info */
  if (!tree->n_files)
    info = 0, comment = 0;
  else if (!(info = calloc (tree->n_files, sizeof *info)) ||
	   !(comment = calloc (tree->n_files, sizeof *comment))) {
    if (info) free (info);
    free (filename);
    goto memory;
  }

  if (!remove_thumbdirs) {
    for (i = 0; i < tree->n_files; i++) {
      unsigned len;
      char* s;
      FILE* f;
      len = strlen (tree->files[i].name) + 1;
      if (!(s = realloc (filename, dirnamelen + len))) {
	(void) fputs ("out of memory\n", stderr);
	status = 1;
	break; /* out of memory */
      }
      filename = s;
      memcpy (filename + dirnamelen, tree->files[i].name, len);

#ifdef HAVE_LINK
      if (linkdirlen &&
	  create_link (filename, filename + uplen + linkdirlen, linkdir_sym))
	continue; /* ignore unlinkable files */
#endif /* HAVE_LINK */

      if (tree->files[i].kind != FILE_JPEG)
	continue; /* ignore non-JPEG files here */

      if (!(f = fopen (filename + uplen + linkdirlen, "rb"))) {
	(void) fputs (filename + uplen + linkdirlen, stderr);
	(void) fflush (stderr);
	perror (": fopen");
	continue; /* ignore unreadable files */
      }
      s = (char*) readinfo (info + i, comment + i, f);
      (void) fclose (f);
      if (s) {
	/* ignore errors while reading image info */
	(void) fputs (filename + uplen + linkdirlen, stderr);
	(void) fputs (": ", stderr), (void) fputs (s, stderr);
	(void) putc ('\n', stderr);
	freeinfo (comment[i]); comment[i] = 0;
	/* clear the info structure for this file */
	memset (info + i, 0, sizeof *info);
      }
    }
  }
  free (filename);

  if (remove_thumbdirs) {
    if (!status)
      status = remove_thumbs (tree);
  }
  else {
    /* generate the thumbnails */
    if (!status)
      status = make_thumbnails (tree, info, comment);
    if (!status && dump_html)
      status = dumphtmls (tree, level,
			  info, (const struct comment**) comment,
			  top, prev, next);
  }

  /* deallocate the image comments */
  for (i = tree->n_files; i--; )
    freeinfo (comment[i]);
  if (tree->n_files)
    free (info), free (comment);
  return status;
}

/** Dump the image file directory tree
 * @param tree	the directory tree; n_images and n_images_dirs may be updated
 * @param level	number of '/' characters in the root directory name
 * @param top	name of parent directory
 * @param prev	name of previous directory
 * @param next	name of next directory
 * @return	zero on success, nonzero on failure
 */
static int
dumptree (struct dirtree* tree,
	  unsigned level,
	  const char* top,
	  const char* prev,
	  const char* next)
{
  unsigned i;
  for (i = 0; i < tree->n_dirs; i++) {
#ifdef HAVE_LINK
    if (linkdirlen && make_directory (tree->name + linkdirlen))
      return 2;
#endif /* HAVE_LINK */
    if (dumptree (tree->dirs + i,
		  level,
		  tree->name,
		  i > 0 ? tree->dirs[i - 1].name : 0,
		  i < tree->n_dirs - 1 ? tree->dirs[i + 1].name : 0))
      return 1;
    tree->n_images_dirs +=
      tree->dirs[i].n_images_dirs + tree->dirs[i].n_images;
  }
  return dumpfiles (tree, level, top, prev, next);
}

/** predicate for pruning directories by name
 * @param name	directory name
 * @return	0 if the directory should be considered, nonzero otherwise
 */
static int
prune_dirs (const char* name)
{
  struct thumb_spec* spec;
  for (spec = thumbs; spec; spec = spec->next)
    if (!strcmp (name, spec->thumbdir))
      return 1;
  return 0;
}

/** identify files
 * @param name	file name
 * @return	type of the file
 */
static enum file_type
identify_images (const char* name)
{
  FILE* f;
  if (!(f = fopen (name, "rb"))) {
    (void) fputs (name, stderr), (void) fflush (stderr);
    perror (": open (reading)");
    return FILE_OTHER;
  }
  else {
    int is_jpeg =
      getc (f) == 0xff &&
      getc (f) == 0xd8;
    int is_eof = feof (f);
    (void) fclose (f);
    if (is_jpeg)
      return FILE_JPEG;
    if (is_eof)
      return FILE_OTHER;
  }

#ifdef HAVE_CONVERTER
  /* Assume that all files can be converted into an image. */
  if (converter)
    return FILE_IMAGE;
#endif /* HAVE_CONVERTER */
  return FILE_OTHER;
}

/** create the CSS file if it does not exist
 * @param path	the path name to the directory
 * @return	0 if successful
 */
static int
create_css (const char* path)
{
  /** first part of CSS definitions */
  static const char css1[] =
    "/*\n"
    " * Sample style sheet for pHoToMoLo-generated image gallery HTML\n"
    " * Written in 2003,2006 by Marko Makela (marko.makela (at) iki.fi).\n"
    " * This file is in the public domain.\n"
    " * The layout has been tested with the following browsers:\n"
    " * Lynx, Opera 6, Opera 8.5, IE 5.01, IE 5.5, IE 6.0, Mozilla 1.x.\n"
    " * This layout does not work well on Netscape 4.\n"
    " */\n"
    "\n"
    "a:link,a:visited,a:active,a:hover{color:red;background:black;}\n"
    "a:link,a:visited{text-decoration:none;}\n"
    "a:hover,a:active{text-decoration:underline;}\n"
    "\n";
  /** second part of CSS definitions */
  static const char css2[] =
    "body{color:white;background:black;"
    "font-family:verdana,helvetica,sans-serif;}\n"
    "body,h1,p,table,tr,td,dl,img{border:0;margin:0;}\n"
    "p.img{line-height:0;}\n"
    "\n"
    "h1{text-align:left;font-weight:bold;font-size:150%;}\n"
    "\n"
    "dd{margin:0 0 0 1em;}\n"
    "dl{font-weight:bold;font-style:normal;}\n"
    "dl dl{font-weight:normal;}\n"
    "dl dl dl{font-style:oblique;}\n"
    "dl dl dl dl{font-size:smaller;}\n";
  FILE* f;
  unsigned pathlen;
  char* s;
  if (*hopts.css == '/' || strstr (hopts.css, "://"))
    return 0; /* absolute location */
  pathlen = strlen (path);
  if (!(s = malloc (pathlen + 1 + cssnamelen))) {
    (void) fputs (path, stderr);
    (void) fputs (": out of memory\n", stderr);
    return 1;
  }
  memcpy (s, path, pathlen);
  s[pathlen++] = '/';
  memcpy (s + pathlen, hopts.css, cssnamelen);
  if ((f = fopen (s, "rt")))
    goto done;
  if (!(f = fopen (s, "wt"))) {
    (void) fputs (s, stderr), (void) fflush (stderr);
    perror (": open (writing)");
    free (s);
    return 2;
  }
  if ((sizeof css1) - 1 != fwrite (css1, 1, (sizeof css1) - 1, f) ||
      (sizeof css2) - 1 != fwrite (css2, 1, (sizeof css2) - 1, f)) {
    (void) fputs (s, stderr), (void) fflush (stderr);
    perror (": write");
    free (s);
    (void) fclose (f);
    return 2;
  }
 done:
  free (s);
  (void) fclose (f);
  return 0;
}

/** The main function
 * @param argc	argument count
 * @param argv	argument vector
 * @return	0 if successful
 */
int
main (int argc, char** argv)
{
  jpeg_create_decompress (&cin);
  jpeg_create_compress (&cout);

  (void) atexit (cleanup);
  (void) setlocale (LC_ALL, "");

  if (argc == 1) {
  Usage:
    (void) fputs
      (PHOTOMOLO_VERSION "\n"
       "Create thumbnails and HTML navigation\n"
       "Usage: photomolo [options] directory1 directory2 ...\n"
       "Options:\n"
       "-h"
       "\tusage\n"
       "-T <dir>/<spec>\n"
       "\tgenerate thumbnail directory <dir> according to <spec>\n"
       "-H htmlfile\n"
       "\tname of HTML files (default=\"index.html\")\n"
       "-C cssfile\n"
       "\tname of CSS files (default=\"index.css\")\n"
       "-c charset\n"
       "\tHTML and JPEG comment character set (default=\"iso-8859-1\")\n"
       "-i iconfile\n"
       "-I file.ico\n"
       "\tname of icon files (default=none)\n",
       stderr);
    (void) fputs
      ("-t"
       "\tgenerate thumbnails only (no HTML or CSS)\n"
       "-r"
       "\tremove the thumbnail directories listed with -T\n"
       "-q quality\n"
       "\tJPEG compression quality (default=75)\n"
       "-d"
       "\tomit timestamps from the HTML\n"
#ifdef HAVE_LINK
       "-l ../path\n"
       "\tgenerate symlinks to the image files\n"
       "-L ../path\n"
       "\tgenerate hard links to the image files\n"
#endif /* HAVE_LINK */
#ifdef HAVE_PROGRESS
       "-P command\n"
       "\toutput progress indication to an external command\n"
#endif /* HAVE_PROGRESS */
#ifdef HAVE_CONVERTER
       "-J command\n"
       "\tconvert non-JPEG files to JPEG with an external command\n"
#endif /* HAVE_CONVERTER */
       "--\tdisable processing further options\n",
       stderr);
    return 0;
  }

  main_argc = argc;
  main_argv = argv;

  /* process the options */
  for (argv++; argc > 1 && **argv == '-'; argv++, argc--) {
    const char* opts = *argv;
    if (!strcmp (*argv, "--")) { /* disable processing further options */
      argv++;
      argc--;
      break;
    }

    while (*++opts) { /* process all flags */
      switch (*opts) {
      default:
	(void) putc ('-', stderr);
	(void) putc (*opts, stderr);
	(void) fputs (": unknown option (use -h for help)\n", stderr);
	return 1;
      case '?':
      case 'h':
	goto Usage;
      case 'T':
	if (argc <= 2) {
	missing:
	  (void) fprintf (stderr, "-%c: missing argument (use -h for help)\n",
			  *opts);
	  return 1;
	}
	else {
	  struct thumb_spec* spec = parse_spec (*++argv);
	  argc--;
	  if (!spec)
	    return 1;
	  if (!thumbs)
	    thumbs = spec;
	  else {
	    struct thumb_spec* last = thumbs;
	    while (last->next)
	      last = last->next;
	    last->next = spec;
	  }
	}
	break;
      case 'H':
	if (argc <= 2)
	  goto missing;
	argc--;
	if (!*(hopts.html = *++argv)) {
	  (void) fputs ("-H: argument must be nonempty (use -h for help)\n",
			stderr);
	  return 1;
	}
	else if (strchr (hopts.html, '/')) {
	  (void) fputs ("-H ", stderr);
	  (void) fputs (hopts.html, stderr);
	  (void) fputs (": name must not contain '/'", stderr);
	  return 1;
	}
	break;
      case 'C':
	if (argc <= 2)
	  goto missing;
	argc--;
	if (!*(hopts.css = *++argv)) {
	  (void) fputs ("-C: argument must be nonempty (use -h for help)\n",
			stderr);
	  return 1;
	}
	break;
      case 'c':
	if (argc <= 2)
	  goto missing;
	argc--;
	hopts.charset = *++argv; /* no sanity check */
	break;
      case 'i':
	if (argc <= 2)
	  goto missing;
	argc--;
	if (!*(hopts.icon = *++argv))
	  hopts.icon = 0;
	break;
      case 'I':
	if (argc <= 2)
	  goto missing;
	argc--;
	if (!*(hopts.msico = *++argv))
	  hopts.msico = 0;
	break;
      case 't':
	dump_html = 0;
	break;
      case 'r':
	remove_thumbdirs = 1;
	break;
      case 'd':
	hopts.notim = 1;
	break;
      case 'q':
	if (argc <= 2)
	  goto missing;
	else {
	  char* endp;
	  unsigned long q = strtoul (*++argv, &endp, 0);
	  argc--;
	  if (q < 1L || q > 100L || *endp || !*argv) {
	    (void) fputs ("-q: quality must be between 1 and 100\n", stderr);
	    return 1;
	  }
	  quality = q;
	}
	break;
#ifdef HAVE_LINK
      case 'l':
      case 'L':
	if (argc <= 2)
	  goto missing;
	argc--;
	linkdir = *++argv;
	linkdirlen = 0;
	if (!*linkdir)
	  linkdir = 0;
	if (linkdir) {
	  linkdirlen = strlen (linkdir);
	  /* strip trailing slashes */
	  while (linkdirlen && linkdir[linkdirlen - 1] == '/')
	    linkdirlen--;
	  linkdirlen++;
	}
	linkdir_sym = *opts == 'l';
	break;
#endif /* HAVE_LINK */
#ifdef HAVE_PROGRESS
      case 'P':
	if (argc <= 2)
	  goto missing;
	if (progress)
	  pclose (progress);
	argc--;
	fflush (stderr);
	fflush (stdout);
	if (*++argv) {
	  progress = popen (*argv, "w");
	  if (!progress) {
	    fprintf (stderr, "-P %s", *argv);
	    perror (": popen");
	  }
	}
	else
	  progress = 0;
	break;
#endif /* HAVE_PROGRESS */
#ifdef HAVE_CONVERTER
      case 'J':
	if (argc <= 2)
	  goto missing;
	argc--;
	converter = *++argv;
	if (!*converter)
	  converter = 0;
	break;
#endif /* HAVE_CONVERTER */
      }
    }
  }

  if (!thumbs) {
    (void) fputs ("expected at least one -T parameter (use -h for help)\n",
		  stderr);
    return 1;
  }

  indexnamelen = strlen (hopts.html) + 1;
  cssnamelen = strlen (hopts.css) + 1;
  for (; argc > 1; argv++, argc--) {
    struct dirtree dirs;
    /* number of slashes in the specified directory */
    unsigned num_slash;
    if (!*argv)
      continue; /* ignore empty directory names */
#ifdef HAVE_LINK
    if (linkdir && make_directory (*argv))
      return 2;
#endif /* HAVE_LINK */
    if (dump_html && !remove_thumbdirs && create_css (*argv))
      return 2;
    else {
      char* c;
      for (num_slash = 0, c = *argv; *c; c++)
#if defined WIN32 || defined __WIN32
	if (*c == '\\') *c = '/', num_slash++; else
#endif
	if (*c == '/')
	  num_slash++;
      /* discount trailing slashes */
      while (c-- > *argv && *c == '/')
	num_slash--;
      c[1] = 0;
    }

#ifdef HAVE_LINK
    if (linkdir) {
      char* path;
      size_t namelen = strlen (*argv);

      if (!(path = malloc (linkdirlen + namelen + 1))) {
	(void) fputs (*argv, stderr);
	(void) fputs (": out of memory\n", stderr);
	return 2;
      }

      memcpy (path, linkdir, linkdirlen - 1);
      path[linkdirlen - 1] = '/';
      memcpy (path + linkdirlen, *argv, namelen + 1);

      inittree (&dirs, path);
      free (path);
    }
    else
#endif /* HAVE_LINK */
      inittree (&dirs, *argv);
    if (readtree (&dirs, prune_dirs, identify_images)) {
    tree_error:
      freetree (&dirs);
      return 2;
    }
#ifdef HAVE_PROGRESS
    progresstotal += dirs.n_files + dirs.n_files_dirs;
#endif /* HAVE_PROGRESS */
    if (dumptree (&dirs, num_slash, 0, 0, 0))
      goto tree_error;
    freetree (&dirs);
  }

  return 0;
}
