/**
 * @file jpegnail.c
 * Remove or embed thumbnails in Exif or JFIF JPEG images
 * @author Marko Mkel <marko.makela@iki.fi>
 */

/* Copyright  2003,2004,2005 Marko Mkel.

   This file is part of JPEGNAIL, a program for removing or embedding
   JPEG thumbnails in images in Exif or JFIF format.

   JPEGNAIL is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   JPEGNAIL is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

#if defined WIN32 || defined __WIN32
# undef HAVE_PROGRESS
# undef __STRICT_ANSI__
# include <sys/types.h>
# include <sys/utime.h>
#else
# define HAVE_PROGRESS
# ifndef _POSIX_SOURCE
#  define _POSIX_SOURCE
# endif
# ifndef _POSIX_C_SOURCE
#  define _POSIX_C_SOURCE 200112L /* for popen(3) */
# endif
# include <sys/types.h>
# include <utime.h>
#endif
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>
#include "scale.h"
#include "rename.h"
#include "jerror.h"

/** Copy the timestamp of a file to another
 * @param name	the file name whose timestamp is to be rewritten
 * @param ref	the reference file
 * @return	zero if successful
 */
static int
touch_file (const char* name,
	    const char* ref)
{
  int status;
  struct stat statbuf;
  if ((status = stat (ref, &statbuf))) {
    (void) fputs (ref, stderr), (void) fflush (stderr);
    perror (": stat");
  }
  else {
    struct utimbuf utim;
    utim.actime = utim.modtime = statbuf.st_mtime;
    if ((status = utime (name, &utim))) {
      (void) fputs (ref, stderr), (void) fflush (stderr);
      perror (": utime");
    }
  }
  return status;
}

/** libjpeg error manager */
struct exif_error_mgr
{
  /** "public" fields */
  struct jpeg_error_mgr pub;
  /** for returning to caller */
  jmp_buf setjmp_buffer;
};

/**
 * Custom error exit handler
 * @param cinfo		the libjpeg context
 */
static void
exif_error_exit (struct jpeg_common_struct* cinfo)
{
  longjmp (((struct exif_error_mgr*) cinfo->err)->setjmp_buffer, 1);
}

/** current input file name */
static const char* srcname;

/**
 * Custom message output handler
 * @param cinfo		the libjpeg context
 */
static void
exif_output_message (struct jpeg_common_struct* cinfo)
{
  /** error message buffer for libjpeg */
  static char jpeg_msg[JMSG_LENGTH_MAX];
  cinfo->err->format_message (cinfo, jpeg_msg);
  (void) fputs (srcname, stderr);
  (void) fputs (": ", stderr);
  (void) fputs (jpeg_msg, stderr);
  (void) putc ('\n', stderr);
}

/** error messages from the Exif parser */
static const char* exif_message_table[] = {
  0,
  "thumbnail image is too large (try smaller -q, -x or -y)",
};

/** error codes from the Exif parser */
enum exif_message {
  EXIF_FIRSTCODE = 1000,	/**< first message code */
  THUMBNAIL_WRITE,		/**< transformed thumbnail too large */
  EXIF_LASTCODE
};

/** dummy function for image compression into memory buffer
 * @param cinfo	the compression context
 */
static void
thumb_out_dummy (struct jpeg_compress_struct* cinfo
#ifdef __GNUC__
		 __attribute__ ((unused))
#endif /* __GNUC__ */
		 )
{
}

/** Signal that the thumbnail compression buffer is full
 * @param cinfo	the compression context
 * @exception	THUMBNAIL_WRITE
 * @return	FALSE, to signal that the data cannot be accepted
 */
static boolean
thumb_out_empty (struct jpeg_compress_struct* cinfo)
{
  ERREXIT (cinfo, THUMBNAIL_WRITE);
  return FALSE; /* give up */
}

/** thumbnail compression buffer */
static JOCTET thumb_buf[16384];
/** thumbnail length */
static size_t thumb_length;
/** maximum thumbnail length */
static size_t thumb_maxlen;
/** maximum thumbnail image width */
static unsigned thumb_width;
/** maximum thumbnail image height */
static unsigned thumb_height;
/** thumbnail image quality */
static unsigned thumb_quality = 75;

/** transform a thumbnail image in memory
 * @param cinfo	image decompression context
 */
static void
thumb_create (struct jpeg_decompress_struct* cinfo)
{
  /** libjpeg compression context */
  struct jpeg_compress_struct dstinfo;
  /** libjpeg error handler */
  struct exif_error_mgr jerr;

  dstinfo.err = jpeg_std_error (&jerr.pub);
  jpeg_create_compress (&dstinfo);

  jerr.pub.error_exit = exif_error_exit;
  jerr.pub.output_message = exif_output_message;
  jerr.pub.addon_message_table = exif_message_table;
  jerr.pub.first_addon_message = EXIF_FIRSTCODE;
  jerr.pub.last_addon_message = EXIF_LASTCODE;

  if (setjmp (jerr.setjmp_buffer)) {
    exif_output_message ((j_common_ptr) &dstinfo);
    jpeg_abort_compress (&dstinfo);
    jpeg_destroy_compress (&dstinfo);
    thumb_length = 0;
    return;
  }

  jpeg_stdio_dest (&dstinfo, 0);
  dstinfo.dest->init_destination =
    dstinfo.dest->term_destination = thumb_out_dummy;
  dstinfo.dest->empty_output_buffer = thumb_out_empty;
  dstinfo.dest->next_output_byte = thumb_buf;
  dstinfo.dest->free_in_buffer = thumb_maxlen;

  dstinfo.image_width = thumb_width;
  dstinfo.image_height = thumb_height;
  scale_prepare (cinfo, &dstinfo, 0);
  dstinfo.optimize_coding = TRUE;
  dstinfo.write_JFIF_header = FALSE;
  jpeg_set_quality (&dstinfo, thumb_quality, FALSE);
  scale_jpeg (cinfo, &dstinfo);

  (void) jpeg_finish_compress (&dstinfo);
  thumb_length = thumb_maxlen - dstinfo.dest->free_in_buffer;
  jpeg_destroy_compress (&dstinfo);
}

/** Byte order of the Exif data */
enum endianness {
  LSB = 0,	/**< least significant byte first (Intel) */
  MSB = 1,	/**< most significant byte first (Motorola) */
  UNKNOWN	/**< unknown (no Exif data read) */
};

/** The byte order of the last Exif block read */
static enum endianness exif_endian;

/** Append a 2-octet quantity to the Exif marker
 * @param cinfo	image compression context
 * @param data	the data
 */
static void
exif_write2 (struct jpeg_compress_struct* cinfo,
	     unsigned data)
{
  if (exif_endian)
    jpeg_write_m_byte (cinfo, data >> 8), jpeg_write_m_byte (cinfo, data);
  else
    jpeg_write_m_byte (cinfo, data), jpeg_write_m_byte (cinfo, data >> 8);
}

/** Append a 4-octet quantity to the Exif marker
 * @param cinfo	image compression context
 * @param data	the data
 */
static void
exif_write4 (struct jpeg_compress_struct* cinfo,
	     unsigned data)
{
  /* The high-order word is zero, as the length
   * of the Exif marker is limited to 65533.
   */
  if (exif_endian)
    jpeg_write_m_byte (cinfo, 0), jpeg_write_m_byte (cinfo, 0),
      jpeg_write_m_byte (cinfo, data >> 8), jpeg_write_m_byte (cinfo, data);
  else
    jpeg_write_m_byte (cinfo, data), jpeg_write_m_byte (cinfo, data >> 8),
      jpeg_write_m_byte (cinfo, 0), jpeg_write_m_byte (cinfo, 0);
}

/** Append an unsigned short data item to the Exif marker
 * @param cinfo	image compression context
 * @param data	the data
 */
static void
exif_write_ushort (struct jpeg_compress_struct* cinfo,
		   unsigned data)
{
  exif_write2 (cinfo, 3); /* data type: unsigned short */
  exif_write4 (cinfo, 1); /* number of components */
  exif_write2 (cinfo, data);
  exif_write2 (cinfo, 0); /* padding */
}

/** Append an unsigned long data item to the Exif marker
 * @param cinfo	image compression context
 * @param data	the data
 */
static void
exif_write_ulong (struct jpeg_compress_struct* cinfo,
		  unsigned data)
{
  exif_write2 (cinfo, 4); /* data type: unsigned short */
  exif_write4 (cinfo, 1); /* number of components */
  exif_write4 (cinfo, data);
}

/** Copy markers saved in the given source object to the destination object.
 * Copy the thumbnail in a JFXX or Exif marker.  Discard existing thumbnails
 * in JFIF or JFXX markers.
 * @param srcinfo		source image information
 * @param dstinfo		destination image information
 */
static void
copy_markers (struct jpeg_decompress_struct* srcinfo,
	      struct jpeg_compress_struct* dstinfo)
{
  jpeg_saved_marker_ptr marker;
  unsigned i;

  /* To avoid confusion, we do not output Adobe APP14 markers
   * if the encoder library already wrote one.
   */
  for (marker = srcinfo->marker_list; marker != NULL; marker = marker->next) {
    if (marker->marker == (UINT8) JPEG_APP0 &&
	marker->data_length >= 5 &&
	GETJOCTET (marker->data[0]) == (JOCTET) 0x4A &&
	GETJOCTET (marker->data[1]) == (JOCTET) 0x46 &&
	((GETJOCTET (marker->data[2]) == (JOCTET) 0x49 &&
	  GETJOCTET (marker->data[3]) == (JOCTET) 0x46) ||
	 (GETJOCTET (marker->data[2]) == (JOCTET) 0x58 &&
	  GETJOCTET (marker->data[3]) == (JOCTET) 0x58)) &&
	GETJOCTET (marker->data[4]) == (JOCTET) 0)
      continue; /* reject JFIF and JFXX markers */
    if (dstinfo->write_Adobe_marker &&
	marker->marker == (UINT8) (JPEG_APP0 + 14) &&
	marker->data_length >= 5 &&
	GETJOCTET (marker->data[0]) == (JOCTET) 0x41 &&
	GETJOCTET (marker->data[1]) == (JOCTET) 0x64 &&
	GETJOCTET (marker->data[2]) == (JOCTET) 0x6F &&
	GETJOCTET (marker->data[3]) == (JOCTET) 0x62 &&
	GETJOCTET (marker->data[4]) == (JOCTET) 0x65)
      continue; /* reject duplicate Adobe */
    if (thumb_length && exif_endian != UNKNOWN &&
	marker->marker == (UINT8) (JPEG_APP0 + 1) &&
	marker->data_length >= 8 &&
	GETJOCTET (marker->data[0]) == (JOCTET) 0x45 &&
	GETJOCTET (marker->data[1]) == (JOCTET) 0x78 &&
	GETJOCTET (marker->data[2]) == (JOCTET) 0x69 &&
	GETJOCTET (marker->data[3]) == (JOCTET) 0x66 &&
	GETJOCTET (marker->data[4]) == (JOCTET) 0x00 &&
	GETJOCTET (marker->data[5]) == (JOCTET) 0x00) {
      /* writing Exif marker with embedded thumbnail */
      jpeg_write_m_header (dstinfo, marker->marker, marker->data_length +
			   94 + thumb_length);
      for (i = 0; i < marker->data_length; i++)
	jpeg_write_m_byte (dstinfo, marker->data[i]);
      /* write Exif IFD1 */
      exif_write2 (dstinfo, 6); /* number of entries */
      exif_write2 (dstinfo, 0x103); /* tag 0x103: Compression */
      exif_write_ushort (dstinfo, 6); /* JPEG compression */
      exif_write2 (dstinfo, 0x11a); /* tag 0x11a: XResolution */
      exif_write2 (dstinfo, 5); /* data type: unsigned rational */
      exif_write4 (dstinfo, 1); /* number of components */
      exif_write4 (dstinfo, marker->data_length + (94 - 6 - 16)); /* pointer */
      exif_write2 (dstinfo, 0x11b); /* tag 0x11b: YResolution */
      exif_write2 (dstinfo, 5); /* data type: unsigned rational */
      exif_write4 (dstinfo, 1); /* number of components */
      exif_write4 (dstinfo, marker->data_length + (94 - 6 - 8)); /* pointer */
      exif_write2 (dstinfo, 0x128); /* tag 0x128: ResolutionUnit */
      exif_write_ushort (dstinfo, 2); /* inch */
      exif_write2 (dstinfo, 0x201); /* tag 0x201: JpegIFOffset */
      exif_write_ulong (dstinfo, marker->data_length + (94 - 6));
      exif_write2 (dstinfo, 0x202); /* tag 0x202: JpegIFByteCount */
      exif_write_ulong (dstinfo, thumb_length);
      /* null link to IFD2 */
      exif_write4 (dstinfo, 0);
      /* XResolution: 72/1 dots per inch */
      exif_write4 (dstinfo, 72), exif_write4 (dstinfo, 1);
      /* YResolution: 72/1 dots per inch */
      exif_write4 (dstinfo, 72), exif_write4 (dstinfo, 1);
      for (i = 0; i < thumb_length; i++)
	jpeg_write_m_byte (dstinfo, thumb_buf[i]);
      continue;
    }
#ifdef NEED_FAR_POINTERS
    /* We could use jpeg_write_marker if the data weren't FAR... */
    jpeg_write_m_header (dstinfo, marker->marker, marker->data_length);
    for (i = 0; i < marker->data_length; i++)
      jpeg_write_m_byte (dstinfo, marker->data[i]);
#else
    jpeg_write_marker (dstinfo, marker->marker,
		       marker->data, marker->data_length);
#endif
  }
}

/** Read a 2-octet quantity from the Exif record
 * @param data		offset to the Exif data
 * @return		the quantity read
 */
static unsigned
exif_read2 (JOCTET FAR* data)
{
  return exif_endian
    ? (unsigned) GETJOCTET (data[0]) << 8 | GETJOCTET (data[1])
    : (unsigned) GETJOCTET (data[0]) | GETJOCTET (data[1]) << 8;
}

/** Read a 4-octet quantity from the Exif record
 * @param data		offset to the Exif data
 * @return		the quantity read
 */
static unsigned long
exif_read4 (JOCTET FAR* data)
{
  return exif_endian
    ? ((unsigned long) GETJOCTET (data[0])) << 24 |
    ((unsigned long) GETJOCTET (data[1])) << 16 |
    ((unsigned long) GETJOCTET (data[2])) << 8 |
    ((unsigned long) GETJOCTET (data[3]))
    : ((unsigned long) GETJOCTET (data[0])) |
    ((unsigned long) GETJOCTET (data[1])) << 8 |
    ((unsigned long) GETJOCTET (data[2])) << 16 |
    ((unsigned long) GETJOCTET (data[3])) << 24;
}

/** Parse the Exif information
 * @param cinfo		decompression structure
 * @param rmifd1	flag: remove IFD1 (containing the thumbnail image)
 */
static void
exif_parse (struct jpeg_decompress_struct* cinfo,
	    int rmifd1)
{
  jpeg_saved_marker_ptr marker;
  exif_endian = UNKNOWN;
  thumb_maxlen = (sizeof thumb_buf) / (sizeof *thumb_buf);

  for (marker = cinfo->marker_list; marker; marker = marker->next) {
    /** current image file directory (IFD) offset */
    unsigned long offset;

    if (marker->marker != (UINT8) (JPEG_APP0 + 1) ||
	marker->data_length < 14)
      continue; /* Exif data is in an APP1 marker of at least 14 octets */

    if (GETJOCTET (marker->data[0]) != (JOCTET) 0x45 ||
	GETJOCTET (marker->data[1]) != (JOCTET) 0x78 ||
	GETJOCTET (marker->data[2]) != (JOCTET) 0x69 ||
	GETJOCTET (marker->data[3]) != (JOCTET) 0x66 ||
	GETJOCTET (marker->data[4]) != (JOCTET) 0x00 ||
	GETJOCTET (marker->data[5]) != (JOCTET) 0x00)
      continue; /* no Exif header */

    if (GETJOCTET (marker->data[6]) == (JOCTET) 0x4d &&
	GETJOCTET (marker->data[7]) == (JOCTET) 0x4d &&
	GETJOCTET (marker->data[8]) == (JOCTET) 0x00 &&
	GETJOCTET (marker->data[9]) == (JOCTET) 0x2a)
      exif_endian = MSB;
    else if (GETJOCTET (marker->data[6]) == (JOCTET) 0x49 &&
	     GETJOCTET (marker->data[7]) == (JOCTET) 0x49 &&
	     GETJOCTET (marker->data[8]) == (JOCTET) 0x2a &&
	     GETJOCTET (marker->data[9]) == (JOCTET) 0x00)
      exif_endian = LSB;
    else {
      fprintf (stderr, "%s: ignoring Exif marker with byte order "
	       "0x%02x%02x%02x%02x\n",
	       srcname,
	       (unsigned char) GETJOCTET (marker->data[6]),
	       (unsigned char) GETJOCTET (marker->data[7]),
	       (unsigned char) GETJOCTET (marker->data[8]),
	       (unsigned char) GETJOCTET (marker->data[9]));
      continue;
    }

    /* Get first IFD offset (offset to IFD0) */
    offset = exif_read4 (&marker->data[10]);
    if (offset > (unsigned long) marker->data_length - (10 + 2)) {
      fprintf (stderr, "%s: Exif IFD0 offset out of bounds: %lu>%lu\n",
	       srcname,
	       offset, (unsigned long) marker->data_length - (10 + 2));
    errexit:
      fprintf (stderr, "%s: leaving Exif block intact due to errors\n",
	       srcname);
      exif_endian = UNKNOWN;
      thumb_length = 0;
      return;
    }
    else {
      /** pointer to the start of the Exif data */
      JOCTET FAR* exif_data = &marker->data[6];
      /** offset to IFD1 */
      unsigned long ifd1_offset;
      /** length of Exif data */
      unsigned exif_length = marker->data_length - 6;

      /* Get the number of IFD0 entries */
      offset += 2 + 12 * exif_read2 (&exif_data[offset]);
      if (offset > exif_length - 4) {
	fprintf (stderr, "%s: Exif IFD0 ends prematurely\n", srcname);
	goto errexit;
      }
      ifd1_offset = exif_read4 (&exif_data[offset]);
      if (ifd1_offset) {
	if (ifd1_offset < offset + 4) {
	  fprintf (stderr, "%s: Exif IFD1 offset out of bounds: %lu<%lu\n",
		   srcname,
		   ifd1_offset, offset + 4);
	  goto errexit;
	}
	else if (ifd1_offset > exif_length - 2) {
	  fprintf (stderr, "%s: Exif IFD1 offset out of bounds: %lu>%lu\n",
		   srcname,
		   ifd1_offset, (unsigned long) exif_length - 2);
	  goto errexit;
	}
      }
      if (ifd1_offset) {
	/* remove the existing IFD1 */
	exif_length = ifd1_offset;
	marker->data_length = exif_length + 6;
	if (rmifd1)
	  exif_data[offset] =
	    exif_data[offset + 1] =
	    exif_data[offset + 2] =
	    exif_data[offset + 3] = 0;
      }
      else if (!rmifd1) {
	/* make the IFD1 pointer point to the end of the Exif block */
	switch (exif_endian) {
	case LSB:
	  exif_data[offset] = exif_length;
	  exif_data[offset + 1] = exif_length >> 8;
	  exif_data[offset + 2] = exif_data[offset + 3] = 0;
	  break;
	case MSB:
	  exif_data[offset] = exif_data[offset + 1] = 0;
	  exif_data[offset + 2] = exif_length >> 8;
	  exif_data[offset + 3] = exif_length;
	  break;
	case UNKNOWN:
	  break;
	}
      }
      if (thumb_maxlen > (65533 - 94) - marker->data_length)
	thumb_maxlen = (65533 - 94) - marker->data_length;
      break;
    }
  }
}

/**
 * Construct a file name
 * @param base		the base file name
 * @param baselen	length of the base file name
 * @param dir		the directory name
 * @param dirlen	length of the directory name
 * @return		the constructed file name, or 0 on error
 */
static char*
createfilename (const char* base,
		unsigned baselen,
		const char* dir,
		unsigned dirlen)
{
  const char* s;
  char* fnam;
  for (s = base + baselen; s > base; s--)
    if (*s == '/'
#if defined WIN32 || defined __WIN32
	|| *s == '\\' || *s == ':'
#endif
	) { s++; break; }
  baselen -= s - base;
  fnam = malloc (dirlen + baselen + 2);
  if (fnam) {
    memcpy (fnam, dir, dirlen);
    fnam[dirlen] = '/';
    memcpy (fnam + dirlen + 1, s, baselen + 1);
  }
  return fnam;
}

/** Transform the image
 * @param srcinfo	image decompression context
 * @param dstinfo	image compression context
 * @param thumbname_	name of thumbnail file to embed
 *			(NULL = compute thumbnail)
 * @param dstname_	destination image file name (NULL = stdin/stdout)
 * @param progressive	flag: use progressive coding
 * @param optimize	flag: compute optimal Huffman coding tables
 * @return		0=ok, 1=read error, 2=write error
 */
static int
jpeg_transform (struct jpeg_decompress_struct* srcinfo,
		struct jpeg_compress_struct* dstinfo,
		const char* thumbname_,
		const char* dstname_,
		int progressive,
		int optimize)
{
  /** input file */
  static FILE* src = 0;
  /** output file */
  static FILE* dst = 0;
  /** name of thumbnail file to embed (NULL = compute thumbnail) */
  const char* thumbname = thumbname_;
  /** destination image file name (NULL = stdin/stdout) */
  const char* dstname = dstname_;

  /* clear possible previous errors */
  errno = 0;

  /* set up jump context for input */
  if (setjmp (((struct exif_error_mgr*) srcinfo->err)->setjmp_buffer)) {
    if (errno)
      perror (dstname ? srcname : "(standard input)");
    else
      exif_output_message ((j_common_ptr) srcinfo);
  ignore:
    jpeg_abort_decompress (srcinfo);
    jpeg_abort_compress (dstinfo);
    if (src) (void) fclose (src), src = 0;
    if (dst) (void) fclose (dst), dst = 0;
    return 1;
  }

  /* set up jump context for output */
  if (setjmp (((struct exif_error_mgr*) dstinfo->err)->setjmp_buffer)) {
    if (errno)
      perror (dstname ? dstname : "(standard output)");
    else
      exif_output_message ((j_common_ptr) dstinfo);
    if (src) (void) fclose (src), src = 0;
    if (dst) (void) fclose (dst), dst = 0;
    return 2;
  }

  if (dstname && !(src = fopen (srcname, "rb"))) {
    (void) fputs (srcname, stderr), (void) fflush (stderr);
    perror (": open (reading)");
    return 1;
  }
  jpeg_stdio_src (srcinfo, dstname ? src : stdin);
  /* save COM markers */
  jpeg_save_markers (srcinfo, JPEG_COM, 0xFFFF);
  /* Save APP0..APP15 markers */
  {
    unsigned m;
    for (m = JPEG_APP0; m < JPEG_APP0 + 16; m++)
      jpeg_save_markers (srcinfo, m, 0xFFFF);
  }
  (void) jpeg_read_header (srcinfo, TRUE);
  exif_parse (srcinfo, !thumbname && !thumb_width && !thumb_height);
  /* read or create thumbnail image for JFIF or Exif files */
  if (thumbname) {
    FILE* thumbfile;
    if (!(thumbfile = fopen (thumbname, "rb"))) {
      (void) fputs (thumbname, stderr), (void) fflush (stderr);
      perror (": open (reading)");
      goto ignore;
    }
    thumb_length = fread (thumb_buf, 1, sizeof thumb_buf, thumbfile);
    if (ferror (thumbfile)) {
      (void) fputs (thumbname, stderr), (void) fflush (stderr);
      perror (": read");
      (void) fclose (thumbfile);
      goto ignore;
    }

    (void) fclose (thumbfile);

    if (thumb_length == sizeof thumb_buf) {
      fprintf (stderr, "%s: thumbnail image is too large\n", thumbname);
      goto ignore;
    }
    else if (!thumb_length)
      fprintf (stderr, "%s: ignoring empty thumbnail image file\n", thumbname);
  }
  else if ((thumb_width || thumb_height) &&
      (exif_endian != UNKNOWN || srcinfo->saw_JFIF_marker)) {
    thumb_create (srcinfo);
    if (!thumb_length)
      goto ignore; /* thumbnail too large */
    /* re-read the file after constructing the thumbnail */
    if (fseek (src, 0, SEEK_SET)) {
      (void) fputs (srcname, stderr), (void) fflush (stderr);
      perror (": fseek");
      goto ignore;
    }
    (void) jpeg_finish_decompress (srcinfo);
    jpeg_stdio_src (srcinfo, src);
    (void) jpeg_read_header (srcinfo, TRUE);
    exif_parse (srcinfo, 0); /* need to patch the Exif marker again */
  }
  else
    thumb_length = 0;

  if (dstname) {
    /* Open the destination file */
    int fd;
    (void) unlink (dstname);
    fd = open (dstname,
#if defined WIN32 || defined __WIN32
	       O_WRONLY | O_CREAT | O_EXCL | O_BINARY, 0666
#else
	       O_WRONLY | O_CREAT | O_EXCL, 0444
#endif
	       );
    if (fd < 0 || !(dst = fdopen (fd, "wb"))) {
      (void) fputs (dstname, stderr), (void) fflush (stderr);
      perror (": open (writing)");
      (void) fclose (src);
      return 2;
    }
  }
  jpeg_stdio_dest (dstinfo, dstname ? dst : stdout);
  jpeg_copy_critical_parameters (srcinfo, dstinfo);
  dstinfo->optimize_coding = optimize;
  dstinfo->write_JFIF_header =
    srcinfo->saw_JFIF_marker && exif_endian == UNKNOWN;
  dstinfo->JFIF_major_version = (UINT8) 1;
  dstinfo->JFIF_minor_version = (UINT8) 2;
  if (progressive)
    jpeg_simple_progression (dstinfo);
  jpeg_write_coefficients (dstinfo, jpeg_read_coefficients (srcinfo));
  /* write thumbnail in JFXX marker for JFIF images */
  if (dstinfo->write_JFIF_header && thumb_length) {
    /** JFXX marker header for JPEG thumbnail */
    static const unsigned char jfxx[] =
      { 0x4A, 0x46, 0x58, 0x58, 0x00, 0x10 };
    unsigned i;
    jpeg_write_m_header (dstinfo, JPEG_APP0,
			 thumb_length + sizeof jfxx);
    for (i = 0; i < sizeof jfxx; i++)
      jpeg_write_m_byte (dstinfo, jfxx[i]);
    for (i = 0; i < thumb_length; i++)
      jpeg_write_m_byte (dstinfo, thumb_buf[i]);
  }
  /* copy other markers (and embed thumbnail in Exif marker) */
  copy_markers (srcinfo, dstinfo);
  jpeg_finish_compress (dstinfo);
  if (src && fclose (src)) {
    (void) fputs (srcname, stderr), (void) fflush (stderr);
    perror (": close");
    /* ignore error on closing input file */
  }
  if (dst && fclose (dst)) {
    (void) fputs (dstname, stderr), (void) fflush (stderr);
    perror (": close");
    return 2;
  }
  return 0;
}

int
main (int argc, char **argv)
{
  /** libjpeg decompression context */
  struct jpeg_decompress_struct srcinfo;
  /** libjpeg compression context */
  struct jpeg_compress_struct dstinfo;
  /** libjpeg error handlers */
  struct exif_error_mgr jsrcerr, jdsterr;
  /** exit status */
  int status = 0;

  /** flag: disable the processing of options */
  int ignopt = 0;
  /** flag: use the progressive JPEG process */
  int progressive = 1;
  /** flag: compute optimal Huffman coding tables for the image */
  int optimize = 1;
  /** directory for modified image files (0=replace the original files) */
  char* imagedir = 0;
  /** imagedir length */
  unsigned imagedirlen = 0;
  /** directory for thumbnail image files (0=compute thumbnails) */
  char* thumbdir = 0;
  /** thumbdir length */
  unsigned thumbdirlen = 0;
  /** destination file name */
  char* dstname = 0;
  /** thumbnail file name */
  char* thumbname = 0;
#ifdef HAVE_PROGRESS
  /** progress indicator */
  FILE* progress = 0;
  /** progress counter */
  int progresscnt = 0;
#endif /* HAVE_PROGRESS */

  srcinfo.err = jpeg_std_error (&jsrcerr.pub);
  jsrcerr.pub.addon_message_table = exif_message_table;
  jsrcerr.pub.first_addon_message = EXIF_FIRSTCODE;
  jsrcerr.pub.last_addon_message = EXIF_LASTCODE;
  dstinfo.err = jpeg_std_error (&jdsterr.pub);

  jpeg_create_decompress (&srcinfo);
  jpeg_create_compress (&dstinfo);

  jsrcerr.pub.error_exit = jdsterr.pub.error_exit = exif_error_exit;
  jsrcerr.pub.output_message = jdsterr.pub.output_message =
    exif_output_message;

  for (argv++; argc > 1; argv++, argc--) {
    const char* opts = ignopt || **argv != '-' ? 0 : *argv;

    if (opts && !strcmp (opts, "--")) {
      argv++;
      argc--;
      ignopt = 1;
      continue;
    }

#ifdef HAVE_PROGRESS
    if (progress) {
      if (0 > fprintf (progress, "%u\n",
		       100 * (progresscnt - argc) / progresscnt) ||
	  fflush (progress)) {
	perror ("write(progress)");
	status = 2;
	goto done;
      }
    }
#endif /* HAVE_PROGRESS */

    while (opts && *++opts) { /* process all flags */
      switch (*opts) {
      default:
	(void) putc ('-', stderr);
	(void) putc (*opts, stderr);
	(void) fputs (": unknown option (use -h for help)\n", stderr);
	status = 1;
	goto done;
      case '?':
      case 'h':
	(void) fputs
	  ("jpegnail 1.2.4\n"
	   "Remove or embed thumbnails in Exif or JFIF JPEG images\n"
	   "Usage: jpegnail [options] file ...\n"
	   "Options:\n"
	   "-h"
	   "\tusage\n"
	   "-q num"
	   "\tthumbnail quality (0..100)\n"
	   "-x num"
	   "\tmaximum thumbnail width\n"
	   "-y num"
	   "\tmaximum thumbnail height\n"
	   "-b"
	   "\tbaseline JPEG\n"
	   "-p"
	   "\tprogressive JPEG (default)\n",
	   stderr);
	(void) fputs
	  ("-n"
	   "\tdo not compute optimal Huffman coding for the image\n"
	   "-S dir\n"
	   "\tread image thumbnails from dir instead of computing them\n"
	   "-s dir\n"
	   "\tsave the image files in dir instead of replacing originals\n"
#ifdef HAVE_PROGRESS
	   "-P command\n"
	   "\toutput progress indication to an external command\n"
#endif /* HAVE_PROGRESS */
	   "--\tdisable processing further options\n",
	   stderr);
	goto done;

      case 'b': progressive = 0; break;
      case 'p': progressive = 1; break;

      case 'n': optimize = 0; break;

#ifdef HAVE_PROGRESS
      case 'P':
	if (argc <= 2)
	  goto missing;
	if (progress)
	  pclose (progress);
	argc--;
	fflush (stderr);
	fflush (stdout);
	if (*++argv) {
	  progress = popen (*argv, "w");
	  if (!progress) {
	    fprintf (stderr, "-P %s", *argv);
	    perror (": popen");
	  }
	}
	else
	  progress = 0;
	progresscnt = argc;
	break;
#endif /* HAVE_PROGRESS */

      case 's': case 'S':
	if (argc <= 2) {
	missing:
	  (void) putc ('-', stderr), (void) putc (*opts, stderr);
	  (void) fputs (": missing argument (use -h for help)\n", stderr);
	  status = 1;
	  goto done;
	}
	argc--;
	if (!**++argv) {
	  /* empty string argument => restore default behaviour */
	  if (*opts == 's')
	    imagedir = 0, imagedirlen = 0;
	  else
	    thumbdir = 0, thumbdirlen = 0;
	}
	else {
	  /* make sure that the specified directory exists */
	  char* dir;
	  unsigned dirlen;
#if defined WIN32 || defined __WIN32
	  char* s = *argv;
	  while (*s) if (*s++ == '\\') s[-1] = '/';
#endif
	  dirlen = strlen (dir = *argv);
	  /* try to create directory for output images, but not for inputs */
	  if (*opts == 's' &&
	      mkdir (*argv
#if !defined WIN32 && !defined __WIN32
		     , 0777
#endif
		     ) && errno != EEXIST) {
	    (void) fputs (*argv, stderr), (void) fflush (stderr);
	    perror (": mkdir");
	    goto quit;
	  }
	  else {
	    struct stat statbuf;
	    char* s = malloc (dirlen + 3);
	    if (!s) {
	      (void) putc ('-', stderr), (void) putc (*opts, stderr);
	      (void) fputs (": out of memory\n", stderr);
	      goto quit;
	    }
	    memcpy (s, *argv, dirlen);
	    memcpy (s + dirlen, "/.", 3);
	    if (stat (s, &statbuf)) {
	      (void) fputs (s, stderr), (void) fflush (stderr);
	      perror (": stat");
	      free (s);
	      goto quit;
	    }
	    free (s);
	  }

	  if (*opts == 's')
	    imagedir = dir, imagedirlen = dirlen;
	  else
	    thumbdir = dir, thumbdirlen = dirlen;
	}
	break;
      case 'q':
      case 'x':
      case 'y':
	if (argc <= 2)
	  goto missing;
	else {
	  char* endp;
	  unsigned long q = strtoul (*++argv, &endp, 0);
	  argc--;
	  if (q < 1L || q > 65535L || *endp || !*argv) {
	  args:
	    (void) putc ('-', stderr);
	    (void) putc (*opts, stderr);
	    (void) fputs (": invalid argument\n", stderr);
	    goto quit;
	  }
	  if (*opts == 'q') {
	    if (q > 100L)
	      goto args;
	    thumb_quality = q;
	  }
	  else if (*opts == 'x')
	    thumb_width = q;
	  else
	    thumb_height = q;
	}
	break;
      }
    }

    if (opts && opts != *argv + 1)
      continue; /* get next parameters */

    /* if opts != 0 at this point, it denotes the file name '-' */
    srcname = *argv;
    if (!opts) { /* file name not '-' */
      unsigned len = strlen (*argv);
      if (!(dstname = imagedir
	    ? createfilename (*argv, len, imagedir, imagedirlen)
	    : malloc (len + 1))) {
      oom:
	(void) fputs (*argv, stderr);
	(void) fputs (": out of memory\n", stderr);
	goto quit;
      }
      if (!imagedir) {
	memcpy (dstname, *argv, len + 1);
	dstname[len - 1] = '0'; /* change the last character to '0' */
      }
      if (thumbdir) {
	/* read thumbnail images from a directory */
	if (!(thumbname =
	      createfilename (*argv, len, thumbdir, thumbdirlen))) {
	  free (dstname);
	  goto oom;
	}
      }
      else
	thumbname = 0;
    }
    else
      dstname = thumbname = 0;

    switch (jpeg_transform (&srcinfo, &dstinfo, thumbname, dstname,
			    progressive, optimize)) {
    case 0:
      free (thumbname);
      break;
    case 1:
      free (thumbname);
      free (dstname);
      continue;
    default:
      free (thumbname);
      free (dstname);
      status = 2;
      goto done;
    }
    if (!opts) {
      (void) touch_file (dstname, srcname);
      if (!imagedir && rename_file (dstname, srcname)) {
      quit:
	free (dstname);
	status = 3;
	goto done;
      }
      free (dstname);
    }
    (void) jpeg_finish_decompress (&srcinfo);
  }

 done:
  jpeg_destroy_compress (&dstinfo);
  jpeg_destroy_decompress (&srcinfo);
#ifdef HAVE_PROGRESS
  if (progress)
    pclose (progress);
#endif /* HAVE_PROGRESS */
  return status;
}
