/** @file rename.c
 * Atomic renaming of files
 * @author Marko Mkel <msmakela@nic.funet.fi>
 */

/* Copyright  2004 Marko Mkel.

   This file is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   The software is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

#if defined WIN32 || defined __WIN32
# include <windows.h>
#endif
#include <stdio.h>

/** Rename a file
 * @param oldpath       old file name
 * @param newpath       new file name
 * @return              zero on success, nonzero on error
 */
int
rename_file (const char* oldpath,
             const char* newpath)
{
#if defined WIN32 || defined __WIN32
  /** flag: running on Windows NT? -1=no, 1=yes, 0=unknown */
  static int is_nt;
  const char* op;
  if (!is_nt)
    is_nt = GetVersion () & 0x80000000 ? -1 : 1;
  if (is_nt > 0) {
    op = "MoveFileEx";
    if (!MoveFileEx (oldpath, newpath, MOVEFILE_REPLACE_EXISTING))
      goto error;
  }
  else {
    op = "MoveFile";
    (void) DeleteFile (newpath);
    if (!MoveFile (oldpath, newpath)) {
      char* msg;
      long err;
    error:
      err = GetLastError ();
      FormatMessage (FORMAT_MESSAGE_ALLOCATE_BUFFER |
                     FORMAT_MESSAGE_FROM_SYSTEM |
                     FORMAT_MESSAGE_IGNORE_INSERTS,
                     0,
                     err,
                     0, /* default language */
                     (LPTSTR) &msg,
                     0,
                     0);
      AnsiToOem (msg, msg);
      (void) fprintf (stderr, "%s -> %s: %s failed with code %ld: %s\n",
                      oldpath, newpath, op, err, msg);
      LocalFree (msg);
      return -1;
    }
  }
  return 0;
#else
  int status = rename (oldpath, newpath);
  if (status) {
    (void) fputs (oldpath, stderr);
    (void) fputs (" -> ", stderr);
    (void) fputs (newpath, stderr);
    (void) fflush (stderr);
    perror (": rename");
  }
  return status;
#endif
}
