 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for ipostscr
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lipostscr
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
static void set_default_values PROTO((double *, double *, double *, double *,
				      double *, double *, double, double, int,
				      int, int, int, int, int, int));
static int print_grayscale_image PROTO((kfile *, kobject, int, int));
static int print_color_image PROTO((kfile *, kobject, int, int));
static int print_bit_image PROTO((kfile *, kobject, int, int));
static void _kprint_header PROTO((kfile *, double, double, double, double, 
				  int, int));
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lipostscr - convert input image object to postscript
* 
*       Purpose: lipostscr formats an image for output on a Postscript 
*		 printer.  The width and height are specified in inches.  
*		 The X and Y offset specifies the location of the lower 
*		 left corner of the image from the lower left corner of 
*		 the page.
*
*		 lipostscr will generate either a binary, grayscale, 
*		 truecolor image from the input object.  If the data set 
*		 contains a map, then the first three columns (assuming 
*		 that there are at least three) will be interpreted a the 
*		 red, green, and blue components and will be used to 
*		 construct a truecolor postscript image by mapping the 
*		 value data through the map.  If a map exists, but it does 
*		 not have three or more columns, then the existing columns 
*		 will be padded out to three columns with 0's.
*
*		 If the data set does not contain a map, then if its data 
*		 type is KBIT, it will generate a bitmapped black and white 
*		 postscript.  Otherwise, the data set will be cast to 
*		 unsigned byte and a grayscale image will be generated.
*
*		 This program can be used to generate either Postscript or 
*		 Encapsulated Postscript images in either Landscape or 
*		 Portrait mode.  The -eps flag is used to specify that 
*		 encapsulated postscript should be output.
*
*		 If you only specify the width, or the height, the other 
*		 dimension will be calculated, so that you will get a 
*		 postscript image which reflects the true dimensions of the 
*		 image.
*         Input: obj     - Object to be output in postscript form
*		 width   - Desired width in inches of the printed image 
*		 height  - Desired height of printed image in inches
*		 xoffset - Width offset in inches from bottom left corner 
*			   of printed page
*		 yoffset - Height offset in inches from bottom left corner 
*			   of printed page
*		 wflg    - if TRUE, use width input parameter
*		 hflg    - if TRUE, use height input parameter
*		 xflg    - if TRUE, use xoffset input parameter
*		 yflg    - if TRUE, use yoffset input parameter
*		 landscape - if TRUE, output as landscape, otherwise portrait
*		 eps     - if TRUE, output as encapsulated postscrip, otherwise
*		           output as
*		 showpage - if TRUE, issue showpage at end of output
*        Output: file     - output postscript file
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: 
*    Written By: Mark Young, Scott Wilson, Jeremy Worley
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: *
*   Declaration: int lipostscr(
*		 !   kfile *file, 
*		 !   kobject obj,
*		 !   double width, 
*		 !   double height, 
*		 !   double xoffset, 
*		 !   double yoffset, 
*		 !   int wflg, 
*		 !   int hflg, 
*		 !   int xflg, 
*		 !   int yflg,
*		 !   int landscape,
*		 !   int eps,
*		 !   int showpage)
****************************************************************/
/* -library_def */
int
lipostscr(
   kfile *file, 
   kobject obj,
   double width, 
   double height, 
   double xoffset, 
   double yoffset, 
   int wflg, 
   int hflg, 
   int xflg, 
   int yflg,
   int landscape,
   int eps,
   int showpage)
/* -library_def_end */

/* -library_code */
{
   int rows;
   int cols;
   int bands;
   int type;
   double ax;
   double ay;
   double pwidth = width;
   double pheight = height;
   double pxoffset = xoffset;
   double pyoffset = yoffset;
   double maxpageheight;
   double maxpagewidth;

   /*
    * first things first:  reference the object so that no changes in this
    * routine get propagated as side effects.
    */
   obj = kpds_reference_object(obj);

   kpds_set_attribute(obj, KPDS_MAPPING_MODE, KMAPPED);
   
   
   /*	
    * Intiailize values for the "set_default_values" routine.  This
    * routine is used to make some assumtions about the aspect ratio
    * and also the position of the output image on the page. 
    */

   if (!kpds_get_attribute(obj, KPDS_POINT_SIZE, &ax, &ay, NULL, NULL, NULL))
   {
      kinfo (KSTANDARD, "lipostscr:  Unable to get the KPDS_POINT_SIZE "
	     "attribute.  Initializing it to a default value of 1.0 x 1.0.\n");
   
      ax = ay = 1.0;
   }

   /*
    * get some handy attributes
    */
   if (!kpds_get_attributes(obj, KPDS_VALUE_DATA_TYPE, &type,
			    KPDS_VALUE_SIZE, &cols, &rows, NULL, NULL, &bands,
			    NULL))
   {
      kerror("image", "lipostscr", "Unable to get either the data type or "
	     "The size of the object to be output.");
      return (FALSE);
   }

   /*
    * set a whole pile of default values.
    */
   set_default_values(&pwidth, &pheight, &pxoffset, &pyoffset, &maxpagewidth, 
		      &maxpageheight, ax, ay, rows, cols, wflg, hflg, 
		      xflg, yflg, landscape);

   if ((pxoffset + pwidth) > maxpagewidth || 
       (pyoffset + pheight) > maxpageheight)
   {
      kerror ("image", "lipostscr", "Requested image layout too large "
	      "for page!");
      return (FALSE);
   }

   /*
    * Print the header.
    */
   _kprint_header(file, pwidth, pheight, pxoffset, pyoffset, eps, landscape);
      
   /*
    * dump the image data.
    */
   if (kpds_query_map(obj) || bands > 1)
      print_color_image(file, obj, rows, cols);
   else if (type == KBIT)
      print_bit_image(file, obj, rows, cols);
   else
      print_grayscale_image(file, obj, rows, cols);

   /*
    * dump the footer.
    */
   kfprintf(file, "\ngrestore\n");

   if (showpage)
      kfprintf(file, "showpage\n");

   kfprintf(file, "%%%%Trailer\n");

   return (TRUE);
}

static void
set_default_values(
   double *width, 
   double *height, 
   double *xoffset, 
   double *yoffset, 
   double *maxpagewidth, 
   double *maxpageheight,

   double ax, 
   double ay, 
                   
   int rows, 
   int cols, 
   int wflg, 
   int hflg, 
   int xflg, 
   int yflg,
   int landscape)
{
   /*
    * set the maximum page width and height according to whether the
    * user requested landscape 
    */
   if (landscape)
   {
      *maxpagewidth = 11.0;
      *maxpageheight = 8.5;
   }
   else
   {
      *maxpagewidth = 8.5;
      *maxpageheight = 11.0;
   }

   /*  
    * Setup default to 4x4, 4x? or ?x4 picture  
    */
   if (wflg == 0 && hflg == 0)  
   {
      if (rows*ay >= cols*ax)
      {
	 *height = 4.0;
	 *width = 4.0*(cols*ax)/(rows*ay);
      }
      else
      {
	 *width = 4.0;
	 *height = 4.0*(rows*ay)/(cols*ax);
      }
   }
   else if (wflg == 0 && hflg != 0) 
      *width = (*height) * (cols * ax)/(rows * ay);

   else if (wflg != 0 && hflg == 0) 
      *height = (*width) * (rows * ay)/(cols * ax);

   if (xflg == 0) 
      *xoffset = (*maxpagewidth - *width)/2.0;

   if (yflg == 0) 
      *yoffset = (*maxpageheight - *height)/2.0;

   return;
}

static int
print_grayscale_image(
   kfile *file, 
   kobject obj,
   int rows,
   int cols)
{
   unsigned char *data;
   unsigned char *tmp;
   int i;
   int j;
   
   kfprintf(file, "/picstr %d string def\n",cols);
   kfprintf(file, "%d %d 8 [%d 0 0 -%d 0 %d]\n",cols,rows,cols,rows,rows);
   kfprintf(file, "{currentfile picstr readhexstring pop} image\n");

   kpds_set_attributes(obj, KPDS_VALUE_DATA_TYPE, KUBYTE,
		       KPDS_VALUE_POSITION, 0, 0, 0, 0, 0, NULL);
   
   if ((data = kpds_get_data(obj, KPDS_VALUE_ALL, NULL)) == NULL)
   {
      kerror("image", "ipostscr", "Get of image data failed.\n");
      return (FALSE);
   }

   tmp = data;
   
   for (i = 0; i < rows; i++)
   {
      for (j = 0; j < cols; j++)
	 kfprintf(file, "%02x", *data++);
      
      kfprintf(file, "\n");
   }
   
   kfree(tmp);
   
   return (TRUE);
}

static int
print_color_image(
   kfile *file, 
   kobject obj,
   int rows,
   int cols)
{
   unsigned char *data;
   unsigned char *p1;
   unsigned char *p2;
   unsigned char *p3;
   int i;
   int j;
   
   /* 
    * Declare three strings, one for RED, GREEEN, and BLUE 
    */
   kfprintf(file, "/rstr %d string def\n",cols);
   kfprintf(file, "/gstr %d string def\n",cols);
   kfprintf(file, "/bstr %d string def\n",cols);

   /* 
    * Set up the source and mapping 
    */
   kfprintf(file, "%d %d 8 [%d 0 0 -%d 0 %d]\n",cols,rows,cols,rows,rows);

   /* 
    * Set up the read procedures 
    */
   kfprintf(file, "{currentfile rstr readhexstring pop}\n");
   kfprintf(file, "{currentfile gstr readhexstring pop}\n");
   kfprintf(file, "{currentfile bstr readhexstring pop}\n");
   
   /* 
    * Set up the colorimage operator with multiple procedures and 3 colors 
    */
   kfprintf(file,"true 3\ncolorimage\n");

   kpds_set_attributes(obj, KPDS_VALUE_DATA_TYPE, KUBYTE,
		       KPDS_VALUE_SIZE, -1, -1, 1, 1, 3,
		       KPDS_VALUE_POSITION, 0, 0, 0, 0, 0, NULL);
   
   if ((data = kpds_get_data(obj, KPDS_VALUE_ALL, NULL)) == NULL)
   {
      kerror("image", "ipostscr", "Get of image data failed.\n");
      return (FALSE);
   }
   
   p1 = data;
   p2 = p1 + rows*cols;
   p3 = p2 + rows*cols;

   for (i=0; i< rows; i++)
   {
      for (j=0; j<cols; j++) /* RED */
	 kfprintf(file,"%02x",*p1++);
      kfprintf(file,"\n");

      for (j=0; j<cols; j++) /* GREEN */
	 kfprintf(file,"%02x",*p2++);
      kfprintf(file,"\n");

      for (j=0; j<cols; j++) /* BLUE */
	 kfprintf(file,"%02x",*p3++);
      kfprintf(file,"\n");
   }

   kfree(data);
   
   return (TRUE);
}
      
static int
print_bit_image(
   kfile *file, 
   kobject obj,
   int rows, 
   int cols)
{
   unsigned char *data;
   unsigned char *tmp;
   int k;
   int i;
   int j;

   k = (cols + 7) / 8;

   kfprintf(file, "/picstr %d string def\n", k);

   kfprintf(file, "%d %d 1 [%d 0 0 -%d 0 %d]\n", cols, rows, cols, rows, rows);

   kfprintf(file, "{currentfile picstr readhexstring pop} image\n	");

   kpds_set_attributes(obj, KPDS_VALUE_POSITION, 0, 0, 0, 0, 0, NULL);
   
   if ((data = kpds_get_data(obj, KPDS_VALUE_ALL, NULL)) == NULL)
   {
      kerror("image", "ipostscr", "Get of image data failed.\n");
      return (FALSE);
   }

   tmp = data;
   
   for (i = 0; i < rows; i++)
   {
      for (j = 0; j < k; j++)
	 kfprintf(file, "%02x", *data++);

      kfprintf(file, "\n");
   }

   kfree(tmp);
   
   return (TRUE);
}

static void
_kprint_header(
   kfile *file,
   double width, 
   double height, 
   double xoffset, 
   double yoffset,
   int eps, 
   int landscape)
{
   time_t date_clock;
   int sx;
   int sy;
   int dx; 
   int dy;
   
   time(&date_clock);

   /*
    * The scaling factor and translation distance is in terms of 72
    * pixels per inch.  So we multiple the number of inches by 72 and
    * truncate to the nearest pixel value. 
    */
   sx = (int) (width * 72.0);
   sy = (int) (height * 72.0);
   dx = (int) (xoffset * 72.0);
   dy = (int) (yoffset * 72.0);
   if (eps)
   {
      kfprintf(file, "%%!PS-Adobe-2.0 EPSF-1.2\n");
      kfprintf(file, "%%%%Title: Khoros Postscript Image Document\n");
      kfprintf(file, "%%%%Creator: Khoros 2.0 ipostscr\n");
      kfprintf(file, "%%%%CreationDate: %s",ctime(&date_clock));
      kfprintf(file, "%%%%BoundingBox: %d %d %d %d\n",dx,dy, dx+sx, dy+sy);
      kfprintf(file, "%%%%Pages: 1\n");
      kfprintf(file, "%%%%EndComments\n");
      kfprintf(file, "gsave\n");
      kfprintf(file, "%%%%EndProlog	\n");
      kfprintf(file, "%%%%Page: 1 1\n");
   }
   else
   {
      kfprintf(file, "%%!PS-Adobe-2.0\n");
      kfprintf(file, "%%%%BoundingBox: %d %d %d %d\n",dx,dy, dx+sx, dy+sy);
      kfprintf(file, "%%%%Creator: Khoros 2.0 ipostscr\n");
      kfprintf(file, "%%%%CreationDate: %s",ctime(&date_clock));
      kfprintf(file, "%%%%Pages: 1\n");

      if (landscape)
	 kfprintf(file, "%%%%Orientation: Landscape\n");
      else
	 kfprintf(file, "%%%%Orientation: Portrait\n");
      
      kfprintf(file, "%%%%EndComments\n");
      kfprintf(file, "gsave\n");
      kfprintf(file, "%%%%EndProlog\n");
      kfprintf(file, "%%%%Page: 1 1\n");
   }

   if (landscape)
   {
      kfprintf(file, "90 rotate\n");
      kfprintf(file, "0 -%d translate\n", sy);
      kfprintf(file, "%d -%d translate\n",dy,dx);
   }
   else
      kfprintf(file, "%d %d translate\n",dx,dy);

   kfprintf(file, "%d %d scale\n",sx,sy);

   return;
}
/* -library_code_end */
