 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for igeowarp
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	ligeowarp
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: ligeowarp - perform direct bilinear warping in the WxH plane
* 
*       Purpose: perform direct bilinear warping in the WxH plane. This
*                is a reasonable way to register one image to another
*                provided that the distortion is not too severe. Output
*                pixel values are calculated using bilinear interpolation. 
*
*         Input: in_obj - input object to be processed
*                a00 - constant term coeff in W mapping function
*                a01 - w constant term coeff in W mapping function
*                a10 - h constant term coeff in W mapping function
*                a11 - wh constant term coeff in W mapping function
*                b00 - constant term coeff in H mapping function
*                b01 - w constant term coeff in H mapping function
*                b10 - h constant term coeff in H mapping function
*                b11 - wh constant term coeff in H mapping function
*                planes - if non-zero, process date by full planes
*                             instead of prisms. Can be MUCH faster by planes
*                             than prisms, but is less well adapted for large 
*                             data sets.
*
*        Output: out_obj - output object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Scott Wilson
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int ligeowarp(
kobject in_obj,
double a00,
double a01,
double a10,
double a11,
double b00,
double b01,
double b10,
double b11,
int planes,
kobject out_obj)
/* -library_def_end */

/* -library_code */
{
        char *lib = "kdatamanip", *rtn = "ligeowarp";
        klist *objlist=NULL;
        unsigned int i,j,k;
        int map_exists,mask_exists;
	unsigned int wc,hc,dc,tc,ec; /* Dimensions of input data set */
	unsigned int wout,hout,dout,tout,eout; /* Dimensions of output data */
	unsigned int wro,hro,dro; /* Dimensions of output region */
	unsigned int wri,hri,dri; /* Dimensions of input region */
        double *bd=NULL,*bm=NULL,*bmask=NULL,*bdata=NULL;
        double in,kn,z1,z2,z3,z4,za,zb,zp;
        int jl,jh,il,ih;
        int beg[5],end[5];

        /* Make sure we have valid objects */
        if (in_obj == KOBJECT_INVALID || out_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus input or output object");
            return(FALSE);
          }

        /* Reference the input object to avoid side effects, then add the
           reference to the list of goodies to be autmatically free'd on
           error. */
        KCALL((in_obj = kpds_reference_object(in_obj)));
        objlist = klist_add(objlist,in_obj,"KOBJECT");

        /* Create the output value segment if it's not there already and
           copy it's size and data type attributes from the input obj.  */
        KCALL(kpds_copy_object(in_obj,out_obj));

        mask_exists=kpds_query_mask(in_obj);
        map_exists =kpds_query_map(in_obj);

        /* If map exists, push all the data thru the map */
        if (map_exists)
          {
            KCALL(kpds_set_attribute(in_obj,KPDS_MAPPING_MODE,KMAPPED));
            KCALL(kpds_destroy_map(out_obj));
          }

        /* Constrain the input data to double real for reading */
        KCALL(kpds_set_attributes(in_obj,
                                 KPDS_VALUE_DATA_TYPE, KDOUBLE,
                                 KPDS_VALUE_COMPLEX_CONVERT, KREAL,
                                 KPDS_VALUE_INTERPOLATE,KPAD,NULL));

        if (mask_exists) KCALL(kpds_set_attributes(in_obj,
                                 KPDS_MASK_DATA_TYPE,KDOUBLE,
                                 KPDS_VALUE_INTERPOLATE,KPAD,NULL));
 
        /* See how big the data is */
        KCALL(kpds_get_attributes(in_obj,
                                 KPDS_VALUE_SIZE,&wc,&hc,&dc,&tc,&ec,NULL));

        /* Constrain to 3D spatial coordinates */
        if (ec != 1 || tc != 1)
          {
            kerror(lib, rtn, "no support for rotation of data on E or T dimensions");
            (void)klist_free(objlist, (kfunc_void)lkcall_free); \
            return(0);
          }

        wout = wc;
        hout = hc;
        dout = dc;
        tout = 1;
        eout = 1;

        if (!planes)
          {
            /* Set the region sizes */
            wri = 2;  hri = 2;  dri = dc;
            wro = wout; hro = 1;  dro = dout;
          }
        else
          {
            /* Set the region sizes */
            wri = wc; hri = hc; dri = 1;
            wro = wout; hro = hout; dro = 1;
          }

        /* Set up the output object, reference it, and set presentation vals */
        KCALL(kpds_set_attributes(out_obj,
                                 KPDS_VALUE_SIZE,wout,hout,dout,1,1,NULL));
        if (mask_exists)
          KCALL(kpds_set_attributes(out_obj,
                                 KPDS_MASK_SIZE,wout,hout,dout,1,1,NULL));
        KCALL((out_obj = kpds_reference_object(out_obj)));
        objlist = klist_add(objlist,in_obj,"KOBJECT");
        KCALL(kpds_set_attributes(out_obj,
                                 KPDS_VALUE_DATA_TYPE, KDOUBLE,NULL));
        if (mask_exists)          {
            KCALL(kpds_set_attributes(out_obj,
                                 KPDS_MASK_DATA_TYPE,KDOUBLE,NULL));
            KCALL(kpds_set_attributes(out_obj,
                                 KPDS_MASK_SIZE,wout,hout,dout,1,1,NULL));
          }

        /* Set the region sizes on the input and output objects */
        KCALL(kpds_set_attribute(in_obj,
                                  KPDS_VALUE_REGION_SIZE,wri,hri,dri,1,1));
        KCALL(kpds_set_attribute(out_obj,
                                  KPDS_VALUE_REGION_SIZE,wro,hro,dro,1,1));
        if (mask_exists)
          { 
            KCALL(kpds_set_attribute(in_obj,
                                   KPDS_MASK_REGION_SIZE,wri,hri,dri,1,1));
            KCALL(kpds_set_attribute(out_obj,
                                   KPDS_MASK_REGION_SIZE,wro,hro,dro,1,1));
          }

        /* Set up output buffer */
        bd = (double *)kmalloc(wro*hro*dro*sizeof(double));
        objlist = klist_add(objlist,bd,"KMALLOC");
        if (mask_exists)
          {
            bm = (double *)kmalloc(wro*hro*dro*sizeof(double));
            objlist = klist_add(objlist,bm,"KMALLOC");
          }

        if (!planes)
          {
            for (i=0; i<hout; i++)
              {
                for (j=0; j<wout; j++)
                  {
                    kn = a00 + a01*j + a10*i + a11*i*j;
                    in = b00 + b01*j + b10*i + b11*i*j;
                    jl = floor(kn); jh = jl+1;
                    il = floor(in); ih = il+1;
                    /* Go under KPDS to KDMS to speed the large data set process
                    ing case. This is NOT the way to do this in general, but is
                    a last grasp for speed and works by avoiding the overhead
                    imposed by KPDS. This will only work if the index order is
                    whdte, a condition that can be imposed by a kpds_open_object
                    call. This is normally the case for use by a kroutine
                    driver, but could be a problem for folks using this routine
                    in thier own processing code. */
                    beg[0] = jl; beg[1]=il; beg[2]=0;    beg[3]=0; beg[4]=0;
                    end[0] = jh; end[1]=ih; end[2]=dc-1; end[3]=1; end[4]=1;
                    bdata=kdms_get_data(in_obj,KDMS_SEGMENT_VALUE,beg,end,(kaddr)bdata);
                    if (mask_exists)
                      bdata=kdms_get_data(in_obj,KDMS_SEGMENT_MASK,beg,end,(kaddr)bmask);
                    for (k=0; k<dro; k++)
                      {
                        z1 = bdata[4*k+0]; z2 = bdata[4*k+1];
                        z4 = bdata[4*k+3]; z3 = bdata[4*k+2];
                        za = z1*(ih-in)+z4*(in-il);
                        zb = z2*(ih-in)+z3*(in-il);
                        zp = za*(jh-kn)+zb*(kn-jl);
                        bd[k*wro+j] = zp;
                        if (mask_exists)
                          if (bmask[4*k+0] != 0 && bmask[4*k+1] != 0 &&
                            bmask[4*k+3] != 0 && bmask[4*k+2] != 0)
                              bm[k*wro+j] = 1;
                      }
                  }
                KCALL(kpds_put_data(out_obj,KPDS_VALUE_REGION,(kaddr)bd));
                if (mask_exists)
                  KCALL(kpds_put_data(out_obj,KPDS_MASK_REGION,(kaddr)bm));
              }
          }
        else /* Process data by full planes instead of prisms */
          {
            for (k=0; k<dout; k++)
              {
                bdata=kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)bdata);
                if (mask_exists)
                  bmask=kpds_get_data(in_obj,KPDS_MASK_REGION,(kaddr)bmask);
                for (i=0; i<hout; i++)
                  {
                    for (j=0; j<wout; j++)
                      {
                        kn = a00 + a01*j + a10*i + a11*i*j;
                        in = b00 + b01*j + b10*i + b11*i*j;
                        jl = floor(kn); jh = jl+1;
                        il = floor(in); ih = il+1;
                        if (in >= 0 && in < hc &&
                            kn >= 0 && kn < wc &&
                            jl >= 0 && jl < wc &&
                            jh >= 0 && jh < wc &&
                            il >= 0 && il < hc &&
                            ih >= 0 && ih < hc)
                          {
                            z1 = bdata[il*wc+jl]; z2 = bdata[il*wc+jh];
                            z4 = bdata[ih*wc+jl]; z3 = bdata[ih*wc+jh];
                            za = z1*(ih-in)+z4*(in-il);
                            zb = z2*(ih-in)+z3*(in-il);
                            zp = za*(jh-kn)+zb*(kn-jl);
                            bd[i*wout+j] = zp;
                          }
                        else
                          {
                            bd[i*wout+j] = 0;
                          } 
                        if (mask_exists)
                          if (bmask[il*wc+jl] != 0 && bmask[il*wc+jh] != 0 &&
                            bmask[ih*wc+jl] != 0 && bmask[ih*wc+jh] != 0)
                              bm[i*wout+j] = 1;
                      }
                  }
                KCALL(kpds_put_data(out_obj,KPDS_VALUE_REGION,(kaddr)bd));
                if (mask_exists)
                  KCALL(kpds_put_data(out_obj,KPDS_MASK_REGION,(kaddr)bm));
              }
          }

        /* Free up everything that has been allocated */
        kfree(bdata);
        if (mask_exists) kfree(bmask);
        (void)klist_free(objlist, (kfunc_void)lkcall_free);
 
        return TRUE;
  }
/* -library_code_end */
