
/*********************************************************************

Copyright (C) 1993, Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus is is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory

  "this software is 100% hand-crafted by a human being in the USA"

*********************************************************************/

#include <design.h>
#include <geometry.h>
#include "rmonster_objs.h"
#include "camera.h"
#include "poly.h"
#include "rmatrix.h"
#include "shader.h"
#include "sort_trans.h"
#include "imaging.h"
#include "lights.h"
#include "zbuffer.h"

    /**
      * here, we have two types of functions.  one is a set of shader
      * functions.  we have flat shaders, gouroud shaders, phong shaders,
      * etc.  the other type of functions are the initialization functions.
      * for flat shaded polygons, for example, we can compute the dot
      * of the surface normal with all lights, thus precomputing the
      * diffuse component at once rather than at each pixel covered
      * by the facet.
      *
      * we also have one set of shader functions used to color opaque
      * objects, and another set used to color transparent objects.
      * the opaque routines will write dots to the framebuffer directly,
      * whereas those that deal with transparent pixels dispatch the
      * color and opacity off to the bucketized z buffer which will
      * be subsequently composited after all the transparent objects
      * are rendererd.
      * future work:  the specular component should be handled a bit
      * differently for transparnet objects.  the diffuse and ambient
      * terms will be attenuated by opacity, whereas the opacity part
      * of the specular component should be a function of the
      * specular component only. for now, all components are dealt
      * with identically.
    **/

static int default_shader=KSHADER_FLAT;
void opaque_flat_shaderfunc();
void opaque_nolight_shaderfunc();
void opaque_gouroud_shaderfunc();
void opaque_phong_shaderfunc();

void transparent_flat_shaderfunc();
void transparent_nolight_shaderfunc();
void transparent_gouroud_shaderfunc();
void transparent_phong_shaderfunc();

void
(*opaque_shaderfuncs[])() =
{
    opaque_nolight_shaderfunc,
    opaque_flat_shaderfunc,
    opaque_gouroud_shaderfunc,
    opaque_phong_shaderfunc
};
int num_opaque_shaderfuncs = knumber(opaque_shaderfuncs);

void
(*transparent_shaderfuncs[])() =
{
    transparent_nolight_shaderfunc,
    transparent_flat_shaderfunc,
    transparent_gouroud_shaderfunc,
    transparent_phong_shaderfunc
};
int num_transparent_shaderfuncs = knumber(transparent_shaderfuncs);

void init_flat_shaderfunc PROTO(( surface_attribs *, surface_colors_rgba *, vertex_3d *, vertex_3d *, Poly *));
void init_nolight_shaderfunc PROTO(( surface_attribs *, surface_colors_rgba *, vertex_3d *, vertex_3d *, Poly *));
void init_gouroud_shaderfunc PROTO(( surface_attribs *, surface_colors_rgba *, vertex_3d *, vertex_3d *, Poly *));
void init_phong_shaderfunc PROTO(( surface_attribs *, surface_colors_rgba *, vertex_3d *, vertex_3d *, Poly *));

void
(*init_shaderfuncs[])() =
{
    init_nolight_shaderfunc,
    init_flat_shaderfunc,
    init_gouroud_shaderfunc,
    init_phong_shaderfunc
};
int num_init_shaderfuncs = knumber(init_shaderfuncs);

static surface_attribs flat_surface_attribs;
static surface_colors_rgba flat_surface_colors;
static surface_attribs gouroud_surface_attribs;
static surface_colors_rgba gouroud_surface_colors;
static float amb[3],diff[3],spec[3];
float f_flat_shade[4];
float f_nolight_shade[4];

static float gouroud_amb[3]; /* these are coefficients */
static float gouroud_diff[3]; /* these are coefficients */
static float gouroud_spec[3]; /* these are coefficients */

static float diffuse_dot;

void nvertex_scaled_shade_spec_per_face_norms PROTO((surface_colors_rgba *,float *, surface_colors_rgba *,int,float *,vertex_3d *));

void
opaque_nolight_shaderfunc(x,y,p)
int x,y;
Poly_vert *p;
{
    int r,g,b;
    float zb;
    float opacity=1.;

    zbuf_read(x,y,&zb);
    if (p->sz <= zb)
    {
	write_dot(x,y,f_nolight_shade[0],f_nolight_shade[1],f_nolight_shade[2]);
	zbuf_write(x,y,p->sz);
#if 0
	abuf_write(x,y,opacity);  /* temporary */
#endif
    }
}

void    
opaque_flat_shaderfunc(x,y,p)
int x,y;
Poly_vert *p;
{
    /**
      * in this function, we only copy over the color.  in the setup
      * routine, we compute the facet color based upon the dot product
      * of the light and object normal, so no computation is necessary.
    **/
    int r,g,b;
    float zb;
    float opacity=1.;
    float fr,fg,fb;

    zbuf_read(x,y,&zb);
    if (p->sz <= zb)
    {
#if 0	
	fr = p->r;
	fg = p->g;
	fb = p->b;
	write_dot(x,y,fr,fg,fb);
#endif
	write_dot(x,y,f_flat_shade[0],f_flat_shade[1],f_flat_shade[2]); 
	zbuf_write(x,y,p->sz);
    }
}

void    
opaque_gouroud_shaderfunc(x,y,p)
int x,y;
Poly_vert *p;
{
    /**
      * in this function, we only copy over the color.  in the setup
      * routine, we compute the facet color based upon the dot product
      * of the light and object normal, so no computation is necessary.
      * the interpolation of colors is handled in the polygon scan
      * converter.
    **/
    float fr,fg,fb;
    float zb;
    float opacity=1.0; /* temp */

    zbuf_read(x,y,&zb);
    if (p->sz <= zb)
    {
	fr = p->r;
	fg = p->g;
	fb = p->b;

	write_dot(x,y,fr,fg,fb);
	zbuf_write(x,y,p->sz);
/*	abuf_write(x,y,opacity); */
    }
}

void    
opaque_phong_shaderfunc(x,y,p)
int x,y;
Poly_vert *p;
{
    /**
      * below is the guts of the gouroud shaderfunc.  here, we want to
      * recompute the color at this pixel based upon lighting values
      * and the surface normal which is interpolated in the calling
      * routine.
      *
      * in other words; the code below is incomplete, wrong, etc.
      * 8/18/94
    **/
    float fr,fg,fb;
    float zb;
    float opacity=1.0; /* temp */

    zbuf_read(x,y,&zb);
    if (p->sz <= zb)
    {
	fr = p->r;
	fg = p->g;
	fb = p->b;

	write_dot(x,y,fr,fg,fb);
	zbuf_write(x,y,p->sz);
/*	abuf_write(x,y,opacity); */
    }
}

void
transparent_nolight_shaderfunc(x,y,p)
int x,y;
Poly_vert *p;
{
    int r,g,b;
    float zb;
    float opacity=1.;

    zbuf_read(x,y,&zb);
    if (p->sz < zb)
    {
	zb = p->sz;
	add_zbucket_entry(x,y,zb,f_nolight_shade[0],f_nolight_shade[1],f_nolight_shade[2],f_nolight_shade[3]);
    }
}

void    
transparent_flat_shaderfunc(x,y,p)
int x,y;
Poly_vert *p;
{
    /**
      * in this function, we only copy over the color.  in the setup
      * routine, we compute the facet color based upon the dot product
      * of the light and object normal, so no computation is necessary.
    **/
    int r,g,b;
    float zb;
    float opacity=1.;

    zbuf_read(x,y,&zb);
    if (p->sz < zb)
    {
	zb = p->sz;
	add_zbucket_entry(x,y,zb,f_flat_shade[0],f_flat_shade[1],f_flat_shade[2],f_flat_shade[3]);
    }
}

void    
transparent_gouroud_shaderfunc(x,y,p)
int x,y;
Poly_vert *p;
{
    /**
      * in this function, we only copy over the color.  in the setup
      * routine, we compute the facet color based upon the dot product
      * of the light and object normal, so no computation is necessary.
      * the interpolation of colors is handled in the polygon scan
      * converter.
    **/
    int r,g,b;
    float fr,fg,fb,fa;
    float zb;
    float opacity=1.0; /* temp */

    zbuf_read(x,y,&zb);
    if (p->sz < zb)
    {
	zb = p->sz;
	fr = p->r;
	fg = p->g;
	fb = p->b;
	fa = p->a;
	add_zbucket_entry(x,y,zb,fr,fg,fb,fa);
    }
}

void    
transparent_phong_shaderfunc(x,y,p)
int x,y;
Poly_vert *p;
{
    /**
      * below is the guts of the gouroud shaderfunc.  here, we want to
      * recompute the color at this pixel based upon lighting values
      * and the surface normal which is interpolated in the calling
      * routine.
      *
      * in other words; the code below is incomplete, wrong, etc.
      * 8/18/94
    **/
    int r,g,b;
    float fr,fg,fb,fa;
    float zb;
    float opacity=1.0; /* temp */

    zbuf_read(x,y,&zb);
    if (p->sz < zb)
    {
	zb = p->sz;
	fr = p->r;
	fg = p->g;
	fb = p->b;
	fa = p->a;
	add_zbucket_entry(x,y,zb,fr,fg,fb,fa);
    }
}

void
init_nolight_shaderfunc(
surface_attribs *sa,
surface_colors_rgba *sc,
vertex_3d *normal,  /* normal and a point on the surface */
vertex_3d *pt,
Poly *p)
{
    /* this is the same as the nolight shaderfunc */
    int i;
    double mag;

    for (i=0;i<4;i++)
	 f_nolight_shade[i] = 0.;

    mag = 1./p->n;
    for (i=0;i<p->n;i++)
    {
	f_nolight_shade[0] += p->vert[i].r;
	f_nolight_shade[1] += p->vert[i].g;
	f_nolight_shade[2] += p->vert[i].b;
	f_nolight_shade[3] += p->vert[i].a;
    }

    for (i=0;i<4;i++)
	 f_nolight_shade[i] *= mag;
}

void
init_flat_shaderfunc(
   surface_attribs *sa,
   surface_colors_rgba *sc,
   vertex_3d *normal,  /* normal and a point on the surface */
   vertex_3d *pt,
   Poly *p)
{
    /* we could do something intelligent about not copying to both
       int and float versions.  the int is for the opaque shader, the
       float part is for the transparent shader. */
    int i;
    double mag;

    for (i=0;i<4;i++)
	 f_flat_shade[i] = 0.;

    mag = 1./p->n;
    for (i=0;i<p->n;i++)
    {
	f_flat_shade[0] += p->vert[i].r;
	f_flat_shade[1] += p->vert[i].g;
	f_flat_shade[2] += p->vert[i].b;
	f_flat_shade[3] += p->vert[i].a;
    }

    for (i=0;i<4;i++)
	 f_flat_shade[i] *= mag;

}

void
init_gouroud_shaderfunc(
   surface_attribs *sa,
   surface_colors_rgba *sc,
   vertex_3d *normal,  /* normal and a point on the surface */
   vertex_3d *pt,
   Poly *p)
{
}

void
init_phong_shaderfunc(
   surface_attribs *sa,
   surface_colors_rgba *sc,
   vertex_3d *normal,  /* normal and a point on the surface */
   vertex_3d *pt,
   Poly *p)
{
}

int
set_def_shader(shader)
int shader;
{
    default_shader = shader;
    return(CHILL);
}

int
get_def_shader()
{
    return(default_shader);
}

/**
  * the order of the routines below corresponds to the order of
  * #define's in lights.h.
**/

static void to_point_light_func PROTO((vertex_3d *verts,vertex_3d *L, vertex_3d *out, int nverts));
static void to_directional_light_func PROTO((vertex_3d *verts,vertex_3d *L, vertex_3d *out,int nverts));
static void (*tolight_funcs[])() =
{
    to_directional_light_func,
    to_directional_light_func,
    to_point_light_func
};

/**
  * the following routine is to be used in compute the shade at a list
  * of vertices.
  *
  * verts = input x,y,z points
  * normals = input x,y,z vertex normals.  MUST BE one vertex per normal.
  * nverts = int, number of vertices to shade
  * in_fcolors = surface_colors_rgba *, possibly null, of color at each
  *      vertex.  if null, then the global colors in *sc will be used.
  * out_fcolors - output array of colors (RGBA), computed by this routine.
  * colorveclen - should be either 3 (for RGB) or 4 for (RGBA).  this is
  *       valid for in_fcolors ONLY, as out_fcolors will ALWAYS contain
  *       an alpha component.
  *
  *
  * NOTE: if out_fcolors is not defined upon entry, we
  *     will kmalloc enough space, and do the usual computations.  if it
  *     IS defined, on the other hand, it is assumed that we will be
  *     ADDING the results of the computation into the existing array.
  *     bounds checking (ie, 0..1 in RGBA is done at the end of the
  *     routine).
  *
  * the vertex shade is computed as the sum of:
  *     a. the contribution from the ambient light source
  *     b. for each light source,
  *         1. the contribution from the diffuse component
  *         2. (not yet implemented) the specular component
  *
**/
int vertex_shader(vertex_3d *verts,
		  vertex_3d *normals,
		  int nverts, 
		  surface_colors_rgba *in_fcolors,
		  surface_colors_rgba **out_fcolors, 
		  surface_attribs *sa,
		  float *global_alpha,
		  int shader)
{
    register surface_colors_rgba *rp;
    register int i,n;
    int ltype;
    int nlights,on;
    float w[3],fs;
    vertex_3d *vwork,*vwork2,*vwork3,ldirection,light_color;
    
    float *fwork;
    vertex_3d eyepoint;
    float two = 2.0;

    vwork = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nverts);
    vwork2 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nverts);
    vwork3 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nverts);
    fwork = (float *)kmalloc(sizeof(float)*nverts);
    
    if (*out_fcolors == NULL)
    {
	rp = *out_fcolors = (surface_colors_rgba *)kmalloc(sizeof(surface_colors_rgba)*nverts);

	for (i=0;i<nverts;i++,rp++)
	    rp->r = rp->g = rp->b = 0.;

	if (in_fcolors == NULL)
	    for (i=0,rp= *out_fcolors;i<nverts;i++,rp++)
		rp->a = *global_alpha;
	else
	{
	    /* copy over alpha component from input colors array. */
	    rp = *out_fcolors;
	    for (i=0;i<nverts;i++,rp++)
		rp->a = in_fcolors[i].a * *global_alpha;
		
	}
    }

    /* check to see if we're using the no-light shader.  if so, copy
       over the colors from input to output and return. */

    if (shader == KSHADER_NO_LIGHT)
    {
	i = nverts * sizeof(surface_colors_rgba);
	kmemcpy(*out_fcolors,in_fcolors,i);
	kfree(vwork);
	kfree(vwork2);
	kfree(vwork3);
	kfree(fwork);
	return(CHILL);
    }

#if 0
    /* get the ambient light color, scale by the value in the surface_attribs
       struct, and write that in to the output array. this corresponds to
       scaling the color of each component (R,G,B) of light by the user-
       specified ambient light attentuation coefficients. */

    get_ambient_light_color((vertex_3d *)w);

    n = nverts;
    for (i=0;i<3;i++)
	w[i] *= sa->amb;

    rp = *out_fcolors;
    for (i=0;i<nverts;i++,rp++)
    {
	rp->r += w[0];
	rp->g += w[1];
	rp->b += w[2];
    }
#endif
    /**
      * change in how ambient light term is computed.  the luminance of
      * the ambient light color is used to scale the exist4ing surface
      * color of the object.  the color of the ambient light doesn't
      * matter.  this seems to produce more asthetically pleasing retults.
      * if you want to use the ambient light color, un-ifdef the above
      * lines of code, and ifdef out the following lines of code.
    **/
    get_ambient_light_color((vertex_3d *)w);
    fs = w[0] * 0.3 + w[1] * 0.6 + w[2] * 0.1;
    fs *= sa->amb;
    rp = *out_fcolors;
    for (i=0;i<nverts;i++,rp++)
    {
	rp->r += fs * in_fcolors[i].r;
	rp->g += fs * in_fcolors[i].g;
	rp->b += fs * in_fcolors[i].b;
    }


    /**
      * now, for each light, compute the diffuse component.  this uses
      * the usual Lambertian reflection model.  in this model, the
      * intensity of the diffuse reflection  depends only on the
      * angle between  the direction to the light source and the
      * surface normal.
      *

      * for now, we assume that the sole light source is a directional
      * light source.
    **/
       
    get_num_lights(&nlights);

    /* check to see if the diffuse coefficient in the surface attributes
       is zero. we can skip this stuff if it is. */

    if (sa->diff > 0.)
    {
	for (n=0;n<nlights;n++)
	{
	    get_light_onoff(n,&on);
	    if (on == LIGHT_ON)
	    {
		get_light_type(n,&ltype);
		if (ltype == POINT_LIGHT)
		{
		    get_light_position(n,&ldirection);
#if 0
		    ldirection.v[2] *= -1.; /* change handedness of light */
#endif
		}
		else if ((ltype == DIRECTIONAL_LIGHT) ||
			 (ltype == BIDIRECTIONAL_LIGHT))
		    get_light_direction(n,&ldirection);
		else
		    continue; /* bail on this light */

		/**
		  * compute the direction L to the light from the
		  * i'th vertex.  these are normalized inside the routine.
		  * the results of the L vectors, one per vertex, are
		  * placed in vwork
		**/
		
		(*tolight_funcs[ltype])(verts,&ldirection,vwork,nverts);

		/**
		  * compute dot product of L with Normals (the normals must
		  * have been unitized somewhere else.
		**/
		if (ltype == BIDIRECTIONAL_LIGHT)
		    nvertex_dot_abs(vwork,normals,fwork,nverts);
		else
		    nvertex_dot(vwork,normals,fwork,nverts);

/*		nvertex_dot(vwork,normals,fwork,nverts); */

		/**
		  * now, scale this array of dot products by the diffuse
		  * reflection coefficient specified by the user.
		**/
		f_ax(fwork,fwork,&(sa->diff),nverts);
		
		/**
		  * compute the contribution of diffuse light, using the
		  * dot product from above.
		**/
		nvertex_scaled_shade(*out_fcolors,fwork,in_fcolors,
				     nverts);
	    }
	}
    }
    
    /**
      * now, for each light, compute the specular component.  
      * 
      * we employ the Phong model of computing the specular component.
      * later, in the scanline polygon renderer, we can choose to do
      * color (Gouroud) interpolation, or Phong interpolation where th
      * surface normal is interpolated across the face of the polygon.
      *
      * for now, we assume that the sole light source is a directional
      * light source.
    **/
       
    /* check to see if the diffuse coefficient in the surface attributes
       is zero. we can skip this stuff if it is. */

    if (sa->spec > 0.)
    {
	/**
	  * compute V[i], or "to viewer" vector for each vertex.  these
	  * will remain constant regardless of the light type.
	**/
	get_eye(&eyepoint);
/*	eyepoint.v[2] *= -1; */
	to_point_light_func(verts,&eyepoint,vwork3,nverts);
	
	for (n=0;n<nlights;n++)
	{
	    get_light_onoff(n,&on);
	    if (on == LIGHT_ON)
	    {
		get_light_type(n,&ltype);
		if (ltype == POINT_LIGHT)
		{
		    get_light_position(n,&ldirection);
		}
		else if (ltype == DIRECTIONAL_LIGHT)
		    get_light_direction(n,&ldirection);
		else
		    continue; /* bail on this light */

		get_light_color(n,&(light_color.v[0]),&(light_color.v[1]),
				&(light_color.v[2]));

		/**
		  * compute the direction L to the light from the
		  * i'th vertex.  these are normalized inside the routine.
		  * the results of the L vectors, one per vertex, are
		  * placed in vwork
		**/
		
		(*tolight_funcs[ltype])(verts,&ldirection,vwork,nverts);

		/**
		  * compute dot product of L with Normals (the normals must
		  * have been unitized somewhere else.
		**/
		nvertex_dot(vwork,normals,fwork,nverts);

		/**
		  * in fwork is L dot N[i].  we next scale N by
		  * 2(L dot N[i]).
		**/
		vert_aBX(vwork2,normals,&two,fwork,nverts);

		/**
		  * now, vwork2 contains: 2N[i](N[i] dot L[i])
		  * next, compute (2N[i](N[i] dot L[i]) - L[i])
		**/
		npoints_diff(vwork2,vwork,vwork2,nverts);

		/**
		  * take the dot product of the vectors contained in vwork2
		  * wich represents the reflected light vector from each
		  * vertex, with the vectors representing the to viewer
		  * vector.
		**/
		nvertex_dot(vwork2,vwork3,fwork,nverts);

		/**
		  * scale the dotproducts by the specular reflection
		  * coefficient.
		**/
		f_ax(fwork,fwork,&(sa->spec),nverts);
		
		/**
		  * these dot products represent the cosine of the angle
		  * formed by each pair of "to viewer"/"reflected light"
		  * pairs.  we need to implement raising this to the power
		  * indicated in sa->spec_exp.  for now, skip it...
		**/
		f_pow(fwork,fwork,&(sa->spec_exp),nverts); 

		/**
		  * compute the contribution of diffuse light, using the
		  * dot product from above.
		**/
		nvertex_scaled_shade_spec(*out_fcolors,fwork,in_fcolors,nverts,
					  &(sa->spec_metal),&light_color);
	    }
	}
    }

    
    kfree(vwork);
    kfree(vwork2);
    kfree(vwork3);
    kfree(fwork);

    return(CHILL);
}

/**
  * the following routine is to be used in compute the shade at a list
  * of vertices.
  *
  * verts = input x,y,z points
  * normals = input x,y,z vertex normals.  MUST BE one normal per face.
  * nverts = int, number of vertices to shade
  * in_fcolors = surface_colors_rgba *, possibly null, of color at each
  *      vertex.  if null, then the global colors in *sc will be used.
  * out_fcolors - output array of colors (RGBA), computed by this routine.
  * colorveclen - should be either 3 (for RGB) or 4 for (RGBA).  this is
  *       valid for in_fcolors ONLY, as out_fcolors will ALWAYS contain
  *       an alpha component.
  *
  *
  * NOTE: if out_fcolors is not defined upon entry, we
  *     will kmalloc enough space, and do the usual computations.  if it
  *     IS defined, on the other hand, it is assumed that we will be
  *     ADDING the results of the computation into the existing array.
  *     bounds checking (ie, 0..1 in RGBA is done at the end of the
  *     routine).
  *
  * the vertex shade is computed as the sum of:
  *     a. the contribution from the ambient light source
  *     b. for each light source,
  *         1. the contribution from the diffuse component
  *         2. (not yet implemented) the specular component
  *
  * there are special problems caused by this configuration of colors and
  * normals.  the output array of colors is basically 3*nvertices, because
  * each vertex can be a member of up to three different faces.
  *
  * the ambient term is calculated on a per-vertex basis first.
  * next, the work required for the diffuse and specular components
  * are done on a per-face basis.  finally, when the 3x expansion is
  * done, the various components of the final vertex shade are reassembled.
**/

int vertex_per_face_normal_shader(vertex_3d *verts,
				  vertex_3d *normals,
				  int nverts, 
				  surface_colors_rgba *in_fcolors,
				  surface_colors_rgba **out_fcolors, 
				  surface_attribs *sa,
				  float *global_alpha,
				  int shader)
{
    register surface_colors_rgba *rp;
    register int i,n;
    int ltype;
    int nlights,on;
    float w[3],fs;
    vertex_3d *vwork,*vwork2,*vwork3,ldirection,light_color;
    vertex_3d *loc_verts;
    int loc_nverts;
    int out_ncolors;
    int j;
    
    float *fwork;
    vertex_3d eyepoint;
    float two = 2.0;

    vwork = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nverts);
    vwork2 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nverts);
    vwork3 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nverts);
    fwork = (float *)kmalloc(sizeof(float)*nverts);
    
    if (*out_fcolors == NULL)
    {
	/* expand from tstrip per vertex colors to something that
	 looks like disjoint triangles per-vertex colors */
	
	out_ncolors = (nverts-2)*3; /* # faces * # verts per face */
	
	rp = *out_fcolors = (surface_colors_rgba *)kmalloc(sizeof(surface_colors_rgba)*out_ncolors);

	/*  set them all to zero */
	for (i=0;i<out_ncolors;i++,rp++)
	    rp->r = rp->g = rp->b = 0.;

	if (in_fcolors == NULL)
	    for (i=0,rp= *out_fcolors;i<out_ncolors;i++,rp++)
		rp->a = *global_alpha;
	else
	{
	    /* copy over alpha component from input colors array. */
	    rp = *out_fcolors;
	    for (i=0;i<out_ncolors;i++,rp++)
		rp->a = in_fcolors[(i/3) + (i%3)].a * *global_alpha;
	}
    }

    loc_nverts = nverts-2;
    loc_verts = (vertex_3d *)kmalloc(sizeof(vertex_3d)*(loc_nverts));

    for (i=0;i<loc_nverts;i++)
    {
	for (j=0;j<3;j++)
	    loc_verts[i].v[j] = (verts[i].v[j] + verts[i+1].v[j] +
				 verts[i+2].v[j]) * 0.33333333334;
    }

    /* check to see if we're using the no-light shader.  if so, copy
       over the colors from input to output and return. */

    if (shader == KSHADER_NO_LIGHT)
    {
	rp = *out_fcolors;
	for (i=0;i<out_ncolors;i++,rp++)
	    kmemcpy(rp,in_fcolors+((i/3)+(i%3)),sizeof(surface_colors_rgba));
	
	kfree(vwork);
	kfree(vwork2);
	kfree(vwork3);
	kfree(fwork);
	kfree(loc_verts);
	return(CHILL);
    }

#if 0
    /* get the ambient light color, scale by the value in the surface_attribs
       struct, and write that in to the output array. this corresponds to
       scaling the color of each component (R,G,B) of light by the user-
       specified ambient light attentuation coefficients. */

    get_ambient_light_color((vertex_3d *)w);

    n = nverts;
    for (i=0;i<3;i++)
	w[i] *= sa->amb;

    rp = *out_fcolors;
    for (i=0;i<out_ncolors;i++,rp++)
    {
	rp->r += w[0];
	rp->g += w[1];
	rp->b += w[2];
    }
#endif
    /**
      * change in how ambient light term is computed.  the luminance of
      * the ambient light color is used to scale the exist4ing surface
      * color of the object.  the color of the ambient light doesn't
      * matter.  this seems to produce more asthetically pleasing retults.
      * if you want to use the ambient light color, un-ifdef the above
      * lines of code, and ifdef out the following lines of code.
    **/
    get_ambient_light_color((vertex_3d *)w);
    fs = w[0] * 0.3 + w[1] * 0.6 + w[2] * 0.1;
    fs *= sa->amb;
    rp = *out_fcolors;
    for (i=0;i<out_ncolors;i++,rp++)
    {
	rp->r += fs * in_fcolors[(i/3) + (i%3)].r;
	rp->g += fs * in_fcolors[(i/3) + (i%3)].g;
	rp->b += fs * in_fcolors[(i/3) + (i%3)].b;
    }


    /**
      * now, for each light, compute the diffuse component.  this uses
      * the usual Lambertian reflection model.  in this model, the
      * intensity of the diffuse reflection  depends only on the
      * angle between  the direction to the light source and the
      * surface normal.
      *

      * for now, we assume that the sole light source is a directional
      * light source.
    **/
       
    get_num_lights(&nlights);

    /* check to see if the diffuse coefficient in the surface attributes
       is zero. we can skip this stuff if it is. */

    if (sa->diff > 0.)
    {
	for (n=0;n<nlights;n++)
	{
	    get_light_onoff(n,&on);
	    if (on == LIGHT_ON)
	    {
		get_light_type(n,&ltype);
		if (ltype == POINT_LIGHT)
		{
		    get_light_position(n,&ldirection);
#if 0
		    ldirection.v[2] *= -1.; /* change handedness of light */
#endif
		}
		else if ((ltype == DIRECTIONAL_LIGHT) ||
			 (ltype == BIDIRECTIONAL_LIGHT))
		    get_light_direction(n,&ldirection);
		else
		    continue; /* bail on this light */

		/**
		  * compute the direction L to the light from the
		  * i'th vertex.  these are normalized inside the routine.
		  * the results of the L vectors, one per vertex, are
		  * placed in vwork
		**/
		
		(*tolight_funcs[ltype])(loc_verts,&ldirection,vwork,loc_nverts);

		/**
		  * compute dot product of L with Normals (the normals must
		  * have been unitized somewhere else.
		  *
		  * nverts-2 because we have per-face normals
		**/
		if (ltype == BIDIRECTIONAL_LIGHT)
		    nvertex_dot_abs(vwork,normals,fwork,loc_nverts);
		else
		    nvertex_dot(vwork,normals,fwork,loc_nverts);

		/**
		  * now, scale this array of dot products by the diffuse
		  * reflection coefficient specified by the user.
		**/
		f_ax(fwork,fwork,&(sa->diff),loc_nverts);
		
		/**
		  * compute the contribution of diffuse light, using the
		  * dot product from above.  note that we assume
		  * per-vertex input colors.
		**/

		nvertex_scaled_shade_per_face_norms(*out_fcolors,fwork,
						    in_fcolors,
						    loc_nverts);

	    }
	}
    }
    
    /**
      * now, for each light, compute the specular component.  
      * 
      * we employ the Phong model of computing the specular component.
      * later, in the scanline polygon renderer, we can choose to do
      * color (Gouroud) interpolation, or Phong interpolation where th
      * surface normal is interpolated across the face of the polygon.
      *
      * for now, we assume that the sole light source is a directional
      * light source.
    **/
       
    /* check to see if the diffuse coefficient in the surface attributes
       is zero. we can skip this stuff if it is. */

    if (sa->spec > 0.)
    {
	/**
	  * compute V[i], or "to viewer" vector for each vertex.  these
	  * will remain constant regardless of the light type.
	**/
	get_eye(&eyepoint);
/*	eyepoint.v[2] *= -1; */
	to_point_light_func(verts,&eyepoint,vwork3,nverts);
	
	for (n=0;n<nlights;n++)
	{
	    get_light_onoff(n,&on);
	    if (on == LIGHT_ON)
	    {
		get_light_type(n,&ltype);
		if (ltype == POINT_LIGHT)
		{
		    get_light_position(n,&ldirection);
		}
		else if (ltype == DIRECTIONAL_LIGHT)
		    get_light_direction(n,&ldirection);
		else
		    continue; /* bail on this light */

		get_light_color(n,&(light_color.v[0]),&(light_color.v[1]),
				&(light_color.v[2]));

		/**
		  * compute the direction L to the light from the
		  * i'th vertex.  these are normalized inside the routine.
		  * the results of the L vectors, one per vertex, are
		  * placed in vwork
		**/
		
		(*tolight_funcs[ltype])(loc_verts,&ldirection,vwork,loc_nverts);

		/**
		  * compute dot product of L with Normals (the normals must
		  * have been unitized somewhere else.
		**/
		nvertex_dot(vwork,normals,fwork,loc_nverts);

		/**
		  * in fwork is L dot N[i].  we next scale N by
		  * 2(L dot N[i]).
		**/
		vert_aBX(vwork2,normals,&two,fwork,loc_nverts);

		/**
		  * now, vwork2 contains: 2N[i](N[i] dot L[i])
		  * next, compute (2N[i](N[i] dot L[i]) - L[i])
		**/
		npoints_diff(vwork2,vwork,vwork2,loc_nverts);

		/**
		  * take the dot product of the vectors contained in vwork2
		  * wich represents the reflected light vector from each
		  * vertex, with the vectors representing the to viewer
		  * vector.
		**/
		nvertex_dot(vwork2,vwork3,fwork,loc_nverts);

		/**
		  * scale the dotproducts by the specular reflection
		  * coefficient.
		**/
		f_ax(fwork,fwork,&(sa->spec),loc_nverts);
		
		/**
		  * these dot products represent the cosine of the angle
		  * formed by each pair of "to viewer"/"reflected light"
		  * pairs.  we need to implement raising this to the power
		  * indicated in sa->spec_exp.  for now, skip it...
		**/
		f_pow(fwork,fwork,&(sa->spec_exp),loc_nverts); 

		/**
		  * compute the contribution of diffuse light, using the
		  * dot product from above.
		**/
		nvertex_scaled_shade_spec_per_face_norms(*out_fcolors,
							 fwork,in_fcolors,
							 loc_nverts,
							 &(sa->spec_metal),
							 &light_color);
	    }
	}
    }

    
    kfree(vwork);
    kfree(vwork2);
    kfree(vwork3);
    kfree(fwork);
    kfree(loc_verts);

    return(CHILL);
}


/**
  * the following routine is to be used in compute the shade on a per-facet
  * basis for an array of vertices which form a connected triangle strip.
  *
  * nverts = int, number of vertices
  * verts = input x,y,z points, of which there are "nverts".
  * normals = input x,y,z facet normals.  there will be "nverts-2" of these.
  * in_fcolors = surface_colors_rgba *, possibly null, of color at each
  *      vertex.  if null, then the global colors in *sc will be used.
  * out_fcolors - output array of colors (RGBA), computed by this routine.
  * colorveclen - should be either 3 (for RGB) or 4 for (RGBA).  this is
  *       valid for in_fcolors ONLY, as out_fcolors will ALWAYS contain
  *       an alpha component.
  *
  * NOTE: if out_fcolors is not defined upon entry, we
  *     will kmalloc enough space, and do the usual computations.  if it
  *     IS defined, on the other hand, it is assumed that we will be
  *     ADDING the results of the computation into the existing array.
  *     bounds checking (ie, 0..1 in RGBA is done at the end of the
  *     routine).
  *
  * the facet shade is computed as the sum of:
  *     a. the contribution from the ambient light source
  *     b. for each light source,
  *         1. the contribution from the diffuse component
  *         2. (not yet implemented) the specular component
  *
**/
int tstrip_facet_shader(vertex_3d *verts,
			vertex_3d *normals,
			int nverts, 
			surface_colors_rgba *in_fcolors,
			surface_colors_rgba **out_fcolors, 
			surface_attribs *sa,
			float *global_alpha,
			int shader)
{
    register surface_colors_rgba *rp;
    register int i,n,j;
    int ltype;
    int nlights,on;
    float w[3],fs;
    vertex_3d *vwork,*vwork2,*vwork3,ldirection,light_color;
    vertex_3d *centers;
    int ntriangles;
    
    float *fwork;
    vertex_3d eyepoint;
    float two = 2.0;

    ntriangles = nverts-2;

    /* misc scratch arrays. */
    vwork = (vertex_3d *)kmalloc(sizeof(vertex_3d)*ntriangles);
    vwork2 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*ntriangles);
    vwork3 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*ntriangles);
    fwork = (float *)kmalloc(sizeof(float)*nverts);

    /* array containing vertex averages, "centroid" of each triangle. */
    centers = (vertex_3d *)kmalloc(sizeof(vertex_3d)*ntriangles);
    
    if (*out_fcolors == NULL)
    {
	rp = *out_fcolors = (surface_colors_rgba *)kmalloc(sizeof(surface_colors_rgba)*ntriangles);

	for (i=0;i<ntriangles;i++,rp++)
	    rp->r = rp->g = rp->b = 0.;

	if (in_fcolors == NULL)
	    for (i=0,rp= *out_fcolors;i<ntriangles;i++,rp++)
		rp->a = *global_alpha;
	else
	{
	    /* copy over alpha component from input colors array. */
	    rp = *out_fcolors;
	    for (i=0;i<ntriangles;i++,rp++)
		rp->a = in_fcolors[i].a * *global_alpha;
		
	}
    }

    /* check to see if we're using the no-light shader.  if so, copy
       over the colors from input to output and return. */

    if (shader == KSHADER_NO_LIGHT)
    {
	i = ntriangles * sizeof(surface_colors_rgba);
	kmemcpy(*out_fcolors,in_fcolors,i);
	kfree(vwork);
	kfree(vwork2);
	kfree(vwork3);
	kfree(fwork);
	kfree(centers);
	return(CHILL);
    }

    /**
      * change in how ambient light term is computed.  the luminance of
      * the ambient light color is used to scale the exist4ing surface
      * color of the object.  the color of the ambient light doesn't
      * matter.  this seems to produce more asthetically pleasing retults.
      * if you want to use the ambient light color, un-ifdef the above
      * lines of code, and ifdef out the following lines of code.
    **/
    get_ambient_light_color((vertex_3d *)w);
    fs = w[0] * 0.3 + w[1] * 0.6 + w[2] * 0.1;
    fs *= sa->amb;
    rp = *out_fcolors;
    for (i=0;i<ntriangles;i++,rp++)
    {
	rp->r += fs * in_fcolors[i].r;
	rp->g += fs * in_fcolors[i].g;
	rp->b += fs * in_fcolors[i].b;
    }

    /**
      * average the vertices.
    **/
    for (i=0;i<ntriangles;i++)
    {
	for (j=0;j<3;j++)
	{
	    centers[i].v[j] = verts[i].v[j] + verts[i+1].v[j] + verts[i+2].v[j];
	    centers[i].v[j] *= 0.33333333333;
	}
    }


    /**
      * now, for each light, compute the diffuse component.  this uses
      * the usual Lambertian reflection model.  in this model, the
      * intensity of the diffuse reflection  depends only on the
      * angle between  the direction to the light source and the
      * surface normal.
      *

      * for now, we assume that the sole light source is a directional
      * light source.
    **/
       
    get_num_lights(&nlights);

    /* check to see if the diffuse coefficient in the surface attributes
       is zero. we can skip this stuff if it is. */

    if (sa->diff > 0.)
    {
	for (n=0;n<nlights;n++)
	{
	    get_light_onoff(n,&on);
	    if (on == LIGHT_ON)
	    {
		get_light_type(n,&ltype);
		if (ltype == POINT_LIGHT)
		{
		    get_light_position(n,&ldirection);
#if 0
		    ldirection.v[2] *= -1.; /* change handedness of light */
#endif
		}
		else if ((ltype == DIRECTIONAL_LIGHT) ||
			 (ltype == BIDIRECTIONAL_LIGHT))
		    get_light_direction(n,&ldirection);
		else
		    continue; /* bail on this light */

		/**
		  * compute the direction L to the light from the
		  * i'th vertex.  these are normalized inside the routine.
		  * the results of the L vectors, one per vertex, are
		  * placed in vwork
		**/

		(*tolight_funcs[ltype])(centers,&ldirection,vwork,ntriangles);

		/**
		  * compute dot product of L with Normals (the normals must
		  * have been unitized somewhere else.
		**/
		if (ltype == BIDIRECTIONAL_LIGHT)
		    nvertex_dot_abs(vwork,normals,fwork,ntriangles);
		else
		    nvertex_dot(vwork,normals,fwork,ntriangles);

		/**
		  * now, scale this array of dot products by the diffuse
		  * reflection coefficient specified by the user.
		**/
		f_ax(fwork,fwork,&(sa->diff),ntriangles);
		
		/**
		  * compute the contribution of diffuse light, using the
		  * dot product from above.
		**/
		nvertex_scaled_shade(*out_fcolors,fwork,in_fcolors,
				     ntriangles);
	    }
	}
    }
    
    /**
      * now, for each light, compute the specular component.  
      * 
      * we employ the Phong model of computing the specular component.
      * later, in the scanline polygon renderer, we can choose to do
      * color (Gouroud) interpolation, or Phong interpolation where th
      * surface normal is interpolated across the face of the polygon.
      *
      * for now, we assume that the sole light source is a directional
      * light source.
    **/
       
    /* check to see if the diffuse coefficient in the surface attributes
       is zero. we can skip this stuff if it is. */

    if (sa->spec > 0.)
    {
	/**
	  * compute V[i], or "to viewer" vector for each vertex.  these
	  * will remain constant regardless of the light type.
	**/
	get_eye(&eyepoint);
/*	eyepoint.v[2] *= -1; */

	to_point_light_func(centers,&eyepoint,vwork3,ntriangles);
	
	for (n=0;n<nlights;n++)
	{
	    get_light_onoff(n,&on);
	    if (on == LIGHT_ON)
	    {
		get_light_type(n,&ltype);
		if (ltype == POINT_LIGHT)
		{
		    get_light_position(n,&ldirection);
		}
		else if (ltype == DIRECTIONAL_LIGHT)
		    get_light_direction(n,&ldirection);
		else
		    continue; /* bail on this light */

		get_light_color(n,&(light_color.v[0]),&(light_color.v[1]),
				&(light_color.v[2]));

		/**
		  * compute the direction L to the light from the
		  * i'th vertex.  these are normalized inside the routine.
		  * the results of the L vectors, one per vertex, are
		  * placed in vwork
		**/

		(*tolight_funcs[ltype])(centers,&ldirection,vwork,ntriangles);

		/**
		  * compute dot product of L with Normals (the normals must
		  * have been unitized somewhere else.
		**/
		nvertex_dot(vwork,normals,fwork,ntriangles);

		/**
		  * in fwork is L dot N[i].  we next scale N by
		  * 2(L dot N[i]).
		**/
		vert_aBX(vwork2,normals,&two,fwork,ntriangles);

		/**
		  * now, vwork2 contains: 2N[i](N[i] dot L[i])
		  * next, compute (2N[i](N[i] dot L[i]) - L[i])
		**/
		npoints_diff(vwork2,vwork,vwork2,ntriangles);

		/**
		  * take the dot product of the vectors contained in vwork2
		  * wich represents the reflected light vector from each
		  * vertex, with the vectors representing the to viewer
		  * vector.
		**/
		nvertex_dot(vwork2,vwork3,fwork,ntriangles);

		/**
		  * scale the dotproducts by the specular reflection
		  * coefficient.
		**/
		f_ax(fwork,fwork,&(sa->spec),ntriangles);
		
		/**
		  * these dot products represent the cosine of the angle
		  * formed by each pair of "to viewer"/"reflected light"
		  * pairs.  we need to implement raising this to the power
		  * indicated in sa->spec_exp.  for now, skip it...
		**/
		f_pow(fwork,fwork,&(sa->spec_exp),ntriangles); 

		/**
		  * compute the contribution of diffuse light, using the
		  * dot product from above.
		**/
		nvertex_scaled_shade_spec(*out_fcolors,fwork,in_fcolors,ntriangles,
					  &(sa->spec_metal),&light_color);
	    }
	}
    }

    kfree(centers);
    kfree(vwork);
    kfree(vwork2);
    kfree(vwork3);
    kfree(fwork);

    return(CHILL);
}

/** the following routines ought to be in lights.c?? **/
static void
to_point_light_func(vertex_3d *verts,
		    vertex_3d *L,
		    vertex_3d *out,
		    int nverts)
{
    register float *f1,*f2,fs;
    double dummy;
    int i;

    /**
      * compute: out[i] = L - verts[i]
    **/

    f1 = (float *)&(verts->v[0]);
    f2 = (float *)&(out->v[0]);
    fs = L->v[0];
    for (i=0;i<nverts;i++)
    {
	*f2++ = L->v[0] - *f1++;
	*f2++ = L->v[1] - *f1++;
	*f2++ = L->v[2] - *f1++;
    }

    /* now normalize them */
    
    for (i=0;i<nverts;i++)
	vertex_unit(out+i,&dummy);

}

static void
to_directional_light_func(vertex_3d *verts,
			  vertex_3d *L,
			  vertex_3d *out,
			  int nverts)
{
    /* assume L is normalized */
    int n,i;
    vertex_3d l;

    n = sizeof(vertex_3d);
    
    for (i=0;i<nverts;i++)
	kmemcpy(out++,L,n);
}

void
nvertex_scaled_shade(surface_colors_rgba *out,
		     float *scalar,
		     surface_colors_rgba *in,
		     int n)
{
    /**
      * compute out[i].[r,g,b] += s[i]*in[i].[r,g,b].
    **/

    register float *d,*s;
    register int i,j;

    d = (float *)&(out->r);
    s = (float *)&(in->r);

    for (i=0;i<n;i++,d+=4,s+=4)
    {
	d[0] += s[0] * scalar[i];
	d[1] += s[1] * scalar[i];
	d[2] += s[2] * scalar[i];
    }
    
    d = (float *)&(out->r);
    for (i=0;i<n;i++,d+=4,s+=4)
    {
	d[0] = (d[0] > 1.) ? 1. : (d[0] < 0.) ? 0 : d[0];
	d[1] = (d[1] > 1.) ? 1. : (d[1] < 0.) ? 0 : d[1];
	d[2] = (d[2] > 1.) ? 1. : (d[2] < 0.) ? 0 : d[2];
    }
}

void
nvertex_scaled_shade_per_face_norms(surface_colors_rgba *out,
				    float *scalar,
				    surface_colors_rgba *in,
				    int n)
{
    /**
      * compute out[i].[r,g,b] += s[i]*in[i].[r,g,b].
    **/

    register float *d;
    register int i,j;

    for (i=0;i<n*3;i++)
    {
	out[i].r += in[(i/3) + (i%3)].r * scalar[i/3];
	out[i].g += in[(i/3) + (i%3)].g * scalar[i/3];
	out[i].b += in[(i/3) + (i%3)].b * scalar[i/3];
    }
    
    d = (float *)&(out->r);
    for (i=0;i<n*3;i++,d+=4)
    {
	d[0] = (d[0] > 1.) ? 1. : (d[0] < 0.) ? 0 : d[0];
	d[1] = (d[1] > 1.) ? 1. : (d[1] < 0.) ? 0 : d[1];
	d[2] = (d[2] > 1.) ? 1. : (d[2] < 0.) ? 0 : d[2];
    }
}

int
volume_vertex_shader(vertex_3d *verts,
		     vertex_3d *normals,
		     int nverts, 
		     surface_colors_rgba *in_fcolors,
		     surface_colors_rgba **out_fcolors, 
		     surface_attribs *sa,
		     float *global_alpha,
		     int shader)
{
    register surface_colors_rgba *rp;
    register int i,n;
    int ltype;
    int nlights,on;
    float w[3],fs;
    vertex_3d *vwork,*vwork2,*vwork3,ldirection;
    float *fwork;
    vertex_3d eyepoint;
    float two = 2.0;

    vwork = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nverts);
    vwork2 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nverts);
    vwork3 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nverts);
    fwork = (float *)kmalloc(sizeof(float)*nverts);
    
    if (*out_fcolors == NULL)
    {
	rp = *out_fcolors = (surface_colors_rgba *)kmalloc(sizeof(surface_colors_rgba)*nverts);
	
	for (i=0;i<nverts;i++,rp++)
	    rp->r = rp->g = rp->b = 0.;


	/*** HACK HACK HACK : copy over stuff from input colors to
	 output colors **/

	i = sizeof(surface_colors_rgba)*nverts;
	kmemcpy(*out_fcolors,in_fcolors,i);
	
	if (in_fcolors == NULL)
	    for (i=0,rp= *out_fcolors;i<nverts;i++,rp++)
		rp->a = *global_alpha;
	else
	{
	    /* copy over alpha component from input colors array. */
	    rp = *out_fcolors;
	    for (i=0;i<nverts;i++,rp++)
		rp->a = in_fcolors[i].a * *global_alpha;
		
	}
    }

#if 0
    /* check to see if we're using the no-light shader.  if so, copy
       over the colors from input to output and return. */

    if (shader == KSHADER_NO_LIGHT)
    {
	i = nverts * sizeof(surface_colors_rgba);
	kmemcpy(*out_fcolors,in_fcolors,i);
	kfree(vwork);
	kfree(vwork2);
	kfree(vwork3);
	kfree(fwork);
	return(CHILL);
    }

    /* get the ambient light color, scale by the value in the surface_attribs
       struct, and write that in to the output array. this corresponds to
       scaling the color of each component (R,G,B) of light by the user-
       specified ambient light attentuation coefficients. */

    get_ambient_light_color((vertex_3d *)w);

    n = nverts;
    for (i=0;i<3;i++)
	w[i] *= sa->amb;

    rp = *out_fcolors;
    for (i=0;i<nverts;i++,rp++)
    {
	rp->r += w[0];
	rp->g += w[1];
	rp->b += w[2];
    }

    /**
      * now, for each light, compute the diffuse component.  this uses
      * the usual Lambertian reflection model.  in this model, the
      * intensity of the diffuse reflection  depends only on the
      * angle between  the direction to the light source and the
      * surface normal.
      *

      * for now, we assume that the sole light source is a directional
      * light source.
    **/
       
    get_num_lights(&nlights);

    /* check to see if the diffuse coefficient in the surface attributes
       is zero. we can skip this stuff if it is. */

    if (sa->diff > 0.)
    {
	for (n=0;n<nlights;n++)
	{
	    get_light_onoff(n,&on);
	    if (on == LIGHT_ON)
	    {
		get_light_type(n,&ltype);
		if (ltype == POINT_LIGHT)
		{
		    get_light_position(n,&ldirection);
		}
		else if (ltype == DIRECTIONAL_LIGHT)
		    get_light_direction(n,&ldirection);
		else
		    continue; /* bail on this light */

		/**
		  * compute the direction L to the light from the
		  * i'th vertex.  these are normalized inside the routine.
		  * the results of the L vectors, one per vertex, are
		  * placed in vwork
		**/
		
		(*tolight_funcs[ltype])(verts,&ldirection,vwork,nverts);

		/**
		  * compute dot product of L with Normals (the normals must
		  * have been unitized somewhere else.
		**/
		if (ltype == BIDIRECTIONAL_LIGHT)
		    nvertex_dot_abs(vwork,normals,fwork,nverts);
		else
		    nvertex_dot(vwork,normals,fwork,nverts);


		/**
		  * now, scale this array of dot products by the diffuse
		  * reflection coefficient specified by the user.
		**/
		f_ax(fwork,fwork,&(sa->diff),nverts);
		
		/**
		  * compute the contribution of diffuse light, using the
		  * dot product from above.
		**/
		nvertex_scaled_shade(*out_fcolors,fwork,in_fcolors,
				     nverts);
	    }
	}
    }
    
    /**
      * now, for each light, compute the specular component.  
      * 
      * we employ the Phong model of computing the specular component.
      * later, in the scanline polygon renderer, we can choose to do
      * color (Gouroud) interpolation, or Phong interpolation where th
      * surface normal is interpolated across the face of the polygon.
      *
      * for now, we assume that the sole light source is a directional
      * light source.
    **/
       
    /* check to see if the diffuse coefficient in the surface attributes
       is zero. we can skip this stuff if it is. */

    if (sa->spec > 0.)
    {
	/**
	  * compute V[i], or ~to viewer" vector for each vertex.  these
	  * will remain constant regardless of the light type.
	**/
	get_eye(&eyepoint);
	to_point_light_func(verts,&eyepoint,vwork3,nverts);
	
	for (n=0;n<nlights;n++)
	{
	    get_light_onoff(n,&on);
	    if (on == LIGHT_ON)
	    {
		get_light_type(n,&ltype);
		if (ltype == POINT_LIGHT)
		{
		    get_light_position(n,&ldirection);
		}
		else if ((ltype == DIRECTIONAL_LIGHT) ||
			 (ltype == BIDIRECTIONAL_LIGHT))
		    get_light_direction(n,&ldirection);
		else
		    continue; /* bail on this light */

		/**
		  * compute the direction L to the light from the
		  * i'th vertex.  these are normalized inside the routine.
		  * the results of the L vectors, one per vertex, are
		  * placed in vwork
		**/
		
		(*tolight_funcs[ltype])(verts,&ldirection,vwork,nverts);

		/**
		  * compute dot product of L with Normals (the normals must
		  * have been unitized somewhere else.
		**/

		if (ltype == BIDIRECTIONAL_LIGHT)
		    nvertex_dot_abs(vwork,normals,fwork,nverts);
		else
		    nvertex_dot(vwork,normals,fwork,nverts);

		/**
		  * in fwork is L dot N[i].  we next scale N by
		  * 2(L dot N[i]).
		**/
		i = nverts * sizeof(vertex_3d);
		vert_aBX(vwork2,normals,&two,fwork,nverts);

		/**
		  * now, vwork2 contains: 2N[i](N[i] dot L[i])
		  * next, compute (2N[i](N[i] dot L[i]) - L[i])
		**/
		npoints_diff(vwork2,vwork,vwork2,nverts);

		/**
		  * take the dot product of the vectors contained in vwork2
		  * wich represents the reflected light vector from each
		  * vertex, with the vectors representing the to viewer
		  * vector.
		**/
		nvertex_dot(vwork2,vwork3,fwork,nverts);

		/**
		  * these dot products represent the cosine of the angle
		  * formed by each pair of "to viewer"/"reflected light"
		  * pairs.  we need to implement raising this to the power
		  * indicated in sa->spec_exp.  for now, skip it...
		**/
		f_ax(fwork,fwork,&(sa->spec),nverts);
		
		/**
		  * compute the contribution of diffuse light, using the
		  * dot product from above.
		**/
		nvertex_scaled_shade(*out_fcolors,fwork,in_fcolors,
				     nverts);
	    }
	}
    }

#endif    
    kfree(vwork);
    kfree(vwork2);
    kfree(vwork3);
    kfree(fwork);

    return(CHILL);
}


void
blend_vertex_shade(surface_colors_rgba *d,
		   float *blend_coeff,
		   vertex_3d *blend_color,
		   int nverts)
{
    /**
      * compute d[i] = (1-blend_coeff)*d[i] + (blend_coeff)*blend_color,
      *   i=0,nverts-1
    **/
    register float *f,c;
    register int i,j;
    vertex_3d metal_color;

    for (i=0;i<3;i++)
	metal_color.v[i] = (1. - *blend_coeff) * blend_color->v[i];

    f = &(d->r);
    c = *blend_coeff;
    
    for (i=0;i<nverts;i++)
    {
	for (j=0;j<3;j++)
	{
	    *f = *f * c + metal_color.v[j];
	    f++;
	}
	f++;
    }
}


void
nvertex_scaled_shade_spec(surface_colors_rgba *out,
			  float *scalar,
			  surface_colors_rgba *in,
			  int n,
			  float *blend_coeff,
			  vertex_3d *blend_color )
{
    /**
      * (metal == blend_coeff)
      * spec_color[i] = (1-metal)*light_color + metal*object_color
      * compute out[i].[r,g,b] += s[i]*spec_color[i]
    **/

    register float *d,*s,bc;
    register int i,j;
    vertex_3d light_color_component;

    d = (float *)&(out->r);
    s = (float *)&(in->r);

    for (i=0;i<3;i++)
	light_color_component.v[i] = (1.- *blend_coeff) * blend_color->v[i];

    bc = *blend_coeff;

    for (i=0;i<n;i++,d+=4,s+=4)
    {
	d[0] += scalar[i] * (s[0]*bc + light_color_component.v[0]);
	d[1] += scalar[i] * (s[1]*bc + light_color_component.v[1]);
	d[2] += scalar[i] * (s[2]*bc + light_color_component.v[2]);
    }
    
    d = (float *)&(out->r);
    for (i=0;i<n;i++,d+=4,s+=4)
    {
	d[0] = (d[0] > 1.) ? 1. : (d[0] < 0.) ? 0 : d[0];
	d[1] = (d[1] > 1.) ? 1. : (d[1] < 0.) ? 0 : d[1];
	d[2] = (d[2] > 1.) ? 1. : (d[2] < 0.) ? 0 : d[2];
    }
}

void
nvertex_scaled_shade_spec_per_face_norms(surface_colors_rgba *out,
					 float *scalar,
					 surface_colors_rgba *in,
					 int n,
					 float *blend_coeff,
					 vertex_3d *blend_color )
{
    /**
      * (metal == blend_coeff)
      * spec_color[i] = (1-metal)*light_color + metal*object_color
      * compute out[i].[r,g,b] += s[i]*spec_color[i]
    **/

    register float *d,*s,bc;
    register int i,j,index;
    vertex_3d light_color_component;

    d = (float *)&(out->r);
    s = (float *)&(in->r);

    for (i=0;i<3;i++)
	light_color_component.v[i] = (1.- *blend_coeff) * blend_color->v[i];

    bc = *blend_coeff;

    for (i=0;i<n*3;i++)
    {
	index = (i/3) + (i%3);
	out[i].r += scalar[i/3] * (in[index].r * bc + light_color_component.v[0]);
	out[i].g += scalar[i/3] * (in[index].g * bc + light_color_component.v[1]);
	out[i].b += scalar[i/3] * (in[index].b * bc + light_color_component.v[2]);
#if 0
	d[0] += scalar[i] * (s[0]*bc + light_color_component.v[0]);
	d[1] += scalar[i] * (s[1]*bc + light_color_component.v[1]);
	d[2] += scalar[i] * (s[2]*bc + light_color_component.v[2]);
#endif
    }
    
    d = (float *)&(out->r);
    for (i=0;i<n*3;i++,d+=4)
    {
	d[0] = (d[0] > 1.) ? 1. : (d[0] < 0.) ? 0 : d[0];
	d[1] = (d[1] > 1.) ? 1. : (d[1] < 0.) ? 0 : d[1];
	d[2] = (d[2] > 1.) ? 1. : (d[2] < 0.) ? 0 : d[2];
    }
}
