
/*********************************************************************

Copyright (C) 1993, Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus is is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory

  "this software is 100% hand-crafted by a human being in the USA"

*********************************************************************/


#include <design.h>
#include <geometry.h>
#include "rmonster.h"
#include "poly.h"
#include "rmatrix.h"
#include "camera.h"
#include "imaging.h"
#include "sort_trans.h"
#include "zbuffer.h"
#include "rmonster_mouse.h"

extern void (*software_primfuncs[])();

static void render_opaque_subtree PROTO((rmonsterobj *, matrix4x4 *, matrix4x4 *, int, int, int, vertex_3d *, int, int));
static void render_transparent_subtree PROTO((rmonsterobj *, matrix4x4 *, matrix4x4 *, int, int, int, vertex_3d *, int, int));
static int is_oqaque PROTO((rmonsterobj *));
int have_any_transparent_objs PROTO((rmonsterobj *r));

int is_opaque PROTO((rmonsterobj *));
int map_to_renderprims PROTO((int));

int
software_render_image()
{
    /* see also: gl_render_func in gl.c */
    rmonsterobj *r;
    matrix4x4 xfrm,view;
    vertex_3d trans_vec;
    int i,have_transparent_objs;
    int w,h;
    int projection;
    int hither_flag, yon_flag;
    int image_display_choice;

    hither_flag = 1;		/* front clipping always on */
    get_yon_clip_flag(&yon_flag);

    get_image_display_choice(&image_display_choice);

    if (image_display_choice != sNO_DISPLAY) 
       show_render_cursor();
    
    clear_framebuffer();

    get_image_dims(&w,&h);
    
    identity_4x4(&xfrm);
    get_view_xform(&view);
    reset_zbuffer();
    reset_abuffer();
    init_bucket_zbuffer(w,h);

    get_projection(&projection);
    r = get_root_object();

    have_transparent_objs = have_any_transparent_objs(r);

    trans_vec.v[0] = trans_vec.v[1] = trans_vec.v[2] = 0.;

    /* render the opaque objects first */
    render_opaque_subtree(r,&xfrm,&view,w,h,projection,&trans_vec,
			  hither_flag,yon_flag); 
    
    if (have_transparent_objs == TRUE)
    {
	render_transparent_subtree(r,&xfrm,&view,w,h,projection,&trans_vec,
				   hither_flag,yon_flag);
	composite_buckets();
    }

    if (image_display_choice != sNO_DISPLAY) 
       clear_render_cursor();
    flush_output();
    
    return(CHILL);
}

static void
render_opaque_subtree(rmonsterobj *r,
		      matrix4x4 *xfrm,
		      matrix4x4 *view,
		      int w,int h,
		      int projection,
		      vertex_3d *tv,
		      int hither_flag,
		      int yon_flag)
{
    int status;
    int display_status,opaque_status;
    int nprims,i;
    matrix4x4 m;
    vertex_3d trans_vec;
    rmonster_prim *p;
    vertex_3d *v,*n2, *norm;
    vertex_3d *obj_space,*v2,*view_space;
    vertex_3d work;
    float *radii;
    double mag;
    float fmag;
    int prim_type;
    float radius_xy_scale,radius_depth_scale;

    /* premultiply object transformation with the current xform matrix. */
/*    mmul_4x4(&(r->xfrm),xfrm,&m); */
    /* postmultiply object transformation with the current xform matrix. */
    /** transformation on the current level is composed of:
      
      1. -center
      2. scale*rotate
      3 +center+translate.
      
    **/
    mmul_4x4(xfrm,&(r->xfrm),&m);

    if (r->nchildren == 0) /* a leaf node */
    {
	get_object_display_status(r,&display_status);
	opaque_status = is_opaque(r);
	if ((display_status == TRUE) && (opaque_status == TRUE))
	{
	    nprims = r->nprims;
	    for (i=0;i<nprims;i++)
	    {
		p = r->prims+i;
		n2 = NULL;
		radii = NULL;

		/* create space for the vertices: one copy for the
		   object space (transformed) vertices, one for the
		   view space vertices. */
		
		v = p->verts;
		obj_space = v2 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*p->nverts);
		view_space = (vertex_3d *)kmalloc(sizeof(vertex_3d)*p->nverts);

		if (p->type != KGEOM_SPHERES)
		{
		    norm = p->prim_specifics.normals;
		    if (norm)
			n2 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*p->nthings);
		}
		else
		{
		    radii = p->prim_specifics.radii;
		    if (radii)
		    {
			radii = (float *)kmalloc(sizeof(float)*p->nthings);
			memcpy((char *)radii,(char *)(p->prim_specifics.radii),
			       sizeof(float)*p->nthings);
		    }
		}

		/**
		  * apply the geometric transformations in object space
		**/
		geom_xform_points(&(r->xfrm),&(r->translate_vect),&(r->center),
				  v,obj_space,xfrm,tv,p->nverts); 
		
		/* transform the normals, if there are any. note: the
		   transformation of the normals does not involve any
		   translational component, so there is a specific
		   routine to deal with this transformation. */
		if (p->type != KGEOM_SPHERES)
		{
		    nnormal_xfrm(norm,&m,n2,p->nthings);
		    /* the normals are not run thru the view transformation
		       matrix since 1. this coordinate system is not
		       necessarily orthonormal to the world coord system,
		       and 2. lighting calculations are performed in the
		       world coord system. */
		    nnormal_unit(n2,n2,p->nthings);
		}
		else
		{
#if 0
		    /* the radii values must be scaled by an amount which
		       is the maximum of the scale components of the
		       object transformation matrix. */
		    work.v[0] = m.m[0][0] * view->m[0][0];
		    work.v[1] = m.m[1][1] * view->m[1][1];
		    work.v[2] = m.m[2][2] * view->m[2][2];
		    vertex_mag(&work,&mag);
		    fmag = mag;
#endif
		    get_world_to_view_scale(&radius_xy_scale,&radius_depth_scale);
		    scale_vector(radii,radius_xy_scale,radii,p->nthings,1);
		}

		/* now, do the viewing transformation */
		npoint_xfrm(&(obj_space->v[0]),view,&(view_space->v[0]),p->nverts);

		/* at this point, we have polygons with vertices
		   in the a normalized space. */

		prim_type = map_to_renderprims(p->type);
		(*software_primfuncs[prim_type])(p,w,h,obj_space,view_space,
						 n2,radii,projection,
						 opaque_status,r,
						 hither_flag,yon_flag);
		kfree(obj_space);
		kfree(view_space);
		if (radii)
		    kfree(radii);
		if (n2)
		    kfree(n2);
		    
	    }
	}
    }
    else
    {
	matrix4x4 minusC,SR,C,composite;
	
	for (i=0;i<3;i++)
	    trans_vec.v[i] = tv->v[i] + r->translate_vect.v[i];

	/**
	  * build composite transformation matrix:
	  * M = -CSRCT, where
	  *    C = object's center point,
	  *    SR = scale and rotate components.
	  *    T = translate vector.
	  *
	**/
	identity_4x4(&minusC);
	identity_4x4(&SR);
	identity_4x4(&C);
	
	identity_4x4(&composite);
	
	minusC.m[3][0] = -1. * r->center.v[0];
	minusC.m[3][1] = -1. * r->center.v[1];
	minusC.m[3][2] = -1. * r->center.v[2];
	
	matrix_4x4copy(&SR,&(r->xfrm));
	
	C.m[3][0] = r->center.v[0];
	C.m[3][1] = r->center.v[1];
	C.m[3][2] = r->center.v[2];
	
	mmul_4x4(&minusC,&SR,&composite);
	mmul_4x4(&composite,&C,&composite);

	mmul_4x4(xfrm,&composite,&m);

	for (i=0;i<r->nchildren;i++)
	{
	    render_opaque_subtree(r->children[i],&m,view,w,h,projection,
				  &trans_vec,hither_flag,yon_flag);
	}
    }
}

static void
render_transparent_subtree(rmonsterobj *r,
			   matrix4x4 *xfrm,
			   matrix4x4 *view,
			   int w,
			   int h,
			   int projection,
			   vertex_3d *tv,
			   int hither_flag,
			   int yon_flag)
{
    int status;
    int display_status,opaque_status;
    int nprims,i;
    matrix4x4 m;
    vertex_3d trans_vec;
    rmonster_prim *p;
    vertex_3d *v,*n2, *norm;
    vertex_3d *obj_space,*v2,*view_space;
    vertex_3d work;
    float *radii;
    double mag;
    float fmag;
    int prim_type;

    /* premultiply object transformation with the current xform matrix. */
/*    mmul_4x4(&(r->xfrm),xfrm,&m); */
    /* postmultiply object transformation with the current xform matrix. */
    mmul_4x4(xfrm,&(r->xfrm),&m);

    if (r->nchildren == 0) /* a leaf node */
    {
	get_object_display_status(r,&display_status);
	opaque_status = is_opaque(r);
	if ((display_status == TRUE) && (opaque_status == FALSE))
	{
	    nprims = r->nprims;
	    for (i=0;i<nprims;i++)
	    {
		p = r->prims+i;
		n2 = NULL;
		radii = NULL;

		/* create space for the vertices: one copy for the
		   object space (transformed) vertices, one for the
		   view space vertices. */
		
		v = p->verts;
		obj_space = v2 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*p->nverts);
		view_space = (vertex_3d *)kmalloc(sizeof(vertex_3d)*p->nverts);

		if (p->type != KGEOM_SPHERES)
		{
		    norm = p->prim_specifics.normals;
		    if (norm)
			n2 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*p->nthings);
		}
		else
		{
		    radii = p->prim_specifics.radii;
		    if (radii)
		    {
			radii = (float *)kmalloc(sizeof(float)*p->nthings);
			memcpy((char *)radii,(char *)(p->prim_specifics.radii),
			       sizeof(float)*p->nthings);
		    }
		}

		/**
		  * apply the geometric transformations in object space
		**/
		geom_xform_points(&(r->xfrm),&(r->translate_vect),&(r->center),
				  v,obj_space,xfrm,tv,p->nverts);
		
		/* transform the normals, if there are any. note: the
		   transformation of the normals does not involve any
		   translational component, so there is a specific
		   routine to deal with this transformation. */
		if (p->type != KGEOM_SPHERES)
		{
		    nnormal_xfrm(norm,&m,n2,p->nthings);
		    /* the normals are not run thru the view transformation
		       matrix since 1. this coordinate system is not
		       necessarily orthonormal to the world coord system,
		       and 2. lighting calculations are performed in the
		       world coord system. */
		    nnormal_unit(n2,n2,p->nthings);
		}
		else
		{
		    /* the radii values must be scaled by an amount which
		       is the maximum of the scale components of the
		       object transformation matrix. */
		    work.v[0] = m.m[0][0] * view->m[0][0];
		    work.v[1] = m.m[0][1] * view->m[1][1];
		    work.v[2] = m.m[0][2] * view->m[2][2];
		    vertex_mag(&work,&mag);
		    fmag = mag;
		    scale_vector(radii,fmag,radii,p->nthings,1);
		}

		/* now, do the viewing transformation */
		npoint_xfrm(&(obj_space->v[0]),view,&(view_space->v[0]),p->nverts);

		/* at this point, we have polygons with vertices
		   in the a normalized space. */

		prim_type = map_to_renderprims(p->type);
		(*software_primfuncs[prim_type])(p,w,h,obj_space,view_space,
						 n2,radii,projection,
						 opaque_status,r,
						 hither_flag,yon_flag);
		kfree(obj_space);
		kfree(view_space);
		if (radii)
		    kfree(radii);
		if (n2)
		    kfree(n2);
		    
	    }
	}
    }
    else
    {
	matrix4x4 minusC,SR,C,composite;
	
	for (i=0;i<3;i++)
	    trans_vec.v[i] = tv->v[i] + r->translate_vect.v[i];

	/**
	  * build composite transformation matrix:
	  * M = -CSRCT, where
	  *    C = object's center point,
	  *    SR = scale and rotate components.
	  *    T = translate vector.
	  *
	**/
	identity_4x4(&minusC);
	identity_4x4(&SR);
	identity_4x4(&C);
	identity_4x4(&composite);

	minusC.m[3][0] = -1. * r->center.v[0];
	minusC.m[3][1] = -1. * r->center.v[1];
	minusC.m[3][2] = -1. * r->center.v[2];

	matrix_4x4copy(&SR,&(r->xfrm));
	
	C.m[3][0] = r->center.v[0];
	C.m[3][1] = r->center.v[1];
	C.m[3][2] = r->center.v[2];

	mmul_4x4(&minusC,&SR,&composite);
	mmul_4x4(&composite,&C,&composite);

	mmul_4x4(xfrm,&composite,&m);

	for (i=0;i<r->nchildren;i++)
	    render_transparent_subtree(r->children[i],&m,view,w,h,projection,
				       &trans_vec,hither_flag,yon_flag);
    }
}

int map_to_renderprims(int ktype)
{
    /* convert from khoros primitve type to one of the rmonster
       supported types. */
    int val;
    
    switch (ktype)
    {
    case KGEOM_POLYLINE_DISJOINT:
	val = POLYLIN_DIS;
	break;
    case KGEOM_POLYLINE_CONNECTED:
	val = POLYLIN_CON;
	break;
    case KGEOM_TRIANGLES_DISJOINT:
	val = POLYTRI_DIS;
	break;
    case KGEOM_TRIANGLES_CONNECTED:
	val = POLYTRI_CON;
	break;
    case KGEOM_SPHERES:
	val = SPHERE;
	break;
    case KGEOM_QUADMESH:
	val = QMESH;
	break;
    case KGEOM_OCTMESH:
	val = OMESH;
	break;

	/* everything else isn't supported yet. */
    default:
	val = NO_PRIM;
	break;
    }
    return(val);
}

int
is_opaque(rmonsterobj *r)
{
    /* this is computed in the conversion from kobject to rmonsterobj
       routine.  it is based on a function of the object's rgba
       value (the alpha component), and the list of colors, which
       if present, is searched for non-unity opacity values. */
    
    if (r->is_transparent)
	return(FALSE);
    else
	return(TRUE);
}

int
have_any_transparent_objs (rmonsterobj *r)
{
    int i,rval;
    if (r->is_transparent)
	return(TRUE);
    else
    {
	for (i=0;i<r->nchildren;i++)
	{
	    rval = have_any_transparent_objs(r->children[i]);
	    if (rval == TRUE)
		return(rval);
	}
    }
    return(FALSE);
}
