/*
 * Khoros: $Id$
 */
 
#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Dithering routines
   >>>>    The routines in this file are taken from the source code
   >>>>    for ppmdither, by Christos Zoulas.  We modified some of
   >>>>    them slightly to work with Khoros data structures.  The
   >>>>    full copyright is included below.
   >>>>                             Kevin Campbell
   >>>>                             Nancy Johnston
   >>>>
   >>>>  Static:  dith_value
   >>>>           dith_matrix
   >>>>           dith_setup
   >>>>           dith_color
   >>>>           dith_dither
   >>>>             
   >>>>  Public:  ppmdither
   >>>>             
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "rmonster.h"

/* ppmdither.c - Ordered dithering of a color ppm file to a specified number
**               of primary shades.
**
** Copyright (C) 1991 by Christos Zoulas.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/

#define DEBUG 0

typedef unsigned char pixval;

typedef struct { pixval r, g, b; } pixel;

#define PPM_GETR(p) ((p).r)
#define PPM_GETG(p) ((p).g)
#define PPM_GETB(p) ((p).b)

#define PPM_ASSIGN(p,red,grn,blu) \
  { (p).r = (red); (p).g = (grn); (p).b = (blu); }

typedef unsigned char ubyte;

#define NC 256			/* Total number of colors		*/
#define NUM_SHADES 256		/* Max number of shades in primary	*/

static int dith_nr  =   8;	/* number of red shades			*/
static int dith_ng  =   8;	/* number of green shades		*/
static int dith_nb  =   4;	/* number of blue shades		*/
static int dith_nc  = 256;	/* total number of colors 8 x 8 x 4	*/
static int dith_dim =   4;	/* dimension of the dither matrix	*/
static int dith_dm2 =  16;	/* dim square				*/
static int **dith_mat; 		/* the dithering matrix			*/
static char *fn = "ppmdither()";

	/* COLOR():
	 *	returns the index in the color table for the
	 *      r, g, b values specified.
	 */
#define COLOR(r,g,b) (((r) * dith_ng + (g)) * dith_nb + (b))

	/* LEVELS():
	 *	Returns the total number of levels after dithering.
	 */
#define LEVELS(s)     ((double)(dith_dm2 * ((s) - 1) + 1))

	/* DITHER():
	 *	Returns the dithered color for a single primary.
	 *      p = the input pixel
	 *      d = entry in the dither matrix
	 *      s = the number of levels of the primary
	 *
	 */
#define DITHER(p,d,s) ((ubyte) (((double)(LEVELS(s) * (double)(p) + (double)(d))) / ((double)((double)dith_dm2 * NUM_SHADES))))


/* dith_value():
 *	Return the value of a dither matrix of size x size at x, y 
 *	[graphics gems, p. 714]
 */
static int
dith_value(y, x, size)
int y, x, size;
{
    register int d;

    /*
     * Think of d as the density. At every iteration, d is shifted
     * left one and a new bit is put in the low bit based on x and y.
     * If x is odd and y is even, or visa versa, then a bit is shifted in.
     * This generates the checkerboard pattern seen in dithering.
     * This quantity is shifted again and the low bit of y is added in.
     * This whole thing interleaves a checkerboard pattern and y's bits
     * which is what you want.
     */
    for (d = 0; size-- > 0; x >>= 1, y >>= 1)
	d = (d << 2) | (((x & 1) ^ (y & 1)) << 1) | (y & 1);
    return(d);
} /* end dith_value */


/* dith_matrix():
 *	Form the dithering matrix for the dimension specified
 *	(Scaled by NUM_SHADES)
 */
static void
dith_matrix(dim)
int dim;
{
    int x, y, *dat;

    dith_dim = (1 << dim);
    dith_dm2 = dith_dim * dith_dim;

    dith_mat = (int **) malloc((dith_dim * sizeof(int *)) + /* pointers */
			       (dith_dm2 * sizeof(int)));   /* data */

    if (dith_mat == NULL) 
      kerror(NULL, "dith_matrix", "out of memory");

    dat =  (int *) &dith_mat[dith_dim];
    for (y = 0; y < dith_dim; y++)
	dith_mat[y] = &dat[y * dith_dim];

    for (y = 0; y < dith_dim; y++) {
	for (x = 0; x < dith_dim; x++) {
	     dith_mat[y][x] = NUM_SHADES * dith_value(y, x, dim);
	}
    }
} /* end dith_matrix */

    
/* dith_setup():
 *	Setup the dithering parameters, lookup table and dithering matrix
 */
static void
dith_setup(dim, nr, ng, nb, ptab)
int dim, nr, ng, nb;
pixel *ptab;
{
    register int r, g, b, i;

    dith_dim = 4;
    dith_dm2 = 16;
    dith_nr  = nr;
    dith_ng  = ng;
    dith_nb  = nb;
    dith_nc  = nr * ng * nb;

    if (dith_nc > NC)
	kerror(NULL, "dith_setup","too many shades %d, max %d", dith_nc, NC);
    if (dith_nr < 2) 
	kerror(NULL, "dith_setup","too few shades for red, minimum of 2");
    if (dith_ng < 2) 
	kerror(NULL, "dith_setup","too few shades for green, minimum of 2");
    if (dith_nb < 2) 
	kerror(NULL, "dith_setup","too few shades for blue, minimum of 2");
    
    for (r = 0; r < dith_nr; r++) 
	for (g = 0; g < dith_ng; g++) 
	    for (b = 0; b < dith_nb; b++) {
		i = COLOR(r,g,b);
		PPM_ASSIGN(ptab[COLOR(r,g,b)], 
		           (r * (NC-1) / (dith_nr - 1)),
		           (g * (NC-1) / (dith_ng - 1)),
		           (b * (NC-1) / (dith_nb - 1)));
	    }
    
    dith_matrix(dim);
} /* end dith_setup */


/* dith_color():
 *  Return the closest color index for the one we ask
 */
int
dith_color(r, g, b)
float r, g, b;
{
    int rr, gg, bb;

    rr = r * (dith_nr - 1);
    gg = g * (dith_ng - 1);
    bb = b * (dith_nb - 1);
    return((int) COLOR(rr, gg, bb));
} /* end dith_color */


/* dith_dither():
 *  Dither height scanlines at a time
 */
void
dith_dither(
	    int w,
	    int h,
	    pixel *t,
	    float *f,
	    unsigned char *o)
{
    int y, dm = (dith_dim - 1);
    register int x, d;
    register int *m;
    unsigned char red, green, blue;

	/* first convert input matrix of floats (range 0-1) to bytes and
	   then do the color dithering */
    for (y = 0; y < h; y++)
	for (m = dith_mat[y & dm], x = w; --x >= 0;) {
	    d = m[x & dm];
	    red = (unsigned char)((*f++)*255.0);
	    green = (unsigned char)((*f++)*255.0);
	    blue = (unsigned char)((*f++)*255.0);
	    *o++ = COLOR(DITHER(red, d, dith_nr), 
			 DITHER(green, d, dith_ng), 
			 DITHER(blue, d, dith_nb));

	  }
    free(dith_mat);
} /* end dith_dither */

int
ppmdither(float *f,
	  int cols,
	  int rows,
	  unsigned char *out_img,
	  unsigned char *rmap,
	  unsigned char *gmap,
	  unsigned char *bmap)
{
  pixel ptab[256];
  pixel *pixels;
  pixel *pP;
  int row, col, i;

  /*
   *  reformat 24-bit pic24 image (3 floats per pixel) into 2-dimensional
   *  array of pixel structures
   */

  if (DEBUG) fprintf(stderr,"%s: remapping to ppm-style internal fmt\n", fn);
  
  dith_setup(4, dith_nr, dith_ng, dith_nb, ptab);
  dith_dither(cols, rows, ptab, f, out_img);

  /* convert ptab into three separate maps, one each for red, green, 
     and blue */

  for (i=0; i<256; i++) {
    rmap[i] = ptab[i].r;
    gmap[i] = ptab[i].g;
    bmap[i] = ptab[i].b;
  }
  if (DEBUG) fprintf(stderr,"%s: done in ppmdither\n", fn);

  return 1;
}


/*-----------------------------------------------------------
|
|  Routine Name: template
|
|       Purpose: 
|                
|                
|
|         Input: 
|                
|                
|
|        Output: 
|                
|       Returns: 
|
|    Written By:
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

