 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for gspheres
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lgspheres
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
static void make_nocolor_spheres PROTO((float *, int, float *));
static void make_colored_spheres_from_scalar_data PROTO((float *, rgba_vertex *, int, vis_cmap *, float *));
static void make_colored_spheres_from_color_data PROTO((float *, float *, rgba_vertex *, int, float *));
static void make_constant_radius_spheres PROTO((float *,int ,float *));
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lgspheres - *
* 
*       Purpose: Library Routine for sphere_maker
*         Input: 
*        Output: 
*       Returns: CHILL (?) on success, WHACKED (?) on failure
*  Restrictions: only processes, at most,  w,h,d (3d) data. t is ignored.
*    Written By: wes
*          Date: Apr 12, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 8 aug 94 -  if 00 lines need to be removed post-testing
*                of lset_common_geom_attributes()
****************************************************************/
/* -library_def */
int lgspheres(kobject datafile,
		  kobject mapfile,
		  kobject outfile,
		  int have_mapfile,
		  float *radius_scale,
		  char *name)
/* -library_def_end */

/* -library_code */
{
    register float scale;
    vis_cmap map;
    int w,h,d,e,t,dims;
    int i,j,k;
    int result;
    vertex_3d *centers;
    float *radii=NULL,*in_data=NULL;
    rgba_vertex *colors=NULL;
    float ft,*temp;
    int offset;
    int do_colors=0;
    vertex_3d bmin,bmax,t1,t2,center;

    temp = &ft;
    
    /**
      * ok, now verify that we have the proper combination of things.
      * we will permit the following:
      * 1. scalar datafile, no colormap
      * 2. scalar datafile, with colormap (map must be RGBA "vis" colormap)
      * 3. RGBA datafile, no colormap
      *
      * in all instances, the input data MUST HAVE 3-space location data.
      **/

    /* first, check to make sure we have 3D location data. */
    kpds_get_attribute(datafile,KPDS_LOCATION_SIZE,&w,&h,&d,&dims);
    if (dims != 3)
    {
	errno = KINVALID_DATATYPE;
	kerror("GEOMETRY","gspheres","The location data must have dimensions=3. The input file has dimensions=%d.\n",dims);
	return(WHACKED);
    }

    /* check the dimensions of the data */
    if (kpds_query_value(datafile))
        kpds_get_attribute(datafile,KPDS_VALUE_SIZE,&w,&h,&d,&t,&e);
    else
        e = 0;

    if (e != 1) /* not scalar data....*/
    {
	if ((e != 4) &&  /* can't possibly take anything but RGBA data */
	    (e != 0))  /* or zero-element data */
	{
	    errno = KINVALID_DATATYPE;
	    kerror("GEOMETRY","gspheres","This datafile has elements = %d.  Only elements=0,1 or 4 (interpreted as RGBA) are supported.",e);
	    return(WHACKED);
	}
	if (e != 0)
	    do_colors = 1;
    }
    else
    {
	/* check out the colormap */
	if ((have_mapfile) && (e == 1)) /* colormap + scalar data */
	{
	    do_colors = 1;
	    result = lkobj_to_vismap(mapfile,&map);
	    if (result == WHACKED)
	    {
		errno = KINVALID_DATATYPE;
		kerror("GEOMETRY","lsphere_maker","The mapfile is not a 'vis map' as created by lvismap_to_kobj.");
		return(WHACKED);
	    }
	}
	else if ((have_mapfile) && (e != 1))
	{
	    errno = KINVALID_DATATYPE;
	    kerror("GEOMETRY","lsphere_maker","When using an optional colormap, e must be equal to 1 in the datafile.  This datafile's e is %d.\n",e);
	    return(WHACKED);
	}
    }

    /* create the primitive list */
    kgeom_create_primitive_list(outfile);

    
    /* set object level attributes in the output object */
    lset_common_geom_attributes(outfile);

    /* set object level attributes in the output object */
    kgeom_set_attribute(outfile,KGEOM_OBJECT,KGEOM_NAME,name);
    kgeom_set_attribute(outfile,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,1);

    /* ok, now ask for floating point presentation of location and
       value segments. */

    /* set prim level attributes. */
    kgeom_set_attribute(outfile,KGEOM_SPHERES,KGEOM_NUMBER_VERTICES,(w*h*d));

    /* create some work arrays. */
    centers = (vertex_3d *)kmalloc(sizeof(vertex_3d)*w*h*d);
    if (do_colors)
    {
	colors = (rgba_vertex *)kmalloc(sizeof(rgba_vertex)*w*h*d);
	
	/* if have colors, set the appropriate flags */
#if 00
	kgeom_set_attribute(outfile,KGEOM_OBJECT,KGEOM_COLORSPACE,KRGB);
	kgeom_set_attribute(outfile,KGEOM_SPHERES,KGEOM_COLOR_DATA_TYPE,KFLOAT);
#endif
	/* set has_alpha to 1.  this is not really necessary at this
	   point - we don't explicitly enable transparent spheres at the
	   per-sphere level at this point in time, but may in the future. */
	
	kgeom_set_attribute(outfile,KGEOM_OBJECT,KGEOM_HAS_ALPHA,TRUE);
    }
    
    kpds_set_attribute(datafile,KPDS_VALUE_POSITION,0,0,0,0,0);
    kpds_set_attribute(datafile,KPDS_VALUE_DATA_TYPE,KFLOAT);

    if (do_colors && !have_mapfile)
    {
	in_data = (float *)kmalloc(sizeof(float)*w*h*d*e);
	kpds_get_data(datafile,KPDS_VALUE_ALL,in_data);
    }
    else if (e != 0)
	radii = kpds_get_data(datafile,KPDS_VALUE_ALL,NULL);
    else
	radii = (float *)kmalloc(sizeof(float)*w*h*d);

    offset = 0;
    scale = *radius_scale;

    if (e == 1)
    {
	if (have_mapfile)
	    make_colored_spheres_from_scalar_data(radii,colors,w*h*d,&map,radius_scale);
	else
	    make_nocolor_spheres(radii,w*h*d,radius_scale);
    }
    else if (e == 0)
	make_constant_radius_spheres(radii,w*h*d,radius_scale);
    else
	make_colored_spheres_from_color_data(radii,in_data,colors,w*h*d,radius_scale);

    /* the centers are computed the same regardless of color, radius
       computation, etc. */
    
    offset = 0;
    for (k=0;k<d;k++)
	for (j=0;j<h;j++)
	    for (i=0;i<w;i++)
	    {
		lget_coords(datafile,i,j,k,centers+offset);
		offset++;
	    }
	    
    /* stuff the data */
    kgeom_put_data(outfile,KGEOM_SPHERES,(float *)&(centers[0].v[0]),colors,radii);

    /* compute the bounding box for the aggregate object formed by the
       set of spheres. */

    for (i=0;i<3;i++)
    {
	bmin.v[i] = KGEOM_BIGNUM;
	bmax.v[i] = KGEOM_SMALLNUM;
    }

    for (i=0;i<w*h*d;i++)
    {
	for (j=0;j<3;j++)
	{
	    t1.v[j] = centers[i].v[j] + radii[i];
	    t2.v[j] = centers[i].v[j] - radii[i];
	}
	for (j=0;j<3;j++)
	{
	    if (t1.v[j] > bmax.v[j])
		bmax.v[j] = t1.v[j];
	    if (t2.v[j] < bmin.v[j])
		bmin.v[j] = t2.v[j];
	}
    }

    /* compute the center of the aggregate object based upon the centroid
       formed by the min/max computed above */

    for (j=0;j<3;j++)
	center.v[j] = (bmax.v[j] - bmin.v[j]) * 0.5 + bmin.v[j];

    kgeom_set_attribute(outfile,KGEOM_OBJECT,KGEOM_BOUNDING_BOX,&(bmin.v[0]),&(bmax.v[0]));
    kgeom_set_attribute(outfile,KGEOM_OBJECT,KGEOM_CENTER,&(center.v[0]));
    
    /* free up work arrays */
    kfree(centers);
    kfree(radii);
    
    return(CHILL);
}

static void
make_nocolor_spheres(float *radii,
		     int n,
		     float *s)
{
    /**
      * here, we just take the original data (in the r array), and
      * scale it by the amount specified by the user.
    **/
    register float scale,*r;
    int i;

    scale = *s;
    r = radii;
    
    for (i=0;i<n;i++,r++)
	*r = *r * scale;
    
}

static void
make_constant_radius_spheres(float *radii,
			     int n,
			     float *s)
{
    /**
      * here, we just copy *s over to *radii.
    **/
    register float scale,*r;
    int i;

    scale = *s;
    r = radii;
    
    for (i=0;i<n;i++,r++)
	*r = scale;
    
}

static void
make_colored_spheres_from_scalar_data(float *scalar,
				      rgba_vertex *colors,
				      int n,
				      vis_cmap *map,
				      float *radius_scale)
{
    /**
      * the radius value is computed as a function of the input
      * data and the alpha channel of the colormap.  the index
      * into the colormap is computed using the library call, then
      * the alpha channel of the map is scaled according to what
      * the user has specified.
      *
      * the color of the sphere is simply the color at the
      * computed colormap index.  the alpha component of the color
      * is set to 1.0.  at a later point, we may want to do something
      * with some type of variable transparency function, but for now,
      * we marry the alpha channel of the colormap to the radius size
      * only.
      *
      * the index into the colormap is computed as a function of the
      * input scalar data, and the vis colormap.
    **/
    
    register float scale,*r,*r2;
    int i,index;

    scale = *radius_scale;
    r = scalar;

    for (i=0;i<n;i++,r++)
    {
	index = lget_map_index(map,r);
	r2 = (float *)(&colors[i].rgba[0]);
	lget_map_rgba(map,index,r2,r2+1,r2+2,r2+3);
#if 0
	colors[i].rgba[0] = map->r[index];
	colors[i].rgba[1] = map->g[index];
	colors[i].rgba[2] = map->b[index];
	colors[i].rgba[3] = 1.;
#endif	
	*r = scale * *(r2+3);

	*(r2+3) = 1.0; /* constant opacity on output? */
    }
}

static void
make_colored_spheres_from_color_data(float *radii,
				     float *in_data,
				     rgba_vertex *colors,
				     int n,
				     float *radius_scale)
{
    /**
      * here, the in_data[] is an array of RGBA tuples.  the
      * output radii[p] values are computed as the product of the
      * A component of the in_data[], and the colors[] are copied
      * over directly from the in_data[] (variable opacity).
      * 
      **/
    register float scale;
    register float *r;
    register rgba_vertex *in,*c;
    register int i;

    r = radii;
    in = (rgba_vertex *)in_data;
    scale = *radius_scale;
    c = colors;
    
    for (i=0;i<n;i++,r++,in++,c++)
    {
	*r = in->rgba[3] * scale;
	memcpy((char *)c,(char *)in,sizeof(rgba_vertex));
    }
}
/* -library_code_end */
