 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for goctmesh
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lgoctmesh
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
#include "internals.h"

static int ok_octmesh_input PROTO(( kobject ));
static int make_oct_mesh PROTO(( kobject, kobject, int *, vis_cmap *, int));
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lgoctmesh - *
* 
*       Purpose: to create an octmesh geom object from 3D data.
*                the geom may be optionally colorized with
*                a colormap, whose presence is optional.
*
*         Input: inobj - an opened kobject containing the source data
*                cmapobj - an opened kobject containing a visualization cmap
*                name - a char string containing the name for the geom object
*
*        Output: outobj - an opened kobject containing the geom object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Yo Momma
*          Date: Apr 13, 1995
*      Verified: 
*  Side Effects: do not drive heavy machinery while operating this software
* Modifications: 
****************************************************************/
/* -library_def */
int lgoctmesh(kobject inobj,
	     int have_mapfile,
	     kobject cmapobj,
	     kobject outobj,
	     char *name)
/* -library_def_end */

/* -library_code */
{
    vis_cmap cmap;
    int i,num_dims,dims[5];

    /** first, verify that the input object is processable **/

    /** at present, we want 3 dimensional data and 3d (or no) coords and
      will barf if we get anything else **/

    if (ok_octmesh_input(inobj) == WHACKED)
	return(WHACKED);

    /* next, convert the colormap if it's present to a vis_cmap */
    if (have_mapfile)
    {
	if (lkobj_to_vismap(cmapobj,&cmap) == WHACKED)
        {
	    errno = KINVALID_DATATYPE;
	    kerror("GEOMETRY","loctmesh","The object supplied for a vis colormap is invalid. \n");
	    return(WHACKED);
	}
    }

    /* set up the output object */
    kgeom_set_attribute(outobj,KGEOM_OBJECT,KGEOM_NAME,name);
    kgeom_create_primitive_list(outobj);
    kgeom_set_attribute(outobj,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,1);
    kgeom_set_attribute(outobj,KGEOM_OBJECT,KGEOM_LAYOUT,KPER_VERTEX);
    kgeom_set_attribute(outobj,KGEOM_OBJECT,KGEOM_HAS_ALPHA,TRUE);

    /* -- xvisual needs these --*/
    kdms_copy_attributes(cmapobj, outobj, NULL,
                         KCOLOR_ALPHA_MIN, KCOLOR_ALPHA_MAX, NULL);

    lset_common_geom_attributes(outobj);
    
    /* build the output object */

    /* figure out if we are workign with 2 or 3d data. */
    kpds_get_attribute(inobj,KPDS_VALUE_SIZE,dims,dims+1,dims+2,
		       dims+3,dims+4);
	
    /* in the following loop, we`ll scan through the sizes associated
       with each value segment dimension (EXCEPT for elements) and
       count if the value is greater than 1 */

    num_dims=0;
    for (i=0;i<4;i++)
	if (dims[i] > 1)
	    num_dims++;

    /* if num_dims==3, then we'll (try to) make an octmesh.  otherwise,
       we'll barf. */

    if (num_dims == 3)
	make_oct_mesh(inobj,outobj,dims,&cmap,have_mapfile); 
    else
    {
	errno = KINVALID_DATATYPE;
	kerror("GEOMETRY","loctmesh","There must be 3 dimmensions of value data to make an octmesh.  You have given me %d\n",num_dims);
	return(WHACKED);
    }
    return(CHILL);
}

static int
ok_octmesh_input(kobject inobj)
{
    int lw,lh,ld,ldims;
    int vw,vh,vd,ve,vt;

    if (kpds_query_location(inobj))
    {
       kpds_get_attribute(inobj,KPDS_LOCATION_SIZE,&lw,&lh,&ld,&ldims);
    
       if (ldims != 3)
       {
	  errno = KINVALID_DATATYPE;
	  kerror("GEOMETRY","loctmesh","The location data must have dimensions=3.  The input file has dimensions=%d.\n",ldims);
	  return(WHACKED);
       }
    }
    else
    {
       errno = KINVALID_DATATYPE;
       kerror("GEOMETRY","loctmesh","The input file must have explicit location data\n");
       return(WHACKED);
    }

    /* set up a happy presentation for the location data */
    kpds_set_attribute(inobj, KPDS_LOCATION_DATA_TYPE, KFLOAT);

    kpds_get_attribute(inobj,KPDS_VALUE_SIZE,&vw,&vh,&vd,&vt,&ve);
    if (ve != 1)  /* it's not scalar data .. */
    {
	if (ve != 4) /* can't possibly be anything but RGBA data */
	{
	    errno = KINVALID_DATATYPE;
	    kerror("GEOMETRY","loctmesh","This datafile has elements = %d.  Only elements=1 or elements=4 (interpreted as RGBA) are supported.");
	    return(WHACKED);
	}
    }

    /* set up a happy presentation for the value data */
    kpds_set_attribute(inobj, KPDS_VALUE_DATA_TYPE, KFLOAT);

    return(CHILL);
}

static int
make_oct_mesh(kobject kin,
	      kobject kout,
	      int *dims,
	      vis_cmap *cmap,
	      int use_map)
{
    int i;
    int usize,vsize,wsize;
    float *iloc=NULL,*ival=NULL,*oloc=NULL,*temp=NULL,*ocolors=NULL;
    int vw,vh,vd,vt,ve;
    int index;
    vertex_3d bmin,bmax,center;

    /* assign values to usize,vsize and wsize.  from the calling routine,
       dims[] holds the size of the value data.  i'm assuming that
       there are at three and only three values in this array that are > 1.
       such an assumption eliminates a lot of boring error checking
       code, but the below code will seg fault if this is not the case */
    
    for (i=0;i<4;i++)
	if (dims[i] != 1)
	    break;
    
    usize = dims[i];
    for (i=i+1;i<4;i++)
	if (dims[i] != 1)
	    break;
    vsize = dims[i];
    
    for (i=i+1;i<4;i++)
	if (dims[i] != 1)
	    break;
    wsize = dims[i];

    /* grab the location data from the input. this code below assumes
     3d location data for both input and output, present on a per-grid
     point basis (curvilinear).  this needs to be enhanced to accomodate
     varying grid types. */

    /* at some point, we'll want to do this in a more intelligent way
       than to just malloc space for the entire shootin' match.
       maybe loop over columns or something */

    iloc = (float *)kmalloc(sizeof(float)*usize*vsize*wsize*3); 
    oloc = (float *)kmalloc(sizeof(float)*usize*vsize*wsize*3); 

    kpds_set_attribute(kin,KPDS_LOCATION_POSITION,0,0,0,0);
    kpds_get_data(kin,KPDS_LOCATION_ALL,iloc);

    kgeom_set_attribute(kout,KGEOM_OCTMESH,KGEOM_OCTMESH_SIZE,usize,vsize,wsize);

    kgeom_set_attribute(kout,KGEOM_OCTMESH,KGEOM_OCTMESH_LOCATION_POSITION,
			0,0);

    /* unscramble location data from XXXYYYZZZ to XYZXYZXYZ */

    {
	float *x,*y,*z,*d;
	int j;
	
	x = iloc;
	y = x + usize*vsize*wsize;
	z = y + usize*vsize*wsize;

	d = oloc;
	
	for (j=0;j<usize*vsize*wsize;j++)
	{
	    *d++ = *x++;
	    *d++ = *y++;
	    *d++ = *z++;
	}
    }

    kgeom_put_data(kout,KGEOM_OCTMESH_LOCATION_ALL,oloc);

    /**
      * compute the bounding box and center point.
    **/

    lcompute_bbox((vertex_3d *)oloc,usize*vsize*wsize,&bmin,&bmax);
    for (i=0;i<3;i++)
	center.v[i] = bmin.v[i] + (bmax.v[i]-bmin.v[i])*0.5;
    kgeom_set_attribute(kout,KGEOM_OBJECT,KGEOM_BOUNDING_BOX,&(bmin.v[0]),&(bmax.v[0]));
    kgeom_set_attribute(kout,KGEOM_OBJECT,KGEOM_CENTER,&(center.v[0]));
			
    /**
     * the possibilities for colorizing the output octmesh are as
     * follows:
     * 1. use_map == TRUE, && scalar data,
     * 2. use_map == FALSE, && e==4 value data (interpreted as
     *    RGBA in [0..1] )
    **/
    kpds_get_attribute(kin,KPDS_VALUE_SIZE,&vw,&vh,&vd,&vt,&ve);
    if (use_map)
    {
	if (ve != 1)
	{
	    errno = KINVALID_DATATYPE;
	    kerror("GEOMETRY","loctmesh","When using a colormap with 2D data, the elements size of the value segment MUST be equal to 1.  The file you supplied has something different.");
	    return(WHACKED);
	}
	ival = (float *)kmalloc(sizeof(float)*usize*vsize*wsize);
	temp = ocolors = (float *)kmalloc(sizeof(float)*usize*vsize*wsize*4);
	kpds_set_attribute(kin,KPDS_VALUE_POSITION,0,0,0,0,0);
	kpds_get_data(kin,KPDS_VALUE_ALL,ival);

	for (i=0;i<usize*vsize*wsize;i++,temp+=4)
	{
	    index = lget_map_index(cmap,ival+i);
	    lget_map_rgba(cmap,index,temp,temp+1,temp+2,temp+3);
	}

	kgeom_put_data(kout,KGEOM_OCTMESH_COLOR_ALL,ocolors);
    }
    kfree(iloc);
    
    return(CHILL);
}


void
lcompute_3dgrad(vertex_3d *dg_slice0,
		int usize,
		int vsize,
		float *d_slice0,
		float *d_slice1,
		float *d_slice2,
		vertex_3d *c_slice0,
		vertex_3d *c_slice1,
		vertex_3d *c_slice2,
		vertex_3d *work,
		int flip)
{
    /**
      * PARMS:
      *
      * dg_slice0 - destination array for the gradients
      * usize,vsize - represent the width and height of a slice of data
      * d_slice0,d_slice1,d_slice2 - three "slices" of data
      * c_slice0,c_slice1,c_slice2 - three "slices" of coordinate information
      * work - a scratch area of size usize*vsize*3 for use by this routine
    **/

    /**
      * the "gradient" is computed as a function of both the data
      * to be contoured, and the space in which it exists.
      *
      * without considering the space, the data gradient is:
      *
      * G.x = data(x+1,y,z) - data(x-1,y,z)
      * G.y = data(x,y+1,z) - data(x,y-1,z)
      * G.z = data(x,y,z+1) - data(x,y,z-1)
      *
      * There is a similar gradient computed for each of the
      * x,y, and z axes of the coordinate space:
      *
      * GX.x = coords(x+1,y,z).x - coords(x-1,y,z).x
      * GX.y = coords(x+1,y,z).y - coords(x-1,y,z).y
      * GX.z = coords(x+1,y,z).z - coords(x-1,y,z).z
      *
      * and similarly for GY and GZ.
      *
      * the "final" gradient is computed as a function of the data
      * gradient and the spatial gradient.   this will provide the
      * correct gradient in curvilinear grids:
      *
      * grad.x = G.x*GX.x + G.x*GX.y + G.x*GX.z
      * grad.y = G.y*GY.x + G.y*GY.y + G.y*GY.z
      * grad.z = G.z*GZ.x + G.z*GZ.y + G.z*GZ.z
    **/
    
    register float *d0,*d1;
    register vertex_3d *g;
    register double t;
    register vertex_3d *c0,*c1;
    register vertex_3d *sx,*sy,*sz;
    double flip_sign;

    int i,j;

    if (flip)
	flip_sign= -1.;
    else
	flip_sign = 1.;
    
    /* do z gradient first */
    
    d0 = d_slice0;
    d1 = d_slice2;
    g = dg_slice0;
    
    for (j=0;j<vsize;j++)
    {
	for (i=0;i<usize;i++)
	{
	    g->v[2] = (*d1 - *d0) * 0.5;
	    d1++;
	    d0++;
	    g++;
	}
    }

    /* next, do X gradients */

    for (j=0;j<vsize;j++)
    {
	g = dg_slice0 + j*usize;
	d0 = d_slice1 + j*usize;

	g->v[0] = (*(d0+1) - *d0) * 0.5;
	d0++;
	g++;
	
	for (i=1;i<usize-1;i++)
        {
	    g->v[0] = (*(d0+1) - *(d0-1)) * 0.5;
	    d0++;
	    g++;
	}
	g->v[0] = (*(d0) - *(d0-1)) * 0.5;
    }

    /* finally, do the Y gradients. */

    d0 = d_slice1;
    d1 = d0 + usize;
    g = dg_slice0;

    /* do first row. */
    for (i=0;i<usize;i++)
    {
	g->v[1] = (*d1 - *d0) * 0.5;
	d1++;
	d0++;
	g++;
    }

    /* do subseqeunt rows (except the last one */
    d0 = d_slice1;
    d1 = d0 + usize*2; /* skip forward two rows. */
	
    for (j=1;j<vsize-1;j++)
    {
	for (i=0;i<usize;i++)
	{
	    g->v[1] = (*d1 - *d0) * 0.5;
	    d1++;
	    d0++;
	    g++;
	}
    }

    d1 = d0 + usize;
    /* do last row. */
    for (i=0;i<usize;i++)
    {
	g->v[1] = (*d1 - *d0) * 0.5;
	d1++;
	d0++;
	g++;
    }

    /* now compute the gradients in space */
    
    /* do the Z coords first. */
    c0 = c_slice0;
    c1 = c_slice2;
    g = work+usize*vsize*2;
    
    for (j=0;j<vsize;j++)
    {
	for (i=0;i<usize;i++)
	{
	    g->v[0] = (c1->v[0] - c0->v[0]);
	    g->v[1] = (c1->v[1] - c0->v[1]);
	    g->v[2] = (c1->v[2] - c0->v[2]);
	    c1++;
	    c0++;
	    g++;
	}
    }

    /* next, do the Y coords. */

    c0 = c_slice1;
    c1 = c0 + usize;
    g = work + usize*vsize;

    /* do first row. */
    for (i=0;i<usize;i++)
    {
	g->v[0] = (c1->v[0] - c0->v[0]);
	g->v[1] = (c1->v[1] - c0->v[1]);
	g->v[2] = (c1->v[2] - c0->v[2]);
	c1++;
	c0++;
	g++;
    }

    /* do subseqeunt rows (except the last one */
    c0 = c_slice1;
    c1 = c0 + usize*2; /* skip forward two rows. */
	
    for (j=1;j<vsize-1;j++)
    {
	for (i=0;i<usize;i++)
	{
	    g->v[0] = (c1->v[0] - c0->v[0]);
	    g->v[1] = (c1->v[1] - c0->v[1]);
	    g->v[2] = (c1->v[2] - c0->v[2]);
	    d1++;
	    d0++;
	    g++;
	}
    }

    c1 = c0 + usize;
    /* do last row. */
    for (i=0;i<usize;i++)
    {
	g->v[0] = (c1->v[0] - c0->v[0]);
	g->v[1] = (c1->v[1] - c0->v[1]);
	g->v[2] = (c1->v[2] - c0->v[2]);
	d1++;
	d0++;
	g++;
    }

    /* next, do X gradients */

    for (j=0;j<vsize;j++)
    {
	g = work + j*usize;
	c0 = c_slice1 + j*usize;

	g->v[0] = ((c0+1)->v[0] - c0->v[0]);
	g->v[1] = ((c0+1)->v[1] - c0->v[1]);
	g->v[2] = ((c0+1)->v[2] - c0->v[2]);
	c0++;
	g++;
	
	for (i=1;i<usize-1;i++)
        {
	    g->v[0] = ((c0+1)->v[0] - (c0-1)->v[0]);
	    g->v[1] = ((c0+1)->v[1] - (c0-1)->v[1]);
	    g->v[2] = ((c0+1)->v[2] - (c0-1)->v[2]);
	    c0++;
	    g++;
	}
	g->v[0] = (c0->v[0] - (c0-1)->v[0]);
	g->v[1] = (c0->v[1] - (c0-1)->v[1]);
	g->v[2] = (c0->v[2] - (c0-1)->v[2]);
    }

    /* compute the produce of data and spatial gradients */
    sx = work;
    sy = sx + usize*vsize;
    sz = sy + usize*vsize;

    g = dg_slice0;

    for (j=0;j<vsize;j++)
    {
	for (i=0;i<usize;i++)
	{
	    g->v[0] = g->v[0]*sx->v[0] + g->v[0]*sx->v[1] + g->v[0]*sx->v[2];
	    g->v[1] = g->v[1]*sy->v[0] + g->v[1]*sy->v[1] + g->v[1]*sy->v[2];
	    g->v[2] = g->v[2]*sz->v[0] + g->v[2]*sz->v[1] + g->v[2]*sz->v[2];
	}
    }
    

    /* now, normalize them all */
    g = dg_slice0;
    for (j=0;j<vsize;j++)
	for (i=0;i<usize;i++)
	{
	    t = g->v[0]*g->v[0] + g->v[1]*g->v[1] + g->v[2]*g->v[2];
	    if (t != 0.)
	    {
		t = ksqrt(t);
		t = flip_sign/t;
		g->v[0] *= t;
		g->v[1] *= t;
		g->v[2] *= t;
	    }
	    g++;
	}
}
/* -library_code_end */
