 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for gimpfacet
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lgimpfacet
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lgimpfacet - read a facet file and convert into geometry
* 
*       Purpose: This routine will input a file containing facet 
*		 information into a data object using geometry 
*		 services.
*
*                The file will contain one or more "big parts", with
*                each big part consisting of a list of XYZ points.
*                For each big part, there will be one or more "small
*                parts", with each small part consisting of a number
*                of facets.  Each facet is simply a triangle specified
*                by three index numbers into the ver- tices list
*                contained in the big part.
*
*                The resulting geometry will consist of a series of
*                disjoint triangle primitives, with one primitive for
*                every small part.  Note that each disjoint triangle
*                primitive contains several triangles.
*
*                All triangles in the format are specified to be RH.
*                RH nor- mals will be generated for each triangle.
*
*		 The following example illustrates the format of a
*		 facet file.  The explanations, separated by '->'
*		 should not appear in the facet file.
*
*   ! information line          
*   ! 		1                    <- 1 big part
*   ! square with fin                <- name of big part
*   ! 0, 0.00 1.00 0.00 0.00         <- mirror, ax+by+cz-d = 0
*   ! 		5                    <- #nodes in big part
*   ! 0.00  0.00  0.00               <- xyz of node 1
*   ! 5.00  0.00  0.00                  separated by at least 1 space
*   ! 5.00  5.00  0.00
*   ! 5.00  0.00  0.00
*   ! 5.00  5.00  4.00
*   !           2                    <- number of small parts
*   ! square                         <- name of 1st small part
*   !		2                    <- number of facets in 1st small part
*   ! 1 2 3   1 0011 1               <- facet 1 has nodes 1,2,3 followed by
*   ! 1 3 4   1 0011 2                  material index, group #, facet #
*   ! fin                            <- name of 2nd small part
*   !		2                    <- number of facets in 2nd small part
*   ! 1 3 5   1 0011 1               
*
*		The material index, group #, and facet # are consumed
*		but not used.  In the future, the material index may
*		be used to add color to the geometry.
*
*	         The facet file format is also known as the General
*	         Dynamics ACAD format.
*
*         Input: in  - input file containing facet information
*
*        Output: out - output data object containing geometry data
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steve Kubica
*          Date: Apr 13, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lgimpfacet(
   kfile  *in,
   char   *geom_name,
   kobject out)
/* -library_def_end */

/* -library_code */
{
   char   *file = NULL;
   char    string[KLENGTH];
   char    name[KLENGTH];
   char   *s;
   int     num_big;
   int     num_small;
   int     numparts;
   int     numnodes;
   int     i = 0;
   int     j = 0;
   int     t = 0;
   int     flag;
   int     icoat;
   int     comp_num;
   int     facet_num;
   int     nb;
   float   a,b,c,d;
   float   min[3] = {0.0, 0.0, 0.0};
   float   max[3] = {0.0, 0.0, 0.0};
   float   ctr[3] = {0.0, 0.0, 0.0};

   float  *pbuffer = NULL;
   float  *p = NULL;
   int    *fbuffer = NULL;
   int    *f = NULL;

   float  *locs;
   float  *l;
   float  *norms;
   float  *n;

   /* -- verify that we have some kind of input -- */
   if (in == NULL)
   {
      kerror("geometry", "lgimpfacet", "Invalid NULL input.");
      return FALSE;
   }

   /* -- verify that we have some kind of output -- */
   if (out == NULL)
   {
      kerror("geometry", "lgimpfacet", "Invalid NULL output.");
      return FALSE;
   }

   /* -- if they really didn't want to have a name, then add one anyway -- */
   if (geom_name == NULL)
      geom_name = "facet";

   /* -- create an output primitive list -- */
   if (!kgeom_create_primitive_list(out))
   {
      kerror("geometry", "lgimpfacet", "Unable to create primitive list.");
      return FALSE;
   }
   
   /* -- read in the initial comment -- */
   kfgets(string, KLENGTH, in); 
   string[kstrlen(string)-1] = '\0';
   kinfo(KVERBOSE, "Reading : %s\n", string);

   /* -- get the number of big parts -- */
   kfgets(string, KLENGTH, in);
   ksscanf(string, "%d", &num_big); 

   for (nb = 0; nb < num_big; nb++)
   {
      /* -- read in the name of the big part -- */
      kfgets(name, KLENGTH, in);
      name[kstrlen(name)-2] = '\0';

      /* -- read the flag and the a b c d part -- */
      kfgets(string, KLENGTH, in);  
      ksscanf(string, "%d %d &d &d &d", &flag, &a, &b, &c, &d); 

      /* -- read the number of nodes in this big part -- */
      kfgets(string, KLENGTH, in);  
      ksscanf(string, "%d", &numnodes);

      kinfo(KVERBOSE, "\t-- big part   %d : '%s' with %d nodes --\n", nb+1,
	    name, numnodes);

      pbuffer = kmalloc(numnodes*3*sizeof(float *));
      for (j = 0, p = pbuffer; j < numnodes; j++)
      {
	 s = kfgets(string, KLENGTH, in);
	 *p++ = (float) strtod(s, &s);
	 *p++ = (float) strtod(s, &s);
	 *p++ = (float) strtod(s, &s);
      }

      /* -- get the number of small parts -- */
      kfgets(string, KLENGTH, in);
      ksscanf(string, "%d", &num_small); 
      for (i = 0; i < num_small; i++)
      {
	 kfgets(name, KLENGTH, in);
	 name[kstrlen(name)-2] = '\0';

	 kfgets(string, KLENGTH, in);  
	 ksscanf(string, "%d", &numparts);

	 kinfo(KVERBOSE, "\t   small part %d : '%s' with %d vertices\n", 
	       i+1, name, numparts);

	 fbuffer = kmalloc(numparts*3*sizeof(int *));
	 for (j = 0, f = fbuffer; j < numparts; j++)
	 {
	    s = kfgets(string, KLENGTH, in);

	    *f++ = (int) strtol(s, &s, 10);
	    *f++ = (int) strtol(s, &s, 10);
	    *f++ = (int) strtol(s, &s, 10);

	    icoat     = (int) strtol(s, &s, 10);
	    comp_num  = (int) strtol(s, &s, 10);
	    facet_num = (int) strtol(s, &s, 10);
	 }

	 /* -- get the triangle locations first -- */
	 locs = kmalloc(numparts*9*sizeof(float));
	 for (j = 0, l = locs; j < numparts*9; j++, l++)
	 {
	    *l = pbuffer[ (fbuffer[j/3]-1)*3 + j%3 ];
	    if (*l < min[j%3]) min[j%3] = *l;
	    if (*l > max[j%3]) max[j%3] = *l;
	 }
	 
	 /* -- construct normals from triangle locations -- */
	 norms = kmalloc(numparts*9*sizeof(float));
	 for (j = 0, n = norms, l = locs; j < numparts; j++, n+=9, l+=9)
	 {
	    /* -- RH normal is replicated one-per vertex -- */
	    n[0] = n[3] = n[6] = 
	       (l[4]-l[1])*(l[8]-l[2]) - (l[5]-l[2])*(l[7]-l[1]);
	    n[1] = n[4] = n[7] = 
	       (l[5]-l[2])*(l[6]-l[0]) - (l[3]-l[0])*(l[8]-l[2]);
	    n[2] = n[5] = n[8] = 
	       (l[3]-l[0])*(l[7]-l[1]) - (l[4]-l[1])*(l[6]-l[0]);
	 }

	 /* -- write out the small part -- */
	 kgeom_set_attribute(out, KGEOM_OBJECT,  
			     KGEOM_NUMBER_PRIMITIVES, (i+1));
	 kgeom_set_attribute(out, KGEOM_TRIANGLES_DISJOINT, 
			     KGEOM_NUMBER_VERTICES, numparts*3);
	 kgeom_put_data(out, KGEOM_TRIANGLES_DISJOINT, 
			locs, NULL, norms, NULL);

	 kfree(locs);
	 kfree(norms);
	 kfree(fbuffer);
      }
      kfree(pbuffer);
   }

   /* -- at this point, we should have read the entire input file -- */

   /* -- calculate the center from the begin and end point -- */
   ctr[0] = min[0] + (max[0]-min[0])/2;
   ctr[1] = min[1] + (max[1]-min[1])/2;
   ctr[2] = min[2] + (max[2]-min[2])/2;

   /* -- set interesting object level attributes -- */
   kgeom_set_attribute(out, KGEOM_OBJECT, 
		       KGEOM_NAME, geom_name);
   kgeom_set_attribute(out, KGEOM_OBJECT, 
		       KGEOM_BOUNDING_BOX, min, max);
   kgeom_set_attribute(out, KGEOM_OBJECT, 
		       KGEOM_CENTER, ctr);
   return TRUE;
}
/* -library_code_end */
