#include <design.h>
#include <geometry.h>

#include "gisosurface.h"

void iso_main PROTO((kobject, kobject, double, int, int, char *));
void add_triangle PROTO((vertex **, vertex **, kobject, int));

static void init_triangle_buffer PROTO((void));
static void reset_triangle_buffer PROTO((void));
static void flush_triangles PROTO((vertex *, vertex *, kobject, int));

static void form_indeces PROTO((int **,unsigned char **,unsigned char **,int,int));
void generate_triangles PROTO((int,int,int,int,kobject,slice *,slice *,slice *,slice *,double,int,int));

static int khoros_init_input PROTO(( kobject, int *, int *,int *));
static int khoros_init_output PROTO(( kobject, char *));

static void malloc_slice PROTO((slice *,int,int));
static void free_slice PROTO((slice *));
static void load_slice PROTO((slice *, int,int,int,kobject,double *));
static void malloc_2d_iarray PROTO((int ***,int,int));
static void free_2d_iarray PROTO((int ***));

/**
  * static buffer to hold triangles (w/normals).  triangles are
  * accumulated until the buffer is full, then the triangles
  * are flushed to the geom library.  the BUFSIZE indicates the
  * number of triangles per output primitive.
**/

#define BUFSIZE 2048

static vertex nbuf[BUFSIZE][3];
static vertex vbuf[BUFSIZE][3];
static int total_triangles,ntriangles;
static vertex bbox_min,bbox_max;


void
iso_main(kobject in,
	 kobject out,
	 double level,
	 int do_norms,
	 int flip,
	 char *name)
{
    int u,v,w;
    int status;
    vertex_3d center,*min,*max;
    vis_cmap vmap;

    status = khoros_init_input(in,&u,&v,&w);
    if (status != CHILL)
	return;

    status = khoros_init_output(out,name);
    if (status != CHILL)
	return;

    bbox_min.x = bbox_min.y = bbox_min.z = KGEOM_BIGNUM;
    bbox_max.x = bbox_max.y = bbox_max.z = -1. * KGEOM_BIGNUM;
    
    init_triangle_buffer();

    /* set up the output object info */
    
    make_surf(in,out,u,v,w,level,flip,do_norms);

    min = (vertex_3d *)&(bbox_min);
    max = (vertex_3d *)&(bbox_max);
    
    for (u=0;u<3;u++)
	center.v[u] = (max->v[u] + min->v[u]) * 0.5;
    
    flush_triangles(&(vbuf[0][0]),&(nbuf[0][0]),out,do_norms);

    kgeom_set_attribute(out,KGEOM_OBJECT,KGEOM_BOUNDING_BOX,&(min->v[0]),&(max->v[0]));
    kgeom_set_attribute(out,KGEOM_OBJECT,KGEOM_CENTER,&(center.v[0]));
}

void
make_surf(kobject f,
	  kobject isosurf,
	  int u,
	  int v,
	  int w,
	  double level,
	  int flip,
	  int do_norms)
{
    /**
      * allocate space for data, coordinate and normal buffers.  each
      * of these buffers must be large enough to accomodate 4 slices
      * of data, where each slice is u*v.
      *
      * the mask buffer is used to indicate whether or not a given
      * vertex is above (mask==1) the threshold, or below (mask==0)
      * the threshold.
    **/

    int i,j,k;
    register slice *slice_prev,*slice0,*slice1,*slice_next,*slice_temp;
    double dlevel;
    int **indices;

    dlevel = level;

    slice_prev = (slice *)kmalloc(sizeof(slice));
    slice0 = (slice *)kmalloc(sizeof(slice));
    slice1 = (slice *)kmalloc(sizeof(slice));
    slice_next = (slice *)kmalloc(sizeof(slice));
    
    malloc_slice(slice_prev,u,v);
    malloc_slice(slice0,u,v);
    malloc_slice(slice1,u,v);
    malloc_slice(slice_next,u,v);
    malloc_2d_iarray(&indices,u,v);

    load_slice(slice_prev,0,u,v,f,&dlevel);
    load_slice(slice0,0,u,v,f,&dlevel);
    load_slice(slice1,1,u,v,f,&dlevel);

    if (w == 2)
	load_slice(slice_next,1,u,v,f,&dlevel);
    else
        load_slice(slice_next,2,u,v,f,&dlevel);

    for (k=0;k<w-1;k++)
    {
	form_indeces(indices,slice0->mask,slice1->mask,u,v);
	
	for (j=0;j<v-1;j++)
	{
	    for (i=0;i<u-1;i++)
	    {
		if (indices[j][i] != 0)
		    generate_triangles(indices[j][i],i,j,k,isosurf,slice_prev,
				       slice0,slice1,slice_next,dlevel,
				       do_norms,flip);
		
	    }
	}

	slice_temp = slice_prev;
	slice_prev = slice0;
	slice0 = slice1;
	slice1 = slice_next;
	slice_next = slice_temp;

	if (k < w-3)
	    load_slice(slice_next,k+3,u,v,f,&dlevel);
    }

    free_slice(slice_prev);
    free_slice(slice0);
    free_slice(slice1);
    free_slice(slice_next);
    free_2d_iarray(&indices);

    kfree(slice_prev);
    kfree(slice0);
    kfree(slice1);
    kfree(slice_next);
}

static void
form_indeces(indices,mask0,mask1,u,v)
int *indices[];
unsigned char **mask0,**mask1;
int u,v;
{
    int i,j,r;
        
    for (j=0;j<v-1;j++)
    {
	for (i=0;i<u-1;i++)
	{
	    r = 0;
	    if (mask0[j][i])
		r += 1;
	    if (mask0[j][i+1])
		r += 2;
	    if (mask0[j+1][i+1])
		r += 4;
	    if (mask0[j+1][i])
		r += 8;
	    if (mask1[j][i])
		r += 16;
	    if (mask1[j][i+1])
		r += 32;
	    if (mask1[j+1][i+1])
		r += 64;
	    if (mask1[j+1][i])
		r += 128;
	    
	    indices[j][i] = r;
	}
    }
}

#if 0
void
load_data_slice(double *d,
		int w,
		int u,
		int v,
		kobject f)
{
    /**
      * in this routine, what we want to do is grab a "slice" or
      * "plane" of data.  here, we are assuming that the size of
      * a plane of data is u*v, and that the calling routine has
      * malloc'ed enough memory to hold this much data.
    **/

    kpds_set_attribute(f,KPDS_VALUE_POSITION,0,0,w,0,0);
    kpds_get_data(f,KPDS_VALUE_PLANE,d);
    
}

void
load_coord_slice(vertex *d,
		 int w,
		 int u,
		 int v,
		 kobject f,
		 vertex_3d *min,
		 vertex_3d *max)
{
    vertex *t;
    int i,j,k;

    t = d;
    for (j=0;j<v;j++)
    {
	for (i=0;i<u;i++)
	{
	    lget_coords(f,i,j,w, (vertex_3d *) t);
	    if (t->x < min->v[0])
		min->v[0] = t->x;
	    if (t->x > max->v[0])
		max->v[0] = t->x;
	    
	    if (t->y < min->v[1])
		min->v[1] = t->y;
	    if (t->y > max->v[1])
		max->v[1] = t->y;
	    
	    if (t->z < min->v[2])
		min->v[2] = t->z;
	    if (t->z > max->v[2])
		max->v[2] = t->z;
	    t++;
	}
    }
}
#endif

static int num_buffers_put = 1;

static void
init_triangle_buffer(void)
{
    num_buffers_put = 1;
    total_triangles = 0;
    ntriangles = 0;
}

static void
reset_triangle_buffer(void)
{
    total_triangles = 0;
    ntriangles = 0;
}

void
add_triangle(vertex **vlist,
	     vertex **nlist,
	     kobject isosurf,
	     int do_normals)
{
    int i;
    for (i=0;i<3;i++)
    {
	kmemcpy(&(vbuf[ntriangles][i]),vlist[i],sizeof(vertex));

	if (do_normals)
	    kmemcpy(&(nbuf[ntriangles][i]),nlist[i],sizeof(vertex));
    }
    
    ntriangles++;

    if (ntriangles == BUFSIZE)
    {
	flush_triangles(&(vbuf[0][0]),&(nbuf[0][0]),isosurf,do_normals);
	reset_triangle_buffer();
    }
}

static void
flush_triangles(vertex *vlist,
		vertex *nlist,
		kobject isosurf,
		int do_normals)
{
    int i;

    kinfo(KVERBOSE," writing buffer number %d, which contains %d triangles.\n",
	  num_buffers_put, ntriangles);

    if (ntriangles == 0)
       return;
    
    kgeom_set_attribute(isosurf,KGEOM_OBJECT,
			KGEOM_NUMBER_PRIMITIVES,num_buffers_put++);
    kgeom_set_attribute(isosurf,KGEOM_TRIANGLES_DISJOINT,
			KGEOM_NUMBER_VERTICES,ntriangles*3);

    {
	/* code to compute the bounding box */
	register float *rf,test;
	register int i;

	/* X minimum */
	rf = &(vlist[0].x);
	test = bbox_min.x;
	for (i=0;i<ntriangles*3;i++,rf+=3)
	{
	    if (*rf < test)
		test = *rf;
	}
	bbox_min.x = test;

	/* X maximum */
	test = bbox_max.x;
	rf = &(vlist[0].x);
	
	for (i=0;i<ntriangles*3;i++,rf+=3)
	{
	    if (*rf > test)
		test = *rf;
	}
	bbox_max.x = test;

	/* Y minimum */
	rf = &(vlist[0].y);
	test = bbox_min.y;
	for (i=0;i<ntriangles*3;i++,rf+=3)
	{
	    if (*rf < test)
		test = *rf;
	}
	bbox_min.y = test;

	/* Y maximum */
	test = bbox_max.y;
	rf = &(vlist[0].y);
	
	for (i=0;i<ntriangles*3;i++,rf+=3)
	{
	    if (*rf > test)
		test = *rf;
	}
	bbox_max.y = test;
	
	/* Z minimum */
	rf = &(vlist[0].z);
	test = bbox_min.z;
	for (i=0;i<ntriangles*3;i++,rf+=3)
	{
	    if (*rf < test)
		test = *rf;
	}
	bbox_min.z = test;

	/* Z maximum */
	test = bbox_max.z;
	rf = &(vlist[0].z);
	
	for (i=0;i<ntriangles*3;i++,rf+=3)
	{
	    if (*rf > test)
		test = *rf;
	}
	bbox_max.z = test;
    }
    
    if (do_normals)
	kgeom_put_data(isosurf,KGEOM_TRIANGLES_DISJOINT,vlist,NULL,nlist,NULL);
    else
	kgeom_put_data(isosurf,KGEOM_TRIANGLES_DISJOINT,vlist,NULL,NULL,NULL);

#if 0
    for (i=0;i<ntriangles*3;i++)
	kprintf("\t%d\t%g,%g,%g\t%g,%g,%g\n",i,vlist[i].x,vlist[i].y,vlist[i].z,nlist[i].x,nlist[i].y,nlist[i].z);
#endif
}



static int
khoros_init_input(kobject in,
		  int *u,
		  int *v,
		  int *w)
{
    int val_avail,dw,dh,dd,dt,de;
    
    /* get dimensions of value data */
    val_avail = kpds_query_value(in);
    if (val_avail != FALSE)
    {
	kpds_get_attribute(in,KPDS_VALUE_SIZE,&dw,&dh,&dd,&dt,&de);

	/* do sanity check on size, type of data */

	if (de != 1)
	{
/* KERROR CAT: "isosurface_old" */
	    kerror("GEOMETRY","data_to_geom","The Isosurface module can process only SCALAR (elements = 1) data.");
	    return(WHACKED);
	}
	if (dt != 1)
        {
/* KERROR CAT: "isosurface_old" */
	    kerror("GEOMETRY","data_to_geom","The Isosurface module can process only T==1 data.");
	    return(WHACKED);
	}
	if ((dw < 2) || (dh < 2) || (dd < 2))
	{
/* KERROR CAT: "isosurface_old" */
	    kerror("GEOMETRY","data_to_geom","The isosurface module requires that width, height and depth ALL be greater than 1.");
	    return(WHACKED);
	}
	
	*u = dw;
	*v = dh;
	*w = dd;

	/* insist on "double" presentation of data */
	kpds_set_attribute(in,KPDS_VALUE_DATA_TYPE,KDOUBLE);
    }

    /* -- ensure that we have location data on the input  //SK -- */
    if (!kpds_query_location(in))
    {
       int wid,hgt,dep;
       
       /* -- we know from the khoros_init_input that we have value -- */
       kpds_get_attribute(in,KPDS_VALUE_SIZE,&wid,&hgt,&dep,NULL,NULL);
       
       /* we are going to set the uniform location min and max to be from 
	  (0,0,0) <-> (wid,hgt,dep).   The rest of the routine will 
	  process as if we had regular curvilinear location data present */

       /* -- create uniform location data -- */
       kpds_set_attribute(in, KPDS_LOCATION_GRID, KUNIFORM);
       kpds_create_location(in);
       kpds_set_attributes(in, 
			   KPDS_LOCATION_BEGIN, 0.,0.,0.,
			   KPDS_LOCATION_END,(double)wid,
			   (double)hgt, (double)dep, NULL);
       
    }

    /* -- make sure that we process location as floating point -- */
    kpds_set_attribute(in,KPDS_LOCATION_DATA_TYPE,KFLOAT);

    return(CHILL);
}

static int
khoros_init_output(kobject kobj, char *name)
{
    /* the object has been created already.  here we set the
       attributes of interest. */

    kgeom_create_primitive_list(kobj);
    lset_common_geom_attributes(kobj);
    kgeom_set_attribute(kobj, KGEOM_OBJECT, KGEOM_NAME, name);
    
    return(CHILL);
}

static void
malloc_slice(slicep,u,v)
slice *slicep;
int u,v;
{
    int i,t;
    vertex *v2;
    double *d2;
    unsigned char *m2;
    float *fx,*fy,*fz;

    slicep->usize = u;
    slicep->vsize = v;

#if 0
    v2 = (vertex *)kmalloc(sizeof(vertex)*u*v);
    slicep->points = (vertex **)kmalloc(sizeof(vertex *)*v);
#endif
    fx = (float *)kmalloc(sizeof(float)*u*v);
    slicep->xpts = (float **)kmalloc(sizeof(float *)*v);

    fy = (float *)kmalloc(sizeof(float)*u*v);
    slicep->ypts = (float **)kmalloc(sizeof(float *)*v);

    fz = (float *)kmalloc(sizeof(float)*u*v);
    slicep->zpts = (float **)kmalloc(sizeof(float *)*v);

    d2 = (double *)kmalloc(sizeof(double)*u*v);
    slicep->data = (double **)kmalloc(sizeof(double *)*v);

    m2 = (unsigned char *)kmalloc(sizeof(unsigned char)*u*v);
    slicep->mask = (unsigned char **)kmalloc(sizeof(unsigned char *)*v);
    
    for (i=0,t=0;i<v;i++,t+=u)
    {
/*	slicep->points[i] = v2 + t; */
	slicep->xpts[i] = fx + t;
	slicep->ypts[i] = fy + t;
	slicep->zpts[i] = fz + t;
	slicep->data[i] = d2 + t;
	slicep->mask[i] = m2 + t;
    }
}

static void
free_slice(slicep)
slice *slicep;
{
    int i;
    
/*    kfree(slicep->points[0]); */
    kfree(slicep->xpts[0]);
    kfree(slicep->ypts[0]);
    kfree(slicep->zpts[0]);
    kfree(slicep->data[0]);
    kfree(slicep->mask[0]);

/*    kfree(slicep->points); */
    kfree(slicep->xpts);
    kfree(slicep->ypts);
    kfree(slicep->zpts);
    kfree(slicep->data);
    kfree(slicep->mask);
}

static void
malloc_2d_iarray(iarray,u,v)
int ***iarray;
int u,v;
{
    int i;
    int **t;
    int *t2;
    int index;

    t2 = (int *)kmalloc(sizeof(int)*u*v);
    t = (int **)kmalloc(sizeof(int *)*v);
    
    for (i=0,index=0;i<v;i++,index+=u)
    {
	t[i] = t2 + index;
    }
    *iarray = t;
}

static void
free_2d_iarray(iarray)
int ***iarray;
{
    int i;
    int **t;
    t = *iarray;
    kfree(t[0]);
    kfree(t);
}

static void
load_slice(slicep,w,u,v,input,dlevel)
slice *slicep;
int w,u,v;
kobject input;
double *dlevel;
{
    double dlev;
    int i,j;
    vertex *coord_ptr;
    double *data_ptr;
    unsigned char *mask_ptr;

    /* grab a plane of data */
    kpds_set_attribute(input,KPDS_VALUE_POSITION,0,0,w,0,0);
    kpds_get_data(input,KPDS_VALUE_PLANE,(float *)&(slicep->data[0][0]));

    /* grab a plane of coordinates */
    kpds_set_attribute(input,KPDS_LOCATION_POSITION,0,0,w,0);
    kpds_get_data(input,KPDS_LOCATION_PLANE,(float *)&(slicep->xpts[0][0]));
    
    kpds_set_attribute(input,KPDS_LOCATION_POSITION,0,0,w,1);
    kpds_get_data(input,KPDS_LOCATION_PLANE,(float *)&(slicep->ypts[0][0]));

    kpds_set_attribute(input,KPDS_LOCATION_POSITION,0,0,w,2);
    kpds_get_data(input,KPDS_LOCATION_PLANE,(float *)&(slicep->zpts[0][0]));

    /* compute the binary threshold mask */
    
    dlev = *dlevel;
    for (j=0;j<v;j++)
    {
	data_ptr = &(slicep->data[j][0]);
	mask_ptr = &(slicep->mask[j][0]);
	
	for (i=0;i<u;i++,data_ptr++,mask_ptr++)
	{
	    if (*data_ptr >= dlev)
		*mask_ptr = 1;
	    else
		*mask_ptr = 0;
	}
    }
}


