  /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */


/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>           ScatterPlot Routines
   >>>>
   >>>>  Private:
   >>>>           spc_init_plot_info()
   >>>>           spc_set_plotdata()
   >>>>           spc_find_plotdata_min_max()
   >>>>           spc_change_plot_col()
   >>>>           spc_paint_from_plot()
   >>>>           spc_identify_plot_point()
   >>>>           spc_update_plot_position()
   >>>>           spc_apply_plot_function()
   >>>>           spc_highlight_pointer_position()
   >>>>   Static:
   >>>>           highlight_cluster_from_plot_position()
   >>>>           update_plot_position_labels()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/

#include "spectrum.h"

static int last_cluster = -1;
static void highlight_cluster_from_plot_position PROTO((xvobject, int));
static void update_plot_position_labels          PROTO((int));


/*-----------------------------------------------------------
|
|  Routine Name: spc_init_plot_info
|
|       Purpose: Allocates & initializes information
|                associated with the scatter plot.
|         Input: None
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Oct 27, 1993
| Modifications:
|
------------------------------------------------------------*/
void spc_init_plot_info(void)
{
        int i, j;

	spc_scatter->wcxmin = 0.0;
	spc_scatter->wcxmax = 100.0;
	spc_scatter->wcymin = 0.0;
	spc_scatter->wcymax = 100.0;

	if (spc_maps == NULL) return;

        j = 0;
        for (i = 0; i < spc_map_colsize; i++)
        {
            if (spc_count[i] > 0) j++;
        }
        spc_unique_values = j;

        if (spc_scatter->plotdata != NULL) 
	    kfree(spc_scatter->plotdata);
        spc_scatter->plotdata = (Coord *) kmalloc(spc_unique_values * 
					          sizeof(Coord));

        if (spc_scatter->cluster_numbers != NULL) 
	    kfree(spc_scatter->cluster_numbers);
        spc_scatter->cluster_numbers = (int *) kmalloc(spc_unique_values * 
						       sizeof(int));
        if (spc_scatter->plot_lookup != NULL) 
	    kfree(spc_scatter->plot_lookup);
        spc_scatter->plot_lookup = (int *) kmalloc(spc_map_colsize * 
						   sizeof(int));
        j = 0;
        for (i = 0; i < spc_map_colsize; i++)
        {
            if (spc_count[i] > 0)
            {
                spc_scatter->plotdata[j].x = spc_maps[i][0];
                spc_scatter->plotdata[j].y = spc_maps[i][1];
                spc_scatter->plotdata[j].z = 0.0;
                spc_scatter->plotdata[j].d = i;
                spc_scatter->cluster_numbers[j] = i;
                spc_scatter->plot_lookup[i] = j;
                j++;
            }
	    else spc_scatter->plot_lookup[i] = -1;
        }

/*
kfprintf(kstderr, "plot index\tcluster number\n");
for (i = 0; i < spc_unique_values; i++)
 	kfprintf(kstderr, "%d\t\t%d\n", i, 
		 spc_scatter->cluster_numbers[i]);

kfprintf(kstderr, "\n\ncluster index\tplot lookup\n");
for (i = 0; i < spc_map_colsize; i++)
        if (spc_count[i] > 0)
 	    kfprintf(kstderr, "%d\t\t%d\n", i, spc_scatter->plot_lookup[i]);
*/

	spc_find_plotdata_min_max(spc_scatter->plotdata, spc_unique_values,
                                  &spc_scatter->wcxmin, &spc_scatter->wcxmax, 
			          &spc_scatter->wcymin, &spc_scatter->wcymax);
	spc_scatter->plot_x_focus = 
			(spc_scatter->wcxmax - spc_scatter->wcxmin)/2.0;
	spc_scatter->plot_y_focus = 
			(spc_scatter->wcymax - spc_scatter->wcymin)/2.0;

	spc_scatter->xcol = 0;
	spc_scatter->ycol = 1;



}

/*-----------------------------------------------------------
|
|  Routine Name: spc_set_plotdata
|
|       Purpose: Assuming that the plot object exists and 
|                the plotdata is valid, associates plot data
|                with plot object, and saves DC arrays.
|         Input: none
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Oct 27, 1993
| Modifications:
|
------------------------------------------------------------*/
int spc_set_plotdata(void)
{
	int status;

	if (spc_scatter == NULL)
	    return(FALSE);
	if (spc_scatter->plotdata == NULL)
	    return(FALSE);
	if (spc_scatter->plot == NULL)
	    return(FALSE);

        status = xvw_set_attributes(spc_scatter->plot,
                           XVW_PLOT2D_POINTS,        spc_scatter->plotdata,
                           XVW_PLOT2D_PLOTSIZE,      spc_unique_values,
                           XVW_GRAPHICS_RESET_WORLD, TRUE,
                           NULL);
	return(status);
}


/*-----------------------------------------------------------
|
|  Routine Name: spc_find_plotdata_min_max
|
|       Purpose: Finds the WC min's and max's of the plot data
|         Input: plotdata  - the plot data
|                size      - the size of the plot data
|        Output: wcxmin    - the WC xmin
|                wcxmax    - the WC xmax
|                wcymin    - the WC ymin
|                wcymax    - the WC ymax
|       Returns:
|    Written By: Danielle Argiro
|          Date: Oct 27, 1993
| Modifications:
|
------------------------------------------------------------*/
void spc_find_plotdata_min_max(
    Coord  *plotdata,
    int    size,
    double *wcxmin,
    double *wcxmax,
    double *wcymin,
    double *wcymax)
{
        int i;

        *wcxmax = plotdata[0].x;  *wcxmin = plotdata[0].x;
        *wcymax = plotdata[0].y;  *wcymin = plotdata[0].y;

        for (i = 1; i < size; i++)
        {
             if (plotdata[i].x < *wcxmin) *wcxmin = plotdata[i].x;
             if (plotdata[i].x > *wcxmax) *wcxmax = plotdata[i].x;
             if (plotdata[i].y < *wcymin) *wcymin = plotdata[i].y;
             if (plotdata[i].y > *wcymax) *wcymax = plotdata[i].y;
        }
}

/*----------------------------------------------------------
|
|  Routine Name: spc_change_plot_col
|
|       Purpose: When the user clicks on the 'X Column', 'Y Column'
|                or 'Z Column' buttons, changes the map column that 
|                specifies the X, Y, or Z values of the scatter plot.
|
|         Input: blank_struct - struct assoc. w/ Blank (-b) line in UIS
|				following Pane Action (-a) line
|		 flag         - one of SPC_X or SPC_Y
|
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Oct 28, 1993
| Modifications: Updated from Khoros 1.0 
|
--------------------------------------------------------*/

void spc_change_plot_col(
    kform_struct *blank_struct,
    int          flag)
{
	int             i, j, col_num;
	char            **choices = NULL; 
	char            *new_colname;
        xvw_list_struct *list_return;

	char *promptX = "Select band to plot as X values",
	     *promptY = "Select band to plot as Y values",
	     *promptZ = "Select band to plot as Z values",
	     *label   = "  ", *prompt;

	/* appropriate prompt depending on color band */
	if (flag == SPC_X)      prompt = promptX;
        else if (flag == SPC_Y) prompt = promptY;
        else if (flag == SPC_Z) prompt = promptZ;
	else
	{
	    kerror(NULL, "spc_change_plot_col", "internal error; bogus flag");
	    return;
	}

	/* create the list of choices of Map Columns */
	choices = spc_create_strings_mapcol_list();

	/* put up the list widget to get new column */
	list_return = xvu_run_list_wait(spc_mapcol_names, spc_map_colnum,  
		      			prompt, label, 0, FALSE); 
	
	if (list_return == NULL) 
	{
	    kfree(choices);
	    return;
	}
	col_num = list_return->list_index;
	new_colname = list_return->string;

	j = 0;
        for (i = 0; i < spc_map_colsize; i++)
        {
            if (spc_count[i] > 0)
            {
		if (flag == SPC_X)
                    spc_scatter->plotdata[j].x = spc_maps[i][col_num];
		else if (flag == SPC_Y)
                    spc_scatter->plotdata[j].y = spc_maps[i][col_num];
		else if (flag == SPC_Z)
                    spc_scatter->plotdata[j].z = spc_maps[i][col_num];
                j++;
            }
        }

	/* update info displayed on form with Blank selection */
	xvf_set_attribute(blank_struct, XVF_TITLE, new_colname);
	xvw_set_attributes(spc_scatter->plot,
                           XVW_PLOT2D_PLOTSIZE,      spc_unique_values,
                           XVW_PLOT2D_POINTS,        spc_scatter->plotdata,
	                   XVW_GRAPHICS_RESET_WORLD, TRUE,
                           NULL);


	if (flag == SPC_X)
	{
            spc_scatter->xcol = col_num;
	    xvw_set_attribute(spc_scatter->xaxis, XVW_AXIS_LABEL, new_colname);
	}
	else if (flag == SPC_Y)
	{
            spc_scatter->ycol = col_num;
	    xvw_set_attribute(spc_scatter->yaxis, XVW_AXIS_LABEL, new_colname);
	}

	karray_free(choices, spc_map_colnum, NULL);
	kfree(list_return);
}


/*----------------------------------------------------------
|
|  Routine Name: spc_paint_from_plot
|
|       Purpose: Adds a cluster to the currently selected class or
|                deletes a cluster from the currently selected class
|                when the mouse is depressed and moved over a point in the
|                scatter plot which represents a cluster.
|
|         Input: object      - the plot workspace
|                client_data - not used
|                call_data   - not used
|        Output: dispatch    - whether to allow event to propogate; unused
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Oct 29, 1993
| Modifications: Updated from Khoros 1.0 
|
--------------------------------------------------------*/

/* ARGSUSED */
void spc_paint_from_plot(
    xvobject object,
    kaddr    client_data,
    XEvent   *event,
    int      *dispatch)
{
	int  cluster, point_index;

	/* no current class - return */
        if (spc_current_class == NULL) return;

	/* button not depressed - return */
        if (event->xbutton.state == 0) return;

	/* see if we are over a plot point - if not, return */
        if (!(spc_identify_plot_point(event->xbutton.x, event->xbutton.y,
                                      &cluster, &point_index)))
        {
            *dispatch = FALSE;
            return;
        }

	/*
	 *  add the cluster to the current class or 
	 *  delete the cluster from the current class
	 */
	spc_add_or_delete_cluster(cluster);
	update_plot_position_labels(cluster);
	*dispatch = FALSE;
}




/*----------------------------------------------------------
|
|  Routine Name: spc_identify_plot_point
|
|       Purpose: Passed an (x,y) point in device coordinates,
|                attempts to identify the cluster represented by
|                that point in the scatter plot.
|
|         Input: (x, y)      - the coordinate of the plot location in DC's
|        Output: point_index - index into plot data
|                cluster     - index into spc_map array
|       Returns: TRUE if the DC point is near enough to a scatterplot
|                point to represent a cluster, FALSE otherwise
|    Written By: Danielle Argiro
|          Date: Oct 29, 1993
| Modifications: Updated from Khoros 1.0
|
--------------------------------------------------------*/
int spc_identify_plot_point(
    int   x, 
    int   y,
    int   *cluster,
    int   *point_index)
{
        int     i, id;
        double  xpos1, xpos2, ypos1, ypos2;
        Coord   c1, c2, c;

        id = (int) (xvw_widget(spc_scatter->plot));

        xpos1 = x - 3; ypos1 = y - 3;
        xpos2 = x + 3; ypos2 = y + 3;
        X2D_convert_point_dc_to_wc(id, xpos1, ypos1, &c1);
        X2D_convert_point_dc_to_wc(id, xpos2, ypos2, &c2);
        X2D_convert_point_dc_to_wc(id, (Real) x, (Real) y, &c);

        xpos1 = kabs(c1.x - c2.x);
        ypos1 = kabs(c1.y - c2.y);
        for (i = 0; i < spc_unique_values; i++)
        {
           if ( (kabs(c.x - spc_scatter->plotdata[i].x) < xpos1) && 
                (kabs(c.y - spc_scatter->plotdata[i].y) < ypos1) )
	   {
	      *cluster = spc_scatter->cluster_numbers[i];
	      *point_index = spc_scatter->plot_lookup[*cluster];
              return(TRUE);
	   }
        }
        return(FALSE);
}


/*----------------------------------------------------------
|
|  Routine Name: spc_update_plot_position
|
|       Purpose: Updates the world coordinates of the point in
|                the plot as indicated by the mouse position.
|
|         Input: object      - the plot workspace
|                client_data - not used
|                call_data   - not used
|        Output: dispatch    - whether to allow event to propogate; unused
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Oct 29, 1993
| Modifications: Updated from Khoros 1.0
|
--------------------------------------------------------*/

/* ARGSUSED */
void spc_update_plot_position(
    xvobject object,
    kaddr    clientData,
    XEvent   *event,
    int      *dispatch)
{
        int   cluster, point_index;

	if (!xvw_check_visible(spc_scatter->parent))
            return;

	/* button depressed - return */
        if (event->xbutton.state != 0) return;

	/* 
	 * see what point we're at 
	 */
        if (!(spc_identify_plot_point(event->xbutton.x,
				      event->xbutton.y,
                                      &cluster, &point_index)))
	{
	    /*
             * if we are not on an actual point in the scatter plot, delete
	     * the last pixel from the colorcell list; this will cause the
	     * color of that pixel to revert to normal.
             */
	    if (last_cluster != -1)
                xvw_set_attributes(spc_scatter->colorcell, 
			           XVW_COLORCELL_RESTORE, TRUE,
			           XVW_COLORCELL_DELETE,  last_cluster,
			           NULL);
	    last_cluster = -1; 
            return;
	}

	/*
         * highlight cluster in image according to position of cursor in plot
         */
	 highlight_cluster_from_plot_position(object, cluster);

	/*
         *  update plot position labels under scatterplot
         */
	update_plot_position_labels(cluster);

}

/*----------------------------------------------------------
|
|  Routine Name: spc_apply_plot_function
|
|       Purpose: Updates the plot data of scatter plot as 
|                as by a function of bands.
|
|         Input: func_string - string defining function to apply
|	         flag        - whether to apply function to X or Y values;
|	                       one of: SPC_X or SPC_Y
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Aug 10, 1994
| Modifications: 
|
--------------------------------------------------------*/
void spc_apply_plot_function(
   char *func_string,
   int   flag)
{

        float   fvalue;
        double  *maps;
        int     i, var_num; 
	int     *bands;
        char    **var_list; 
	char    *function, *function_def;
        char    init_string[KLENGTH], temp[KLENGTH];

        if (flag == SPC_X)
           ksprintf(init_string, "X(");
        else ksprintf(init_string, "Y(");

        /* parse variables MX out of function string */
        if ((var_list = CreateVariableList(func_string, spc_map_colnum,
                                           &var_num)) == NULL)
           return;

        function_def = CreateFunctionDefinition(func_string, init_string,
                                                var_list, var_num);

        /* send function definition to kexpr parser */
        if (!(kexpr_evaluate_float(SPC_ID, function_def, NULL, temp)))
        {
            kerror("spectrum", "spc_apply_plot_function", temp);
            kfree(function_def);
            karray_free(var_list, var_num, NULL);
            return;
        }

        bands = (int *) kmalloc(var_num * sizeof(int));
        for(i = 0; i < var_num; i++)
        {
            ksprintf(temp, var_list[i]);
            bands[i] = atoi(&temp[1]);
        }

	kpds_set_attributes(spc_image,
                KPDS_MAP_DATA_TYPE,   KDOUBLE,
                KPDS_MAP_POSITION,    0, 0, 0, 0, 0,
                KPDS_MAP_REGION_SIZE, spc_map_colnum, spc_map_rownum, 1, 1, 1,
                NULL);
	maps = (double *) kpds_get_data(spc_image, KPDS_MAP_REGION, NULL);

        for (i = 0; i < spc_map_rownum; i++)
        {
             function = CreateFunctionCall(i, bands, var_num, spc_map_colnum,
                                           init_string, maps);
             kexpr_evaluate_float(SPC_ID, function, &fvalue, temp);
	     if (flag == SPC_X)
	         spc_scatter->plotdata[i].x = (double) fvalue;
	     else spc_scatter->plotdata[i].y = (double) fvalue;
             kfree(function);
        }
        kfree(bands);
        kfree(function_def);

	xvw_set_attributes(spc_scatter->plot,
                           XVW_PLOT2D_PLOTSIZE,      spc_unique_values,
                           XVW_PLOT2D_POINTS,        spc_scatter->plotdata,
                           XVW_GRAPHICS_RESET_WORLD, TRUE,
                           NULL);
}

/*----------------------------------------------------------
|
|  Routine Name: spc_highlight_pointer_position
|
|       Purpose: Updates the location of the marker in the 
|                scatterplot according to the location of 
|                the pointer in the image.
|
|         Input: object      - the image workspace, zoom workspace, 
|			       or scatterplot
|                client_data - not used
|                call_data   - not used
|        Output: dispatch    - whether to allow event to propogate; unused
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Oct 17, 1994
| Modifications:
|
--------------------------------------------------------*/
/* ARGSUSED */
void spc_highlight_pointer_position(
    xvobject object, 
    kaddr    client_data, 
    XEvent   *event, 
    int      *dispatch)
{
	int    cluster;
        double tmp_double;
	
	if (!xvw_check_visible(spc_scatter->parent))
            return;

	/*
         *  event came from the main spectrum workspace or from zoom window. 
	 *  get cluster value at that point.
         */
	xvw_get_attribute(object, XVW_IMAGE_VALUE,  &tmp_double);
        cluster = (int) tmp_double;

	/*
	 *  relocate the scatter plot marker to that position.
	 */
        xvw_set_attributes(spc_scatter->plot_marker,
		  XVW_MARKER_XPLACEMENT, spc_scatter->plotdata[cluster].x,
		  XVW_MARKER_YPLACEMENT, spc_scatter->plotdata[cluster].y,
		  NULL);
        /*
         *  highlight cluster in image according to position of cursor in plot
         */
        highlight_cluster_from_plot_position(object, cluster);

	/*
	 *  update plot position labels under scatterplot
	 */
	update_plot_position_labels(cluster);

}

/*----------------------------------------------------------
|
|  Routine Name: highlight_cluster_from_plot_position
|
|       Purpose: Highlights the cluster in the image that corresponds
|                to the plot point under the cursor.
|
|         Input: object - scatterplot area, main spectrum workspace, or
|			  zoom workspace
|		 cluster - current cluster number
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Feb 16, 1995
| Modifications:
|
--------------------------------------------------------*/
static void highlight_cluster_from_plot_position(
    xvobject object,
    int      cluster)
{
	int red, green, blue;

	/*	 
	 * cursor in image or zoom workspace: only want to
	 * update color cell itself, NOT the image.
	 */
	if (object != spc_scatter->area)
        {

            /*
             * delete old pixel from the colorcell list
             */
            xvw_set_attribute(spc_scatter->colorcell,
                              XVW_COLORCELL_DELETE, last_cluster);
            /*
	     * add new pixel to colorcell list, causing colorcell
	     * to reflect color of pixel under cursor
	     */
            xvw_set_attribute(spc_scatter->colorcell,
                              XVW_COLORCELL_ADD, cluster);
	}

	/*
	 * only if cursor is in scatter plot: turn the corresponding 
	 * clusters in the image a highlighted color by adding
         * them to the index list of the colorcell, and then changing
         * the color of the colorcell.
         */
	else
	{
            if (last_cluster == cluster) return;

            /*
             * restore color of old pixel in the colorcell list,
	     * then delete it from the list (only want one pixel in
	     * the list at any one time).
             */
	    if (last_cluster != -1)
                xvw_set_attributes(spc_scatter->colorcell,
			           XVW_COLORCELL_RESTORE, TRUE,
                                   XVW_COLORCELL_DELETE,  last_cluster,
			           NULL);
            /*
             * add new pixel to colorcell list, then cause colorcell
             * to update to reflect color of pixel under cursor
             */
            xvw_set_attributes(spc_scatter->colorcell,
                               XVW_COLORCELL_ADD, cluster,
			       XVW_COLORCELL_UPDATE,   TRUE,
                               NULL);

            xvw_get_attributes(spc_scatter->colorcell,
                               XVW_COLORCELL_REDVAL,   &red,
                               XVW_COLORCELL_GREENVAL, &green,
                               XVW_COLORCELL_BLUEVAL,  &blue,
                               NULL);
            red   = 0;
            green = 0;
            blue  = 0;
            xvw_set_attributes(spc_scatter->colorcell,
                               XVW_COLORCELL_REDVAL,   red,
                               XVW_COLORCELL_GREENVAL, green,
                               XVW_COLORCELL_BLUEVAL,  blue,
		               XVW_COLORCELL_UPDATE,   TRUE,
                               NULL);
	}
        last_cluster = cluster;

}

/*----------------------------------------------------------
|
|  Routine Name: update_plot_position_labels
|
|       Purpose: Updates the text under the scatterplot that
|		 displays the plot position and the 
|		 current cluster number (with class).
|
|         Input: cluster - current cluster number
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Feb 16, 1995
| Modifications:
|
--------------------------------------------------------*/

static void update_plot_position_labels(
    int cluster)
{
	char temp[KLENGTH];
	int  point_index;

	point_index = spc_scatter->plot_lookup[cluster];

        /* update label showing plot position */
        ksprintf(temp, "Plot position = (%3g x %3g)",
                spc_scatter->plotdata[point_index].x,
                spc_scatter->plotdata[point_index].y);
        xvw_set_attribute(spc_scatter->position_label, XVW_LABEL, temp);

        /* update label showing cluster number */
        if ((spc_legend_lookup != NULL) &&
            (spc_legend_lookup[cluster] != NULL))
            ksprintf(temp, "Cluster Number %d, member of class %s",
                    cluster, spc_legend_lookup[cluster]->classname);
        else ksprintf(temp, "Cluster Number %d, not member of class",
                     cluster);
        xvw_set_attribute(spc_scatter->cluster_label, XVW_LABEL, temp);
}
