 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */



/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>  	Display Environment File Routines
   >>>>
   >>>>  Private:
   >>>>			spc_read_environ_file()
   >>>>   Static:
   >>>>			parse_environ_info()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/

#include "spectrum.h"

/*-----------------------------------------------------------
|
|  Routine Name: spc_read_environ_file
|
|       Purpose: Reads an ascii display environment file
|
|         Input: filename - name of file to read
|        Output: None
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Aug 8, 1994
| Modifications: 
|
------------------------------------------------------------*/

static char *pixel_dim_x_scan = "%s%*[ ]'%lg'";
static char *pixel_dim_y_scan = "%s%*[ ]'%lg'";
static char *utm_00_n_scan    = "%s%*[ ]'%lg'";
static char *utm_00_e_scan    = "%s%*[ ]'%lg'";
static char *utm_zone_scan    = "%s%*[ ]%*['`]%[^'`]%*['`]%*[ ]";
static char *mapcol_scan      = "M%d:%*[ ]%*['`]%[^'`]%*['`]%*[ ]";
static char *transform_scan   = "%[^:]";
static char *red_scan         = "Red:%*[ ]%*['`]%[^'`]%*['`]%*[ ]:";
static char *green_scan       = "Green:%*[ ]%*['`]%[^'`]%*['`]%*[ ]:";
static char *blue_scan        = "Blue:%*[ ]%*['`]%[^'`]%*['`]%*[ ]:";

static char *get_environ_entry   PROTO((kfile *));
static int   get_utm_coordinates PROTO((kfile *));

#define OMIT_UTM 2
#define FAILED   0

int spc_read_environ_file(
    char *filename)
{
	int   i, done, status, colnum;
	char  temp[KLENGTH];
	char  *line;
	kfile *file;
        

	/* open environ file */
	if ((file = kfopen(filename, "r")) == NULL)
        {
	   kerror(NULL, "spc_read_environ_file", 
	          "Unable to open '%s' to read display environment.", filename);
	   kinfo(KHOSTILE, "Get it right!");
           return(FALSE);
        }

	/* premature end of file */
        if (kfeof(file))
           return(FALSE);

	/*
	 *  get pixel x & y dimensions, UTM N & E coordinates & UTM zone
	 */
	status = get_utm_coordinates(file);
	if (status == FALSE)
	   return(FALSE);

	/*
         *  create the label that will display the UTM position
         */
	if ((status != OMIT_UTM) && (spc_display != NULL))
	{
            spc_display->utm_position= xvw_create_labelstr(spc_display->parent,
                                                           "position_utm");
            ksprintf(temp, "0N, 0E, Zone ");
            xvw_set_attributes(spc_display->utm_position,
                               XVW_BELOW,           spc_display->workspace,
                               XVW_LEFT_OF,         NULL,
                               XVW_LABEL,           temp,
                               XVW_FORCE_REDISPLAY, TRUE,
                               NULL);
	    xvw_add_event(spc_display->workspace, PointerMotionMask,
                          spc_update_utm_position, NULL);
	}

	/*
         * get names for map columns
         */
	line = get_environ_entry(file);
	/* no map column names or transformations included in file, ok */
        if (line == NULL) return(TRUE);

	while (line[0] == 'M')
	{
            status = sscanf(line, mapcol_scan, &colnum, temp);
            if (status != 2)
            {
                kerror(NULL, "spc_read_environ_file",
                       "Unable to read line of in display environment file giving band names, which should read as in the following example:\nM1: 'TM 1 blue'\n");
                return(FALSE);
            }
	    if (colnum >= spc_map_colnum)
 	       spc_mapcol_names = (char **) krealloc(spc_mapcol_names, 
				            (colnum+1) * sizeof(char *));
            spc_mapcol_names[colnum] = kstrdup(temp);
            kfprintf(kstderr, "M%d: %s\n", colnum, spc_mapcol_names[colnum]);
            kfree(line);
	    line = get_environ_entry(file);
	}

	/*
	 * get equations for transformations
	 */
	/* no equations for transformations: ok */
	if (line == NULL) return(TRUE);
	done = FALSE;
	spc_trans_num = 0;
	while (!done)
	{
	    status = sscanf(line, transform_scan, temp);
	    if (status != 1)
            {
                kerror(NULL, "spc_read_environ_file",
                       "Unable to read 1st line of transformation info set in display environment file, which should read as in the following example:\nColorIR :");
                return(FALSE);
            }
	    spc_trans_info = (TransformStruct **) krealloc(spc_trans_info,
                                (spc_trans_num+1)*sizeof(TransformStruct *));
	    spc_trans_info[spc_trans_num]  = (TransformStruct *) 
				  	    kcalloc(1, sizeof(TransformStruct));

	    spc_trans_info[spc_trans_num]->name = kstrdup(temp);

	    line = get_environ_entry(file);
	    status = sscanf(line, red_scan, temp);
            if (status != 1)
            {
                kerror(NULL, "spc_read_environ_file",
                       "Unable to read 2nd line of transformation info set in display environment file, which should read as in the following example:\nRed: 'M3+M1'");
                return(FALSE);
            }
            spc_trans_info[spc_trans_num]->red_equation = kstrdup(temp);

	    line = get_environ_entry(file);
	    status = sscanf(line, green_scan, temp);
            if (status != 1)
            {
                kerror(NULL, "spc_read_environ_file",
                       "Unable to read 3rd line of transformation info set in display environment file, which should read as in the following example:\nGreen: 'M2/M0'");
                return(FALSE);
            }
            spc_trans_info[spc_trans_num]->green_equation = kstrdup(temp);

	    line = get_environ_entry(file);
	    status = sscanf(line, blue_scan, temp);
            if (status != 1)
            {
                kerror(NULL, "spc_read_environ_file",
                       "Unable to read 4th line of transformation info set in display environment file, which should read as in the following example:\nBlue: 'M4+M0/M2'");
                return(FALSE);
            }
            spc_trans_info[spc_trans_num]->blue_equation = kstrdup(temp);

	    line = get_environ_entry(file);
	    if (line == NULL) done = TRUE;
	    spc_trans_num++;
	}
	spc_update_GUI_from_mapcol_names();

	for (i = 0; i < spc_trans_num; i++)
	{
	    kfprintf(kstderr, "%s :\n",    spc_trans_info[i]->name);
	    kfprintf(kstderr, "Red: '%s'\n", spc_trans_info[i]->red_equation);
	    kfprintf(kstderr, "Green: '%s'\n", spc_trans_info[i]->green_equation);
	    kfprintf(kstderr, "Blue: '%s'\n", spc_trans_info[i]->blue_equation);
	    kfprintf(kstderr, "\n");
	}
	if (spc_trans_num > 0)
	    xvf_set_attribute(gui_info->Disp->disp->t_struct,
			      XVF_ACTIVATE, 1);

	/*
         * update GUI to reflect filename
         */
        xvf_set_attribute(gui_info->Files->files->in_environ_struct,
                          XVF_FILE_NAME, filename);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: get_environ_entry
|
|       Purpose: Reads a single entry from the display environment file
|
|         Input: file - open stream to display environment file
|        Output: None
|       Returns: a single entry from the display environment file
|    Written By: Danielle Argiro
|          Date: Aug 8, 1994
| Modifications:
|
------------------------------------------------------------*/

static char *get_environ_entry(
       kfile *file)
{
	int  j, ready;
	char temp[KLENGTH];

	ready = FALSE;
        while (!ready)
        {
           j = 0;
           if (kfgets(temp, KLENGTH, file) == NULL)
                return(NULL);

           /* skip over any leading whitespace */
           while(temp[j] == ' ' || temp[j] == '\t') j++;

           /* skip over any blank lines;  comments have # symbol */
            if ((temp[j] != '\n') && (temp[j] != '#'))
                ready = TRUE;
        }

	if (ready)
	    return(kstrdup(&temp[j]));
	else return(NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: get_utm_coordinates
|
|       Purpose: Gets UTM coordinate parameters from the 
|		 display environment file, including:
|		 pixel_dim_x,
|		 pixel_dim_y,
|		 utm_00_N,
|		 utm_00_E, and
|		 utm_00_zone,
|		 and stores these values in the global spc_display structure.
|
|         Input: file - open stream to display environment file
|        Output: None
|       Returns: a single entry from the display environment file
|    Written By: Danielle Argiro
|          Date: Aug 8, 1994
| Modifications:
|
------------------------------------------------------------*/

static int get_utm_coordinates(
    kfile *file)
{
	int  status, returnval = TRUE;
	char *line;
	char temp[KLENGTH];
	char zone[KLENGTH];

	/*
	 * get pixel dimension for x 
	 */
	line = get_environ_entry(file);
        if (line == NULL) 
        {
            kerror(NULL, "spc_read_environ_file", "Premature End of File read");
            return(FALSE);
        }
	status = sscanf(line, pixel_dim_x_scan, temp, 
		        &(spc_display->pixel_dim_x));
	if ((status == 1) && (kstrcmp(temp, "pixel_dim_x:") == 0))
	{
	    kwarn(NULL, "spc_read_environ_file", "pixel_dim_x value not specified in display environment file; UTM coordinates will not be displayed");
	    returnval = OMIT_UTM;
	}
	else if (status != 2)
	{
	    kerror(NULL, "spc_read_environ_file",
		   "Unable to read first line of info in display environment file, which should read as in the following example:\npixel_dim_x: '45.0'\n");
	    return(FALSE);
	}
	kfprintf(kstderr, "pixel_dim_x: %g\n", spc_display->pixel_dim_x);
	kfree(line);
	

	/*
         * get pixel dimension for y
         */
        line = get_environ_entry(file);
        if (line == NULL) 
        {
            kerror(NULL, "spc_read_environ_file",
                  "Premature End of File read");
            return(FALSE);
        }
        status = sscanf(line, pixel_dim_y_scan, temp,
		        &(spc_display->pixel_dim_y));
	if ((status == 1) && (kstrcmp(temp, "pixel_dim_y:") == 0))
        {
            kwarn(NULL, "spc_read_environ_file", "pixel_dim_y value not specified in display environment file; UTM coordinates will not be displayed");
            returnval = OMIT_UTM;
        }
        else if (status != 2)
        {
            kerror(NULL, "spc_read_environ_file",
                   "Unable to read second line of info in display environment file, which should read as in the following example:\npixel_dim_y: '45.0'\n");
            return(FALSE);
        }
        kfprintf(kstderr, "pixel_dim_y: %g\n", spc_display->pixel_dim_y);
        kfree(line);

	/*
         * get UTM 00 N coordinate 
         */
        line = get_environ_entry(file);
        if (line == NULL)
        {
            kerror(NULL, "spc_read_environ_file",
                  "Premature End of File read");
            return(FALSE);
        }
        status = sscanf(line, utm_00_n_scan, temp,
			&(spc_display->utm_00_N));
	if ((status == 1) && (kstrcmp(temp, "utm_00_N:") == 0))
        {
            kwarn(NULL, "spc_read_environ_file", "utm_00_N value not specified in display environment file; UTM coordinates will not be displayed");
            returnval = OMIT_UTM;
        }
        else if (status != 2)
        {
            kerror(NULL, "spc_read_environ_file",
                   "Unable to read third line of info in display environment file, which should read as in the following example:\nutm_00_N: '4570950.0'\n");
            return(FALSE);
        }
        kfprintf(kstderr, "utm_00_n: %g\n", spc_display->utm_00_N);
        kfree(line);

        /*
         * get UTM 00 E coordinate
         */
        line = get_environ_entry(file);
        if (line == NULL) 
        {
            kerror(NULL, "spc_read_environ_file",
                  "Premature End of File read");
            return(FALSE);
        }
        status = sscanf(line, utm_00_e_scan, temp,
		        &(spc_display->utm_00_E));
        if ((status == 1) && (kstrcmp(temp, "utm_00_E:") == 0))
        {
            kwarn(NULL, "spc_read_environ_file", "utm_00_E value not specified in display environment file; UTM coordinates will not be displayed");
            returnval = OMIT_UTM;
        }
        else if (status != 2)
        {
            kerror(NULL, "spc_read_environ_file",
                   "Unable to read fourth line of info in display environment file, which should read as in the following example:\nutm_00_E: '258720.0'\n");
            return(FALSE);
        }
        kfprintf(kstderr, "utm_00_e: %g\n", spc_display->utm_00_E);
        kfree(line);


	/*
         * get UTM zone 
         */
        line = get_environ_entry(file);
        if (line == NULL) 
	{
	    kerror(NULL, "spc_read_environ_file",
		  "Premature End of File read");
	    return(FALSE);
	}
        status = sscanf(line, utm_zone_scan, temp, zone);
        if ((status == 1) && (kstrcmp(temp, "utm_zone:") == 0))
        {
            kwarn(NULL, "spc_read_environ_file", "utm_zone value not specified in display environment file; UTM coordinates will not be displayed");
            returnval = OMIT_UTM;
        }
        else if (status != 2)
        {
            kerror(NULL, "spc_read_environ_file",
                   "Unable to read fifth line of info in display environment file, which should read as in the following example:\nutm_zone: '18'\n");
            return(FALSE);
        }
	spc_display->utm_zone = kstrdup(zone);
        kfprintf(kstderr, "utm_zone: %s\n", spc_display->utm_zone);
        kfree(line);
	
	return(returnval);
}
