 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Functionality routines for master spectrum
   >>>> 
   >>>>  Private: 
   >>>>          spectrum_add_class
   >>>>          spectrum_delete_class
   >>>>          spectrum_select
   >>>>          spectrum_add_cluster_manually
   >>>>          spectrum_delete_cluster_manually
   >>>>          spectrum_autoclass
   >>>>          spectrum_reassign
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "spectrum.h"

/*-----------------------------------------------------------
| 
|  Routine Name: spectrum_add_class
| 
|       Purpose: Do routine which is called when
|                  pane action button add_class is used
| 
|         Input: master_info - ptr to FormInfo struct for spectrum
| 
|        Output: None
|    Written By: Danielle Argiro
|          Date: Nov 24, 1993
| Modifications: 
| 
------------------------------------------------------------*/
/* ARGSUSED */
void spectrum_add_class(
     gui_info_struct *master_info)
{
	int          i;
	LegendEntry *legend_ptr;

        /*
         * make sure they've named the last class
         * before letting them start a new one
         */
        legend_ptr = spc_legend_list;
        while (legend_ptr != NULL)
            legend_ptr = legend_ptr->next;

	/* 
	 * add the legend entry that represents the class
	 */
        legend_ptr = spc_add_legend_entry(-1, 0, 0, 0, "New Category",
                                          NULL, 0, SPC_USER_CREATED);

	spc_set_current_class(legend_ptr);

kfprintf(kstderr, "Classes\n\n");
legend_ptr = spc_legend_list;
while (legend_ptr != NULL)
{
   kfprintf(kstderr, "Class %d -- %s\n", legend_ptr->class, legend_ptr->classname);
   kfprintf(kstderr, "            clusters:\n");
   for (i = 0; i < legend_ptr->clusternum; i++)
       kfprintf(kstderr,"                       %d\n", legend_ptr->clusters[i]);
   legend_ptr = legend_ptr->next;
}
kfprintf(kstderr, "\n\n");
}


/*-----------------------------------------------------------
| 
|  Routine Name: spectrum_delete_class
| 
|       Purpose: Do routine which is called when
|                  pane action button delete_class is used
| 
|         Input: master_info - ptr to FormInfo struct for spectrum
| 
|        Output: None
|    Written By: Danielle Argiro
|          Date: Nov 24, 1993
| Modifications: 
| 
------------------------------------------------------------*/
/* ARGSUSED */
void spectrum_delete_class(
     gui_info_struct *master_info)
{
	char            prompt[KLENGTH], label[KLENGTH];
	int             listsize;
	char           **classnames;
	LegendEntry     *legend_ptr;
	xvw_list_struct *class_list_return;

	if (spc_image == NULL) 
	{
	    kerror(NULL, "spectrum_delete_class",
		   "You must display an image w/ classes before deleting classes!");
	    kinfo(KHOSTILE, "Don't you have any common sense AT ALL?");
	    return;
	}
	if (spc_legend_list == NULL)
	{
	    kerror(NULL, "spectrum_delete_class",
		   "No classes to delete!");
	    kinfo(KHOSTILE, "Can't you see that? Why should I have to tell you?");
	    return;
	}

        /*
         * prompt to see which class they want to delete
         */
        ksprintf(prompt, "Select Class To Delete:");
        ksprintf(label,  "Current Classes");
        classnames = spc_create_strings_category_list(NULL, SPC_ALL,
                                                      &listsize, NULL, 0);
        class_list_return = xvu_run_list_wait(classnames, listsize, prompt,
                                               label, -1, FALSE);
        if (class_list_return == NULL) return;

        /*
         *  find class to delete from user's choice of list items
         */
        legend_ptr = spc_find_legend_from_classname(class_list_return->string);

	/*
	 * delete the legend entry that represents the class
	 */
	spc_delete_legend_entry(legend_ptr);

}


/*-----------------------------------------------------------
| 
|  Routine Name: spectrum_select
| 
|       Purpose: Do routine which is called when
|                  pane action button select is used
| 
|         Input: master_info - ptr to FormInfo struct for spectrum
| 
|        Output: None
|    Written By: Danielle Argiro
|          Date: Nov 24, 1993
| Modifications: 
| 
------------------------------------------------------------*/
/* ARGSUSED */
void spectrum_select(
     gui_info_struct *master_info)
{
        int             listsize, dummy = 0;
        char            **classnames;
        char            label[KLENGTH], prompt[KLENGTH];
        LegendEntry     *legend_ptr;
        xvw_list_struct *class_list_return;

        if (spc_legend_list == NULL)
        {
            kerror(NULL, "spectrum_select", "No classes to select from");
	    kinfo(KHOSTILE, "Obviously!  (What a pinhead).");
            return;
        }

        /*
         *  set up prompts w/ old map column names entered
         */
        if (gui_info->cluster_operation_val == SPC_ADD)
            ksprintf(prompt, "Select Class To Which to Add Clusters:");
        else ksprintf(prompt, "Select Class From Which to Delete Clusters:");

        ksprintf(label,  "Current Classes");

        /*
         * prompt to see which category they want
         */
        classnames = spc_create_strings_category_list(NULL, SPC_ALL,
                                                      &listsize, NULL, 0);
        class_list_return = xvu_run_list_wait(classnames, listsize, prompt,
                                               label, dummy, FALSE);
        if (class_list_return == NULL)
                return;

        /*
         *  find current class from user's choice of list items
         */
        legend_ptr = spc_find_legend_from_classname(class_list_return->string);

        /*
         * set the current class
         */
        spc_set_current_class(legend_ptr);
}

/*-----------------------------------------------------------
| 
|  Routine Name: spectrum_add_cluster_manually
| 
|       Purpose: Do routine which is called when
|                  pane action button add_cluster_manually is used
| 
|         Input: master_info - ptr to FormInfo struct for spectrum
| 
|        Output: None
|    Written By: Danielle Argiro
|          Date: Jul 15, 1994
| Modifications: 
| 
------------------------------------------------------------*/
/* ARGSUSED */
#define QUERY_NUM 10

void spectrum_add_cluster_manually(
     gui_info_struct *master_info)
{
	int  i, cluster;
	char temp[KLENGTH];
	char **answers;

	if ((spc_current_class == NULL) || (spc_legend_lookup == NULL))
	{
	    kerror(NULL, "spectrum_add_cluster_manually",
		   "There is no current class to which to add clusters!");
	    kinfo(KHOSTILE, "Use your head, will you?");
	    return;
	}

	answers  = (char **) kcalloc(QUERY_NUM, sizeof(char *));
	for (i = 0; i < QUERY_NUM; i++)
	    answers[i] = (char *) kcalloc(1, KLENGTH * sizeof(char));

	/*
         *  prompt for new names
         */
        if (!(xvu_query_wait("Specify Clusters To be Added",
                              NULL, "ok", answers, QUERY_NUM, 15)))
	{
	    karray_free(answers, QUERY_NUM, NULL);
	    return;
	}

	for (i = 0; i < QUERY_NUM; i++)
        {
	    kstring_cleanup(answers[i], temp);
	    if (kstrlen(temp) == 0) continue;
	    else cluster = atoi(answers[i]);

	    if (spc_legend_lookup[cluster] != NULL)
		kinfo(KSTANDARD, 
		      "Cluster number %d has already been assigned to class %s",
	              cluster, spc_legend_lookup[cluster]->classname);
	    else if (spc_add_cluster_to_class(spc_current_class, cluster))
            {
                kfprintf(kstderr,
                         "Adding cluster %d to current class '%s' \n",
                         cluster, spc_legend_lookup[cluster]->classname);
            }

            /* if this is the first cluster we add to a class, need to
                   set the pseudocolor list */
            if ((spc_current_class->clusters != NULL) &&
                (spc_current_class->clusters[0] == cluster))
                spc_set_pseudo_color(spc_current_class);

	}
}


/*-----------------------------------------------------------
| 
|  Routine Name: spectrum_delete_cluster_manually
| 
|       Purpose: Do routine which is called when
|                  pane action button delete_cluster_manually is used
| 
|         Input: master_info - ptr to FormInfo struct for spectrum
| 
|        Output: None
|    Written By: Danielle Argiro
|          Date: Jul 15, 1994
| Modifications: 
| 
------------------------------------------------------------*/
/* ARGSUSED */
void spectrum_delete_cluster_manually(
     gui_info_struct *master_info)
{
	int  i, cluster;
        char temp[KLENGTH];
        char **answers;

        if ((spc_current_class == NULL) || (spc_legend_lookup == NULL))
        {
            kerror(NULL, "spectrum_delete_cluster_manually",
                   "There is no current class to which to delete clusters!");
            kinfo(KHOSTILE, "Use your head, will you?");
            return;
        }

        answers  = (char **) kcalloc(QUERY_NUM, sizeof(char *));
        for (i = 0; i < QUERY_NUM; i++)
            answers[i] = (char *) kcalloc(1, KLENGTH * sizeof(char));

        /*
         *  prompt for new names
         */
        if (!(xvu_query_wait("Specify Clusters To be Deleted",
                              NULL, "ok", answers, QUERY_NUM, 15)))
        {
            karray_free(answers, QUERY_NUM, NULL);
            return;
        }

        for (i = 0; i < QUERY_NUM; i++)
        {
            kstring_cleanup(answers[i], temp);
            if (kstrlen(temp) == 0) continue;
            else cluster = atoi(answers[i]);

            if (spc_legend_lookup[cluster] != spc_current_class)
                kinfo(KSTANDARD,
                      "Cluster number %d is not a member of class %s", 
		      cluster, spc_current_class->classname);
            else if (spc_delete_cluster_from_class(spc_current_class, cluster))
            {
                kfprintf(kstderr,
                         "Deleting cluster %d from current class '%s' \n",
                         cluster, spc_current_class->classname);
            }

            /* if this is the first cluster we add to a class, need to
                   set the pseudocolor list */
            if ((spc_current_class->clusters != NULL) &&
                (spc_current_class->clusters[0] == cluster))
                spc_set_pseudo_color(spc_current_class);

        }
}


/*-----------------------------------------------------------
| 
|  Routine Name: spectrum_autoclass
| 
|       Purpose: Do routine which is called when
|                  pane action button autoclass is used
| 
|         Input: master_info - ptr to FormInfo struct for spectrum
| 
|        Output: None
|    Written By: Danielle Argiro
|          Date: Sep 14, 1994
| Modifications: 
| 
------------------------------------------------------------*/
/* ARGSUSED */
void spectrum_autoclass(
     gui_info_struct *master_info)
{
	int w, h, cluster;
	int *new_class_column = NULL;
        int  i, zero_count = FALSE;
	LegendEntry *legend_ptr;

        /*
         *  Lots of error checking!
         */

	/* can't autoclassify without covariance matrix */
	if (spc_covar_matrix == NULL)
	{
            kerror(NULL, "spectrum_autoclass", "Your input image has no covariance matrix included in the codebook. The autoclassing procedure can only be performed on images that include a covariance matrix in the codebook. Sorry.");
            return;
	}
	/* can't autoclassify with any pixels having 0 count*/
	for (i = 0; i < spc_map_rownum; i++)
	{
    	    if (spc_count[i] == 0)
		zero_count = TRUE;
	}
	if (zero_count)	
	{
    	    kinfo(KSTANDARD, "Codebook contains entries indexed by pixel values that do not appear in the image (ie, the codebook contains pixels with a zero count); this means that you cannot use the AutoClassify feature with this image.");
	    return;
	}

	/* can't autoclassify if there are no classes established */
        if (spc_legend_list == NULL)
        {
            kerror(NULL, "spectrum_autoclass", "No classes currently exist. You must create at least two classes (more classes are usually better) and add at least one cluster to each class before using the 'Autoclass' feature.  Note that the more carefully you choose the clusters in each class, the more effective the autoclass procedure will be.");
            return;
        }

	/* can't autoclassify if there are not at least 2 classes established */
        if (spc_legend_classnum < 2)
        {
            kerror(NULL, "spectrum_autoclass", "Only one class currently exists. The autoclassing procedure will be worthless unless you first identify and add cluster(s) to at least two classes. Actually, more than two classes is recommended.");
            return;
        }

	/*
        legend_ptr = spc_legend_list;
        while (legend_ptr != NULL)
        {
            if (legend_ptr->clusternum == 0)
            {
                kwarn(NULL, "spectrum_autoclass", "The '%s' class does not have any clusters assigned to it.  You must have at least one cluster assigned to each class in order for the autoclassing procedure to work.", legend_ptr->classname);
            }
            legend_ptr = legend_ptr->next;
        }
	*/

        /*
         *  get the width and height of the image, then call the routine
         *  to classify the remaining clusters.  this routine will return
         *  a new class column;  over-write our current class column with
         *  the class column returned.
         */

	kpds_get_attribute(spc_image, KPDS_VALUE_SIZE,
                           &w, &h, NULL, NULL, NULL);
	new_class_column = classifyRemainingClusters(spc_image,
                                  h, w, spc_map_colnum, spc_map_rownum, 
	                          spc_count, spc_maps, spc_covar_matrix, 
                                  spc_classes, new_class_column);

	kfree(spc_classes);
	spc_classes = new_class_column;

	/*
	 * now, go through the new class column and make sure that the
	 * cluster corresponding to each index is added to the legend.
	 * update the display of Class contents subform on the way.
	 */
	for (cluster = 0; cluster < spc_map_rownum; cluster++)
	{ 
	     if (spc_legend_lookup[cluster] == NULL)
	     {
		legend_ptr = spc_find_class_from_id(spc_classes[cluster]);
	        if (legend_ptr != NULL)
	        {
                    (void) spc_add_cluster_to_class(legend_ptr, cluster);
                    spc_update_classcontents_label(legend_ptr);
		}
	     }

	}

}


/*-----------------------------------------------------------
| 
|  Routine Name: spectrum_cluster_operation
| 
|       Purpose: Do routine which is called when
|                  toggle selection cluster_operation is used
| 
|         Input: master_info - ptr to FormInfo struct for spectrum
| 
|        Output: None
|    Written By: Danielle Argiro
|          Date: Nov 15, 1994
| Modifications: 
| 
------------------------------------------------------------*/
/* ARGSUSED */
void spectrum_cluster_operation(
     gui_info_struct *master_info)
{
	/* PUT YOUR CODE HERE ! */
}


/*-----------------------------------------------------------
| 
|  Routine Name: spectrum_reassign
| 
|       Purpose: Do routine which is called when
|                  pane action button reassign is used
| 
|         Input: master_info - ptr to FormInfo struct for spectrum
| 
|        Output: None
|    Written By: Danielle Argiro
|          Date: Feb 14, 1995
| Modifications: 
| 
------------------------------------------------------------*/
/* ARGSUSED */
void spectrum_reassign(
     gui_info_struct *master_info)
{
	int     *unique_values; 
	int     i, w, h, d, t, e, found, dest_size, offset; 
	int     dest_cluster;
	int     unique_cluster_number, reassign_cluster_number;
	int     roi_x, roi_y;
	int     *image_values = NULL, *mask_values = NULL;
	kobject tmp_roi, data_object;
	char    **destinations = NULL, **cluster_list;
	char    *Cprompt = "Choose cluster(s) to be reassigned to new class";
	char    *Clabel  = "Unique cluster numbers appearing in polygon";
	char    *Dprompt = "Choose new class for clusters";
	char    *Dlabel  = "Eligible classes";
	LegendEntry *legend_ptr;
	int     reassign_clusters[KLENGTH];

	xvw_list_struct *dest_list_return = NULL, **clust_list_return = NULL;

	/*
	 * create list of classes having at least one cluster assigned, 
	 * that can be potential destinations for re-assigned clusters.
	 * if none exist, might as well return immediately.
	 */
	destinations = spc_create_strings_category_list(NULL, SPC_NOTEMPTY,
				&dest_size, NULL, 0);
	if (dest_size == 0)
	{
	    kerror(NULL, "spectrum_reassign", "There are currently no classes with at least one cluster assigned. Note that an established class must contain at least one previously assigned cluster before you can reassign other clusters to that class.");
	    return;
	}

	/*
	 * get ROI from zoom window
	 */
	if (gui_info->Zoom->zoom->zoommode == KZOOM_UM_BUTTON_PRESS)
	{
	    xvw_set_attribute(zoom_workspace, XVW_IMAGE_ROI_SHAPE,
			      KIMAGE_ROI_POLYLINE);
	    xvw_get_attribute(zoom_workspace, XVW_IMAGE_ROI, &tmp_roi);
	    xvw_get_attribute(zoom_workspace, XVW_IMAGE_IMAGEOBJ, &data_object);

	}
	/*
	 * get ROI from main spectrum workspace
	 */
	else
	{
            xvw_set_attribute(spc_display->workspace, XVW_IMAGE_ROI_SHAPE,
                              KIMAGE_ROI_POLYLINE);
            xvw_get_attribute(spc_display->workspace, XVW_IMAGE_ROI, &tmp_roi);
	    xvw_get_attribute(spc_display->workspace, XVW_IMAGE_IMAGEOBJ, 
			      &data_object);
	}
	data_object = kpds_reference_object(data_object);

	/*
	 * from the ROI, get an integer array of unique clusters
	 */
	unique_values = spc_find_unique_values_from_polygon(tmp_roi, 
							&unique_cluster_number);

	/*
	 * from the integer array of unique clusters, get an array of 
	 * strings containing "Cluster Number %d" for prompting the user.
	 */
	cluster_list = spc_create_strings_cluster_list(unique_values, 
						       unique_cluster_number);

	/*
	 * less than one unique cluster: some sort of bizarre error
	 */
	if (unique_cluster_number < 1)
	{
	    kerror(NULL, "spectrum_reassign", "Region of interest specified has no whole clusters inside");
	    return;
	}
	/*
	 *  just one unique cluster: don't bother prompting
	 */
	else if (unique_cluster_number == 1)
	    reassign_cluster_number = 1;
	/*
	 *  more than one unique cluster: prompt for which ones to reassign
	 */
	else 
	{
	    offset = 0;
	    if (unique_cluster_number > 1) offset = 1;
	    clust_list_return = xvu_run_list_multsel_wait(cluster_list, 
						 unique_cluster_number +offset, 
						  Cprompt, Clabel, 
						  FALSE, FALSE, 
					          &reassign_cluster_number);
	    /* no clusters chosen for re-assignment; return */
            if (clust_list_return == NULL) return;
	}


	/* 
	 * based on user's choice from displayed list of cluster numbers,
	 * construct array of clusters for reference later.
	 */
	if ((reassign_cluster_number == 1) && (unique_cluster_number == 1))
	{
	    /* there was only one cluster in question to begin with */
	    ksscanf(cluster_list[0], "Cluster Number %d",
                    &reassign_clusters[0]);
	}
	else
	{
	    if (kstrcmp(clust_list_return[0]->string, "ALL Clusters") == 0)
	    {
		/* all available clusters to be reassigned */
                for (i = 0; i < unique_cluster_number; i++)
                    ksscanf(cluster_list[i], "Cluster Number %d",
                            &reassign_clusters[i]);
		reassign_cluster_number = unique_cluster_number;
	    }
	    else if (reassign_cluster_number == 1)
	    {
	        /* more than one cluster possible, but only one chosen */
	        ksscanf(clust_list_return[0]->string, "Cluster Number %d",
                        &reassign_clusters[0]);
	    }
	    else 
	    { 
	        /* more than one cluster possible, more than one chosen */
	        for (i = 0; i < reassign_cluster_number; i++)
	            ksscanf(clust_list_return[i]->string, "Cluster Number %d",
                            &reassign_clusters[i]);
	    }
	}

	/*
	 * prompt user for choice of possible destination classes.
	 */
	dest_list_return = xvu_run_list_wait(destinations, dest_size, Dprompt,
                                             Dlabel, 0, FALSE);
        if (dest_list_return == NULL) return;

	/*
	 * get the chosen destination class from it's name;  reassigned 
	 * clusters will take on the cluster value of whichever cluster
	 * happens to appear first in the new class.
	 */
	legend_ptr = spc_find_legend_from_classname(dest_list_return->string);
	dest_cluster = legend_ptr->clusters[0];

	/*
	 * get all the value & mask data in a single region from the ROI.
	 */
        kpds_set_attribute(tmp_roi, KPDS_VALUE_DATA_TYPE,   KINT);
        kpds_get_attribute(tmp_roi, KPDS_VALUE_SIZE,       &w, &h, &d, &t, &e);
	kpds_set_attribute(tmp_roi, KPDS_VALUE_REGION_SIZE, w, h, d, t, e);
	kpds_set_attribute(tmp_roi, KPDS_VALUE_POSITION,    0, 0, 0, 0, 0);

        kpds_set_attribute(data_object, KPDS_VALUE_DATA_TYPE, KINT);
	image_values = (int *) kpds_get_data(tmp_roi, KPDS_VALUE_REGION, NULL);

	kpds_set_attribute(tmp_roi, KPDS_MASK_REGION_SIZE,  w, h, d, t, e);
        kpds_set_attribute(tmp_roi, KPDS_MASK_DATA_TYPE,  KINT);
	kpds_set_attribute(tmp_roi, KPDS_MASK_POSITION,  0, 0, 0, 0, 0);

	mask_values  = (int *) kpds_get_data(tmp_roi, KPDS_MASK_REGION,  NULL);

	/*
	 * go through the entire ROI, replacing image data values with
	 * the new cluster value whenever appropriate.
	 */
	for (i = 0; i < w*h*d*t*e; i++)
	{
	    /* do not replace value if it was masked out (not in the polygon) */
	    if (mask_values[i] != 0)
	    {
		/* do not replace value if it was not selected from the list */
		found = karray_locate((kaddr)reassign_clusters, 
				       (kaddr)image_values[i], 
				       reassign_cluster_number);

		/* replace unmasked, selected values with new cluster value */
		if (found != -1) image_values[i] = dest_cluster;
	    }
	}

	/*
	 *  put the region with the modified image data values 
	 *  back into the original image in its original location.
	 */
        kpds_get_attribute(tmp_roi, KPDS_SUBOBJECT_POSITION,
                           &roi_x, &roi_y, NULL, NULL, NULL);
	kpds_set_attribute(data_object, KPDS_VALUE_POSITION, 
			   roi_x, roi_y, 0, 0, 0);
	kpds_set_attribute(data_object, KPDS_VALUE_REGION_SIZE, w, h, d, t, e);

	/*
	 *  put modified data back into image 
	 */
	kpds_put_data(data_object, KPDS_VALUE_REGION, image_values);

	/* cleanup */
	kfree(image_values); 
	kfree(mask_values);
	kpds_close_object(data_object);
}

