#include <xvinclude.h>

/*
 * This program creates an image object which displays the ball image.
 * Then, a zoom object is created, and is associated with the same information
 * as the image object.  Without installing any event handlers, the zoom 
 * object will automatically zoom in on the image displayed in the image object
 * when the pointer is moved across the image, since they are both using the 
 * same data object.
 *
 * A button created beneath the image has a callback installed which will
 * switch the zoom update mode from "Continuous" to "Button Press" and 
 * vice versa.
 */
static void set_updatemode PROTO((xvobject, kaddr, kaddr));

void main(
   int  argc,
   char *argv[],
   char *envp[])
{
	kobject  object;
	char     *filename = "image:mandril";
	xvobject image, zoom, position, parent, button;

        /* initialize Khoros program */
        khoros_initialize(argc, argv, envp, "ENVISION");

	/* allow different images to be used, as in "% example image:lizard" */
	if (argc > 1)
	   filename = argv[1];

	/* open the data object associated with the input image */
	object = kpds_open_input_object(filename);

	/* initialize the xvwidgets lib */
	if (!xvw_initialize(XVW_MENUS_XVFORMS))
	{
	   kerror(NULL, "main", "unable to open display");
	   kexit(KEXIT_FAILURE);
	}

	/*
	 * a manager will be the backplane for the image and position objects
	 */
	parent = xvw_create_manager(NULL, "parent");

	/*
	 *  create the image object; associate it with the data object
         *  representing the input file
	 */
	image  = xvw_create_image(parent, "image");
	xvw_set_attribute(image, XVW_IMAGE_IMAGEOBJ, object);

	/*
	 *  create the position object under the image object; have it
         *  use the same data object.
	 */
	position = xvw_create_position(parent, "position");
	xvw_set_attributes(position,
			   XVW_BELOW,           image,
			   XVW_POSITION_OBJECT, object,
			   XVW_LEFT_OF,         NULL,
			   XVW_RIGHT_OF,        NULL,
			   NULL);

	/*
	 *  create the zoom object in an independant window. 
         *  specify the associated image object to be the same data.
	 */
	zoom = xvw_create_zoom(NULL, "zoom");
	xvw_set_attributes(zoom, 
			   XVW_IMAGE_IMAGEOBJ,  object,
			   XVW_ZOOM_FACTOR,     85.0,
			   XVW_ZOOM_UPDATEMODE, KZOOM_UM_BUTTON_PRESS,
			   XVW_MAXIMUM_WIDTH,   200,
			   XVW_MAXIMUM_HEIGHT,  200,
			   NULL);

	/*
	 * create a button that the user can click on to change
	 * the update mode of the zoom object. (by default, the
         * update mode is KZOOM_UM_CONTINUOUS
	 */
	button = xvw_create_button(parent, "button");
	xvw_set_attributes(button,
                           XVW_BELOW,    image,
			   XVW_RIGHT_OF, NULL,
			   XVW_LABEL,    "Button Press",
			   NULL);

	xvw_add_callback(button, XVW_BUTTON_SELECT, set_updatemode, zoom);

	/* display and run */
	xvf_run_form();
}

/*
 *  the callback for the button simply queries the current update mode
 *  on the zoom object, and switches it to the other type.
 */
void set_updatemode(
    xvobject object,
    kaddr    client_data,
    kaddr    call_data)
{
	int  updatemode;
	char label[KLENGTH];
	xvobject zoom = (xvobject) client_data;

	xvw_get_attribute(zoom, XVW_ZOOM_UPDATEMODE, &updatemode);

	if (updatemode == KZOOM_UM_CONTINUOUS)
	{
	    updatemode = KZOOM_UM_BUTTON_PRESS;
	    ksprintf(label, "Button Press");
	}
	else 
	{
	    updatemode = KZOOM_UM_CONTINUOUS;
	    ksprintf(label, "Continuous");
	}

	xvw_set_attribute(zoom, XVW_ZOOM_UPDATEMODE, updatemode);

	xvw_set_attribute(object, XVW_LABEL, label);
	
}
