#include <xvinclude.h>

/*
 * Like the first example, this program creates an image object which displays 
 * the kitten image, and then creates a colorcell object.  The event handler
 * which is installed for button press in the image will register the value 
 * of the first pixel that you click on.  Further button press events in the
 * image will cause pixels to take on the same color as the first pixel.
 *
 * The "restore" button will restore the pixels to their original colors,
 * and clear the colorcell list of all the previously registered pixels
 * so that the process may be repeated.
 */

void  restore_colorcell	PROTO((xvobject, kaddr, kaddr));
void  quit_cb          	PROTO((xvobject, kaddr, kaddr));
void  update_colorcell	PROTO((xvobject, kaddr, XEvent *, Boolean *));

void main(
   int  argc,
   char **argv,
   char **envp)
{
	char *filename = "image:kitten";
	xvobject manager, image, colorcell, restore, quit;

        /* initialize Khoros program */
        khoros_initialize(argc, argv, envp, "ENVISION");

        /* initialize the xvwidgets lib */
	if (!xvw_initialize(XVW_MENUS_XVFORMS))
	{
	   kerror(NULL, "main", "unable to open display");
	   kexit(KEXIT_FAILURE);
	}

	if (argc > 1)
	   filename = argv[1];

        /* Create a manager to parent the image and colorcell objects  */
	manager = xvw_create_manager(NULL, "manager");

	/*
	 *  create the colorcell object, and associate it with the
         *  kitten image.  setting XVW_COLORCELL_UPDATE_ONADD to TRUE
         *  will cause pixels to take on the color displayed by the
         *  colorcell when they are added to the colorcell pixel list.
	 */
	colorcell = xvw_create_colorcell(manager, "colorcell");
	xvw_set_attributes(colorcell,
		XVW_COLORCELL_UPDATE_ONADD, TRUE,
		XVW_COLOR_COLORFILE, filename,
		XVW_BELOW,	     NULL,
		XVW_LEFT_OF,	     NULL,
		XVW_WIDTH,           35,
		XVW_HEIGHT,          35,
		NULL);

	image = xvw_create_image(manager, "image");
	xvw_set_attributes(image,
		XVW_IMAGE_IMAGEFILE, filename,
		XVW_IMAGE_BACKING,   FALSE,
		XVW_BELOW,	     colorcell,
		NULL);

	/* add the event handler to update the colorcell */
	xvw_add_event(image, ButtonPressMask,
                      update_colorcell, (kaddr)colorcell);

	/* add a restore button to restore colors of pixels */
	restore = xvw_create_button(manager, "restore");
	xvw_set_attributes(restore, 
			   XVW_LABEL,   "restore",
			   XVW_LEFT_OF,  colorcell,
			   XVW_RIGHT_OF, NULL,
			   NULL);
	xvw_add_callback(restore, XVW_BUTTON_SELECT,
		         restore_colorcell, colorcell);

	/* add a quit button */
        quit = xvw_create_button(manager, "quit");
        xvw_set_attributes(quit /*restore*/,
                           XVW_LABEL,   "quit",
                           XVW_LEFT_OF,  colorcell,
                           XVW_RIGHT_OF, restore,
                           NULL);

        xvw_add_callback(quit, XVW_BUTTON_SELECT,
                         quit_cb, manager);

	/* display & run */
	xvf_run_form();
}


/*
 *  event handler to update the colorcell index 
 */
void  update_colorcell(
    xvobject object, 
    kaddr    clientData,  
    XEvent   *event,
    Boolean  *dispatch)
{
	xvobject colorcell = (xvobject) clientData;
	double   value;

	/* get the value of the pixel where the mouse was clicked */
	xvw_get_attribute(object, XVW_IMAGE_VALUE, &value);

	/* 
	 * add the pixel to the colorcell pixel list.  since
         * XVW_COLORCELL_UPDATE_ONADD has been set to TRUE, the
         * pixel will be immediately recolored to match the current
         * color displayed by the colorcell.  if this is the first
         * time the event handler has been called, the pixel to be
         * added will determine the color of the colorcell.
         */ 
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, (int) value);
}

/*
 *  callback to restore colorcell
 */
void  restore_colorcell(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	xvobject colorcell = (xvobject) client_data;

	/* 
	 * first, restore all the pixels in the colorcell list 
	 * to their original colors
         */
	xvw_set_attribute(colorcell, XVW_COLORCELL_RESTORE, TRUE);

	/*
	 *  then, clear the colorcell list so that the process
         *  may be repeated with new pixels.
	 */
	xvw_set_attribute(colorcell, XVW_COLORCELL_CLEAR, TRUE);
}


/*
 *  callback to quit program
 */
void  quit_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
        xvobject manager = (xvobject) client_data;
	xvw_destroy(manager);
	kexit(KEXIT_SUCCESS);
}

