 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Utility routines for guise
   >>>> 
   >>>>  Private: 
   >>>> 	update_form
   >>>> 	display_empty_form
   >>>> 	xvf_calculate_y_position
   >>>> 	output_display_form
   >>>> 	reload_form
   >>>> 	get_selected_gui_items
   >>>> 	check_all_variables
   >>>> 
   >>>>   Static: 
   >>>> 	check_selection_variables
   >>>> 	get_selnames
   >>>>   Public: 
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "guise.h"

static int get_selnames     PROTO((char ***, kselection *, int *));
static char *check_selection_variables PROTO((kselection *));


/*-----------------------------------------------------------
|
|  Routine Name: update_form
|
|       Purpose: This is the callback routine for the detection of
|                a change in the UIS file. Unmaps the old form, creates
|                and maps the new form.
|
|         Input: object      - object associated with file
|                filename    - name of UIS file
|                client_data - client_data passed in
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro & Mark Young
|          Date: Mon Dec 7, 1992
| Modifications: Modified from Khoros 1.0 preview
|
------------------------------------------------------------*/
int update_form(
   xvobject object,
   char     *filename,
   kaddr    client_data)
{
        int   x, y, width, height;
	char  alternate_filepath[KLENGTH];
	options_create *create_info = (options_create *) client_data;

        if (display_form != NULL)
        {
	    /* 
	     * see if the UIS file in the output file has actually changed 
	     * if not, no need to reload the form -- simply return.
	     */
	    if (identical_output_file(filename))
		return(TRUE);

	    /*
	     *  The GUI in the file IS DIFFERENT.  
             *  Save geometry, and destroy the old form
	     */
            if (display_form != NULL)
	       if (display_form->toplevel != NULL)
                   xvf_form_geometry(display_form, &x, &y, &width, &height);
            else xvf_subform_geometry(display_form->subform, &x, &y, 
	 			      &width, &height);
	    xvf_unmap_form(display_form);
            xvf_destroy_form(display_form);
        }
        else x = y = -1;

	/*
	 *  create & display the new form.
	 */
        display_form = xvf_create_form(filename, NONE, run_display, 
				       gui_info, x, y, XVF_FULL_EDIT);
	if (display_form != NULL) 
	{
            if (editor_up)
	    {
		if (display_form->toplevel != NULL)
	            xvw_busy(display_form->toplevel, TRUE);
		else if (display_form->subform->toplevel != NULL)
	            xvw_busy(display_form->subform->toplevel, TRUE);
	        else xvw_busy(NULL, TRUE);
	    }
	    alternate_filepath[0] = '\0';
/*
	    output_display_form(create_info->o, create_info->force,
				alternate_filepath);
*/
	    if (kstrlen(alternate_filepath) > 0)
	    {
		xvf_set_attribute(create_info->o_struct, XVF_FILE_NAME,
				  alternate_filepath);
		kfree(create_info->o);
		create_info->o = kstrdup(alternate_filepath);
	    }
            install_gui_attribute_callbacks();
	    put_gui_in_edit_mode();

	}
        return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: display_empty_form
|
|       Purpose: Displays the empty form as a pallette for form creation.
|
|         Input: outfile  - name of UIS file to output
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Tues Dec  8, 1992 
| Modifications: 
|
------------------------------------------------------------*/
int display_empty_form(
   char *outfile)
{
        display_form = xvf_create_form(EMPTY_FORM_UIS, NONE, run_display,
				       gui_info, -1, -1, XVF_FULL_EDIT);
	if (display_form != NULL)
	    put_gui_in_edit_mode();

	if (outfile != NULL)
 	    kcopyfile(EMPTY_FORM_UIS, outfile);

        if (display_form == NULL) 
	    return(FALSE);
	else 
	{
	    output_display_form(output_filename, TRUE, NULL);
	    return(TRUE);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: output_display_form
|
|       Purpose: Writes out the UIS file associated with the
|                currently displayed GUI.
|
|         Input: filename           - name of file to write out 
|                force              - TRUE if we are forcing over-write
|
|        Output: alternate_filename - this should be passed in as a 
|                                     static character array or NULL.  
|                                     If we could not write to the file 
|                                     specified, then we will try to use the 
|                                     directory specified by TMPDIR, or 
|                                     /usr/tmp if TMPDIR is not set.  In this 
|                                     case, alternate_filename has the filepath
|                                     where the UIS file was written, so
|                                     that we can appropriately call
|                                     xvf_set_attribute(create_info->o_struct).
|				      if alternate_filename is passed as NULL,
|                                     it is assumed that the caller does not
|                                     care what the actual filepath was.
|
|       Returns: TRUE if there is such a selection in the selection list,
|                FALSE otherwise
|    Written By: Danielle Argiro
|          Date: Wed Dec 9, 1992 
| Modifications: 
|
------------------------------------------------------------*/

int output_display_form(
   char *filename,
   int  force,
   char *alternate_filename)
{
	kfile *file;
	char  *tmp_dir;
	char  basename[KLENGTH];
	char  directory[KLENGTH];
	char  filepath[KLENGTH];

	if (alternate_filename != NULL)
	    alternate_filename[0] = '\0';

	if (display_form == NULL)
	    return(FALSE);

	if (filename == NULL)
	{
            kerror(NULL, "output_display_form", "NULL filename specified");
	    return(FALSE);
	}

	if ((force) || (koverwrite(KSTANDARD, filename)))
	{
	    (void) kdirname(filename, directory);
            if (kaccess(directory, W_OK) == -1)
            {
		(void) kbasename(filename, basename);
		tmp_dir = kgetenv("TMPDIR");
                if (tmp_dir == NULL)
                {
                    tmp_dir = "/usr/tmp";
                    kerror(NULL, "output_display_form",
                           "Directory %s is not accessible.  Attempting to write %s to %s instead.", directory, basename, tmp_dir);
                }
                else
                {
                    kerror(NULL, "get_uis_fullpath",
                          "Directory %s is not accessible.  Attempting to write %s to %s (as specified by your TMPDIR environment variable) instead.", directory, basename, tmp_dir);
                }
		ksprintf(filepath, "%s/%s", tmp_dir, basename);
		if (alternate_filename != NULL)
		    ksprintf(alternate_filename, "%s", filepath);
	    }
	    else ksprintf(filepath, "%s", filename);

	    if ((file = kfopen(filepath, "w")) == NULL)
            {
                kerror(NULL, "output_display_form",
                       "Cannot write to file %s", filename);
                return(FALSE);
            }

            if (!(kvf_print_UIS(file, display_form, force, FALSE)))
	        return(FALSE);

	    if (kstrcmp(filename, gui_info->options->create->o) == 0)
		set_save_notneeded(gui_info->options->create->save_struct);
	    else set_save_needed(gui_info->options->create->save_struct, NULL, 
				NULL, gui_info->options->create->save_struct);
            kfclose(file);

	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: reload_form
|
|       Purpose: Re-displays the currently displayed form.
|         Input: infile  - input file to read
|                outfile - output file to write (if empty form only)
|        Output: None
|       Returns: TRUE if there is such a selection in the selection list,
|                FALSE otherwise
|    Written By: Danielle Argiro
|          Date: Wed Dec 9, 1992 
| Modifications: 
|
------------------------------------------------------------*/
void reload_form(
   char *infile,
   char *outfile)
{
	options_create *create_info = gui_info->options->create;

	if (infile == NULL)
	{
	    if (display_form != NULL)
                xvf_destroy_form(display_form);
	    if (!(display_empty_form(outfile)))
		kexit(KEXIT_FAILURE);
	}

	else
	{
	    if (display_form != NULL)
	    {
	        xvf_unmap_form(display_form);
		xvf_destroy_form(display_form);
	    }
            display_form = xvf_create_form(infile, NONE, run_display,
                                           gui_info, -1, -1, XVF_FULL_EDIT);
	    xvw_sync(FALSE);
	}

	if (display_form != NULL)
	{
	    install_gui_attribute_callbacks();
	    if (display_form->master != NULL)
		xvf_set_attribute(create_info->subform_button_struct, 
			          XVF_TITLE, "New Subform");
	}

	output_display_form(output_filename, TRUE, NULL);
}


/*-----------------------------------------------------------
|
|  Routine Name: get_selected_gui_items
|
|       Purpose: Allocates and returns an array of xvobjects that are
|                the backplanes of GUI items on the displayed GUI
|                which are "selected" (have control markers bracketing 
|                them by the manager object).  Does error checking 
|                to make sure that selected GUI items are all on the 
|                same backplane.
|
|         Input: None
|        Output: num_selected    - the number of selected GUI items
|                display_control - the backplane of the selected GUI items 
|                control_type    - type of control: KPANE, KGUIDEPANE, KMASTER
|       Returns: Allocated array of kformstructs
|    Written By: Danielle Argiro
|          Date: Nov 12, 1993
| Modifications:
|
------------------------------------------------------------*/
xvobject *get_selected_gui_items(
    int      *num_selected,
    kcontrol **display_control,
    int      *control_type)
{
	ksubform *subform;
	kcontrol *pane;
	kguide   *guide;
	xvobject *selbacks, *master_selbacks, *gp_selbacks, *pane_selbacks;
	xvobject *selback_array;
	kcontrol *display_master, *display_gp, *display_pane;
	int      num_master_selected, num_gp_selected, num_pane_selected;

	*num_selected = 0;

        /* don't do anything if there is no displayed form */
        if (display_form == NULL)
            return(NULL);

	/*
	 *  error checking: have to make sure that all selected items
         *  are on the same backplane.  check for selected items on 
         *  master, subform, and pane and compare.
	 */
	num_master_selected = 0;
	num_gp_selected     = 0;
	num_pane_selected   = 0;
	master_selbacks     = NULL;
	gp_selbacks         = NULL;
	pane_selbacks       = NULL;
	*control_type        = 0;
	    
        /* might have selected items on master */
        display_master = find_displayed_control(NULL, NULL, KMASTER,
                                                &subform, &guide, &pane);
	if (display_master != NULL)
	    xvw_get_attributes(display_master->back,
                           XVW_NUM_SELECTIONS, &num_master_selected,
                           XVW_SELECTIONS,     &master_selbacks,
                           NULL);

        /* might have selected items on guidepane */
        display_gp = find_displayed_control(NULL, NULL, KGUIDEPANE,
                                                   &subform, &guide, &pane);
	if (display_gp != NULL)
            xvw_get_attributes(display_gp->back,
                           XVW_NUM_SELECTIONS, &num_gp_selected,
                           XVW_SELECTIONS,     &gp_selbacks,
                           NULL);

        /* might have selected items on pane */
        display_pane = find_displayed_control(NULL, NULL, KPANE,
                                              &subform, &guide, &pane);
        if (display_pane != NULL)
            xvw_get_attributes(display_pane->back,
                           XVW_NUM_SELECTIONS, &num_pane_selected,
                           XVW_SELECTIONS,     &pane_selbacks,
                           NULL);

	/* if no items selected anywhere, give an error */
	if ((num_master_selected == 0) &&
	    (num_gp_selected     == 0) &&
	    (num_pane_selected   == 0))
	{
	     kerror(NULL, "get_selected_gui_items",
	            "No items selected!");
	     *num_selected = 0;
	     return(NULL);
	}

	/* if items selected on more than one backplane, give an error */
	if ( ((num_master_selected  > 0)  && (num_gp_selected     > 0)) ||
	     ((num_gp_selected      > 0)  && (num_pane_selected   > 0)) ||
	     ((num_pane_selected    > 0) &&  (num_master_selected > 0)) ) 
	{
	    kerror(NULL, "get_selected_gui_items",
		   "For any group operation, all selected items MUST be on the same backplane.  You have selected items on different backplanes;  please make sure all selected items have the same backplane, and try again.");
	    *num_selected = 0;
	    return(NULL);
	}

	/* set array of selected items appropriately to backplane origin */
	*display_control = (kcontrol *) kcalloc(1, sizeof(kcontrol));
	if (num_master_selected > 0)
	{
	    *num_selected = num_master_selected;
	    *control_type = KMASTER;
	    *display_control = display_master;
	    selbacks = master_selbacks;
	}
	else if (num_gp_selected > 0)
	{
	    *num_selected = num_gp_selected;
	    *control_type = KGUIDEPANE;
	    *display_control = display_gp;
	    selbacks = gp_selbacks;
	}
	else if (num_pane_selected > 0)
	{
	    *num_selected = num_pane_selected;
	    *control_type = KPANE;
	    *display_control = display_pane;
	    selbacks = pane_selbacks;
	}
	else return(NULL);

	selback_array = (xvobject *) 
			 karray_copy((char **)selbacks, *num_selected, FALSE);
        return(selback_array);
}

/*-----------------------------------------------------------
|
|  Routine Name: check_selection_variables
|
|       Purpose: makes sure there are no duplicate variable names
|         Input: none
|        Output: none
|       Returns: NULL if there are NO duplicate variable names 
|                (implying success), the 1st duplicate variable name 
|                if there is one (implying failure)
|     
|    Written By: Danielle Argiro
|          Date: Nov 23, 1993
| Modifications:
|
------------------------------------------------------------*/
static char *check_selection_variables(
    kselection *sel_list)
{
	int  i, j, k;
	int  sel_count = 0;
	char **sel_names = NULL;
	char *return_string = NULL;
	
	if (!(get_selnames(&sel_names, sel_list, &sel_count)))
	    return(FALSE);
	for (i = 0; i < sel_count; i++)
	{
	    for (j = 0; j < sel_count; j++)
	    {
		if ((i != j) && (kstrcmp(sel_names[i], sel_names[j]) == 0))
		{
		    return_string = kstrdup(sel_names[i]);
		    for (k = 0; k < sel_count; k++)
		        kfree(sel_names[k]);    
		    kfree(sel_names);
		    return(return_string);
		}
	    }
	}
	for (k = 0; i < sel_count; k++)
	    kfree(sel_names[k]);    
	kfree(sel_names);
	return(NULL);
}


/*-----------------------------------------------------------
|
|  Routine Name: check_all_variables
|
|       Purpose: Creates a list of all the variable names off 
|                all the selections in a GUI
|         Input: none
|        Output: sel_names - pointer to the array of strings w/ variable names
|                count     - size of sel_names array
|       Returns:
|    Written By: Danielle Argiro
|          Date: Nov 23, 1993
| Modifications:
|
------------------------------------------------------------*/

char *check_all_variables(void)
{
	kform    *form;
        ksubform *subform;
        kguide   *guide;
	char     *dup_varname;

	if (display_form == NULL) return(FALSE);
	form = display_form;
	if (form->master != NULL)
	{
	    dup_varname = check_selection_variables(form->master->sel_list);
	    if (dup_varname != NULL)
		return(dup_varname); 
	    subform = form->master->subform_list;
	}
	else subform = form->subform;

	while (subform != NULL)
	{
	    if (subform->guidepane != NULL)
	    {
	        dup_varname = 
			check_selection_variables(subform->guidepane->sel_list);
	        if (dup_varname != NULL)
		    return(dup_varname); 
	        guide = subform->guidepane->guide_list;
	    }
	    else guide = subform->guide;

	    while (guide != NULL)
	    {
		dup_varname = check_selection_variables(guide->pane->sel_list);
                if (dup_varname != NULL)
                    return(dup_varname);
		guide = guide->next;
	    }
	    subform = subform->next;
	}
	return(NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: get_selnames
|
|       Purpose: gets the variable names off all the selections in 
|                the selection list passed in
|         Input: sel_list  - header of the selection list
|        Output: sel_names - pointer to the array of strings w/ variable names
|                count     - size of sel_names array
|       Returns:
|    Written By: Danielle Argiro
|          Date: Nov 23, 1993
| Modifications:
|
------------------------------------------------------------*/
static int get_selnames(
    char       ***sel_names,
    kselection  *sel_list,
    int         *count)
{
	kselection *selection;
	Line_Info  line_info;
	int        help_num = 0;
	int        quit_num = 0;

	selection = sel_list;
	kvf_clear_line_info(&line_info);
	while (selection != NULL)
	{
	    line_info.variable = NULL;
	    kvf_gen_parse(selection->line, &line_info);

	    /* this is kind of a nasty little kludge, to take care
               of the case that each pane can have a help button named "help",
	       but there cannot be two buttons named "help" on the same pane;
	       and ditto with the case of "quit" buttons.
	       So we count the occurances of "help" variables, and only add
               them into the list of selection names unless there's 2 or more */

	    if ((kstring_to_token("help") == selection->var_token) ||
		(kstring_to_token("quit") == selection->var_token))
	    {
		if (kstring_to_token("help") == selection->var_token)
		    help_num++;
		else quit_num++;
		if ((help_num > 1) || (quit_num > 1))
		{
	            *sel_names = (char **) krealloc(*sel_names, 
					(*count+2) *sizeof(char *));
		    if (*sel_names == NULL)
		        return(FALSE);
		    (*sel_names)[*count] = kstrdup(line_info.variable);
	            (*count)++;
		    (*sel_names)[*count] = kstrdup(line_info.variable);
	            (*count)++;
		}
	    }

	    /* general case, simply add variable name 
	       into array of selection names */
	    else if (line_info.variable != NULL)
	    {
	        *sel_names = (char **) krealloc(*sel_names, 
					(*count+1) *sizeof(char *));
		if (*sel_names == NULL)
		    return(FALSE);
		(*sel_names)[*count] = kstrdup(line_info.variable);
	        (*count)++;
	    }
	    kvf_free_line_info_strings(&line_info);

	    /* if this is an ME, MI, or loose group, gotta recurse */
	    if (selection->group_next != NULL)
	    {
		if (!(get_selnames(sel_names, selection->group_next, 
				    count)))
		    return(FALSE);
	    }
	    selection = selection->next;
	}
	
	return(TRUE);
}
