 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>              
   >>>>   Utilities for Updating GUI of guise
   >>>>
   >>>>  Private:    set_save_needed
   >>>>              set_save_notneeded
   >>>>              adjust_gui_from_infile
   >>>>              add_uis_misc_file
   >>>>
   >>>>              install_gui_attribute_callbacks
   >>>>              add_save_needed_callback
   >>>>   Static:
   >>>>              set_selection_save_needed
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "guise.h"
#include <kcms/private.h>

static void set_selection_save_needed PROTO((kselection *));

/*-----------------------------------------------------------
|
|  Routine Name: set_save_needed
|
|       Purpose: Sets the "Save" button to say "Save (Needed)" instead 
|                of "Save" & sets global save_needed to TRUE
|
|         Input: save_struct - the kform struct associated w/ save button
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Aug 20, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void set_save_needed(
    kform_struct *kformstruct,
    char         *attribute,
    kaddr        attr_value,
    kaddr        client_data)
{
        kform_struct *save_struct = (kform_struct *) client_data;

	/*
	 * update the Save button to say, "Save (Needed)"
	 */
        xvf_set_attribute(save_struct, XVF_TITLE, "Save (Needed)");
	save_needed = TRUE;

	/*
	 * also, don't know if the Create Master/New Subform button
         * has the correct label (they could have just deleted the last
         * subform button on a master, and reverted back to a single subform.
         * probably not, but we don't know, since that would have been done
         * directly using menuforms.  Update the button title just in case.
	 */
	if (display_form->master == NULL)
	    xvf_set_attribute(gui_info->options->create->subform_button_struct,
			      XVF_TITLE, "Create Master");	
	else xvf_set_attribute(gui_info->options->create->subform_button_struct,
                              XVF_TITLE, "New Subform");      
}

/*-----------------------------------------------------------
|
|  Routine Name: set_save_notneeded
|
|       Purpose: Sets the "Save" button to say "Save" instead 
|                of "Save (Needed)" & sets global save_needed to FALSE
|
|         Input: save_struct - the kform struct associated w/ save button
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Sept 20, 1993
| Modifications:
|
------------------------------------------------------------*/
void set_save_notneeded(
	kform_struct *save_struct)
{
	xvf_set_attribute(save_struct, XVF_TITLE, "Save");
	save_needed = FALSE;
}

/*-----------------------------------------------------------
|
|  Routine Name: adjust_gui_from_infile
|
|       Purpose: Activates or de-activates the "New Pane", 
|                "New Subform", and control destination toggle
|                depending on the ending of the UIS file 
|                (*.form, *.pane)
|
|         Input: filename    - name of UIS file
|                create_info - ptr to PaneInfo struct for the "create" pane
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Sept 21, 1993
| Modifications: 
|
------------------------------------------------------------*/
void adjust_gui_from_infile(
    char           *infile,
    options_create *create_info)
{
	char temp[KLENGTH];

        if (kstrstr(infile, ".form") != NULL)
        {
            xvf_set_attribute(create_info->subform_button_struct,
                              XVF_ACTIVATE, 1);
            xvf_set_attribute(create_info->guide_button_struct,
                              XVF_ACTIVATE, 1);
            xvf_set_attribute(create_info->control_dest_struct,
                              XVF_ACTIVATE, 1);
            ksprintf(temp, "%d", KPANE);
            xvf_set_attribute(create_info->control_dest_struct,
                              XVF_TOGGLE_VAL, temp);
	    create_info->control_dest_val = KPANE;
        }
        else if ((kstrstr(infile, ".sub") != NULL) ||
		 (kstrstr(infile, ".subform") != NULL))
        {
	    xvf_set_attribute(create_info->subform_button_struct,
                              XVF_ACTIVATE, 0);
	    xvf_set_attribute(create_info->guide_button_struct,
                              XVF_ACTIVATE, 1);
            xvf_set_attribute(create_info->control_dest_struct,
                              XVF_ACTIVATE, 1);
            ksprintf(temp, "%d", KPANE);
            xvf_set_attribute(create_info->control_dest_struct,
                              XVF_TOGGLE_VAL, temp);
	    create_info->control_dest_val = KPANE;
	}
	else if (kstrstr(infile, ".pane") != NULL)
	{
            xvf_set_attribute(create_info->subform_button_struct,
                              XVF_ACTIVATE, 0);
            xvf_set_attribute(create_info->guide_button_struct,
                              XVF_ACTIVATE, 0);
            xvf_set_attribute(create_info->control_dest_struct,
                              XVF_ACTIVATE, 0);
            ksprintf(temp, "%d", KPANE);
            xvf_set_attribute(create_info->control_dest_struct,
                              XVF_TOGGLE_VAL, temp);
	    create_info->control_dest_val = KPANE;
        }


}

/*-----------------------------------------------------------
|
|  Routine Name: install_gui_attribute_callbacks
|
|       Purpose: Installs the "set_save_needed()" callback
|                (which simply updates the "Save" button to "Save (Needed)")
|                on each and every GUI item of the displayed form,
|                so that changes to a pre-created displayed form
|                are registered on the label of the Save button.
|
|		 Also installs the "add_uis_misc_file()" callback
|                on subforms and panes, so that if the user uses the
|                menuform of the subform or pane to write out a new
|                *.subform or *.pane file to the {object}/uis directory,
|                that new file is added to the cms database.
|
|         Input: None
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Sept 22, 1993
| Modifications: 
|
------------------------------------------------------------*/
void install_gui_attribute_callbacks(void)
{
	ksubform       *subform;
	kguide         *guide;

	/* update Save button if items on master form change */
	if (display_form->master != NULL)
	{
	    add_save_needed_callback(display_form->master->back_kformstruct);
	    set_selection_save_needed(display_form->master->sel_list);

	    subform = display_form->master->subform_list;
	    while (subform != NULL)
            {
	        add_save_needed_callback(subform->back_kformstruct);
	        if (subform->back_kformstruct != NULL)
		    xvf_add_gui_callback(subform->back_kformstruct,
				         XVF_PRINT_SUBFORM,
					 add_uis_misc_file, NULL);
				         
		subform = subform->next;
	    }
	    subform = display_form->master->subform_list;
	}
	else 
	{
	    subform = display_form->subform;
	    add_save_needed_callback(subform->back_kformstruct);
                if (subform->back_kformstruct != NULL)
                    xvf_add_gui_callback(subform->back_kformstruct,
                                         XVF_PRINT_SUBFORM,
                                         add_uis_misc_file, NULL);
	}

	while (subform != NULL)
	{
            /* update Save button if items on guidepane change */
            if (subform->guidepane != NULL)
            {
		add_save_needed_callback(subform->guidepane->back_kformstruct);
                set_selection_save_needed(subform->guidepane->sel_list);

                guide = subform->guidepane->guide_list;
		while (guide != NULL)
	        {
		    add_save_needed_callback(guide->back_kformstruct);
		    add_save_needed_callback(guide->pane->back_kformstruct);
                    set_selection_save_needed(guide->pane->sel_list);
		    xvf_add_gui_callback(guide->pane->back_kformstruct,
			 	         XVF_PRINT_PANE,
				         add_uis_misc_file, NULL);
                    guide = guide->next;
	        }
                guide = subform->guidepane->guide_list;
            }
            else 
	    {
	 	guide = subform->guide;
		add_save_needed_callback(guide->back_kformstruct);
		add_save_needed_callback(guide->pane->back_kformstruct);
                set_selection_save_needed(guide->pane->sel_list);
		xvf_add_gui_callback(guide->pane->back_kformstruct,
			 	     XVF_PRINT_PANE, add_uis_misc_file, NULL);
	    }
	    subform = subform->next;
	}
}
/*-----------------------------------------------------------
|
|  Routine Name: add_save_needed_callback
|
|       Purpose: installs the "set_save_needed" callback
|                (which simply updates the "Save" button to "Save (Needed)")
|                on the kformstruct passed in.
|         Input: subform_list - list of subform pointers
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: July 28, 1994
| Modifications:
|
--------------------------------------------------------------*/

void add_save_needed_callback(
	kform_struct *kformstruct)
{
	options_create *create_info = gui_info->options->create;
	
	if (kformstruct != NULL)
            xvf_add_gui_callback(kformstruct, 
			         "xvf_all_attributes",
                                 set_save_needed,
                                 (kaddr) create_info->save_struct);
}


/*-----------------------------------------------------------
|
|  Routine Name: set_selection_save_needed
|
|       Purpose: installs the "set_save_needed" callback
|                (which simply updates the "Save" button to "Save (Needed)")
|                on each and every selection of a control, so that changes to 
|                a pre-created selections are registered on the label 
|                of the Save button.  Note that if the objects for the
|                selection are not created yet, we cannot install the
|                handler since the handler takes the back_kformstruct
|                which will still be NULL.
|
|         Input: subform_list - list of subform pointers
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Sept 22, 1993
| Modifications: 
|
--------------------------------------------------------------*/

static void set_selection_save_needed(
    kselection     *sel_list)
{
        kselection     *selection;

	selection = sel_list;

	while (selection != NULL)
	{
	    add_save_needed_callback(selection->back_kformstruct);
	    if (selection->toggle_next != NULL)
	        set_selection_save_needed(selection->toggle_next);
	    if (selection->group_next != NULL)
	        set_selection_save_needed(selection->group_next);
	    selection = selection->next;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: add_uis_misc_file
|
|       Purpose: If the user uses the menuform of the subform or pane 
|                to write out a new *.subform or *.pane file to the 
|                {object}/uis directory, this gui attribute callback
|                makes sure that the new file is added to the cms database.
|
|         Input: None
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: June 10, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void add_uis_misc_file(
    kform_struct *kformstruct,
    char         *attribute,
    kaddr        attr_value,
    kaddr        client_data)
{
	char    *filepath;
	char    *uis_misc_filename;
	klist   *miscfile_list;
	kobject file_object;
	char    temp[KLENGTH];

	if (kstrcmp(attribute, XVF_PRINT_SUBFORM) == 0)
	    xvf_get_attribute(kformstruct, XVF_SUBFORM_FILE, &filepath);

	else if (kstrcmp(attribute, XVF_PRINT_PANE) == 0)
	    xvf_get_attribute(kformstruct, XVF_PANE_FILE, &filepath);

	if (filepath == NULL) return;

	/*
	 * the UIS file in question does not exist in a program object
	 */
	if (!(kcms_valid_object_path(filepath)))
	    return;

	/*
	 *  no program object associated w/ UIS file being displayed
	 */
	if (display_program == NULL) return;
	  
	/*
         * get list of file objects representing miscellanous UIS files
         */
        if (!(kcms_get_attribute(display_program, KCMS_CMOBJ_UIS_MISC,
                                  &miscfile_list)))
            return;

        /*
         * 1st time any *.subform file has been created - 
         * create file object, so that kcms registers UIS_MISC in the kcms 
         * database
         */
        if (miscfile_list == NULL)
        {
            ksprintf(temp, "%s", filepath);
            file_object = kcms_create_fileobj(display_program, temp, NULL,
                                              KCMS_FOBJ_TYPE_UIS,
                                              KCMS_FOBJ_SUBTYPE_MISC,
                                              KCMS_FOBJ_GEN_NONE,
                                              KCMS_FOBJ_ACCESS_RDWR);
	    kcms_sync(display_program);
	    return;
        }

	/*
	 * there is an existing list of UIS_MISC files;  search through
         * them looking for the name of the *.subform file in question
	 */
	while (miscfile_list != NULL)
	{
	    file_object = klist_clientdata(miscfile_list);
	    kcms_get_attribute(file_object, KCMS_PATH, &uis_misc_filename);
	    if (kstrcmp(uis_misc_filename, filepath) == 0)
		return;
	    miscfile_list = klist_next(miscfile_list);
	}

	/*
	 *  if the current *.subform file had been in the list of 
	 *  UIS_MISC files, it would have been found above, and we would
         *  already returned;  the fact that we got here means it wasn't there.
	 *  add it to the list of UIS_MISC files, then.
	 */
	ksprintf(temp, "%s", filepath);
        file_object = kcms_create_fileobj(display_program, temp, NULL,
                                          KCMS_FOBJ_TYPE_UIS,
                                          KCMS_FOBJ_SUBTYPE_MISC,
                                          KCMS_FOBJ_GEN_NONE,
                                          KCMS_FOBJ_ACCESS_RDWR);
	kcms_sync(display_program);
}

