 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>            Query xvobject Utilitie		      <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>             xvu_create_query()			      <<<<              
   >>>>   Static:                                             <<<<
   >>>>             quit_query()			      <<<<              
   >>>>             cancel_query()			      <<<<              
   >>>>   Public:                                             <<<<
   >>>>             xvu_query_wait()			      <<<<              
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	

static int  xvu_query_count = 0;
static int  xvu_query_done  = FALSE;
static int  xvu_query_sig;

static void quit_query   PROTO((xvobject, kaddr));
static void cancel_query PROTO((xvobject, kaddr));
static void kill_cr      PROTO((xvobject, kaddr));

typedef struct _Query_Info {
        xvobject toplevel;
	xvobject ok_obj;
	xvobject cancel_obj;
	xvobject *text;
	char     **answers;
	int      num;
	} Query_Info;

/************************************************************
*
*  Routine Name: xvu_query_wait - pop up prompt widget; wait for response
*
*       Purpose: Creates a pop-up query object in order to prompt
*                more responses from the user; xvu_query_wait may
*                be used to obtain a set of strings, floats, integers,
*                or responses of mixed types from the user.
*
*		 Will not return control to application program until
*		 the user selects "Ok" or "Cancel".
*
*                The prompts[] and answers[] arrays must be of the same
*                size, where that size is specified as 'num_prompts'.
*
*                The prompts[] array must be completely filled out 
*                with 'num_prompts' strings giving appropriate prompt 
*                for each response desired.
*		
*		 The answers[] array must be dynamically allocated, and
*                each element of the string array should be initialized
*                to contain the string representation of the default (if any).
*                Strings should be passed in as NULL if no default is 
*                appropriate).  The defaults will appear in the text objects 
*                when the query object is popped up; when the user clicks on 
*                "Ok", the contents of the answers[] array will be freed if 
*                non-NULL, and the responses of the user will be substituted 
*                in their place.  
*
*                When the caller is prompting for float, int, or other 
*                non-string responses, it is expected that the caller will
*                convert the responses returned in the answers[] array to
*                their appropriately typed counterparts before use.
*
*		 The 'size' argument is provided to ensure that the text
*                objects are physically large enough to accomodate the
*                user's response.
*
*                
*         Input: top_label    - label to appear at top of query object
*		 prompts[]    - array of prompts
*                button       - label for acknowledgement button;
*                               NULL will give default of "Ok"
*		 answers[]    - array of strings in which to return the
*				user's responses, one for each of the prompts.
*                               Default values may be provided here if desired.
*		 num_prompts  - size of prompts[] and answers[] arrays
*		 size	      - size (in characters) to make each text 
*                               object for the users' responses
*
*        Output: answers[] - set of strings provided by the caller; 
*                            defaults will be returned when the user does not
*                            change a string;  NULL will be returned if the
*                            user did not provide a string and no default was
*                            provided.
*
*	Returns:       1 if user selected "Ok"
*                      0 if user selected "Cancel"
*
*
*  Restrictions: 
*
*    Written By: Danielle Argiro
*          Date: May 19, 1994
*      Verified:
*  Side Effects: The strings in the answers[] array will be freed if non-NULL,
*                and the responses of the user substituted.  If the user
*                leaves the default in place, that will be returned;  if the
*                user blanks out a string, NULL is returned. 
*
* Modifications:
*
*************************************************************/

int  xvu_query_wait(
   char *top_label,
   char *prompts[],
   char *button,
   char *answers[],
   int  num_prompts,
   int  size)
{
	xvobject toplevel;

	if (answers == NULL)
	{
	    kerror("xvutils", "xvu_query_wait",
		   "Invalid call to xvu_query_wait with answers array = NULL; you must provide an allocated array of strings of size num_prompts.  For no defaults, strings should be initialized to \"\".");
	    return(FALSE);
	           
	}
	xvu_query_done = FALSE;

	xvw_busy(NULL, TRUE);

	toplevel  = xvu_create_query(top_label, prompts, button, answers,
		                     num_prompts, size);
	if (toplevel == NULL)
	    return(FALSE);

	while(!xvu_query_done)
	    xvw_process_event();  

	xvw_unmap(toplevel);
	xvw_destroy(toplevel);
	xvw_busy(NULL, FALSE);
	return(xvu_query_sig);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvu_create_query()
|
|       Purpose: Creates a query object. 
|         Input: top_label    - label to appear at top of query object
|                prompts[]    - array of prompts
|                button       - label for acknowledgement button;
|                               NULL will give default of "Ok"
|                answers[]    - array of strings in which to return the
|                               user's responses, one for each of the prompts.
|                               Default values may be provided here if desired.
|                num_prompts  - size of prompts[] and answers[] arrays
|                size         - size (in characters) to make each text
|                               object for the users' responses
|
|        Output: Returns the query object's toplevel 
|    Written By: Danielle Argiro
|          Date: Jul 9, 1992
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/

#define MaxNumQuerysInViewport 10 

xvobject xvu_create_query(
   char *top_label,
   char **prompts,
   char *button,
   char **answers,
   int  num_prompts,
   int  size)
{
	xvobject back;		/* backplane for query form */
	xvobject label_wid;     /* initial label */
	xvobject label;		/* label for each prompt */
	xvobject offset;     	/* object to offset from */
	float  width, prompt_width = 0.0, label_width, ok_width, cancel_width;
	int    i; 
	char name[KLENGTH];
	Query_Info *query_info;
	
	if (num_prompts < 1)
	{
	    errno = KINVALID_PARAMETER;
	    kerror("xvutils", "xvu_create_query", "'num_prompts' must be > 0");
	    return(FALSE);
	}

	query_info = (Query_Info *) (kcalloc (1, sizeof(Query_Info)));

	/*
	 *  allocate space for answers array, text & label arrays
	 */
	if (answers == NULL)
	    answers = (char **) kcalloc(1, num_prompts * sizeof(char *));
	for (i = 0; i < num_prompts; i++)
	    if (answers[i] == NULL)
		answers[i] = (char *) kmalloc(125*sizeof(char));

	query_info->text  = (xvobject *) kmalloc(num_prompts*sizeof(xvobject)); 

	/*
	 * create identifying object name
	 */
	ksprintf(name, "%d_query", xvu_query_count++);

	if (top_label == NULL)
	{
	    top_label = kstring_copy("Please Respond", NULL);
	    label_width = 15.0;
	}
	else label_width = (float) xvu_find_string_width(top_label);

	/*
	 * determine size of longest prompt (in pixels);
	 */
	if (prompts == NULL)
	    prompt_width = 0.0;
	else
	{
	    for (i = 0; i < num_prompts; i++)
	    {
	        if (prompts[i] != NULL)
		    width = (float) kstrlen(prompts[i]);
		else width = 0.0;
                if (width > prompt_width) prompt_width = width;
	    }
	    prompt_width += 1.0;
	}

	/*
	 * determine size of ok object (in pixels);
	 */
	if (button == NULL)
	{
	    ok_width = 3.0;
	    button = kstring_copy("Ok", NULL);
	}
	else ok_width = (float) (kstrlen(button)+1);

	cancel_width = 7.0;

	/*
	 * create the query box's toplevel object, and add it
	 * to the list of toplevels that is used with journal playback.
	 */
	query_info->toplevel = xvw_create_transient_shell(name, NULL, NULL);

	/*
         * create the backplane object
         */
        back = xvw_create_manager(query_info->toplevel, "back");


	/* 
	 * create the label object 
	 */ 
	width = prompt_width + size - ok_width - cancel_width;
	if (width > label_width) label_width = width;

	label_wid = xvw_create_label(back, "label");

        xvw_set_attributes(label_wid,
		XVW_LABEL,            top_label,           /* label         */
                XVW_BORDER_WIDTH,     0,                   /* no border     */
                XVW_RIGHT_OF,         NULL,                /* centered      */
                XVW_BELOW,            NULL,                /* at top        */
                NULL);


	/* 
	 * create cancel button 
	 */
	query_info->cancel_obj = xvw_create_button(back, "cancel");
        xvw_set_attributes(query_info->cancel_obj,
		XVW_LABEL,            "Cancel",        /* label           */
		XVW_CHAR_WIDTH,       cancel_width,    /* set width       */
                XVW_CHAR_HEIGHT,      1.0,	       /* set height      */
                XVW_LEFT_OF,          NULL,            /* upper R corner  */
                XVW_BELOW,            label_wid,       /* at top          */
                XVW_MAP_WHEN_MANAGED, TRUE,            /* mapped          */
                NULL);

	/* 
	 * create ok button 
	 */
	query_info->ok_obj = xvw_create_button(back, "ok");
        xvw_set_attributes(query_info->ok_obj,
		XVW_LABEL,            button,        /* button label    */
		XVW_CHAR_WIDTH,       ok_width,      /* set width       */
                XVW_CHAR_HEIGHT,      1.0,	       /* set height     */
                XVW_LEFT_OF,   query_info->cancel_obj, /* L of "cancel"  */
                XVW_BELOW,            label_wid,     /* at top          */
                XVW_MAP_WHEN_MANAGED, TRUE,          /* mapped          */
                NULL);
	/*
	 *  create the set of prompt & answer object pairs
	 */
	offset = query_info->ok_obj;
	for (i = 0; i < num_prompts; i++)
	{
	    /* 
	     * create label object 
	     */
	    if ((prompts != NULL) && (prompts[i] != NULL))
	    {

	        ksprintf(name,"%d_label",i);
	        label = xvw_create_label(back, name);
	        xvw_set_attributes(label,
		    XVW_LABEL,            prompts[i],    /* label            */
		    XVW_CHAR_WIDTH,       prompt_width,  /* set width        */
                    XVW_CHAR_HEIGHT,      1.0,           /* set width        */
                    XVW_BELOW,            offset,        /* under last pair  */
                    XVW_RESIZABLE,        TRUE,          /* resizable        */
                    XVW_MAP_WHEN_MANAGED, TRUE,          /* mapped           */
		    XVW_LABEL_JUSTIFY,    KLABEL_JUSTIFY_CENTER, /* centered */
                    NULL);
	    }
	    else label = NULL;

	    /* 
	     * create text object 
	     */
	    ksprintf(name, "%d_text", i);
	    query_info->text[i] = xvw_create_text(back, name);
	
	    width = size;
	    xvw_set_attributes(query_info->text[i],
                XVW_TEXT_STRING,      answers[i],       /* initial default */
		XVW_CHAR_WIDTH,       width,            /* set width       */
                XVW_CHAR_HEIGHT,      1.1,              /* set height      */
                XVW_BELOW,            offset,           /* under last pair */
                XVW_RIGHT_OF,         label,            /* R of label      */
                XVW_RESIZABLE,        TRUE,             /* resizable       */
                XVW_BORDER_WIDTH,     2,                /* border          */
                XVW_TEXT_LENGTH,      KLENGTH,          /* resizable       */
                XVW_TEXT_LEFT_MARGIN, XVU_LEFT_MARGIN,  /* L margin        */
                NULL);

	   if (answers[i] != NULL)
	       xvw_set_attribute(query_info->text[i], 
			         XVW_TEXT_STRING,  answers[i]);

	    xvw_add_action(query_info->text[i], "<Key>Return",
                          kill_cr, NULL, TRUE);


	    offset = query_info->text[i];
	}

	query_info->answers = answers;
	query_info->num     = num_prompts;
	xvw_add_callback(query_info->ok_obj, XVW_BUTTON_SELECT,
			quit_query, query_info);
	xvw_add_callback(query_info->cancel_obj, XVW_BUTTON_SELECT,
			cancel_query, query_info);

	xvw_add_protocol(query_info->toplevel, "WM_DELETE_WINDOW", 
			 cancel_query, query_info->toplevel);

	xvw_place(query_info->toplevel, NULL);
	return(query_info->toplevel);
}




/*-----------------------------------------------------------
|
|  Routine Name: quit_query()
|
|       Purpose: Quits the query object
|         Input: object      - the "ok" object
|                client_data - not used
|        Output: none
|          Date: Jul 9, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/
/* ARGSUSED */
static void quit_query(
   xvobject object,
   kaddr    client_data)
{
	int  i;
	char *response;

	Query_Info *query_info = (Query_Info *) client_data;

	for (i = 0; i < query_info->num; i++)
	{
	    xvw_get_attribute(query_info->text[i], XVW_TEXT_STRING, 
			      &response);
	    if (kstrlen(response) == 0)
	       query_info->answers[i] = NULL;
	    else ksprintf(query_info->answers[i], "%s", response);
	}
	xvw_remove_callback(query_info->ok_obj, XVW_BUTTON_SELECT,
			    quit_query, query_info);
	xvw_remove_callback(query_info->cancel_obj, XVW_BUTTON_SELECT,
			    cancel_query, query_info->toplevel);

	kfree(query_info);
	xvu_query_done = TRUE;
	xvu_query_sig = 1;

}

/*-----------------------------------------------------------
|
|  Routine Name: cancel_query()
|
|       Purpose: Cancels the query object
|         Input: object      - the "cancel" object
|                client_data - not used
|                call_data   - not used
|        Output: none
|          Date: July 9, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/
/* ARGSUSED */
static void cancel_query(
   xvobject object,
   kaddr    client_data)
{
	Query_Info *query_info = (Query_Info *) client_data;

	xvw_remove_protocol(query_info->toplevel, "WM_DELETE_WINDOW", 
			    cancel_query, query_info->toplevel);
	xvw_remove_callback(query_info->ok_obj, XVW_BUTTON_SELECT,
			    quit_query, query_info);
	xvw_remove_callback(query_info->cancel_obj, XVW_BUTTON_SELECT,
			    cancel_query, query_info->toplevel);
	xvu_query_done = TRUE;

	kfree(query_info);
	xvu_query_sig = 0;

}


/*-----------------------------------------------------------
|
|  Routine Name: kill_cr()
|
|       Purpose: This bogus little routine prevents the
|                user from hitting <cr> in the string parameter box
|                and loosing track of their input
|         Input: object      - the "cancel" object
|                client_data - not used
|        Output: none
|          Date: Dec 19, 1993
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
static void kill_cr(
   xvobject object,
   kaddr    client_data)
{
        return;
}
