 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                Layout Widget Routines
   >>>>
   >>>>	Private:
   >>>>			ClassInitialize()
   >>>>			ConstraintInitialize()
   >>>>			ChangeManaged();
   >>>>			Resize()
   >>>>			SetValues()
   >>>>			ConstraintDestroy()
   >>>>
   >>>>			ChangeHandler()
   >>>>			ChangeSelected()
   >>>>			Layout()
   >>>>	Public:
   >>>>			xvw_create_layout()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvobjects/LayoutP.h>

static void ClassInitialize   PROTO((void));
static void ConstraintInitialize PROTO((Widget, Widget, ArgList, Cardinal *));
static void ChangeManaged     PROTO((Widget));
static void Resize	      PROTO((Widget));
static void ConstraintDestroy PROTO((Widget));
static Boolean SetValues      PROTO((Widget, Widget, Widget, ArgList,
				   Cardinal *));
static void ChangeHandler     PROTO((xvobject, XEvent *));
static void ChangeSelected    PROTO((xvobject));
static void Layout	      PROTO((Widget));
static XtGeometryResult GeometryManager   PROTO((Widget, XtWidgetGeometry *,
                                                 XtWidgetGeometry *));




/*-------------------------------------------------------
|
|   Full class attributes
|
--------------------------------------------------------*/

static xvattribute attributes[] = {
{XVW_LAYOUT_AREA_JUSTIFICATION,     NULL,    XtRInt,	   NULL},
{XVW_LAYOUT_BUFFER_SIZE,	    NULL,    XtRInt,	   NULL},
{XVW_LAYOUT_BORDER_SIZE,	    NULL,    XtRInt,	   NULL},
{XVW_LAYOUT_CALLBACK,	            NULL,    XtRCallback,  NULL},
{XVW_LAYOUT_NUMBER_ACROSS,	    NULL,    XtRInt,	   NULL},
{XVW_LAYOUT_SELECTED_CHILD,	    NULL,    XtRObject,    XtRWidget},
};

/*-------------------------------------------------------
|
|   Full class record constant
|
--------------------------------------------------------*/

#define offset(field) XtOffsetOf(XvwLayoutWidgetRec, layout.field)

static XtResource resources[] = { 
{XVW_LAYOUT_AREA_JUSTIFICATION, NULL, XtRInt, sizeof(int),
      offset(area_justification), XtRImmediate,(XtPointer) KLAYOUT_AREA_CENTER},
{XVW_LAYOUT_BUFFER_SIZE, NULL, XtRInt, sizeof(int),
      offset(buffer_size), XtRImmediate, (XtPointer) 5},
{XVW_LAYOUT_BORDER_SIZE, NULL, XtRInt, sizeof(int),
      offset(border_size), XtRImmediate, (XtPointer) 2},
{XVW_LAYOUT_CALLBACK, NULL, XtRCallback, sizeof(XtPointer),
      offset(layout_callback), XtRCallback, (XtPointer) NULL},
{XVW_LAYOUT_NUMBER_ACROSS, NULL, XtRInt, sizeof(int),
      offset(number_across), XtRImmediate, (XtPointer) 2},
{XVW_LAYOUT_SELECTED_CHILD, NULL, XtRWidget, sizeof(Widget),
      offset(selected_child), XtRImmediate, (XtPointer) NULL},
};
#undef offset


#define SUPERCLASS (&xvwManagerWidgetClassRec)

XvwLayoutWidgetClassRec xvwLayoutWidgetClassRec =
{
  {
    (WidgetClass) SUPERCLASS,		/* superclass		  */	
    "Layout",				/* class_name		  */
    sizeof(XvwLayoutWidgetRec),			/* size			  */
    ClassInitialize,			/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    NULL,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    XtInheritRealize,			/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    XtNumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    TRUE,				/* compress_motion	  */
    XtExposeCompressMaximal,		/* compress_exposure	  */
    TRUE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    NULL,				/* destroy		  */
    NULL,				/* resize		  */
    XtInheritExpose,			/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    XtInheritTranslations,		/* tm_table		  */
    NULL,				/* query_geometry	  */
    XtInheritDisplayAccelerator,	/* display_accelerator	  */
    NULL				/* extension		  */
  },  /* CoreClass fields initialization */
  {
    GeometryManager,			/* geometry_manager	  */
    ChangeManaged,			/* change_managed	  */
    XtInheritInsertChild,		/* insert_child	  	  */
    XtInheritDeleteChild,		/* delete_child	  	  */
    NULL,				/* extension	 	  */
  },  /* CompositeClass fields initialization */
  {
    NULL,			            /* subresources       */
    0,					    /* subresources_count */
    sizeof(XvwManagerWidgetConstraintsRec), /* constraint_size    */
    ConstraintInitialize,		    /* initialize         */
    ConstraintDestroy,                      /* destroy            */
    NULL,				    /* set_values         */
    NULL,                                   /* extension          */
  },  /* ConstraintClass fields initialization */
  {
    XtInheritLayout,			    /* child layout routine  */
    XtInheritChangeSel,			    /* change selected proc   */
    XtInheritEraseSel,			    /* erase selected proc    */
    XtInheritRefreshSel,		    /* refresh selection proc */
    Resize,				    /* resize		      */
    XtInheritGeometryManager,		    /* geometry_manager	      */
  }, /* XvwManagerWidgetClass fields initialization */
  {
    NULL,                                     /* field not used    */
  },  /* XvwLayoutWidgetClass fields initialization */
};
#undef SUPERCLASS

/*
 * xvwLayoutWidgetClass for public consumption
 */
WidgetClass xvwLayoutWidgetClass = (WidgetClass) &xvwLayoutWidgetClassRec;


/*-----------------------------------------------------------
|
|  Routine Name: ClassInitialize
|
|       Purpose: This method is called the first time an  
|                instance of a xvwLayoutWidgetClass has been created. 
|                It will initialize all the class attributes. 
|
|         Input: None
|
|        Output: None
|
|    Written By: Mark Young and John M. Salas
|          Date: Aug 15, 1992 12:31
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ClassInitialize(void)
{
        xvw_init_attributes(xvwLayoutWidgetClass, attributes,
		XtNumber(attributes), NULL, 0, 
		"$DESIGN/objects/library/xvobjects/uis/Layout.pane");
	xvw_load_resources("$DESIGN/objects/library/xvobjects/app-defaults/Layout");
}

/*-----------------------------------------------------------
|
|  Routine Name: ConstraintInitialize
|
|       Purpose: This method is used to overide the translations
|                of a child of the XvwManagerWidget and to
|                install the accelerators from the XvwManagerWidget
|                into the child.
|
|         Input: request - widget child with requested constraints
|                new     - widget child with overidden translations and
|                          installed accelerators
|    Written By: Mark Young
|          Date: Nov 01, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ConstraintInitialize(
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	xvobject object = xvw_object(new);

	if (XtIsSubclass(new, xvwManagerWidgetClass) == TRUE)
	{
	   xvw_add_event(object, ButtonPressMask, ChangeHandler, NULL);
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: ChangeManaged
|
|       Purpose: This method is called once while the 
|                instance of layoutWidget is being realized.
|                It handles the initial layout of the 
|                widget.  It will also be called to unmanage a 
|                managed widget, or to manage an unmanaged one.
|
|         Input: widget - the layoutWidget
|    Written By: Mark Young
|          Date: Apr 10, 1995
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ChangeManaged(
   Widget widget)
{
	Layout(widget);
}


/*-----------------------------------------------------------
|
|  Routine Name: GeometryManager
|
|       Purpose: This method is called once while the 
|                instance of layoutWidget is being realized.
|                It handles the initial layout of the 
|                widget.  It will also be called to unmanage a 
|                managed widget, or to manage an unmanaged one.
|         Input: widget - the layoutWidget
|    Written By: Steven Kubica and John M. Salas
|          Date: Oct 23, 1992 15:49
|
------------------------------------------------------------*/
/* ARGSUSED */
static XtGeometryResult GeometryManager(
   Widget           widget,
   XtWidgetGeometry *request,
   XtWidgetGeometry *reply)
{
	return(XtGeometryDone);
}

/*-----------------------------------------------------------
|
|  Routine Name: Resize
|
|       Purpose: This method is used to recalculate the layout
|		 of the widget in the event of a resize. 
|
|         Input: widget - the widget containing the new position
|			  and size information 
|    Written By: Mark Young
|          Date: Apr 11, 1995
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Resize(
   Widget widget)
{
	Layout(widget);
	ManagerRequestLayout(widget, FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: SetValues
|
|       Purpose: Determine what the widget should do by
|		 comparing the current public settable values 
|		 versus the new settings.
|
|         Input: current - the widget containing current settings
|                request - the widget containing requested settings
|                new     - the widget processed through all set values methods
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|    Written By: John M. Salas
|          Date: Oct 23, 1992 16:38
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean SetValues(
   Widget   current,
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
       XvwLayoutWidget cxwid = (XvwLayoutWidget) current;
       XvwLayoutWidget nxwid = (XvwLayoutWidget) new;


       if (cxwid->layout.selected_child != nxwid->layout.selected_child ||
           cxwid->layout.border_size    != nxwid->layout.border_size)
       {
	  ChangeSelected(xvw_object(nxwid->layout.selected_child));	  
       }
     
       if ( (cxwid->layout.number_across   != nxwid->layout.number_across) || 
         (cxwid->layout.buffer_size        != nxwid->layout.buffer_size)   ||
         (cxwid->layout.area_justification != nxwid->layout.area_justification))
       {
	  Layout(new);
       }
       return(FALSE);
}


/*-----------------------------------------------------------
|
|  Routine Name: ConstraintDestroy
|
|       Purpose: This method is used to change the selected area to the
|		 first area in the list, if the child being destroyed is
|		 the currently selected area.
|
|         Input: child   - child widget to be deleted
|    Written By: Mark Young
|          Date: Apr 10, 1995
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ConstraintDestroy(
   Widget child)
{
	XvwLayoutWidget xwid = (XvwLayoutWidget) XtParent(child);

	int num;
	xvobject object;

	if (!xwid->core.being_destroyed && xwid->layout.selected_child == child)
	{
	   num = xwid->composite.num_children;
	   if (num > 0)
	   {
	      object = xvw_object(xwid->composite.children[num-1]);
	      ChangeSelected(object);
	   }
	   else
	   {
	      object = NULL;
	      xwid->layout.selected_child = NULL;
	   }
	   XtCallCallbacks((Widget) xwid, XVW_LAYOUT_CALLBACK, &object);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ChangeHandler 
|
|       Purpose: Handler to call ChangeSelected and also invoke
|		 the inform callback.
|
|         Input: object - the newly selected object
|		 event  - the event
|    Written By: Mark Young
|          Date: Dec 6, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ChangeHandler(
   xvobject object,
   XEvent   *event)
{
	XvwLayoutWidget xwid = (XvwLayoutWidget) xvw_widget(xvw_parent(object));

	object = xvw_object(xvw_widget(object));
	ChangeSelected(object);
	XtCallCallbacks((Widget) xwid, XVW_LAYOUT_CALLBACK, &object);
}

/*-----------------------------------------------------------
|
|  Routine Name: ChangeSelected 
|
|       Purpose: When the selected child attribute
|		 has been changed, this routine will update
|		 the child so it "looks" selected by puttin
|		 a border around it.
|
|         Input: object - the newly selected object
|    Written By: Mark Young
|          Date: Apr 10, 1995
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ChangeSelected(
   xvobject object)
{
	XvwLayoutWidget xwid = (XvwLayoutWidget) xvw_widget(xvw_parent(object));

	int      i;
	Pixel    pixel;
	xvobject cobject;
	Widget   child;

	xwid->layout.selected_child = xvw_widget(object);
	for (i = 0; i < xwid->composite.num_children; i++)
	{
	   child = xwid->composite.children[i];
	   if (XtIsSubclass(child, ManagerWidgetClass) == TRUE)
	   {
	      cobject = xvw_object(child);

	      if (child == xwid->layout.selected_child)
                 xvw_get_attribute(cobject, XVW_FOREGROUND_PIXEL, &pixel);
              else
	         xvw_get_attribute(cobject, XVW_BACKGROUND_PIXEL, &pixel);

              xvw_set_attributes(cobject,
			XVW_BORDER_PIXEL, pixel,
			XVW_BORDER_WIDTH, xwid->layout.border_size,
			NULL);
 	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: Layout
|
|       Purpose: This routine will layout the children of the
|		 layoutWidget.  
|
|         Input: widget - the layout widget 
|    Written By: Steven Kubica and John M. Salas and Mark Young
|          Date: Dec 7, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Layout(
   Widget widget)
{
	XvwLayoutWidget xwid = (XvwLayoutWidget) widget;

	Widget   *children;
	xvobject object, right_of, below;

	int i, j, area_list[KLAYOUT_MAX_AREAS], just;
	int num_children, woff, hoff;
	int width, height, area_width, area_height;
	int border, buffer, num_areas, num_across, num_down, num_across_bottom;


	/* Don't Layout if there are no children */
	if (xwid->composite.num_children == 0)
	   return;

	/* Get the relevant and useful information from the widget */ 
	border = xwid->layout.border_size;
	buffer = xwid->layout.buffer_size;
	just   = xwid->layout.area_justification;
	children     = xwid->composite.children;
	num_across   = xwid->layout.number_across;
	num_children = xwid->composite.num_children;

	/*
	 *  Find the number of areas, also make array of "pointers"
	 *  into the composite list of children, so I can easily access 
	 *  the areas in order later on.
	 */
	for (i = 0, num_areas = 0; i < num_children; i++)
	{
	   object = xvw_object(children[i]);
	   if (xvw_check_subclass(object, ManagerWidgetClass) &&
	       xvw_check_managed(object))
	   {
	      area_list[num_areas] = i; 
	      num_areas++; 
	   }
	}

	if (num_areas == 0)
	   return;

	if (num_areas < num_across) num_across = num_areas;

	/*
	 *  Find all of the appropriate widths and heights needed
	 *  for all the areas.  Note that all areas will
	 *  have the same width and height, with the exception of the 
	 *  on the bottom row which will be different depending 
	 *  on the justification.
	 */

	/* Find overall height and width we have to work with */ 
	height = (int) xwid->core.height;
	width  = (int) xwid->core.width; 

	/* Based on the number of areas, and the number across,
	 * find the number down and the number across the bottom 
	 */ 
	num_down = (int) kceil((double)num_areas/(double)num_across);
	num_across_bottom = num_areas % num_across;

	if ( num_areas < num_across ) num_across = num_areas;

	/* Find the height and the width */
/*
	area_height = (height - 2*num_down - (num_down+1)*border)/num_down;
	area_width  = (width - 2*num_across - (num_across+1)*border)/num_across;
 */
	area_width  = (width  - num_across*(2*border+buffer))/num_across;
	area_height = (height - num_down*(2*border+buffer))/num_down;

	/* Find left and bottom kludge values */
	woff = (int) xwid->core.width -
		((int) xwid->core.width/num_across) * num_across;
	hoff = (int) xwid->core.height -
		((int) xwid->core.height/num_down) * num_down;

	/* Place all the areas, except for those in the last row */
	for (i = 0, below = NULL; i < num_down-1; i++)
	{
	   for (j = 0, right_of = NULL; j < num_across; j++)
	   {
	      /* Want to place the current guy */
	      object = xvw_object(children[ area_list[j+i*num_across]]);

	      if (j == num_across-1 ) area_width += (int) woff;

	      /* Set the appropriate attributes to place the current guy */
	      xvw_set_attributes(object,
			     XVW_BELOW,	         below,
			     XVW_RIGHT_OF,	 right_of,
			     XVW_BORDER_WIDTH,	 xwid->layout.border_size,
			     XVW_MINIMUM_WIDTH,  area_width,
			     XVW_MINIMUM_HEIGHT, area_height,
			     XVW_MAXIMUM_WIDTH,  area_width,
			     XVW_MAXIMUM_HEIGHT, area_height, NULL );

	      if (j == num_across-1)
		 area_width -= (int) woff;

	      right_of = object;
	   }
	   below = right_of;
	}

	if (num_across_bottom == 0)
	   num_across_bottom = num_across;

	/* width will change for full justification */
	if (just == KLAYOUT_AREA_FULL)
	{
	   area_width  = (width - 2*num_across_bottom - 
		    (num_across_bottom+1)*border)/num_across_bottom;
	   woff = (int) xwid->core.width -
		 ((int) xwid->core.width/num_across_bottom) * num_across_bottom;
	}

	/* Place last row specially */
	area_height += hoff;
	for (j = 0, right_of = NULL; j < num_across_bottom; j ++)
	{
	    object = xvw_object(children[area_list[j+(num_down-1)*num_across]]);

	    if (j == (num_across_bottom - 1) && num_across_bottom > 1) 
	       area_width += woff;

	    /* Set the appropriate attributes to place the current */
	    xvw_set_attributes(object,
			  XVW_BELOW,	      below,
			  XVW_BORDER_WIDTH,   xwid->layout.border_size,
			  XVW_RIGHT_OF,	      right_of,
			  XVW_MINIMUM_WIDTH,  area_width,
			  XVW_MINIMUM_HEIGHT, area_height,
			  XVW_MAXIMUM_WIDTH,  area_width,
			  XVW_MAXIMUM_HEIGHT, area_height, NULL );

	    right_of = object;
	}
}


/************************************************************
*
*  Routine Name: xvw_create_layout - create a layout object
*
*       Purpose: The layout object is designed for doing layout of 
*		 objects that are subclassed from the Manager widget.  
*		 Only objects subclassed from the manager widget (\fInot\fP
*		 the manager object) may be created as children of the
*		 layout object.  Such objects include area objects, 
*		 image objects, zoom objects, viewport objects, and so on.  
*		 See Chapter 1, "Introduction", of the Programming Services 
*		 Manual, Volume 3, for diagrams depicting the objects that 
*		 are subclassed from the Manager widget.  
*
*		 The layout object allows you to do quick and easy
*		 layout when a variety of such objects share a common
*		 backplane.  You may specify the number of objects
*		 that should appear in a single row;  relative layout
*		 specifications are not needed.  
*
*		 The layout object is especially effective when laying
*		 out objects \fIof the same type\fP, as it will preserve
*		 proportionality between the objects.  It is often used
*		 in applications such as \fHxprism\fP, where the user
*		 will be interactively creating new objects to display
*		 data.  The application need not implement special
*		 code to do appropriate layout of new objects as they
*		 are created, since the layout object does it 
*		 automatically according to initial specifications.   
*
*         Input: parent - the parent of the layout object; NULL will cause a
*                         default toplevel to be created automatically
*                name   - the name with which to reference layout object 
*
*        Output:
*	Returns: The layout object on success, NULL on failure
*  Restrictions:
*    Written By: Mark Young and John M. Salas
*          Date: Jul 31, 1992 10:41
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xvobject xvw_create_layout(
   xvobject parent,
   char     *name)
{
	xvobject object;

	/*
	 *  Call the xvw_create() routine to do the actual creation.
	 */
	object = xvw_create(parent, FALSE, TRUE, name, LayoutWidgetClass);
	return(object);
}
