/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                Image Widget Routines
   >>>>
   >>>>	Private:
   >>>>	 Static:
   >>>>			ImageInit()
   >>>>			ImageReloadXData()
   >>>>			ApplyRGBAlphaData()
   >>>>			ImageReload()
   >>>>			ImageUpdatePosition()
   >>>>			ImageSetPosition()
   >>>>			ImageGetName()
   >>>>			ImageSetName()
   >>>>  Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvisual/ImageP.h>


/*-----------------------------------------------------------
|
|  Routine Name: ImageInit - initialize the image
|
|       Purpose: Initializes the ImageClass for a given object.
|		 Given an object to be displayed we initialize
|		 the object within the image so that it can be
|		 properly displayed.
|         Input: 
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Jun 24, 1994
| Modifications:
|
------------------------------------------------------------*/

void ImageInit(
  Widget widget)
{
	XvwImageWidget xwid = (XvwImageWidget) widget;

	kobject  object;
	char     *filename;
	unsigned long shiftval;
	double   padval, alpha_min = 0.0, alpha_max = 255.0;
	int      w, h, d, t, e, type, colorspace, has_alpha;

	/*
	 *  Sanity check to see if we have the necessary information in which
	 *  to initialize the ImageClass.
	 */
	if (xwid->image.iobject == NULL)
	   return;
	else
	   object = xwid->image.iobject;

	/*
	 *  Make sure we got value data, otherwise there ain't nothing we
	 *  can do...
	 */
	if (kpds_query_value(object) == FALSE)
	{
	   filename = "Unknown";
	   kpds_get_attribute(object, KPDS_NAME, &filename);
	   kerror(XVISUAL, "ImageInit", "No image data exists for the \
data object '%s'",  filename);
	   return;
	}

	/*
	 *  Get the value data dimensionality
	 */
	kpds_get_attributes(object,
		KPDS_VALUE_SIZE, &w, &h, &d, &t, &e,
		KPDS_VALUE_DATA_TYPE, &type,
		NULL);
	kcolor_get_attributes(object,
		KCOLOR_COLORSPACE, &colorspace,
		KCOLOR_HAS_ALPHA,  &has_alpha,
		NULL);
	if (kcolor_query_attribute(object,KCOLOR_ALPHA_MIN,NULL,NULL,NULL,NULL))
	   kcolor_get_attribute(object, KCOLOR_ALPHA_MIN, &alpha_min);
	if (kcolor_query_attribute(object,KCOLOR_ALPHA_MAX,NULL,NULL,NULL,NULL))
	   kcolor_get_attribute(object, KCOLOR_ALPHA_MAX, &alpha_max);

	if (type == KDOUBLE  || type == KFLOAT ||
	    type == KCOMPLEX || type == KDCOMPLEX)
	{
	   kpds_set_attributes(object,
		KPDS_VALUE_SCALING,   KNONE,
		KPDS_VALUE_NORM_MIN,  0.0,
		KPDS_VALUE_NORM_MAX,  255.0,
		KPDS_VALUE_DATA_TYPE, KDOUBLE,
		KPDS_VALUE_POSITION,  0, 0, 0, 0, 0,
		KPDS_VALUE_COMPLEX_CONVERT, xwid->image.complex_convert,
		NULL);

	   /*
	    *  It's to be normalized, but we need to set the pad value to
	    *  something better than 0.0
	    */ 
	   padval = 0.0;
	   (void) kpds_get_data(object, KPDS_VALUE_POINT, &padval);
	   kpds_set_attributes(object,
		KPDS_VALUE_PAD_VALUE, padval, 0.0,
		KPDS_VALUE_DATA_TYPE, type,
		KPDS_VALUE_SCALING,   KNORMALIZE,
		NULL);
	}
	else if (xwid->color.info != NULL)
	{
	   if (xwid->color.info->minval != 0 || xwid->color.info->scale != 1.0)
	   {
	      shiftval = - xwid->color.info->minval;
	      kpds_set_attributes(object,
		   KPDS_VALUE_SCALING, KSCALE,
		   KPDS_VALUE_SCALE_FACTOR, xwid->color.info->scale,
		   KPDS_VALUE_SCALE_OFFSET, (double) shiftval, (double) 0.0,
		   NULL);
	   }
	   else
	      kpds_set_attribute(object, KPDS_VALUE_SCALING, KNONE);
	}

	/*
	 *  Check to see if we got to deal with alpha
	 */
	xwid->image.alpha_offset = -alpha_min;
	xwid->image.alpha_factor = 1.0/(alpha_max - alpha_min);
	if (e == 4 && has_alpha)
	   xwid->image.has_alpha = TRUE;
	else
	   xwid->image.has_alpha = FALSE;

	/*
	 *  Set the maximum band number so that if the band number is
	 *  set we know the maximum band.
	 */
	if (colorspace == KRGB && (e == 3 || (e == 4 && has_alpha)))
	{
	   xwid->image.image_depth = 3, e = 1;
	}
	else
	   xwid->image.image_depth = 1;

	if ((xwid->image.band_dimensions & KIMAGE_ELEMENTS) == 0)
	   e = 1;
	if ((xwid->image.band_dimensions & KIMAGE_DEPTH) == 0)
	   d = 1;
	if ((xwid->image.band_dimensions & KIMAGE_TIME) == 0)
	   t = 1;

	xwid->image.wsize = w;
	xwid->image.hsize = h;
	xwid->image.esize = e;
	xwid->image.dsize = d;
	xwid->image.tsize = t;
	xwid->image.band_maxnum = e*d*t - 1;

	if (xwid->image.band_number > xwid->image.band_maxnum)
	   xwid->image.band_number = xwid->image.band_maxnum;
	else if (xwid->image.band_number < 0)
	   xwid->image.band_number = 0;

	if (xwid->color.info != NULL)
	   kfree(xwid->color.info->histogram);
}

/*-----------------------------------------------------------
|
|  Routine Name: FastConvertImageData - fast convert image data
|
|       Purpose: 
|         Input: 
|        Output: 
|       Returns:
|
|    Written By: Mark Young
|          Date: May 06, 1994
| Modifications:
|
------------------------------------------------------------*/
#define FastConvertImageData(xwid, idata, odata, mdata, ximage,		\
				pixels, w, size)			\
{									\
	if (mdata != NULL)						\
	{								\
	   for (i = 0; i < size; i++, idata++)				\
	   {								\
	      if ((i % w) == 0 && i != 0) odata += ximage->width - w;	\
	      *odata++ = !*mdata++ ? xwid->core.background_pixel :	\
			pixels[*idata];					\
	   }								\
	}								\
	else if (ximage->width != w)					\
	{								\
	   for (i = 0; i < size; i++)					\
	   {								\
	      if ((i % w) == 0 && i != 0) odata += ximage->width - w;	\
	      *odata++ = pixels[*idata++];				\
	   }								\
	}								\
	else /* default case */						\
	{								\
	   while (size-- > 0)						\
	      *odata++ = pixels[*idata++];				\
	}								\
}

/*-----------------------------------------------------------
|
|  Routine Name: SlowConvertImageData - slow convert image data
|
|       Purpose: 
|         Input: 
|        Output: 
|       Returns:
|
|    Written By: Mark Young
|          Date: May 06, 1994
| Modifications:
|
------------------------------------------------------------*/
#define SlowConvertImageData(xwid, idata, odata, mdata, ximage,		\
				xcolors, pixels, w, size)		\
{                                                                       \
        if (mdata != NULL)                                              \
        {                                                               \
           for (i = 0; i < size; i++, idata++)                          \
           {                                                            \
              if ((i % w) == 0 && i != 0) odata += ximage->width - w;   \
              *odata++ = !*mdata++ ? xwid->core.background_pixel :      \
                        !xcolors[*idata].flags ?                        \
                        ColorAllocate(widget, *idata) :                 \
                        pixels[*idata];                                 \
           }                                                            \
        }                                                               \
        else if (ximage->width != w)                                    \
        {                                                               \
           for (i = 0; i < size; i++)                                   \
           {                                                            \
              if ((i % w) == 0 && i != 0) odata += ximage->width - w;   \
              *odata++ = !xcolors[*idata].flags ?                       \
                        ColorAllocate(widget, *idata++) :               \
			pixels[*idata++];                               \
           }                                                            \
        }                                                               \
        else /* default case */                                         \
        {                                                               \
           while (size-- > 0)                                           \
           {                                                            \
              *odata++ = !xcolors[*idata].flags ?                       \
                        ColorAllocate(widget, *idata++) :               \
                        pixels[*idata++];                               \
           }                                                            \
        }                                                               \
}

/*-----------------------------------------------------------
|
|  Routine Name: ConvertImageData - convert image data
|
|       Purpose: 
|         Input: 
|        Output: 
|       Returns:
|
|    Written By: Mark Young
|          Date: May 06, 1994
| Modifications:
|
------------------------------------------------------------*/
#define ConvertImageData(xwid, histogram, idata, odata, mdata, ximage,	\
				xcolors, pixels, w, size)		\
{                                                                       \
        if (histogram == NULL)                                          \
        {                                                               \
	   SlowConvertImageData(xwid, idata, odata, mdata, ximage,	\
			xcolors, pixels, w, size)			\
        }                                                               \
        else								\
        {                                                               \
	   FastConvertImageData(xwid, idata, odata, mdata, ximage,	\
			pixels, w, size)				\
        }                                                               \
}

/*-----------------------------------------------------------
|
|  Routine Name: ConvertRGBImageData - convert RGB image data
|
|       Purpose:
|         Input:
|        Output:
|       Returns:
|
|    Written By: Mark Young
|          Date: May 06, 1994
| Modifications:
|
------------------------------------------------------------*/
#define ConvertRGBImageData(xwid, rptr, gptr, bptr, odata, mdata, ximage,      \
			    xcolors, w, size, rmask, gmask, bmask, rshift,     \
			    gshift, bshift)				       \
{									       \
	if (mdata != NULL)						       \
	{								       \
	   unsigned char *rmdata, *gmdata, *bmdata;			       \
	   rmdata = mdata; gmdata = &mdata[size]; bmdata = &mdata[size*2];     \
	   for (i = 0; i < size; i++, idata++, rptr++, gptr++, bptr++)	       \
	   {								       \
	      if ((i % w) == 0 && i != 0) odata += ximage->width - w;	       \
	      *odata++ = (*rmdata++ ? ((xcolors[*rptr].red >> 8) & rmask) <<   \
			  rshift : 0) |        				       \
                 (*gmdata++ ? ((xcolors[*gptr].green >> 8) & gmask) <<         \
			  gshift : 0) |        				       \
                 (*bmdata++ ? ((xcolors[*bptr].blue >> 8) & bmask) <<          \
			  bshift : 0);         				       \
	   }								       \
	}								       \
	else if (ximage->width != w)					       \
	{								       \
	   for (i = 0; i < size; i++)					       \
	   {								       \
	      if ((i % w) == 0 && i != 0) odata += ximage->width - w;	       \
              *odata++ = ((xcolors[*rptr++].red >> 8)   & rmask) << rshift |   \
                         ((xcolors[*gptr++].green >> 8) & gmask) << gshift |   \
                         ((xcolors[*bptr++].blue >> 8)  & bmask) << bshift;    \
	   }								       \
	}								       \
	else /* default case */						       \
	{								       \
	   while (size-- > 0)						       \
	   {								       \
              *odata++ = ((xcolors[*rptr++].red >> 8)   & rmask) << rshift |   \
                         ((xcolors[*gptr++].green >> 8) & gmask) << gshift |   \
                         ((xcolors[*bptr++].blue >> 8)  & bmask) << bshift;    \
	   }								       \
	}								       \
}

/*-----------------------------------------------------------
|
|  Routine Name: Convert24to8ImageData - convert 24 to 8 image data
|
|       Purpose:
|         Input:
|        Output:
|       Returns:
|
|    Written By: Mark Young
|          Date: May 06, 1994
| Modifications:
|
------------------------------------------------------------*/

#define Convert24to8ImageData(xwid, red, green, blue, odata, mdata, ximage,    \
			    xcolors, w, size)				       \
{									       \
	if (mdata != NULL)						       \
	{								       \
	   unsigned char *rmdata, *gmdata, *bmdata;			       \
	   rmdata = mdata; gmdata = &mdata[size]; bmdata = &mdata[size*2];     \
	   for (i = 0; i < size; i++, idata++, red++, green++, blue++)	       \
	   {								       \
	      if ((i % w) == 0 && i != 0) odata += ximage->width - w;	       \
	      *odata++ = (*rmdata++ ? (*red & (unsigned) 0xe0) : 0) |          \
                         (*gmdata++ ? (*green & (unsigned) 0xe0) >> 3 : 0) |   \
                         (*bmdata++ ? (*blue & (unsigned) 0xc0) >> 6 : 0);     \
	   }								       \
	}								       \
	else if (ximage->width != w)					       \
	{								       \
	   for (i = 0; i < size; i++)					       \
	   {								       \
	      if ((i % w) == 0 && i != 0) odata += ximage->width - w;	       \
              *odata++ = (*red++   & (unsigned) 0xe0) |     	               \
                         (*green++ & (unsigned) 0xe0) >> 3 |   		       \
                         (*blue++  & (unsigned) 0xc0) >> 6;   		       \
	   }								       \
	}								       \
	else /* default case */						       \
	{								       \
	   while (size-- > 0)						       \
	   {								       \
              *odata++ = (*red++ & (unsigned) 0xe0) |	   		       \
                         (*green++ & (unsigned) 0xe0) >> 3 | 		       \
                         (*blue++ & (unsigned) 0xc0) >> 6;		       \
	   }								       \
	}								       \
}

/*-----------------------------------------------------------
|
|  Routine Name: ImageReloadXData - routine for loading the image data into the
|				 ximage data structure.
|
|       Purpose: This routine is used to map the image data into the ximage
|		 data structure.
|
|         Input: widget - the image widget
|                type   - the image datatype
|        Output:
|       Returns:
|
|    Written By: Mark Young
|          Date: Apr 25, 1994
| Modifications:
|
------------------------------------------------------------*/
void ImageReloadXData(
   Widget widget,
   int    type,
   int    xoff,
   int    yoff,
   int    w,
   int	  h)
{
	XvwImageWidget xwid = (XvwImageWidget) widget;

        int      rshift, gshift, bshift;
        unsigned long rmask, gmask, bmask;
	register int i, ncolors, size = w*h;
	register unsigned char *red, *green, *blue, *mdata;
	register double *alpha;

	unsigned long *histogram;
	int      depth = xwid->image.image_depth;
	XImage   *ximage  = (XImage *) xwid->image.ximage;
	register Pixel  *pixels  = (Pixel *)  ColorGetPixels(widget);
	register XColor *xcolors = (XColor *) ColorGetXColors(widget);

	/*
	 *  Sanity check...
	 */
	if (xcolors == NULL || pixels == NULL)
	   return;

	/*
	 *  Allocate the pixels that we need before 
	 */
	if ((histogram = xwid->color.info->histogram) != NULL)
	{
	   ncolors = xwid->color.info->ncolors;
	   for (i = 0; i < ncolors; i++)
	   {
	      if (histogram[i] > 0 && xcolors[i].flags == 0)
	         (void) ColorAllocate(widget, i);
	   }
	}

	alpha = xwid->image.alpha;
	if (xwid->image.mdata)
	   mdata = xwid->image.mdata;
	else if (xwid->image.cdata)
	   mdata = xwid->image.cdata;
	else
	   mdata = NULL;

	if (ximage->bits_per_pixel == 8 && depth == 1 && type == KUBYTE)
	{
	   register unsigned char *odata;
	   register unsigned char *idata;
	   idata = (unsigned char *) xwid->image.idata;
	   odata = (unsigned char *) ximage->data + yoff*ximage->width + xoff;

	   ConvertImageData(xwid, histogram, idata, odata, mdata, ximage,
				xcolors, pixels, w, size);
	}
	else if (ximage->bits_per_pixel == 8 && depth == 1 &&
		 (type == KUSHORT || type == KSHORT))
	{
	   register unsigned char *odata;
	   register unsigned short *idata;
	   idata = (unsigned short *) xwid->image.idata;
	   odata = (unsigned char *) ximage->data + yoff*ximage->width + xoff;

	   ConvertImageData(xwid, histogram, idata, odata, mdata, ximage,
				xcolors, pixels, w, size);
	}
	else if (ximage->bits_per_pixel == 8 && depth == 1 &&
		 (type == KUINT || type == KINT))
	{
	   register unsigned char *odata;
	   register unsigned int *idata;
	   idata = (unsigned int *) xwid->image.idata;
	   odata = (unsigned char *) ximage->data + yoff*ximage->width + xoff;

	   ConvertImageData(xwid, histogram, idata, odata, mdata, ximage,
				xcolors, pixels, w, size);
	}
	else if (ximage->bits_per_pixel == 8 && depth == 1 &&
		 (type == KULONG || type == KLONG))
	{
	   register unsigned char *odata;
	   register unsigned long *idata;
	   idata = (unsigned long *) xwid->image.idata;
	   odata = (unsigned char *) ximage->data + yoff*ximage->width + xoff;

	   ConvertImageData(xwid, histogram, idata, odata, mdata, ximage,
				xcolors, pixels, w, size);
	}
	else if (ximage->bits_per_pixel == 16 && depth == 1 && type == KUBYTE)
	{
	   register unsigned short *odata;
	   register unsigned char *idata;
	   idata = (unsigned char *) xwid->image.idata;
	   odata = (unsigned short *) ximage->data + yoff*ximage->width + xoff;

	   ConvertImageData(xwid, histogram, idata, odata, mdata, ximage,
				xcolors, pixels, w, size);
	}
	else if (ximage->bits_per_pixel == 16 && depth == 1 &&
		 (type == KUSHORT || type == KSHORT))
	{
	   register unsigned short *odata;
	   register unsigned short *idata;
	   idata = (unsigned short *) xwid->image.idata;
	   odata = (unsigned short *) ximage->data + yoff*ximage->width + xoff;

	   ConvertImageData(xwid, histogram, idata, odata, mdata, ximage,
				xcolors, pixels, w, size);
	}
	else if (ximage->bits_per_pixel == 16 && depth == 1 &&
		 (type == KUINT || type == KINT))
	{
	   register unsigned short *odata;
	   register unsigned int *idata;
	   idata = (unsigned int *) xwid->image.idata;
	   odata = (unsigned short *) ximage->data + yoff*ximage->width + xoff;

	   ConvertImageData(xwid, histogram, idata, odata, mdata, ximage,
				xcolors, pixels, w, size);
	}
	else if (ximage->bits_per_pixel == 16 && depth == 1 &&
		 (type == KULONG || type == KLONG))
	{
	   register unsigned short *odata;
	   register unsigned long *idata;
	   idata = (unsigned long *) xwid->image.idata;
	   odata = (unsigned short *) ximage->data + yoff*ximage->width + xoff;

	   ConvertImageData(xwid, histogram, idata, odata, mdata, ximage,
				xcolors, pixels, w, size);
	}
        else if (ximage->bits_per_pixel == 32 && depth == 1 && type == KUBYTE)
        {
           register unsigned int *odata;
	   register unsigned char *idata;
           idata = (unsigned char *) xwid->image.idata;
           odata = (unsigned int *) ximage->data + yoff*ximage->width + xoff;
	   ximage->byte_order = (kmach_order(KMACH_LOCAL) == 
			KORDER_BIG_ENDIAN) ? MSBFirst : LSBFirst;

	   ConvertImageData(xwid, histogram, idata, odata, mdata, ximage,
				xcolors, pixels, w, size);
        }
        else if (ximage->bits_per_pixel == 32 && depth == 1 &&
		 (type == KUSHORT || type == KSHORT))
        {
           register unsigned int *odata;
           register unsigned short *idata;
           idata = (unsigned short *) xwid->image.idata;
           odata = (unsigned int *) ximage->data + yoff*ximage->width + xoff;
	   ximage->byte_order = (kmach_order(KMACH_LOCAL) == 
			KORDER_BIG_ENDIAN) ? MSBFirst : LSBFirst;

	   ConvertImageData(xwid, histogram, idata, odata, mdata, ximage,
				xcolors, pixels, w, size);
        }
        else if (ximage->bits_per_pixel == 32 && depth == 1 &&
		 (type == KUINT || type == KINT))
        {
           register unsigned int *odata;
           register unsigned int *idata;
           idata = (unsigned int *) xwid->image.idata;
           odata = (unsigned int *) ximage->data + yoff*ximage->width + xoff;
	   ximage->byte_order = (kmach_order(KMACH_LOCAL) == 
			KORDER_BIG_ENDIAN) ? MSBFirst : LSBFirst;

	   ConvertImageData(xwid, histogram, idata, odata, mdata, ximage,
				xcolors, pixels, w, size);
        }
        else if (ximage->bits_per_pixel == 32 && depth == 1 &&
		 (type == KULONG || type == KLONG))
        {
           register unsigned int *odata;
           register unsigned long *idata;
           idata = (unsigned long *) xwid->image.idata;
           odata = (unsigned int *) ximage->data + yoff*ximage->width + xoff;
	   ximage->byte_order = (kmach_order(KMACH_LOCAL) == 
			KORDER_BIG_ENDIAN) ? MSBFirst : LSBFirst;

	   ConvertImageData(xwid, histogram, idata, odata, mdata, ximage,
				xcolors, pixels, w, size);
        }
        else if (ximage->bits_per_pixel == 32 && depth == 3 &&
		 (type == KUINT || type == KINT))
        {
           Visual   *visual = xwid->manager.visual;
           register unsigned int  *odata;
           register unsigned int  *idata, *ruint, *guint, *buint;

           idata  = (unsigned int *) xwid->image.idata;
           ruint  = (unsigned int *) &idata[0];
           guint  = (unsigned int *) &idata[size];
           buint  = (unsigned int *) &idata[size*2];
           odata  = ((unsigned int *) ximage->data) + yoff*ximage->width+xoff;
	   ximage->byte_order = (kmach_order(KMACH_LOCAL) == 
			KORDER_BIG_ENDIAN) ? MSBFirst : LSBFirst;

	   rmask = xwid->color.rmask;
	   gmask = xwid->color.gmask;
	   bmask = xwid->color.bmask;
	   rshift = xwid->color.rshift;
	   gshift = xwid->color.gshift;
	   bshift = xwid->color.bshift;

	   ConvertRGBImageData(xwid, ruint, guint, buint, odata, mdata,
			       ximage, xcolors, w, size, rmask, gmask, bmask,
			       rshift, gshift, bshift);
        }
        else if (ximage->bits_per_pixel == 32 && depth == 3 &&
		 (type == KUSHORT || type == KSHORT))
        {
           Visual   *visual = xwid->manager.visual;
           register unsigned int  *odata;
           register unsigned short  *idata, *rushort, *gushort, *bushort;

           idata    = (unsigned short *) xwid->image.idata;
           rushort  = (unsigned short *) &idata[0];
           gushort  = (unsigned short *) &idata[size];
           bushort  = (unsigned short *) &idata[size*2];
           odata  = ((unsigned int *) ximage->data) + yoff*ximage->width + xoff;
	   ximage->byte_order = (kmach_order(KMACH_LOCAL) == 
			KORDER_BIG_ENDIAN) ? MSBFirst : LSBFirst;

	   rmask = xwid->color.rmask;
	   gmask = xwid->color.gmask;
	   bmask = xwid->color.bmask;
	   rshift = xwid->color.rshift;
	   gshift = xwid->color.gshift;
	   bshift = xwid->color.bshift;

	   ConvertRGBImageData(xwid, rushort, gushort, bushort, odata, mdata,
			       ximage, xcolors, w, size, rmask, gmask, bmask,
			       rshift, gshift, bshift);
        }
        else if (ximage->bits_per_pixel == 32 && depth == 3)
        {
           Visual   *visual = xwid->manager.visual;
           register unsigned int  *odata;
           register unsigned char *idata;

           idata = (unsigned char *) xwid->image.idata;
           red   = (unsigned char *) &idata[0];
           green = (unsigned char *) &idata[size];
           blue  = (unsigned char *) &idata[size*2];
           odata = ((unsigned int *) ximage->data) + yoff*ximage->width+xoff;
	   ximage->byte_order = (kmach_order(KMACH_LOCAL) == 
			KORDER_BIG_ENDIAN) ? MSBFirst : LSBFirst;

	   rmask = xwid->color.rmask;
	   gmask = xwid->color.gmask;
	   bmask = xwid->color.bmask;
	   rshift = xwid->color.rshift;
	   gshift = xwid->color.gshift;
	   bshift = xwid->color.bshift;

	   ConvertRGBImageData(xwid, red, green, blue, odata, mdata, ximage,
                               xcolors, w, size, rmask, gmask, bmask, rshift,
                               gshift, bshift);
        }
	else if (ximage->bits_per_pixel == 8 && depth == 3)
	{
	   register unsigned char *odata;
           register unsigned char *idata;

           idata = (unsigned char *) xwid->image.idata;
           red   = (unsigned char *) &idata[0];
           green = (unsigned char *) &idata[size];
           blue  = (unsigned char *) &idata[size*2];
           odata = (unsigned char *) ximage->data + yoff*ximage->width + xoff;

	   Convert24to8ImageData(xwid, red, green, blue, odata, mdata, ximage,
                                 xcolors, w, size);

	   /*
	    *  Now that we have performed our 3-3-2, we need to allocate the
	    *  Pixels.  We can re-use the ConvertImageData(), but it's probably
	    *  easier to just loop thru it.
	    */
	   size = w*h;
	   odata = (unsigned char *) ximage->data + yoff*ximage->width + xoff;
	   while (size-- > 0)
	   {
	      *odata++ = !xcolors[*odata].flags ?
                        ColorAllocate(widget, *odata) :
                        xcolors[*odata].pixel;          
	   }
	}
	xwid->image.recolor = FALSE;
}

/*-----------------------------------------------------------
|
|  Routine Name: ApplyRGBAlphaData - apply alpha to the RGB data 
|
|       Purpose:
|         Input:
|        Output:
|       Returns:
|
|    Written By: Mark Young
|          Date: May 06, 1994
| Modifications:
|
------------------------------------------------------------*/
static void ApplyRGBAlphaData(
   Widget widget,
   int	  type,
   int    w,
   int    h)
{
	XvwImageWidget xwid = (XvwImageWidget) widget;

	int      i, size = w*h;
	register double *alpha = xwid->image.alpha;


	if (alpha == NULL)
	   return;

	if (type == KUBYTE  || type == KBYTE)
	{
	   unsigned char *red, *green, *blue, *idata;
	   idata = (unsigned char *) xwid->image.idata;
	   red = idata; green = &idata[size]; blue = &idata[size*2];

	   for (i = 0; i < size; i++)
	      red[i] *= alpha[i], green[i] *= alpha[i], blue[i] *= alpha[i];
	}
	else if (type == KUINT   || type == KINT)
	{
	   unsigned int *red, *green, *blue, *idata;
	   idata = (unsigned int *) xwid->image.idata;
	   red = idata; green = &idata[size]; blue = &idata[size*2];

	   for (i = 0; i < size; i++)
	      red[i] *= alpha[i], green[i] *= alpha[i], blue[i] *= alpha[i];
	}
	else if (type == KULONG  || type == KLONG)
	{
	   unsigned long *red, *green, *blue, *idata;
	   idata = (unsigned long *) xwid->image.idata;
	   red = idata; green = &idata[size]; blue = &idata[size*2];

	   for (i = 0; i < size; i++)
	      red[i] *= alpha[i], green[i] *= alpha[i], blue[i] *= alpha[i];
	}
	else if (type == KUSHORT || type == KSHORT)
	{
	   unsigned short *red, *green, *blue, *idata;
	   idata = (unsigned short *) xwid->image.idata;
	   red = idata; green = &idata[size]; blue = &idata[size*2];

	   for (i = 0; i < size; i++)
	      red[i] *= alpha[i], green[i] *= alpha[i], blue[i] *= alpha[i];
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ImageReload
|
|       Purpose: Reload the ximage data structure from the image
|		 data object.  The cb is the pds callback data which
|		 represents the region to be refreshed.  If NULL then
|		 the whole image is reloaded. 
|
|         Input: widget - the image widget to load the image data
|		 cb     - the pds callback data indicating the region
|			  to reload.
|
|        Output: none
|
|    Written By: Mark Young
|          Date: Nov 21, 1992 12:52
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void ImageReload(
   Widget	widget,
   kdms_callback *cb)
{
	XvwImageWidget xwid = (XvwImageWidget) widget;

	XImage   *ximage;
	kobject  iobject;
	double   offset, factor;
	register unsigned char *data, *mdata, *cdata;
	int      i, xoff, yoff, size, region_num, num_bytes, scale;
	int      type, ctype, x, y, w, h, d, t, e, tmpt, tmpd, tmpe;


	/*
	 *  Sanity check to make sure that the image object exists and that
	 *  the ximage as well.
	 */
	if (!XtWindow(widget) || (ximage = xwid->image.ximage) == NULL ||
	    (iobject = xwid->image.iobject) == NULL)
	   return;

	tmpe = xwid->image.band_number % xwid->image.esize;
	tmpd = xwid->image.band_number/xwid->image.esize % xwid->image.dsize;
	tmpt = xwid->image.band_number/(xwid->image.esize*xwid->image.dsize) %
			xwid->image.tsize;

	/*
	 *  Set the current position of in the image in which we want to
	 *  get the image band.
	 */
	if (cb)
	{
	   x = xoff = cb->begin.w;
	   y = yoff = cb->begin.h;
	   w = (cb->end.w - cb->begin.w) + 1;
	   h = (cb->end.h - cb->begin.h) + 1;
	   xoff -= xwid->image.xoffset;
	   yoff -= xwid->image.yoffset;
	   region_num = 1;
	}
	else
	{
	   if ((region_num = xwid->image.region_num) > 1)
	   {
	      w = xwid->image.region_width;
	      h = xwid->image.region_height;
	   }
	   else
	   {
	      if ((w = xwid->image.wsize) > ximage->width)
		 w = ximage->width;

	      if ((h = xwid->image.hsize) > ximage->height)
		 h = ximage->height;
	   }
	   x = y = xoff = yoff = 0;
	   x += xwid->image.xoffset;
	   y += xwid->image.yoffset;
	}


	/*
	 *  Get the processing type...
	 */
	kpds_get_attribute(iobject, KPDS_VALUE_DATA_TYPE, &type);
	switch (type)
	{
	   case KSHORT:
	   case KUSHORT:
		size = sizeof(short); ctype = type;
		break;

	   case KINT:
	   case KUINT:
		size = sizeof(int); ctype = type;
		break;

	   case KLONG:
	   case KULONG:
		size = sizeof(int); ctype = type;
		break;

	   default:
		size = 1; ctype = KUBYTE;
		break;
	}

	/*
	 *  All the following is really expensive, and under certain conditions
	 *  this routine will be called, but all we really want to do is call
	 *  ImageReloadXData() with the right parameters.  So check to see if
	 *  xwid->image.reload is really set...
	 */
	if (xwid->image.reload == TRUE)
	{
           /*
	    *  Set up the begin and end points for the region get
	    */
	   kpds_set_attributes(iobject,
		KPDS_VALUE_DATA_TYPE, ctype,
		KPDS_VALUE_POSITION, x, y, tmpd, tmpt, tmpe,
		NULL);
	   xvw_set_type(iobject, xwid->image.reload_type, IMAGE_REFRESH_UPDATE);

	   if (region_num == 1)
	   {
	      kpds_set_attribute(iobject, KPDS_VALUE_REGION_SIZE, w, h, 1, 1,
			xwid->image.image_depth);
	      kfree(xwid->image.idata);
	      xwid->image.idata = kpds_get_data(iobject, KPDS_VALUE_REGION,
			                        xwid->image.idata);
	   }
	   else
	   {
	      kpds_set_attribute(iobject, KPDS_VALUE_REGION_SIZE, w, h, 1, 1,1);
	      if ((data = (unsigned char *) xwid->image.idata) == NULL)
	      {
		 num_bytes = region_num*size*w*h*xwid->image.image_depth;
	         xwid->image.idata = kmalloc(num_bytes);
	         data = (unsigned char *) xwid->image.idata;
	      }

	      for (i = 0; i < region_num*xwid->image.image_depth; i++)
	      {
		 if ((i % region_num) == 0)
		 {
	            kpds_set_attribute(iobject, KPDS_VALUE_POSITION, 0, 0,
			tmpd, tmpt, i/region_num);
		 }

                 data = kpds_get_data(iobject, KPDS_VALUE_REGION, data);
	         if (data == NULL)
                 {
	            kerror(XVISUAL, "ImageReload", "Unable to get image data");
	            return;
	         }
	         data += w*h*size;
	      }
	   }

	   /*
	    *  If we have alpha data, we better get that too..
	    */
	   if (xwid->image.has_alpha)
	   {
              kpds_get_attributes(iobject,
                   KPDS_VALUE_SCALING, &scale,
                   KPDS_VALUE_SCALE_FACTOR, &factor,
                   KPDS_VALUE_SCALE_OFFSET, &offset, NULL,
                   NULL);
              kpds_set_attributes(iobject,
                   KPDS_VALUE_SCALING, KSCALE,
		   KPDS_VALUE_DATA_TYPE, KDOUBLE,
                   KPDS_VALUE_SCALE_OFFSET, xwid->image.alpha_offset, 0.0,
                   KPDS_VALUE_SCALE_FACTOR, xwid->image.alpha_factor,
		   KPDS_VALUE_REGION_SIZE, w, h, 1, 1, 1,
		   KPDS_VALUE_POSITION, x, y, tmpd, tmpt, 3,
                   NULL);
	      xwid->image.alpha = kpds_get_data(iobject, KPDS_VALUE_REGION,
			 xwid->image.alpha);
              kpds_set_attributes(iobject,
		   KPDS_VALUE_SCALING, scale,
		   KPDS_VALUE_SCALE_OFFSET, offset, 0.0,
		   KPDS_VALUE_SCALE_FACTOR, factor,
		   KPDS_VALUE_DATA_TYPE, type,
		   NULL);
	      ApplyRGBAlphaData(widget, ctype, w, h*region_num);
	   }
	   else
	      kpds_set_attribute(iobject, KPDS_VALUE_DATA_TYPE, type);

	   /*
	    *  If we have masks, we better get them too..
	    */
	   if (kpds_query_mask(iobject) == TRUE)
	   {
              kpds_set_attributes(iobject,
                   KPDS_MASK_DATA_TYPE, KUBYTE,
                   KPDS_MASK_POSITION, x, y, tmpd, tmpt, tmpe,
                   KPDS_MASK_REGION_SIZE, w, h, 1, 1, 1,
                   NULL);

              if ((mdata = (unsigned char *) xwid->image.mdata) == NULL)
              {
		 num_bytes = region_num*w*h*xwid->image.image_depth;
                 xwid->image.mdata = kmalloc(num_bytes);
                 mdata = (unsigned char *) xwid->image.mdata;
              }

              for (i = 0; i < region_num*xwid->image.image_depth; i++)
              {
                 mdata = kpds_get_data(iobject, KPDS_MASK_REGION, mdata);
                 mdata += w*h;
              }
	   }

	   /*
	    *  If we have a clip object, we better get the clip data and combine
	    *  it with the mask data.
	    */
	   if (xwid->image.cobject)
	   {
	      kpds_get_attribute(xwid->image.cobject,
		   KPDS_VALUE_SIZE, NULL, NULL, &d, &t, &e);
	      kpds_set_attribute(xwid->image.cobject,
		   KPDS_VALUE_DATA_TYPE, KUBYTE);
	      kpds_set_attribute(xwid->image.cobject,
		   KPDS_VALUE_POSITION, x, y, tmpd % d, tmpt % t, tmpe % e);
	      kpds_set_attribute(xwid->image.cobject,
		   KPDS_VALUE_REGION_SIZE, w, h, 1, 1, 1);

	      if ((cdata = (unsigned char *) xwid->image.cdata) == NULL)
	      {
		 num_bytes = region_num*w*h*xwid->image.image_depth;
	         xwid->image.cdata = kmalloc(num_bytes);
	         cdata = (unsigned char *) xwid->image.cdata;
	      }

	      for (i = 0; i < region_num*xwid->image.image_depth; i++)
	      {
                 cdata = kpds_get_data(xwid->image.cobject, KPDS_VALUE_REGION,
				cdata);
		 cdata += w*h;
	      }

	      if (xwid->image.mdata != NULL)
	      {
	         mdata = (unsigned char *) xwid->image.mdata;
	         cdata = (unsigned char *) xwid->image.cdata;
	         for (i = 0; i < region_num*w*h*xwid->image.image_depth; i++)
	            *mdata++ = (*cdata++ && *mdata) ? *mdata : 0;
	      }
	   }
	}

	/*
	 *  Map the data thru the object's xcolor array
	 */
	ImageReloadXData(widget, ctype, xoff, yoff, w, h*region_num);
	xwid->image.reload = FALSE;
}

/*-----------------------------------------------------------
|
|  Routine Name: ImageSetPosition
|
|	Purpose: This routine is used to set the x&y image position at which
|		 the user is currently focused.  The image widget uses data
|		 services to define the protocol that keeps all visual objects
|		 in sync.  Access callbacks are install on the data in order
|		 to keep all visual objects in sync.  When the focus changes
|		 (button press or pointer motion in the image by the user,
|		  or explicit sets of XVW_IMAGE_XPOSITION and/or
|		  XVW_IMAGE_YPOSITON by the programmer) the image
|		 widget will perform a get data at that position.  The visual
|		 objects that have installed access callbacks on the data will
|		 see this access and know to update their displays.  In order
|		 reduce the number of extraneous updates the image widget
|		 sets the "KUPDATE_TYPE" attribute on the data object.  The
|		 visual objects inspect the update attribute to see why the
|		 callback has been fired and respond appropriately.
|
|         Input: widget - widget data structure for image
|		 x      - the new x position
|		 y      - the new y position
|		 update_type - the reason for the update
|        Output:
|    Written By: Mark Young
|          Date: May 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void ImageSetPosition(
   Widget widget,
   int    x,
   int    y,
   int    update_type)
{
	XvwImageWidget xwid = (XvwImageWidget) widget;

	unsigned char value;
	int type, w, h, d, t, e;

	/*
	 *  Need to get some information from the value segment such as the
	 *  width, height.  We also need the location of the value segment's
	 *  depth, time, and element.  We use this to set the position at
	 *  which the user is focused.  We also are going to temporarily
	 *  override the current data type, so we need to save it in order
	 *  to restore later.
	 */ 
	kpds_get_attributes(xwid->image.iobject,
		KPDS_VALUE_SIZE, &w, &h, NULL, NULL, NULL,
		KPDS_VALUE_POSITION, NULL, NULL, &d, &t, &e,
		KPDS_VALUE_DATA_TYPE, &type,
		NULL);

	/*
	 *  Update the Image widget's xposition & yposition to the x&y position
	 *  that the user focused.
	 */
	xwid->image.xposition = x;
	xwid->image.yposition = y;

	/*
	 *  The position is where we get the data from in order to force the
	 *  callback.  Also, we want to save the value at that point so set
	 *  the type to be double.
	 */
	kpds_set_attributes(xwid->image.iobject,
		KPDS_VALUE_POSITION, x, y, d, t, e,
		KPDS_VALUE_DATA_TYPE, KDOUBLE,
		NULL);

	/*
	 *  Update our "reload" who and our "update" why that we want to get
	 *  this data.
	 */
	xvw_set_type(xwid->image.iobject, xwid->image.reload_type, update_type);
	kpds_get_data(xwid->image.iobject,KPDS_VALUE_POINT, &xwid->image.value);

	value = (unsigned char) xwid->image.value;
	xwid->image.pixel = ColorAllocate(widget, value);
	kpds_set_attribute(xwid->image.iobject, KPDS_VALUE_DATA_TYPE, type);
}

/*-----------------------------------------------------------
|
|  Routine Name: ImageUpdatePosition
|
|       Purpose: updates the position of the cursor in the image object
|		 so that if there is other devices that have any access
|		 access callbacks we will know it.
|
|         Input: widget      - image widget
|                client_data - client data (unused)
|                event       - type of event that triggered us
|
|        Output: dispatch    - whether to continue dispatching this event
|       Returns: none
|
|    Written By: Mark Young
|          Date: Aug 19, 1992
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void ImageUpdatePosition(
   xvobject object,
   kaddr    client_data,
   XEvent   *event,
   int      *dispatch)
{
	int x, y, xpos, ypos, type;
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);


	/*
	 *  Can't allow position updates on anything, but the ImageWidget.
	 *  Why?  Better come ask...
	 */
	if (xwid->image.reload_type != IMAGE_OBJECT_RELOAD &&
	    xwid->image.reload_type != ZOOM_OBJECT_RELOAD)
	   return;

	if (event->type == MotionNotify)
	{
	   x = event->xmotion.x; y = event->xmotion.y;
	   type = MOTION_EVENT_UPDATE;
	}
	else if (event->type == ButtonPress)
	{
	   x = event->xbutton.x; y = event->xbutton.y;
	   type = BUTTONPRESS_EVENT_UPDATE;
	}
	else
	   return;

	if (x < 0) x = 0;
	if (y < 0) y = 0;
	if (x >= (int) xwid->core.width)  x = ((int) xwid->core.width) -1;
	if (y >= (int) xwid->core.height) y = ((int) xwid->core.height) -1;
	xpos = x + xwid->image.xoffset;
	ypos = y + xwid->image.yoffset;
	if (xpos != xwid->image.xposition || ypos != xwid->image.yposition ||
	    type == BUTTONPRESS_EVENT_UPDATE)
	{
	   ImageSetPosition((Widget) xwid, xpos, ypos, type);
	   kinfo(KDEBUG, "Image: update position (%d,%d)", xpos, ypos);
	}
	else
	   kinfo(KDEBUG,"Image: update position but position hasn't changed");
}

/*-----------------------------------------------------------
|
|  Routine Name: ImageGetName
|
|       Purpose: ImageGetName is used to get the filename associated
|		 with the attribute.
|
|         Input: widget    - the widget in which we will be getting the
|			     filename
|		 attribute - the old object to be closed
|
|        Output: calldata  - pointer to retrieve the object name
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 21, 1992 12:52
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int ImageGetName(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	kobject image;
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);
	String *filename = (String *) calldata;


	if (kstrcmp(attribute, XVW_IMAGE_IMAGEFILE) == 0)
	   image = xwid->image.iobject;
        else if (kstrcmp(attribute, XVW_IMAGE_CLIPFILE) == 0)
	   image = xwid->image.cobject;
	else
	{
	   kerror(XVISUAL, "ImageGetName", "invalid attribute %s",
			attribute);
	   return(FALSE);
	}
	*filename = NULL;
	return(kpds_get_attribute(image, KPDS_NAME, filename));
}

/*-----------------------------------------------------------
|
|  Routine Name: ImageSetName
|
|       Purpose: ImageSetName is used to set the filename associated
|		 with the attribute.
|
|         Input: object    - the object in which we will be getting the
|			     filename
|		 attribute - the old object to be closed
|		 calldata  - the filename to be set
|
|        Output:
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 21, 1992
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int ImageSetName(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int	locate = TRUE;
	kobject out, image = NULL;
	XvwImageWidget xwid = (XvwImageWidget) xvw_widget(object);
	String *filename = (String *) calldata;


	if (kstrcmp(attribute, XVW_IMAGE_SAVEIMAGE) == 0)
	{
	   if ((out = kpds_open_object(*filename, KOBJ_WRITE)) == NULL)
	   {
	      kerror(XVISUAL, "ImageSetName", "Unable to open output file '%s' \
for saving the image to.", filename);
	      return(FALSE);
	   }

           /*
            *  copy the attributes & the data of the displayed image
            *  to the output image
            */
	   kdms_copy_object(xwid->image.iobject, out, TRUE, TRUE);
 
           /*
            *  that's all... close the output object
            */
           kpds_close_object(out);
	   return(TRUE);
	}

	if (*filename != NULL && (image = kdms_locate(*filename)) == NULL)
	{
	   locate = FALSE;
	   if ((image = kpds_open_object(*filename, KOBJ_READ)) == NULL)
	   {
	      kerror(XVISUAL, "ImageSetName", "Failed to load image '%s'",
			*filename);
	      return(TRUE);
	   }
	}

	if (kstrcmp(attribute, XVW_IMAGE_IMAGEFILE) == 0)
	   xvw_set_attribute(object, XVW_IMAGE_IMAGEOBJ, image);
        else if (kstrcmp(attribute, XVW_IMAGE_CLIPFILE) == 0)
	   xvw_set_attribute(object, XVW_IMAGE_CLIPOBJ, image);
        else if (kstrcmp(attribute, XVW_IMAGE_COLORMAPFILE) == 0)
	   xvw_set_attribute(object, XVW_IMAGE_COLORMAPOBJ, image);
	else
	{
	   kerror(XVISUAL, "ImageSetName", "invalid attribute %s", attribute);
	   if (!locate) kpds_close_object(image);
	   return(FALSE);
	}
	if (!locate) kpds_close_object(image);
	
	return(TRUE);
}
