 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Color Utility Routines
   >>>>
   >>>>   Static:
   >>>>                 ColorMapInitialize()
   >>>>                 ColorTimeout()
   >>>>                 ColorReadCodebook()
   >>>>			ColorReloadCallback()
   >>>>  Private:
   >>>>                 ColorUtilCallback()
   >>>>			ColorInit()
   >>>>			GetColorPart()
   >>>>			ColorGetPixels()
   >>>>			ColorGetXColors()
   >>>>			ColorGetHistogram()
   >>>>			ColorAllocate()
   >>>>			ColorFree()
   >>>>			ColorStore()
   >>>>			ColorSort()
   >>>>			ColorReload()
   >>>>			ColorReloadXData()
   >>>>			ColorSetUtil()
   >>>>			ColorGetUtil()
   >>>>                 ColorFileDetect()
   >>>>			ColorChangeMapCol()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvisual/ColorP.h>

#define XVW_COLOR_INFO "colorInfo"

static void ColorTimeout        PROTO((XtPointer, XtIntervalId *));
static void ColorMapInitialize  PROTO((Widget));
static void ColorReadCodebook   PROTO((XvwColorPart *, kobject));
static void ColorReloadCallback PROTO((kobject,char *,Widget,kdms_callback *));


/*-----------------------------------------------------------
|
|  Routine Name: ColorTimeout - routine to update colors in Image, Palette,
|                                Animate, Zoom, (etc) objects.
|
|       Purpose: This routine simply calls the object's color reload
|		 procedure and any color callbacks installed by the
|		 application.  We have ColorTimeout installed as a
|		 timeout, so that we can wait till all the colormap
|		 changes have been done before the color reload procedure
|		 is done, since otherwise color update is unbearably slow
|		 as we chug through the same process umpty-ump times.
|
|         Input: calldata - the color widget
|                id       - the timer id
|        Output:
|       Returns:
|    Written By: Mark Young & Danielle Argiro
|          Date: May 23, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ColorTimeout(
   XtPointer    calldata,
   XtIntervalId *id)
{
        Widget    widget = (Widget) calldata;

	XvwColorPart *color = GetColorPart(widget);

	color->id = 0;
	widget = color->info->widget;
        if (XtIsSubclass(widget, xvwColorGadgetClass))
	{
	   XvwColorGadgetClass class;

	   class = (XvwColorGadgetClass) widget->core.widget_class;
	   if (class->color_class.color_reload != NULL)
              class->color_class.color_reload(widget, NULL);
	}
        else if (XtIsSubclass(widget, xvwColorWidgetClass))
	{
	   XvwColorWidgetClass class;

	   class = (XvwColorWidgetClass) widget->core.widget_class;
	   if (class->color_class.color_reload != NULL)
              class->color_class.color_reload(widget, NULL);
	}
	XtCallCallbacks(widget, XVW_COLOR_CALLBACK, NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorMapInitialize - 
|
|       Purpose: initialize the object's maps...
|    Written By: Mark Young
|          Date: Jul 25, 1994
| Modifications:
|
------------------------------------------------------------*/

static void ColorMapInitialize(
   Widget widget)
{
        XvwColorPart *color = GetColorPart(widget);

	kobject object = color->object;
	int     e, width, height, type, colorspace, has_alpha, autocolor;


	/*
	 *  Make sure that the colormap exists...
	 */
	if (kpds_query_map(object) == TRUE)
	   return;

	kpds_get_attributes(object,
                KPDS_VALUE_DATA_TYPE, &type,
		KPDS_VALUE_SIZE, NULL, NULL, NULL, NULL, &e,
		NULL);

	kcolor_get_attributes(object,
		KCOLOR_COLORSPACE, &colorspace,
		KCOLOR_HAS_ALPHA, &has_alpha,
		NULL);

	if (color->visual_depth == 8 && colorspace == KRGB &&
	    (e == 3 || (e == 4 && has_alpha)))
	{
	   autocolor = KMAP332; 
	}
	else
	{
	   autocolor = KGREYSCALE; 
	   (void) ColorGetHistogram(widget);
	}

	if (has_alpha == TRUE)
	   width = 4;
	else
	   width = 3;

	if (type == KSHORT || type == KUSHORT ||
	    type == KINT   || type == KUINT ||
	    type == KLONG  || type == KULONG ||
	    type == KBYTE  || (type == KUBYTE && e == 1 && color->norm_ubyte))
	{
	   height = (color->info->maxval - color->info->minval) + 1;
	   if (height > 65535)
	      height = 65535;

	   if (has_alpha == TRUE)
	      type = KDOUBLE;
	   else if (type == KBYTE)
	      type = KUBYTE;
	}
	else if (type == KBIT)
	{
	   height = 2;
	   type = KUBYTE;
	}
	else
	{
	   height = 256;
	   if (has_alpha == TRUE)
	      type = KDOUBLE;
	   else
	      type = KUBYTE;
	}
	kpds_create_map(object);
	kpds_set_attribute(object, KPDS_MAP_SIZE, width, height, 1, 1, 1);
	kpds_set_attribute(object, KPDS_MAP_DATA_TYPE, type);
	kpds_set_attribute(object, KPDS_COUPLING, KUNCOUPLED);
	kdms_update_references(object, KDMS_SEGMENT_MAP);
	kcolor_set_attribute(object, KCOLOR_MAP_AUTOCOLOR, autocolor);

        color->info->map_width  = width;
        color->info->map_height = height;
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorReadCodebook
|       Purpose: Reads information out of the map columns.
|
|                The "colorMapContents" (VIFF 2.0) attribute, or
|                image->ispare1 (VIFF 1.0, xvimage field) is a mask which
|                tells us what is stored in the map columns.
|                You read it like this:
|                0 0 0 0 0 0 - no codebook available (colormap data only)
|                0 0 0 0 0 1 - count
|                0 0 0 0 1 0 - class
|                0 0 0 1 0 0 - covariance matrix (upper triangle)
|                0 0 1 0 0 0 - covariance diagonal
|
|                The following diagram illustrates the layout of the 
|                codebook information as stored in the map data:
|
|            |-----------|               |--------------------------------
|            |        | 2|-------        |  .  .  .  |  |  |        | c d|
|            | pixel  ---|       ---------->.  .  .  | c| c|        | o i|
|            | values    |               | colormap  | o| l|        | v a|
|            | stored    |               |   data    | u| a|        | a g|
|            | in VIFF   |               |  .  .  .  | n| s| covar  | r o|
|            | imagedata |               |  .  .  .  | t| s| matrix | i n|
|            |           |             | |  .  .  .  |  |  |        | a a|  |
|            |  values   |             | |  .  .  .  |  |  |        | n l|  |
|            | 0 - (m-1) |             V |  .  .  .  |  |  |        | c  |  V
|            |-----------|             m |---------->n------------------>N  M
|
|
|		N = # cols in entire map
|		M = # rows in entire map
|
|		n = #columns in = vector dimension
|		  = image->ispare2 in 1.0.5 VIFF
|		  = "colorMapColnum" in 2.0 VIFF
|
|               m = # mean vectors total = M
|
|         Input:
|        Output: none
|       Returns: 
|    Written By: Danielle Argiro & Mark Young
|          Date: June 23, 1994
| Modifications:
|
---------------------------------------------------------------*/

/*
 * contents of map data
 */
#define HISTOGRAM_COLUMN (1L << 0)
#define CLASS_COLUMN     (1L << 1)
#define COVAR_MATRIX     (1L << 2)
#define COVAR_DIAGONAL   (1L << 3)

static void ColorReadCodebook(
    XvwColorPart *color,
    kobject      data_object)
{
	int map_contents, map_colnum, mapcol_count;
	int map_width, map_height;
	int ispare1, ispare2;

	map_width  = color->info->map_width;
	map_height = color->info->map_height;

        /*
         * read in the double map data
         */
        kpds_set_attribute(data_object, KPDS_MAP_DATA_TYPE, KDOUBLE);

        /*
         * K1 VIFF has map  contents = ispare1, 
         *             column number = ispare2
         * K2 VIFF has map contents  = "colorMapContents" attribute,
         *             column number = "colorMapColnum"   attribute
         */
        if (!(kpds_get_attribute(data_object, "colorMapContents",
                                 &map_contents)))
        {
            if (kpds_get_attribute(data_object, "ispare1", &ispare1))
               map_contents = (int) ispare1;
            else map_contents = 0;

        }
        if (!(kpds_get_attribute(data_object, "colorMapColnum", &map_colnum)))
        {
            if (kpds_get_attribute(data_object, "ispare2", &ispare2))
                map_colnum   = (int) ispare2;
            else map_colnum = 0;
        }

	/*
	 *  check to see if colorMapContents & colorMapColnum are valid
	 *  (to the best of our ability)
	 */
	if (map_contents == 0)
	    return;

	mapcol_count = map_colnum;
	if (map_contents & HISTOGRAM_COLUMN) 
	    mapcol_count++;
	if (map_contents & CLASS_COLUMN) 
            mapcol_count++;
	if (map_contents & COVAR_MATRIX)
	   mapcol_count = mapcol_count + ((map_colnum * (map_colnum+1)) / 2);
	else if (map_contents & COVAR_DIAGONAL)
	   mapcol_count = mapcol_count + map_colnum;
	if (mapcol_count != map_width)
	   return;

	if (map_contents & HISTOGRAM_COLUMN)
        {
            kpds_set_attributes(data_object,
			KPDS_MAP_DATA_TYPE,   KUINT,
			KPDS_MAP_INTERPOLATE, KNONE,
			KPDS_MAP_POSITION,    map_colnum, 0, 0, 0, 0,
			KPDS_MAP_REGION_SIZE, 1, map_height, 1, 1, 1,
			NULL);
            color->info->histogram = kpds_get_data(data_object, 
						   KPDS_MAP_REGION, NULL);
            if (color->info->histogram == NULL)
            {
                kwarn(XVISUAL, "ColorReadCodebook",
                      "The colorMapContents attribute for your input image (if it's a  VIFF image) or the ispare1 value (if it's an xvimage) contains a mask which indicates the presence of a histogram.  However, there is no histogram column stored in the maps of the image.  Somehow, your histogram seems to have been lost from your map data.");
            }
        }
	color->info->map_width = map_colnum;
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorReloadCallback
|
|       Purpose: Color Reload the xcolor data structure from the color
|                data object.
|
|         Input: widget - the image widget to load the image data
|
|        Output: none
|
|    Written By: Mark Young
|          Date: May 18, 1993 14:44
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ColorReloadCallback(
	kobject object,
	char    *segment,
	Widget  widget,
	kdms_callback *cb)
{
	XvwColorPart *color = GetColorPart(widget);
	int indx, num;

	color->info->initialized = TRUE;
	indx = cb->begin.h;
	num  = (cb->end.h - cb->begin.h) + 1;

	/*
	 *  free the allocated color for the pixel
	 */
	if (color->alloc_policy == KCOLOR_ALLOC_READONLY)
	   ColorFree(widget, indx, num);

	/*
	 *  reload the XColor array from the changed map data
	 */
	ColorReload(widget, indx, num);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorUtilCallback - routine to update colors in Image, Palette,
|                                Animate, Zoom, (etc) objects.
|
|       Purpose: This routine simply calls the object's color reload
|		 procedure.  We have ColorUtilCallback installed as a
|		 callback, so that when a color change is done we will be
|		 informed and will refresh our own display.
|
|         Input: calldata - the color widget
|                id       - the timer id
|        Output:
|       Returns:
|    Written By: Mark Young & Danielle Argiro
|          Date: May 23, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void ColorUtilCallback(
   xvobject object,
   kaddr    clientdata,
   kaddr    calldata)
{
        Widget    widget = (Widget) clientdata;

	XvwColorPart *color = GetColorPart(widget);

	color->id = 0;
        if (XtIsSubclass(widget, xvwColorGadgetClass))
	{
	   XvwColorGadgetClass class;

	   class = (XvwColorGadgetClass) widget->core.widget_class;
	   if (class->color_class.color_reload != NULL)
              class->color_class.color_reload(widget, NULL);
	}
        else if (XtIsSubclass(widget, xvwColorWidgetClass))
	{
	   XvwColorWidgetClass class;

	   class = (XvwColorWidgetClass) widget->core.widget_class;
	   if (class->color_class.color_reload != NULL)
              class->color_class.color_reload(widget, NULL);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorInit - initialize the color
|
|       Purpose: Initializes the ColorClass for a given object.
|		 Given an object to be displayed we initialize
|		 the ColorClass's red, green, blue function.
|         Input: 
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Jun 10, 1994
| Modifications:
|
------------------------------------------------------------*/

void ColorInit(
  Widget widget)
{
	XvwColorPart *color = GetColorPart(widget);

	kobject  object;
	xvobject colorobj;
	char	 *function, *filename, temp[KLENGTH];
	int      type, i, w, h, rcol = -1, bcol = -1, gcol = -1;


	/*
	 *  Sanity check to see if we have the necessary information in which
	 *  to initialize the ColorClass.
	 */
	if (!color) return;

	/*
	 *  In case there was previously a callback placed remove it, otherwise
	 *  we'll end up confusing ourselves.
	 */ 
	if (color->info != NULL && color->info->widget != NULL &&
	    color->info->widget != widget)
	{
	   xvw_remove_callback(xvw_object(color->info->widget),
		XVW_COLOR_CALLBACK, ColorUtilCallback, (XtPointer) widget);
	}

	/*
	 *  Now check to see if the object is NULL, if so then just return...
	 */
	if (color->object == NULL)
	   return;

	/*
	 *  Allocate the color->info structure if one doesn't exist, otherwise
	 *  share it with the other objects.
	 */
	object = color->object;
	if (!kdms_query_attribute(object, NULL, XVW_COLOR_INFO, NULL,
				  NULL, NULL, NULL))
	{
	   color->allocated = TRUE;
	   color->info = (ColorObjInfo *) kcalloc(1, sizeof(ColorObjInfo));
	   color->info->references = 1;
	   kdms_create_attribute(object, NULL, XVW_COLOR_INFO, 1, 1,
				 KULONG, FALSE, TRUE);
	   kpds_set_attribute(object, XVW_COLOR_INFO, color->info);
	   kpds_get_attribute(object, KPDS_NAME, &filename);

	   /*
	    *  Make sure that we have maps...
	    */
	   color->info->scale  = 1.0;
	   color->info->minval = color->info->maxval = 0;
	   if (!kpds_query_map(color->object))
	      ColorMapInitialize(widget);
	   else color->info->initialized = TRUE;

	   /*
	    *  If the map are of type double, float, or complex then we
	    *  need to make sure the data is normalized between 0 and 255.
	    */
	   kpds_get_attributes(object,
			KPDS_MAP_DATA_TYPE, &type,
	         	KPDS_MAP_SIZE, &w, &h, NULL, NULL, NULL,
			NULL);
	   color->info->map_width  = w;
	   color->info->map_height = h;
	   ColorReadCodebook(color, object);
	   color->info->pixels  = (Pixel *)  kcalloc(h, sizeof(Pixel));
	   color->info->xcolors = (XColor *) kcalloc(h, sizeof(XColor));
	   for (i = 0; i < h; i++) color->info->pixels[i] = (Pixel) -1;

	   color->info->widget   = widget;
	   color->info->filename = kstrdup(filename);

	   /*
	    *  Error check to make sure that the red, green, blue mappings
	    *  are less than our map width.
	    */
	   colorobj = xvw_object(widget);
	   xvw_get_attributes(colorobj,
			XVW_COLOR_RED_MAPCOL, &rcol,
			XVW_COLOR_GREEN_MAPCOL, &gcol,
			XVW_COLOR_BLUE_MAPCOL, &bcol,
			NULL);

	   ksprintf(temp, "M%d", w-1);
	   if (rcol >= w)
	   {
	      kfree(color->red_function);
	      color->red_function = kstrdup(temp);
	   }
	   if (gcol >= w)
	   {
	      kfree(color->green_function);
	      color->green_function = kstrdup(temp);
	   }
	   if (bcol >= w)
	   {
	      kfree(color->blue_function);
	      color->blue_function = kstrdup(temp);
	   }

	   /*
	    *  Color Reload to get the data
	    */
	   ColorReload(widget, 0, -1);
	   kpds_add_map_callback(object, KPDS_CALLBACK_CHANGE,
		ColorReloadCallback, (kaddr) widget);
	}
	else
	{
	   kpds_get_attribute(object, XVW_COLOR_INFO, &color->info);
	   if (color->info->widget == widget)
	   {
	      color->allocated = TRUE;
	      ColorReload(widget, 0, -1);
	      kpds_add_map_callback(object, KPDS_CALLBACK_CHANGE,
			ColorReloadCallback, (kaddr) widget);
	   }
	   else
	   {
	      color->allocated = FALSE;

	      /*
	       *  Add a callback on the widget that allocated the color info
	       *  structure.  This callback is used to inform us when the color
	       *  info structure has changed.
	       */
	      xvw_insert_callback(xvw_object(color->info->widget),
			XVW_COLOR_CALLBACK, FALSE, ColorUtilCallback,
			(XtPointer) widget);
	   }
	   color->info->references++;
	}


	/*
	 *  Check to see if an existing "red" function exists for the object.
	 */
	if (!kdms_query_attribute(object, NULL, XVW_COLOR_RED_FUNCTION, NULL,
				  NULL, NULL, NULL))
	{
	   kdms_create_attribute(object, NULL, XVW_COLOR_RED_FUNCTION, 1, 1,
				 KSTRING, FALSE, FALSE);
	   kpds_set_attribute(object, XVW_COLOR_RED_FUNCTION, NULL);
	   function = NULL;
	}
	else
	   kpds_get_attribute(object, XVW_COLOR_RED_FUNCTION, &function);

	if (function != NULL)
	{
	   kfree(color->red_function);
	   color->red_function = kstrdup(function);
	}

	/*
	 *  Check to see if an existing "green" function exists for the object.
	 */
	if (!kdms_query_attribute(object, NULL, XVW_COLOR_GREEN_FUNCTION, NULL,
				  NULL, NULL, NULL))
	{
	   kdms_create_attribute(object, NULL, XVW_COLOR_GREEN_FUNCTION, 1, 1,
				 KSTRING, FALSE, FALSE);
	   kpds_set_attribute(object, XVW_COLOR_GREEN_FUNCTION, NULL);
	   function = NULL;
	}
	else
	   kpds_get_attribute(object, XVW_COLOR_GREEN_FUNCTION, &function);

	if (function != NULL)
	{
	   kfree(color->green_function);
	   color->green_function = kstrdup(function);
	}

	/*
	 *  Check to see if an existing "blue" function exists for the object.
	 */
	if (!kdms_query_attribute(object, NULL, XVW_COLOR_BLUE_FUNCTION, NULL,
				  NULL, NULL, NULL))
	{
	   kdms_create_attribute(object, NULL, XVW_COLOR_BLUE_FUNCTION, 1, 1,
				 KSTRING, FALSE, FALSE);
	   kpds_set_attribute(object, XVW_COLOR_BLUE_FUNCTION, NULL);
	   function = NULL;
	}
	else
	   kpds_get_attribute(object, XVW_COLOR_BLUE_FUNCTION, &function);

	if (function != NULL)
	{
	   kfree(color->blue_function);
	   color->blue_function = kstrdup(function);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorReload
|
|       Purpose: Reload the xcolor structure from the color
|		 data object's colormap.
|
|         Input: widget - the widget to load the colormap for
|		 indx   - the index to actually start loading from
|		 length - the number of colors to load
|
|        Output: none
|
|    Written By: Mark Young
|          Date: Nov 21, 1992
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void ColorReload(
   Widget  widget,
   int     indx,
   int     length)
{
	XvwColorPart *color = GetColorPart(widget);

	kobject object = color->object;
	int i, available, ncolors, enable = KOPTIONAL, map_width, map_height;


	/*
	 *  Sanity check to make sure that the color object exists
	 */
	if (!object || !color->allocated || !color->info)
	   return;

	/*
	 *  Re-allocate the maps if they need to bigger or smaller...
	 */
	ncolors = color->info->ncolors;
	map_width  = color->info->map_width;
	map_height = color->info->map_height;
        available = kpds_query_map(object);

	/*
	 *  Ready to load the xcolor array, but need to get the map data
	 */
	if (available)
	{
	   /*
	    *  Get the map plane
	    */
	   kpds_set_attributes(object,
			KPDS_MAP_DATA_TYPE, KDOUBLE,
			KPDS_MAP_POSITION, 0, 0, 0, 0, 0,
			KPDS_MAP_REGION_SIZE, map_width, map_height, 1, 1, 1,
			NULL);

	   /*
	    *  Get the map plane
	    */
	   if ((color->info->maps = (double *) kpds_get_data(color->object,
			KPDS_MAP_REGION, color->info->maps)) == NULL)
	   {
	      kerror(XVISUAL, "ColorReload", "Unable to get maps. \
Initializing maps to be linear grey scale.");
	      available =  FALSE;
	   }
	}

	if (map_height != ncolors)
	{
	   if (available && enable == KFORCED)
	   {
	      color->info->mapping = (int *) krealloc(color->info->mapping,
				map_height * sizeof(int));
	   }
	   else
	      kfree(color->info->mapping);

	   color->info->xcolors = (XColor *) krealloc(color->info->xcolors,
				map_height * sizeof(XColor));
	   color->info->pixels  = (Pixel *) krealloc(color->info->pixels,
				map_height * sizeof(Pixel));
	   color->info->red   = (double *) krealloc(color->info->red,
				map_height * sizeof(double));
	   color->info->green = (double *) krealloc(color->info->green,
				map_height * sizeof(double));
	   color->info->blue  = (double *) krealloc(color->info->blue,
				map_height * sizeof(double));

	   for (i = ncolors; i < map_height; i++)
	   {
	      color->info->pixels[i] = (Pixel) -1;
	      color->info->xcolors[i].flags = 0;
	   }
	   color->info->ncolors = map_height;
	}


	/*
	 *  According to the RGB mapping functions, load the red, green, blue
	 *  map arrays.
	 */
	ApplyMapcolFunction(widget, RED_COLOR);
	ApplyMapcolFunction(widget, GREEN_COLOR);
	ApplyMapcolFunction(widget, BLUE_COLOR);
	if ((enable == KFORCED) && available)
	{
	   ColorSort(widget, color->info->maps, color->info->mapping,
		     map_width, ncolors);
	}
	ColorReloadXData(widget);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorReloadXData
|
|       Purpose: Reload the xcolor array according to the normalization
|		 scheme and RGB mapping functions.
|         Input: widget - color widget
|        Output: 
|       Returns: 
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Jun 22, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void ColorReloadXData(
  Widget widget)
{
	XColor *xcolors = ColorGetXColors(widget);
        Pixel  *pixels  = ColorGetPixels(widget);
	XvwColorPart *color = GetColorPart(widget);

	int    i;
	double *red, *green, *blue, rtmp, gtmp, btmp,
	       rscale, roffset, gscale, goffset, bscale, boffset;


	/*
	 *  Sanity check to make sure that the color object exists
	 */
	if (!xcolors || !pixels || !color || !color->allocated || !color->info)
	   return;

	/*
	 *  Get the stored information from the color information structure.
	 */
	red   = color->info->red;
	green = color->info->green;
	blue  = color->info->blue;

	/*
	 *  Compute the normalization scale and offset factors according to
	 *  our normalization policy.
	 */
	ComputeNormalization(widget, &rscale, &roffset, &gscale, &goffset,
			     &bscale, &boffset);

	for (i = 0; i < color->info->ncolors; i++)
	{
	   rtmp = (roffset + *red++)   * rscale;
	   gtmp = (goffset + *green++) * gscale;
	   btmp = (boffset + *blue++)  * bscale;
	   xcolors[i].red   = krange(0, rtmp, 65535);
	   xcolors[i].green = krange(0, gtmp, 65535);
	   xcolors[i].blue  = krange(0, btmp, 65535);

	   if (color->alloc_policy == KCOLOR_ALLOC_READWRITE && 
	       xcolors[i].flags != 0 && xcolors[i].flags != -1)
	   {
	      XStoreColor(XtDisplay(widget),widget->core.colormap, &xcolors[i]);
	   }
	   else if ((xcolors[i].flags == 0 || xcolors[i].flags == -1) &&
		    widget->core.depth == 24)
	   {
	      if (pixels[i] == xcolors[i].pixel) pixels[i] = (Pixel) -1;
	      xcolors[i].flags = -1;
	      xcolors[i].pixel =
		  ((xcolors[i].red >> 8)    & color->rmask) << color->rshift |
                  ((xcolors[i].green >> 8) & color->gmask) << color->gshift |
                  ((xcolors[i].blue >> 8)  & color->bmask) << color->bshift;
	      if (pixels[i] == (Pixel) -1) pixels[i] = xcolors[i].pixel;
	   }
	   kinfo(KDEBUG, "%3d: (%3d %3d %3d)", i,
			xcolors[i].red >> 8,
			xcolors[i].green >> 8,
			xcolors[i].blue >> 8);
	}

	/*
	 * if color->id is not 0, the timeout has already been added.
	 */
	if (color->id == 0 && color->alloc_policy == KCOLOR_ALLOC_READONLY)
	{
           color->id = XtAppAddTimeOut(XtWidgetToApplicationContext(widget), 0,
                                        ColorTimeout, (XtPointer) widget);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: GetColorPart
|
|       Purpose: Gets the XvwColorPart for either a color object
|		 or color widget (xvwColorWidgetClass or
|		 xvwColorGadgetClass)
|
|         Input: widget - the widget to the color part for
|
|        Output: none
|       Returns: returns the color part, or NULL upon failure
|
|    Written By: Mark Young
|          Date: Nov 21, 1992
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
XvwColorPart *GetColorPart(
	Widget widget)
{
	if (widget == NULL)
	   return(NULL);
	else if (XtIsSubclass(widget, xvwColorWidgetClass))
	   return(&((XvwColorWidget) widget)->color);
	else if (XtIsSubclass(widget, xvwColorGadgetClass))
	   return(&((XvwColorGadget) widget)->color);

	return(NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorGetPixels
|
|       Purpose: Gets the Pixel array assocaited with the color info
|		 structure.  This routine is meant as a convience routine
|		 since it will work for a ColorWidget or ColorGadget, and
|		 also pass back NULL if the info structure has not been
|		 initialized.
|
|         Input: widget - the widget to the color part for
|        Output: none
|       Returns: returns the pixel array, or NULL upon failure
|
|    Written By: Mark Young
|          Date: Jun 11, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
Pixel *ColorGetPixels(
	Widget widget)
{
	XvwColorPart *color = GetColorPart(widget);

	if (color == NULL || color->info == NULL)
	   return(NULL);

	return(color->info->pixels);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorGetXColors
|
|       Purpose: Gets the XColors array assocaited with the color info
|		 structure.  This routine is meant as a convience routine
|		 since it will work for a ColorWidget or ColorGadget, and
|		 also pass back NULL if the info structure has not been
|		 initialized.
|
|         Input: widget - the widget to the color part for
|        Output: none
|       Returns: returns the xcolor array, or NULL upon failure
|
|    Written By: Mark Young
|          Date: Jun 11, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
XColor *ColorGetXColors(
	Widget widget)
{
	XvwColorPart *color = GetColorPart(widget);

	if (color == NULL || color->info == NULL)
	   return(NULL);

	return(color->info->xcolors);
}

/*-----------------------------------------------------------
|
|  Routine Name: UnsignedHistogram - compute an unsigned histogram
|
|       Purpose: 
|         Input: 
|        Output: 
|       Returns:
|
|    Written By: Mark Young
|          Date: May 06, 1994
| Modifications:
|
------------------------------------------------------------*/
#define UnsignedHistogram(data, histogram, num_pts, maxval)		\
{									\
	for (j = 0; j < num_pts; j++)					\
	   maxval = kmax((long) data[j], maxval);			\
									\
	if (maxval < MaxMapLength) {					\
	   for (j = 0; j < num_pts; j++)				\
	      histogram[(int) data[j]]++;				\
	}								\
}

/*-----------------------------------------------------------
|
|  Routine Name: SignedHistogram - compute a signed histogram
|
|       Purpose: 
|         Input: 
|        Output: 
|       Returns:
|
|    Written By: Mark Young
|          Date: May 06, 1994
| Modifications:
|
------------------------------------------------------------*/
#define SignedHistogram(data, histogram, num_pts, minval, maxval)	\
{									\
	for (j = 0; j < num_pts; j++) {					\
	   maxval = kmax((long) data[j], maxval);			\
	   minval = kmin((long) data[j], minval);			\
	}								\
	if (maxval - minval < MaxMapLength) {				\
	   if (minval >= 0) {						\
	      for (j = 0; j < num_pts; j++)				\
	         histogram[data[j]]++;					\
	   } else {							\
	      for (j = 0; j < num_pts; j++)				\
	         histogram[data[j] - minval]++;				\
	   }								\
	}								\
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorGetHistogram
|
|       Purpose: Creates a histogram for the image if there was not one
|		 provided.  This is done by counting the occurances
|                of each pixel value in the image data, and entering
|		 that number at the location indexed by the pixel number.
|         Input: color - the color widget part
|        Output: 
|       Returns: 
|
|    Written By: Mark Young
|          Date: Aug 12, 1994
| Modifications:
|
------------------------------------------------------------*/

/* ARGSUSED */
unsigned long *ColorGetHistogram(
   Widget widget)
{
	XvwColorPart *color = GetColorPart(widget);

	double scale = 1.0;
	long minval = 0, maxval = 0;
	kobject object = color->object;
	int height = 0, i, j, w, h, d, t, e, type, num_rgns, num_pts;

        kaddr data = NULL;
        unsigned long histogram[MaxMapLength];
 

	if (color->info == NULL)
	   return(NULL);
	else if (color->info->histogram != NULL)
	   return(color->info->histogram);

	if (kpds_query_value(object) == TRUE)
	{
           kpds_get_attributes(object,
		KPDS_VALUE_DATA_TYPE, &type,
		KPDS_VALUE_OPTIMAL_REGION_SIZE, &w,&h,&d,&t,&e, &num_rgns,
		NULL);
	   num_pts = w*h*d*t*e;

	   kmemset(histogram, 0, MaxMapLength*sizeof(unsigned long));
	   if (type == KFLOAT || type == KDOUBLE ||
	       type == KCOMPLEX || type == KDCOMPLEX)
	   {
	      type = KUBYTE;
              kpds_set_attributes(object,
                   KPDS_VALUE_SCALING, KNORMALIZE,
                   KPDS_VALUE_NORM_MIN, 0.0,
                   KPDS_VALUE_NORM_MAX, 255.0,
		   KPDS_VALUE_COMPLEX_CONVERT, KLOGMAGP1,
		   KPDS_VALUE_DATA_TYPE, type,
                   NULL);
	   }
	   else if (type == KBIT)
	   {
	      height = 2;
	      minval = num_pts = num_rgns = 0;
	      maxval = histogram[0] = histogram[1] = 1;
	   }
	}
	else
	{
	   num_pts = num_rgns = 0;
	   maxval = color->info->ncolors;
	   for (i = 0; i < maxval && i < MaxMapLength; i++)
	      histogram[i] = 1;

	   height = ((maxval - minval) + 1) * scale;
	}

	if (num_pts > 0 && num_rgns > 0)
	{
           kpds_set_attributes(object,
		    KPDS_VALUE_REGION_SIZE, w, h, d, t, e,
		    KPDS_VALUE_POSITION, 0, 0, 0, 0, 0,
		    NULL);
	   for (i = 0; i < num_rgns; i++)
	   {
              data = kpds_get_data(object, KPDS_VALUE_REGION, data);
	      switch (type)
	      {
	         case KUSHORT:
		   {
	              register unsigned short *usdata = (unsigned short *) data;
		      UnsignedHistogram(usdata, histogram, num_pts, maxval);
	           }
		   break;

	         case KUINT:
		   {
	              register unsigned int *uidata = (unsigned int *) data;
		      UnsignedHistogram(uidata, histogram, num_pts, maxval);
		   }
	           break;

	         case KULONG:
		   {
	              register unsigned long *uldata = (unsigned long *) data;
		      UnsignedHistogram(uldata, histogram, num_pts, maxval);
		   }
	           break;

	         case KUBYTE:
		   {
	              register unsigned char *ubdata = (unsigned char *) data;
		      UnsignedHistogram(ubdata, histogram, num_pts, maxval);
		   }
	           break;

	         case KSHORT:
		   {
	              register short *sdata = (short *) data;
		      SignedHistogram(sdata, histogram, num_pts, minval,maxval);
		   }
	           break;

	         case KINT:
		   {
	              register int *idata = (int *) data;
		      SignedHistogram(idata, histogram, num_pts, minval,maxval);
		   }
	           break;

	         case KLONG:
		   {
	              register long *ldata = (long *) data;
		      SignedHistogram(ldata, histogram, num_pts, minval,maxval);
		   }
	           break;

	         case KBYTE:
		   {
	              register char *bdata = (char *) data;
		      SignedHistogram(bdata, histogram, num_pts, minval,maxval);
		   }
	           break;
	      }
           }
	   kfree(data);
	   height = ((maxval - minval) + 1) * scale;
	}
	color->info->minval = minval;
	color->info->maxval = maxval;
	color->info->scale  = scale;
	height = kmax(height, color->info->ncolors);
        if (height <= MaxMapLength)
	   color->info->histogram = kdupalloc(histogram, height * sizeof(long));
	else color->info->histogram = NULL;

	return(color->info->histogram);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorAllocate
|
|       Purpose: Allocate the color...
|
|         Input: widget - the color widget to allocate the color for
|		 indx   - the corresponding index in the xcolor array
|
|        Output: none
|       Returns: none
|
|    Written By: Mark Young
|          Date: Nov 21, 1992
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
Pixel ColorAllocate(
	Widget       widget,
	register int indx)
{
	XvwColorPart *color = GetColorPart(widget);

	int      screen;
	XColor   *xcolors;
	Display  *display;
	Colormap colormap;
	Pixel    plane, *pixels;
	register int red, green, blue;
	int      i, diff, temp, pixel = 0;


	/*
	 *  Check to see that we have an allocated xcolor array to work from
	 */
	if (!color->info || indx >= color->info->ncolors || indx < 0 ||
	    (color = GetColorPart(color->info->widget)) == NULL)
	{
	   kinfo(KDEBUG, "Bad  index %d", indx);
	   return(-1);
	}

	/*
	 *  Get the real color information structure.
	 */
	widget   = color->info->widget;
	xcolors  = ColorGetXColors(widget);
	pixels   = ColorGetPixels(widget);
	colormap = widget->core.colormap;
	display  = XtDisplay(widget);
        screen   = XScreenNumberOfScreen(XtScreen(widget));

	/*
	 *  First make sure that color hasn't already been allocated
	 */
	if (xcolors[indx].flags != 0)
	   return(pixels[indx]);

	/*
	 *  Go ahead and set the flags to use the red, green, blue component
	 *  of the pixel allocation.
	 */
	if (pixels[indx] == xcolors[indx].pixel) pixels[indx] = (Pixel) -1;
	xcolors[indx].flags = DoRed | DoGreen | DoBlue;
	kinfo(KDEBUG, "Allocating index %d", indx);

	/*
	 *  Allocate the pixel.  If we succeed then return, otherwise if we
	 *  fail then we need to perform a closest search of the available
	 *  pixels.
	 */
	if (color->alloc_policy == KCOLOR_ALLOC_READONLY)
	{
	   if (XAllocColor(display, colormap, &xcolors[indx]))
	   {
	      kinfo(KDEBUG, "for index %d found pixel %d (readonly)", indx,
			xcolors[indx].pixel);
	      xcolors[indx].pixel;
	      if (pixels[indx] == (Pixel) -1)
		 pixels[indx] = xcolors[indx].pixel;
	      return(pixels[indx]);
	   }
	}
	else
	{
	   if (XAllocColorCells(display, colormap, TRUE, &plane, 0,
			&xcolors[indx].pixel, 1))
	   {
	      kinfo(KDEBUG, "for index %d found pixel %d (readwrite)", indx,
			xcolors[indx].pixel);
	      XStoreColor(display, colormap, &xcolors[indx]);
	      if (pixels[indx] == (Pixel) -1)
		 pixels[indx] = xcolors[indx].pixel;
	      return(pixels[indx]);
	   }
	}

	if (color->private_cmap == TRUE &&
	    colormap == DefaultColormap(display, screen))
	{
	   colormap = XCopyColormapAndFree(display, colormap);
	   xvw_set_attribute(xvw_object(widget), XVW_COLORMAP, colormap);
	   return(ColorAllocate(widget, indx));
	}
	xcolors[indx].flags = 0;

	for (diff = -1, i = 0; i < color->info->ncolors; i++)
	{
	   if (!xcolors[i].flags) 
	      continue;

	   red   = kabs(((int) xcolors[i].red)   - ((int) xcolors[indx].red));
	   green = kabs(((int) xcolors[i].green) - ((int) xcolors[indx].green));
	   blue  = kabs(((int) xcolors[i].blue)  - ((int) xcolors[indx].blue));

	   temp  = ksqr(red >> 8) + ksqr(green >> 8) + ksqr(blue >> 8);
	   if (diff == -1 || temp < diff)
	   {
	      pixel = xcolors[i].pixel;
	      diff  = temp;
	   }
	}
	kinfo(KDEBUG, "for index %d (NEAREST) to pixel %d\n\n", indx, pixel);
	xcolors[indx].flags = -1;
	xcolors[indx].pixel = pixel;
	if (pixels[indx] == (Pixel) -1) pixels[indx] = pixel;
	return(pixel);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorFree
|
|       Purpose: Free the xcolors associated with a color widget
|		 or object.
|
|         Input: widget - the color widget/object to free the colors for
|		 indx   - the index to start freeing from
|		 length - the number of colors to actually free
|
|    Written By: Mark Young
|          Date: Nov 21, 1992
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void ColorFree(
   Widget widget,
   int    indx,
   int    length)
{
	XvwColorPart *color = GetColorPart(widget);

	Pixel    pixels[256];
	Display  *display  = XtDisplay(widget);
	Colormap colormap = widget->core.colormap;

	register int i, num, ncolors;
	register XColor *xcolors = ColorGetXColors(widget);


	/*
	 *  Sanity check...  Should be an xcolor array, but if not
	 *  then just return.
	 */
	if (!xcolors || color->allocated == FALSE)
	   return;

	if (length == -1)
	   length = color->info->ncolors;

	ncolors = kmin(indx+length, color->info->ncolors);
	for (i = indx, num = 0; i < ncolors; i++)
	{
	   if (xcolors[i].flags != 0 && xcolors[i].flags != -1)
	   {
	      pixels[num++] = xcolors[i].pixel;
	      if (num == 256)
	      {
		 XFreeColors(display, colormap, pixels, num, 0L);
		 num = 0;
	      }
	   }
	   xcolors[i].flags = 0;
	}
	if (num > 0) XFreeColors(display, colormap, pixels, num, 0L);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorStore
|
|       Purpose: Given an index and a color it frees the currently stored
|		 color and loads the new red, green, blue colors from the
|		 supplied arrays.
|
|         Input: widget - the color widget/object to free the colors for
|		 indx   - the index to store
|		 xcolor - the new xcolor values
|        Output:
|       Returns:
|    Written By: Mark Young
|          Date: Jun 24, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void ColorStore(
   Widget widget,
   int    indx,
   XColor *xcolor)
{
	XvwColorPart *color = GetColorPart(widget);
	XColor *xcolors;

        Display  *display;
        Colormap colormap;


	/*
	 *  if the current xcolor is NULL then just return.  Also, we need
	 *  to get the color class associated with the object that "allocated"
	 *  it.  This means that for the routines that Store, Free, or
	 *  Allocate pixels, the only object that can do this is the one
	 *  that allocated the info structure in the first place.  Hence
	 *  why we re-acquire the color structure associated with
	 *  "color->info->widget" as done below.
	 */
	if (!color->info || (color = GetColorPart(color->info->widget)) == NULL)
	   return;

	/*
	 *  Get the real color information structure.
	 */
	widget   = color->info->widget;
	xcolors  = ColorGetXColors(widget);
	colormap = widget->core.colormap;
	display  = XtDisplay(widget);

	/*
	 *  Assign the new color if it's not NULL
	 */
	if (xcolor != NULL)
	{
	   xcolors[indx].red   = xcolor->red; 
	   xcolors[indx].green = xcolor->green; 
	   xcolors[indx].blue  = xcolor->blue; 
	}

	/*
	 *  Free the current color if currently allocated
	 */
	if (indx >= 0 && indx < color->info->ncolors)
	{
	   if (xcolors[indx].flags != 0 && xcolors[indx].flags != -1)
	   {
	      if (color->alloc_policy == KCOLOR_ALLOC_READWRITE)
	         XStoreColor(display, colormap, &xcolors[indx]);
	      else
	      {
	         XFreeColors(display, colormap, &xcolors[indx].pixel, 1, 0L);
	         xcolors[indx].flags = 0;
	      }
	   }
	   else
	      xcolors[indx].flags = 0;
	}

	/*
	 * if color->id is not 0, the timeout has already been added.
	 */
	if (color->id == 0 && color->alloc_policy == KCOLOR_ALLOC_READONLY)
	{
           color->id = XtAppAddTimeOut(XtWidgetToApplicationContext(widget), 0,
                                        ColorTimeout, (XtPointer) widget);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorSort - routine for sorting color mapping array
|
|       Purpose: This routine sort the color mapping array
|
|         Input: 
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: 
|          Date: 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void ColorSort(
	Widget   widget,
	double   *maps,
	int      *mapping,
	int      w,
	int      h)
{
	int diff, tmpdiff;
	int i, j, ro, ri, go, gi, bo, bi, indx = 0, *used;


	if (!mapping)
	   return;

	ro = go = bo = 0;
	used = (int *) kcalloc(h, sizeof(int));
        for (i = 0; i < h; i++)
	{
           for (j = 0, diff = KMAXLINT; j < h; j++)
	   {
	      if (used[j] == TRUE)
		 continue;

	      ri = maps[3*j]; gi = maps[3*j + 1]; bi = maps[3*j + 2];
	      tmpdiff = ksqr(ro - ri) + ksqr(go - gi) +  ksqr(bo - bi);
	      kinfo(KDEBUG, "%d: diff %d tmpdiff %d\n", j, diff, tmpdiff);
	      if (tmpdiff < diff)
	      {
		 indx = j;
		 diff = tmpdiff;
	      }
	   }
	   used[indx] = TRUE;
	   mapping[i] = indx;
	   ro = maps[3*indx]; go = maps[3*indx + 1]; bo = maps[3*indx + 2];
	   kinfo(KDEBUG, "%d: diff %d\n", i, diff);
	}

	for (i = 0; i < h; i++)
	   kinfo(KDEBUG, "%d: %d %d\n", i, used[i], mapping[i]);

	kfree(used);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorSetUtil
|
|       Purpose: ColorSetUtil is used to set miscellaneous color attributes.
|		 Including:
|				XVW_COLOR_RELOAD
|				XVW_COLOR_RED_MAPCOL
|				XVW_COLOR_GREEN_MAPCOL
|				XVW_COLOR_BLUE_MAPCOL
|
|         Input: object    - the object in which we want to do the action for
|		 attribute - the 
|		 calldata  - the data associated with the action
|
|        Output:
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Jun 24, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int ColorSetUtil(
	xvobject colorobj,
	char     *attribute,
	kaddr    calldata)
{
	Widget	     widget = (Widget) xvw_widget(colorobj);
	XvwColorPart *color = GetColorPart(widget);

	kobject  object = NULL;

	if (kstrcmp(attribute, XVW_COLOR_RELOAD) == 0)
	{
	   /*
	    *  Sanity check to make sure that the color object exists
	    */
	   object = color->object;
	   if (!object || !kpds_query_map(object) ||
	       !color->allocated || !color->info)
	   {
	      return(TRUE);
	   }
	   kprintf("Hey.  I'm a lonely ColorReload that needs to be \
re-implemented\n");
	}
	else if (kstrcmp(attribute, XVW_COLOR_RED_MAPCOL) == 0 ||
		 kstrcmp(attribute, XVW_COLOR_GREEN_MAPCOL) == 0 ||
		 kstrcmp(attribute, XVW_COLOR_BLUE_MAPCOL) == 0)
	{
	   int *mapcol = (int *) calldata;
	   char *function_attribute, temp[KLENGTH];


	   if (*mapcol < 0 || *mapcol >= color->info->map_width)
	   {
	      kerror(XVISUAL, "ColorSetUtil", "Map Column '%d' provided for \
%s is invalid.  The valid range for map columns is %d to %d", *mapcol,
			attribute, 0, color->info->map_width-1);
	      return(FALSE);
	   }

	   if (kstrcmp(attribute, XVW_COLOR_RED_MAPCOL) == 0)
	      function_attribute = XVW_COLOR_RED_FUNCTION;
	   else if (kstrcmp(attribute, XVW_COLOR_GREEN_MAPCOL) == 0)
	      function_attribute = XVW_COLOR_GREEN_FUNCTION;
	   else
	      function_attribute = XVW_COLOR_BLUE_FUNCTION;

	   ksprintf(temp,"M%d", *mapcol);
	   xvw_set_attribute(colorobj, function_attribute, temp);
	}
	else if (kstrcmp(attribute, XVW_COLOR_COLORFILE) == 0)
	{
	    int	locate = TRUE;
	    String  *filename = (String *) calldata;

	    if (*filename != NULL &&
		(object = kdms_locate(*filename)) == NULL)
	    {
	       locate = FALSE;
	       if ((object = kpds_open_object(*filename, KOBJ_READ)) == NULL)
	       {
	          kerror(XVISUAL, "ColorSetUtil", "Failed to load color \
object '%s'", *filename);
	          return(TRUE);
	       }
	    }
	    xvw_set_attribute(colorobj, XVW_COLOR_COLOROBJ, object);
	    if (!locate) kpds_close_object(object);
	}
	else if (kstrcmp(attribute, XVW_COLOR_SAVEMAP) == 0)
	{
	    unsigned char *maps;
	    Pixel    *pixels = color->info->pixels;
	    String   *filename = (String *) calldata;
	    int      i, height = color->info->ncolors;
            XColor   temp, *xcolors = ColorGetXColors(widget);


	    if (!(object = kpds_open_object(*filename, KOBJ_WRITE)))
	    {
	       kerror(XVISUAL, "ColorSetUtil", "Failed to open color \
object '%s'", *filename);
	       return(TRUE);
	    }
	    kpds_create_map(object);
	    maps = (unsigned char *) kmalloc(3 * height);
	    kpds_set_attribute(object, KPDS_MAP_SIZE, 3, height, 1, 1, 1);
	    kpds_set_attribute(object, KPDS_MAP_DATA_TYPE, KUBYTE);

	    temp.flags = DoRed|DoGreen|DoBlue;
	    for (i = 0; i < height; i++)
	    {
	       if (pixels[i] == xcolors[i].pixel || pixels[i] == (Pixel) -1)
	       {
	          maps[i*3]     = xcolors[i].red >> 8;
	          maps[i*3 + 1] = xcolors[i].green >> 8;
	          maps[i*3 + 2] = xcolors[i].blue >> 8;
	       }
	       else
	       {
		  temp.pixel = pixels[i];
		  XQueryColor(XtDisplay(widget), widget->core.colormap, &temp);
	          maps[i*3]     = temp.red >> 8;
	          maps[i*3 + 1] = temp.green >> 8;
	          maps[i*3 + 2] = temp.blue >> 8;
	       }
	    }
	    kpds_put_data(object, KPDS_MAP_PLANE, maps);
	    kfree(maps);
	    kpds_close_object(object);
	}
	else
	{
	   kerror(XVISUAL, "ColorSetUtil", "invalid attribute %s", attribute);
	   return(FALSE);
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorGetUtil
|
|       Purpose: ColorGetUtil is used to get miscellaneous color attributes.
|		 Including:
|				XVW_COLOR_RED_MAPCOL
|				XVW_COLOR_GREEN_MAPCOL
|				XVW_COLOR_BLUE_MAPCOL
|
|         Input: object    - the object in which we want to get the value for
|		 attribute - one of the attributes mentioned above 
|        Output: calldata  - the data in which we return the value
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Jun 24, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int ColorGetUtil(
	xvobject colorobj,
	char     *attribute,
	kaddr    calldata)
{
	Widget	     widget = (Widget) xvw_widget(colorobj);
	XvwColorPart *color = GetColorPart(widget);

	if (kstrcmp(attribute, XVW_COLOR_RED_MAPCOL) == 0   ||
	    kstrcmp(attribute, XVW_COLOR_GREEN_MAPCOL) == 0 ||
	    kstrcmp(attribute, XVW_COLOR_BLUE_MAPCOL) == 0)
	{
	   int  *mapcol = (int *) calldata;
	   char *function, temp[KLENGTH];
	   int  indx = -1;

	   if (kstrcmp(attribute, XVW_COLOR_RED_MAPCOL) == 0)
	      function = color->red_function;
	   else if (kstrcmp(attribute, XVW_COLOR_GREEN_MAPCOL) == 0)
	      function = color->green_function;
	   else
	      function = color->blue_function;

	   /*
	    *  If function string contained more than "Mx" that implies a
	    *  function rather than a simple map column.
	    */
	   temp[0] = '\0';
	   ksscanf(function, "M%d%s", &indx, temp); 

	   if (kstrlen(temp) > 0) 
		*mapcol = -1;
	   else *mapcol = indx;
	}
	else if (kstrcmp(attribute, XVW_COLOR_COLORFILE) == 0)
	{
	   String *filename = (String *) calldata;

	   *filename = NULL;
	   kpds_get_attribute(color->object, KPDS_NAME, filename);
	}
	else if (kstrcmp(attribute, XVW_COLOR_COLORS) == 0)
	{
	   XColor **xcolors = (XColor **) calldata;

	   *xcolors = (!color->info) ? NULL : color->info->xcolors;
	}
	else if (kstrcmp(attribute, XVW_COLOR_NUMCOLORS) == 0)
	{
	   int *ncolors = (int *) calldata;

	   *ncolors = (!color->info) ? 0 : color->info->ncolors;
	}
	else
	{
	   kerror(XVISUAL, "ColorGetUtil", "invalid attribute %s", attribute);
	   return(FALSE);
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorChangeMapCol
|
|       Purpose: ColorChangeMapCol is an interactive routine, called in response
|		 to setting of attribute:
|				XVW_COLOR_CHANGE_RED_MAPCOL
|				XVW_COLOR_CHANGE_GREEN_MAPCOL
|				XVW_COLOR_CHANGE_BLUE_MAPCOL
|
|		 This routine pops up a list of the available map columns,
|		 and allows the user to choose one for application to the
|		 map column in question.  It then just calls xvw_set_attribute
|		 on the corresponding red,green,blue mapcol attribute.
|
|         Input: object    - the object in which we want to do the action for
|		 attribute - the 
|		 calldata  - the data associated with the action
|
|        Output:
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Jun 24, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int ColorChangeMapCol(
	xvobject colorobj,
	char     *attribute,
	kaddr    calldata)
{
	Widget	     widget = (Widget) xvw_widget(colorobj);
	XvwColorPart *color = GetColorPart(widget);

        char **list;
	char *mapcol_attribute;
	char *prompt;
        int  i, indx = -1;
        char temp[KLENGTH];


	if (kstrcmp(attribute, XVW_COLOR_CHANGE_RED_MAPCOL) == 0 ||
	    kstrcmp(attribute, XVW_COLOR_CHANGE_GREEN_MAPCOL) == 0 ||
	    kstrcmp(attribute, XVW_COLOR_CHANGE_BLUE_MAPCOL) == 0)
	{
	   if (kstrcmp(attribute, XVW_COLOR_CHANGE_RED_MAPCOL) == 0)
	   {
	      mapcol_attribute = XVW_COLOR_RED_MAPCOL;
	      prompt = "Select map column to use as RED band";
	   }
	   else if (kstrcmp(attribute, XVW_COLOR_CHANGE_GREEN_MAPCOL) == 0)
	   {
	      mapcol_attribute = XVW_COLOR_GREEN_MAPCOL;
	      prompt = "Select map column to use as GREEN band";
	   }
	   else
	   {
	      mapcol_attribute = XVW_COLOR_BLUE_MAPCOL;
	      prompt = "Select map column to use as BLUE band";
	   }
 
	   /*
	    *  Create the list of map column selections
	    */
	   list = (char **) kcalloc(1, color->info->map_width * sizeof(char *));
           for (i = 0; i < color->info->map_width; i++)
           {
               ksprintf(temp, "Map Column %d (desc)", i);
               list[i] = kstring_copy(temp, NULL);
           }

	   /*
	    *  Prompt the user for which one they want.
	    */
	   if (kchoose(KFORCE, list, color->info->map_width, 0, NULL,
			&indx, prompt) == NULL)
	      indx = -1;
	   else indx--;

	   karray_free(list, color->info->map_width, NULL);

	   /*
	    *  If the indx is not -1 then change the
	    */
	   if (indx != -1)
	      xvw_set_attribute(colorobj, mapcol_attribute, indx);
	}
	else
	{
	   kerror(XVISUAL, "ColorChangeMapCol", "invalid attribute %s",
			attribute);
	   return(FALSE);
	}
	return(TRUE);
}
