/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                3D Get Utilities
   >>>>
   >>>>  Private:
   >>>>  	       X3D_get_projection()
   >>>>  	       X3D_get_viewpoint()
   >>>>  	       X3D_get_camera()
   >>>>  	       X3D_get_focus()
   >>>>  	       X3D_get_view_distance()
   >>>>  	       X3D_get_window()
   >>>>  	       X3D_get_viewport()
   >>>>  	       X3D_get_wc_min_max()
   >>>>  	       X3D_get_win_type()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"


/*-----------------------------------------------------------
|
|  Routine Name: X3D_get_projection
|
|       Purpose: Inquire the 3D projection type for a given X3D
|		 graphics id.  Types of projection:
|			PERSPECTIVE     1
|			ORTHOGRAPHIC    2
|			CAVALIER        3
|			CABINET         4
|
|         Input: id         - X3D graphics structure ID
|
|        Output: projection - Type of projection
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X3D_get_projection (
   int id,
   int *projection)
{
        X3DGraphics *graphics;

        if ((graphics = _X3D_get_graphics(id)) == NULL)
        {
           (void) kfprintf (kstderr,"X3D_get_projection:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
           return(FALSE);
        }

	*projection = graphics->projection;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_get_viewpoint
|
|       Purpose: Inquire the eye orientation for a given X3D graphics
|		 structure.
|
|         Input: id	  - X3D graphics structure ID
|
|        Output: alpha	  - the angle describing the position of the
|			     eye between the Z axis and the XY plane
|		 theta	  - Angle describing the position of the eye
|			     around Z axis with respect to the XY plane
|		 gammaval - Angle describing the rotation of the eye
|			     to tilt the view
|		 eye_dist - Distance of the eye from the object
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X3D_get_viewpoint (
   int  id,
   Real *alpha,
   Real *theta,
   Real *gammaval,
   Real *eye_distance)
{
	X3DGraphics *graphics;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_get_viewpoint:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}
	*eye_distance = graphics->eye_dist;
	*alpha = graphics->alpha;
	*theta = graphics->theta;
	*gammaval = graphics->gamma;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_get_focus
|
|       Purpose: Inquire the point in which the eye is focusing on.
|
|         Input: id    - X3D graphics structure ID
|
|        Output: focus - the point in which the user is focusing
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X3D_get_focus (
   int   id,
   Coord *focus)
{
	X3DGraphics *graphics;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_get_focus:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}
	*focus = graphics->focus;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_get_camera
|
|       Purpose: Inquire the point in which the eye is currently at
|
|         Input: id	- X3D graphics structure ID
|
|        Output: camera	- the camera or eye position
|		 up	- the up rotation
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X3D_get_camera (
   int   id,
   Coord *camera,
   Coord *up)
{
	X3DGraphics *graphics;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_camera:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}
	*up	= graphics->up;
	*camera = graphics->camera;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_get_view_distance
|
|       Purpose: Inquire the eye orientation for a given X3D graphics
|		 structure.
|
|         Input: id	   - X3D graphics structure ID
|
|        Output: view_dist - Distance of the eye from the viewport
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X3D_get_view_distance (
   int  id,
   Real *view_distance)
{
        X3DGraphics *graphics;

        if ((graphics = _X3D_get_graphics(id)) == NULL)
        {
           (void) kfprintf (kstderr,"X3D_get_view_distance:");
           (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
           return(FALSE);
        }
        *view_distance = graphics->view_dist;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_get_window
|
|       Purpose: Inquires the minimum and maximum device coordinate
|		 for a given X11 Window.
|
|         Input: id        - X3D graphics structure ID
|
|        Output: x	- x offset in device coordinates
|		 y	- y offset in device coordinates
|		 width	- width of the window in device coordinates
|		 height	- height of the window in device coordinates
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X3D_get_window (
   int   id,
   int   *x,
   int   *y,
   unsigned int *width,
   unsigned int *height)
{
	X3DGraphics 	*graphics;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_get_window:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}

	if (graphics->device == X11)
	{
	   *x = graphics->X11_xmin;
	   *y = graphics->X11_ymin;
	   *width  = graphics->X11_xmax - *x;
	   *height = graphics->X11_ymax - *y;
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_get_viewport
|
|       Purpose: Inquires the minimum and maximum viewport values
		 for a given X3D graphics structure.
|
|         Input: id     - X3D graphics structure ID
|
|        Output: viewmin - the viewport minimum coordinate value.
|		 viewmax - the viewport maximum coordinate value.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X3D_get_viewport (
   int   id,
   Coord *viewmin,
   Coord *viewmax)
{
	X3DGraphics *graphics;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_get_viewport:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}
	viewmin->x = graphics->xv_min;
	viewmin->y = graphics->yv_min;
	viewmin->z = graphics->zv_min;
	viewmax->x = graphics->xv_max;
	viewmax->y = graphics->yv_max;
	viewmax->z = graphics->zv_max;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_get_wc_min_max
|
|       Purpose: Inquires the minimum and maximum world coordinate
|		 points for a given X3D graphics structure.
|
|         Input: id     - X3D graphics structure ID
|
|        Output: wc_min - A coord containing the minimum world
|			  coordinate values
|		 wc_max - A coord containing the maximum world
|			  coordinate values
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X3D_get_wc_min_max (
   int   id,
   Coord *wc_min,
   Coord *wc_max)
{
	X3DGraphics *graphics;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_get_wc_min_max:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}

	*wc_min = graphics->wc_min;
	*wc_max = graphics->wc_max;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_get_win_type
|
|       Purpose: Inquire the 3D window type for a given
|		 graphics id.  Types of windows:
|			2D	1
|			3D	2
|
|         Input: id	  - X3D graphics structure ID
|
|        Output: win_type - the type of graphics window
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X3D_get_win_type (
   int id,
   int *win_type)
{
        X3DGraphics *graphics;

        if ((graphics = _X3D_get_graphics(id)) == NULL)
        {
           (void) kfprintf (kstderr,"X3D_get_win_type:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
           return(FALSE);
        }

	*win_type = graphics->win_type;
	return(TRUE);
}
