 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            3D Clipping Routines
   >>>>
   >>>>  Private:
   >>>>			_X3D_clip_points()	      	
   >>>>			_X3D_clip_segments()   	 
   >>>>			_X3D_clip_lines() 
   >>>>			_X3D_clip_polygon() 
   >>>>			_X3D_clip_rectangle() 
   >>>>   Static:
   >>>>			static clip_point_3D()
   >>>>			static clip_segment_3D()
   >>>>			static compute_outcode_3D()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"

/*-----------------------------------------------------------
|
|  Routine Name: clip_point_3D
|
|       Purpose: clips a point by making sure that it lies within
|                world coordinate viewing system.  If the point is
|                to the left of wcmin.x then x is set to wcmin.x.
|                The process is repeated for points that are to
|                the right, bottom, & top. 
|
|         Input: graphics - the graphics structure
|                outcode  - the code that defines the points location
|                           in terms of the clipping region
|                xin      - the x point to be clipped
|                yin      - the y point to be clipped
|                zin      - the z point to be clipped
|
|        Output: point - the clipped point
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void clip_point_3D(
   X3DGraphics    *graphics,
   int            outcode,
   register  Real xin,
   register  Real yin,
   register  Real zin,
   Coord          *point)
{
	if (outcode & X3D_LEFT)
	   point->x = graphics->wc_min.x;
	else if (outcode & X3D_RIGHT)
	   point->x = graphics->wc_max.x;
	else
	   point->x = xin;

	if (outcode & X3D_FRONT)
	   point->z = graphics->wc_min.z;
	else if (outcode & X3D_BACK)
	   point->z = graphics->wc_max.z;
	else
	   point->z = zin;

	if (outcode & X3D_TOP)
	   point->y = graphics->wc_max.y;
	else if (outcode & X3D_BOTTOM)
	   point->y = graphics->wc_min.y;
	else
	   point->y = zin;
}

/*-----------------------------------------------------------
|
|  Routine Name: clip_segment_3D
|
|       Purpose: computes the outcodes (Cohen-Sutherland style).
|                The outcode will indicate if the point is to the
|                left, right, top, bttom, or in the center of our
|                world coordinate view.
|
|         Input: graphics   - the graphics structure
|                outcode1   - the code that defines the points location
|                             in terms of the clipping region for (px1,py1,pz1)
|                outcode2   - the code that defines the points location
|                             in terms of the clipping region for (px2,py2,pz2)
|                (px1,py1,pz1) - the begin point of the segment to be
|                             clipped
|                (px2,py2,pz2) - the end point of the segment to be
|                             clipped
|
|        Output: point1, point2 - the resulting clipped output line
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void clip_segment_3D(
   X3DGraphics   *graphics,
   int           outcode1,
   int           outcode2,
   register Real px1,
   register Real py1,
   register Real pz1,
   register Real px2,
   register Real py2,
   register Real pz2,
   Coord         *point1,
   Coord         *point2)
{
	
	if (outcode1 & X3D_LEFT)
	{
	   pz1 = kintercept(pz1, pz2, px1, px2, graphics->wc_min.x);
	   py1 = kintercept(py1, py2, px1, px2, graphics->wc_min.x);
	   px1 = graphics->wc_min.x;
	}
	else if (outcode2 & X3D_LEFT)
	{
	   pz2 = kintercept(pz1, pz2, px1, px2, graphics->wc_min.x);
	   py2 = kintercept(py1, py2, px1, px2, graphics->wc_min.x);
	   px2 = graphics->wc_min.x;
	}

	if (outcode1 & X3D_RIGHT)
	{
	   pz1 = kintercept(pz1, pz2, px1, px2, graphics->wc_max.x);
	   py1 = kintercept(py1, py2, px1, px2, graphics->wc_max.x);
	   px1 = graphics->wc_max.x;
	}
	else if (outcode2 & X3D_RIGHT)
	{
	   pz2 = kintercept(pz1, pz2, px1, px2, graphics->wc_max.x);
	   py2 = kintercept(py1, py2, px1, px2, graphics->wc_max.x);
	   px2 = graphics->wc_max.x;
	}

	if (outcode1 & X3D_BOTTOM)
	{
	   pz1 = kintercept(pz1, pz2, py1, py2, graphics->wc_min.y);
	   px1 = kintercept(px1, px2, py1, py2, graphics->wc_min.y);
	   py1 = graphics->wc_min.y;
	}
	else if (outcode2 & X3D_BOTTOM)
	{
	   pz2 = kintercept(pz1, pz2, py1, py2, graphics->wc_min.y);
	   px2 = kintercept(px1, px2, py1, py2, graphics->wc_min.y);
	   py2 = graphics->wc_min.y;
	}

	if (outcode1 & X3D_TOP)
	{
	   pz1 = kintercept(pz1, pz2, py1, py2, graphics->wc_max.y);
	   px1 = kintercept(px1, px2, py1, py2, graphics->wc_max.y);
	   py1 = graphics->wc_max.y;
	}
	else if (outcode2 & X3D_TOP)
	{
	   pz2 = kintercept(pz1, pz2, py1, py2, graphics->wc_max.y);
	   px2 = kintercept(px1, px2, py1, py2, graphics->wc_max.y);
	   py2 = graphics->wc_max.y;
	}

	if (outcode1 & X3D_FRONT)
	{
	   py1 = kintercept(py1, py2, pz1, pz2, graphics->wc_min.z);
	   px1 = kintercept(px1, px2, pz1, pz2, graphics->wc_min.z);
	   pz1 = graphics->wc_min.z;
	}
	else if (outcode2 & X3D_FRONT)
	{
	   py2 = kintercept(py1, py2, pz1, pz2, graphics->wc_min.z);
	   px2 = kintercept(px1, px2, pz1, pz2, graphics->wc_min.z);
	   pz2 = graphics->wc_min.z;
	}

	if (outcode1 & X3D_BACK)
	{
	   py1 = kintercept(py1, py2, pz1, pz2, graphics->wc_max.z);
	   px1 = kintercept(px1, px2, pz1, pz2, graphics->wc_max.z);
	   pz1 = graphics->wc_max.z;
	}
	else if (outcode2 & X3D_BACK)
	{
	   py2 = kintercept(py1, py2, pz1, pz2, graphics->wc_max.z);
	   px2 = kintercept(px1, px2, pz1, pz2, graphics->wc_max.z);
	   pz2 = graphics->wc_max.z;
	}
	point1->x = px1; point1->y = py1; point1->z = pz1;
	point2->x = px2; point2->y = py2; point2->z = pz2;
}

/*-----------------------------------------------------------
|
|  Routine Name: compute_outcode_3D - determine the outcode for a (x,y,z)
|                                     point
|
|       Purpose: Computes the outcodes (Cohen-Sutherland style).
|                The outcode will indicate if the point is to the
|                left, right, top, bottom, or in the center of our
|                world coordinate view.
|
|         Input: graphics - the graphics structure
|                x   - the x position
|                y   - the y position
|                z   - the z position
|
|        Output: none
|
|       Returns: the outcode for the (x,y,z) point
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static int compute_outcode_3D(
   X3DGraphics    *graphics,
   register  Real x,
   register  Real y,
   register  Real z)
{
	register int outcode;


	if (graphics->clipping == FALSE)
	   return(X3D_CENTER);

	if (x < graphics->wc_min.x)
	   outcode = X3D_LEFT;
	else if (x > graphics->wc_max.x)
	   outcode = X3D_RIGHT;
	else
	   outcode = X3D_CENTER;

	if (y < graphics->wc_min.y)
	   outcode |= X3D_BOTTOM;
	else if (y > graphics->wc_max.y && graphics->depth_clipping == TRUE)
	   outcode |= X3D_TOP;

	if (z < graphics->wc_min.z)
	   outcode |= X3D_FRONT;
	else if (z > graphics->wc_max.z && graphics->depth_clipping == TRUE)
	   outcode |= X3D_BACK;

	return(outcode);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_clip_points - Clips the world coordinate points
|
|       Purpose: Clips the world coordinate points and returns
|                the clipped points that fall within the view
|                of the world coordinate viewing system.
|
|         Input: graphics - the graphics structure
|                coords   - array of world coordinates input
|                size     - number of points to clipped
|
|        Output: ocoords  - points that were clipped
|		 outsize  - the number of clipped points
|
|       Returns: a pointer to the clipped points
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Coord  *_X3D_clip_points(
   X3DGraphics *graphics,
   Coord       *coords,
   int         size,
   int         clip_code,
   Coord       *ocoords,
   int         *outsize)
{
	register Real x, y, z, w;

	Coord    *points;
	int	i, j, outcode;

/*
	if (graphics->clipping == FALSE)
	{
	   *outsize = size;
	   return(coords);
	}
 */

	if (ocoords != NULL)
	   points = ocoords;
	else if ((points = (Coord *) kmalloc((unsigned) size * sizeof(Coord)))							== NULL)
	{
	   kfprintf (kstderr,"_X3D_clip_points:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return(NULL);
	}

	for (i = 0, j = 0; i < size; i++)
	{
	   _X3D_vector_mult(coords[i], graphics->matrix1, x, y, z, w);
	   outcode = compute_outcode_3D(graphics, x, y, z);
	   if (outcode == X3D_CENTER)
	   {
	      points[j].x = x; points[j].y = y; points[j].z = z;
	      points[j].d = coords[i].d;
	      j++;
	   }
	   else if ((outcode & ~clip_code) == X3D_CENTER)
	   {
	      clip_point_3D(graphics, outcode, x, y, z, &points[j]);
	      points[j++].d = coords[i].d;
	   }
	}

	if (j == 0)
	{
	   if (ocoords != points)
	      kfree(points);
	   *outsize = 0;
	   return(NULL);
	}
	*outsize = j;
	return(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_clip_segments - Clips the world coordinate segments
|
|       Purpose: Clips the world coordinate segments and returns
|                the clipped segments that fall within the view
|                of the world coordinate viewing system.
|
|         Input: graphics - the graphics structure
|                coords   - array of world coordinates input
|                size     - number of points to clipped
|
|        Output: ocoords  - points that were clipped
|		 outsize  - the number of clipped points
|
|       Returns: points   - the clipped points
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Coord  *_X3D_clip_segments(
   X3DGraphics *graphics,
   Coord       *coords,
   int         size,
   Coord       *ocoords,
   int         *outsize)
{
	Coord   *points;
	int	i, j, outcode1, outcode2;
	register Real px1, py1, pz1, px2, py2, pz2, w;

/*
	if (graphics->clipping == FALSE)
	{
	   *outsize = size;
	   return(coords);
	}
 */

	if (ocoords != NULL)
	   points = ocoords;
	else if ((points = (Coord *) kmalloc((unsigned) size*2*sizeof(Coord)))
								== NULL)
	{
	   kfprintf (kstderr,"_X3D_clip_points:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return(NULL);
	}

	for (i = 0, j = 0; i+1 < size; i += 2)
	{
	   _X3D_vector_mult(coords[i], graphics->matrix1, px1, py1, pz1, w);
	   _X3D_vector_mult(coords[i+1], graphics->matrix1, px2, py2, pz2, w);
	   outcode1 = compute_outcode_3D(graphics, px1, py1, pz1);
	   outcode2 = compute_outcode_3D(graphics, px2, py2, pz2);
	   if ((outcode1 | outcode2) == X3D_CENTER)
	   {
	      points[j].x   = px1; points[j].y   = py1; points[j].z   = pz1;
	      points[j+1].x = px2; points[j+1].y = py2; points[j+1].z = pz2;
	      points[j].d   = coords[i-1].d;
	      points[j+1].d = coords[i].d;
	      j += 2;
	   }
	   else if ((outcode1 & outcode2) == X3D_CENTER)
	   {
	      clip_segment_3D(graphics, outcode1, outcode2, px1, py1, pz1, px2, py2,
			pz2, &points[j], &points[j+1]);
	      points[j].d = coords[i-1].d;
	      points[j+1].d = coords[i].d;
	      j += 2;
	   }
	}

	if (j == 0)
	{
	   if (ocoords != points)
	      kfree(points);
	   *outsize = 0;
	   return(NULL);
	}
	*outsize = j;
	return(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_clip_lines - Clip the world coordinate polyline
|
|       Purpose: Clips the world coordinate polyline such that
|                the resulting segments or set of polylines lie within
|                the world coordinate viewing system.
|
|         Input: graphics - the graphics structure
|                coords   - array that defines the polyline
|                size     - number of points in the polyline to clipped
|
|        Output: ocoords  - points that were clipped
|		 outsize  - the number of clipped segment points
|
|       Returns: points   - the clipped segments
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Coord  *_X3D_clip_lines(
   X3DGraphics *graphics,
   Coord       *coords,
   int         size,
   Coord       *ocoords,
   int         *outsize)
{
	Coord    *points;
	int	 i, j, outcode1, outcode2;
	register Real px1, py1, pz1, px2, py2, pz2, w;

/*
	if (graphics->clipping == FALSE)
	{
	   *outsize = size;
	   return(coords);
	}
 */

	if (ocoords != NULL)
	   points = ocoords;
	else if ((points = (Coord *) kmalloc((unsigned) size*2*sizeof(Coord)))
					== NULL)
	{
	   kfprintf (kstderr,"_X3D_clip_lines:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return(NULL);
	}

	_X3D_vector_mult(coords[0], graphics->matrix1, px1, py1, pz1, w);
	outcode1 = compute_outcode_3D(graphics, px1, py1, pz1);
	for (i = 1, j = 0; i < size; i++)
	{
	   _X3D_vector_mult(coords[i], graphics->matrix1, px2, py2, pz2, w);
	   outcode2 = compute_outcode_3D(graphics, px2, py2, pz2);
	   if ((outcode1 | outcode2) == X3D_CENTER)
	   {
	      points[j].x   = px1; points[j].y   = py1; points[j].z   = pz1;
	      points[j+1].x = px2; points[j+1].y = py2; points[j+1].z = pz2;
	      points[j].d   = coords[i-1].d;
	      points[j+1].d = coords[i].d;
	      j += 2;
	   }
	   else if ((outcode1 & outcode2) == X3D_CENTER)
	   {
	      clip_segment_3D(graphics, outcode1, outcode2, px1, py1, pz1,
			px2, py2, pz2, &points[j], &points[j+1]);
	      points[j].d = coords[i-1].d;
	      points[j+1].d = coords[i].d;
	      j += 2;
	   }
	   px1 = px2; py1 = py2; pz1 = pz2;
	   outcode1 = outcode2;
	}

	if (j == 0)
	{
	   if (ocoords != points)
	      kfree(points);
	   *outsize = 0;
	   return(NULL);
	}
	*outsize = j;
	return(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_clip_polygon - Clips the world coordinate polygon
|
|       Purpose: Clips the world coordinate polygon such that
|                the resulting polygon or set of polylines lie within
|                the world coordinate viewing system.
|
|         Input: graphics - the graphics structure
|                coords   - array that defines the polygon 
|                size     - number points in the polygon to be clipped
|
|        Output: ocoords  - points that were clipped
|		 outsize  - the number of clipped polygon points
|
|       Returns: points   - the clipped polygon points
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Coord  *_X3D_clip_polygon(
   X3DGraphics *graphics,
   Coord       *coords,
   int         size,
   Coord       *ocoords,
   int         *outsize)
{
	Coord   *points, dummy;
	register Real px1, py1, pz1, px2, py2, pz2, w;
	int	i, j, outcode1, outcode2;

/*
	if (graphics->clipping == FALSE)
	{
	   *outsize = size;
	   return(coords);
	}
 */

	if (ocoords != NULL)
	   points = ocoords;
	else if ((points = (Coord *) kmalloc((unsigned) size*2*sizeof(Coord)))
						== NULL)
	{
	   kfprintf (kstderr,"_X3D_clip_lines:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return(NULL);
	}

	_X3D_vector_mult(coords[0], graphics->matrix1, px1, py1, pz1, w);
	outcode1 = compute_outcode_3D(graphics, px1, py1, pz1);

	j = 0;
	if (outcode1 == X3D_CENTER)
	{
	   points[j].x = px1; points[j].y = py1; points[j].z = pz1;
	   points[j++].d = coords[0].d;
	}

	for (i = 1; i < size; i++)
	{
	   _X3D_vector_mult(coords[i], graphics->matrix1, px2, py2, pz2, w);
	   outcode2 = compute_outcode_3D(graphics, px2, py2, pz2);
	   if ((outcode1 | outcode2) == X3D_CENTER)
	   {
	      points[j].x = px2; points[j].y = py2; points[j].z = pz2;
	      points[j++].d = coords[i].d;
	   }
	   else if (outcode1 == X3D_CENTER)
	   {
	      points[j].d = coords[i].d;
	      clip_segment_3D(graphics, outcode1, outcode2, px1, py1, pz1,
			px2, py2, pz2, &dummy, &points[j]);
	      j++;
	   }
	   else if ((outcode1 & outcode2) == X3D_CENTER)
	   {
	      points[j].d = coords[i-1].d;
	      points[j+1].d = coords[i].d;
	      clip_segment_3D(graphics, outcode1, outcode2, px1, py1, pz1,
			px2, py2, pz2, &points[j], &points[j+1]);
	      j += 2;
	   }
	   px1 = px2; py1 = py2; pz1 = pz2;
	   outcode1 = outcode2;
	}

	if (j == 0)
	{
	   if (ocoords != points)
	      kfree(points);
	   *outsize = 0;
	   return(NULL);
	}
	*outsize = j;
	return(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_clip_rectangles - Clips the world coordinate rectangle
|
|       Purpose: Clips the world coordinate rectangle such that
|                the resulting rectangle lies within the world
|                coordinate viewing system.
|
|         Input: graphics - the graphics structure
|                coords   - array that defines the rectangle
|                size     - number of points in the rectangle to clipped
|
|        Output: ocoords  - points that were clipped
|		 outsize  - the number of clipped rectangle points
|
|       Returns: points   - the clipped rectangle points
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Coord  *_X3D_clip_rectangles(
   X3DGraphics *graphics,
   Coord       *coords,
   int         size,
   Coord       *ocoords,
   int         *outsize)
{
	Coord   *points;
	int	i, j, outcode1, outcode2;

/*
	if (graphics->clipping == FALSE)
	{
	   *outsize = size;
	   return(coords);
	}
 */

	if (ocoords != NULL)
	   points = ocoords;
	else if ((points = (Coord *) kmalloc((unsigned) size*2*sizeof(Coord)))
								== NULL)
	{
	   kfprintf (kstderr,"_X3D_clip_lines:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return(NULL);
	}

	outcode1 = compute_outcode_3D(graphics, coords[0].x, coords[0].y,
			coords[0].z);

	j = 0;
	if (outcode1 == X3D_CENTER)
	   points[j++] = coords[0];

	for (i = 1; i < size; i++)
	{
	   outcode2 = compute_outcode_3D(graphics, coords[i].x, coords[i].y,
			coords[i].z);
	   if ((outcode1 | outcode2) == X3D_CENTER)
	   {
	      points[j++] = coords[i];
	   }
	   else if (outcode1 == X3D_CENTER)
	   {
	      clip_point_3D(graphics, outcode2, coords[i].x, coords[i].y,
			coords[i].z, &points[j]);
	      points[j++].d = coords[i].d;
	   }
	   else if ((outcode1 & outcode2) == X3D_CENTER)
	   {
	      clip_point_3D(graphics, outcode1, coords[i-1].x, coords[i-1].y,
			coords[i-1].z, &points[j]);
	      clip_point_3D(graphics, outcode2, coords[i].x, coords[i].y,
			coords[i].z, &points[j+1]);
	      points[j].d = coords[i-1].d;
	      points[j+1].d = coords[i].d;
	      j += 2;
	   }
	   outcode1 = outcode2;
	}

	if (j == 0)
	{
	   if (ocoords != points)
	      kfree(points);
	   *outsize = 0;
	   return(NULL);
	}
	*outsize = j;
	return(points);
}
