/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>         General GUI Element Creation Routines         <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>		xvf_create_formback()	                      <<<<
   >>>>         xvf_create_selback()	                      <<<<
   >>>>         xvf_create_menubutton()	                      <<<<
   >>>>         xvf_create_button()	                      <<<<
   >>>>         xvf_create_menu_button()                      <<<<
   >>>>         xvf_create_optional_button()	              <<<<
   >>>>         xvf_create_label()	                      <<<<
   >>>>         xvf_create_menu_label()                       <<<<
   >>>>         xvf_create_pseudo_label()                     <<<<
   >>>>         xvf_create_text()	                      <<<<
   >>>>         xvf_create_cr_pixmap()                        <<<<
   >>>>         xvf_create_scroll()                           <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"

static char *get_pixmap_filename PROTO((char *));

/*------------------------------------------------------------
|
|  Routine Name: xvf_create_formback()
|
|       Purpose: Creates a backplane for general use.
|
|	  Input: parent - parent of the backplane object
|		 x      - x location (in characters) of object
|		 y      - y location (in characters) of object
|		 name   - name to reference backplane object
|                mapped - TRUE if backplane is to be mapped, FALSE otherwise
|        Output: Returns the backpane object
|    Written By: Danielle Argiro
|          Date: Apr 08, 1992
| Modifications: 
|
-----------------------------------------------------------------*/   

xvobject xvf_create_formback(
   xvobject parent,
   double   x,
   double   y,
   double   minwidth,
   double   minheight,
   char    *name,
   int      mapped)
{
	xvobject back;

	back = xvw_create_manager(parent, name);
	xvw_set_attributes(back,
		XVW_CHAR_XSNAP,	      1.0,	      /* xchar snapto	    */
		XVW_CHAR_YSNAP,	      0.5,	      /* ychar snapto	    */
	        XVW_CHAR_WIDTH,       minwidth,
	        XVW_CHAR_HEIGHT,      minheight,
		XVW_CURSORNAME,      "left_ptr",
                XVW_MENUABLE,         TRUE,    
                XVW_SELECTABLE,       TRUE,    
		NULL);

	xvw_insert_callback(back, XVW_GEOMETRY_CALLBACK, FALSE,
			    xvf_inform_geometry, NULL);
	return(back);
}

/*------------------------------------------------------------
|
|  Routine Name: xvf_create_selback()
|
|       Purpose: Creates a backplane for a selection.
|
| 	  Input: parent - parent of the backplane object
|		 x      - x location (in characters) of object
|		 y      - y location (in characters) of object
|		 name   - name to reference backplane object
|
|        Output: Returns the backpane object
|    Written By: Danielle Argiro
|          Date: Apr 08, 1992
| Modifications:
|
------------------------------------------------------------------*/

xvobject xvf_create_selback(
   xvobject parent,
   double  x,
   double  y,
   double  width,
   double  height,
   char   *name,
   int    dist)
{
	xvobject back;
 
	back = xvw_create_manager(parent, name);
	xvw_set_attributes(back, 
                XVW_BORDER_WIDTH,     0,              /* no border          */
                XVW_CHAR_XPOS,        x,	      /* set x dist         */
                XVW_CHAR_YPOS,        y,              /* set y dist         */
		XVW_CHAR_XSNAP,	      1.0,	      /* xchar snapto	    */
		XVW_CHAR_YSNAP,	      0.5,	      /* ychar snapto	    */
		XVW_CHAR_WIDTH,       width,          /* width from UIS     */
		XVW_CHAR_HEIGHT,      height,         /* height from UIS    */
                XVW_RESIZABLE,        TRUE,           /* resizable          */
                XVW_CURSORNAME,      "left_ptr",      /* cursor             */
                XVW_DEF_VERT_DIST,    0,              /* dist bwn childs    */
                XVW_MENUABLE,         TRUE,           /* menuable           */
                XVW_SELECTABLE,       TRUE,           /* selectable         */
		XVW_BUFFER_DIST,      dist,	      /* buffer distance    */
                NULL);

	xvw_insert_callback(back, XVW_GEOMETRY_CALLBACK, FALSE,
			    xvf_inform_geometry, NULL);
	return(back);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvf_create_menubutton()
|
|       Purpose: Creates a menubutton object (with menu)
|
|	  Input: parent - parent of the backplane object
|                offset - object for menu to be right of (if applic).
|		 x      - x location (in characters) of object
|		 y      - y location (in characters) of object
|		 width  - width      (in characters) of object
|		 height - height     (in characters) of object
|		 label  - label to display on button object
|		 name   - name to reference backplane object
|        Output: Returns the button object
|    Written By: Danielle Argiro
|          Date: Apr 08, 1992
| Modifications: 
|
------------------------------------------------------------------*/

xvobject xvf_create_menubutton(
   xvobject parent,
   xvobject offset,
   double  x,
   double  y,
   double  width,
   double  height,
   char   *label,
   char   *name,
   char   **button_names,
   int    button_num,
   xvobject **menu_buttons)
{
	int      i;
	xvobject menu; 
	xvobject menubutton;

	/*
 	 *  first, create button which will bring up the menu.
	 *  using a copy of label is a memory leak, but must be
         *  left that way or menu button will not come up under Athena.
         *  set width, height, min height.
	 */
	if (kstrlen(label) > (int) width)
	    label[(int)width] = '\0';

	menubutton = xvw_create_menubutton(parent, name);
	xvw_set_attributes(menubutton, 
		XVW_LABEL,            label,       /* set label      */
		XVW_CHAR_WIDTH,       width,       /* set width      */
		XVW_CHAR_HEIGHT,      height,      /* set height     */
		XVW_CHAR_MIN_HEIGHT,  1.0,         /* minimum height */
		NULL);

	/* if offset was provided, move menubutton to right of offset */
	if (offset != NULL)
	     xvw_set_attribute(menubutton, XVW_RIGHT_OF, offset);

	/* otherwise, use (x,y) position */
	else xvw_set_attributes(menubutton, 
		XVW_CHAR_XPOS,      x,   	  /* set x dist */
		XVW_CHAR_YPOS,      y,            /* set y dist */
		NULL);


	/* don't need to alloc array of menu buttons if there aren't any yet */
	if (button_num == 0) 
	    return(menubutton);

	/* allocate array of menu buttons for use within the pulldown menu */
	*menu_buttons = (xvobject *) kcalloc(1, button_num*sizeof(xvobject));
	if (*menu_buttons == NULL)
	{
	    kerror("xvforms", "xvf_create_menubutton",
		   "Unable to allocate array of buttons for display on menu");
	    return(NULL);
	}
	menu = xvw_retrieve_menu(menubutton);
	for (i = 0; i < button_num; i++)
	    (*menu_buttons)[i] = xvf_create_menu_button(menu, button_names[i], 
						        button_names[i]);
	return(menubutton);
}
			         
/*-------------------------------------------------------------
|
|  Routine Name: xvf_create_button()
|
|       Purpose: Creates a square or rectangular button object
|
|	  Input: parent - parent of the backplane object
|		 x      - x location (in characters) of object
|		 y      - y location (in characters) of object
|		 width  - width      (in characters) of object
|		 height - height     (in characters) of object
|		 label  - label to display on button object
|		 name   - name to reference backplane object
|        Output: Returns the button object
|    Written By: Danielle Argiro
|          Date: Apr 08, 1992
| Modifications: 
|
------------------------------------------------------------------*/

xvobject xvf_create_button(
   xvobject parent,
   double  x,
   double  y,
   double  width,
   double  height,
   char   *label,
   char   *name)
{
	xvobject button;

	button = xvw_create_button(parent, name);
	xvw_set_attributes(button, 
		XVW_CHAR_MIN_HEIGHT,  1.0,      /* minimum height */
		XVW_CHAR_WIDTH,       width,    /* set width      */
		XVW_CHAR_HEIGHT,      height,   /* set height     */
		XVW_CHAR_XPOS,        x,        /* set x pos      */
		XVW_CHAR_YPOS,        y,        /* set y pos      */
		XVW_RESIZABLE,        TRUE,     /* variable size  */
                XVW_MENUABLE,         TRUE,     /* menuable       */
                XVW_SELECTABLE,       TRUE,     /* selectable     */
		NULL);

	/*
         * the label given may specify a pixmap. if so, set pixmap;
         * otherwise, just set the text of the label as given.
         */
        xvf_set_object_label_or_pixmap(button, XVF_BUTTON_OBJ, label);


	return(button);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvf_create_menu_button()
|
|       Purpose: Creates a button object for use on a menu.
|
|	  Input: parent - parent of the backplane object
|		 label  - label to display on button object
|		 name   - name to reference backplane object
|        Output: Returns the button object
|    Written By: Danielle Argiro
|          Date: Apr 08, 1992
| Modifications: 
|
------------------------------------------------------------------*/

xvobject xvf_create_menu_button(
   xvobject parent,
   char   *label,
   char   *name)
{
	xvobject button;

	button = xvw_create_button(parent, name);
	xvw_set_attributes(button,
		XVW_LABEL,	  label,
                XVW_BUTTON_SHAPE, KBUTTON_SHAPE_OVAL,
		NULL);
	return(button);
}


/*-------------------------------------------------------------
|
|  Routine Name: xvf_create_optional_button()
|
|       Purpose: Creates a little square button object 
|		 used for indicating if selections are optional.  
|
|	  Input: parent - parent of the backplane object
|        Output: Returns the button object
|    Written By: Danielle Argiro
|          Date: Apr 08, 1992
| Modifications: 
|
------------------------------------------------------------------*/

xvobject xvf_create_optional_button(
   xvobject parent)
{
	xvobject  opt_button;

	opt_button = xvw_create_button(parent, "opt_button");
	xvw_set_attributes(opt_button,
			   XVW_LABEL,            "",  /* blank label      */
			   XVW_CHAR_WIDTH,       2.0,  /* char width       */
                           XVW_CHAR_HEIGHT,      1.0,  /* char height      */
			   XVW_CHAR_MIN_WIDTH,   2.0,  /* minimum width    */
                           XVW_CHAR_MIN_HEIGHT,  1.0,  /* minimum height   */
			   XVW_CHAR_MAX_WIDTH,   2.0,  /* maximum width    */
                           XVW_CHAR_MAX_HEIGHT,  1.0,  /* maximum height   */
			   XVW_YPOSITION,	 0,
			   NULL);
	return(opt_button);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvf_create_boolean_button()
|
|       Purpose: Creates a round button object 
|
| 	  Input: parent - parent of the backplane object
|		 width  - width  (in characters) of object
|		 height - height (in characters) of object
|		 offset - object to left of boolean object
|		 title  - title of boolean object
|	         name   - name of boolean object
|        Output: Returns the button object
|    Written By: Danielle Argiro
|          Date: Apr 08, 1992
| Modifications: 
|
------------------------------------------------------------------*/

xvobject xvf_create_boolean_button(
   xvobject parent,
   double    width,
   double    height,
   xvobject offset,
   char     *title,
   char     *name)
{
	xvobject  boolean;

	if (kstrlen(title) > (int) width)
	    title[(int)width] = '\0';

	boolean = xvw_create_button(parent, name);
	xvw_set_attributes(boolean, 
		 XVW_BUTTON_SHAPE,     KBUTTON_SHAPE_OVAL, /* oval button  */
	         XVW_CHAR_WIDTH,       width,          /* set width        */
	         XVW_CHAR_HEIGHT,      height,         /* set height       */
		 XVW_CHAR_MIN_HEIGHT,  1.0,            /* minimum height   */
                 XVW_RESIZABLE,        TRUE,           /* variable size    */
	         XVW_RIGHT_OF,         offset,         /* set offset       */
                 XVW_TACK_EDGE,        KMANAGER_TACK_ALL,/* grow vert & horiz */
	         NULL);

	/*
         * the label given may specify a pixmap. if so, set pixmap;
         * otherwise, just set the text of the label as given.
         */
	xvf_set_object_label_or_pixmap(boolean, XVF_BUTTON_OBJ, title);

	xvw_set_attribute(boolean, XVW_MENUABLE,  TRUE);

	return(boolean);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvf_create_label()
|
|       Purpose: Creates a label object 
|
|	  Input: parent - parent of the backplane object
|		 x      - x offset (in characters) to label
|		 y      - y offset (in characters) to label
|		 offset - object to Left of label
|		 label  - label to display in label object
|		 name   - name to reference label object
|		 left_justify - flag TRUE if label is to be left justified
|				     FALSE if label is to be centered
|		 menuable - TRUE if user should be able to modify the 
|				label directly.
|        Output: Returns the label object
|    Written By: Danielle Argiro
|          Date: Apr 08, 1992
| Modifications: 
|
------------------------------------------------------------------*/

xvobject xvf_create_label(
   xvobject parent,
   double    x,
   double    y,
   xvobject offset,
   char     *label,
   char     *name,
   int      left_justify,
   int      selectable)
{
	xvobject  label_object;

	if (label == NULL)
	    label = kstrdup("");

	/* create the label object */
	label_object = xvw_create_label(parent, name);
	xvw_set_attributes(label_object, 
	         XVW_BORDER_WIDTH,     0,        /* no border        */
		 XVW_CHAR_MIN_HEIGHT,  1.0,      /* minimum height   */
                 XVW_RESIZABLE,        TRUE,     /* resizable        */
	         NULL);

	/* 
	 * the label given may specify a pixmap. if so, set pixmap;
	 * otherwise, just set the text of the label as given.
	 */
	xvf_set_object_label_or_pixmap(label_object, XVF_LABEL_OBJ, label);

	/* if geometry is specified, set the position of the label using it */
	if ((x != 0.0) || (y != 0.0))
	{
	    xvw_set_attributes(label_object, 
	         XVW_CHAR_XPOS,        x,        /* set x dist       */
	         XVW_CHAR_YPOS,        y,        /* set y dist       */
		 NULL);
	}

	/* if offset is specified, set the position of the label using it */
	else if (offset != NULL)
	    xvw_set_attribute(label_object, XVW_RIGHT_OF, offset);

	if (selectable)
	    xvw_set_attribute(label_object, XVW_SELECTABLE, TRUE);

	if (left_justify)
	    xvw_set_attribute(label_object, XVW_LABEL_JUSTIFY, 
	  		      KLABEL_JUSTIFY_LEFT);  
	return(label_object);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvf_create_menu_label()
|
|       Purpose: Creates a label object for use on a menu.
|
|	  Input: parent - parent of the backplane object
|		 label  - label to display in label object
|		 name   - name to reference label object
|        Output: Returns the label object
|    Written By: Danielle Argiro
|          Date: Apr 08, 1992
* Modifications: 
|
------------------------------------------------------------------*/

xvobject xvf_create_menu_label(
   xvobject parent,
   char   *label,
   char   *name)
{
	xvobject  menu_label;

	menu_label = xvw_create_label(parent, name);

	if (label == NULL) label = " ";
	xvw_set_attributes(menu_label, 
			   XVW_LABEL,         label, /* label     */
			   XVW_BORDER_WIDTH,  0,     /* no border */
			   NULL); 
	return(menu_label);
}


/*-------------------------------------------------------------
|
|  Routine Name: xvf_create_pseudo_label()
|
|       Purpose: Creates a button object of type that is used both
|
|                to label an KUIS_INPUTFILE or KUIS_OUTPUTFILE selection and
|                to bring up the browser.
|	  Input: parent - parent of the backplane object
|		 width  - width      (in characters) of label
|		 height - height     (in characters) of label
|		 offset - object to Left of label
|		 label  - label to display in label object
|		 left_justify - TRUE if label is to be left justified
|				 FALSE if label is to be centered
|        Output: Returns the label object
|    Written By: Danielle Argiro
|          Date: Apr 08, 1992
| Modifications: 
|
------------------------------------------------------------------*/

xvobject xvf_create_pseudo_label(
   xvobject parent,
   double    width,
   double    height,
   xvobject offset,
   char     *label)
{
	xvobject  pseudo_label;

	if (kstrlen(label) > (int) width)
	    label[(int)width] = '\0';

	pseudo_label = xvw_create_button(parent, "browser_button");
	xvw_set_attributes(pseudo_label, 
	         XVW_CHAR_WIDTH,      width,          /* set width      */
	         XVW_CHAR_HEIGHT,     height,         /* set height     */
		 NULL);

	/* if offset was provided, move pseudo label to right of offset */
	if (offset != NULL)
	     xvw_set_attribute(pseudo_label, XVW_RIGHT_OF, offset);

	/* otherwise, use (x,y) position of (0,0) */
	else xvw_set_attributes(pseudo_label, 
		XVW_XPOSITION,      0,   	  /* set x dist */
		XVW_YPOSITION,      0,            /* set y dist */
		NULL);
	   

	/* 
	 * the label given may specify a pixmap. if so, set pixmap;
	 * otherwise, just set the text of the label as given.
	 */
	xvf_set_object_label_or_pixmap(pseudo_label, XVF_BUTTON_OBJ, label);

	return(pseudo_label);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvf_create_text()
|
|       Purpose: Creates a text object for entering strings and values.
|
|	  Input: parent      - parent of the text object
|		 width       - width  (in characters) of text object
|		 height      - height (in characters) of text object
|		 offset      - object to Left of text object
|		 def_string  - default string to display in text object
|		 spec        - flag TRUE for special case if called by
|			       xvf_create_string_sel w/ geometry height > 1.
|			       will set special attributes to scroll, etc.
|        Output: none
|       Returns: The text object
|    Written By: Danielle Argiro
|          Date: Apr 08, 1992
| Modifications: 
|
------------------------------------------------------------------*/

xvobject xvf_create_text(
   xvobject       parent,
   double          width,
   double          height,
   xvobject       offset,
   char           *def_string,
   int            spec)
{
	xvobject  text_obj;

	text_obj =  xvw_create_text(parent, "ascii");

	if (height > 1.0)
	    xvw_set_attribute(text_obj, XVW_TEXT_MULTILINE, TRUE);
	else xvw_set_attribute(parent, XVW_CHAR_MAX_HEIGHT, 1.0);

	if (width < 2.0) width = 2.0;

	if (!kstrcmp(def_string," "))
		def_string = "";

	xvw_set_attributes(text_obj, 
		XVW_TEXT_TYPE,       KTEXT_TYPE_STRING,/* string contents    */
		XVW_TEXT_STRING,     def_string,  /* default string          */
		XVW_TEXT_INSERT_POS, kstrlen(def_string),/* cursor at end    */
		XVW_TEXT_EDIT_TYPE,  KTEXT_EDIT,         /* editable         */
		XVW_YPOSITION,       0,            /* y position              */
                XVW_CHAR_WIDTH,      width,       /* width                   */
/*		XVW_CHAR_HEIGHT,     height, */	  /* height                  */
		XVW_CHAR_MIN_WIDTH,  3.0,         /* minimum width           */
		XVW_CHAR_MIN_HEIGHT, 1.0,         /* minimum height          */
                XVW_RIGHT_OF,        offset,      /* offset                  */
                XVW_ABOVE,           NULL,        /* offset                  */
                XVW_BELOW,           NULL,        /* offset                  */
          	XVW_TACK_EDGE,	     KMANAGER_TACK_ALL,/* grow vert & horiz  */
                XVW_CURSORNAME,      "xterm",     /* cursor                  */
		XVW_SELECTABLE,       TRUE,
		NULL);

	if (spec)
	    xvw_set_attribute(text_obj, XVW_TEXT_SCROLL_VERT, 
			      KSCROLLBAR_IFNEEDED);

	return(text_obj);
}


/*-------------------------------------------------------------
|
|  Routine Name: xvf_create_cr_pixmap()
|
|       Purpose: Creates a small, square label object containing 
|                a pixmap that represents a carriage return.
|
|   	  Input: parent - parent of the backplane object
|		 y      - y location (in characters) of object
|		 offset - object to Left of pixmap object
|        Output: Returns the pixmap object
|    Written By: Danielle Argiro
|          Date: Apr 08, 1992
| Modifications: 
|
------------------------------------------------------------------*/

xvobject xvf_create_cr_pixmap(
   xvobject parent)
{
	xvobject    pixmap_object;

	pixmap_object = xvw_create_pixmap(parent, "LivePixmap");
	xvw_set_attributes(pixmap_object, 
                XVW_BORDER_WIDTH,      0,                /* no border       */
                XVW_LEFT_OF,           NULL,	         /* set offset      */
                XVW_BELOW,	       NULL,	         /* set offset      */
		NULL);
	return(pixmap_object);
}


/*-------------------------------------------------------------
|
|  Routine Name:  xvf_create_scroll()
|
|       Purpose: Creates a scroll bar object.
|
|	  Input: parent - parent of the backplane object
|		 offset - object to Left of scroll bar
|		 name   - name to reference scroll object by 
|
|        Output: Returns the pixmap object
|    Written By: Danielle Argiro
|          Date: Apr 08, 1992
| Modifications: 
|
------------------------------------------------------------------*/

xvobject  xvf_create_scroll(
   xvobject parent,
   xvobject offset,
   char     *name)
{
	xvobject  scroll;

	scroll = xvw_create_scrollbar(parent, name);
	xvw_set_attributes(scroll, 
		XVW_ORIENTATION,      KSCROLLBAR_ORIENT_HORIZ,   /* horiz    */
                XVW_CHAR_MIN_HEIGHT,  1.0,                /* min height      */
                XVW_CHAR_MAX_HEIGHT,  1.0,                /* min height      */
                XVW_YPOSITION,        0,	          /* set y pos       */
                XVW_RIGHT_OF,         offset,	          /* set offset      */
                XVW_TACK_EDGE,        KMANAGER_TACK_HORIZ,/* grow horiz.     */
		NULL);
	return(scroll);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvf_set_object_label_or_pixmap()
|
|       Purpose: In a UIS line, for the text appearing on a button 
|                or a label, the application may elect to use a
|                pixmap rather than text.  When this is the case,
|                the entry for the label must appear in the UIS line as:
|                "pixmap:{path to pixmap}"
|    
|                This routine sees if the label passed in indeed has
|                that construct, and if so, is a valid pixmap path specified.
|                If so, it sets the pixmap properly, according to whether
|                the object passed in is a button or a label.  Right now,
|                only buttons and labels are supported (and not those
|                appearing in lists or pulldown menus).
|
|         Input: object       - button or label xvobject on which to 
|                               set the label or the pixmap
|                object_type  - XVF_LABEL_OBJ  for label xvobject, 
|                               XVF_BUTTON_OBJ for button xvobject
|                label_string - for a simple label, just the text;
|                               for a pixmap, the words "pixmap:" followed
|                               by the file to the pixmap
|
|        Output: 
|       Returns: XVF_LABEL_OBJ if it changes a label, 
|                XVF_BUTTON_OBJ if it FALSE on failure
|    Written By: Danielle Argiro
|          Date: June 29, 1994
| Modifications: 
|
------------------------------------------------------------------*/

int xvf_set_object_label_or_pixmap(
    xvobject object,
    int      object_type,
    char     *label_string)
{
	char *pixmap_filename;

	/* 
	 * the label given may specify a pixmap. if so, set pixmap;
	 * otherwise, just set the text of the label as given.
	 */
	pixmap_filename = get_pixmap_filename(label_string);
	if (pixmap_filename == NULL)
	{
	    if (object_type == XVF_LABEL_OBJ)
	        xvw_set_attributes(object, 
		     	           XVW_LABEL_TYPE, KLABEL_TYPE_LABEL,
				   XVW_LABEL,  label_string,
				   NULL);
	    else if (object_type == XVF_BUTTON_OBJ)
	        xvw_set_attributes(object, 
		     	           XVW_BUTTON_TYPE, KBUTTON_TYPE_LABEL,
				   XVW_LABEL,  label_string,
				   NULL);
	    else xvw_set_attribute(object, XVW_LABEL,  label_string);
	    return(TRUE);
	}
	if (object_type == XVF_LABEL_OBJ)
	{
	    xvw_set_attributes(object,
			       XVW_PIXMAP_FILENAME, pixmap_filename,
		     	       XVW_LABEL_TYPE, KLABEL_TYPE_PIXMAP,
			       NULL);
	}
	else if (object_type == XVF_BUTTON_OBJ)
	{
	    xvw_set_attributes(object,
			       XVW_PIXMAP_FILENAME, pixmap_filename,
		     	       XVW_BUTTON_TYPE, KBUTTON_TYPE_PIXMAP,
			       NULL);
	}
	else xvw_set_attribute(object, XVW_LABEL,  label_string);

	return(TRUE);
}

    
    

static char *get_pixmap_filename(
    char *label_string)
{
	char filename[KLENGTH];
	
	if (kstrncmp(label_string, "pixmap:", 7) != 0)
	    return(NULL);

	if (ksscanf(label_string, "pixmap:%s", filename) == 1)
	{
	    if (kaccess(filename, R_OK) == 0)
	        return(kstrdup(filename));
	    else 
	    {
	        kerror("xvforms", "get_pixmap_filename",
		       "You have an entry, '%s', that is attempting to specify a pixmap, but the file does not exist", filename);
	        return(NULL);
	    }
	}
	else return(NULL);
}
