/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Image Display Utility Routines
   >>>>
   >>>>  Private:
   >>>>             xvf_init_menuform()
   >>>>   Static:                                            
   >>>>             xvf_menuform_cb()
   >>>>             xvf_update_item()
   >>>>
   >>>>             xvf_update_menuform()
   >>>>             xvf_update_menuform_selections()
   >>>>
   >>>>             xvf_create_menuform()
   >>>>             xvf_map_menuform()
   >>>>             xvf_unmap_menuform()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


static void xvf_update_item   PROTO((xvobject, kselection *));
static void xvf_menuform_cb   PROTO((kform *, ksubform *, kaddr));
static void xvf_update_menuform_selections PROTO((xvobject, kselection *));

static int xvf_update_menuform    PROTO((xvobject, kform *));
static int xvf_map_menuform       PROTO((xvobject, kform *));
static int xvf_unmap_menuform     PROTO((xvobject, kform *));

static kaddr xvf_create_menuform  PROTO((xvobject, String));


/*-----------------------------------------------------------
|
|  Routine Name: xvf_menuform_cb - menu callback routine
|	
|
|       Purpose: This routine is used to process the user's menuform
|		 changes.   Any changes in the menuform are noted
|		 and the object/gadget/GUI-item is updated.  Update is
|                accomplished by using the xvw_set_attributes() to modify
|                the attributes of the object/gadget/GUI-item.
|
|         Input: object - object to be used to build the menuform
|
|        Output: RETURNS: the menuform on success, NULL otherwise
|
|    Written By: Mark Young  
|          Date: Jul 03, 1992 10:15
| Modifications:
|
------------------------------------------------------------*/

static void xvf_menuform_cb(
   kform    *form,
   ksubform *subform,
   kaddr    client_data)
{
	xvobject object = (xvobject) client_data;

	kguide     *guide;

	/*
	 *  Make sure that the object is not NULL.  If so then we return
	 *  failure.
	 */
	if (form == NULL)
	   return;

	/*
	 *  Update the form from the attributes.  What this involves is
	 *  finding the selection from the attributes uis variables name.
	 */
	if (form->master != NULL)
	{
	   xvf_update_item(object, form->master->sel_list);
	   subform = form->master->subform_list;
	}
	else subform = form->subform;

	while (subform != NULL)
	{
	   if (subform->guidepane != NULL)
	   {
	      xvf_update_item(object, subform->guidepane->sel_list);
	      guide = subform->guidepane->guide_list;
	   }
	   else guide = subform->guide;

	   while (guide != NULL)
	   {
	      xvf_update_item(object, guide->pane->sel_list);
	      guide = guide->next;
	   }
	   subform = subform->next;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_update_item 
|
|       Purpose: This routine updates a particular object/gadget/GUI item
|                according to changes made to a single selection list
|                of the menuform.
|
|         Input: object - object to be used to build the menuform
|
|        Output: RETURNS: the menuform on success, NULL otherwise
|
|    Written By: Mark Young
|          Date: Jul 03, 1992 10:15
| Modifications:
|
------------------------------------------------------------*/

static void xvf_update_item(
   xvobject   object,
   kselection *selptr)
{
	Line_Info  line_info;
	int        toggle_type;
	char  	   *name, *stringval;
        char       temp[KLENGTH];
	kselection *selection;

	if (object == NULL) return;
	kvf_clear_line_info(&line_info);

	selection = selptr;
	while (selection != NULL)
	{
	   if (selection->group_next != NULL)
	   {
		xvf_update_item(object, selection->group_next);
	        selection = selection->next;
	        continue;
	   }

	   if (selection->modified == FALSE)
	   {
	       selection = selection->next;
	       continue;
	   }
	   if (!(kvf_gen_parse(selection->line, &line_info)))
           {
		kerror ("xvforms", "xvf_update_item",
			"Internal error! cannot parse '%s'\n",
			selection->line);
		continue;
	   }

	   name = line_info.variable;
	   if ((selection->back_group != NULL) && 
	       (selection->back_group->modified))
	   {
	        switch (line_info.typeflag)
		{
	      	    case KUIS_INPUTFILE:
	      	    case KUIS_OUTPUTFILE:
	      	    case KUIS_STDIN:
	      	    case KUIS_STDOUT:
	            case KUIS_STRING:
	            case KUIS_STRINGLIST:
	            case KUIS_INTEGER:
	            case KUIS_LOGICAL:
	            case KUIS_CYCLE:
	            case KUIS_FLAG:
	            case KUIS_LIST:
	            case KUIS_DISPLAYLIST:
	            case KUIS_TOGGLE:
	            case KUIS_FLOAT:
	            case KUIS_DOUBLE:
                         ksprintf(temp, "%s_sel", name);
                         xvw_set_attribute(object, temp, line_info.opt_sel);
			 kvf_free_line_info_strings(&line_info);
	                 kvf_gen_parse(selection->line, &line_info);
	                 name = line_info.variable;
		         selection->back_group->modified = FALSE;
		         break;
		}
	   }

	   if ((selection->type != KUIS_MUTEXCL) &&
               (selection->type != KUIS_MUTINCL) &&
	       (selection->type != KUIS_GROUP))
	      selection->modified = -1;

	   switch (selection->type)
	   {
	       case KUIS_INPUTFILE:
	       case KUIS_OUTPUTFILE:
	            xvw_set_attribute(object, name, line_info.filename);
		    break;

	      case KUIS_STRING:
		   xvw_set_attribute(object, name, line_info.string_val);
		   break;

	      case KUIS_STRINGLIST:
		   xvw_set_attribute(object, name, line_info.string_val);
		   break;

	      case KUIS_INTEGER:
		   xvw_set_attribute(object, name, line_info.int_val);
		   break;

	      case KUIS_LOGICAL:
		   xvw_set_attribute(object, name, line_info.logical_val);
		   break;

	      case KUIS_FLAG:
		   if (selection->back_group == NULL)
		      xvw_set_attribute(object, name, line_info.opt_sel);
		   break;

	      case KUIS_LIST:
	      case KUIS_CYCLE:
	      case KUIS_DISPLAYLIST:
		   xvw_set_attribute(object, name, line_info.list_val);
		   break;

	      case KUIS_TOGGLE:
		   xvf_get_attributes(selection->back_kformstruct,
				      XVF_TOGGLE_TYPE,  &toggle_type,
				      XVF_TOGGLE_VAL,   &stringval,
				      NULL);
		   if ((toggle_type == KUIS_FLAG) || 
		       (toggle_type == KUIS_LOGICAL))
		       xvw_set_attribute(object, name, line_info.toggle_val);
	 	   else if (toggle_type == KUIS_INTEGER)
			xvw_set_attribute(object, name, atoi(stringval));
	 	   else if (toggle_type == KUIS_FLOAT)
			xvw_set_attribute(object, name, (float) atof(stringval));
	 	   else if (toggle_type == KUIS_DOUBLE)
			xvw_set_attribute(object, name, atof(stringval));
		   else xvw_set_attribute(object, name, stringval);
		   break;

	      case KUIS_FLOAT:
		   xvw_set_attribute(object, name, line_info.float_val);
		   break;

	      case KUIS_DOUBLE:
		   xvw_set_attribute(object, name, line_info.double_val);
		   break;

	      case KUIS_QUIT:
                   if (xvw_activated_menu(object))
		       xvw_inactivate_menu(object);
		   break;

	      case KUIS_PANEACTION:
		   xvw_set_attribute(object, name, line_info.selected);
		   break;
	   }

	   if ((selection->type != KUIS_MUTEXCL) && 
	       (selection->type != KUIS_MUTINCL) &&
	       (selection->type != KUIS_GROUP))
	       selection->modified = FALSE;

	   selection = selection->next;
	   kvf_free_line_info_strings(&line_info);
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_update_menuform - update a menuform
|
|       Purpose: Updates the entire menuform with current attribute 
|                settings for the object/gadget/GUI-item that brought it up.
|
|         Input: object - object to update the user interface menuform
|                selptr - pointer to the header of the selection list
|
|        Output: none
|       Returns: none
|    Written By: Mark Young & Danielle Argiro 
|          Date: Dec 04, 1992
| Modifications:
|
------------------------------------------------------------*/

static int xvf_update_menuform(
   xvobject object,
   kform    *form)
{
	ksubform *subform;
	kguide   *guide;

	/*
	 *  Make sure that the object is not NULL.  If so then we return
	 *  failure.
	 */
	if (form == NULL)
	   return(FALSE);

	/*
	 *  Update the form from the attributes.  What this involves is
	 *  finding the selection from the attributes uis variables name.
	 */
	if (form->master != NULL)
	{
	   xvf_update_menuform_selections(object, form->master->sel_list);
	   subform = form->master->subform_list;
	}
	else subform = form->subform;

	while (subform != NULL)
	{
	   if (subform->guidepane != NULL)
	   {
	      xvf_update_menuform_selections(object, 
					     subform->guidepane->sel_list);
	      guide = subform->guidepane->guide_list;
	   }
	   else guide = subform->guide;

	   /*
	    * make sure pane has been created, then update selections 
	    * on menuform according to their default values 
            */
	   while (guide != NULL)
	   {
	      if (guide->pane->back == NULL)
		  xvf_create_pane(guide->pane, guide->selected);
	      xvf_update_menuform_selections(object, guide->pane->sel_list);
	      guide = guide->next;
	   }
	   subform = subform->next;
	}
	kvf_change_entry((char *) form, TRUE, xvf_menuform_cb, (kaddr) object);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_update_menuform_selections
|
|       Purpose: Updates a selection list on the menuform with the
|                current attribute settings for the object/gadget/GUI-item
|                that brought up the menuform on which the selection appears.
|
|         Input: object - object to update the user interface menuform
|                selptr - head of the selection list to be updated
|
|        Output: none
|       Returns: none
|    Written By: Mark Young & Danielle Argiro
|          Date: Dec 04, 1992 
| Modifications:
|
------------------------------------------------------------*/

static void xvf_update_menuform_selections(
   xvobject   object,
   kselection *selptr)
{
	int	     intval, toggle_type;
	float	     floatval;
	double	     doubleval;
	kform_struct kformstruct;
	Line_Info    line_info;
	char	     *name, *stringval, *toggle_value;
	char         *string = NULL; 
	char         temp[KLENGTH];
	kselection   *selection;

	/*
	 *  Update the menuform from the attributes.  What this involves is
	 *  finding the selection from the attributes uis variables name.
	 */
	kvf_clear_line_info(&line_info);
	selection = selptr;
	while (selection != NULL)
        {
	    kformstruct.type = KSELECTION;

	    if (selection->group_next != NULL)
	    {
	        xvf_update_menuform_selections(object, selection->group_next);
	        selection = selection->next;
	        continue;
	    }

	    if (selection->modified == TRUE)
	    {
	        selection = selection->next;
	        continue;
	    }

	    kvf_gen_parse(selection->line, &line_info);
	    name = line_info.variable;
        
	    kformstruct.Selptr = selection;
	    kformstruct.flag = kformstruct.Selptr->type;
	    kformstruct.callback_list = NULL;

	    if (selection->back_group != NULL)
	    {
	        switch (line_info.typeflag)
	        {
	    	case KUIS_INPUTFILE:
	      	case KUIS_OUTPUTFILE:
	        case KUIS_STRING:
	        case KUIS_STRINGLIST:
	        case KUIS_FLAG:
	        case KUIS_INTEGER:
	        case KUIS_LOGICAL:
	        case KUIS_CYCLE:
	        case KUIS_LIST:
	        case KUIS_DISPLAYLIST:
	        case KUIS_TOGGLE:
	        case KUIS_FLOAT:
	        case KUIS_DOUBLE:
                     ksprintf(temp, "%s_sel", name);
                     if (xvw_get_attribute(object, temp, &intval)  &&
                                        intval != line_info.opt_sel) 
		         xvf_set_attribute(&kformstruct, XVF_OPTSEL, intval);
		     break;
	        }
	    }
	    switch (kformstruct.flag)
	    {
	    case KUIS_INPUTFILE:
	    case KUIS_OUTPUTFILE:
		 if (xvw_get_attribute(object, name, &string) &&
		     kstrcmp(string, line_info.filename) != 0)
		     xvf_set_attribute(&kformstruct, XVF_FILE_NAME, string);
		 break;

	    case KUIS_STRING:
	    case KUIS_STRINGLIST:
		 if (xvw_get_attribute(object, name, &string) &&
		     kstrcmp(string, line_info.string_val) != 0)
		     xvf_set_attribute(&kformstruct, XVF_STRING_VAL, string);
		 break;

	    case KUIS_INTEGER:
		 if (xvw_get_attribute(object, name, &intval) &&
		     intval != line_info.int_val)
		     xvf_set_attribute(&kformstruct, XVF_INT_VAL, intval);
		 break;

	    case KUIS_FLOAT:
		 if (xvw_get_attribute(object, name, &floatval) &&
		     floatval != line_info.float_val)
		     xvf_set_attribute(&kformstruct, XVF_FLOAT_VAL, floatval);
		 break;

	    case KUIS_DOUBLE:
		 if (xvw_get_attribute(object, name, &doubleval) &&
		     doubleval != line_info.double_val)
		     xvf_set_attribute(&kformstruct, XVF_DOUBLE_VAL, doubleval);
		 break;

	    case KUIS_LOGICAL:
		 if (xvw_get_attribute(object, name, &intval) &&
		     intval != line_info.logical_val)
		     xvf_set_attribute(&kformstruct, XVF_LOGIC_VAL, intval);
		 break;

	    case KUIS_FLAG:
		 if (selection->back_group == NULL)
		 {
		     if (xvw_get_attribute(object, name, &intval) &&
			 intval != line_info.opt_sel)
		         xvf_set_attribute(&kformstruct, XVF_OPTSEL, intval);
	         }
		 break;

	    case KUIS_CYCLE:
		 if (xvw_get_attribute(object, name, &intval) &&
		    intval != line_info.list_val)
		     xvf_set_attribute(&kformstruct, XVF_CYCLE_VAL, intval);
		 break;
     
	    case KUIS_LIST:
	    case KUIS_DISPLAYLIST:
		 if (xvw_get_attribute(object, name, &intval) &&
		     intval != line_info.list_val)
		     xvf_set_attribute(&kformstruct, XVF_LIST_VAL, intval);
		 break;

	    case KUIS_TOGGLE:
		 xvf_get_attributes(selection->back_kformstruct,
				    XVF_TOGGLE_TYPE, &toggle_type,
				    XVF_TOGGLE_VAL,  &toggle_value,
				    NULL);
		 if ((toggle_type == KUIS_FLAG) ||
		     (toggle_type == KUIS_LOGICAL))
		 {
		     if (xvw_get_attribute(object, name, &intval) &&
		         intval != line_info.toggle_val)
		         xvf_set_attribute(&kformstruct, XVF_TOGGLE_NUM, intval);
		     break;
		 }
		 else if (toggle_type == KUIS_INTEGER)
		 {
		     if (!xvw_get_attribute(object, name, &intval)) break;
		     ksprintf(temp, "%d", intval);
		 }
		 else if (toggle_type == KUIS_FLOAT)
                 {
                     if (!xvw_get_attribute(object, name, &floatval)) break;
		     ksprintf(temp, "%f", floatval);
                 }
		 else if (toggle_type == KUIS_DOUBLE)
                 {
                     if (!xvw_get_attribute(object, name, &doubleval)) break;
		     ksprintf(temp, "%g", doubleval);
                 }
		 else 
		 {
                     if (!xvw_get_attribute(object, name, &stringval)) break;
		     ksprintf(temp, "%s", stringval);
		 }
		 if (kstrcmp(temp, toggle_value) != 0)
		     xvf_set_attribute(&kformstruct, XVF_TOGGLE_VAL, temp); 
		 break;
	    }
	    selection->modified = FALSE;
	    selection = selection->next;
	    kvf_free_line_info_strings(&line_info);
        }
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_menuform - build a menuform
|
|
|       Purpose: Returns an kform structure that contains the
|                user interface menuform to accesss the objects
|                attributes.
|
|         Input: object - object to be used to build the menuform
|                menuform - the menuform filename to be built
|
|        Output: RETURNS: the menuform on success, NULL otherwise
|
|    Written By: Mark Young
|          Date: Jul 03, 1992 10:15
| Modifications:
|
------------------------------------------------------------*/

static kaddr xvf_create_menuform(
   xvobject object,
   String   menuform)
{
        kform *form = NULL;


        /*
         *  Get the attributes that correspond with the form
         */
        if (menuform != NULL)
        {
	   form = xvf_create_form(menuform, NONE, xvf_menuform_cb,
                                      (kaddr) object, -1, -1, FALSE);
        }
        return((kaddr) form);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_map_menuform
|
|       Purpose: Map the menuform
|
|         Input: object - the object in which the menu is associated
|		 form   - the object's menu form
|        Output: none
|    Written By: Mark Young and John Salas
|          Date: Nov 12, 1992 17:42
| Modifications:
|
------------------------------------------------------------*/

/* ARGSUSED */
static int xvf_map_menuform(
   xvobject object,
   kform    *form)
{
	if (kvf_get_entry((kaddr) form))
	   xvf_map_form(form);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_unmap_menuform
|
|       Purpose: Unmap the menuform
|
|         Input: object - the object in which the menu is associated
|		 form   - the object's menu form
|        Output: none
|    Written By: Mark Young and John Salas
|          Date: Nov 12, 1992 17:42
| Modifications:
|
------------------------------------------------------------*/

/* ARGSUSED */
static int xvf_unmap_menuform(
   xvobject object,
   kform    *form)
{
	if (kvf_get_entry((kaddr) form))
	   xvf_unmap_form(form);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_init_menuform
|
|       Purpose: Initializes the ManagerAction routines so that
|                xvf_create_menuform() & xvf_activate_menu()
|                are used with the menu callback system.
|
|         Input: none
|        Output: none
|    Written By: Mark Young  
|          Date: Jul 03, 1992 10:16
| Modifications:
|
------------------------------------------------------------*/

void xvf_init_menuform(void)
{
	xvf_initialize();
	xvw_initinfo_menu(xvf_create_menuform, xvf_update_menuform, 
			  xvf_map_menuform,    xvf_unmap_menuform);
}
