/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		       Extra Call Routines                    <<<<
   >>>>  Private:					      <<<<
   >>>>   Static:       add_extra_call                        <<<<
   >>>>		        remove_extra_call                     <<<<
   >>>>                                                       <<<<
   >>>>   Public:	xvf_add_extra_call()	              <<<<
   >>>>		        xvf_remove_extra_call()	              <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


static int  call_count = 0; 

static int  add_extra_call    PROTO((kform_struct *, kfunc_void, char *, int));
static void remove_extra_call PROTO((kform_struct *, kfunc_void, char *));

/************************************************************
*
*  Routine Name: xvf_add_extra_call - add function call to GUI item
*
*       Purpose: Some items on the Khoros GUI perform a particular 
*                function automatically, \fIwithout ever returning flow 
*                control to the application program\fP;  they cannot be
*                made "live" in order to force a return of flow control 
*                to the application.  These items include:
*
*              ! (1) subform buttons, which map a subform
*              ! (2) guide buttons, which map a pane
*              ! (3) quit buttons, which close the subform or exit the program
*              ! (4) help buttons, which put up a help page
*              ! (5) answer input files, which update the GUI according to a GUI answer file
*              ! (6) answer output files, which write out a GUI answer file
*             
*              ! With these items, a mechanism is sometimes desired which 
*                will allow the programmer to call a particular routine when 
*                these items are used.  For example, the programmer may want 
*                a certain routine to be called before a subform is mapped, 
*                after a pane is mapped, before the GUI is destroyed, or before
*                a help page is displayed.  For situations like this, 
*                xvf_add_extra_call() may be used to specify an additional
*                routine that will be called by the built-in callback for
*                the GUI item in question.  xvf_add_extra_call() may be
*                called more than once, if calls to more than one extra 
*                routine are needed.
*
*         Input: kformstruct - kform_struct associated with the GUI item,
*			       to which the extra call is to be added.
*			       Note that all candidates for this argument
*                              will be found in the GUI Information structure,
*                              which is automatically generated in the 
*                              "form_info.h" file.
*
*		 routine     - The extra routine to call. This routine MUST
*                              be declared as follows:
*
*                              ! void routine(
*                              !       kaddr client_data)
*                                
*		 client_data   - Client data with which to call extra routine.
*                                The client_data pointer is used when it is 
*                                necessary to pass arguments to a callback 
*                                routine.  First, define a single structure 
*                                containing all the parameters needed. Declare
*                                a pointer to the structure, dynamically 
*                                allocate the pointer, and initialize all the 
*                                fields with the values that you would have 
*                                given to the parameters before passing them to 
*                                the routine in a normal call.  Pass the pointer
*                                to your allocated, initialized structure to 
*                                xvf_add_extra_call() as the client_data.  
*                                Inside the callback, cast the client_data to 
*                                the appropriate structure type, as in:
*                              ! my_struct *my_ptr = (my_struct *) client_data;
*                                At that point, the fields may be accessed as
*                                desired.  They will hold the values to which
*                                they were initialized.
*
*		 call_location - tells whether the specified routine should
*                                be called before the "normal" operation of 
*                                the item is performed, after the "normal"
*                                operation of the item is performed, or 
*                                instead of the "normal" operation of the
*                                item.
*
*                                One of: XVF_CALL_FIRST, XVF_CALL_LAST,
*                                        or XVF_CALL_SUBSTITUTE
*
*        Output: none
*
*       Returns: TRUE on success, FALSE on failure
*
*  Restrictions: This routine is only supported for kformstructs that are 
*                associated with subform buttons, guide buttons, quit buttons,
*                help buttons, answer input file selections, or answer
*                output file selections.  Other GUI items either return control 
*                to the application program by their definition (as with Action
*                buttons) or can be made "live" (as with Toggles, Input Files, 
*                Integer selections, etc); either case makes an additional
*                callback installation redundant.
*
*    Written By: Danielle Argiro 
*          Date: Oct 14, 1992 10:14
*      Verified:
*  Side Effects:
* Modifications: 
*
*************************************************************/

int xvf_add_extra_call(
   kform_struct *kformstruct,
   kfunc_void   routine,
   char         *client_data,
   int          call_location)
{
	int status;

	switch (kformstruct->type)
	{
            case KSUBFORM:
            case KPANE:
            case KGUIDE:
            case KSELECTION:
	         switch(kformstruct->flag)
		 {
	    	     case KUIS_SUBFORMBUTTON:
	    	     case KUIS_GUIDEBUTTON:
	    	     case KUIS_STARTPANE:
	    	     case KUIS_HELP:
	    	     case KUIS_QUIT:
	    	     case KUIS_ROUTINE:
		          status = add_extra_call(kformstruct, routine, 
			                          client_data, call_location);
			  break;

		     default:
			  errno = KINVALID_PARAMETER;
		          kerror("xvforms", "xvf_add_extra_call", 
			         "invalid %s kformstruct representing passed in; only kformstructs representing subform buttons, guide buttons, help buttons, quit buttons, answer inputfile, or answer outputfile selections may be used.",
				 kvf_ascii_typeflag(kformstruct->flag));
	                  status = FALSE;
			  break;
		 }
		 break;

	    default:
		 errno = KINVALID_PARAMETER;
		 kerror("xvforms", "xvf_add_extra_call",
			"invalid kformstruct passed in;  only kformstructs representing subforms, guides, or valid selections may be used");
	         status = FALSE;
	         break;
	}
	return(status);
}

/*-----------------------------------------------------------
|
|  Routine Name: add_extra_call
|
|       Purpose: Adds a new xvf_callback_info node to the list of
|                extra calls for a subform, guide, or selection
|
|         Input: kformstruct   - kform_struct associated with the GUI item
|                routine       - The extra routine to call.
|                client_data   - Client data with which to call extra routine.
|                call_location - One of: XVF_CALL_FIRST, XVF_CALL_LAST,
|                                or XVF_CALL_SUBSTITUTE
|
|        Output: Returns TRUE on success, FALSE on failure
|          Date: Oct 14, 1992
|    Written By: Danielle Argiro
| Modifications: 
|
-------------------------------------------------------------*/

static int add_extra_call(
   kform_struct *kformstruct,
   kfunc_void   routine,
   char         *client_data,
   int          call_location)
{
	int               status;
	ksubform          *subform;
	kguide            *guide;
	kselection        *selection;
	xvf_callback_info *call_info;

	call_info = (xvf_callback_info *) kcalloc(1, sizeof(xvf_callback_info));
	if (call_info == NULL)
	{
	    kerror("xvforms", "xvf_add_extra_call",
		   "Unable to allocate extra call list");
	    return(FALSE);
	}

	call_info->routine        = routine;
	call_info->client_data    = client_data;
	call_info->call_location  = call_location;
	call_count++;

	
	status = FALSE;
	switch (kformstruct->type)
	{
	    case KSUBFORM:
		 subform = kformstruct->Subformptr;
		 subform->extra_call_list = 
			klist_insert(subform->extra_call_list, 
			             (kaddr) kformstruct, (kaddr) call_info, 
				     KLIST_NEXT, TRUE);
		 if (subform->extra_call_list != NULL)
		     status = TRUE;
	         else status = FALSE;
		 break;

            case KPANE:
		 guide = kformstruct->Controlptr->back_guide;
		 guide->extra_call_list =
			klist_insert(guide->extra_call_list, 
		    	            (kaddr) kformstruct, (kaddr) call_info, 
			 	     KLIST_NEXT, TRUE);
		 if (guide->extra_call_list != NULL)
		     status = TRUE;
	         else status = FALSE;
		 break;

            case KGUIDE:
		 guide = kformstruct->Guideptr;
		 guide->extra_call_list =
		    klist_insert(guide->extra_call_list, 
			             (kaddr) kformstruct, (kaddr) call_info, 
				     KLIST_NEXT, TRUE);
		 if (guide->extra_call_list != NULL)
		     status = TRUE;
	         else status = FALSE;
		 break;

            case KSELECTION:
		 selection = kformstruct->Selptr;
		 selection->extra_call_list =
		    klist_insert(selection->extra_call_list, 
			             (kaddr) kformstruct, (kaddr) call_info, 
				     KLIST_NEXT, TRUE);
		 if (selection->extra_call_list != NULL)
		     status = TRUE;
	         else status = FALSE;
		 break;
	}
	return(status);
}

/************************************************************
*
*  Routine Name: xvf_remove_extra_call - remove function call from GUI item
*
*       Purpose: Removes an extra call from a subform button, guide button,
*                quit button, help button, answer infile selection, or
*                answer outfile selection, where the extra call was added
*                earlier with xvf_add_extra_call().
*
*         Input: kformstruct     - kform_struct associated with the GUI item,
*			         from which the extra call is to be removed
*                                (passed to xvf_add_extra_call earlier).
*		 routine       - The extra routine that was being called
*                                (passed to xvf_add_extra_call earlier).
*		 client_data   - Client data with which routine was called
*                                (passed to xvf_add_extra_call earlier).
*
*        Output: none
*
*       Returns: TRUE on success, FALSE on failure
*
*  Restrictions: This routine should only be used for extra calls that were
*                added earlier with a call to xvf_add_extra_call.
*
*    Written By: Danielle Argiro 
*          Date: Oct 14, 1992 10:14
*      Verified:
*  Side Effects:
* Modifications: 
*
*************************************************************/

int xvf_remove_extra_call(
   kform_struct *kformstruct,
   kfunc_void   routine,
   char         *client_data)
{
	int status = FALSE;

	switch (kformstruct->type)
	{
            case KSUBFORM:
            case KGUIDE:
            case KSELECTION:

	         switch(kformstruct->flag)
		 {
	    	     case KUIS_SUBFORMBUTTON:
	    	     case KUIS_GUIDEBUTTON:
	    	     case KUIS_HELP:
	    	     case KUIS_QUIT:
	    	     case KUIS_ROUTINE:
		          remove_extra_call(kformstruct, routine, client_data);
			  status = TRUE;
			  break;

		     default:
			  errno = KINVALID_PARAMETER;
		          kerror("xvforms", "xvf_remove_extra_call", 
			         "invalid kformstruct passed in;  only kformstructs used earlier with xvf_add_extra_call() are valid. ");
			  break;
		 }
		 break;

	    default:
		 errno = KINVALID_PARAMETER;
		 kerror("xvforms", "xvf_remove_extra_call",
			"invalid kformstruct passed in;  only kformstructs used earlier with xvf_add_extra_call() are valid. ");
	         break;
	}
	return(status);
}

/*-----------------------------------------------------------
|
|  Routine Name: remove_extra_call
|
|       Purpose: Removes an old xvf_callback_info node from the list of
|                extra calls for a subform, guide, or selection
|
|         Input: kformstruct     - kform_struct associated with the GUI item
|                routine       - The extra routine which was called.
|                client_data   - Client data being used w/ extra routine.
|
|        Output: none
|          Date: Oct 14, 1992
|    Written By: Danielle Argiro
| Modifications: 
|
-------------------------------------------------------------*/

/* ARGSUSED */
static void remove_extra_call(
   kform_struct *kformstruct,
   kfunc_void   routine,
   char         *client_data)
{
	switch (kformstruct->type)
	{
	    case KSUBFORM:
		 kformstruct->Subformptr->extra_call_list = 
			klist_delete(kformstruct->Subformptr->extra_call_list, 
			             (kaddr) kformstruct);
		 break;

            case KGUIDE:
		 kformstruct->Guideptr->extra_call_list =
			klist_delete(kformstruct->Guideptr->extra_call_list, 
			             (kaddr) kformstruct);
		 break;

            case KSELECTION:
		 kformstruct->Selptr->extra_call_list =
			klist_delete(kformstruct->Selptr->extra_call_list, 
			             (kaddr) kformstruct);
		 break;
	}
}
