/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		       xvobject Creation Routines             <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		xvf_create_backplane()		      <<<<
   >>>>	     		xvf_create_submenu_sel()              <<<<
   >>>>	     		xvf_create_subform_sel()	      <<<<
   >>>>	     		xvf_create_guide_sel()	  	      <<<<
   >>>>	     		xvf_create_quit_master_sel()  	      <<<<
   >>>>	     		xvf_create_quit_subform_sel()  	      <<<<
   >>>>	     		xvf_create_quit_pane_sel()  	      <<<<
   >>>>	     		xvf_create_workspace_sel()  	      <<<<
   >>>>	     		xvf_create_master_action_sel() 	      <<<<
   >>>>	     		xvf_create_subform_action_sel()       <<<<
   >>>>	     		xvf_create_help_sel()       	      <<<<
   >>>>	     		xvf_create_input_sel()       	      <<<<
   >>>>	     		xvf_create_output_sel()       	      <<<<
   >>>>	     		xvf_create_std_sel()       	      <<<<
   >>>>	     		xvf_create_int_sel()       	      <<<<
   >>>>	     		xvf_create_float_sel()       	      <<<<
   >>>>	     		xvf_create_logic_sel()       	      <<<<
   >>>>	     		xvf_create_string_sel()       	      <<<<
   >>>>	     		xvf_create_stringlist_sel()   	      <<<<
   >>>>	     		xvf_create_cycle_sel()       	      <<<<
   >>>>	     		xvf_create_routine_sel()       	      <<<<
   >>>>	     		xvf_create_toggle_sel()       	      <<<<
   >>>>	     		xvf_create_blank_sel()       	      <<<<
   >>>>	     		xvf_create_list_sel()       	      <<<<
   >>>>	     		xvf_create_displaylist_sel()          <<<<
   >>>>	     		xvf_create_pane_action_sel()   	      <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


#define  MaxNumSize 50

static int toggle_count = 0;


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_backplane
|
|       Purpose: Creates a backplane to contain other objects
|
|         Input: line_info - structure containing information
|			      gained from the UIS line.
|		 parent   - parent of backplane being created.
|		 mapped   - TRUE if backplane should be mapped upon creation
|
|        Output: Returns the object which will serve as backplane
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject  xvf_create_backplane(
   kaddr    pointer,
   int      type,
   xvobject parent,
   int      mapped)
{
	xvobject     back = NULL;
	Line_Info    line_info;
	kform        *form;
	ksubform     *subform;
	kcontrol     *guidepane, *pane;
	char         object_name[KLENGTH];
	char         buffer[KLENGTH];
	
	kvf_clear_line_info(&line_info);

	if (type == KUIS_STARTFORM)
	{
	    /*
	     *  create the backplane for the master form
	     */
	    form = (kform *) pointer;
	    kvf_parse_startform_line(form->control_line, &line_info);
	    ksprintf(object_name, "%s_back", line_info.variable);
	    back = xvf_create_formback(parent, line_info.x, line_info.y, 
			               line_info.width, line_info.height,
				       object_name, mapped);
	    xvw_set_attribute(back, XVW_DEF_VERT_DIST, 3);

	    /*
	     * disable editing if specified
	     */
	    if (form->editable == XVF_NO_EDIT)
	    {
		xvw_set_attribute(back, XVW_MENUABLE, FALSE);
		xvw_set_attribute(back, XVW_SELECTABLE, FALSE);
	    }

            /*
	     * associate the appropriate menuform with the master form;
	     * set the kformstruct as the clientdata
	     */
	    if (form->editable)
	        xvw_set_attribute(back, XVW_MENU_FORMFILE, 
		    "$DESIGN/objects/library/xvforms/uis/xvf_edit/Form.pane");
	    else xvw_set_attribute(back, XVW_MENU_FORMFILE, 
		    "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Form.pane");
	    xvw_set_attribute(back, XVW_MENU_CLIENTDATA, 
			      form->master->back_kformstruct);

	    /*
	     * deliberately creating blank label if line_info.title is NULL,
	     * so the title can be changed from nothing to something w/ guise
	     */
	    if (line_info.title != NULL)
	        ksprintf(buffer, "%s", line_info.title);
	    else ksprintf(buffer, " ");

	    form->master->label_object = xvf_create_label(back, line_info.xpos,
	    	                         line_info.ypos, NULL, buffer,
					 "label", FALSE, TRUE);
	    xvw_set_attribute(form->master->label_object, XVW_MENUABLE, FALSE);

	    /*
             * want masterform label to have same clientdata as backplane
             * (so that it works in xvf_set_xxx), plus a max height of 1.
             */
	    xvw_set_attributes(form->master->label_object, 
			  XVW_MENU_CLIENTDATA, form->master->back_kformstruct,
		          XVW_CHAR_MAX_HEIGHT, 1.0,
			  NULL);
	}
	else if (type == KUIS_STARTSUBFORM)
	{
	    /*
	     *  create the backplane for the subform
	     */
	    subform = (ksubform *) pointer;
	    kvf_parse_startsubform_line(subform->control_line, &line_info);
	    ksprintf(object_name, "%s_back",line_info.variable);
	    back = xvf_create_formback(parent, line_info.x, line_info.y, 
				       line_info.width, line_info.height,
				       object_name, mapped);
	    xvw_set_attribute(back, XVW_DEF_VERT_DIST, 3);

	    /*
	     * disable editing if specified
	     */
	    if (subform->back_form->editable == XVF_NO_EDIT)
	    {
		xvw_set_attribute(back, XVW_MENUABLE, FALSE);
		xvw_set_attribute(back, XVW_SELECTABLE, FALSE);
	    }

            /*
	     * associate the appropriate menuform with the subform
	     * set the kformstruct as the clientdata
	     */
	    if (subform->back_form->editable)
	        xvw_set_attribute(back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Subform.pane");
	    else xvw_set_attribute(back, XVW_MENU_FORMFILE, 
                  "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Subform.pane");
	    xvw_set_attribute(back, XVW_MENU_CLIENTDATA, 
			      subform->back_kformstruct);

	    /*
	     * deliberately creating blank label if line_info.title is NULL,
	     * so the title can be changed from nothing to something w/ guise
	     */
	    if (line_info.title != NULL)
                ksprintf(buffer, "%s", line_info.title);
            else ksprintf(buffer, " ");

	    subform->label_object = xvf_create_label(back, line_info.xpos,
	    	                          line_info.ypos, NULL, buffer,
					  "label", FALSE, TRUE);
	    xvw_set_attribute(subform->label_object, XVW_MENUABLE, FALSE);

	    /*
	     * want subform label to have same clientdata as backplane
             * (so that it works in xvf_set_xxx), plus a max height of 1.
	     */
	    xvw_set_attributes(subform->label_object, 
			XVW_MENU_CLIENTDATA, subform->back_kformstruct,
		        XVW_CHAR_MAX_HEIGHT, 1.0,
			NULL);
	}
	else if (type == KUIS_STARTGUIDE)
	{
	    guidepane = (kcontrol *) pointer;
	    kvf_parse_startguide_line(guidepane->control_line, &line_info);
	    ksprintf(object_name, "guide_back");
	    back = guidepane->back_subform->back;

	    /*
	     * deliberately creating blank label if line_info.title is NULL,
	     * so the title can be changed from nothing to something w/ guise
	     */
	    if (line_info.title != NULL)
                ksprintf(buffer, "%s", line_info.title);
            else ksprintf(buffer, " ");

	    guidepane->label_object = xvf_create_label(back, line_info.xpos,
	    	                          line_info.ypos, NULL, buffer,
					  "label", FALSE, TRUE);

	    xvw_set_attribute(guidepane->label_object, XVW_MENUABLE, FALSE);

	    /*
             * want guidepane label to have same clientdata as backplane
             * (so that it works in xvf_set_xxx), plus a max height of 1.
             */
	    xvw_set_attributes(guidepane->label_object, 
			XVW_MENU_CLIENTDATA, guidepane->back_kformstruct,
			XVW_CHAR_MAX_HEIGHT, 1.0,
			NULL);
	}
	else if (type == KUIS_STARTPANE)
	{
	    pane = (kcontrol *) pointer;
	    kvf_parse_startpane_line(pane->control_line, &line_info);
	    ksprintf(object_name,"%s_back",line_info.variable);
	    back = xvf_create_selback(parent, line_info.x, line_info.y, 
				      line_info.width, line_info.height, 
				      object_name, 3);
	    xvw_set_attribute(back, XVW_DEF_VERT_DIST, 3);
	    xvw_set_attribute(back, XVW_MAP_WHEN_MANAGED, mapped);
	    xvw_set_attribute(back, XVW_BORDER_WIDTH, 1);

	    xvw_set_attribute(back, XVW_MENU_CLIENTDATA, 
			      pane->back_kformstruct);

	    if (pane->back_form->editable == XVF_NO_EDIT)
	    {
		xvw_set_attribute(back, XVW_MENUABLE, FALSE);
		xvw_set_attribute(back, XVW_SELECTABLE, FALSE);
	    }

            /*
	     * associate the appropriate menuform with the pane
	     */
	    if (pane->back_form->editable)
	        xvw_set_attribute(back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Pane.pane");
	    else xvw_set_attribute(back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Pane.pane");

	    /*
	     * deliberately creating blank label if line_info.title is NULL,
	     * so the title can be changed from nothing to something w/ guise
	     */
	    if (line_info.title != NULL)
                ksprintf(buffer, "%s", line_info.title);
            else ksprintf(buffer, " ");

	    pane->label_object = xvf_create_label(back, line_info.xpos, 
	    	                          line_info.ypos, NULL, buffer,
					  "label", FALSE, TRUE);
	    xvw_set_attribute(pane->label_object, XVW_MENUABLE, FALSE);

	    /*
             * want pane label to have same clientdata as backplane
             * (so that it works in xvf_set_xxx), plus a max height of 1.
             */
	    xvw_set_attributes(pane->label_object, 
			XVW_MENU_CLIENTDATA, pane->back_kformstruct,
			XVW_CHAR_MAX_HEIGHT, 1.0,
			NULL);
	}

	if (!line_info.activate)
	    xvw_sensitive(back, FALSE);

	kvf_free_line_info_strings(&line_info);

	return(back);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_submenu_sel
|
|       Purpose: Creates a submenu button and associated submenu
|		 which may contain any selection which consists
|                of a button.
|
|         Input: selection - ptr to the selection struct 
|		 parent    - parent of submenu button object being created 
|
|        Output: Returns the object which will serve as submenu button.
|
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_submenu_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *label;
	xvobject     menubutton = NULL; 
	xvobject     submenu = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_startsubmenu_line(selection->line, &line_info)))
	    return(NULL);

	if (line_info.title != NULL)
	    label = kstrdup(line_info.title);  
	else label = kstrdup(" ");

	menubutton = xvf_create_menubutton(parent, NULL, line_info.x, 
					   line_info.y, line_info.width, 
					   line_info.height, label, 
					   line_info.variable, NULL, 0, 
			                   &selection->submenu_buttons);
        submenu = xvw_retrieve_menu(menubutton);

	xvw_set_attributes(menubutton, 
			   XVW_MENUABLE,     TRUE,     /* menuable       */
                           XVW_RESIZABLE,    TRUE,     /* resizable      */
                           XVW_SELECTABLE,   TRUE,     /* translatable   */
			   NULL);

	if (!line_info.activate)
	    xvw_sensitive(menubutton, FALSE);

	xvw_set_attribute(menubutton, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	xvw_set_attribute(submenu, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_form->editable)
	    xvw_set_attribute(menubutton, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Submenu.pane");
	else xvw_set_attribute(menubutton, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Submenu.pane");

	selection->label_object = menubutton;

	kvf_free_line_info_strings(&line_info);
	kfree(label);
	return(submenu);
}



/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_subform_sel
|
|       Purpose: Creates a subform button object
|		 which will bring up a particular subform.
|
|         Input: subform   - ptr to the subform struct
|		 parent    - parent of subform button being created 
|
|        Output: Returns the object which will serve as subform button
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
|
------------------------------------------------------------*/

xvobject xvf_create_subform_sel(
   ksubform *subform,
   xvobject parent)
{
	Line_Info    line_info, tmp_line_info;
	char         *label;
	xvobject     button = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_gen_parse(subform->line, &line_info)))
	    return(NULL);

	kvf_clear_line_info(&tmp_line_info);
	if (!(kvf_parse_startsubform_line(subform->control_line, 
					  &tmp_line_info)))
	    return(NULL);

	if (line_info.title != NULL)
	   label = kstrdup(line_info.title);  
	else label = kstrdup(" ");

	if (subform->back_submenu != NULL)
	    button = xvf_create_menu_button(parent, 
				label, tmp_line_info.variable);
	else button = xvf_create_button(parent, line_info.x, 
				line_info.y, line_info.width, line_info.height, 
				label, tmp_line_info.variable);

	xvw_insert_callback(button, XVW_BUTTON_SELECT, FALSE, xvf_subform_cb,
			 (kaddr) subform->back_kformstruct);

	if (!line_info.activate)
	    xvw_sensitive(button, FALSE);

	kvf_free_line_info_strings(&line_info);
	kfree(label);
	kvf_free_line_info_strings(&tmp_line_info);

	/*
	 *  Set the internal menuform for a subform button...
	 */
	xvw_set_attribute(button, XVW_MENU_CLIENTDATA, 
			  subform->back_kformstruct);
	if (subform->back_submenu == NULL)
	{
	    if (subform->back_form->editable)
	        xvw_set_attribute(button, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Sfb.pane");
	    else xvw_set_attribute(button, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Sfb.pane");
	}
	return(button);

}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_guide_sel
|
|       Purpose: Creates a guide button.
|
|         Input: guide     - ptr to the guide struct
|		 parent    - parent of subform button being created 
|
|        Output: Returns the object which will serve as guide button.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_guide_sel(
   kguide   *guide,
   xvobject parent)
{
	Line_Info    line_info, tmp_line_info;
	char         *label; 
	xvobject     button = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_guide_line(guide->line, &line_info)))
	{
	    kvf_free_line_info_strings(&line_info);
	    return(NULL);
	}

	kvf_clear_line_info(&tmp_line_info);
	if (!(kvf_parse_startpane_line(guide->pane->control_line, 
					&tmp_line_info)))
	{
	    kvf_free_line_info_strings(&tmp_line_info);
	    return(NULL);
	}
	kvf_free_line_info_strings(&tmp_line_info);

	if (line_info.title != NULL)
	    label = kstrdup(line_info.title); 
	else label = kstrdup(" ");				

	if (guide->back_submenu != NULL)
	    button = xvf_create_menu_button(parent, 
	 			label, tmp_line_info.variable);
	else
	    button = xvf_create_button(parent, line_info.x, line_info.y,
                        	line_info.width, line_info.height, 
				label, tmp_line_info.variable);

	xvw_insert_callback(button, XVW_BUTTON_SELECT, FALSE, xvf_guide_cb,
			 (kaddr) guide->back_kformstruct);

	if (!line_info.activate)
	    xvw_sensitive(button, FALSE);

	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for a guide button...
	 */
	xvw_set_attribute(button, XVW_MENU_CLIENTDATA, 
			  guide->back_kformstruct);
	if (guide->back_submenu == NULL)
	{
	    if (guide->back_form->editable)
	        xvw_set_attribute(button, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Gb.pane");
	    else xvw_set_attribute(button, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Gb.pane");
	}
	return(button);
}



/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_quit_sel
|
|       Purpose: Creates a quit button. 
|
|         Input: selection     - ptr to the selection struct
|		 parent        - parent of quit button being created 
|		 num           - number of quit button for naming
|                location_flag - Master, GuidePane, or Pane
|
|        Output: Returns the object which will serve as quit button.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_quit_sel(
   kselection *selection,
   xvobject   parent,
   int        location_flag)
{
	Line_Info    line_info;
	char         *label;
	xvobject     button = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_quit_line(selection->line, &line_info))) return(NULL);
   
	if (line_info.title != NULL) 
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	if (selection->back_submenu != NULL)
	    button = xvf_create_menu_button(parent, 
	 			label, "quit");
	else
	    button = xvf_create_button(parent, line_info.x, line_info.y,
                        	line_info.width, line_info.height, 
				label, "quit");

        if (location_flag == KMASTER)
            xvw_insert_callback(button, XVW_BUTTON_SELECT, FALSE,
		xvf_quit_master_cb, (kaddr) selection->back_kformstruct);
        else if (location_flag == KGUIDEPANE)
            xvw_insert_callback(button, XVW_BUTTON_SELECT, FALSE,
		xvf_quit_subform_cb, (kaddr) selection->back_kformstruct);
	else if (location_flag == KPANE)
	    xvw_insert_callback(button, XVW_BUTTON_SELECT, FALSE,
		xvf_quit_pane_cb, (kaddr) selection->back_kformstruct);

	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for a quit button...
	 */
	xvw_set_attribute(button, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_submenu == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(button, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Quit.pane");
	    else xvw_set_attribute(button, XVW_MENU_FORMFILE, 
                   "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Quit.pane");
	}
	return(button);
}



/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_workspace_sel
|
|       Purpose: Creates a workspace selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of workspace object being created 
|
|        Output: Returns the object which will serve as workspace 
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_workspace_sel(
   kselection *selection,
   xvobject   parent)
{ 
	Line_Info line_info;
	char      temp[KLENGTH];

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_workspace_line(selection->line, &line_info)))
	    return(NULL);
   
	ksprintf(temp, "%s_back", line_info.variable);

	/* create workspace object  -- simply a backplane itself */
	selection->back = xvw_create_manager(parent, temp);
	xvw_set_attributes(selection->back,
                XVW_CHAR_XPOS,      line_info.x,       /* set x position      */
                XVW_CHAR_YPOS,      line_info.y,       /* set y position      */
                XVW_WIDTH,  (int)   line_info.width,   /* set width (pixels)  */
                XVW_HEIGHT, (int)   line_info.height,  /* set height (pixels) */
	        XVW_RESIZABLE,      TRUE,              /* resizable           */
                XVW_DEF_HORIZ_DIST, 0,                 /* dist bwn children   */
                XVW_DEF_VERT_DIST,  0,                 /* dist bwn children   */
                XVW_MENUABLE,       TRUE,              /* menuable            */
                XVW_SELECTABLE,     TRUE,              /* selectable          */
                NULL);
	xvw_insert_callback(selection->back, XVW_GEOMETRY_CALLBACK, FALSE,
			 xvf_inform_geometry, NULL);

	selection->value_object = selection->back;

	line_info.workspace = selection->value_object;
	kvf_deparse_workspace_line(&line_info, &selection->line);

	kvf_free_line_info_strings(&line_info);

	/*
	 *  Set the internal menuform for a workspace object...
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_submenu == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Workspace.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Workspace.pane");
	}
	return(selection->back);
}



/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_master_action_sel
|
|       Purpose: Creates a master action button.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of quit button being created 
|
|        Output: Returns the object which will serve as master action button.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_master_action_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *label;
	xvobject     button = NULL; 

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_master_action_line(selection->line, &line_info)))
	    return(NULL);

	if (line_info.title != NULL)
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	if (selection->back_submenu != NULL)
	    button = xvf_create_menu_button(parent, 
					label, line_info.variable);
	else button = xvf_create_button(parent, line_info.x,
					line_info.y, line_info.width, 
					line_info.height, label, 
					line_info.variable);

	xvw_insert_callback(button, XVW_BUTTON_SELECT, FALSE, xvf_action_cb,
                         (kaddr) selection->back_kformstruct);

	if (!line_info.activate)
	    xvw_sensitive(button, FALSE);

	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for a master action...
	 */
	xvw_set_attribute(button, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_submenu == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(button, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Action.pane");
	    else xvw_set_attribute(button, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Action.pane");
	}
	return(button);
}



/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_subform_action_sel
|
|       Purpose: Creates a subform action button. 
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of subform action button being created 
|
|        Output: Returns object which serves as subform action button.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_subform_action_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *label;
	xvobject     button = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_subform_action_line(selection->line, &line_info)))
	    return(NULL);

	if (line_info.title != NULL)
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	if (selection->back_submenu != NULL)
	    button = xvf_create_menu_button(parent, 
					label, line_info.variable);
	else button = xvf_create_button(parent, line_info.x, 
					line_info.y, line_info.width, 
					line_info.height, label, 
					line_info.variable);

	xvw_insert_callback(button, XVW_BUTTON_SELECT, FALSE, xvf_action_cb,
			 (kaddr) selection->back_kformstruct);

	if (!line_info.activate)
	    xvw_sensitive(button, FALSE);

	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for a subform action...
	 */
	xvw_set_attribute(button, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_submenu == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(button, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Action.pane");
	    else xvw_set_attribute(button, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Action.pane");
	}
	return(button);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_help_sel
|
|       Purpose: Creates a help button.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of help button being created 
|		 num       - number of help button for naming
|
|        Output: Returns object which serves as help button.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_help_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	xvobject     button = NULL; 
	char         *label;
	char         *name;
	char         *ptr = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_help_line(selection->line,&line_info))) return(NULL);

	if (line_info.title != NULL) 
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	ptr = kstrstr(line_info.variable, "license");
	if (ptr != NULL)
	    name = "license";
	else 
	{
	    ptr = kstrstr(line_info.variable, "todo");
    	    if (ptr != NULL)
	        name = "todo";
	    else name = "help";
	}

	if (selection->back_submenu != NULL)
	    button = xvf_create_menu_button(parent, label, name);
	else button = xvf_create_button(parent, line_info.x, 
				line_info.y, line_info.width, line_info.height, 
				label, name);

	xvw_insert_callback(button, XVW_BUTTON_SELECT, TRUE,
		xvf_help_button_cb, (kaddr) selection->back_kformstruct);

	if (!line_info.activate)
	    xvw_sensitive(button, FALSE);

	/*
	 *  Set the internal menuform for a help button...
	 */
	xvw_set_attribute(button, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_submenu == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(button, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Help.pane");
	    else xvw_set_attribute(button, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Help.pane");
	}
	kvf_free_line_info_strings(&line_info);
	kfree(label);
	return(button);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_input_sel
|
|       Purpose: Creates an input selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent object
|
|        Output: Returns backplane of input selection object.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/

xvobject xvf_create_input_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info      line_info;
	char           *label, *object_name; 
	char           temp[KLENGTH];
	float          width; 
	xvobject       offset = NULL;

	/* 
	 * initialize callback data for this input selection 
	 */
	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_input_line(selection->line, &line_info))) return(NULL);
	width = line_info.width;

	object_name = xvf_create_gen_selname(line_info.variable, 
				    selection->back_toggle, &toggle_count);

	/* create input selection backplane object */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
 			          line_info.width, line_info.height, 
				  object_name, 0);
	xvw_set_attribute(selection->back, XVW_CHAR_MAX_HEIGHT, 1.0);
	kfree(object_name);

	/* 
	 * create optional button if necessary
	 */
	if (((line_info.optional) ||
             (selection->back_toggle != NULL)) &&
	     (line_info.opt_sel != 2))
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
	    xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
                         xvf_gen_opt_cb, (kaddr) selection->back_kformstruct);

	    offset = selection->opt_object;
	    width -= 1.0;
	}

	/* 
	 * put in label 
	 */

	if (line_info.title != NULL) 
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	if (selection->back_toggle == NULL)
	{
	    selection->label_object = xvf_create_pseudo_label(selection->back,
				(float) kstrlen(label), line_info.height,
				offset, label);

	    xvw_insert_callback(selection->label_object, XVW_BUTTON_SELECT, 
			     FALSE, xvf_pseudo_label_cb, 
			     (kaddr) selection->back_kformstruct);
	}
	else
	{
	    selection->label_object = 
			xvf_create_label(selection->back, line_info.xpos, 
			line_info.ypos, offset, label, "label", TRUE, 
			FALSE);
	}
	width -= (float) (kstrlen(label));

	/*
	 * create the ascii text object  
	 */
	xvf_set_text_buffer(temp, &line_info);

	if (selection->back_toggle == NULL)
	{
	    selection->value_object  = xvf_create_text(selection->back, width, 
					 line_info.height, 
					 selection->label_object, temp, FALSE);
            xvw_add_action(selection->value_object, "<KeyRelease>",
                           xvf_mark_modified, selection, FALSE);

	    if (line_info.live)
	    {
		selection->pix_object = 
			xvf_create_cr_pixmap(selection->back);
	        xvw_set_attribute(selection->value_object, XVW_LEFT_OF, 
				  selection->pix_object);
	    }
	    else xvw_set_attribute(selection->value_object, XVW_LEFT_OF, NULL);

	    xvw_add_action(selection->value_object, "<Key>Return",
                          xvf_return_live_selection, selection, TRUE);
	}

	selection->opt_selected = line_info.opt_sel;

	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

	if (selection->back_toggle != NULL)
	{
	    if (!selection->back_toggle->opt_selected) 
	    {
	        xvf_set_optional_sensitivity(selection, FALSE);
		if (xvf_desensitize_optionals)
	            xvw_sensitive(selection->opt_object, FALSE);
	    }
	}
	else if (!line_info.opt_sel)
            xvf_set_optional_sensitivity(selection, selection->opt_selected);


	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for an InputFile selection...
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);

	if (selection->back_toggle == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/InputFile.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/InputFile.pane");
	}
	else 
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/ToggleI.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/ToggleI.pane");
	}
	return(selection->back);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_output_sel 
|
|       Purpose: Creates an output selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of output selection being created 
|
|        Output: Returns backplane of output selection object.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_output_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *object_name, *label; 
	char         temp[KLENGTH];
	float        width;
	xvobject     offset = NULL;

	/*
         * initialize callback data for this output selection
         */
	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_output_line(selection->line, &line_info))) return(NULL);
	width = line_info.width;
   
	object_name = xvf_create_gen_selname(line_info.variable, 
				selection->back_toggle, &toggle_count);
   
	/* create output selection backplane object */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
				  line_info.width, line_info.height,
				  object_name, 0);
	xvw_set_attribute(selection->back, XVW_CHAR_MAX_HEIGHT, 1.0);
	kfree(object_name);

	/* create optional button */
	if (((line_info.optional) ||
             (selection->back_toggle != NULL)) &&
	     (line_info.opt_sel != 2))
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
	    xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
                             xvf_gen_opt_cb, (kaddr) selection->back_kformstruct);
	    offset = selection->opt_object;
	    width -= 1.0;
	}

	/* 
	 * put in label 
	 */

	if (line_info.title != NULL) 
	    label= kstrdup(line_info.title);
	else label = kstrdup(" ");

	if (selection->back_toggle == NULL)
	{
	    selection->label_object = xvf_create_pseudo_label(selection->back,
				(float) (kstrlen(label)), line_info.height,
				offset, label);

	    xvw_insert_callback(selection->label_object, XVW_BUTTON_SELECT,
				FALSE, xvf_pseudo_label_cb,
				(kaddr) selection->back_kformstruct);
	}
	else
	{
	    selection->label_object = 
			xvf_create_label(selection->back, line_info.xpos, 
			line_info.ypos, offset, label, "label", TRUE, 
			FALSE);
	}
	width -= (float) (kstrlen(label));

	/*
	 * create the ascii text object
	 */
	xvf_set_text_buffer(temp, &line_info);

	if (selection->back_toggle == NULL)
	{
	    selection->value_object = 
			xvf_create_text(selection->back, width, 
					line_info.height, 
					selection->label_object, 
				        temp, FALSE);
            xvw_add_action(selection->value_object, "<KeyRelease>",
                           xvf_mark_modified, selection, FALSE);

	    if (line_info.live)
	    {
		selection->pix_object = 
			xvf_create_cr_pixmap(selection->back);
	        xvw_set_attribute(selection->value_object, XVW_LEFT_OF, 
				  selection->pix_object);
	    }
	    else xvw_set_attribute(selection->value_object, XVW_LEFT_OF, NULL);

            xvw_add_action(selection->value_object, "<Key>Return",
                          xvf_return_live_selection, selection, TRUE);
	}
   
	selection->opt_selected = line_info.opt_sel;
 
	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

	if (selection->back_toggle != NULL)
	{
	    if (!selection->back_toggle->opt_selected)
	    {
	        xvf_set_optional_sensitivity(selection, FALSE);
		if (xvf_desensitize_optionals)
	            xvw_sensitive(selection->opt_object, FALSE);
	    }
	}
        else if (!line_info.opt_sel)
            xvf_set_optional_sensitivity(selection, selection->opt_selected);

	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for an OutputFile selection...
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);

	if (selection->back_toggle == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/OutputFile.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/OutputFile.pane");
	}
	else 
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/ToggleO.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/ToggleO.pane");
	}
	return(selection->back);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_std_sel
|
|       Purpose: Creates a stdin or stdout selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent object
|
|        Output: Returns backplane of stdin or stdout selection object.
|          Date: Apr 26, 1994
|    Written By: 
| Modifications: 
|
-------------------------------------------------------------*/

xvobject xvf_create_std_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info      line_info;
	char           *label, *object_name; 
	xvobject       offset = NULL;

	/* 
	 * initialize callback data for this input selection 
	 */
	kvf_clear_line_info(&line_info);
	if (!(kvf_gen_parse(selection->line, &line_info))) return(NULL);

	object_name = xvf_create_gen_selname(line_info.variable, 
				    selection->back_toggle, &toggle_count);
	if (line_info.title != NULL) 
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	/* create stdin/stdout selection backplane object */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
 			                     (float)(kstrlen(label) + 4), 1.0, 
				             object_name, 0);
	xvw_set_attribute(selection->back, XVW_CHAR_MAX_HEIGHT, 1.0);
	kfree(object_name);

	/* 
	 * create optional button if necessary
	 */
	if ((line_info.optional) &&
	    (line_info.opt_sel != 2))
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
	    xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
                           xvf_gen_opt_cb, (kaddr) selection->back_kformstruct);
	    offset = selection->opt_object;
	}

	/*
	 *  stdin:  optional box, stdin pixmap, label
	 */
	if (selection->type == KUIS_STDIN)
	{
	    selection->pix_object = xvw_create_pixmap(selection->back, 
						      "StdinPixmap");
            xvw_set_attributes(selection->pix_object,
                               XVW_BORDER_WIDTH, 0,     /* no border       */
                               XVW_BELOW,        NULL,  /* set offset      */
                               XVW_ABOVE,        NULL,  /* set offset      */
                               XVW_RIGHT_OF,     offset, /* set offset      */
                               NULL);

	    selection->label_object = xvf_create_label(selection->back, 
					           0.0, 0.0, 
					           selection->pix_object, 
				                   label, "label", 
					           TRUE, FALSE);
	}
	/*
	 *  stdout:  optional box, label, stdout pixmap
	 */
	else 
	{
	    selection->label_object = xvf_create_label(selection->back, 
					           0.0, 0.0, 
					           offset, label, "label", 
					           TRUE, FALSE);
	    selection->pix_object = xvw_create_pixmap(selection->back, 
						      "StdoutPixmap");
            xvw_set_attributes(selection->pix_object,
                               XVW_BORDER_WIDTH, 0,     /* no border          */
                               XVW_BELOW,        NULL,  /* set offset         */
                               XVW_ABOVE,        NULL,  /* set offset      */
                               XVW_RIGHT_OF,     selection->label_object,
                               NULL);
	}

	selection->opt_selected = line_info.opt_sel;

	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

	if (!line_info.opt_sel)
            xvf_set_optional_sensitivity(selection, selection->opt_selected);


	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for an Stdin selection...
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);

	if (selection->type == KUIS_STDIN)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Stdin.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Stdin.pane");
	}
	else
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Stdout.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Stdout.pane");
	}
	return(selection->back);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_int_sel
|
|       Purpose: Creates an integer selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of int selection being created 
|
|        Output: Returns backplane of int selection object.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_int_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *label, *object_name;
	char         buff[KLENGTH];
	float        text_width;
	xvobject     offset = NULL;

	/*
	 * get info from UIS line
	 */
	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_int_line(selection->line, &line_info))) return(NULL);

	/*
	 * generate a unique name for selection (for journal record/playback)
	 */
	object_name = xvf_create_gen_selname(line_info.variable, 
				selection->back_toggle, &toggle_count);

	/* 
	 * create integer selection backplane 
         */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
				    line_info.width, line_info.height, 
				    object_name, 0);
	xvw_set_attribute(selection->back, XVW_CHAR_MAX_HEIGHT, 1.0);
	kfree(object_name);

	/*
         * create optional button if necessary
         */
	if (((line_info.optional) ||
             (selection->back_toggle != NULL)) &&
	     (line_info.opt_sel != 2))
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
	    xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
			     xvf_gen_opt_cb, (kaddr) selection->back_kformstruct);
	    offset = selection->opt_object;
	}
	selection->opt_selected = line_info.opt_sel;

	/* 
	 * put in label 
         */
	if (line_info.title != NULL)
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	selection->label_object = xvf_create_label(selection->back, 
				   	line_info.xpos, line_info.ypos, 
					offset, label, "label",  TRUE, FALSE);
	/*
	 * create the ascii text object
	 */
	xvf_set_text_buffer(buff, &line_info);
   
	/*
	 * if this integer selection is NOT a toggle member,
         * create the text object, the scrollbar (if applicable),
         * and the <cr> pixmap (if applicable).
	 */
	if (selection->back_toggle == NULL)
	{
	    /*
	     * unbounded int or no scrollbar wanted => 
             * use up remainder of space on text object 
	     */
	    if  ((line_info.upper_int == line_info.lower_int) || 
		 (line_info.special != 1))
	    {
		text_width = xvf_calculate_remaining_width(line_info.width,
				line_info.optional, line_info.opt_sel, 
				line_info.title);
	    }
	    /* 
	     * bounded int w/ potential scrollbar => 
             * calculate space needed for text object 
             */
	    else
	    {
		text_width = 
			xvf_calculate_text_width((double) line_info.upper_int,
                                                 (double) line_info.lower_int,
                                                  line_info.precision,
    						  KUIS_INTEGER);
	    }

	    /* 
	     * create the text object - always at least width 1
	     */
	    if (text_width <= 0) text_width = 1;
	    selection->value_object = xvf_create_text(selection->back, 
					text_width, line_info.height, 
					selection->label_object, buff, FALSE); 

	    /* 
	     * bounded int with space left bar and 'mech' set to TRUE
	     */
            xvw_add_action(selection->value_object, "<KeyRelease>",
                           xvf_mark_modified, selection, FALSE);

	    if ((line_info.upper_int != line_info.lower_int) && 
	        (line_info.special == 1))
	  	 xvf_append_selection_scrollbar(selection, &line_info);
	
	    /*
	     * "live" selections have the <cr> pixmap
	     */
	    if (line_info.live)
	    {
		if (selection->scroll != NULL)
		{
		    selection->pix_object = 
				xvf_create_cr_pixmap(selection->back);
		    xvw_set_attribute(selection->scroll, XVW_LEFT_OF,
				      selection->pix_object);
	        }
		else 
		{
		    selection->pix_object = 
				xvf_create_cr_pixmap(selection->back);
		    xvw_set_attribute(selection->value_object,
                                      XVW_LEFT_OF, selection->pix_object);
		}
	    }
	    else
	    {
		if (selection->scroll != NULL)
		    xvw_set_attribute(selection->scroll, XVW_LEFT_OF, NULL);
		else xvw_set_attribute(selection->value_object, 
				       XVW_LEFT_OF, NULL);
	    }
            xvw_add_action(selection->value_object, "<Key>Return",
                          xvf_return_live_selection, selection, TRUE);
	}

	/*
	 * desensitize backplane if requested
	 */
	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

	/*
	 *  desensitize unselected optional selection if user likes it
	 */
	if (selection->back_toggle != NULL)
	{
	    if (!selection->back_toggle->opt_selected) 
	    {
	        xvf_set_optional_sensitivity(selection, FALSE);
		if (xvf_desensitize_optionals)
	            xvw_sensitive(selection->opt_object, FALSE);
	    }
	}
        else if (!line_info.opt_sel)
            xvf_set_optional_sensitivity(selection, selection->opt_selected);

	/*
	 *  Set the internal menuform for an Integer selection
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
		          selection->back_kformstruct);

	if (selection->back_toggle == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Integer.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Integer.pane");
	}
	else
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Togglei.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Togglei.pane");
	}

	/*
	 * free up memory & return
	 */
	kvf_free_line_info_strings(&line_info);
	kfree(label);

	return(selection->back);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_float_sel
|
|       Purpose: Creates a floating point selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of float selection being created 
|
|        Output: Returns backplane of float selection object.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_float_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *label, *object_name;
	char         buff[4*KLENGTH]; 
	float        text_width;
	xvobject     offset = NULL;

        /*
         * get info from UIS line
         */
	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_float_line(selection->line, &line_info))) return(NULL);

        /*
         * generate a unique name for selection (for journal record/playback)
         */
	object_name = xvf_create_gen_selname(line_info.variable, 
				selection->back_toggle, &toggle_count);

        /*
         * create float selection backplane
         */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
				  line_info.width, line_info.height, 
				  object_name, 0);
	xvw_set_attribute(selection->back, XVW_CHAR_MAX_HEIGHT, 1.0);
	kfree(object_name);

	/*
         * create optional button if necessary
         */
	if (((line_info.optional) ||
             (selection->back_toggle != NULL)) &&
	     (line_info.opt_sel != 2))
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
	    xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
                             xvf_gen_opt_cb, (kaddr) selection->back_kformstruct);
	    offset = selection->opt_object;
	}
	selection->opt_selected = line_info.opt_sel;

	/*
         * put in label
         */
	if (line_info.title != NULL)
	    label= kstrdup(line_info.title);
	else label = kstrdup(" ");

	selection->label_object  = xvf_create_label(selection->back, 
					line_info.xpos, line_info.ypos, 
					offset, label, "label", TRUE, FALSE);
	/*
         * create the ascii text object
         */
	xvf_set_text_buffer(buff, &line_info);
   
        /*
         * if this float selection is NOT a toggle member,
         * create the text object, the scrollbar (if applicable),
         * and the <cr> pixmap (if applicable).
         */
	if (selection->back_toggle == NULL)
	{
	    /*
             * unbounded float or no scrollbar wanted =>
             * use up remainder of space on text object
             */
	    if  ((line_info.upper_float == line_info.lower_float) ||
		 (line_info.special != 1))
	    {
		text_width = xvf_calculate_remaining_width(line_info.width,
				line_info.optional, line_info.opt_sel, 
				line_info.title);
	    }
            /*
             * bounded float w/ potential scrollbar =>
             * calculate space needed for text object
             */
	    else
	    {
		 text_width =
                        xvf_calculate_text_width((double) line_info.upper_float,
                                                 (double) line_info.lower_float,
                                                  line_info.precision,
                                                  KUIS_FLOAT);
	    }

            /*
             * create the text object - always at least width 1
             */
	    if (text_width <= 0) text_width = 1;

	    selection->value_object= xvf_create_text(selection->back, 
					text_width, line_info.height, 
					selection->label_object, buff, FALSE);
	    xvw_set_attribute(selection->value_object, 
			      XVW_CHAR_MAX_HEIGHT, 1.0);

	    /* 
	     * bounded float with space left bar and 'mech' set to TRUE
	     */
            xvw_add_action(selection->value_object, "<KeyRelease>",
                           xvf_mark_modified, selection, FALSE);

	    if ((line_info.upper_float != line_info.lower_float) && 
                (line_info.special == 1))
                 xvf_append_selection_scrollbar(selection, &line_info);

            /*
             * "live" selections have the <cr> pixmap
             */
            if (line_info.live)
            {
                if (selection->scroll != NULL)
                {
                    selection->pix_object =
                                xvf_create_cr_pixmap(selection->back);
                    xvw_set_attribute(selection->scroll, XVW_LEFT_OF,
                                      selection->pix_object);
                }
                else
                {
                    selection->pix_object =
                                xvf_create_cr_pixmap(selection->back);
                    xvw_set_attribute(selection->value_object,
                                      XVW_LEFT_OF, selection->pix_object);
                }
            }
            else
            {
                if (selection->scroll != NULL)
                    xvw_set_attribute(selection->scroll, XVW_LEFT_OF, NULL);
                else xvw_set_attribute(selection->value_object,
                                       XVW_LEFT_OF, NULL);
            }
            xvw_add_action(selection->value_object, "<Key>Return",
                          xvf_return_live_selection, selection, TRUE);
	}

        /*
         * desensitize backplane if requested
         */
	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

        /*
         *  desensitize unselected optional selection if user likes it
         */
	if (selection->back_toggle != NULL)
	{
	    if (!selection->back_toggle->opt_selected) 
	    {
	        xvf_set_optional_sensitivity(selection, FALSE);
		if (xvf_desensitize_optionals)
	            xvw_sensitive(selection->opt_object, FALSE);
	    }
	}
        else if (!line_info.opt_sel)
            xvf_set_optional_sensitivity(selection, selection->opt_selected);

	/*
	 *  Set the internal menuform for a Float selection...
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);

	if (selection->back_toggle == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Float.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Float.pane");
	}
	else
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Togglef.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Togglef.pane");
	}

        /*
         * free up memory & return
         */
	kvf_free_line_info_strings(&line_info);
	kfree(label);
	return(selection->back);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_double_sel
|
|       Purpose: Creates a double point precision selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of double selection being created 
|
|        Output: Returns backplane of double selection object.
|          Date: Apr 30, 1993
|    Written By: Danielle Argiro 
| Modifications: 
|
------------------------------------------------------------*/

xvobject xvf_create_double_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *label, *object_name;
	char         buff[4*KLENGTH]; 
	float        text_width;
	xvobject     offset = NULL;

        /*
         * get info from UIS line
         */
	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_double_line(selection->line, &line_info))) return(NULL);

        /*
         * generate a unique name for selection (for journal record/playback)
         */
	object_name = xvf_create_gen_selname(line_info.variable, 
				selection->back_toggle, &toggle_count);

        /*
         * create double selection backplane
         */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
				  line_info.width, line_info.height, 
				  object_name, 0);
	xvw_set_attribute(selection->back, XVW_CHAR_MAX_HEIGHT, 1.0);
	kfree(object_name);

	/*
         * create optional button if necessary
         */
	if (((line_info.optional) ||
             (selection->back_toggle != NULL)) &&
	     (line_info.opt_sel != 2))
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
	    xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
                             xvf_gen_opt_cb, (kaddr) selection->back_kformstruct);
	    offset = selection->opt_object;
	}
	selection->opt_selected = line_info.opt_sel;

        /*
         * put in label
         */
	if (line_info.title != NULL)
	    label= kstrdup(line_info.title);
	else label = kstrdup(" ");

	selection->label_object = xvf_create_label(selection->back, 
					line_info.xpos, line_info.ypos, 
					offset, label, "label", TRUE, 
				        FALSE);
	/*
         * create the ascii text object
         */
	xvf_set_text_buffer(buff, &line_info);
  
        /*
         * if this double selection is NOT a toggle member,
         * create the text object, the scrollbar (if applicable),
         * and the <cr> pixmap (if applicable).
         */
	if (selection->back_toggle == NULL)
	{
            /*
             * unbounded double or no scrollbar wanted =>
             * use up remainder of space on text object
             */
            if  ((line_info.upper_double == line_info.lower_double) ||
                 (line_info.special != 1))
            {
		text_width = xvf_calculate_remaining_width(line_info.width,
				line_info.optional, line_info.opt_sel, 
				line_info.title);
            }
            /*
             * bounded double w/ potential scrollbar =>
             * calculate space needed for text object
             */
            else
            {
                 text_width =
                        xvf_calculate_text_width(line_info.upper_double,
                                                 line_info.lower_double,
                                                 line_info.precision,
                                                 KUIS_DOUBLE);
            }

            /*
             * create the text object - always at least width 1
             */
            if (text_width <= 0) text_width = 1;

            selection->value_object= xvf_create_text(selection->back,
                                        text_width, line_info.height,
                                        selection->label_object, buff, FALSE);
	    xvw_set_attribute(selection->value_object, 
			      XVW_CHAR_MAX_HEIGHT, 1.0);
            /*
             * bounded double with space left bar and 'mech' set to TRUE
             */
            xvw_add_action(selection->value_object, "<KeyRelease>",
                           xvf_mark_modified, selection, FALSE);

            if ((line_info.upper_double != line_info.lower_double) &&
                (line_info.special == 1))
                 xvf_append_selection_scrollbar(selection, &line_info);

            /*
             * "live" selections have the <cr> pixmap
             */
            if (line_info.live)
            {
                if (selection->scroll != NULL)
                {
                    selection->pix_object =
                                xvf_create_cr_pixmap(selection->back);
                    xvw_set_attribute(selection->scroll, XVW_LEFT_OF,
                                      selection->pix_object);
                }
                else
                {
                    selection->pix_object =
                                xvf_create_cr_pixmap(selection->back);
                    xvw_set_attribute(selection->value_object,
                                      XVW_LEFT_OF, selection->pix_object);
                }
            }
            else
            {
                if (selection->scroll != NULL)
                    xvw_set_attribute(selection->scroll, XVW_LEFT_OF, NULL);
                else xvw_set_attribute(selection->value_object,
                                       XVW_LEFT_OF, NULL);
            }
            xvw_add_action(selection->value_object, "<Key>Return",
                          xvf_return_live_selection, selection, TRUE);
	}

        /*
         * desensitize backplane if requested
         */
	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

        /*
         *  desensitize unselected optional selection if user likes it
         */
	if (selection->back_toggle != NULL)
	{
	    if (!selection->back_toggle->opt_selected) 
	    {
	        xvf_set_optional_sensitivity(selection, FALSE);
		if (xvf_desensitize_optionals)
	            xvw_sensitive(selection->opt_object, FALSE);
	    }
	}
        else if (!line_info.opt_sel)
            xvf_set_optional_sensitivity(selection, selection->opt_selected);

	/*
	 *  Set the internal menuform for a Double selection...
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_toggle == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Double.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Double.pane");
	}
	else 
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Toggleh.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Toggleh.pane");
	}

        /*
         * free up memory & return
         */
	kvf_free_line_info_strings(&line_info);
	kfree(label);
	return(selection->back);
}

/*------------------------------------------------------------
|
|  Routine Name: xvf_create_logic_sel
|
|       Purpose: Creates a logical selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of logical selection being created 
|
|        Output: Returns backplane of logical selection object.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_logic_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *object_name, *boolean_label, *label;
	float        width;
	xvobject     offset = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_logic_line(selection->line,&line_info))) return(NULL);
	width = line_info.width;
   
	object_name = xvf_create_gen_selname(line_info.variable, 
			selection->back_toggle, &toggle_count);

	/* create logical selection backplane object */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
				             line_info.width, line_info.height, 
				             object_name, 0);
	kfree(object_name);

	/*
	 * create optional button if necessary
	 */
	if (((line_info.optional) ||
             (selection->back_toggle != NULL)) &&
	     (line_info.opt_sel != 2))
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
	    xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
                             xvf_gen_opt_cb, (kaddr) selection->back_kformstruct);
	    offset = selection->opt_object;
            width -= 1.0;
	}

	/*
	 * put in label
	 */
	if (line_info.title != NULL)
	    label= kstrdup(line_info.title);
	else label = kstrdup(" ");

	selection->label_object  = xvf_create_label(selection->back, 0.0, 0.0, 
 			                            offset, label, "label", 
					            TRUE, FALSE);

	if (selection->back_toggle == NULL)
	{
	    /*
	     *  determine which string to display as value
	     */
	    if (line_info.logical_val)
	    {
		if (line_info.val_labels[1] != NULL)
		    boolean_label = kstrdup(line_info.val_labels[1]);
		else boolean_label = kstrdup(" ");
	    }
	    else	
	    {
		if (line_info.val_labels[0] != NULL)
		    boolean_label = kstrdup(line_info.val_labels[0]);
		else boolean_label = kstrdup(" ");
	    }
	    width = kmax(kstrlen(line_info.val_labels[0]),
			kstrlen(line_info.val_labels[1])) + 1.0;
	    width += 1.0;
	    if (width < 4.0) width = 4.0;

	    /* add actual value object */
	    selection->value_object  = 
			xvf_create_boolean_button(selection->back, width, 
				line_info.height, selection->label_object,
                                boolean_label, "logic_button");
	    xvw_insert_callback(selection->value_object, XVW_BUTTON_SELECT,
		  FALSE, xvf_logic_val_cb, (kaddr) selection->back_kformstruct);
	    kfree(boolean_label);
	    if (line_info.live)
	    {
		selection->pix_object = 
			xvf_create_cr_pixmap(selection->back);
	        xvw_set_attribute(selection->value_object, XVW_LEFT_OF, 
				  selection->pix_object);
	    }
	    else xvw_set_attribute(selection->value_object, XVW_LEFT_OF, NULL);

	}

	selection->opt_selected = line_info.opt_sel;

	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

	if (selection->back_toggle != NULL)
	{
	    if (!selection->back_toggle->opt_selected) 
	    {
	        xvf_set_optional_sensitivity(selection, FALSE);
		if (xvf_desensitize_optionals)
	            xvw_sensitive(selection->opt_object, FALSE);
	    }
	}
        else if (!line_info.opt_sel)
            xvf_set_optional_sensitivity(selection, selection->opt_selected);

	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for a Logic selection...
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_toggle == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		  "$DESIGN/objects/library/xvforms/uis/xvf_edit/Logic.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		  "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Logic.pane");
	}
	else 
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		  "$DESIGN/objects/library/xvforms/uis/xvf_edit/Togglel.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		  "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Togglel.pane");
	}
	return(selection->back);
}



/*------------------------------------------------------------
|
|  Routine Name: xvf_create_flag_sel
|
|       Purpose: Creates a flag selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of logical selection being created 
|
|        Output: Returns backplane of flag selection object.
|          Date: June 26, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_flag_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *object_name, *label;
	float        width;
	xvobject     offset = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_flag_line(selection->line,&line_info))) return(NULL);
	width = line_info.width;
   
	object_name = xvf_create_gen_selname(line_info.variable, 
			selection->back_toggle, &toggle_count);

	/* create logical selection backplane object */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
				  line_info.width, line_info.height, 
				  object_name, 0);
	xvw_set_attribute(selection->back, XVW_CHAR_MAX_HEIGHT, 1.0);
	kfree(object_name);

	/*
	 * create optional button if necessary
	 */
	if ((line_info.optional) || (selection->back_toggle != NULL)) 
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
	    xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
                          xvf_gen_opt_cb, (kaddr) selection->back_kformstruct);
	    offset = selection->opt_object;
            width -= 1.0;
	}

	/*
	 * put in label
	 */
	if (line_info.title != NULL)
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	selection->label_object = xvf_create_label(selection->back, 0.0, 0.0,
                                                   offset, label, "label",
                                                   TRUE, FALSE);
	if (line_info.live)
	{
	    selection->pix_object = 
			xvf_create_cr_pixmap(selection->back);
	    xvw_set_attribute(selection->label_object, XVW_LEFT_OF, 
			      selection->pix_object);
	}

	selection->value_object = selection->label_object;
	selection->opt_selected = line_info.opt_sel;

	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

	if (selection->back_toggle != NULL)
	{
	    if (!selection->back_toggle->opt_selected) 
	    {
	        xvf_set_optional_sensitivity(selection, FALSE);
		if (xvf_desensitize_optionals)
	            xvw_sensitive(selection->opt_object, FALSE);
	    }
	}

	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for a Flag selection...
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_toggle == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		  "$DESIGN/objects/library/xvforms/uis/xvf_edit/Flag.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		  "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Flag.pane");
	}
	else 
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Togglet.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Togglet.pane");
	}
	return(selection->back);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_string_sel
|
|       Purpose: Creates a string selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of string selection being created 
|
|        Output: Returns backplane of string selection object.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/


xvobject xvf_create_string_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *object_name, temp[10*KLENGTH], *label;
	float        width;
	int          spec;
	xvobject     offset = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_string_line(selection->line, &line_info))) return(NULL);
	width = line_info.width;
   
	object_name = xvf_create_gen_selname(line_info.variable, 
			selection->back_toggle, &toggle_count);

	/* create string selection backplane object */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
				  line_info.width, line_info.height, 
				  object_name, 0);
	kfree(object_name);

	/* create optional button */
	 if (((line_info.optional) ||
             (selection->back_toggle != NULL)) &&
             (line_info.opt_sel != 2))
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
	    xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
                          xvf_gen_opt_cb, (kaddr) selection->back_kformstruct);
	    offset = selection->opt_object;
            width -= 1.0;
	}

	/* put in label */
	if (line_info.title != NULL)
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	selection->label_object = 
		xvf_create_label(selection->back, line_info.xpos, 
		 	         line_info.ypos, offset, label, "label", 
				 TRUE, FALSE);
	width -= (float) kstrlen(label);

	/*
	 * create the ascii text object
	 */
	xvf_set_text_buffer(temp, &line_info);

	if (selection->back_toggle == NULL)
	{
	    if (line_info.height > 1) spec = TRUE; else spec = FALSE;
	    selection->value_object = 
		xvf_create_text(selection->back, width, line_info.height, 
				selection->label_object, temp, spec);

            xvw_add_action(selection->value_object, "<KeyRelease>",
                           xvf_mark_modified, selection, FALSE);

	    if (line_info.live)
	    {
		selection->pix_object = 
			xvf_create_cr_pixmap(selection->back);
	        xvw_set_attribute(selection->value_object, XVW_LEFT_OF, 
				  selection->pix_object);
	    }
	    else xvw_set_attribute(selection->value_object, XVW_LEFT_OF, NULL);

	    if (!spec)
               xvw_add_action(selection->value_object, "<Key>Return",
                          xvf_return_live_selection, selection, TRUE);
	}
   
	selection->opt_selected = line_info.opt_sel;

	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

	if (selection->back_toggle != NULL)
	{
	    if (!selection->back_toggle->opt_selected)
	    {
	    	xvf_set_optional_sensitivity(selection, FALSE);
		if (xvf_desensitize_optionals)
	  	    xvw_sensitive(selection->opt_object, FALSE);
	    }
	}
        else if (!line_info.opt_sel)
            xvf_set_optional_sensitivity(selection, selection->opt_selected);

	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for a String selection...
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_toggle == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/String.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/String.pane");
	}
	else
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Toggles.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Toggles.pane");
	}
	return(selection->back);
}



/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_stringlist_sel
|
|       Purpose: Creates a string list selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of stringlist selection being created 
|
|        Output: Returns backplane of stringlist selection object.
|          Date: Sep 17, 1992
|    Written By: Danielle Argiro
| Modifications: 
|
------------------------------------------------------------*/


xvobject xvf_create_stringlist_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	float        list_width, width;
	int          i, spec;
	char         temp[10*KLENGTH];
	char         *object_name, *label; 
	xvobject     offset = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_stringlist_line(selection->line, &line_info))) 
	    return(NULL);
	width = line_info.width;
   
	object_name = xvf_create_gen_selname(line_info.variable, 
			selection->back_toggle, &toggle_count);

	/* create string selection backplane object */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
				  line_info.width, line_info.height, 
				  object_name, 0);
	kfree(object_name);

	/* create optional button */
	if ((line_info.optional) && (line_info.opt_sel != 2))
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
	    xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
                         xvf_gen_opt_cb, (kaddr) selection->back_kformstruct);
	    offset = selection->opt_object;
            width -= 1.0;
	}

	/* put in label */
	if (line_info.title != NULL)
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	list_width = (float) kstrlen(label) + 3.0;

	selection->label_object = 
		xvf_create_menubutton(selection->back, offset, 0.0, 0.0, 
				      list_width, 1.0, label, "strlist_menu", 
				      line_info.val_labels,
			              line_info.list_num, 
				      &selection->submenu_buttons);
	selection->submenu = xvw_retrieve_menu(selection->label_object);

	for (i = 0; i < line_info.list_num; i++)
	    xvw_insert_callback(selection->submenu_buttons[i],
			     XVW_BUTTON_SELECT, FALSE, xvf_stringlist_val_cb, 
			     (kaddr) selection->back_kformstruct);

	width -= list_width;

	/*
	 * create the ascii text object
	 */
	xvf_set_text_buffer(temp, &line_info);

	if (line_info.height > 1) spec = TRUE; else spec = FALSE;
	selection->value_object = 
		xvf_create_text(selection->back, width, line_info.height, 
				selection->label_object, temp, spec);

        xvw_add_action(selection->value_object, "<KeyRelease>",
                       xvf_mark_modified, selection, FALSE);

	if (line_info.live)
	{
	    selection->pix_object = 
			xvf_create_cr_pixmap(selection->back);
	    xvw_set_attribute(selection->value_object, XVW_LEFT_OF, 
			      selection->pix_object);
	}
	else xvw_set_attribute(selection->value_object, XVW_LEFT_OF, NULL);
 
	if (!spec)
            xvw_add_action(selection->value_object, "<Key>Return",
                          xvf_return_live_selection, selection, TRUE);

	selection->opt_selected = line_info.opt_sel;

	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

	if (!line_info.opt_sel)
            xvf_set_optional_sensitivity(selection, selection->opt_selected);

	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for a StringList selection...
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
        if (selection->back_form->editable)
            xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
	      "$DESIGN/objects/library/xvforms/uis/xvf_edit/StringList.pane");
        else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE,      
	      "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/StringList.pane");

	return(selection->back);
}



/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_cycle_sel
|
|       Purpose: Creates a cycle selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of cycle selection being created 
|
|        Output: Returns backplane of cycle selection object.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_cycle_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info  line_info;
	char       object_name[KLENGTH], *choice_label, *label;
	float      width;
	int        val_index;
	xvobject   offset = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_cycle_line(selection->line,&line_info))) return(NULL);
	width = line_info.width;
   
	ksprintf(object_name,"%s",line_info.variable);

	/* create cycle selection backplane object */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
				  line_info.width, line_info.height, 
				  object_name, 0);

	/* create optional button if necessary */
	if ((line_info.optional) && (line_info.opt_sel != 2))
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
            xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
		     xvf_gen_opt_cb, (kaddr) selection->back_kformstruct); 
	    offset = selection->opt_object;
            width -= 1.0;
	}

	/*
	 * put in label
	 */
	if (line_info.title != NULL)
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	selection->label_object = 
			xvf_create_label(selection->back, line_info.xpos, 
			                 line_info.ypos, offset, label, 
				         "label", TRUE, FALSE);

	/* 
	 *  add actual value object 
	 */
	if (line_info.list_num > 0)
	{
	    val_index = line_info.list_val - line_info.int_val;
	    choice_label = kstrdup(line_info.val_labels[val_index]);
	}
	else choice_label  = kstrdup(" ");

	width = xvf_list_button_width(line_info.val_labels, 
				      line_info.list_num);
	selection->value_object = 
			xvf_create_boolean_button(selection->back, width, 
				line_info.height, selection->label_object, 
				choice_label, "cycle_button");
	if (line_info.list_num > 0)
	    xvw_insert_callback(selection->value_object, XVW_BUTTON_SELECT,
    	 	  FALSE, xvf_cycle_val_cb, (kaddr) selection->back_kformstruct); 

	if (line_info.live)
	{
	    selection->pix_object = 
		xvf_create_cr_pixmap(selection->back);
	    xvw_set_attribute(selection->value_object, XVW_LEFT_OF, 
			      selection->pix_object);
	}
	else xvw_set_attribute(selection->value_object, XVW_LEFT_OF, NULL);

	selection->opt_selected = line_info.opt_sel;

	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

	if (selection->back_toggle != NULL)
	{
	    if (!selection->back_toggle->opt_selected) 
	    {
	        xvf_set_optional_sensitivity(selection, FALSE);
		if (xvf_desensitize_optionals)
	            xvw_sensitive(selection->opt_object, FALSE);
	    }
	}
        else if (!line_info.opt_sel)
            xvf_set_optional_sensitivity(selection, selection->opt_selected);

	kvf_free_line_info_strings(&line_info);
	kfree(label);  
	kfree(choice_label);

	/*
	 *  Set the internal menuform for an Cycle selection...
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_form->editable)
	    xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Cycle.pane");
	else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Cycle.pane");

	return(selection->back);
}

 
/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_routine_sel
|
|       Purpose: Creates a routine button.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of routine button being created 
|
|        Output: Returns object which will serve as routine button 
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_routine_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *label; 
	char         object_name[KLENGTH];
	xvobject     button = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_routine_line(selection->line, &line_info)))
	    return(NULL);

	ksprintf(object_name,"%s",line_info.routine);
   
	if (line_info.title != NULL)
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	if (selection->back_submenu != NULL)
	    button = xvf_create_menu_button(parent, 
	 			label, object_name);
	else
	    button = xvf_create_button(parent, line_info.x, 
				line_info.y, line_info.width, line_info.height, 
				label, object_name);

	xvw_insert_callback(button, XVW_BUTTON_SELECT, TRUE, xvf_routine_cb,
			 (kaddr) selection->back_kformstruct);

	if (!line_info.activate)
	    xvw_sensitive(button, FALSE);

	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for a Routine button...
	 */
	xvw_set_attribute(button, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_form->editable)
	    xvw_set_attribute(button, XVW_MENU_FORMFILE, 
		"$DESIGN/objects/library/xvforms/uis/xvf_edit/Routine.pane");
	else xvw_set_attribute(button, XVW_MENU_FORMFILE, 
		"$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Routine.pane");

	return(button);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_toggle_sel
|
|       Purpose: Creates the backplane for a toggle group.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of toggle backplane being created 
|
|        Output: Returns backplane for toggle group.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_toggle_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *label;
	float        width;
	xvobject     offset = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_toggle_line(selection->line, &line_info))) return(NULL);
	width = line_info.width;

	/* create toggle selection backplane object */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
				  line_info.width, line_info.height, 
				  line_info.variable, 0);

	/* create optional button */
	if ((line_info.optional) && (line_info.opt_sel != 2))
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
	    xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
		     xvf_gen_opt_cb, (kaddr) selection->back_kformstruct); 
            offset = selection->opt_object;
	    width -= 1.0;
	}

	/*
         * put in label
         */
	if (line_info.title != NULL)
            label = kstrdup(line_info.title);
        else label = kstrdup(" ");

	selection->label_object = 
			xvf_create_label(selection->back, line_info.xpos, 
			                 line_info.ypos, offset, label, 
                                         "label", TRUE, TRUE);
	xvw_set_attribute(selection->label_object, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);

	selection->opt_selected = line_info.opt_sel;
	selection->toggle_num   = line_info.toggle_val;

	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

	toggle_count = 0;

	kvf_free_line_info_strings(&line_info);

	/*
	 *  Set the internal clientdata
	 */
	if (selection->back != NULL)
	{
	    xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
			      selection->back_kformstruct);
	    if (selection->back_form->editable)
	        xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                   "$DESIGN/objects/library/xvforms/uis/xvf_edit/Toggle.pane");
	    else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
                   "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Toggle.pane");
	}

	kfree(label);
	return(selection->back);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_blank_sel
|
|       Purpose: Creates a blank selection.
|
|         Input: selection - ptr to the selection struct
|                parent    - parent of blank selection being created 
|
|        Output: Returns object which will serve as blank selection.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_blank_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *label = NULL;
	xvobject     blank  = NULL;

	kvf_clear_line_info(&line_info);
	kvf_parse_blank_line(selection->line, &line_info);
   
	/*
         * put in label
         */
	if (line_info.title != NULL)
            label = kstrdup(line_info.title);
        else label = kstrdup(" ");

	/* create label object */
	if (selection->back_submenu != NULL)
	    blank = xvf_create_menu_label(parent,
				label, line_info.variable);
	else blank = xvf_create_label(parent, line_info.xpos, line_info.ypos,
				      NULL, label, label, TRUE, TRUE);

	xvw_set_attribute(blank, XVW_MENUABLE, TRUE);

	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for a Blank selection...
	 */
 	xvw_set_attribute(blank, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_submenu == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(blank, XVW_MENU_FORMFILE, 
		   "$DESIGN/objects/library/xvforms/uis/xvf_edit/Blank.pane");
	    else xvw_set_attribute(blank, XVW_MENU_FORMFILE, 
		   "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Blank.pane");
	}
	return(blank);
}




/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_list_sel
|
|       Purpose: Creates a list selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of list selection being created 
|
|        Output: Returns button which will bring up list selection.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_list_sel(
   kselection *selection,
   xvobject   parent)
{
	int          i, val_index;
	Line_Info    line_info;
	char         *choice_label, *label;
	float        width;
	xvobject     offset = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_list_line(selection->line, &line_info))) return(NULL);
	width = line_info.width;
   
	/* create list selection backplane object */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
				  line_info.width, line_info.height, 
				  line_info.variable, 0);

	/*
	 * create optional button if necessary
	 */
	if ((line_info.optional) && (line_info.opt_sel != 2))
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
            xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
                          xvf_gen_opt_cb, (kaddr) selection->back_kformstruct); 
	    offset = selection->opt_object;
	    width -= 1.0;
	}

	/*
	 * put in label
	 */
	if (line_info.title != NULL)
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	selection->label_object = 
			xvf_create_label(selection->back, line_info.xpos, 
			                 line_info.ypos, offset, label, 
					 "label", TRUE, FALSE);
	width -= (float) kstrlen(label);

	/* 
	 *  add actual value object 
	 */
	if (line_info.list_num > 0)
	{
	    val_index = line_info.list_val -  line_info.int_val;
	    choice_label = kstrdup(line_info.val_labels[val_index]);
	}
	else choice_label = kstrdup(" ");

	width = xvf_list_button_width(line_info.val_labels, 
				      line_info.list_num);

	selection->value_object 
			= xvf_create_menubutton(selection->back, 
				     selection->label_object, 0.0, 
				     0.0, width, line_info.height, choice_label,
	                             "list_button", line_info.val_labels,
				     line_info.list_num, 
				     &selection->submenu_buttons);
	selection->submenu = xvw_retrieve_menu(selection->value_object);


	for (i = 0; i < line_info.list_num; i++)
	    xvw_insert_callback(selection->submenu_buttons[i], 
			     XVW_BUTTON_SELECT, FALSE, xvf_list_val_cb, 
			     (kaddr) selection->back_kformstruct);

	if (line_info.live)
	{
	    selection->pix_object = xvf_create_cr_pixmap(selection->back);
	    xvw_set_attribute(selection->value_object, XVW_LEFT_OF, 
			      selection->pix_object);
	}
	else xvw_set_attribute(selection->value_object, XVW_LEFT_OF, NULL);

	selection->opt_selected = line_info.opt_sel;

	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

	if (selection->back_toggle != NULL)
	{
	    if (!selection->back_toggle->opt_selected) 
	    {
	        xvf_set_optional_sensitivity(selection, FALSE);
		if (xvf_desensitize_optionals)
	            xvw_sensitive(selection->opt_object, FALSE);
	    }
	}
        else if (!line_info.opt_sel)
            xvf_set_optional_sensitivity(selection, selection->opt_selected);

	kvf_free_line_info_strings(&line_info);
	kfree(label);  
	kfree(choice_label);

	/*
	 *  Set the internal menuform for a List selection...
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
 			  selection->back_kformstruct);
	if (selection->back_form->editable)
	    xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
	      "$DESIGN/objects/library/xvforms/uis/xvf_edit/List.pane");
	else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
              "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/List.pane");

	return(selection->back);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_displaylist_sel
|
|       Purpose: Creates a displaylist selection.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of list selection being created 
|
|        Output: Returns button which will bring up list selection.
|          Date: January 18, 1994
|    Written By: Danielle Argiro 
| Modifications: 
|
------------------------------------------------------------*/

xvobject xvf_create_displaylist_sel(
   kselection *selection,
   xvobject   parent)
{
	int          val_index;
	Line_Info    line_info;
	char         *label;
	xvobject     offset = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_displaylist_line(selection->line, &line_info))) return(NULL);
	/* create displaylist selection backplane object */
	selection->back = xvf_create_selback(parent, line_info.x, line_info.y, 
				             line_info.width, line_info.height, 
				             line_info.variable, 0);
	/*
	 * create optional button if necessary
	 */
	if ((line_info.optional) && (line_info.opt_sel != 2))
	{
	    selection->opt_object = xvf_create_optional_button(selection->back);
            xvw_insert_callback(selection->opt_object, XVW_BUTTON_SELECT, FALSE,
                          xvf_gen_opt_cb, (kaddr) selection->back_kformstruct); 
	    offset = selection->opt_object;
	}

	/*
	 * put in label
	 */
	if (line_info.title != NULL)
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	selection->label_object = 
			xvf_create_label(selection->back, line_info.xpos, 
			                 line_info.ypos, offset, label, 
					 "label", TRUE, FALSE);
	if (line_info.live)
	{
	    selection->pix_object = xvf_create_cr_pixmap(selection->back);
	    xvw_set_attribute(selection->pix_object, XVW_RIGHT_OF, 
			      selection->label_object);
	}

	/* 
	 *  add actual value object 
	 */
	selection->value_object = 
		xvw_create_list(selection->back, "displaylist");
	xvw_set_attributes(selection->value_object,
			   XVW_BELOW,      selection->label_object,
			   XVW_RIGHT_OF,   NULL,
			   XVW_LEFT_OF,    NULL,
			   XVW_ABOVE,      NULL,
			   XVW_TACK_EDGE,  KMANAGER_TACK_ALL,
			   NULL);
	selection->submenu =  xvw_retrieve_list(selection->value_object);

	if ((line_info.val_labels != NULL) && (line_info.list_num > 0))
	    xvw_change_list(selection->submenu, line_info.val_labels, 
			    line_info.list_num, TRUE);

	xvw_set_attributes(selection->label_object, 
			   XVW_RIGHT_OF, NULL,
			   XVW_LEFT_OF, NULL,
			   NULL);

	if (line_info.list_num > 0)
	{
	    val_index = line_info.list_val -  line_info.int_val;
	    if (val_index > -1)
	        xvw_set_attribute(selection->submenu, 
			          XVW_LIST_HIGHLT_ELEM, val_index);
	}

	/* callback added for single click */
	xvw_insert_callback(selection->submenu, XVW_LIST_ITEM_SELECT,
                         FALSE, xvf_displaylist_val_cb,
			 (kaddr) selection->back_kformstruct);

	/* callback also added on double click */
	if (line_info.special)
	    xvw_insert_callback(selection->submenu, XVW_LIST_ITEM_ACTION,
                             FALSE, xvf_displaylist_val_doubleclick_cb,
                             (kaddr) selection->back_kformstruct);

	selection->opt_selected = line_info.opt_sel;

	if (!line_info.activate)
	    xvw_sensitive(selection->back, FALSE);

        if (!line_info.opt_sel)
            xvf_set_optional_sensitivity(selection, selection->opt_selected);

	kvf_free_line_info_strings(&line_info);
	kfree(label);  

	/*
	 *  Set the internal menuform for a DisplayList selection...
	 */
	xvw_set_attribute(selection->back, XVW_MENU_CLIENTDATA, 
 			  selection->back_kformstruct);
	if (selection->back_form->editable)
	    xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
	      "$DESIGN/objects/library/xvforms/uis/xvf_edit/DisplayList.pane");
	else xvw_set_attribute(selection->back, XVW_MENU_FORMFILE, 
             "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/DisplayList.pane");

	return(selection->back);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_pane_action_sel
|
|       Purpose: Creates a pane action button.
|
|         Input: selection - ptr to the selection struct
|		 parent    - parent of pane action button being created 
|
|        Output: Returns object which serves as pane action button.
|          Date: Apr 14, 1992
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

xvobject xvf_create_pane_action_sel(
   kselection *selection,
   xvobject   parent)
{
	Line_Info    line_info;
	char         *label;
	xvobject     button = NULL;

	kvf_clear_line_info(&line_info);
	if (!(kvf_parse_pane_action_line(selection->line, &line_info)))
	    return(NULL);

	if (line_info.title != NULL)
	    label = kstrdup(line_info.title);
	else label = kstrdup(" ");

	if (selection->back_submenu != NULL)
	    button = xvf_create_menu_button(parent, 
	 			label, line_info.variable);
	else
	    button  = xvf_create_button(parent, line_info.x, 
				line_info.y, line_info.width, line_info.height, 
				label, line_info.variable);

	xvw_insert_callback(button, XVW_BUTTON_SELECT, FALSE, xvf_action_cb,
		         (kaddr) selection->back_kformstruct); 

	if (!line_info.activate)
	    xvw_sensitive(button, FALSE);

	kvf_free_line_info_strings(&line_info);
	kfree(label);

	/*
	 *  Set the internal menuform for a pane action...
	 */
	xvw_set_attribute(button, XVW_MENU_CLIENTDATA, 
			  selection->back_kformstruct);
	if (selection->back_submenu == NULL)
	{
	    if (selection->back_form->editable)
	        xvw_set_attribute(button, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_edit/Action.pane");
	    else xvw_set_attribute(button, XVW_MENU_FORMFILE, 
		 "$DESIGN/objects/library/xvforms/uis/xvf_nonedit/Action.pane");
	}
	return(button);
}
