/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>> General & Sundry Utilities for GUI Creation           <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>               xvf_create_gen_selname()                <<<<
   >>>>               xvf_set_text_buffer()                   <<<<
   >>>>               xvf_highlight_objects()                 <<<<
   >>>>               xvf_highlight_toggle()                  <<<<
   >>>>               xvf_highlight_group()                   <<<<
   >>>>               xvf_object_creation_error_mesg()        <<<<
   >>>>               xvf_calculate_text_width()              <<<<
   >>>>               xvf_calculate_remaining_width()         <<<<
   >>>>               xvf_calculate_sb_width()                <<<<
   >>>>               xvf_append_selection_scrollbar()        <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_gen_selname()
|
|       Purpose: Creates the name for the value object of
|                a UIS selection.  The name is calculated if
|                the selection is a member of a toggle; otherwise,
|                the variable name on the UIS line is used.
|
|         Input: variable      - the variable name from the UIS line
|                back_toggle   - non-NULL if this selection is part of toggle
|                toggle_count  - integer used in creating unique name
|
|        Output: toggle_count - increments the toggle count for next time
|                Returns the name for the value object of the selection
|
|          Date: Apr 10, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------------*/

char *xvf_create_gen_selname(
   char       *variable,
   kselection *back_toggle,
   int        *toggle_count)
{
        char buff[KLENGTH];
        char *object_name;

        if (back_toggle != NULL)
        {
            ksprintf(buff,"%d_back", *toggle_count);
            (*toggle_count)++;
        }
        else
        {
            ksprintf(buff,"%s", variable);
        }
        object_name = kstrdup(buff);
        return(object_name);
}


/*------------------------------------------------------------
|
|  Routine Name: xvf_set_text_buffer
|
|       Purpose: Sets the text buffer for a UIS Selection
|                so that the text buffer can be passed to
|                the ascii text object creation routines.
|                Note: result shows up as contents of text object.
|
|         Input: buff      - space for text buffer
|                line_info - line_info struct from UIS line
|        Output: returns the text buffer (filled)
|          Date: Apr 10, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
void xvf_set_text_buffer(
   char      *buff,
   Line_Info *line_info)
{
	char scale[KLENGTH];

        switch (line_info->typeflag)
        {
        case KUIS_INPUTFILE:
        case KUIS_OUTPUTFILE:
             if ((line_info->filename == NULL) &&
                 (line_info->file_def == NULL) &&
                 (line_info->literal == NULL))
                 buff[0] = '\0';
             else if (line_info->literal != NULL)
                 ksprintf(buff, "%s", line_info->literal);
             else if (line_info->filename != NULL)
                 ksprintf(buff, "%s", line_info->filename);
             else
                 ksprintf(buff, "%s", line_info->file_def);
             break;

        case KUIS_INTEGER:
             if (line_info->literal != NULL)
                 ksprintf(buff,"%s", line_info->literal);
             else
                 ksprintf(buff,"%d",line_info->int_val);
             break;

        case KUIS_FLOAT:
             if (line_info->literal != NULL)
                 ksprintf(buff,"%s", line_info->literal);
             else
	     {
		 if (line_info->precision == 0)
		     ksprintf(scale, "%%g");
		 else if (line_info->precision == -1)
		     ksprintf(scale, "%%f");
		 else ksprintf(scale, "%%.%df", line_info->precision);
                 ksprintf(buff, scale, line_info->float_val);
	     }
             break;

        case KUIS_DOUBLE:
             if (line_info->literal != NULL)
                 ksprintf(buff,"%s", line_info->literal);
             else
	     {
		 if (line_info->precision == 0)
		     ksprintf(scale, "%%g");
		 else if (line_info->precision == -1)
		     ksprintf(scale, "%%f");
		 else ksprintf(scale, "%%.%df", line_info->precision);
                 ksprintf(buff, scale, line_info->double_val);
	     }
             break;

        case KUIS_STRING:
        case KUIS_STRINGLIST:
             if ((line_info->string_val == NULL) &&
                 (line_info->string_def == NULL) &&
                 (line_info->literal == NULL))
                buff[0] = '\0';
             else if (line_info->literal != NULL)
                ksprintf(buff, "%s", line_info->literal);
             else if (line_info->string_val != NULL)
                ksprintf(buff, "%s", line_info->string_val);
	     else
                ksprintf(buff, "%s", line_info->string_def);
             break;
        }
}



/*------------------------------------------------------------
|
|  Routine Name: xvf_highlight_objects
|
|       Purpose: Highlights (turns on) appropriate objects
|                such as optional boxes on selections that are
|                "on" by default, the default selections of
|                toggles and mutually exclusive groups, etc.
|
|         Input: selection - pointer to selection list
|        Output: none
|      KUIS_COMMENTs: called by routines in createform.c
|          Date: Mar 27, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/

void xvf_highlight_objects(
   kselection *sel_list)
{
        kselection *selection;

        selection = sel_list;
        while (selection != NULL)
        {
            if ((selection->opt_selected != 0) &&
                (selection->opt_object != NULL))
                xvw_reverse_colors(selection->opt_object, TRUE);

            if (selection->toggle_next != NULL)
                xvf_highlight_toggle(selection);

            if (selection->group_next != NULL)
                xvf_highlight_group(selection->group_next);

            selection = selection->next;
        }
}

/*------------------------------------------------------------
|
|  Routine Name: xvf_highlight_toggle
|
|       Purpose: Highlights (turns on) optional boxes of
|                selections that are part of a toggle.
|
|         Input: selection - pointer to header of toggle
|        Output: none
|          Date: Nov 05, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
void xvf_highlight_toggle(
   kselection *toggle_start)
{
        kselection *toggle;

        toggle = toggle_start->toggle_next;

        while (toggle != NULL)
        {
            if ((toggle->opt_selected == TRUE) &&
                (toggle->opt_object != NULL))
                xvw_reverse_colors(toggle->opt_object, TRUE);
            toggle = toggle->next;
        }
}


/*------------------------------------------------------------
|
|  Routine Name: xvf_highlight_group
|
|       Purpose: Highlights (turns on) optional boxes of
|                selections that are part of a group.
|
|         Input: selection - pointer to header of group
|        Output: none
|          Date: Mar 27, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/

void xvf_highlight_group(
   kselection *group)
{
        while (group != NULL)
        {
            if ((group->opt_selected == TRUE) &&
                (group->opt_object != NULL))
                 xvw_reverse_colors(group->opt_object, TRUE);

            if (group->type == KUIS_TOGGLE)
               xvf_highlight_toggle(group);

            if (group->group_next != NULL)
                xvf_highlight_group(group->group_next);

            group = group->next;
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_object_creation_error_mesg
|
|       Purpose: Calls kerror to give error message upon
|                failure to create the object(s) that comprise
|                some element of the GUI.
|
|         Input: routine_name - name of calling routine
|                item         - name of GUI item that couldn't be created
|
|        Output: none
|          Date: Mar 27, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
void xvf_object_creation_error_mesg(
   char *routine_name,
   char *item)
{
        char mesg[KLENGTH];

        ksprintf(mesg, "Couldn't create %s; aborting form creation.", item);
	errno = KWIDGET_CREATION;
        kerror("xvforms", routine_name, mesg);
}

/*------------------------------------------------------------
|
|  Routine Name: xvf_calculate_text_width
|
|       Purpose: Computes the proper width of the text object for an
|                Integer, Float, or Double selection.
|
|         Input: upper      - upper bound of number, in double representation
|                lower      - lower bound of number, in double representation
|                precision  - precision of selection, from UIS line 
|                type       - type of selection, KUIS_INTEGER, KUIS_FLOAT,
|                                                or KUIS_DOUBLE
|        Output:
|       Returns: The width that the text object should be
|          Date: May 10, 1993
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
double xvf_calculate_text_width(
    double upper,
    double lower,
    int    precision,
    int    type)
{
	char temp[KLENGTH];
	char scale[KLENGTH];
	int  text_length;

	if (type == KUIS_INTEGER)
	{
	    ksprintf(scale, "%%d");
	    ksprintf(temp, scale, (int) upper);
	    text_length = kstrlen(temp);
	    ksprintf(temp, scale, (int) lower);
	    if (kstrlen(temp) > text_length)
               text_length = kstrlen(temp);
	    text_length += 2;
	}
	else 
	{
	    if (precision == 0)
                ksprintf(scale, "%%f");
            else if (precision == -1)
                ksprintf(scale, "%%f");
            else ksprintf(scale, "%%.%df", precision);

	    ksprintf(temp, scale, (double) upper);
	    text_length = kstrlen(temp);
	    ksprintf(temp, scale, (double) lower);
	    if (kstrlen(temp) > text_length)
               text_length = kstrlen(temp);
	    text_length += 2;
	}

	return( (double)text_length);
}
    
/*------------------------------------------------------------
|
|  Routine Name: xvf_calculate_remaining_width
|
|       Purpose: Computes the remaining width from the geometry
|                string of a selection, after width of the optional box
|                (if applicable) and the width of the label have
|                been subtracted.
|
|         Input: total_width  - total width from geometry string of UIS line
|                optional     - TRUE for optional sel's only (from UIS line)
|                opt_sel      - from opt_sel field of UIS line
|                title        - title of selection from UIS line
|        Output:
|       Returns: The remaining width in the geometry string
|          Date: May 10, 1993
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
double xvf_calculate_remaining_width(
	double total_width,
	int   optional,
	int   opt_sel,
	char  *title)
{
	double width;
	double subtract = 0.0;
        
        if ((optional) && (opt_sel != 2))
           subtract += 1.0;

        if (title == NULL)
           subtract += 1.0;
        else subtract += ((double) (kstrlen(title)));

	width = total_width - subtract;

	return(width);
}

/*------------------------------------------------------------
|
|  Routine Name: xvf_calculate_sb_width
|
|       Purpose: Computes the proper width of the scrollbar for an
|                Integer, Float, or Double selection.
|
|         Input: line_info  - line_info struct from UIS line
|                text_width - pass width of text object if known,
|                             zero or negative number otherwise
|        Output:
|       Returns: The width that the scrollbar should be
|          Date: May 10, 1993
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
double xvf_calculate_sb_width(
	double     total_width,
	int        optional,
	int        opt_sel,
	char       *title,
	double     upper,
	double     lower,
	int        precision,
	int        type)
{
	double  sb_width; 
	double  text_width;
	double  remaining_width;

	if ((type != KUIS_INTEGER) &&
	    (type != KUIS_FLOAT)   &&
	    (type != KUIS_DOUBLE))
	{
	    errno = KCALL;
	    kerror("xvforms", "xvf_calculate_sb_width",
	           "Incorrect selection type passed in");
	    return(0.0);
	}
	remaining_width = xvf_calculate_remaining_width(total_width,
				optional, opt_sel, title);

	text_width = xvf_calculate_text_width(upper, lower, precision, type);

	sb_width = remaining_width - text_width;

	return(sb_width);
}
      

/*------------------------------------------------------------
|
|  Routine Name: xvf_append_selection_scrollbar
|
|       Purpose: Puts a scrollbar at the end of an Integer,
|                Float, or Double selection.
|
|         Input: selection - pointer to the selection
|                line_info - line_info struct from UIS line
|        Output: 
|       Returns: TRUE on success, FALSE on failure
|          Date: May 10, 1993
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
int xvf_append_selection_scrollbar(
    kselection *selection,
    Line_Info  *line_info)
{
	double      sb_width, text_width;
	double     lower, upper, val;
	kfunc_void scroll_cb; 

	/*
	 * Depending on selection type, the scrollbar callback and action
         * procedure will vary.  Set lower & upper bounds, and value.
	 */
	if (selection->type == KUIS_INTEGER)
	{
	    scroll_cb   = xvf_int_scroll_cb;
	    lower = (double) line_info->lower_int;
	    upper = (double) line_info->upper_int;
	    val   = (double) line_info->int_val;
	}
	else if (selection->type == KUIS_FLOAT)
	{
	    scroll_cb   = xvf_float_scroll_cb;
	    lower = (double) line_info->lower_float;
	    upper = (double) line_info->upper_float;
	    val   = (double) line_info->float_val;
	}
	else if (selection->type == KUIS_DOUBLE)
        {
            scroll_cb   = xvf_double_scroll_cb;
            lower = line_info->lower_double;
            upper = line_info->upper_double;
            val   = line_info->double_val;
        }
	else
	{
	    kerror("xvforms", "xvf_append_selection_scrollbar",
	         "Only selection types supported are Integer, Float, & Double");
	    return(FALSE);
	}

	/*
         *  text object can no longer take up all remaining space, but
         *  must be resized to be just large enough to accommodate number.
	 *  calculate new width of text object.
         */
	text_width = xvf_calculate_text_width(upper, lower,
                                              line_info->precision,
                                              selection->type);
	/*
	 *  calculate width for the scrollbar; if width is not enough
	 *  for the scrollbar, return.
	 */
	sb_width = xvf_calculate_sb_width(line_info->width, line_info->optional,
				          line_info->opt_sel, line_info->title, 
					  upper, lower, line_info->precision, 
					  line_info->typeflag);
        if (sb_width < 1.0)
	    return(FALSE);

	/*
	 *  resize selection's text object to new text width
	 */
	xvw_set_attributes(selection->value_object, 
			   XVW_CHAR_WIDTH, text_width,
			   XVW_LEFT_OF,    KMANAGER_UNDEFINED,
			   XVW_TACK_EDGE,  KMANAGER_TACK_NONE,
			   NULL);

	/*
	 *  create scrollbar to right of text object
	 */
        selection->scroll = xvf_create_scroll(selection->back,
                                     selection->value_object, "scrollbar");

	/*
	 *  add callbacks for continuous and incremental scrollbar motion,
         *  which will update the number displayed in the text object.
	 */
        xvw_insert_callback(selection->scroll, XVW_SCROLL_CONT_MOTION, FALSE,
                         scroll_cb, (kaddr) selection);

        xvw_insert_callback(selection->scroll, XVW_SCROLL_INCR_MOTION, FALSE,
                         scroll_cb, (kaddr) selection);

	/*
	 *  initialize minimum, maximum, value of scrollbar.
	 */
	xvw_set_attributes(selection->scroll,
                           XVW_SCROLL_MIN,    lower,
                           XVW_SCROLL_MAX,    upper,
                           XVW_SCROLL_VALUE,  val,
                           NULL);

	/*
	 *  scoot <cr> pixmap (if one) to right of scrollbar.
	 */
        if (selection->pix_object != NULL)
            xvw_set_attribute(selection->scroll, XVW_LEFT_OF, 
			      selection->pix_object);
	else xvw_set_attribute(selection->scroll, XVW_LEFT_OF, NULL);


	return(TRUE);
}
