/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		Attribute Utilities			      <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		xvf_get_defined_attribute()	      <<<<
   >>>>	     		xvf_set_defined_attribute()	      <<<<
   >>>>   Static:                                             <<<<
   >>>>	     		find_attribute_entry()	      	      <<<<
   >>>>   Public:                                             <<<<
   >>>>	     		xvf_set_attributes()		      <<<<
   >>>>	     		xvf_get_attributes()		      <<<<
   >>>>	     		xvf_vset_attributes()		      <<<<
   >>>>	     		xvf_vget_attributes()		      <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"
#include "attributes.h"


static AttributeEntry *find_attribute_entry PROTO((char *));

/************************************************************
*
*  Routine Name: xvf_set_attributes - set multiple attributes of a GUI item
*
*       Purpose: Sets attributes associated with GUI items and UIS 
*                lines of GUI items supported by the xvforms library.
*                Note that the variable argument list of attribute/pairs 
*                MUST be terminated by NULL.
*
*         Input: kformstruct - the generic kform_struct associated
*                              with the GUI item (automatically generated
*                              by conductor in the GUI Information Structure)
*                attribute   - The attribute to be changed
*                value       - value to which to set attribute
*                [attr/value]- as many additional attribute/value pairs as 
*                              desired
*                NULL        - attribute/value pairs must be ended with NULL
*
*        Output: 
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Danielle Argiro & Mark Young
*          Date: Jan 31, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xvf_set_attributes(
   kform_struct *kformstruct,
   kvalist)
{
	kva_list list;
	xvobject object = NULL;

	/*
	 * sanity check
	 */
	if (kformstruct == NULL)
	{
	    kerror("xvforms", "xvf_set_attributes",
		   "kformstruct passed in is NULL");
	    return(FALSE);
	}

	/*
	 *  need to get the object associated with the kformstruct so that
         *  we can use xvf_vset_attributes().  Do not create widgets if they
         *  are not already created; instead, call kvf_vset_attributes().
	 */
        kva_start(list, kformstruct);
	if (kformstruct->type == KSUBFORM)
	{
	    if (kformstruct->Subformptr->button != NULL)
		object = xvf_get_xvobject(kformstruct, XVF_BACKPLANE, TRUE);
	}
	else if ((kformstruct->flag == KUIS_STARTPANE) ||
	         (kformstruct->flag == KUIS_GUIDEBUTTON))
	{
	    
	    if (kformstruct->Controlptr->back_guide->button != NULL)
		object = xvf_get_xvobject(kformstruct, XVF_BACKPLANE, TRUE);
	    else object = xvf_get_xvobject(kformstruct, XVF_BACKPLANE, FALSE);
	}
	else object = xvf_get_xvobject(kformstruct, XVF_BACKPLANE, FALSE);

	/*
         *  let xvf_vset_attributes() actually do the work 
         *  of setting the attributes, given that objects are created.
	 */
	if (object != NULL)
            return(xvf_vset_attributes(kformstruct, list));

        /*
         *  let kvf_vset_attributes() actually do the work
         *  of setting the attributes, given that objects are NOT created.
         */
	else return(kvf_vset_attributes(kformstruct, list));

}


/************************************************************
*
*  Routine Name: xvf_get_attributes - get multiple attributes of a GUI item
*
*       Purpose: Gets attributes associated with GUI items
*                and UIS lines of GUI objects supported by
*                the xvforms library.
*
*         Input: kformstruct - the generic kform_struct associated
*                              with the GUI item (automatically generated
*                              by conductor in the GUI Information Structure)
*                attribute   - The attribute to be changed
*                value       - value to which to set attribute
*                [attr/value]- as many additional attribute/value pairs as
*                              desired
*                NULL        - attribute/value pairs must be ended with NULL
*
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Danielle Argiro & Mark Young
*          Date: Jan 31, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xvf_get_attributes(
   kform_struct *kformstruct,
   kvalist)
{
        kva_list list;
        xvobject object;

	/*
	 * sanity check
	 */
	if (kformstruct == NULL)
	{
	    kerror("xvforms", "xvf_get_attributes",
		   "kformstruct passed in is NULL");
	    return(FALSE);
	}

	/*
         *  need to get the object associated with the kformstruct so that
         *  we can use xvf_vset_attributes().  Do not create widgets if they
         *  are not already created; instead, call kvf_vset_attributes().
         */
        kva_start(list, kformstruct);
        object = xvf_get_xvobject(kformstruct, XVF_BACKPLANE, FALSE);

	/*
         *  let xvf_vget_attributes() actually do the work 
         *  of getting the attributes, given that objects are created.
	 */
        if (object != NULL)
            return(xvf_vget_attributes(kformstruct, list));

	/*
         *  let kvf_vget_attributes() actually do the work
         *  of getting the attributes, given that objects are NOT created.
         */
	else return(kvf_vget_attributes(kformstruct, list));
}


/************************************************************
*
*  Routine Name: xvf_vset_attributes - set attributes of GUI items / UIS lines u
sing existing variable argument list
*
*       Purpose: Sets attributes associated with nodes in the form tree
*                when xvobjects for the GUI have not yet been created.
*
*         Input: kformstruct  - the generic kform_struct associated
*                               with a form tree node
*                attribute, value - variable number of attribute/value pairs
*                                   list of pairs; MUST be terminated by NULL.
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Danielle Argiro & Mark Young
*          Date: March 23, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xvf_vset_attributes(
   kform_struct *kformstruct,
   kva_list list)
{
        double         tmpdbl;
        kaddr          tmpstruct;
        kaddr          calldata;
        char           *attribute, *tmpstr;
        int            i, tmpint;
	int            status;
        float          tmpflt;
        AttributeEntry *entry;
	kselection    **save_extensions = NULL;

        /*
         *  for each attribute/value resource pair...
         */
	status = TRUE;
        while ((attribute = kva_arg(list, char *)) != NULL)
        {
            /*
             *  find the attribute in the attribute table
             */
	    entry = find_attribute_entry(attribute);

            /*
             *  shouldn't be possible to get here with xvf_check_attributes,
             *  but you never know.
             */
            if (entry == NULL)
            {
                kerror("xvforms", "xvf_vset_attributes",
                       "Unknown attribute '%s'. Unable to continue setting the rest of the attributes list.", attribute);
                return(FALSE);
            }

            switch(entry->data_type)
            {
                case KINT:
                     tmpint = kva_arg(list, int);
                     calldata = (kaddr) &tmpint;
                     break;

                case KSTRING:
                     tmpstr = kva_arg(list, char *);
                     calldata = (kaddr) &tmpstr;
                     break;

                case KSTRUCT:
                     tmpstruct = kva_arg(list, kaddr);
                     calldata = (kaddr) &tmpstruct;
                     break;

                case KFLOAT:
                     tmpflt = (float) kva_arg(list, double);
                     calldata = (kaddr) &tmpflt;
                     break;

                case KDOUBLE:
                     tmpdbl = kva_arg(list, double);
                     calldata = (kaddr) &tmpdbl;
                     break;

                default:
                     kerror("xvforms", "xvf_vset_attributes",
                            "Unknown attribute data type");
		     return(FALSE);
            }

            status = (*entry->set_attr_routine)
		     (kformstruct, attribute, calldata);

	    if (kstrcmp(attribute, XVF_DELETE) == 0)
	       return(status);

	   /*
            *  special case for selections with extensions to support
            *  setting of attributes menuform selections, making
            *  sure they are reflected by the same attribute in the
            *  corresponding selection on the front panel
            */
            if ((kformstruct->type == KSELECTION)        &&
                (kformstruct->Selptr->extensions != NULL))
            {
                /* only attributes that "travel" are "value" attributes */
                if ((kstrcmp(attribute, XVF_FILE_NAME)  == 0) ||
                    (kstrcmp(attribute, XVF_FILE_DEF)   == 0) ||
                    (kstrcmp(attribute, XVF_STRING_VAL) == 0) ||
                    (kstrcmp(attribute, XVF_STRING_DEF) == 0) ||
                    (kstrcmp(attribute, XVF_TOGGLE_NUM) == 0) ||
                    (kstrcmp(attribute, XVF_TOGGLE_VAL) == 0) ||
                    (kstrcmp(attribute, XVF_LOGIC_VAL)  == 0) ||
                    (kstrcmp(attribute, XVF_LIST_VAL)   == 0) ||
                    (kstrcmp(attribute, XVF_CYCLE_VAL)  == 0) ||
                    (kstrcmp(attribute, XVF_INT_VAL)    == 0) ||
                    (kstrcmp(attribute, XVF_INT_DEF)    == 0) ||
                    (kstrcmp(attribute, XVF_FLOAT_VAL)  == 0) ||
                    (kstrcmp(attribute, XVF_DOUBLE_VAL) == 0) ||
                    (kstrcmp(attribute, XVF_LITERAL)    == 0) ||
                    (kstrcmp(attribute, XVF_OPTSEL)     == 0))
                {
	            kselection  *extension;
		    int          restore_link = FALSE;

		    for (i = 0; i < kformstruct->Selptr->extension_num; i++)
                    {
                        extension = kformstruct->Selptr->extensions[i];
		        if (extension->extensions != NULL)
			{
		    	    restore_link = TRUE;
                            save_extensions = extension->extensions;
                            extension->extensions = NULL;
		 	}

		        switch(entry->data_type)
                        {
                	    case KINT:
		                 status = 
			         xvf_set_attribute(extension->back_kformstruct, 
				                      attribute, tmpint); 
                                 break;
    
                            case KSTRING:
		                 status = 
			         xvf_set_attribute(extension->back_kformstruct, 
				                   attribute, tmpstr); 
                                 break;
    
                            case KSTRUCT:
		                 status = 
			         xvf_set_attribute(extension->back_kformstruct, 
				                   attribute, tmpstruct); 
                                 break;
    
                            case KFLOAT:
		                 status = 
			         xvf_set_attribute(extension->back_kformstruct, 
				                   attribute, tmpflt); 
                                 break;
    
                            case KDOUBLE:
		                 status = 
			         xvf_set_attribute(extension->back_kformstruct, 
				                   attribute, tmpdbl); 
                                 break;
    
                            default:
                                 kerror("xvforms", "xvf_vset_attributes",
                                         "Unknown attribute data type");
                                 break;
                            }

		        extension->modified = TRUE;
		
		        if (restore_link)
                            extension->extensions = save_extensions;
		    }
                }
            }
        }
        return(status);
}

/************************************************************
*
*  Routine Name: xvf_vget_attributes - get attributes of GUI items / UIS lines u
sing existing variable argument list
*
*       Purpose: Gets attributes associated with nodes in the form tree
*                when xvobjects for the GUI have not yet been created.
*
*         Input: kformstruct      -  the generic kform_struct associated
*                                    with a form tree node
*                attribute, value - variable number of attribute/value pairs
*                                   list of pairs; MUST be terminated by NULL.
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Danielle Argiro & Mark Young
*          Date: March 23, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xvf_vget_attributes(
   kform_struct *kformstruct,
   kva_list list)
{
        char    *attribute;
        kaddr   calldata;
        AttributeEntry *entry;

        /*
         *  for each attribute/value resource pair...
         */
        while ((attribute = kva_arg(list, char *)) != NULL)
        {
            /*
             *  find the attribute in the attribute table
             */
	    entry = find_attribute_entry(attribute);

            /*
             *  shouldn't be possible to get here with xvf_check_attributes,
             *  but you never know.
             */
            if (entry == NULL)
            {
                kerror("xvforms", "xvf_vget_attributes",
                       "Unknown attribute '%s'. Unable to continue getting the rest of the attributes list.", attribute);
                return(FALSE);
            }

            calldata = kva_arg(list, kaddr);

            if (!((*entry->get_attr_routine)(kformstruct, attribute, calldata)))
	  	return(FALSE);

        }

        return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_defined_attribute
|       Purpose: Intermediate routine to be called by the xvwidgets
|                library when a menuform is used;  need to call the
|                correct xvforms get attributes routine.
|
|         Input: object    - backplane of selection
|                attribute - the xvforms attribute
|        Output: calldata  - provides call data (if any)
|
|          Date: March 24, 1994
|    Written By: Danielle Argiro & Mark Young
| Modifications:
|
------------------------------------------------------------*/

int xvf_get_defined_attribute(
    xvobject object,
    char     *attribute,
    kaddr    calldata)
{
	kform_struct   *kformstruct;
	AttributeEntry *entry;

        /* item that brought up the menu */
        if (!(xvw_get_attribute(object, XVW_MENU_CLIENTDATA, &kformstruct)))
            return(FALSE);

	/*
	 * find the attribute entry in the xvforms attribute table,
         * and call the corresponding routine to get the attribute.
	 */
        entry = find_attribute_entry(attribute);
        if (entry == NULL)
        {
            kerror("xvforms", "xvf_set_defined_attribute",
                   "Unknown attribute '%s'. Unable to continue setting the rest of the attributes list.", attribute);
	    return(FALSE);
        }
        return((*entry->get_attr_routine)(kformstruct, attribute, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_defined_attribute
|       Purpose: Intermediate routine to be called by the xvwidgets
|                library when a menuform is used;  need to call the
|                correct xvforms set attributes routine.
|
|         Input: object    - backplane of selection
|                attribute - the xvforms attribute
|        Output: calldata  - provides call data (if any)
|
|          Date: March 24, 1994
|    Written By: Danielle Argiro & Mark Young
| Modifications:
|
------------------------------------------------------------*/

int xvf_set_defined_attribute(
    xvobject object,
    char     *attribute,
    kaddr    calldata)
{
        kform_struct   *kformstruct;
	AttributeEntry *entry;

        /* item that brought up the menu */
        if (!(xvw_get_attribute(object, XVW_MENU_CLIENTDATA, &kformstruct)))
            return(FALSE);

	/*
	 * find the attribute entry in the xvforms attribute table,
         * and call the corresponding routine to set the attribute.
	 */
        entry = find_attribute_entry(attribute);
        if (entry == NULL)
        {
            kerror("xvforms", "xvf_set_defined_attribute",
                   "Unknown attribute '%s'. Unable to continue setting the rest of the attributes list.", attribute);
                return(FALSE);
        }
	if (!((*entry->set_attr_routine)(kformstruct, attribute, calldata)))
	    return(FALSE);

	/*
         *  special case for selections with extensions to support
         *  setting of attributes menuform selections, making
         *  sure they are reflected by the same attribute in the
         *  corresponding selection on the front panel
         */
        if ((kformstruct->type == KSELECTION)        &&
            (kformstruct->Selptr->extensions != NULL))
        {
                /* only attributes that "travel" are "value" attributes */
                if ((kstrcmp(attribute, XVF_FILE_NAME)  == 0) ||
                    (kstrcmp(attribute, XVF_FILE_DEF)   == 0) ||
                    (kstrcmp(attribute, XVF_STRING_VAL) == 0) ||
                    (kstrcmp(attribute, XVF_STRING_DEF) == 0) ||
                    (kstrcmp(attribute, XVF_TOGGLE_NUM) == 0) ||
                    (kstrcmp(attribute, XVF_TOGGLE_VAL) == 0) ||
                    (kstrcmp(attribute, XVF_LOGIC_VAL)  == 0) ||
                    (kstrcmp(attribute, XVF_LIST_VAL)   == 0) ||
                    (kstrcmp(attribute, XVF_CYCLE_VAL)  == 0) ||
                    (kstrcmp(attribute, XVF_INT_VAL)    == 0) ||
                    (kstrcmp(attribute, XVF_INT_DEF)    == 0) ||
                    (kstrcmp(attribute, XVF_FLOAT_VAL)  == 0) ||
                    (kstrcmp(attribute, XVF_DOUBLE_VAL) == 0) ||
                    (kstrcmp(attribute, XVF_OPTSEL)     == 0) ||
		    (kstrcmp(attribute, XVF_LITERAL)    == 0))
                {
                    kselection  **save_extensions = NULL;
                    kselection  *extension;
                    int          i, restore_link = FALSE;

		    for (i = 0; i < kformstruct->Selptr->extension_num; i++)
		    {
                        extension = kformstruct->Selptr->extensions[i];
                        if (extension->extensions != NULL)
		        {
                            restore_link = TRUE;
                            save_extensions = extension->extensions;
                            extension->extensions = NULL;
			}
		 
		        if (!(xvf_set_attribute(kformstruct, attribute, 
						calldata)))
			    return(FALSE);

                        extension->modified = TRUE;

                        if (restore_link)
                            extension->extensions = save_extensions;
		    }
                }


	}

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: find_attribute_entry
|       Purpose: Finds an attribute entry from the xvforms attribute
|                entry table given a particular attribute.
|         Input: none
|        Output: none
|       Returns: the attribute entry on success, NULL on failure
|          Date: March 24, 1994
|    Written By: Danielle Argiro & Mark Young
| Modifications:
|
------------------------------------------------------------*/

static AttributeEntry *find_attribute_entry(
    char *attribute)
{
	int            i, token;
	AttributeEntry *entry = NULL;

        /*
         *  find the attribute in the attribute table
         */
        token = kstring_to_token(attribute);
        for (i = 0; i < xvf_attribute_num; i++)
        {
            if (token == xvf_attribute_table[i].token)
            {
                entry = &xvf_attribute_table[i];
                break;
            }
	}
	return(entry);
}
