/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>        Miscellaneous Attribute Actions                 <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		xvf_get_logic_val()		      <<<<
   >>>>	     		xvf_set_logic_val()		      <<<<
   >>>>	     		xvf_get_logic_def()		      <<<<
   >>>>	     		xvf_set_logic_def()		      <<<<
   >>>>	     		xvf_get_logic_label()		      <<<<
   >>>>	     		xvf_set_logic_label()		      <<<<
   >>>>                                                       <<<<
   >>>>	     		xvf_set_list_index()		      <<<<
   >>>>	     		xvf_get_list_index()		      <<<<
   >>>>	     		xvf_set_list_label()		      <<<<
   >>>>	     		xvf_get_list_label()		      <<<<
   >>>>	     		xvf_get_list_size()		      <<<<
   >>>>	     		xvf_set_list_size()		      <<<<
   >>>>	     		xvf_get_list_val()		      <<<<
   >>>>	     		xvf_set_list_val()		      <<<<
   >>>>	     		xvf_get_list_start()		      <<<<
   >>>>	     		xvf_set_list_start()		      <<<<
   >>>>	     		xvf_set_list_choices()		      <<<<
   >>>>	     		xvf_get_list_contents()		      <<<<
   >>>>	     		xvf_set_list_contents()		      <<<<
   >>>>                                                       <<<<
   >>>>	     		xvf_get_exectype()		      <<<<
   >>>>	     		xvf_set_exectype()		      <<<<
   >>>>	     		xvf_get_routine()		      <<<<
   >>>>	     		xvf_set_routine()		      <<<<
   >>>>	     		xvf_get_routine()		      <<<<
   >>>>	     		xvf_set_routine()		      <<<<
   >>>>	     		xvf_get_list_doubleclick()	      <<<<
   >>>>	     		xvf_set_list_doubleclick()	      <<<<
   >>>>                                                       <<<<
   >>>>	     		xvf_set_list_add()	              <<<<
   >>>>	     		xvf_set_list_delete()	              <<<<
   >>>>	     		xvf_set_list_deleteall()	      <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"
#include "attributes.h"


static int update_list_value PROTO((kselection *, int, char *));

/*-----------------------------------------------------------
|
|       Routine: xvf_get_logic_val
|       Purpose: Gets the value of the logical from 
|                the (-l) UIS line of a logical selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LOGIC_VAL ("xvf_logic_val")
|        Output: calldata    - passes back logical value (TRUE or FALSE)
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_logic_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_logic_val(kformstruct, KVF_LOGIC_VAL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_logic_val
|       Purpose: Sets the value of the logical on 
|                the (-l) UIS line of a logical selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LOGIC_VAL ("xvf_logic_val")
|                calldata    - provides boolean value of TRUE or FALSE
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_logic_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *logic_val;
	char         **val_labels;

	logic_val = (int *) calldata;

	if (!(kvf_set_logic_val(kformstruct, KVF_LOGIC_VAL, calldata)))
	    return(FALSE);

	if (!(kvf_get_attribute(kformstruct, KVF_VAL_LABELS, &val_labels)))
            return(FALSE);

	if (!(update_list_value(kformstruct->Selptr, *logic_val, 
			        val_labels[*logic_val])))
            return(FALSE);

	karray_free(val_labels, 2, NULL);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_get_logic_def
|       Purpose: Gets the value of the logical default field
|                from the (-l) UIS line of a logical selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LOGIC_DEF ("xvf_logic_def")
|        Output: calldata    - passes back logical default (TRUE or FALSE)
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_logic_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_logic_def(kformstruct, KVF_LOGIC_DEF, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_logic_def
|       Purpose: Sets the value of the logical default field
|                on the (-l) UIS line of a logical selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LOGIC_DEF ("xvf_logic_def")
|                calldata    - provides boolean value of TRUE or FALSE
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_logic_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_logic_def(kformstruct, KVF_LOGIC_DEF, calldata)))
	    return(FALSE);

	if (!(xvf_set_logic_val(kformstruct, XVF_LOGIC_VAL, calldata)))
	    return(FALSE);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_logic_label
|       Purpose: Gets the label associated with a value of 0 or 1
|                from the (-l) UIS line of a logical selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LOGIC_1LABEL ("xvf_logic_1label") or 
|                              XVF_LOGIC_0LABEL ("xvf_logic_0label")
|        Output: calldata    - passes back title for value of 0 or 1 on logical
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_logic_label(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (kstrcmp(attribute, XVF_LOGIC_0LABEL) == 0)
	    return (kvf_get_logic_label(kformstruct, KVF_LOGIC_0LABEL, calldata));
	else return (kvf_get_logic_label(kformstruct, KVF_LOGIC_1LABEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_logic_label
|       Purpose: Sets the label associated with a value of 0 or 1
|                on the (-l) UIS line of a logical selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LOGIC_1LABEL ("xvf_logic_1label") or 
|                              XVF_LOGIC_0LABEL ("xvf_logic_0label")
|                calldata    - provides string for value of 0 or 1 for logical
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_logic_label(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         **val_labels;
	int          val_index, width, logic_val;

	if (kstrcmp(attribute, XVF_LOGIC_0LABEL) == 0)
	{
	    if (!(kvf_set_logic_label(kformstruct, KVF_LOGIC_0LABEL, calldata)))
	        return(FALSE);
	}
	else
	{
	    if (!(kvf_set_logic_label(kformstruct, KVF_LOGIC_1LABEL, calldata)))
	        return(FALSE);
	}

	if (kstrcmp(attribute, XVF_LOGIC_1LABEL) == 0)
	    val_index = 1;
	else val_index = 0;

	kvf_get_attribute(kformstruct, KVF_LOGIC_VAL, &logic_val);
        if (logic_val == val_index)
        {
	    kvf_get_attribute(kformstruct, KVF_VAL_LABELS, &val_labels);
	    width = kstrlen(val_labels[val_index]) +1;
            xvw_set_attributes(kformstruct->Selptr->value_object,
                               XVW_CHAR_WIDTH, (float) width,
			       XVW_LABEL,      val_labels[val_index],
			       NULL);
	    karray_free(val_labels, 2, NULL);
	}

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}
/*-----------------------------------------------------------
|
|       Routine: xvf_get_list_index
|       Purpose: Gets the value of the value index field
|                from the (-l), (-x), (-z), or (-c) UIS line 
|                of a logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_INDEX ("xvf_list_index")
|        Output: calldata    - passes back integer value of the list index
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_list_index(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_list_index(kformstruct, KVF_LIST_INDEX, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_list_index
|       Purpose: Sets the value of the value index field
|                on the (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_INDEX ("xvf_list_index")
|                calldata    - provides integer value of list_index
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_list_index(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *list_index;
	char         *list_label;

	/* the value to which XVF_VAL_INDEX is to be set */
        list_index  = (int *) calldata;

	if (!(kvf_set_list_index(kformstruct, KVF_LIST_INDEX, calldata)))
	    return(FALSE);

	if (!(kvf_get_attribute(kformstruct, KVF_LIST_LABEL, &list_label)))
	    return(FALSE);

        if (!(update_list_value(kformstruct->Selptr, *list_index, list_label)))
            return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_list_label
|       Purpose: Gets the label associated with a value of n from the
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_LABEL ("xvf_list_label")
|        Output: calldata    - passes back title for value of n on logical
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_list_label(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_list_label(kformstruct, KVF_LIST_LABEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_list_label
|       Purpose: Sets the label associated with a value of N on the
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection.
|                Note that the list value must be set to N first,
|                before the label for that value can be assigned.
|
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_LABEL ("xvf_list_label")
|                calldata    - provides string value of title for 
|                            value of n on logical
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_list_label(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         **list_label;
	kselection   *selection;

	/* the value to which KVF_VAL_LABEL is to be set */
        list_label  = (char **) calldata;

	if (!(kvf_set_list_label(kformstruct, KVF_LIST_LABEL, calldata)))
	    return(FALSE);
	

	/* setting label of currently selected value */
	selection = kformstruct->Selptr;
	if ((selection->type == KUIS_CYCLE) || (selection->type == KUIS_LIST))
	{
            if (!(xvw_set_attribute(selection->value_object,
                                    XVW_LABEL, *list_label)))
		    return(FALSE);

            if (!(xvw_set_attribute(selection->value_object,
                                    XVW_WIDTH, *list_label)))
		    return(FALSE);
        }
	else if (selection->type == KUIS_STRINGLIST)
	{
	   if (!(xvw_set_attribute(selection->value_object,
		                   XVW_TEXT_STRING, *list_label)))
		return(FALSE);
	}

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_list_size
|       Purpose: Gets the number of list choices from the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_SIZE ("xvf_list_size")
|        Output: calldata    - passes back number of choices for list
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_list_size(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_list_size(kformstruct, KVF_LIST_SIZE, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_list_size
|       Purpose: Sets the number of list choices on the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_SIZE ("xvf_list_size")
|                calldata    - provides number of choices for list
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_list_size(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int           i, *list_size, old_list_size;
	kselection    *selection;
	char          *button_label;
	char          **val_labels;

	/* the value to which KVF_LIST_SIZE is to be set */
        list_size = (int *) calldata;

	if (!(kvf_get_attribute(kformstruct, KVF_LIST_SIZE, &old_list_size)))
	    return(FALSE);	

	if (!(kvf_set_list_size(kformstruct, KVF_LIST_SIZE, calldata)))
            return(FALSE);      

	/*
	 *  For lists & stringlists, procedure is more complicated.
	 */
	selection = kformstruct->Selptr;
	if ((selection->type == KUIS_LIST)  ||
	    (selection->type == KUIS_STRINGLIST))
	{
	    /*
	     *  there are now less buttons than before.
	     */
	    if (*list_size < old_list_size)
	    {
		/*
		 *  have to remove old callbacks, destroy the old buttons
		 */
	        if (selection->submenu_buttons != NULL)
	        {
	            for (i = *list_size; i < old_list_size; i++)
	            {
		        if (selection->submenu_buttons[i] != NULL)
		        {
	                    xvw_remove_callback(selection->submenu_buttons[i],
                                                XVW_BUTTON_SELECT, 
						xvf_list_val_cb,
                                               (kaddr) kformstruct);
	                    xvw_destroy(selection->submenu_buttons[i]);
		        }
	            }
	        }

	        /*
	         *  if list size has been set to 0, visually indicate that.
	         */
		if (*list_size == 0)
		{
		    /* stringlist: blank out text parameter box */
	    	    if (selection->type == KUIS_STRINGLIST)
	        	xvw_set_attribute(selection->value_object, 
	   	                  XVW_TEXT_STRING, NULL);

		    /* list: blank out title on value button */
	    	    else if (selection->type == KUIS_LIST)
		 	xvw_set_attribute(selection->value_object, 
					  XVW_LABEL, " ");
	        }
	    }

	    /*
	     * there are now more buttons than before
	     */
	    if (*list_size > old_list_size)
	    {
		/*
                 * for lists & stringlists, have to create the new buttons.
		 */
	        selection->submenu_buttons 
			= (xvobject *) krealloc(selection->submenu_buttons,
						*list_size * sizeof(xvobject));
		if (selection->submenu_buttons == NULL)
		{
		    kerror("xvforms", "xvf_set_list_size",
			   "Unable to allocate additional submenu buttons");
		    return(FALSE);
		}
		kvf_get_attribute(kformstruct, KVF_VAL_LABELS, &val_labels);
	        for (i = old_list_size; i < *list_size; i++)
	        {
	            button_label = kstrdup(val_labels[i]);
	            selection->submenu_buttons[i] = 
			             xvf_create_menu_button(selection->submenu, 
					                    button_label, 
					                    button_label);
		    if (selection->type == KUIS_LIST)
		        xvw_insert_callback(selection->submenu_buttons[i],
                                            XVW_BUTTON_SELECT,  FALSE,
					    xvf_list_val_cb,
                                            (kaddr) kformstruct);
		    else if (selection->type == KUIS_STRINGLIST)
		        xvw_insert_callback(selection->submenu_buttons[i],
                                            XVW_BUTTON_SELECT, FALSE,
					    xvf_stringlist_val_cb,
                                            (kaddr) kformstruct);
		}
	        karray_free(val_labels, old_list_size, NULL);
	    }
	}

	/*
	 * have to do some stuff for cycles, not as much 
	 */
	else if (selection->type == KUIS_CYCLE)
	{
	    /*
	     * setting list size to 0 : 
	     * give visual indication by blanking out button,
             * and remove the callback on the button
	     */
	    if ((*list_size == 0) && (old_list_size > 0))
	    {
		xvw_set_attribute(selection->value_object, XVW_LABEL, " ");
		xvw_remove_callback(selection->value_object, XVW_BUTTON_SELECT,
                                    xvf_cycle_val_cb, (kaddr) kformstruct);
	    }

	    /*
	     *  setting list size to > 0: add the callback back onto button.
	     */
	    else if ((*list_size > 0) && (old_list_size == 0))
	    {
                xvw_insert_callback(selection->value_object, XVW_BUTTON_SELECT, 
			  FALSE, xvf_cycle_val_cb, (kaddr) kformstruct);
	    }
	}

	/*
	 * For DISPLAYLIST selection, have to modify displayed list.
	 */
	else if (selection->type == KUIS_DISPLAYLIST)
	{
	    kvf_get_attribute(kformstruct, KVF_VAL_LABELS, &val_labels);
	    xvw_change_list(selection->submenu, val_labels,
                            *list_size, FALSE);
	    karray_free(val_labels, *list_size, NULL);
	}
	 
	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_list_val
|       Purpose: Gets the current list choice from the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_VAL ("xvf_list_val")
|        Output: calldata    - passes back default choice for list
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_list_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_list_val(kformstruct, KVF_LIST_VAL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_list_val
|       Purpose: Sets the current list choice on the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_VAL ("xvf_list_val")
|                calldata    - provides default choice for list
|        Output: none
|
|          Date: Nov 18, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_list_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          list_index; 
	char         *list_label;

	if (!(kvf_set_list_val(kformstruct, KVF_LIST_VAL, calldata)))
            return(FALSE);

	if (!(kvf_get_attribute(kformstruct, KVF_LIST_INDEX, &list_index)))
            return(FALSE);

	if (!(kvf_get_attribute(kformstruct, KVF_LIST_LABEL, &list_label)))
            return(FALSE);

	if (!(update_list_value(kformstruct->Selptr, list_index, list_label)))
            return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_list_start
|       Purpose: Gets the integer start value from the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_START ("xvf_list_start")
|        Output: calldata    - passes back starting integer for list
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_list_start(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_list_start(kformstruct, KVF_LIST_START, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_list_start
|       Purpose: Sets the integer start value on the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_START ("xvf_list_start")
|                calldata    - provides starting integer for list
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_list_start(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int          list_val;
	int          *list_start;

	/* the value to which KVF_LIST_START is to be set */
        list_start = (int *) calldata;

	if (!(kvf_set_list_start(kformstruct, KVF_LIST_START, calldata)))
            return(FALSE);

	/* 
	 * current list val less than new list start, putting the 
         * list value L of range.  scoot list value up to list start.
	 */
	if (!(kvf_get_attribute(kformstruct, KVF_LIST_VAL, &list_val)))
            return(FALSE);

	if (kformstruct->flag == KUIS_DISPLAYLIST)
            xvw_set_attribute(kformstruct->Selptr->submenu, 
			      XVW_LIST_UNHIGHLT_ELEM,
                              list_val - *list_start);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_set_list_choices
|       Purpose: Sets the list of choices on the 
|                (-l), (-x), (-z), or (-c) UIS line of a 
|                logical, list, displaylist, or cycle selection
|                This is an action attribute routine.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_CHOICES ("xvf_list_choices")
|                calldata    - not used
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_list_choices(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{    
	static char  *top_label = "Strings Representing Choices";
	char         **prompts, **answers, **val_labels;
	char         temp[KLENGTH];
	int          i, value, list_size;

	if (!(kvf_get_attribute(kformstruct, KVF_LIST_SIZE, &list_size)))
	    return(FALSE);

	if (list_size == 0) 
	{
	    kerror("xvforms", "xvf_set_list_choices",
		   "Current number of choices specified is 0. Please specify at least one choice before attempting to set values.");
	    return(FALSE);
	}

	/*
	 *  set up for use of query object
	 */
	prompts = (char **) kcalloc(1, list_size*sizeof(char *));
	if (prompts == NULL)
	{
	    kerror("xvforms", "xvf_set_list_choices",
		   "Cannot allocate room for prompt strings");
	    return(FALSE);
	}

	answers = (char **) kcalloc(1, list_size*sizeof(char *));
	if (answers == NULL)
	{
	    kerror("xvforms", "xvf_set_list_choices",
		   "Cannot allocate room for answers strings");
	    return(FALSE);
	}
	
	kvf_get_attribute(kformstruct, KVF_LIST_START, &value);
	kvf_get_attribute(kformstruct, KVF_VAL_LABELS, &val_labels);
	for (i = 0; i < list_size; i++)
	{
	    ksprintf(temp, "String assoc. w/ %d:", value++);
	    prompts[i] = kstrdup(temp);
	    answers[i] = kcalloc(1, KLENGTH*sizeof(char));
	    if (answers[i] == NULL)
	    {
		kerror("xvforms", "xvf_set_list_choices",
		       "Unable to allocate string for entering list label");
		return(FALSE); 
	    }
	    
	    if ((val_labels != NULL) && (val_labels[i] != NULL))
		ksprintf(answers[i], "%s", val_labels[i]);
	}
	karray_free(val_labels, list_size, NULL);

	/*
	 *  bring up query object w/ sets of prompts
	 */
	if (!(xvu_query_wait(top_label, prompts, "Use", answers, 
		             list_size, 20)))
	    return(FALSE);

	for (i = 0; i < list_size; i++)
	    kfree(prompts[i]);

	if (!(xvf_set_attribute(kformstruct, XVF_LIST_CONTENTS, answers)))
		return(FALSE);

	for (i = 0; i < list_size ; i++)
	    kfree(answers[i]);

	kfree(answers);
	kfree(prompts);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_list_contents
|       Purpose: Gets the list of strings from the 
|                (-l), (-x), (-z), or (-c) UIS line of a
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_CONTENTS ("xvf_list_contents")
|        Output: calldata    - passes back array of strings making up list
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_list_contents(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_list_contents(kformstruct, KVF_LIST_CONTENTS, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_list_contents
|       Purpose: Sets the list of string on the 
|                (-l), (-x), (-z), or (-c) UIS line of a
|                logical, list, displaylist, or cycle selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_CONTENTS ("xvf_list_contents")
|                calldata    - provides array of strings for list
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications: 
| ------------------------------------------------------------*/

/* ARGSUSED */
int xvf_set_list_contents(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          i, list_size;
	kselection   *selection;
	char         ***val_labels;
	char         *set_string, *existing_string;

	/* the value to which KVF_LIST_CONTENTS is to be set */
        val_labels = (char ***) calldata;

	if (!(kvf_set_list_contents(kformstruct, KVF_LIST_CONTENTS, calldata)))
	    return(FALSE);

        /*
         *  update the label on the button (for list & cycle objects only
         */
	selection = kformstruct->Selptr;

        if ((kformstruct->flag == KUIS_LIST) ||
            (kformstruct->flag == KUIS_CYCLE))
        {
	    kvf_get_attribute(kformstruct, KVF_LIST_LABEL, &set_string);

            if (!(xvw_set_attribute(selection->value_object, 
			            XVW_LABEL, set_string)))
                return(FALSE);
        }

        /*
         *  LIST and STRINGLIST selections: re-create the pulldown menu
         */
        if ((selection->type == KUIS_LIST) ||
            (selection->type == KUIS_STRINGLIST))
        {
	    kvf_get_attribute(kformstruct, KVF_LIST_SIZE, &list_size);
            for (i = 0; i < list_size; i++)
            {
                if (!(xvw_set_attribute(selection->submenu_buttons[i],
                                        XVW_LABEL, (*val_labels)[i])))
                    return(FALSE);
            }
        }

        /*
         *  STRINGLIST selections: also update the parameter box
         */
        if (selection->type == KUIS_STRINGLIST)
        {
	    kvf_get_attribute(kformstruct, KVF_LIST_LABEL, &set_string);
	    xvw_get_attribute(selection->value_object, XVW_TEXT_STRING,
		              &existing_string);
	    if (existing_string == NULL)
                xvw_set_attribute(selection->value_object, XVW_TEXT_STRING,
                                  set_string);
        }

	/*
         *  DISPLAYLIST selections: update the displayed list, but
         *                          set NOTHING SELECTED
         */
        if (selection->type == KUIS_DISPLAYLIST)
        {
	    kvf_get_attribute(kformstruct, KVF_LIST_SIZE, &list_size);
	    if  ((list_size == 0) || (*val_labels == NULL))
		xvw_change_list(selection->submenu, NULL, 0, FALSE);
	    else 
	    {
		kvf_get_attribute(kformstruct, KVF_LIST_SIZE, &list_size);
		xvw_change_list(selection->submenu, *val_labels,
                                list_size, FALSE);
	    }
        }

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_get_exectype
|       Purpose: Gets the execution type from the (-R) UIS line 
|                of a routine selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_EXECTYPE ("xvf_exectype")
|        Output: calldata    - passes back execution type for Routine button
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_exectype(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_exectype(kformstruct, KVF_EXECTYPE, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_exectype
|       Purpose: Sets the execution type on the (-R) UIS line 
|                of a routine selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_EXECTYPE ("xvf_exectype")
|                calldata    - provides execution type for Routine button
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications: 
| ------------------------------------------------------------*/

/* ARGSUSED */
int xvf_set_exectype(
   kform_struct *kformstruct,
   char   *attribute,
   kaddr  calldata)
{
	if (!(kvf_set_exectype(kformstruct, KVF_EXECTYPE, calldata)))
	    return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_routine
|       Purpose: Gets the value of the routine field from 
|                the (-R) UIS line of a routine selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_ROUTINE ("xvf_routine")
|        Output: calldata    - passes back string value of routine
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_routine(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_routine(kformstruct, KVF_ROUTINE, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_routine
|       Purpose: Sets the value of the routine field on 
|                the (-R) UIS line of a routine selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_ROUTINE ("xvf_routine")
|                calldata    - provides string value of routine
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_routine(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_routine(kformstruct, KVF_ROUTINE, calldata)))
	    return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_help_path
|       Purpose: Gets the value of the help path from 
|                the (-H) UIS line of a Help selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_HELPPATH ("xvf_help_path")
|        Output: calldata    - passes back string value of help_path
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_help_path(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_help_path(kformstruct, KVF_HELPPATH, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_help_path
|       Purpose: Sets the value of the help path on 
|                the (-H) UIS line of a Help selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_HELPPATH ("xvf_help_path")
|                calldata    - provides string value of help_path
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_help_path(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_help_path(kformstruct, KVF_HELPPATH, calldata)))
	    return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_list_doubleclick
|       Purpose: Gets the value of the doubleclick field
|                from the (-z) DISPLAYLIST UIS line 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_DOUBLECLICK ("xvf_list_doubleclick")
|        Output: calldata    - passes back boolean value of the list index
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_list_doubleclick(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_list_doubleclick(kformstruct, KVF_LIST_DOUBLECLICK, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_list_doubleclick
|       Purpose: Sets the value of the doubleclick field
|                from the (-z) DISPLAYLIST UIS line
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_DOUBLECLICK ("xvf_list_doubleclick")
|                calldata    - provides integer value of list_index
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_list_doubleclick(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_list_doubleclick(kformstruct, KVF_LIST_DOUBLECLICK, calldata)))
	    return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_list_add
|       Purpose: Adds an item to a list of choices on the
|                (-l), (-x), (-z), or (-c) UIS line of a
|                logical, list, displaylist, or cycle selection
|                This is an action attribute routine.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_ADD ("xvf_list_add")
|                calldata    - provides label for new item in list
|                            (index is determined by length of list;
|                             new item is always added at the end>)
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_list_add(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int          list_size;
	char         **new_choice;
	char         **val_labels;
	kselection   *selection;

	new_choice = (char **) calldata;

	if (!(kvf_set_list_add(kformstruct, KVF_LIST_ADD, calldata)))
	    return(FALSE);


	/*
	 *  update LIST selection 
	 */
	kvf_get_attribute(kformstruct, KVF_LIST_SIZE, &list_size);
	selection = kformstruct->Selptr;
	if (selection->type == KUIS_LIST) 
	{
	    /* no previous value: set label on value object */
	    if (list_size == 1) 
	        xvw_set_attribute(selection->value_object, 
				  XVW_LABEL, *new_choice);

	    /* alloc submenu buttons */
	    selection->submenu_buttons = 
		(xvobject *) krealloc(selection->submenu_buttons,
                                      list_size * sizeof(xvobject));
	    if (selection->submenu_buttons == NULL)
	    {
		kerror("xvforms", "xvf_change_listadd",
		       "Unable to allocate additional button(s) for new items");
		return(FALSE);
	    }

	    /* create button on menu for new choice */
	    selection->submenu_buttons[list_size-1] =
		xvf_create_menu_button(selection->submenu, 
				       *new_choice, *new_choice);

	    /* add callback for list object */
	    xvw_insert_callback(selection->submenu_buttons[list_size-1],
                             XVW_BUTTON_SELECT, FALSE, xvf_list_val_cb,
                             (kaddr) kformstruct);

	}

	/*
         *  update STRINGLIST selection
	 */
	else if (selection->type == KUIS_STRINGLIST)
	{
            /* alloc submenu buttons */
            selection->submenu_buttons =
                (xvobject *) krealloc(selection->submenu_buttons,
                                      list_size * sizeof(xvobject));
	    if (selection->submenu_buttons == NULL)
	    {
		kerror("xvforms", "xvf_change_listadd",
		       "Unable to allocate additional button(s) for new items");
		return(FALSE);
	    }

            /* create button on menu for new choice */
            selection->submenu_buttons[list_size-1] =
                xvf_create_menu_button(selection->submenu, 
				       *new_choice, *new_choice);

	    /* add callback for stringlist object */
	    xvw_insert_callback(selection->submenu_buttons[list_size-1],
                             XVW_BUTTON_SELECT, FALSE, xvf_stringlist_val_cb,
                             (kaddr) kformstruct);
	}

	/*
	 *  update CYCLE selection
	 */
	else if (selection->type == KUIS_CYCLE)
	{
	    if (list_size == 1)
	    {
	        xvw_set_attribute(selection->value_object, XVW_LABEL, 
			          *new_choice);
	        xvw_insert_callback(selection->value_object, XVW_BUTTON_SELECT,
                         FALSE, xvf_cycle_val_cb, (kaddr) kformstruct);
	    }
	}

	/*
	 * update DISPLAYLIST selection
	 */
	else if (selection->type == KUIS_DISPLAYLIST)
	{
	    kvf_get_attribute(kformstruct, KVF_VAL_LABELS, &val_labels);
	    xvw_change_list(selection->submenu, val_labels,
                            list_size, FALSE);
	    karray_free(val_labels, list_size, NULL);
	}
	
	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_set_list_delete
|       Purpose: Deletes an item from the list of choices on the
|                (-l), (-x), (-z), or (-c) UIS line of a
|                logical, list, displaylist, or cycle selection
|                This is an action attribute routine.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_DELETE ("xvf_list_delete")
|                calldata    - provides index of choice to be deleted
|        Output: none
|
|          Date: Jan 31, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_list_delete(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        kselection   *selection;
        char         **val_labels;
        int          old_index = -1; 
	int          list_size;

	if (!(kvf_get_attribute(kformstruct, KVF_LIST_INDEX, &old_index)))
            return(FALSE);

	if (!(kvf_set_list_delete(kformstruct, KVF_LIST_DELETE, calldata)))
            return(FALSE);

	if (!(kvf_get_attribute(kformstruct, KVF_LIST_SIZE, &list_size)))
            return(FALSE);

	selection = kformstruct->Selptr;
	if (selection->type == KUIS_LIST)
	{
	    xvw_remove_callback(selection->submenu_buttons[old_index], 
			        XVW_BUTTON_SELECT,
                                xvf_list_val_cb, (kaddr) kformstruct);
	    xvw_destroy(selection->submenu_buttons[old_index]);
	    if (list_size == 0)
	    {
	        selection->submenu_buttons = NULL;
                xvw_set_attribute(selection->value_object, XVW_LABEL, " ");
	    }
	}
	else if (selection->type == KUIS_STRINGLIST)
        {
            xvw_remove_callback(selection->submenu_buttons[old_index], 
                                XVW_BUTTON_SELECT,
                                xvf_stringlist_val_cb, (kaddr) kformstruct);
            xvw_destroy(selection->submenu_buttons[old_index]);
	    if (list_size == 0)
                xvw_set_attribute(selection->value_object, XVW_TEXT_STRING, "");
        }
	else if (selection->type == KUIS_CYCLE)
	{
	    if (list_size == 0)
	    {
                xvw_set_attribute(selection->value_object, XVW_LABEL, " ");
                xvw_remove_callback(selection->value_object, 
                                    XVW_BUTTON_SELECT,
                                    xvf_cycle_val_cb, (kaddr) kformstruct);
	    }
	}
	else if (selection->type == KUIS_DISPLAYLIST)
	{
	    kvf_get_attribute(kformstruct, KVF_VAL_LABELS, &val_labels);
	    xvw_change_list(selection->submenu, val_labels,
                            list_size, FALSE);
	    karray_free(val_labels, list_size, NULL);
	}

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_set_list_deleteall
|       Purpose: Deletes all items from the list of choices on the
|                (-l), (-x), (-z), or (-c) UIS line of a
|                logical, list, displaylist, or cycle selection
|                This is an action attribute routine.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIST_DELETEALL ("xvf_list_deleteall")
|                calldata    - provides integer value of TRUE (ignored)
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_list_deleteall(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          i;
	int          list_size;
	char         **val_labels;
	kselection   *selection;

	if (!(kvf_get_attribute(kformstruct, KVF_LIST_SIZE, &list_size)))
            return(FALSE);

	selection = kformstruct->Selptr;
	for (i = 0; i < list_size; i++)
	{
	    if (selection->type == KUIS_LIST)
            {
                xvw_remove_callback(selection->submenu_buttons[i],
                                    XVW_BUTTON_SELECT,
                                    xvf_list_val_cb, (kaddr) kformstruct);
                xvw_destroy(selection->submenu_buttons[i]);
	    }	
	    else if (selection->type == KUIS_STRINGLIST)
            {
                xvw_remove_callback(selection->submenu_buttons[i],
                                    XVW_BUTTON_SELECT,
                                    xvf_stringlist_val_cb, (kaddr) kformstruct);
                xvw_destroy(selection->submenu_buttons[i]);
	    }
	}

	if (!(kvf_set_list_deleteall(kformstruct, KVF_LIST_DELETEALL, calldata)))
	    return(FALSE);

	if (selection->type == KUIS_LIST)
        {
            selection->submenu_buttons = NULL;
            xvw_set_attribute(selection->value_object, XVW_LABEL, " ");
        }
        else if (selection->type == KUIS_STRINGLIST)
        {
            xvw_set_attribute(selection->value_object, XVW_TEXT_STRING, "");
        }
        else if (selection->type == KUIS_CYCLE)
        {
            xvw_set_attribute(selection->value_object, XVW_LABEL, " ");
            xvw_remove_callback(selection->value_object,
                                XVW_BUTTON_SELECT,
                                xvf_cycle_val_cb, (kaddr) kformstruct);
        }
	else if (selection->type == KUIS_DISPLAYLIST)
	{
	    kvf_get_attributes(kformstruct,
			       KVF_LIST_SIZE,  &list_size,
			       KVF_VAL_LABELS, &val_labels,
			       NULL);
	    xvw_change_list(selection->submenu, val_labels,
                            list_size, FALSE);
	    karray_free(val_labels, list_size, NULL);
	}

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: update_list_value
|       Purpose: This utility routine updates the *displayed*
|                value of a list, displaylist, stringlist, or cycle selection;
|                ie, it updates the label on the button for a cycle or list,
|                the highlighted item on a displaylist, the word in the
|                text parameter box of the stringlist.
|
|         Input: selection  - selection to have value updated
|                list_index - index to update to
|                list_label - label to update with
|        Output: none
|
|          Date: Feb 21, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
static int update_list_value(
    kselection *selection,
    int 	list_index,
    char       *list_label)
{
	int   last_selected;

        /*
         * for LIST and CYCLE selections, need to update title on value button
         */
        if ((selection->type == KUIS_LIST) || 
	    (selection->type == KUIS_CYCLE)||
	    (selection->type == KUIS_LOGICAL))
        {
	    xvf_set_object_label_or_pixmap(selection->value_object, 
					   XVF_BUTTON_OBJ, list_label);
                return(FALSE);
        }

        /*
         * for STRINGLIST selection, update string in parameter box
         */
        else if (selection->type == KUIS_STRINGLIST)
        {
            if (!(xvw_set_attribute(selection->value_object,
                                    XVW_TEXT_STRING, list_label)))
                return(FALSE);
        }

        /*
         * for DISPLAYLIST selection, highlight desired item
         */
        else if (selection->type == KUIS_DISPLAYLIST)
        {
            xvw_get_attribute(selection->submenu, XVW_LIST_HIGHLT_ELEM,
                              &last_selected);
            if (last_selected != -1)
                xvw_set_attribute(selection->submenu, XVW_LIST_UNHIGHLT_ELEM,
                                  last_selected);
	    if (list_index > -1)
	    {
                if (!(xvw_set_attribute(selection->submenu, 
				        XVW_LIST_HIGHLT_ELEM, list_index)))
		    return(FALSE);
	    }
        }
	return(TRUE);
}
