/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		    General Attribute Actions                 <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		xvf_get_activate()		      <<<<
   >>>>	     		xvf_set_activate()		      <<<<
   >>>>	     		xvf_get_selected()		      <<<<
   >>>>	     		xvf_set_selected()		      <<<<
   >>>>	     		xvf_get_optional()		      <<<<
   >>>>	     		xvf_set_optional()		      <<<<
   >>>>	     		xvf_get_optsel()		      <<<<
   >>>>	     		xvf_set_optsel()		      <<<<
   >>>>	     		xvf_get_live()			      <<<<
   >>>>	     		xvf_set_live()			      <<<<
   >>>>	     		xvf_get_variable()		      <<<<
   >>>>	     		xvf_set_variable()		      <<<<
   >>>>	     		xvf_get_description()		      <<<<
   >>>>	     		xvf_set_description()		      <<<<
   >>>>	     		xvf_get_title()			      <<<<
   >>>>	     		xvf_set_title()			      <<<<
   >>>>	     		xvf_get_buttontitle()		      <<<<
   >>>>	     		xvf_set_buttontitle()		      <<<<
   >>>>	     		xvf_get_guidepanetitle()	      <<<<
   >>>>	     		xvf_set_guidepanetitle()	      <<<<
   >>>>	     		xvf_get_literal()		      <<<<
   >>>>	     		xvf_set_literal()		      <<<<
   >>>>	     		xvf_set_delete()		      <<<<
   >>>>	     		xvf_set_display_subform()	      <<<<
   >>>>	     		xvf_set_display_pane()		      <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"
#include "attributes.h"


/*-----------------------------------------------------------
|
|       Routine: xvf_get_activate
|       Purpose: Gets the value of the activate field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_ACTIVATE ("xvf_activate")
|        Output: calldata    - passes back TRUE if GUI item is activated
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/

/* ARGSUSED */
int xvf_get_activate(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        return (kvf_get_activate(kformstruct, KVF_ACTIVATE, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_activate
|       Purpose: Sets the value of the activate field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_ACTIVATE ("xvf_activate")
|                calldata    - provides boolean value of TRUE or FALSE
|        Output: none
|
|          Date: Nov 10, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_activate(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int        *active;
	xvobject   object;

        active = (int *) calldata;

        if (!(kvf_set_activate(kformstruct, KVF_ACTIVATE, calldata)))
            return(FALSE);

	if (kformstruct->type == KSUBFORM)
	{
            if (kformstruct->Subformptr->button != NULL)
                xvw_sensitive(kformstruct->Subformptr->button, *active);
	    if (kformstruct->Subformptr->guidepane != NULL)
	    {
		kguide *guide;
		xvf_set_selections_sensitive(kformstruct->Subformptr->guidepane->sel_list, *active);
		guide = kformstruct->Subformptr->guidepane->guide_list;
		while (guide != NULL)
		{
		    xvw_sensitive(guide->button, *active);
		    xvf_set_selections_sensitive(guide->pane->sel_list, 
                                                 *active);
		    guide = guide->next;
		}
	    }
	    else xvf_set_selections_sensitive(kformstruct->Subformptr->guide->pane->sel_list, *active);

	}
	else if (kformstruct->type == KGUIDE)
        {
            if (kformstruct->Guideptr->button != NULL)
                xvw_sensitive(kformstruct->Guideptr->button, *active);
	    xvf_set_selections_sensitive(kformstruct->Controlptr->sel_list, 
				         *active);
        }
	else if (kformstruct->type == KPANE)
        {
            if (kformstruct->Controlptr->back_guide->button != NULL)
                xvw_sensitive(kformstruct->Controlptr->back_guide->button,
                              *active);
	    xvf_set_selections_sensitive(kformstruct->Controlptr->sel_list, 
				         *active);
        }
	else if (kformstruct->flag == KUIS_STARTSUBMENU)
        {
            if (kformstruct->Selptr->label_object != NULL)
                xvw_sensitive(kformstruct->Selptr->label_object, *active);
        }
	else 
	{
	    object = xvf_get_xvobject(kformstruct, XVF_BACKPLANE, FALSE);
	    if (object != NULL)
	        xvw_sensitive(object, *active);
	}


	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_selected
|       Purpose: Gets the value of the selected field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_SELECTED ("xvf_selected")
|        Output: calldata    - passes TRUE if GUI item is selected
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_selected(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        return (kvf_get_selected(kformstruct, KVF_SELECTED, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_selected
|       Purpose: Sets the value of the selected field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_SELECTED ("xvf_selected")
|                calldata    - provides boolean value of TRUE or FALSE
|        Output: none
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_selected(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        if (!(kvf_set_selected(kformstruct, KVF_SELECTED, calldata)))
            return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);

}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_optional
|       Purpose: Gets the value of the optional field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_OPTIONAL ("xvf_optional")
|        Output: calldata    - passes back boolean value of TRUE or FALSE
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_optional(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        return (kvf_get_optional(kformstruct, KVF_OPTIONAL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_optional
|       Purpose: Sets the value of the optional field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_OPTIONAL ("xvf_optional")
|                calldata    - provides boolean value of TRUE or FALSE
|
|        Output: none
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_optional(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int        *optional;
	int        opt_sel;
	kselection *selection;

	/* the value to which XVF_OPTIONAL is to be set */
	optional  = (int *) calldata;

	/*
	 *  setting selection to optional.
	 */
	selection = kformstruct->Selptr;
	if (*optional)
	{
            if (selection->opt_object == NULL)
		xvf_add_opt_button(kformstruct);
	}
	else 
	{
	    /* 
	     * if we are changing to not optional, must restore opt_sel to 1.
	     * note that the code below CANNOT be substituted with a single
             * call to xvw_set_attributes() on XVF_OPTSEL because of the
             * catch-22 situation that will occur with illegal combination
             * 1 0 0 0 in UIS line.  the call to xvw_set_attributes() is 
             * redundant, but is made to restore the menupane cycle selection
             * for opt_sel.
	     */
	    kvf_get_attribute(kformstruct, KVF_OPTSEL, &opt_sel);
	    if (opt_sel != 1)
	    {
		if (!(xvf_set_attribute(kformstruct, XVF_OPTSEL, 1)))
		    return(FALSE);
	    }
	    if (selection->opt_object != NULL)
		xvf_delete_opt_button(kformstruct);
	}

        if (!(kvf_set_optional(kformstruct, KVF_OPTIONAL, calldata)))
            return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_optsel
|       Purpose: Gets the value of the opt sel field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_OPTSEL ("xvf_optsel")
|        Output: calldata    - passes back integer value of 0, 1, or 2
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_optsel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        return (kvf_get_optsel(kformstruct, KVF_OPTSEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_optsel
|       Purpose: Sets the value of the opt sel field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_OPTSEL ("xvf_optsel")
|                calldata    - provides integer value of 0, 1, or 2
|        Output: none
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_optsel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int           *optsel, current;
	kselection    *selection;
	int           change, optional;

	selection = kformstruct->Selptr;
	if (selection->back_group != NULL)
	{
	    xvf_call_do_group(selection);
	    return(TRUE);
	}

	/* the value to which XVF_OPTSEL is to be set */
        optsel    = (int *) calldata;

	if (!(kvf_get_attribute(kformstruct, KVF_OPTSEL, &current)))
            return(FALSE);

        if (!(kvf_set_optsel(kformstruct, KVF_OPTSEL, calldata)))
            return(FALSE);

	if ((current != *optsel) || (selection->modified))
            change = TRUE;
        else change = FALSE;

	/* was here */
        selection->opt_selected = *optsel;

        /*
         * opt_sel is Selected (1) or Not Selected (0),
         * must have an optional button; highlight for Selected,
         * un-highlight for Not Selected.
         */
        if (*optsel != 2)
        {
	    kvf_get_attribute(kformstruct, KVF_OPTIONAL, &optional);
            if ((selection->opt_object == NULL) && (optional))
	    {
                xvf_add_opt_button(kformstruct);
		if (change && (current != 2)) 
		    xvw_reverse_colors(selection->opt_object, *optsel);
	    }
            else if ((change)  && (selection->opt_object != NULL))
	    {
		xvw_reverse_colors(selection->opt_object, *optsel);
	    }
	    if ((selection->type != KUIS_FLAG) && 
		(selection->back_group == NULL))
                xvf_set_optional_sensitivity(selection, *optsel);
        }

        /*
         * opt_sel is Selected But Not Shown (2);
         * delete optional button if necessary.
         */
        else
        {
             if (selection->opt_object != NULL)
                 xvf_delete_opt_button(kformstruct);
        }

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_live
|       Purpose: Gets the value of the live field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIVE ("xvf_live")
|        Output: calldata    - passes back boolean value of TRUE or FALSE
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_live(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        return (kvf_get_live(kformstruct, KVF_LIVE, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_live
|       Purpose: Sets the value of the live field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LIVE ("xvf_live")
|                calldata    - provides boolean value of TRUE or FALSE
|        Output: none
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_live(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *live;
	kselection   *selection;

	/* the current value that "xvf_live" is to be set */
        live  = (int *) calldata;

        if (!(kvf_set_live(kformstruct, KVF_LIVE, calldata)))
            return(FALSE);

	selection = kformstruct->Selptr;

	/*
         *  the selection WAS live;  when making it non-live, must re-adjust 
         *  layout of selection, then destroy the <cr> pixmap object
         */
        if (!(*live))
        {
            switch (selection->type)
            {
                case KUIS_INPUTFILE:
                case KUIS_OUTPUTFILE:
                case KUIS_STRING:
                case KUIS_LOGICAL:
                case KUIS_STRINGLIST:
                case KUIS_CYCLE:
                case KUIS_LIST:
                case KUIS_FLAG:
	             xvw_set_attribute(selection->value_object, XVW_LEFT_OF, 
                                      NULL);
		     break;

                case KUIS_FLOAT:
                case KUIS_INTEGER:
                case KUIS_DOUBLE:
                     if (selection->scroll  != NULL)
		     {
			 xvw_set_attribute(selection->scroll, XVW_LEFT_OF, 
                                           NULL);
		     }
                     else 
		     { 
			 xvw_set_attribute(selection->value_object, XVW_LEFT_OF,
                                           NULL);
		     }
		     break;
	    }

	    if (selection->pix_object != NULL)
                xvw_destroy(selection->pix_object);
            selection->pix_object = NULL;
        }

        /*
         *  the selection WAS non-live;  when making it live, must
         *  create the <cr> pixmap object at the end of the selection
         *  and re-adjust layout of selection.
         */
        else
        {
            switch (selection->type)
            {
                case KUIS_INPUTFILE:
                case KUIS_OUTPUTFILE:
                case KUIS_STRING:
                case KUIS_LOGICAL:
                case KUIS_STRINGLIST:
                case KUIS_CYCLE:
                case KUIS_LIST:
                case KUIS_FLAG:
                     selection->pix_object = 
				xvf_create_cr_pixmap(selection->back);
		     xvw_set_attribute(selection->value_object, XVW_LEFT_OF, 
				       selection->pix_object);
                     break;

                case KUIS_DISPLAYLIST:
                     selection->pix_object = 
				xvf_create_cr_pixmap(selection->back);
		     break;

                case KUIS_FLOAT:
                case KUIS_INTEGER:
                case KUIS_DOUBLE:
                     if (selection->scroll  != NULL)
		     {
                         selection->pix_object = 
				xvf_create_cr_pixmap(selection->back);
			 xvw_set_attribute(selection->scroll, XVW_LEFT_OF, 
                                           selection->pix_object);
			 xvw_set_attribute(selection->value_object, XVW_LEFT_OF,
					   KMANAGER_UNDEFINED);
		     }
                     else 
		     { 
			 selection->pix_object = 
				xvf_create_cr_pixmap(selection->back);
			 xvw_set_attribute(selection->value_object, XVW_LEFT_OF,
                                           selection->pix_object);
		     }
                     break;

            }
        }

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_variable
|       Purpose: Gets the value of the variable field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_VARIABLE ("xvf_variable")
|        Output: calldata    - passes back string value of variable
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_variable(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        return (kvf_get_variable(kformstruct, KVF_VARIABLE, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_variable
|       Purpose: Sets the value of the variable field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_VARIABLE ("xvf_variable")
|                calldata    - provides string value of variable
|        Output: none
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_variable(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        if (!(kvf_set_variable(kformstruct, KVF_VARIABLE, calldata)))
            return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_description
|       Purpose: Gets the value of the description field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DESCRIPTION ("xvf_description")
|        Output: calldata    - passes back string value of description
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_description(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        return (kvf_get_description(kformstruct, KVF_DESCRIPTION, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_description
|       Purpose: Sets the value of the description field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DESCRIPTION ("xvf_description")
|                calldata    - provides string value of description
|        Output: none
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_description(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        if (!(kvf_set_description(kformstruct, KVF_DESCRIPTION, calldata)))
            return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_title
|       Purpose: Gets the value of the title field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_TITLE ("xvf_title")
|        Output: calldata    - passes back string value of title
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_title(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        return (kvf_get_title(kformstruct, KVF_TITLE, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_title
|       Purpose: Sets the value of the title field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_TITLE ("xvf_title")
|                calldata    - provides string value of title
|        Output: none
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_title(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	xvobject object, label = NULL;
	float    new_size;
	char     *new_title;

        if (!(kvf_set_title(kformstruct, KVF_TITLE, calldata)))
            return(FALSE);

	if (kformstruct->type == KFORM)
	    label = kformstruct->Formptr->master->label_object;
	else if (kformstruct->type == KSUBFORM)
	{
	    object = xvf_get_xvobject(kformstruct, XVF_BACKPLANE, TRUE);
	    if (object == kformstruct->Subformptr->button)
	        label = object;
	    else label = kformstruct->Subformptr->label_object;
	}
	else if (kformstruct->type == KGUIDE)
	    label = kformstruct->Guideptr->back_subform->guidepane->label_object;
	else if (kformstruct->type == KPANE)
	    label = kformstruct->Controlptr->label_object;
        else if (kformstruct->type == KMASTER)
            label = kformstruct->Controlptr->label_object;
        else if (kformstruct->type == KGUIDEPANE)
            label = kformstruct->Controlptr->label_object;
        else if (kformstruct->type == KCONTROL)
            label = kformstruct->Controlptr->label_object;
	else if (kformstruct->type == KSELECTION)
	    label = kformstruct->Selptr->label_object;
	else return(FALSE);

	if (label == NULL) return(FALSE);

        if (label != NULL)
        {
            if (kformstruct->type == KSELECTION)
            {
                if (kformstruct->Selptr->opt_object != NULL)
                    xvw_set_attribute(label, XVW_RIGHT_OF,
                                      kformstruct->Selptr->opt_object);
            }
	    
	    kvf_get_attribute(kformstruct, KVF_TITLE, &new_title);
	    if (!new_title) new_title = kstrdup(" ");

	    new_size = (float) kstrlen(new_title);
	    if (kformstruct->Selptr->type == KUIS_STRINGLIST) new_size += 3.0;

	    if ((kformstruct->Selptr->type == KUIS_MASTERACTION)  ||
                (kformstruct->Selptr->type == KUIS_SUBFORMACTION) ||
                (kformstruct->Selptr->type == KUIS_GUIDEBUTTON)   ||
                (kformstruct->Selptr->type == KUIS_HELP)          ||
                (kformstruct->Selptr->type == KUIS_QUIT)          ||
                (kformstruct->Selptr->type == KUIS_ROUTINE)       ||
                (kformstruct->Selptr->type == KUIS_PANEACTION))
                 xvf_set_object_label_or_pixmap(label, XVF_BUTTON_OBJ, 
					        new_title);
	    else xvf_set_object_label_or_pixmap(label, XVF_LABEL_OBJ, 
						new_title);
	    kfree(new_title);
        }

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_buttontitle
|       Purpose: Gets the value of the title field
|                from the UIS line of a GUI button
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_BUTTONTITLE ("xvf_buttontitle")
|        Output: calldata    - passes back string value of title
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_buttontitle(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        return (kvf_get_buttontitle(kformstruct, KVF_BUTTONTITLE, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_buttontitle
|       Purpose: Sets the value of the title field
|                on the UIS line of a GUI button
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_BUTTONTITLE ("xvf_buttontitle")
|                calldata    - provides string value of title
|        Output: none
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_buttontitle(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	xvobject      button = NULL;
	float         new_size, current_size;
	char          *new_title;

        if (!(kvf_set_buttontitle(kformstruct, KVF_BUTTONTITLE, calldata)))
            return(FALSE);

	if (kformstruct->type == KSUBFORM)
	    button = kformstruct->Subformptr->button;
	else if (kformstruct->type == KPANE)
	    button = kformstruct->Controlptr->back_guide->button;
	else if (kformstruct->type == KGUIDE)
	    button = kformstruct->Guideptr->button;
	else if (kformstruct->type == KSELECTION)
	    button = kformstruct->Selptr->back;
	else return(FALSE);

	if (button == NULL) return (FALSE);
	
	kvf_get_buttontitle(kformstruct, KVF_BUTTONTITLE, &new_title);
	new_size = (float) (kstrlen(new_title));
        xvw_get_attribute(button, XVW_CHAR_WIDTH, &current_size);

	xvf_set_object_label_or_pixmap(button, XVF_LABEL_OBJ, new_title);
	if (kstrncmp(new_title, "pixmap:", 7) != 0)
	{
	    xvw_set_attribute(button, XVW_CHAR_MIN_WIDTH, new_size);
	    if (current_size < new_size)
                xvw_set_attribute(button, XVW_CHAR_WIDTH, new_size);
	}

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_get_guidepanetitle
|       Purpose: Gets the value of the title field
|                from the -G StartGuide UIS line of a subform
|
|         Input: kformstruct - generic kformstruct holding ptr to subform
|                attribute   - XVF_GUIDEPANETITLE ("xvf_guidepanetitle")
|        Output: calldata    - passes back string value of guide pane title
|
|          Date: November 28, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_guidepanetitle(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        return (kvf_get_guidepanetitle(kformstruct, 
				       KVF_GUIDEPANETITLE, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_guidepanetitle
|       Purpose: Sets the value of the title field
|                on the -G StartGuide UIS line of a subform
|
|         Input: kformstruct - generic kformstruct holding ptr to subform
|                attribute   - XVF_GUIDEPANETITLE ("xvf_title")
|                calldata    - provides string value of guide pane title
|        Output: none
|
|          Date: November 28, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_guidepanetitle(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	ksubform *subform;
	char     **new_title;
	xvobject label;

	new_title = (char **) calldata;

	if (!(kvf_set_guidepanetitle(kformstruct, 
				     KVF_GUIDEPANETITLE, calldata)))
            return(FALSE);

	subform = kformstruct->Subformptr;
	label = xvf_get_xvobject(subform->guidepane->back_kformstruct, 
				  XVF_LABEL_OBJ, TRUE);
	xvf_set_object_label_or_pixmap(label, XVF_LABEL_OBJ, *new_title);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|
|       Routine: xvf_get_literal
|       Purpose: Gets the value of the literal field
|                from the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LITERAL ("xvf_literal")
|        Output: calldata    - passes back string value of literal
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_literal(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        return (kvf_get_literal(kformstruct, KVF_LITERAL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_literal
|       Purpose: Sets the value of the literal field
|                on the UIS line of a GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_LITERAL ("xvf_literal")
|                calldata    - provides string value of literal
|        Output: none
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_literal(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char **literal;

        if (!(kvf_set_literal(kformstruct, KVF_LITERAL, calldata)))
            return(FALSE);

	literal = (char **) calldata;
	if (kformstruct->Selptr->value_object != NULL)
	    xvw_set_attribute(kformstruct->Selptr->value_object,
			      XVW_TEXT_STRING, *literal);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|
|       Routine: xvf_get_modified
|       Purpose: Gets the value of the "modified" field
|                directly from the kselection structure
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_MODIFIED ("xvf_modified")
|        Output: calldata    - passes back int value of modified
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_modified(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        return (kvf_get_modified(kformstruct, KVF_MODIFIED, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_modified
|       Purpose: Sets the value of the "modified" field
|                directly from the kselection structure
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_MODIFIED ("xvf_modified")
|                calldata    - provides int value of modified
|        Output: none
|
|          Date: March 21, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_modified(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        if (!(kvf_set_modified(kformstruct, KVF_MODIFIED, calldata)))
            return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|
|       Routine: xvf_get_clientdata
|       Purpose: Gets the value of the client_data field
|                directly from the kformstruct
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_CLIENTDATA ("kvf_clientdata")
|        Output: calldata    - passes back address of client data
|
|          Date: October 25, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_clientdata(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        kaddr  *client_data;

        /* set the pointer to be returned */
        client_data = (kaddr *) calldata;

        *client_data = kformstruct->client_data;

        return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_clientdata
|       Purpose: Sets the value of the client_data field
|                directly in the kformstruct
|
|         Input: kformstruct - generic kformstruct holding ptr to GUI item
|                attribute   - KVF_CLIENTDATA ("kvf_clientdata")
|                calldata    - provides address of client data
|        Output: none
|
|          Date: October 25, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_clientdata(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        kaddr  *client_data;

        /* the value to which KVF_CLIENTDATA is to be set */
        client_data  = (kaddr *) calldata;

        kformstruct->client_data = *client_data;

        return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_delete
|       Purpose: Called via the editable menuform of a Pane,
|                Guidepane, or Master, this routine deletes a
|                GUI item
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DELETE ("xvf_delete")
|                calldata    - not used
|        Output: none
|
|          Date: Dec 7, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_delete(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	ksubform     *subform;
	int          status;

	if (kformstruct->type == KSELECTION)
	{
	    if (!(xvf_delete_selection(kformstruct->Selptr)))
	        return(FALSE);
	}
	else if (kformstruct->type == KSUBFORM)
	{
	    subform = kformstruct->Subformptr;
	    if ((subform->back_form->master != NULL) &&
		(subform->back_form->master->subform_list == subform) && 
		(subform->next == NULL))
	        status = kprompt(KSTANDARD, "Delete", "Cancel", 0,
	 		         "About to delete the only subform button on the master form (making the master form obsolete).  The master form will be deleted, and any selections on the master form will be moved to the subform.  Are you SURE you want to delete this subform button (and the master it's on)?");

	   else status = kprompt(KSTANDARD, "Delete", "Cancel", 0,
	 		         "About to delete the subform button, as well as the entire subform which is displayed when the user clicks on the subform button.  Are you SURE you want to delete this subformbutton & subform?");
	    if (!status) return(TRUE);

	    if (!(xvf_delete_subform(subform->back_form, subform)))
	        return(FALSE);
	}
	else if (kformstruct->type == KGUIDE)
	{
	    status = kprompt(KSTANDARD, "Delete", "Cancel", 0,
			     "About to delete the guide button, as well as the entire pane which is displayed when the user clicks on the guide button.  Are you SURE you want to delete this guidebutton & pane?");
	    if (!status) return(TRUE);

	    if (!(xvf_delete_guide(kformstruct->Guideptr)))
	        return(FALSE);
	}
	else if (kformstruct->type == KPANE)
        {
	    if (kformstruct->Controlptr->back_guide->button != NULL)
	       status = kprompt(KSTANDARD, "Delete", "Cancel", 0,
                             "About to delete the pane, as well as the guide button that brings up this pane.  Are you SURE you want to delete this pane & guidebutton?");
	    else status = kprompt(KSTANDARD, "Delete", "Cancel", 0,
                             "About to delete the pane. Are you SURE you want to delete this pane?");

            if (!status) return(TRUE);

            if (!(xvf_delete_guide(kformstruct->Controlptr->back_guide)))
                return(FALSE);
        }
	else
        {
            kerror("xvforms", "xvf_set_delete",
                   "kformstruct of unsupported type passed in");
            return(FALSE);
        }


	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_set_display_subform
|       Purpose: This routine sets a subform to displayed or un-displayed
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DISPLAY_SUBFORM ("xvf_display_subform")
|                calldata    - not used
|        Output: none
|
|          Date: Jan 27, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_display_subform(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        ksubform     *subform, *old_subform;
	int          *display;

	/* the value to which XVF_DISPLAY_SUBFORM is to be set */
        display  = (int *) calldata;

	if (kformstruct->type != KSUBFORM)
        {
            kerror("xvforms", "xvf_set_display_subform",
                   "kformstruct is not of type subform");
            return(FALSE);
        }
	subform = kformstruct->Subformptr;

	/*
	 * if we have mutually exclusive subforms, and we are 
         * turning on a new subform, need to turn off the old subform
	 */
	if ((subform->back_form->master != NULL) && (*display) &&
            (subform->back_form->master->excl_subforms == 1))
        {
            old_subform = subform->back_form->master->subform_list;
	    while (old_subform != NULL)
            {
                if ((old_subform->selected == TRUE) && 
                    (old_subform != subform))
                {
                    if (old_subform->button != NULL)
                    {
                        xvw_reverse_colors(old_subform->button, FALSE);
                        xvf_unmap_subform(old_subform);
                    }
	        }
	        old_subform = old_subform->next;
	    }
	}
	
	if (!(kvf_set_attribute(kformstruct, KVF_DISPLAY_SUBFORM,
				*display)))
	    return(FALSE);

	if (subform->button != NULL)
        {
            if (subform->back_submenu == NULL)
                xvw_reverse_colors(subform->button, *display);
            if (*display == TRUE)
                xvf_map_subform(subform);
           else xvf_unmap_subform(subform);
        }

	if (!(xvf_collect_check_data(subform->back_form)))
	    return(FALSE);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_set_display_pane
|       Purpose: This routine sets a pane to displayed.
|                The new pane is displayed, its guide button is highlighted
|                accordingly.  The old pane is unmapped, and its guide button
|                is unhighlighted.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DISPLAY_PANE ("xvf_display_pane")
|                calldata    - not used
|        Output: none
|
|          Date: Jan 28, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_display_pane(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        kguide       *guide, *guideptr, *guide_list;
	int          *display;

	/* the value to which XVF_DISPLAY_PANE is to be set */
        display  = (int *) calldata;

	if ((*display != 0) && (*display != 1))
        {
            errno = KINVALID_PARAMETER;
            kerror("xvforms", "xvf_set_display_pane",
                   "You must provide value of either 1 or 0");
            return(FALSE);
        }

        if (kformstruct->flag == KUIS_STARTPANE)
	{
            guide_list = kformstruct->Controlptr->back_subform->guidepane->guide_list;
	    guide = kformstruct->Controlptr->back_guide;
	}
        else if (kformstruct->flag == KUIS_GUIDEBUTTON)
	{
            guide_list = kformstruct->Guideptr->back_subform->guidepane->guide_list;
	    guide = kformstruct->Guideptr;
	}
        else
        {
            kerror("xvforms", "xvf_set_display_pane",
                   "Internal Error: unexpected kformstruct type");
            return(FALSE);
        }


        /*
         * if we are turning on new guide button,
         * we must first turn off the old one
         */
        if (*display)
        {
	    guideptr = guide_list;
            while (guideptr != NULL)
            {
                if ((guideptr->selected == TRUE) && (guideptr->pane != NULL))
                {
                    if (guideptr->button != NULL)
                    {
                        xvw_reverse_colors(guideptr->button, FALSE);
		        if (guideptr->pane->back != NULL)
                            xvw_unmap(guideptr->pane->back);
                    }
                }
                guideptr = guideptr->next;
            }
        }

	if (!(kvf_set_attribute(kformstruct, KVF_DISPLAY_PANE, *display)))
	    return(FALSE);

        /*
         * find the guide button in question
         */
        if (guide->button != NULL)
        {
            if (guide->back_submenu == NULL)
                xvw_reverse_colors(guide->button, *display);
            if (*display)
            {
                if (guide->pane->back == NULL)
                {
                    if (!(xvf_create_pane(guide->pane, guide->selected)))
                        return(FALSE);
                }
                xvw_map(guide->pane->back);
            }
            else xvw_unmap(guide->pane->back);
        }

	if (!(xvf_collect_check_data(guide->back_form)))
	    return(FALSE);

        return(TRUE);
}

