/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		Attribute Actions For Double Selections       <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		xvf_get_double_gt0_sel()	      <<<<
   >>>>	     		xvf_set_double_gt0_sel()	      <<<<
   >>>>	     		xvf_get_double_ge0_sel()	      <<<<
   >>>>	     		xvf_set_double_ge0_sel()	      <<<<
   >>>>	     		xvf_get_double_lt0_sel()	      <<<<
   >>>>	     		xvf_set_double_lt0_sel()	      <<<<
   >>>>	     		xvf_get_double_le0_sel()	      <<<<
   >>>>	     		xvf_set_double_le0_sel()	      <<<<
   >>>>	     		xvf_get_double_upper()		      <<<<
   >>>>	     		xvf_set_double_upper()		      <<<<
   >>>>	     		xvf_get_double_lower()		      <<<<
   >>>>	     		xvf_set_double_lower()		      <<<<
   >>>>	     		xvf_get_double_def()		      <<<<
   >>>>	     		xvf_set_double_def()		      <<<<
   >>>>	     		xvf_get_double_val()		      <<<<
   >>>>	     		xvf_set_double_val()		      <<<<
   >>>>	     		xvf_get_precision()		      <<<<
   >>>>	     		xvf_set_precision()		      <<<<
   >>>>                                                       <<<<
   >>>>	     		xvf_change_doublebounds()	      <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"
#include "attributes.h"


/*-----------------------------------------------------------
|
|       Routine: xvf_get_double_gt0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an double selection are > 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_GT0_SEL ("xvf_double_gt0_sel")
|        Output: calldata    - passes back TRUE if bounds > 0 set
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_double_gt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return(kvf_get_double_gt0_sel(kformstruct, KVF_DOUBLE_GT0_SEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_double_gt0_sel
|       Purpose: Sets the bounds of an double selection to > 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_GT0_SEL ("xvf_double_gt0_sel")
|                calldata    - provides value of TRUE if bounds > 0 are set
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_double_gt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_double_gt0_sel(kformstruct, KVF_DOUBLE_GT0_SEL, calldata)))
	    return(FALSE);

	xvf_change_doublebounds(kformstruct, "gt0");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);
	
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_double_ge0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an double selection are >= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_GE0_SEL ("xvf_double_ge0_sel")
|        Output: calldata    - passes back TRUE if bounds are set >= 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_double_ge0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return(kvf_get_double_ge0_sel(kformstruct, KVF_DOUBLE_GE0_SEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_double_ge0_sel
|       Purpose: Sets the bounds of an double selection to >= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_GE0_SEL ("xvf_double_ge0_sel")
|                calldata    - provides boolean value of TRUE if bounds are >= 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_double_ge0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_double_ge0_sel(kformstruct, KVF_DOUBLE_GE0_SEL, calldata)))
	    return(FALSE);

	xvf_change_doublebounds(kformstruct, "ge0");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);
	
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_double_lt0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an double selection are < 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_LT0_SEL ("xvf_double_lt0_sel")
|        Output: calldata    - passes back TRUE if bounds are set to < 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_double_lt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return(kvf_get_double_lt0_sel(kformstruct, KVF_DOUBLE_LT0_SEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_double_lt0_sel
|       Purpose: Sets the bounds of an double selection to < 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_LT0_SEL ("xvf_double_lt0_sel")
|                calldata    - provides boolean value of TRUE if bounds are < 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_double_lt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_double_lt0_sel(kformstruct, KVF_DOUBLE_LT0_SEL, calldata)))
	    return(FALSE);

	xvf_change_doublebounds(kformstruct, "lt0");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);
	
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_double_le0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an double selection are <= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_LE0_SEL ("xvf_double_le0_sel")
|        Output: calldata    - passes back TRUE if bounds are set to <= 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_double_le0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return(kvf_get_double_le0_sel(kformstruct, KVF_DOUBLE_LE0_SEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_double_le0_sel
|       Purpose: Sets the bounds of an double selection to <= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_LE0_SEL ("xvf_double_le0_sel")
|                calldata    - provides boolean value of TRUE if bounds are <= 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_double_le0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_double_le0_sel(kformstruct, KVF_DOUBLE_LE0_SEL, calldata)))
	    return(FALSE);

	xvf_change_doublebounds(kformstruct, "le0");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_double_upper
|       Purpose: Gets the value of the upper bound for an double,
|                assuming that "xvf_double_upper_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_UPPER ("xvf_double_upper")
|        Output: calldata    - passes back double upper bound value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_double_upper(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return(kvf_get_double_upper(kformstruct, KVF_DOUBLE_UPPER, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_double_upper
|       Purpose: Sets the value of the upper bound for an double,
|                assuming that "xvf_double_upper_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_UPPER ("xvf_double_upper")
|                calldata    - provides double upper bound value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_double_upper(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_double_upper(kformstruct, KVF_DOUBLE_UPPER, calldata)))
	    return(FALSE);

	xvf_change_doublebounds(kformstruct, "bounds");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_double_lower
|       Purpose: Gets the value of the lower bound for an double,
|                assuming that "xvf_double_lower_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_LOWER ("xvf_double_lower")
|        Output: calldata    - passes back double lower bound value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_double_lower(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return(kvf_get_double_lower(kformstruct, KVF_DOUBLE_LOWER, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_double_lower
|       Purpose: Sets the value of the lower bound for an double,
|                assuming that "xvf_double_lower_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_LOWER ("xvf_double_lower")
|                calldata    - provides double lower bound value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_double_lower(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_double_lower(kformstruct, KVF_DOUBLE_LOWER, calldata)))
	    return(FALSE);

	xvf_change_doublebounds(kformstruct, "bounds");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_get_double_def
|       Purpose: Gets the value of the default for an double
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_DEF ("xvf_double_def")
|        Output: calldata    - passes back double default value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_double_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return(kvf_get_double_def(kformstruct, KVF_DOUBLE_DEF, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_double_def
|       Purpose: Sets the value of the default for an double
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_DEF ("xvf_double_def")
|                calldata    - provides double default value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_double_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	double *double_def;

	if (!(kvf_set_double_def(kformstruct, KVF_DOUBLE_DEF, calldata)))
	    return(FALSE);

	double_def = (double *) calldata;
	if (!(xvf_set_double_val(kformstruct, XVF_DOUBLE_VAL, double_def)))
	    return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_get_double_val
|       Purpose: Gets the value of the double
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_VAL ("xvf_double_val")
|        Output: calldata    - passes back double value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_double_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return(kvf_get_double_val(kformstruct, KVF_DOUBLE_VAL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_double_val
|       Purpose: Sets the value of an double
|                menuform field => current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_VAL ("xvf_double_val")
|                calldata    - provides double value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_double_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	kselection   *selection;
	Line_Info    line_info;
	char         scale[KLENGTH], temp[KLENGTH];

	if (!(kvf_set_double_val(kformstruct, KVF_DOUBLE_VAL, calldata)))
	    return(FALSE);

	selection = kformstruct->Selptr;

	if ((selection->value_object != NULL) ||
            (selection->scroll != NULL))
	{
	    kvf_clear_line_info(&line_info);
	    kvf_parse_double_line(selection->line, &line_info);

	    if (line_info.precision == 0)
	        ksprintf(scale, "%%g");
	    else if (line_info.precision == -1)
	        ksprintf(scale, "%%f");
	    else ksprintf(scale, "%%.%df", line_info.precision);
	    ksprintf(temp, scale, line_info.double_val);

	    if (selection->value_object != NULL)
                xvw_set_attribute(selection->value_object, 
				 XVW_TEXT_STRING, temp);

	    if (selection->scroll != NULL)
            {
                xvw_set_attributes(selection->scroll,
                                   XVW_SCROLL_MIN,   line_info.lower_double,
                                   XVW_SCROLL_MAX,   line_info.upper_double,
                                   XVW_SCROLL_VALUE, line_info.double_val,
                                   NULL);
            }
	    kvf_free_line_info_strings(&line_info);
	}



	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_precision
|       Purpose: Gets the number of decimal places for a float or double
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_PREC ("xvf_double_prec") or 
|                              XVF_FLOAT_PREC  ("xvf_float_prec")
|        Output: calldata    - passes back int value according to # decimal places
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_precision(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (kstrcmp(attribute, XVF_DOUBLE_PREC) == 0)
	    return(kvf_get_precision(kformstruct, KVF_DOUBLE_PREC, calldata));

	else return (kvf_get_precision(kformstruct, KVF_FLOAT_PREC, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_precision
|       Purpose: Sets the number of decimal places for a float or double
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_DOUBLE_PREC ("xvf_double_prec") or 
|                              XVF_FLOAT_PREC  ("xvf_float_prec")
|                calldata    - provides number of decimal places
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_precision(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         temp[KLENGTH];
	char         scale[KLENGTH];
	int          *precision; 
	double       doubleval;
	float        floatval;

        if (kstrcmp(attribute, XVF_DOUBLE_PREC) == 0)
        {
            if (!(kvf_set_precision(kformstruct, KVF_DOUBLE_PREC, calldata)))
                return(FALSE);
        }
        else if (kstrcmp(attribute, XVF_FLOAT_PREC) == 0)
        {
            if (!(kvf_set_precision(kformstruct, KVF_FLOAT_PREC, calldata)))
                return(FALSE);
        }

	if (kformstruct->Selptr->value_object != NULL)
	{
	    precision = (int *) calldata;

	    if (*precision == 0)
	        ksprintf(scale, "%%g");
	    else if (*precision == -1)
	        ksprintf(scale, "%%f");
	    else ksprintf(scale, "%%.%df", *precision);
	    if (kstrcmp(attribute, XVF_DOUBLE_PREC) == 0)
	    {
	        kvf_get_attribute(kformstruct, KVF_DOUBLE_VAL, &doubleval);
	        ksprintf(temp, scale, doubleval);
	    }
	    else if (kstrcmp(attribute, XVF_FLOAT_PREC) == 0)
	    {
	        kvf_get_attribute(kformstruct, KVF_FLOAT_VAL, &floatval);
	        ksprintf(temp, scale, floatval);
	    }
	    xvw_set_attribute(kformstruct->Selptr->value_object,
			      XVW_TEXT_STRING, temp);
	}

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_change_doublebounds
|
|       Purpose: Changes the bounds for a double selection
|
|         Input: kformstruct - generic kform_struct pointer
|		 string    - string rep of double bounds, may be one of:
|			     "gt0"        : bounds >  0
|			     "ge0"        : bounds >= 0
|			     "lt0"        : bounds <  0
|			     "le0"        : bounds <= 0
|                            "lower = %f" : lower bound 
|                            "upper = %f" : upper bound 
|			     "nobounds"   : no bounds 
|
|        Output: Returns TRUE on success, FALSE on Failure
|          Date: March 20, 1994 
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

int xvf_change_doublebounds(
   kform_struct *kformstruct,
   char         *string)
{
	kselection *selection;
	Line_Info  line_info;
	int        scrollbar = FALSE;
	char       temp[KLENGTH], scale[KLENGTH];

	kvf_clear_line_info(&line_info);
	selection = kformstruct->Selptr;
        kvf_parse_double_line(selection->line, &line_info);

	if (kstrcmp(string, "bounds") == 0)
	   scrollbar = TRUE;

	/*
         *  take off scrollbar if there is one
         */
        if ((!scrollbar) && (selection->scroll != NULL))
        {
             xvw_set_attribute(selection->back, XVF_MECHANISM, 0);
        }
        
	if (selection->value_object != NULL)
	{
            if (line_info.precision == 0)
                ksprintf(scale, "%%g");
            else if (line_info.precision == -1)
                ksprintf(scale, "%%f");
            else ksprintf(scale, "%%.%df", line_info.precision);
            ksprintf(temp, scale, line_info.double_val);
            xvw_set_attribute(selection->value_object, XVW_TEXT_STRING, temp);
	}

	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}

