#include "xvinclude.h"

/*
 *  In this example, a list GUI object offering a choice of three animals is
 *  displayed.  The user is allowed to click on an item, causing a popup
 *  with information about the animal to be displayed.
 *  The callback installed on list item selection needs an array
 *  of structures passed in, which is allocated and passed into the callback 
 *  using the client_data structure.  Because it is a list GUI object, the 
 *  call_data pointer is also used.
 */


/*
 *  here is the structure containing the information we need, 
 *  an array of which to be passed to the callback as the client_data
 */
typedef struct _AnimalInfo {
	
    int   id;
    char  *habitat;
    char  *diet;
    int   endangered;

} AnimalInfo;
    
/*
 *  defines to identify animal
 */
#define MONKEY   0
#define WOLF     1
#define ELEPHANT 2

/*
 *  prototype for callback
 */
void print_info PROTO((xvobject, kaddr, kaddr));

/*
 *  here's the main program
 */
void main(
   int  argc,
   char **argv,
   char **envp)
{
	xvobject    comp_list; 
        xvobject    actual_list;
	xvobject    manager;
	int         foreign;
	AnimalInfo **animal_info;

	static char *animals[] = { "monkey", "wolf", "elephant" };
	int         num = knumber(animals);

        /* initialize Khoros program */
        khoros_initialize(argc, argv, envp, "DESIGN");

	/* initialize xvwidgets lib */
	if (!xvw_initialize(XVW_MENUS_XVFORMS))
	{
	   kerror("example", "main", "Cannot open display");
           kexit(KEXIT_FAILURE);
	}

	/* create manager backplane, width 10, height 4 */
	manager = xvw_create_manager(NULL, "back");
	xvw_set_attributes(manager,
			   XVW_CHAR_WIDTH, 15.0,
			   XVW_CHAR_HEIGHT, 4.0,
			   NULL);

	/* create compound list object, tack it to parent */
	comp_list = xvw_create_list(manager, "comp_list");
	xvw_set_attribute(comp_list, XVW_TACK_EDGE, KMANAGER_TACK_ALL);

	/* get actual list object from compound list object */
	actual_list = xvw_retrieve_list(comp_list);

	/* add list contents to actual list */
	xvw_change_list(actual_list, animals, num, TRUE);

	/*
	 * allocate & initialize client_data
 	 */
	animal_info = (AnimalInfo **) kcalloc(3, sizeof(AnimalInfo *));

	animal_info[MONKEY] = (AnimalInfo *) kcalloc(1, sizeof(AnimalInfo));
	animal_info[MONKEY]->id         = 1230;
	animal_info[MONKEY]->habitat    = kstrdup("South America");
	animal_info[MONKEY]->diet       = kstrdup("Omniverous"); 
	animal_info[MONKEY]->endangered = FALSE;

	animal_info[WOLF] = (AnimalInfo *) kcalloc(1, sizeof(AnimalInfo));
	animal_info[WOLF]->id           = 1894;
	animal_info[WOLF]->habitat      = kstrdup("North America"); 
	animal_info[WOLF]->diet         = kstrdup("Carnivorous"); 
	animal_info[WOLF]->endangered   = TRUE;

	animal_info[ELEPHANT] = (AnimalInfo *) kcalloc(1,sizeof(AnimalInfo));
	animal_info[ELEPHANT]->id         = 2113;
	animal_info[ELEPHANT]->habitat    = kstrdup("Africa"); 
	animal_info[ELEPHANT]->diet       = kstrdup("Herbivorous"); 
	animal_info[ELEPHANT]->endangered = TRUE;
	
	/* 
         * add callback to list selection that will print info. 
         * pass the "animal_info" data structure as client_data
         */
	xvw_add_callback(actual_list, XVW_LIST_ITEM_SELECT, print_info, 
                         animal_info); 
	
	/* display and run */
	xvf_run_form();
}

/*
 *  callback prints information about the animal chosen from the list
 */
void print_info(
    xvobject button,
    kaddr    client_data,
    kaddr    call_data)  
{
	AnimalInfo    **animal_info; /* array of animal info structures */
	xvw_list_struct *list_return; /* holds info about selected item */ 
	int             animal_index; /* index of selected animal       */
	char            *animal_name; /* name of selected animal        */

	/* cast client data pointer to expected data type (defined by us) */
	animal_info = (AnimalInfo **) client_data;

	/* cast call_data pointer to data type (defined by list gui object) */
	list_return = (xvw_list_struct *) call_data;

	if ( list_return == NULL || animal_info == NULL ) return;

	/* index and string associated w/ selection stored in list struct */
	animal_index = list_return->list_index;
	animal_name  = list_return->string;

	if ( animal_index == -1 || animal_name == NULL ) return;

	kinfo(KSTANDARD, "Selected %s,\n Index %d,\n ID %d,\n Habitat %s,\n %s Diet", 
              animal_name, animal_index, 
              animal_info[animal_index]->id, 
              animal_info[animal_index]->habitat, 
              animal_info[animal_index]->diet);

	if (animal_info[animal_index]->endangered)
	   kinfo(KSTANDARD, "Warning! This is an endangered species");
}
