 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros XPM Data Service Routines
   >>>>
   >>>>   Static:
   >>>>  Private:
   >>>>             xpm_check()
   >>>>             xpm_input()
   >>>>             xpm_output()
   >>>>             _init()
   >>>>             xpm_destroy()
   >>>>   Public:
   >>>>             None - no public should ever exist as these are
   >>>>                    internal routines only accessible via the
   >>>>                    DataServiceInformation *services[] structure.
   >>>>
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


#if !defined(KXPM_DEF)
DataServiceInformation xpm_format[] =
{NULL};

#else

#include "kdatafmt/xpm.h"


static int xpm_check PROTO((int));
static int xpm_input PROTO((kobject, int, int));
static int xpm_output PROTO((kobject, int, int));
static int xpm_destroy PROTO((kobject));
static int xpm_order PROTO((char *, int *));
static int xpm_architecture PROTO((kobject));

DataServiceInformation xpm_format[] =
{
   {
      "X11 Pixmap File Format (xpm)",
      "xpm",
      xpm_check,
      xpm_input,
      xpm_output,
      xpm_destroy,
      NULL,
      NULL,
      xpm_order,
      xpm_architecture,
      NULL,
      NULL,
   }
};

/*
 *  Internal Resource Structure for the following data services
 *
 *              xpm  -  X11 Pixmap File Format Utilities (xpm)
 */
typedef struct
{
   kfile *file;
   xpm *image;
}
ResourceStruct;


/*-----------------------------------------------------------
|
|  Routine Name: _init - Creates an xpm image
|
|       Purpose: This function is used to create an xpm image.  Which
|                means that we use the current dimension values in order
|                to create a proper xpm.
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:53
| Modifications:
|
------------------------------------------------------------*/

static int
_init(kobject object)
{
   ResourceStruct *resources;


   /*
    * Create the image to the object file
    */
   resources = (ResourceStruct *) kcalloc(1, sizeof(ResourceStruct));

   if (resources == NULL)
      return (FALSE);

   _kdms_glue_set_resources(object, resources);
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xpm_check - Checks to see if the file is
|                                an xpm
|
|       Purpose: This function is used to check the first few bytes of
|                the stream to see if it is an xpm.
|
|         Input: object - the xpm object to be initialized
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:53
| Modifications:
|
------------------------------------------------------------*/

static int
xpm_check(int fid)
{
   /*
    * Seems like the easiest way to do this is to check the header
    */
   if (xpm_readheader(fid) == NULL)
      return (FALSE);
   else
      return (TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: xpm_input - Reda a xpm image
|
|       Purpose: This function is used to read in the image if the
|                supplied data is xpm image.  The data was checked
|                by the xpm_check() routine in which the xpm
|                identifier indicated that the data is a valid xpm.
|                Since the data was valid we can assume simply read
|                the header in and initialize that data segments.
|
|         Input: object - the xpm object to be initialized
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:53
| Modifications:
|
------------------------------------------------------------*/

/*ARGSUSED */
static int
xpm_input(kobject object, int fid, int flags)
{
   ResourceStruct *resources;
   xpm *image;

   int type = KUBYTE,
      i,
      j;
   int size[KDMS_MAX_DIM],
      order[KDMS_MAX_DIM],
      begin[KDMS_MAX_DIM],
      end[KDMS_MAX_DIM];

   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * Read the image from the object file and go ahead and set the xpm image
    * within the resource structure.
    */
   image = resources->image = xpm_fdread(fid);
   if (image == NULL)
   {
      _kdms_set_error(KDMS_EFMT_FAILURE);
      return (FALSE);
   }

   /*
    * Initialize the data segments according to the data found in the xpm
    * structure.
    * 
    * The xpm structure only contains value (image->data)
    */

   /* check the value data */
   if (image->width != 0 && image->height != 0)
   {
      type = KUBYTE;

      size[0] = image->width;
      size[1] = image->height;
      size[2] = size[3] = size[4] = 1;

      order[0] = KWIDTH;
      order[1] = KHEIGHT;
      order[2] = KELEMENTS;
      order[3] = KDEPTH;
      order[4] = KTIME;

      for (i = 0; i < KDMS_MAX_DIM; i++)
      {
         end[i] = size[i] - 1;
         begin[i] = 0;
      }

      _kdms_set_segment(object, KDMS_SEGMENT_VALUE, image->data, type,
                        order, size, begin, end, 5);

      /*
       * it is vitally important that you include this step because if the
       * data set larger than the buffer threshold, then spurous flushes
       * will ensue that should not happen...it turns out that xvimages and
       * viffs will cope with that situation (though not very efficiently),
       * but file formats that buffer the whole thing in memory won't.
       */
      kdms_set_attribute(object, KDMS_SEGMENT_VALUE, KDMS_COUPLING,KUNCOUPLED);
   }

   /* do the maps */
   if (image->ncolors != 0)
   {
      unsigned short *data;

      size[0] = 3;
      size[1] = image->ncolors;
      size[2] = size[3] = size[4] = 1;

      data = (unsigned short *)kmalloc(size[0] * size[1] *
                                       sizeof(unsigned short));

      order[0] = KMAP_WIDTH;
      order[1] = KMAP_HEIGHT;
      order[2] = KMAP_ELEMENTS;
      order[3] = KMAP_DEPTH;
      order[4] = KMAP_TIME;

      for (i = 0; i < KDMS_MAX_DIM; i++)
      {
         end[i] = size[i] - 1;
         begin[i] = 0;
      }

      for (i = 0, j = 0; i < size[1]; i++)
      {
         data[j++] = image->maps[i].red;
         data[j++] = image->maps[i].green;
         data[j++] = image->maps[i].blue;
      }

      _kdms_set_segment(object, KDMS_SEGMENT_MAP, data, KUSHORT,
                        order, size, begin, end, 5);

      /*
       * it is vitally important that you include this step because if the
       * data set larger than the buffer threshold, then spurous flushes
       * will ensue that should not happen...it turns out that xvimages and
       * viffs will cope with that situation (though not very efficiently),
       * but file formats that buffer the whole thing in memory won't.
       */
      kdms_set_attribute(object, KDMS_SEGMENT_MAP, KDMS_COUPLING,KUNCOUPLED);
   }
   return (TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: xpm_output - Closes the xpm image
|
|       Purpose: This function is used to close the xpm image.  Which
|                means if the image was modified we then re-write image
|                back out.
|
|         Input: object - the xpm object to be closed
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:53
| Modifications:
|
------------------------------------------------------------*/
#define XPM_MAX_VAL 65535.0

/*ARGSUSED */
static int
xpm_output(kobject object, int fid, int flags)
{
   ResourceStruct *resources;
   unsigned short *data;
   xpm *image;
   int *size,
      i,
      j;
   int vorder[KDMS_MAX_DIM] = {KWIDTH, KHEIGHT, KDEPTH, KTIME, KELEMENTS};
   int morder[KDMS_MAX_DIM] ={KMAP_WIDTH, KMAP_HEIGHT, KMAP_DEPTH,KMAP_TIME, 
			      KMAP_ELEMENTS};
   int begin[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
   int end[KDMS_MAX_DIM];
   
   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (resources->image == NULL)
   {
      resources->image = xpm_create();
      if (!resources->image)
      {
         _kdms_traceback("xpm_output");
         _kdms_set_error(KDMS_EFMT_FAILURE);
         return (FALSE);
      }
   }
   image = resources->image;

   
   /*
    * Get the value segment.  Xpm is limited to char data only, with a
    * specific index order. So, first we gonna set the data type and 
    * index order, then sync downward and then get the segment.
    */
   if (!kdms_set_attributes(object, KDMS_SEGMENT_VALUE, KDMS_DATA_TYPE, KBYTE,
                            KDMS_INDEX_ORDER, vorder, NULL))
      return (FALSE);

   kdms_get_attribute(object, KDMS_SEGMENT_VALUE, KDMS_SIZE, &size);
   
   end[0] = size[0] - 1;
   end[1] = size[1] - 1;
   end[2] = end[3] = end[4] = 0;
   
   image->width = (int)size[0];
   image->height = (int)size[1];

   image->data = kdms_get_data(object, KDMS_SEGMENT_VALUE, begin, end, NULL);
   
   /*
    * if the map segment exists, then set the datatype to short and 
    * normalize in the range 2^16 - 1.
    */
   if (kdms_query_segment(object, KDMS_SEGMENT_MAP))
   {
      kdms_set_attributes(object, KDMS_SEGMENT_MAP, KDMS_DATA_TYPE, KUSHORT, 
			  KDMS_INDEX_ORDER, morder, KDMS_SCALING, KNORMALIZE,
			  KDMS_NORM_MIN, 0.0, KDMS_NORM_MAX, XPM_MAX_VAL,
			  NULL);

      /* get the size and override the width to 3 */
      kdms_get_attribute(object, KDMS_SEGMENT_MAP, KDMS_SIZE, &size);
      size[0] = 3;
      kdms_set_attribute(object, KDMS_SEGMENT_MAP, KDMS_SIZE, size);

      end[0] = size[0] - 1;
      end[1] = size[1] - 1;
      end[2] = size[2] - 1;
      end[3] = size[3] - 1;
      end[4] = size[4] - 1;
      
      data = kdms_get_data(object, KDMS_SEGMENT_MAP, begin, end, NULL);
      image->maps = (xpmcolor *) kmalloc(size[1] * sizeof(xpmcolor));

      for (i = 0, j = 0; i < size[1]; i++)
      {
         image->maps[i].red = data[j++];
         image->maps[i].green = data[j++];
         image->maps[i].blue = data[j++];
         image->maps[i].flags = 0xff;
         image->maps[i].val = 0;
      }
      image->ncolors = size[1];
   } 
   else 
   {
      image->maps = (xpmcolor *) kmalloc(256 * sizeof(xpmcolor));
      for (i = 0; i < 256; i++)
      {
	 image->maps[i].red = image->maps[i].green = image->maps[i].blue = 
	    i << 8;
	 image->maps[i].flags = 0xff;
	 image->maps[i].val = NULL;
      }
      image->ncolors = 256;
   }
   
   /*
    * Write the xpm structure out to the specified filename
    */
   return(xpm_fdwrite(fid, image));
}

/*-----------------------------------------------------------
|
|  Routine Name: xpm_destroy - Frees an xpm image
|
|       Purpose: This function is used to create an xpm image.  Which
|                means that we use the current dimension values in order
|                to create a proper xpm.
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:53
| Modifications:
|
------------------------------------------------------------*/

static int
xpm_destroy(kobject object)
{
   ResourceStruct *resources;

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * free the xpmer structure
    */
   xpm_free(resources->image);

   /*
    * free the resources
    */
   kfree(resources);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xpm_order
|
|       Purpose:
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:53
| Modifications:
|
------------------------------------------------------------*/

static int
xpm_order(char *seg, int *ord)
{
   if (kstring_to_token(seg) == kstring_to_token(KDMS_SEGMENT_MAP))
   {
      ord[0] = KMAP_WIDTH;
      ord[1] = KMAP_HEIGHT;
      ord[2] = KMAP_DEPTH;
      ord[3] = KMAP_TIME;
      ord[4] = KMAP_ELEMENTS;
   }
   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: xpm_architecture
|
|       Purpose: Return the architecture that the data was
|                generated on in khoros-speak.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:54
| Modifications:
|
------------------------------------------------------------*/

static int
xpm_architecture(kobject object)
{
   ResourceStruct *resources;

   /*
    * get the resource structure
    */
   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (resources == NULL)
      return (KMACH_LOCAL);
   else
      return (kfile_getmachtype(kfileno(resources->file)));

}

#endif /* KXPM_DEF */

/* don`t add after the endif */
