 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros RAW Data Service Routines
   >>>>
   >>>>   Static:
   >>>>  Private:
   >>>>             raw_check()
   >>>>             raw_input()
   >>>>             raw_output()
   >>>>             _init()
   >>>>             raw_destroy()
   >>>>   Public:
   >>>>             None - no public should ever exist as these are
   >>>>                    internal routines only accessible via the
   >>>>                    DataServiceInformation *services[] structure.
   >>>>
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static int raw_check PROTO((int));
static int raw_input PROTO((kobject, int, int));
static int raw_output PROTO((kobject, int, int));
static int raw_destroy PROTO((kobject));
static kaddr raw_read_data PROTO((kobject, int, kaddr *, int *, int *));
static int raw_write_data PROTO((kobject, int, kaddr, int *, int *));
static int raw_get_architecture PROTO((kobject));
static int raw_set_architecture PROTO((kobject, int));

DataServiceInformation raw_format[] =
{
   {
      "The Raw Un-format",
      "raw",
      raw_check,
      raw_input,
      raw_output,
      raw_destroy,
      raw_read_data,
      raw_write_data,
      NULL,
      raw_get_architecture,
      raw_set_architecture,
      NULL,
   }
};

typedef struct
{
   kfile *file;
}
ResourceStruct;


/*-----------------------------------------------------------
|
|  Routine Name: _init - Creates an raw image
|
|       Purpose: This function is used to create an raw image.  Which
|                means that we use the current dimension values in order
|                to create a proper raw.
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
_init(kobject object)
{
   ResourceStruct *resources;


   /*
    * Create the image to the object file
    */
   resources = (ResourceStruct *) kcalloc(1, sizeof(ResourceStruct));

   if (resources == NULL)
      return (FALSE);

   _kdms_glue_set_resources(object, resources);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: raw_info(object)
|
|       Purpose: This routine is used primarily to query
|                the nature of a segment in the raw.
|                the main piece of information is offset.
|
|         Input: object - object.
|
|        Output: offset - the offset of the data segment for
|                         the object in question.
|
|       Returns: TRUE (1) when that segment does exist in this
|                file format, FALSE (0) when the segment does
|                not exist in this file format.
|
|    Written By: Jeremy Worley
|          Date: Jan 29, 1993 01:22
| Modifications:
|
------------------------------------------------------------*/

/*ARGSUSED */
static int
raw_info(kobject object, char *segment, int *offset)
{
   *offset = 0;
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: raw_check - Checks to see if the file is
|                                an raw
|
|       Purpose: This function is used to check the first few bytes of
|                the stream to see if it is an raw.
|
|         Input: object - the raw object to be initialized
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
raw_check(int fid)
{
   if (fid < 0)
      return (FALSE);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: raw_input - Reda a raw image
|
|       Purpose: This function is used to read in the image if the
|                supplied data is raw image.  The data was checked
|                by the raw_check() routine in which the raw
|                identifier indicated that the data is a valid raw.
|                Since the data was valid we can assume simply read
|                the header in and initialize that data segments.
|
|         Input: object - the raw object to be initialized
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
raw_input(kobject object, int fid, int flags)
{
   ResourceStruct *resources;
   char type[KLENGTH];
   int kflags;

   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   kflags = _kdms2kfile_flags(flags);

   resources->file = kfdopen(fid, kflags_to_type(kflags, type));

   if (resources->file == NULL)
   {
      _kdms_set_error(KDMS_EFMT_OPENFAIL);
      return (FALSE);
   }

   _kdms_set_segment(object, KDMS_SEGMENT_VALUE, NULL, 0,
                     NULL, NULL, NULL, NULL, 5);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: raw_output - Closes the raw image
|
|       Purpose: This function is used to close the raw image.  Which
|                means if the image was modified we then re-write image
|                back out.
|
|         Input: object - the raw object to be closed
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
raw_output(kobject object, int fid, int flags)
{
   ResourceStruct *resources;
   char type[KLENGTH];
   int kflags;

   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   kflags = _kdms2kfile_flags(flags);

   if ((resources->file = kfdopen(fid, kflags_to_type(kflags, type))) 
       == NULL)
      return (FALSE);
   

   if (resources->file == NULL)
   {
      _kdms_set_error(KDMS_EFMT_OPENFAIL);
      return (FALSE);
   }

   /*
    * whats left to do?
    */
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: raw_destroy - Frees an raw image
|
|       Purpose: This function is used to create an raw image.  Which
|                means that we use the current dimension values in order
|                to create a proper raw.
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
raw_destroy(kobject object)
{
   ResourceStruct *resources;

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * block and wait until I can have the descriptor to myself
    */
   kflock(kfileno(resources->file), KLOCK_EX);
   kfclose(resources->file);

   /*
    * free the resources
    */
   kfree(resources);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: raw_read_data
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 02, 1993 07:04
| Modifications:
|
------------------------------------------------------------*/

static kaddr
raw_read_data(kobject object, int seg, kaddr * data, int *begin, int *end)
{
   kpresentation *pres;
   int offset;
   ResourceStruct *resources;

   if (!_kdms_get_segment_info(object, seg, &pres) ||
       !pres)
      return (FALSE);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   raw_info(object, ktoken_to_string(seg), &offset);

   *data = _kdms_glue_read_segment_data(
                                          resources->file,
                                kfile_getmachtype(kfileno(resources->file)),
                                          kfread_generic,
                                          kfseek,
                                          offset,
                                          pres->segment->datatype,
                                          pres->segment->dimension,
                                          pres->segment->size,
					  NULL,
                                          data,
                                          begin,
                                          end);

   return (*data);
}

/*-----------------------------------------------------------
|
|  Routine Name: raw_write_data
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 02, 1993 07:09
| Modifications:
|
------------------------------------------------------------*/

static int
raw_write_data(kobject object, int seg, kaddr data, int *begin, int *end)
{
   ResourceStruct *resources;
   kpresentation *pres;
   int offset;

   if (!_kdms_get_segment_info(object, seg, &pres) ||
       !pres)
      return (FALSE);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   raw_info(object, ktoken_to_string(seg), &offset);

   return (_kdms_glue_write_segment_data(
                                           resources->file,
                                kfile_getmachtype(kfileno(resources->file)),
                                           kfwrite_generic,
                                           kfseek,
                                           offset,
                                           pres->segment->datatype,
                                           pres->segment->dimension,
                                           pres->segment->size,
					   NULL,
                                           data,
                                           begin,
                                           end));
}


/*-----------------------------------------------------------
|
|  Routine Name: raw_get_architecture
|
|       Purpose: Return the architecture that the data was
|                generated on in khoros-speak.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 11:35
| Modifications:
|
------------------------------------------------------------*/

static int
raw_get_architecture(kobject object)
{
   ResourceStruct *resources;

   /*
    * get the resource structure
    */
   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (resources == NULL)
      return (KMACH_LOCAL);
   else
      return (kfile_getmachtype(kfileno(resources->file)));

}

/*-----------------------------------------------------------
|
|  Routine Name: raw_set_architecture
|
|       Purpose: Return the architecture that the data was
|                generated on in khoros-speak.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 11:35
| Modifications:
|
------------------------------------------------------------*/

static int
raw_set_architecture(kobject object, int mach)
{
   ResourceStruct *resources;

   /*
    * get the resource structure
    */
   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (resources == NULL)
      return (FALSE);
   else
       return (kfile_setmachtype(kfileno(resources->file), mach) == 0);

}
