 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros EPS Data Service Routines
   >>>>
   >>>>   Static:
   >>>>		genrev
   >>>>		set_default_values
   >>>>		print_byte_image
   >>>>		print_bit_image
   >>>>		dump_postscript
   >>>>  Private:
   >>>>         eps_output()
   >>>>         eps_destroy()
   >>>>   Public:
   >>>>             None - no public should ever exist as these are
   >>>>                    internal routines only accessible via the
   >>>>                    DataServiceInformation *services[] structure.
   >>>>
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


#if !defined(KEPS_DEF)
DataServiceInformation eps_format[] =
{NULL};

#else

static int eps_output PROTO((kobject, int, int));
static int eps_destroy PROTO((kobject));

DataServiceInformation eps_format[] =
{
   {
      "Encapsulated Postscript Output",
      "eps",
      NULL,
      NULL,
      eps_output,
      eps_destroy,
      NULL,
      NULL,
      NULL,
      NULL,
      NULL,
      NULL,
   }
};

typedef struct
{
   kfile *file;
}
ResourceStruct;


/*-----------------------------------------------------------
|
|  Routine Name: (static) genrev
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Scott Wilson & Mark Young
|          Date: Aug 05, 1994 09:27
| Modifications:
|
------------------------------------------------------------*/

static void
genrev(unsigned char *p)
{
   /* Generate an array of bit-reverse characters */
    int i;
    unsigned char j,k;

    for (i=0; i<256; i++)
    {
        j = (unsigned char)i;
        k = 0;
        k |= (j >> 7) & (1 << 0); /* Bit 7 to Bit 0 */
        k |= (j >> 5) & (1 << 1); /* Bit 6 to Bit 1 */
        k |= (j >> 3) & (1 << 2); /* Bit 5 to Bit 2 */
        k |= (j >> 1) & (1 << 3); /* Bit 4 to Bit 3 */
        k |= (j << 1) & (1 << 4); /* Bit 3 to Bit 4 */
        k |= (j << 3) & (1 << 5); /* Bit 2 to Bit 5 */
        k |= (j << 5) & (1 << 6); /* Bit 1 to Bit 6 */
        k |= (j << 7) & (1 << 7); /* Bit 0 to Bit 7 */
        p[i]=k;
    }
}


/*-----------------------------------------------------------
|
|  Routine Name: (static) set_default_values
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Aug 05, 1994 09:35
| Modifications:
|
------------------------------------------------------------*/
static void
set_default_values(
   double *width, 
   double *height, 
   double *xoffset, 
   double *yoffset, 
   int rows,
   int cols,
   int xflg,
   int yflg,
   double maxpagewidth,
   double maxpageheight) 
{
   *width = cols / 72.0;
   *height = rows / 72.0;

   if (xflg == 0) 
      *xoffset = (maxpagewidth - *width)/2.0;
   
   if (yflg == 0) 
      *yoffset = (maxpageheight - *height)/2.0;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) print_byte_image
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Scott Wilson & Mark Young
|          Date: Aug 05, 1994 09:35
| Modifications:
|
------------------------------------------------------------*/

static void
print_byte_image(
   kfile *file, 
   unsigned char *data,
   int rows, 
   int cols,
   int bands,
   unsigned char *maps)
{
   int        i, j;
   unsigned char *p1,*p2,*p3; /* Data access variables */

   if (bands == 1 && maps == NULL)
   {
      /* Have a standard 1 plane 8 bit greyscale image */
      kfprintf(file, "/picstr %d string def\n",cols);
      kfprintf(file, "%d %d 8 [%d 0 0 -%d 0 %d]\n",cols,rows,cols,rows,rows);
      kfprintf(file, "{currentfile picstr readhexstring pop} image\n");
 
      for (i = 0; i < rows; i++)
      {
	 for (j = 0; j < cols; j++)
	    kfprintf(file, "%02x", *data++);
	 
	 kfprintf(file, "\n");
      }
   }
   else  /* Have a color image to deal with */
   {
      /* Declare three strings, one for RED, GREEEN, and BLUE */
      kfprintf(file, "/rstr %d string def\n",cols);
      kfprintf(file, "/gstr %d string def\n",cols);
      kfprintf(file, "/bstr %d string def\n",cols);
      
      /* Set up the source and mapping */
      kfprintf(file, "%d %d 8 [%d 0 0 -%d 0 %d]\n",cols,rows,cols,rows,rows);

      /* Set up the read procedures */
      (void)kfprintf(file,"{currentfile rstr readhexstring pop}\n");
      (void)kfprintf(file,"{currentfile gstr readhexstring pop}\n");
      (void)kfprintf(file,"{currentfile bstr readhexstring pop}\n");

      /* 
       * Set up the colorimage operator with multiple procedures and 3
       * colors 
       */
      (void)kfprintf(file,"true 3 colorimage\n");
      
      /* Start burping out the data */
      if (bands == 3)
      {
	 /* Have a 24 bit, 3 plane RGB image */
	 p1 = data;
	 p2 = p1 + rows*cols;
	 p3 = p2 + rows*cols;
	 for (i=0; i< rows; i++)
	 {
	    for (j=0; j<cols; j++) /* RED */
	       (void)kfprintf(file,"%02x",*p1++);
	    kfprintf(file,"\n");
	    for (j=0; j<cols; j++) /* GREEN */
	       (void)kfprintf(file,"%02x",*p2++);
	    kfprintf(file,"\n");
	    for (j=0; j<cols; j++) /* BLUE */
	       (void)kfprintf(file,"%02x",*p3++);
	    kfprintf(file,"\n");
	 }
      }
      else if (bands == 1)
      {
	 /* Hav	e a 1 plane, 8-bit image that has a 3-band map, of type RGB */
	 for (i=0; i<rows; i++)
	 {
	    for (j=0; j<cols; j++) /* RED */
	       (void)kfprintf(file,"%02x",maps[3 * data[i * cols + j]]);
	    (void)kfprintf(file,"\n");
	    for (j=0; j<cols; j++) /* GREEN */
	       (void)kfprintf(file,"%02x",maps[3 * data[i * cols + j] + 1]);
	    (void)kfprintf(file,"\n");
	    for (j=0; j<cols; j++) /* BLUE */
	       (void)kfprintf(file,"%02x",maps[3 * data[i * cols + j] + 2]);
	    (void)kfprintf(file,"\n");
	 }
      } 
   }
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) print_bit_image
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Mark Young & Scott Wilson
|          Date: Aug 05, 1994 09:37
| Modifications:
|
------------------------------------------------------------*/

static void
print_bit_image(
   kfile *file,
   unsigned char *data,
   int rows,
   int cols, 
   int photo_neg)
{
   int        k, i, j;
   unsigned   char *ptr, rev[256];

   k = (cols+7)/8;

   /* Flip the bits around to match the POSTSCRIPT idea of pixels */
   ptr = (unsigned char *) data;
   genrev(rev);
   for (i = 0; i < k*rows; i++)
   {
      *ptr = rev[*ptr]; ptr++;
   }
   
   /* Do photonegative if requested */
   if (photo_neg)
   {
      ptr = (unsigned char *) data;
      for (i = 0; i < k*rows; i++)
      {
	 *ptr = ~(*ptr); ptr++;
      }
   }

   kfprintf(file, "/picstr %d string def\n",k);
   kfprintf(file, "%d %d 1 [%d 0 0 -%d 0 %d]\n",cols,rows,cols,
	    rows,rows);
   kfprintf(file, "{currentfile picstr readhexstring pop} image\n");

   /* Sp	it it out */
   ptr = (unsigned char *) data;
   for (i = 0; i < rows; i++)
   {
      for (j = 0; j < k; j++)
	 kfprintf(file, "%02x", *ptr++);
      
      kfprintf(file, "\n");
   }
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) dump_postscript
|       Purpose: This function was converted from the K1
|		 lvpostscr routine.
|         Input:
|    		 file        - file descriptor to write the output to
|    
|    		 force       - flag to force output of the current page
|    
|    		 photo_neg   - flag to  indicate  that  photonegative  output
|			       is desired
|    
|    		 wflg        - flag to indicate that the given  width  is  to
|			       be used rather than the computed one
|    
|    		 hflg        - flag to indicate that the given height  is  to
|			       be used rather than the computed one
|    
|    		 xflg        - flag to indicate that the given X position  is
|			       to be used rather than the computed one
|    
|    		 yflg        - flag to indicate that the given Y position  is
|			       to be used rather than the computed one
|    
|    		 def_width   - tells what the desired output width is (inches)
|    
|    		 def_height  - tells what the desired output height is (inches)
|    
|    		 def_xoffset - tells what the desired X offset is (inches)
|    
|    		 def_yoffset - tells what the desired Y offset is (inches)
|    
|    		landscape    - indicates if output should be in landscape mode
|        Output:
|    file           this descriptor receives the Postscript output
|
|       Returns: TRUE on success, FALSE otherwise
|    Written By: Scott Wilson, Mark Young
|          Date: Aug 05, 1994 09:14
| Modifications:
|
------------------------------------------------------------*/

static int
dump_postscript(
   kfile *file, 
   unsigned char *data,
   int type, /* byte or bit */
   int cols,
   int rows,
   int num_bands,
   unsigned char *maps,
   int force, 
   int photo_neg, 
   int xflg, 
   int yflg,
   double def_width, 
   double def_height, 
   double def_xoffset, 
   double def_yoffset, 
   int landscape)
{
   time_t     tclock;
   int        sx, sy, dx, dy;
   double      width = def_width, height = def_height, 
              xoffset = def_xoffset, yoffset = def_yoffset;
   double maxpagewidth, maxpageheight;

   /*	
    * Intiailize values for the "set_default_values" routine.  This
    * routine is used to make some assumtions about the aspect ratio
    * and also the position of the output image on the page. 
    */
   if (landscape)
   {
      maxpagewidth = 11.0;
      maxpageheight = 8.5;
   }
   else
   {
      maxpagewidth = 8.5;
      maxpageheight = 11.0;
   }

   set_default_values(&width, &height, &xoffset, &yoffset, rows, 
		      cols, xflg, yflg, 
		      maxpagewidth, maxpageheight);

   if (photo_neg && num_bands != 1)
      return(0);

   if ((xoffset + width) > maxpagewidth || (yoffset + height) > maxpageheight)
      return(0);

   /*
    * The scaling factor and translation distance is in terms of 72
    * pixels per inch.  So we multiple the number of inches by 72 and
    * truncate to the nearest pixel value. 
    */
   sx = (int) (width * 72.0);
   sy = (int) (height * 72.0);
   dx = (int) (xoffset * 72.0);
   dy = (int) (yoffset * 72.0);
      
   /*
    * Write header information.
    */
   kfprintf(file,"%%!PS-Adobe-2.0 EPSF-1.2\n");
   kfprintf(file,"%%%%DocumentFonts: Courier\n");
   kfprintf(file,"%%%%Title: Khoros Postscript Image Document\n");
   kfprintf(file,"%%%%Creator: Data Services\n");
   (void) time(&tclock);
   kfprintf(file,"%%%%CreationDate: %s",ctime(&tclock));
   kfprintf(file,"%%%%BoundingBox: %d %d %d %d\n",dx,dy, dx+sx, dy+sy);
   kfprintf(file,"%%%%Pages: 1\n");
   kfprintf(file,"%%%%EndComments\n");
   kfprintf(file,"initgraphics\n");
   kfprintf(file, "save\n");
   kfprintf(file,"%%%%EndProlog\n");
   kfprintf(file,"%%%%Page: 1 1\n");

      
   /* 
    * Print scale and translation information. 
    */
   if (landscape)
   {
      kfprintf(file, "90 rotate\n");
      kfprintf(file, "0 -%d translate\n", sy);
      kfprintf(file, "%d -%d translate\n",dy,dx);
   }
   else
   {
      kfprintf(file, "%d %d translate\n",dx,dy);
   }
   kfprintf(file, "%d %d scale\n",sx,sy);

   if (type == KUBYTE)
   {
      if (photo_neg)
      {
	 kinfo(KSTANDARD, "photo negative still not supported...\n");
      }
      print_byte_image(file, data, rows, cols, num_bands, maps);
   }
   else /* BIT */
   {
      print_bit_image(file, data, rows, cols, photo_neg);
   }

   kfprintf(file, "restore\n");
   if (force)
   {
      kfprintf(file, "showpage\n");
   }   /* flush the file to write the last little bit in the buffer */
   kfflush(file);
   return(1);
}


/*-----------------------------------------------------------
|
|  Routine Name: eps_output
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Aug 05, 1994 10:00
| Modifications:
|
------------------------------------------------------------*/

/*ARGSUSED*/
static int
eps_output(kobject object, int fid, int flags)
{
   kfile *file;
   int begin[5] = {0, 0, 0, 0, 0};
   int end[5];
   int *size;
   int *msize;
   unsigned char *data;
   unsigned char *maps = NULL;
   int type;
   int order[KDMS_MAX_DIM] = {KWIDTH, KHEIGHT, KDEPTH, KTIME, KELEMENTS};

   if ((file = kfdopen(fid, "w")) == NULL)
      return (FALSE);
   
   kdms_get_attribute(object, KDMS_SEGMENT_VALUE,
		      KDMS_PHYSICAL_DATA_TYPE, &type);
   
   if (type == KBIT)
   {
      kdms_set_attribute(object, KDMS_SEGMENT_VALUE, KDMS_INDEX_ORDER, order);
      kdms_get_attribute(object, KDMS_SEGMENT_VALUE, KDMS_SIZE, &size);

      end[0] = size[0] - 1;
      end[1] = size[1] - 1;
      end[2] = size[2] - 1;
      end[3] = size[3] - 1;
      end[4] = size[4] - 1;

      data = kdms_get_data(object, KDMS_SEGMENT_VALUE, begin, end, NULL);
      
      dump_postscript(file, data, KBIT, size[0], size[1], size[4],
		      NULL, TRUE, FALSE, FALSE, FALSE, 
		      1.0, 1.0, 1.0, 1.0, FALSE);
   }
   else
   {
      if (kdms_query_segment(object, KDMS_SEGMENT_VALUE) == FALSE)
	 return (FALSE);
      
      kdms_set_attributes(object, KDMS_SEGMENT_VALUE, 
			  KDMS_DATA_TYPE, KUBYTE,
			  KDMS_INDEX_ORDER, order, NULL);
      
      kdms_get_attribute(object, KDMS_SEGMENT_VALUE, KDMS_SIZE, &size);

      end[0] = size[0] - 1;
      end[1] = size[1] - 1;
      end[2] = size[2] - 1;
      end[3] = size[3] - 1;
      end[4] = size[4] - 1;
      
      data = kdms_get_data(object, KDMS_SEGMENT_VALUE, begin, end, NULL);

      if (kdms_query_segment(object, KDMS_SEGMENT_MAP) == TRUE)
      {
	 kdms_set_attribute(object, KDMS_SEGMENT_MAP, KDMS_DATA_TYPE, KUBYTE);
	 kdms_set_attributes(object, KDMS_SEGMENT_MAP, 
			     KDMS_DATA_TYPE, KUBYTE,
			     KDMS_INDEX_ORDER, order, 
			     KDMS_SCALING, KNORMALIZE,
			     KDMS_NORM_MIN, 0.0,
			     KDMS_NORM_MAX, 255.0, 
			     NULL);
      
	 kdms_get_attribute(object, KDMS_SEGMENT_MAP, KDMS_SIZE, &msize);
      
	 end[0] = msize[0] - 1;
	 end[1] = msize[1] - 1;
	 end[2] = msize[2] - 1;
	 end[3] = msize[3] - 1;
	 end[4] = msize[4] - 1;
	 
	 maps = kdms_get_data(object, KDMS_SEGMENT_MAP, begin, end, NULL);
      }

      dump_postscript(file, data, KUBYTE, size[0], size[1], size[4],
		      maps, TRUE, FALSE, FALSE,
		      FALSE, 1.0, 1.0, 1.0, 1.0, FALSE);
   }
   
   kfclose(file);
   kfree(data);
   kfree(maps);
   
   return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: eps_destroy
|       Purpose: 
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Aug 05, 1994 10:06
| Modifications:
|
------------------------------------------------------------*/

static int
eps_destroy(kobject object)
{
   ResourceStruct *resources;

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   kfree(resources);

   return (TRUE);
}
#endif
