 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */



/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Ascii File Format Utilities
   >>>>
   >>>>	    The "ascii" format describes the conversion used for
   >>>>	    the format:
   >>>>
   >>>>			Ascii File Format
   >>>>
   >>>>	    The format was orignially developed by the company/persons:
   >>>>
   >>>>			
   >>>>	    
   >>>>	    The following routines are used by the conversion routines
   >>>>	    to read and write the ascii files.  These routines are not a
   >>>>	    standard, just an a set of routines used by the Khoros routines
   >>>>	    to manipulate Ascii File Format.
   >>>>
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>   Public:
   >>>>             ascii_readheader()
   >>>>             ascii_read()
   >>>>             ascii_writeheader()
   >>>>             ascii_write()
   >>>>             ascii_free()
   >>>>             ascii_create()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <kdatafmt/ascii.h>
#define  KDATAFMT "kdatafmt"

#define ALLOCSIZE 10000


/************************************************************
*
*  Routine Name: ascii_readheader - reads a ascii header structure
*				      from the specified kfile id
*
*       Purpose: This routines reads a ascii header structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the ascii header.
*
*         Input: fid - the kfile id opened previously with kopen()
*        Output:
*       Returns: image - explanation
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Nov 14, 1993  
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

/*ARGSUSED*/
ascii *ascii_readheader(
   int	fid)
{
	ascii *image;


	/* 
	 * Sanity check to make sure that the image pointer is not NULL
	 */
	if ((image = (ascii *) kcalloc(1, sizeof(ascii))) == NULL)
	   return(NULL);

	return(image);
}

/************************************************************
*
*  Routine Name: ascii_fdread - read a ascii structure from the
*				specified file descriptor
*
*       Purpose: This routines reads a ascii structure and it's
*		 data from the specified file descriptor.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the data and it's header.
*
*         Input: fid - file descriptor in which we will be reading
*			    the ascii
*        Output:
*       Returns: returns the newly read ascii or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 14, 1993 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

ascii *ascii_fdread(int fid)
{
   kfile   *file;
   ascii   *image;
   int	i, c, row_size, width, size = 0;
   char	dummy[512], character[2], *line, *temp;
   
   double rval, ival;
   kdcomplex *cmplx  = NULL;
   double *values = NULL;

   /*
    *  Read the Ascii File Format header
    */
   if ((image = ascii_readheader(fid)) == NULL)
      return(NULL);

   /*
    *  Now read the data associated with the
    *  Ascii File Format header
    */
   file = kfdopen(fid, "r");

   /* read data and determine number of points */
   i = 0; width = 0;
   while ((line = kfgets(dummy, 512, file)) != NULL)
   {
      row_size = 0;
      do
      {
	 rval = strtod(line, &temp);
	 if (line != temp)
	 {
	    if (cmplx != NULL)
	    {
	       errno = KINVALID_INPUT;
	       kfree(values);
	       kfree(cmplx);
	       return(NULL);
	    }
	    
	    if (i == size)
	    {
	       size += ALLOCSIZE;
	       if ((values = (double *) krealloc(values, (unsigned) size *
						 sizeof(double))) == NULL)
		  return(NULL);
	    }
	    values[i] = rval;
	    i++; row_size++;
	 }
	 else if (ksscanf(line, " (%lg,%lg) %[^\n]", &rval, &ival,
			  temp) >= 2)
	 {
	    if (values != NULL)
	    {
	       errno = KINVALID_INPUT;
	       kfree(values);
	       kfree(cmplx);
	       return(NULL);
	    }
	    
	    if (i == size)
	    {
	       size += ALLOCSIZE;
	       if ((cmplx = (kdcomplex *) krealloc(cmplx, (unsigned) size *
						   sizeof(kdcomplex))) == NULL)
		  return(NULL);
	    }
	    cmplx[i].r = rval;
	    cmplx[i].i = ival;
	    i++; row_size++;
	 }
	 else
	 {
	    /*
	     *  Check to see if we encountered a comment.  We do this
	     *  by seeing if the comment begins with a "%" "*" "#" or
	     *  "!".  If no comment was found then print an error message
	     *  and return NULL, otherwise read the rest of line.
	     */
	    if (ksscanf(line, "%[#]", character) == 1)
	    {
	       /*
		*  Read to the rest of the comment which should be until
		*  the end of the line.
		*/
	       temp = NULL;
	    }
	    else if (ksscanf(line, "%[/]%[*]", character, character) == 2)
	    {
	       /*
		*  Read to the end of the comment.
		*/
	       while (*temp != '\0')
	       {
		  if (*temp == '*')
		  {
		     temp++;
		     if (*temp == '/')
		     {
			temp++;
			break;
		     }
		  }
		  temp++;
	       }

	       if (*temp == '\0')
	       {
		  for (;;)
		  {
		     while ((c = kfgetc(file)) != '*' && c!=EOF && c!='\0')
			;
		     
		     if (c == '*')
		     {
			if ((c = kfgetc(file)) == '/')
			   break;
		     }
		     else
			break;
		  }
	       }
	    }
	    else
	    {
	       while (isspace(*temp))
		  temp++;
	       
	       if (*temp != '\0')
	       {
		  errno = KINVALID_INPUT;
		  kfree(values);
		  kfree(cmplx);
		  return(NULL);
	       }
	    }
	 }
	 line = temp;
      } while (line != NULL && *line != '\0');
      
      if (row_size > 0) width = row_size;
   }

   if (i == 0)
   {
      ascii_free(image);
      return(NULL);
   }
   if (width <= 0 || (i % width) != 0) width = 1;
   image->width  = width;
   image->height = i/width;
   if (values != NULL)
   {
      image->type = KDOUBLE;
      image->data = (kaddr) values;
   }
   else
   {
      image->type = KDCOMPLEX;
      image->data = (kaddr) cmplx;
   }
   return(image);
}

/************************************************************
*
*  Routine Name: ascii_read - read a ascii structure from the
*				specified filename
*
*       Purpose: This routines reads a ascii structure and it's
*		 data from the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the data and it's header.
*
*         Input: filename - filename in which we will be reading
*			    the ascii
*        Output:
*       Returns: returns the newly read ascii or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 14, 1993 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

ascii *ascii_read(
   char	*filename)
{
   int fid;
   ascii *image;
   
   /* 
    * Open the input file 
    */
   if ((fid = kinput(filename)) < 0)
      return(NULL);

   /*
    *  Lock the transport file descriptor if the transport has
    *  data permanence.
    */
   if (kfile_getpermanence(filename) == TRUE)
      kflock(fid, KLOCK_SH);

   image = ascii_fdread(fid);
   
   return image;
}

/************************************************************
*
*  Routine Name: ascii_fdwrite - write a ascii structure to the
*				 specified file descriptor
*
*       Purpose: This routines writes a ascii structure and it's
*		 data into the specified file descriptor.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the data.
*
*         Input: fid - the file descriptor in which we will be writing
*			    the ascii image and associated data
*                image    - the ascii structure to be written
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young 
*          Date: Nov 14, 1993  
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int ascii_fdwrite(int fid, ascii *image)
{
   kfile  *file;
   double    *values;
   kdcomplex *cmplx;
   int    i, j;
   
   /*
    *  Read the Ascii File Format header
    */
   if (!ascii_writeheader(fid, image))
      return(FALSE);

   /*
    *  Now writer the data associated with the Ascii File Format
    *  header
    */
   file = kfdopen(fid, "w");

   if (image->type == KDCOMPLEX)
   {
      cmplx = (kdcomplex *) image->data;
      
      for (j = 0; j < image->height; j++)
      {
	 for (i = 0; i < image->width; i++)
	 {
	    kfprintf(file, "(%g,%g) ", cmplx->r, cmplx->i);
	    cmplx++;
	 }
	 kfprintf(file, "\n");
      }
   }
   else
   {
      values = (double *) image->data;
      
      for (j = 0; j < image->height; j++)
      {
	 for (i = 0; i < image->width; i++)
	    kfprintf(file, "%g ", *values++);
	 
	 kfprintf(file, "\n");
      }
   }

   return(TRUE);
}

/************************************************************
*
*  Routine Name: ascii_write - write a ascii structure to the
*				 specified filename
*
*       Purpose: This routines writes a ascii structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the data.
*
*         Input: filename - the filename in which we will be writing
*			    the ascii image and associated data
*                image    - the ascii structure to be written
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young 
*          Date: Nov 14, 1993  
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
   
int ascii_write(
   char		*filename,
   ascii	*image)
{
   int fid;
   int status;
   
   /* 
    * Sanity check to make sure that the image pointer is not NULL
    */
   if (image == NULL)
   {
      errno = KNULL_PARAMETER;
      return(FALSE);
   }
   
   /* 
    * Open the output file 
    */
   if ((fid = koutput(filename)) < 0)
      return(FALSE);

   /*
    *  Lock the transport file descriptor if the transport has
    *  data permanence.
    */
   if (kfile_getpermanence(filename) == TRUE)
      kflock(fid, KLOCK_EX);

   status = ascii_fdwrite(fid, image);
   
   kclose(fid);
   
   return status;
}


/************************************************************
*
*  Routine Name: ascii_writeheader - writes a ascii header structure
*				      from the specified kfile id
*
*       Purpose: This routines writes a ascii header structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the ascii header.
*
*         Input: fid - the kfile id opened previously with kopen()
*		 image - the image header to be written
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Nov 14, 1993  
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

/*ARGSUSED*/
int ascii_writeheader(
   int		fid,
   ascii	*image)
{

	/*
	 *  Write the (ascii) header out
	 */
	return(TRUE);
}


/************************************************************
*
*  Routine Name: ascii_free - frees a ascii structure and it's
*				associated data
*
*       Purpose: This routine frees an khoros ascii structure and
*		 it's associated data.
*
*         Input: image - a pointer to an khoros ascii structure that
*			 contains the image structure to be freed.
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Nov 14, 1993 
*      Verified:
*  Side Effects: Once this routine is called no further reference
*		 the image should be made.
* Modifications:
*
*************************************************************/

int ascii_free(
   ascii	*image)
{
	/*
	 *  Sanity check to make sure that the image structure is not NULL.
	 */
	if (!image)
	   return(FALSE);

	/*
	 *  Free the data and all associated resources associated with
	 *  the image.
	 */
	kfree(image->data);

	/*
	 *  Free the ascii image structure itself, which was previously
	 */
	kfree(image);
	return(TRUE);
}


/************************************************************
*
*  Routine Name: ascii_create - creates a ascii structure and it's
*				associated data
*
*       Purpose: This routine creates an khoros ascii structure and
*		 it's associated data.
*
*         Input: image - a pointer to an khoros ascii structure that
*			 contains the image structure to be freed.
*
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Nov 14, 1993 
*      Verified:
*  Side Effects: 
* Modifications:
*
*************************************************************/

ascii *ascii_create(
   void)
{
	ascii *image;

	/* 
	 * Sanity check to make sure that the image pointer is not NULL
	 */
	if ((image = (ascii *) kcalloc(1, sizeof(ascii))) == NULL)
	   return(NULL);

	return(image);
}
