
/*
 * Khoros: $Id$
 */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

#include "copyright.h"		/* Khoros copyright */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>         ARF File Format Utility Routines
   >>>>
   >>>>  Private:
   >>>>	
   >>>>            arf_get_main_hdr()
   >>>>	           arf_get_info()
   >>>>            arf_get_colormap()
   >>>>            arf_get_comment()
   >>>>            arf_get_multi_band()
   >>>>            arf_get_frame_data()
   >>>>            arf_get_ladar_param()
   >>>>            arf_get_footer()
   >>>>            arf2khoros_datatype()
   >>>>            khoros2arf_datatype()
   >>>>            arf_put_info()
   >>>>            arf_put_colormap()
   >>>>            arf_put_comment()
   >>>>            arf_put_multi_band()
   >>>>            arf_put_frame_data()
   >>>>            arf_put_ladar_param()
   >>>>            arf_read_data()
   >>>>
   >>>>   Static:
   >>>> 
   >>>>   Public:
   >>>>           read_xdr_string()
   >>>>           write_xdr_string()
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include <dataserv.h>
#include <kdatafmt/arf.h>

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_get_main_hdr
  |
  |       Purpose: reads ARF_MAIN_HDR structure from ARF file
  |                
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |	 Output:   hdr - ARF_MAIN_HDR structure pointer
  |
  |       Returns: TRUE (1) on success, FALSE (0) on failure
  |    Written By: Tim Williams 
  |          Date: 16 May 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int arf_get_main_hdr(kfile *file, ARF_MAIN_HDR *hdr)
{
  unsigned long tmpval;
  
  
  kfread_generic(file, &hdr->_magic_num, 1, KULONG);
  kfread_generic(file, &hdr->_version, 1, KULONG);
  kfread_generic(file, &hdr->_num_rows, 1, KULONG);
  kfread_generic(file, &hdr->_num_cols, 1, KULONG);
  kfread_generic(file, &tmpval, 1, KULONG);
  hdr->_image_type = (ARF_IMAGE_TYPE)tmpval;
  kfread_generic(file, &hdr->_num_frames, 1, KULONG);
  kfread_generic(file, &hdr->_image_offset, 1, KULONG);
  kfread_generic(file, &hdr->_subheader_flags, 1, KULONG);
  
  return (TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_get_info
  |
  |       Purpose: reads a ARF_INFO structure from an ARF file
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                subh - ARF_SUBHEADER structure 
  |
  |
  |	 Output: ARF_INFO structure, info_length members of ARF_SUBHEADER
  |
  |       Returns: byte length of subheader in file on success, 
  |                FALSE (0) on failure
  |
  |    Written By: Tim Williams, (modified John Horger's arf_get_sub_info)
  |          Date:  10 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int arf_get_info(kfile *file, ARF_SUBHEADER *subh)
{
  unsigned long tmpval;
  int subh_length;
  char *tmpstr;
  
  
  
  tmpstr = (char *)kcalloc(ARF_MAX_SIMPLE_STR_SIZE +1, 1);
  if (!tmpstr) 
  {
    kerror("data services", "arf_get_info", "error in kcalloc for tmpstr\n");
    return (FALSE);
  }
  
  /******  Read the Values  ******/
  /*******************************/
  kfread_generic(file, &tmpval, 1, KULONG);
  subh->info.image_source = (ARF_IMAGE_SOURCE)tmpval;
  kfread_generic(file, &subh->info.start_x, 1, KULONG);
  kfread_generic(file, &subh->info.start_y, 1, KULONG);
  kfread_generic(file, &subh->info.num_avg, 1, KULONG);
  kfread_generic(file, &subh->info.capture_rate, 1, KULONG);
  
  /* next 6 shorts are ARF_IRIG_TIME */
  kfread_generic(file, &tmpval, 1, KULONG);
  subh->info.capture_time.yy = (unsigned short)tmpval;
  kfread_generic(file, &tmpval, 1, KULONG);
  subh->info.capture_time.dd = (unsigned short)tmpval;
  kfread_generic(file, &tmpval, 1, KULONG);
  subh->info.capture_time.hh = (unsigned short)tmpval;
  kfread_generic(file, &tmpval, 1, KULONG);
  subh->info.capture_time.mm = (unsigned short)tmpval;
  kfread_generic(file, &tmpval, 1, KULONG);
  subh->info.capture_time.ss = (unsigned short)tmpval;
  kfread_generic(file, &tmpval, 1, KULONG);
  subh->info.capture_time.ms = (unsigned short)tmpval;
  
  read_xdr_string(file, tmpstr);
  kstrcpy(subh->info.capture_loc, tmpstr);
  
  read_xdr_string(file, tmpstr);
  kstrcpy(subh->info.sensor_name, tmpstr);
  
  read_xdr_string(file, tmpstr);
  kstrcpy(subh->info.digitizer, tmpstr);
  
  kfread_generic(file, &subh->info.sensor_hor_fov, 1, KFLOAT);
  kfread_generic(file, &subh->info.sensor_ver_fov, 1, KFLOAT);
  
  kfread_generic(file, &subh->info.samples_per_dwell, 1, KULONG);
  
  /** update the subhsize ***/
  /* 12 longs + 2 floats + 3 string lengths (ints) */
  subh_length = 68;
  
  tmpval = kstrlen(subh->info.capture_loc);
  if (tmpval%4)
    tmpval += (4 - (tmpval%4));
  subh_length += tmpval;
  
  tmpval = kstrlen(subh->info.sensor_name);
  if (tmpval%4) tmpval += (4 - (tmpval%4));
  subh_length += tmpval;
  
  tmpval = kstrlen(subh->info.digitizer);
  if (tmpval%4) tmpval += (4 - (tmpval%4));
  subh_length += tmpval;
  
  kfree(tmpstr);
  return (subh_length);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_get_colormap
  |
  |       Purpose: reads a ARF_COLORMAP structure from an ARF file
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                subh - ARF_SUBHEADER structure
  |
  |	 Output: ARF_COLORMAP structure member of ARF_SUBHEADER
  |
  |       Returns: byte length of subheader in file on success, 
  |                FALSE (0) on failure
  |
  |    Written By: Tim Williams, (modified John Horger's arf_get_sub_colormap)
  |          Date: 10 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int arf_get_colormap(kfile  *file, ARF_SUBHEADER *subh)
{
  unsigned long tmparray[ARF_COLORMAP_SIZE];
  int i;
  int subh_length;
  unsigned char *ptr;
  
  
  
  kfread_generic(file, tmparray, ARF_COLORMAP_SIZE, KULONG);
  for (i=0, ptr=(unsigned char *)subh->colormap.red; 
       i < ARF_COLORMAP_SIZE; 
       i++, ptr++)
    *ptr = (unsigned char)tmparray[i];
  
  kfread_generic(file, tmparray, ARF_COLORMAP_SIZE, KULONG);
  for (i=0, ptr=(unsigned char*)subh->colormap.green; 
       i < ARF_COLORMAP_SIZE; 
       i++, ptr++)
    *ptr = (unsigned char)tmparray[i];
  
  kfread_generic(file, tmparray, ARF_COLORMAP_SIZE, KULONG);
  for (i=0, ptr=(unsigned char *)subh->colormap.blue;
       i < ARF_COLORMAP_SIZE;
       i++, ptr++)
    *ptr = (unsigned char)tmparray[i];
  
  subh_length = ARF_COLORMAP_SIZE * 4 * 3; /* 256*4*3=3072 */
  return (subh_length);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_get_comment
  |
  |       Purpose: reads an ARF_COMMENT structure from an ARF file
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                subh - ARF_SUBHEADER structure
  |
  |	 Output: comment - ARF_COMMENT structure member of ARF_SUBHEADER
  |
  |       Returns: byte length of subheader in file on success, 
  |                FALSE (0) on failure
  |
  |    Written By: Tim Williams, (modified John Horger's arf_get_sub_comment)
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/
int arf_get_comment (kfile *file, ARF_SUBHEADER *subh)
{
  char *tmpstr;
  int subh_length;
  int length;
  
  
  
  tmpstr = (char *)kmalloc(ARF_MAX_COMMENT_SIZE +1);
  if (!tmpstr)
  {
    kerror("data services", "arf_get_comment", 
	   "not enough memory for tmpstr");
    return (FALSE);
  }
  
  
  read_xdr_string(file, tmpstr);
  kstrcpy(subh->comment.bs, tmpstr);
  length = kstrlen(subh->comment.bs);
  
  subh_length = length%4 ? (4+length+(4-length%4)) : (4+length);
  
  kfree(tmpstr);
  return (subh_length);
  
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_get_multi_band
  |
  |       Purpose: reads a ARF_MULTI_BAND from an ARF file
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                subh - ARF_SUBHEADER structure
  |
  |	 Output: multi_band - ARF_MULTI_BAND structure member of ARF_SUBHEADER
  |
  |       Returns: byte length of subheader in file on success, 
  |                FALSE (0) on failure
  |
  |   Written By: Tim Williams, (modified John Horger's arf_get_sub_multi_band)
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/
int arf_get_multi_band (kfile *file, ARF_SUBHEADER *subh)
{
  unsigned long tmp_ulong;
  long tmpval, i;
  char *tmpstr;
  int empty_multi_band_length;
  int subh_length;
  
  
  
  subh_length = 0;
  tmpstr = (char *)kmalloc(ARF_MAX_NAME_SIZE +1);
  if (!tmpstr) 
  {
    kerror("data services", "arf_get_multi_band", 
	   "not enough memory for tmpstr");
    return (FALSE);
  }
  
  /******  Get the general data for the multi-band  ******/
  /*******************************************************/
  read_xdr_string(file, tmpstr);
  kstrcpy(subh->multi_band.name, tmpstr);
  kfread_generic(file, &tmp_ulong, 1, KULONG);
  subh->multi_band.num_bands = (unsigned short)tmp_ulong;
  
  tmpval = kstrlen(subh->multi_band.name);
  subh_length += tmpval%4 ? 8+tmpval+(4-tmpval%4) : 8+tmpval;
  
  for (i=0 ; i< (int)subh->multi_band.num_bands ; i++)
  {
    read_xdr_string(file, tmpstr);
    kstrcpy(subh->multi_band.band[i].name, tmpstr);
    kfread_generic(file, &tmp_ulong, 1, KULONG);
    subh->multi_band.band[i].type = (ARF_IMAGE_TYPE)tmp_ulong;
    if (subh->multi_band.band[i].type != subh->multi_band.band[0].type)
    {
      kerror("data services", "arf_get_multi_band",
	     "bands must all be of same data type");
      return (FALSE);
    }
    tmpval = kstrlen(subh->multi_band.band[i].name);
    subh_length += 
      tmpval%4 ? 8+tmpval+(4-tmpval%4) : 8+tmpval;
  }     
  
  /* increment subhsize for empty bands up to 16 (ARF_MAX_BANDS) */
  empty_multi_band_length = 8*(ARF_MAX_BANDS-subh->multi_band.num_bands);
  subh_length += empty_multi_band_length;
  
  /* position file past blank bands */
  kfseek(file, empty_multi_band_length, KSEEK_CUR);
  
  kfree(tmpstr);
  return (subh_length);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_get_frame_data
  |
  |       Purpose: reads a ARF_FRAME_DATA structure from an ARF file
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                subh - ARF_SUBHEADER structure
  |
  |	 Output: frame_data - ARF_FRAME_DATA structure member of ARF_SUBHEADER
  |
  |       Returns: byte length of subheader in file on success, 
  |                FALSE (0) on failure
  |
  |   Written By: Tim Williams, (modified John Horger's arf_get_sub_frame_data)
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/
int arf_get_frame_data (kfile *file, ARF_SUBHEADER *subh)
{
  int subh_length;
  
  kfread_generic(file, &subh->frame_data.footer_flags, 1, KULONG);
  subh_length = 4;
  
  return (subh_length);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_get_ladar_param
  |
  |       Purpose: reads a ARF_LADAR_PARAM structure from an ARF file
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                subh - ARF subheader structure
  |
  |	 Output: ladar_param - ARF_LADAR_PARAM structure member 
  |                            of ARF_SUBHEADER
  |
  |       Returns: byte length of subheader in file on success, 
  |                FALSE (0) on failure
  |
  |    Written By: Tim Williams, 
  |                (modified John Horger's arf_get_sub_ladar_param)
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/
int arf_get_ladar_param (kfile *file, ARF_SUBHEADER *subh)
{
  int subh_length;
  
  kfread_generic(file, &subh->ladar_param.fm_scale, 1, KFLOAT);
  kfread_generic(file, &subh->ladar_param.resolved_range_bias, 1, KFLOAT);
  kfread_generic(file, &subh->ladar_param.hor_ifov, 1, KFLOAT);
  kfread_generic(file, &subh->ladar_param.ver_ifov, 1, KFLOAT);
  kfread_generic(file, &subh->ladar_param.coarse_res, 1, KFLOAT);
  kfread_generic(file, &subh->ladar_param.fine_res, 1, KFLOAT);
  
  subh_length = 24;
  return (subh_length);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_get_footer
  |
  |       Purpose: reads ARF footers from an ARF file
  |                
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                footer_flags - footer flags
  |
  |                frame_footer - pointer to ARF_FRAME_FOOTER structure
  |
  |	 Output: 
  |
  |       Returns: byte length of footer in file on success, 
  |                FALSE (0) on failure
  |
  |    Written By: Tim Williams 
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int arf_get_footer (kfile *file, 
		    unsigned long footer_flags, 
		    ARF_FRAME_FOOTER *frame_footer)
{
  unsigned long tmpval;
  int footer_length;
  
  footer_length = 0;
  if (footer_flags & ARF_FRAME_INFO_FLAG)
  {
    kfread_generic(file, &(frame_footer->info.x), 1, KULONG);
    kfread_generic(file, &(frame_footer->info.y), 1, KULONG);
    kfread_generic(file, &(frame_footer->info.azimuth), 1, KFLOAT);
    kfread_generic(file, &(frame_footer->info.elevation), 1, KFLOAT);
    
    /* irig_time */
    kfread_generic(file, &tmpval, 1, KULONG);
    frame_footer->info.irig_time.yy = (unsigned short)tmpval;
    kfread_generic(file, &tmpval, 1, KULONG);
    frame_footer->info.irig_time.dd = (unsigned short)tmpval;
    kfread_generic(file, &tmpval, 1, KULONG);
    frame_footer->info.irig_time.hh = (unsigned short)tmpval;
    kfread_generic(file, &tmpval, 1, KULONG);
    frame_footer->info.irig_time.mm = (unsigned short)tmpval;
    kfread_generic(file, &tmpval, 1, KULONG);
    frame_footer->info.irig_time.ss = (unsigned short)tmpval;
    kfread_generic(file, &tmpval, 1, KULONG);
    frame_footer->info.irig_time.ms = (unsigned short)tmpval;
    
    footer_length += 40;
  }
  
  if (footer_flags & ARF_PLATFORM_ATTITUDE_FLAG)
  {
    kfread_generic(file, &(frame_footer->attitude.vel_north), 1, KDOUBLE);
    kfread_generic(file, &(frame_footer->attitude.vel_east), 1, KDOUBLE);
    kfread_generic(file, &(frame_footer->attitude.vel_vertical), 1, KDOUBLE);
    kfread_generic(file, &(frame_footer->attitude.acc_north), 1, KDOUBLE);
    kfread_generic(file, &(frame_footer->attitude.acc_east), 1, KDOUBLE);
    kfread_generic(file, &(frame_footer->attitude.acc_vertical), 1, KDOUBLE);
    kfread_generic(file, &(frame_footer->attitude.latitude), 1, KDOUBLE);
    kfread_generic(file, &(frame_footer->attitude.longitude), 1, KDOUBLE);
    kfread_generic(file, &(frame_footer->attitude.altitude), 1, KDOUBLE);
    kfread_generic(file, &(frame_footer->attitude.elevation), 1, KDOUBLE);
    kfread_generic(file, &(frame_footer->attitude.bank), 1, KDOUBLE);
    kfread_generic(file, &(frame_footer->attitude.heading), 1, KDOUBLE);
    
    footer_length += 96;
  }
  
  return (footer_length);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: read_xdr_string
  |
  |       Purpose: reads a string from a file in XDR format
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                string - pointer to char string
  |
  |	 Output: none
  |
  |       Returns: nothing
  |    Written By: Tim Williams, 
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/
void read_xdr_string(kfile *file, char *string)
{
  unsigned long length;
  
  kfread_generic(file, &length, 1, KULONG);
  kfread_generic(file, string, 
		 (int)(length%4 ? length+(4-length%4) : length), 
		 KUBYTE);
  string[length]=0; /* make sure string is terminated */
  
  return;
}

/*-----------------------------------------------------------
  |
  |  Routine Name: write_xdr_string
  |
  |       Purpose: writes a string in XDR format
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                string - pointer to char string
  |
  |	 Output: none
  |
  |       Returns: nothing
  |    Written By: Tim Williams, 
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/
void write_xdr_string(kfile *file, char *string)
{
  unsigned long length;
  static char pad[4] = {0,0,0,0};
  
  length = kstrlen(string);
  kfwrite_generic(file, &length, 1, KULONG);
  
  /* write string, pad with zeros to 4-byte multiple (XDR) */
  kfwrite_generic(file, string, (int)length, KUBYTE);
  if (length%4)
    kfwrite_generic(file, pad, (int)(4-length%4), KUBYTE);
  
  return;
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf2khoros_datatype
  |
  |       Purpose: The routine converts the arf datatype
  |		 passed in as input into a khoros 2.0
  |		 data format specification.
  |
  |         Input: arf_type - arf data type.
  |
  |        Output: k_type   - Khoros 2.0 data type
  |
  |       Returns: TRUE (1) on success, FALSE (0) on failure
  |
  |    Written By: Tim Williams (modified from Jeremy Worley's code)
  |          Date: 13 May 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int arf2khoros_datatype(ARF_IMAGE_TYPE arf_type, int *k_type)
{
  switch (arf_type) 
  {
  case arf_eight_bit:
    *k_type = (int)KUBYTE;
    break;
  case arf_ten_bit:
  case arf_twelve_bit:
  case arf_short_word:
    *k_type = (int)KUSHORT;
    break;
  case arf_short_int:
    *k_type = (int)KSHORT;
    break;
  case arf_long_int:
    *k_type = (int)KLONG;
    break;
  case arf_long_word:
    *k_type = (int) KULONG;
    break;
  case arf_float_single:
    *k_type = (int)KFLOAT;
    break;
  case arf_float_double:
    *k_type = (int)KDOUBLE;
    break;
  case arf_complex:
    *k_type = (int)KCOMPLEX;
    break;
  case arf_multi_band:
    /* KSTRUCT's not supported yet, so bail out */
    return ((int)FALSE);
  case arf_complex_double:
    *k_type = (int)KDCOMPLEX;
    break;
  default:
    kerror("data services","arf2khorosdatatype",
	   "invalid data type");
    return ((int)FALSE);
  }
  return ((int)TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: khoros2arf_datatype
  |
  |       Purpose: The routine converts the Khoros 2.0 datatype
  |		 passed in as input into a ARF
  |		 data format specification.
  |
  |         Input: k_type - Khoros data type.
  |
  |        Output: arf_type - Arf data type corresponding to 
  |		 the input argument
  |
  |       Returns: TRUE (1) on success, FALSE (0) on failure.
  |
  |    Written By: Tim Williams (modified from Jeremy Worley's code)
  |          Date: 13 May 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int khoros2arf_datatype(int k_type, ARF_IMAGE_TYPE *arf_type)
{
  switch (k_type) 
  {
  case KBYTE:
  case KUBYTE:
    *arf_type = (ARF_IMAGE_TYPE) arf_eight_bit;
    break;
  case KSHORT:
    *arf_type = (ARF_IMAGE_TYPE) arf_short_int;
    break;
  case KUSHORT:
    *arf_type = (ARF_IMAGE_TYPE) arf_short_word;
    break;
  case KINT:
  case KLONG:
    *arf_type = (ARF_IMAGE_TYPE) arf_long_int;
    break;
  case KULONG:
  case KUINT:
    *arf_type = (ARF_IMAGE_TYPE) arf_long_word;
    break;
  case KFLOAT:
    *arf_type = (ARF_IMAGE_TYPE) arf_float_single;
    break;
  case KDOUBLE:
    *arf_type = (ARF_IMAGE_TYPE) arf_float_double;
    break;
  case KCOMPLEX:
    *arf_type = (ARF_IMAGE_TYPE) arf_complex;
    break;
  case KDCOMPLEX:
    *arf_type = (ARF_IMAGE_TYPE) arf_complex_double;
    break;
  case KBIT: /* no corresponding ARF type */
  case KSTRUCT: /* not supported yet - needed for multi-band */
  default:
    return ((int)FALSE);
  }
  return ((int)TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_put_main_hdr
  |
  |       Purpose: writes the 32 byte ARF_MAIN_HDR structure containing the
  |                ARF main header to an ARF file
  |                
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |
  |       Returns: TRUE (1) on success, FALSE (0) on failure
  |    Written By: Tim Williams 
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int arf_put_main_hdr(kfile *file, ARF_MAIN_HDR *hdr)
{
  
  
  kfwrite_generic(file, &hdr->_magic_num, 1, KULONG);
  kfwrite_generic(file, &hdr->_version, 1, KULONG);
  kfwrite_generic(file, &hdr->_num_rows, 1, KULONG);
  kfwrite_generic(file, &hdr->_num_cols, 1, KULONG);
  kfwrite_generic(file, &hdr->_image_type, 1, KULONG);
  kfwrite_generic(file, &hdr->_num_frames, 1, KULONG);
  kfwrite_generic(file, &hdr->_image_offset, 1, KULONG);
  kfwrite_generic(file, &hdr->_subheader_flags, 1, KULONG);
  
  return ((int)TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_put_info
  |
  |       Purpose: writes ARF_INFO subheader to an ARF file
  |                
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                info - ARF_INFO structure
  |
  |	 Output: None
  |
  |       Returns: TRUE (1) on success, FALSE (0) on failure
  |    Written By: Tim Williams (modified John Horger's arf_put_sub_info)
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int arf_put_info(kfile *file, ARF_INFO *info)
{
  unsigned long tmpval;
  
  /******  Write the Values  ******/
  /********************************/
  
  kfwrite_generic(file, &info->image_source, 1, KULONG);
  kfwrite_generic(file, &info->start_x, 1, KULONG);
  kfwrite_generic(file, &info->start_y, 1, KULONG);
  kfwrite_generic(file, &info->num_avg, 1, KULONG);
  kfwrite_generic(file, &info->capture_rate, 1, KULONG);
  
  tmpval = info->capture_time.yy;
  kfwrite_generic(file, &tmpval, 1, KULONG);
  tmpval = info->capture_time.dd;
  kfwrite_generic(file, &tmpval, 1, KULONG);
  tmpval = info->capture_time.hh;
  kfwrite_generic(file, &tmpval, 1, KULONG);
  tmpval = info->capture_time.mm;
  kfwrite_generic(file, &tmpval, 1, KULONG);
  tmpval = info->capture_time.ss;
  kfwrite_generic(file, &tmpval, 1, KULONG);
  tmpval = info->capture_time.ms;
  kfwrite_generic(file, &tmpval, 1, KULONG);
  
  write_xdr_string(file, info->capture_loc);
  write_xdr_string(file, info->sensor_name);
  write_xdr_string(file, info->digitizer);
  
  kfwrite_generic(file, &info->sensor_hor_fov, 1, KFLOAT);
  kfwrite_generic(file, &info->sensor_ver_fov, 1, KFLOAT);
  
  kfwrite_generic(file, &info->samples_per_dwell, 1, KULONG);
  
  return ((int)TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_put_colormap
  |
  |       Purpose: writes ARF_COLORMAP subheader to an ARF file
  |                
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                colormap - ARF_COLORMAP structure 
  |
  |	 Output: None
  |
  |       Returns: TRUE (1) on success, FALSE (0) on failure
  |    Written By: Tim Williams (modified John Horger's arf_put_sub_colormap)
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int arf_put_colormap(kfile *file, ARF_COLORMAP *colormap)
{
  unsigned long tmparray[ARF_COLORMAP_SIZE];
  unsigned char *ptr;
  int i;
  
  for (i=0, ptr=colormap->red; i < ARF_COLORMAP_SIZE; i++, ptr++)
    tmparray[i] = (unsigned long)*ptr;
  kfwrite_generic(file, tmparray, ARF_COLORMAP_SIZE, KULONG);
  
  for (i=0, ptr=colormap->green; i < ARF_COLORMAP_SIZE; i++, ptr++)
    tmparray[i] = (unsigned long)*ptr;
  kfwrite_generic(file, tmparray, ARF_COLORMAP_SIZE, KULONG);
  
  for (i=0, ptr=colormap->blue; i < ARF_COLORMAP_SIZE; i++, ptr++)
    tmparray[i] = (unsigned long)*ptr;
  kfwrite_generic(file, tmparray, ARF_COLORMAP_SIZE, KULONG);

return (TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_put_comment
  |
  |       Purpose: writes ARF_COMMENT subheader to ARF file
  |                
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                comment - ARF_COMMENT structure
  |
  |	 Output: None
  |
  |       Returns: TRUE (1) on success, FALSE (0) on failure
  |    Written By: Tim Williams (modified John Horger's arf_put_sub_comment)
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int arf_put_comment(kfile  *file, ARF_COMMENT *comment)
{
  write_xdr_string(file, comment->bs);
  return (TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_put_multi_band
  |
  |       Purpose: writes ARF_MULTI_BAND subheader to ARF file
  |                
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                multi_band - ARF_MULTI_BAND structure
  |
  |	 Output: None
  |
  |       Returns: TRUE (1) on success, FALSE (0) on failure
  |    Written By: Tim Williams (modified John Horger's arf_put_sub_multi_band)
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int arf_put_multi_band(kfile *file, ARF_MULTI_BAND *multi_band)
{
  unsigned long tmpval;
  int i;
  
  /******  Get the general data for the multi-band  ******/
  /*******************************************************/
  
  write_xdr_string(file, multi_band->name);
  
  tmpval = (unsigned long)multi_band->num_bands;
  kfwrite_generic(file, &tmpval, 1, KULONG);
  
  /******  Deal with the bands  ******/
  /***********************************/
  for (i=0 ; i< (int)multi_band->num_bands ; i++)
  {
    write_xdr_string(file, multi_band->band[i].name);
    kfwrite_generic(file, &multi_band->band[i].type, 1, KULONG);
  }
  return ((int)TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_put_frame_data
  |
  |       Purpose: writes ARF_FRAME_DATA subheader to ARF file
  |                
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                frame_data - ARF_FRAME_DATA structure
  |
  |	 Output: None
  |
  |       Returns: TRUE (1) on success, FALSE (0) on failure
  |    Written By: Tim Williams (modified John Horger's arf_put_sub_frame_data)
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int arf_put_frame_data(kfile *file, ARF_FRAME_DATA *frame_data)
{
  kfwrite_generic(file, &frame_data->footer_flags, 1, KULONG);
  return ((int)TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_put_ladar_param
  |
  |       Purpose: writes ARF_LADAR_PARAM subheader to ARF file
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                ladar_param - ARF_LADAR_PARAM structure
  |
  |	 Output: None
  |
  |       Returns: TRUE (1) on success, FALSE (0) on failure
  |   Written By: Tim Williams (modified John Horger's arf_put_sub_ladar_param)
  |          Date: 13 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int arf_put_ladar_param(kfile *file, ARF_LADAR_PARAM *ladar_param)
{
  kfwrite_generic(file, &ladar_param->fm_scale, 1, KFLOAT);
  kfwrite_generic(file, &ladar_param->resolved_range_bias, 1, KFLOAT);
  kfwrite_generic(file, &ladar_param->hor_ifov, 1, KFLOAT);
  kfwrite_generic(file, &ladar_param->ver_ifov, 1, KFLOAT);
  kfwrite_generic(file, &ladar_param->coarse_res, 1, KFLOAT);
  kfwrite_generic(file, &ladar_param->fine_res, 1, KFLOAT);
  
  return ((int)TRUE);
}

/*-----------------------------------------------------------
  |
  |  Routine Name: arf_put_footer
  |
  |       Purpose: writes ARF footers to an ARF file
  |                
  |
  |         Input: file - kfile pointer to ARF file
  |                       kfile_setmachtype was already called to set the
  |                       architecture to KARCH_SUN since ARF uses XDR.
  |
  |                footer_flags - footer flags
  |
  |                frame_footer - ARF_FRAME_FOOTER structure
  |
  |	 Output: 
  |
  |       Returns: byte length of footer in file on success, 
  |                FALSE (0) on failure
  |
  |    Written By: Tim Williams 
  |          Date: 17 June 1994
  | Modifications:
  |
  ------------------------------------------------------------*/

int arf_put_footer (kfile *file, 
		    unsigned long footer_flags, 
		    ARF_FRAME_FOOTER *frame_footer)
{
  unsigned long tmpval;
  int footer_length;
  
  footer_length = 0;

  if (footer_flags & ARF_FRAME_INFO_FLAG)
  {
    kfwrite_generic(file, &(frame_footer->info.x), 1, KULONG);
    kfwrite_generic(file, &(frame_footer->info.y), 1, KULONG);
    kfwrite_generic(file, &(frame_footer->info.azimuth), 1, KFLOAT);
    kfwrite_generic(file, &(frame_footer->info.elevation), 1, KFLOAT);
    
    /* irig_time */
    tmpval = frame_footer->info.irig_time.yy;
    kfwrite_generic(file, &tmpval, 1, KULONG);
    tmpval = frame_footer->info.irig_time.dd;
    kfwrite_generic(file, &tmpval, 1, KULONG);
    tmpval = frame_footer->info.irig_time.hh;
    kfwrite_generic(file, &tmpval, 1, KULONG);
    tmpval = frame_footer->info.irig_time.mm;
    kfwrite_generic(file, &tmpval, 1, KULONG);
    tmpval = frame_footer->info.irig_time.ss;
    kfwrite_generic(file, &tmpval, 1, KULONG);
    tmpval = frame_footer->info.irig_time.ms;
    kfwrite_generic(file, &tmpval, 1, KULONG);
    
    footer_length += 40;
  }
  
  if (footer_flags & ARF_PLATFORM_ATTITUDE_FLAG)
  {
    kfwrite_generic(file, &(frame_footer->attitude.vel_north), 1, KDOUBLE);
    kfwrite_generic(file, &(frame_footer->attitude.vel_east), 1, KDOUBLE);
    kfwrite_generic(file, &(frame_footer->attitude.vel_vertical), 1, KDOUBLE);
    kfwrite_generic(file, &(frame_footer->attitude.acc_north), 1, KDOUBLE);
    kfwrite_generic(file, &(frame_footer->attitude.acc_east), 1, KDOUBLE);
    kfwrite_generic(file, &(frame_footer->attitude.acc_vertical), 1, KDOUBLE);
    kfwrite_generic(file, &(frame_footer->attitude.latitude), 1, KDOUBLE);
    kfwrite_generic(file, &(frame_footer->attitude.longitude), 1, KDOUBLE);
    kfwrite_generic(file, &(frame_footer->attitude.altitude), 1, KDOUBLE);
    kfwrite_generic(file, &(frame_footer->attitude.elevation), 1, KDOUBLE);
    kfwrite_generic(file, &(frame_footer->attitude.bank), 1, KDOUBLE);
    kfwrite_generic(file, &(frame_footer->attitude.heading), 1, KDOUBLE);
    
    footer_length += 96;
  }
  
  return (footer_length);
}

