 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Polymorphic Data Services
   >>>>
   >>>>   Static:
   >>>>			_kpds_lookup_primitive()
   >>>>			_nop_set()
   >>>>			_nop_copy()
   >>>>			_line_info_get()
   >>>>			_line_info_match()
   >>>>			_line_info_query()
   >>>>			_line_info_print()
   >>>>			_plane_info_get()
   >>>>			_plane_info_match()
   >>>>			_plane_info_query()
   >>>>			_plane_info_print()
   >>>>			_volume_info_get()
   >>>>			_volume_info_match()
   >>>>			_volume_info_query()
   >>>>			_volume_info_print()
   >>>>			_vector_info_get()
   >>>>			_vector_info_match()
   >>>>			_vector_info_query()
   >>>>			_vector_info_print()
   >>>>			_region_info_get()
   >>>>			_region_info_match()
   >>>>			_region_info_query()
   >>>>			_region_info_print()
   >>>>			_optimal_region_size_get()
   >>>>			_optimal_region_size_match()
   >>>>			_optimal_region_size_query()
   >>>>			_optimal_region_size_print()
   >>>>			_kpds_region_corners()
   >>>>
   >>>>  Private:
   >>>>         	kpds_init()
   >>>>			kpds_get_histogram()
   >>>>			kpds_get_region()
   >>>>			kpds_put_region()
   >>>>
   >>>>   Public:
   >>>>			kpds_get_attribute()
   >>>>			kpds_get_attributes()
   >>>>			kpds_set_attribute()
   >>>>			kpds_set_attributes()
   >>>>			kpds_match_attribute()
   >>>>			kpds_match_attributes()
   >>>>			kpds_copy_attribute()
   >>>>			kpds_copy_attributes()
   >>>>			kpds_query_attribute()
   >>>>			kpds_print_attribute()
   >>>>			kpds_get_data()
   >>>>			kpds_put_data()
   >>>>			kpds_open_object()
   >>>>			kpds_reference_object()
   >>>>			kpds_copy_object_attr()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

kaddr kpds_get_histogram PROTO((kobject, kaps_primitive *, kaddr));
kaddr kpds_get_region PROTO((kobject, kaps_primitive *, kaddr));
int kpds_put_region PROTO((kobject, kaps_primitive *, kaddr));

int pds_initialized = FALSE;

extern char *_pds_seg_mask;
extern char *_pds_seg_value;
extern char *_pds_seg_location;
extern char *_pds_seg_map;
extern char *_pds_seg_time;

extern int value_token;
extern int mask_token;
extern int location_token;
extern int time_token;
extern int map_token;
extern int width_token;
extern int height_token;
extern int depth_token;

/*
 * I hate this.  These are bogus initialization arrays for the internal
 * versions of the optimal region constraint and flag stuff.
 */
static int zeros[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
static int ones[KDMS_MAX_DIM] = {1, 1, 1, 1, 1};

kaps_primitive kpds_primitives[] =
{

   {KPDS_VALUE_POINT, 0, 0, KDMS_SEGMENT_VALUE,
    {0, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_VALUE_VECTOR, 0, 0, KDMS_SEGMENT_VALUE,
    {0, 0, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_VALUE_LINE, 0, 0, KDMS_SEGMENT_VALUE,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_VALUE_PLANE, 0, 0, KDMS_SEGMENT_VALUE,
    {1, 1, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_VALUE_VOLUME, 0, 0, KDMS_SEGMENT_VALUE,
    {1, 1, 1, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_VALUE_ALL, 0, 0, KDMS_SEGMENT_VALUE,
    {1, 1, 1, 1, 1},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_VALUE_REGION, 0, 0, KDMS_SEGMENT_VALUE,
    {0, 0, 0, 0, 0},
    kpds_get_region, kpds_put_region},
   {KPDS_VALUE_HISTOGRAM, 0, 0, KDMS_SEGMENT_VALUE,
    {0, 0, 0, 0, 0},
    kpds_get_histogram, NULL},


   {KPDS_MASK_POINT, 0, 0, KDMS_SEGMENT_MASK,
    {0, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_MASK_VECTOR, 0, 0, KDMS_SEGMENT_MASK,
    {0, 0, 0, 0, 1},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_MASK_LINE, 0, 0, KDMS_SEGMENT_MASK,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_MASK_PLANE, 0, 0, KDMS_SEGMENT_MASK,
    {1, 1, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_MASK_VOLUME, 0, 0, KDMS_SEGMENT_MASK,
    {1, 1, 1, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_MASK_ALL, 0, 0, KDMS_SEGMENT_MASK,
    {1, 1, 1, 1, 1},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_MASK_REGION, 0, 0, KDMS_SEGMENT_MASK,
    {0, 0, 0, 0, 0},
    kpds_get_region, kpds_put_region},


   {KPDS_LOCATION_POINT, 0, 0, KDMS_SEGMENT_LOCATION,
    {0, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_LOCATION_VECTOR, 0, 0, KDMS_SEGMENT_LOCATION,
    {0, 0, 0, 1, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_LOCATION_LINE, 0, 0, KDMS_SEGMENT_LOCATION,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_LOCATION_PLANE, 0, 0, KDMS_SEGMENT_LOCATION,
    {1, 1, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_LOCATION_VOLUME, 0, 0, KDMS_SEGMENT_LOCATION,
    {1, 1, 1, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_LOCATION_ALL, 0, 0, KDMS_SEGMENT_LOCATION,
    {1, 1, 1, 1, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_LOCATION_REGION, 0, 0, KDMS_SEGMENT_LOCATION,
    {0, 0, 0, 0, 0},
    kpds_get_region, kpds_put_region},

   {KPDS_LOCATION_WIDTH_POINT, 0, 0, KDMS_SEGMENT_WIDTH,
    {0, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_LOCATION_WIDTH_ALL, 0, 0, KDMS_SEGMENT_WIDTH,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_LOCATION_WIDTH_REGION, 0, 0, KDMS_SEGMENT_WIDTH,
    {0, 0, 0, 0, 0},
    kpds_get_region, kpds_put_region},

   {KPDS_LOCATION_HEIGHT_POINT, 0, 0, KDMS_SEGMENT_HEIGHT,
    {0, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_LOCATION_HEIGHT_ALL, 0, 0, KDMS_SEGMENT_HEIGHT,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_LOCATION_HEIGHT_REGION, 0, 0, KDMS_SEGMENT_HEIGHT,
    {0, 0, 0, 0, 0},
    kpds_get_region, kpds_put_region},

   {KPDS_LOCATION_DEPTH_POINT, 0, 0, KDMS_SEGMENT_DEPTH,
    {0, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_LOCATION_DEPTH_ALL, 0, 0, KDMS_SEGMENT_DEPTH,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_LOCATION_DEPTH_REGION, 0, 0, KDMS_SEGMENT_DEPTH,
    {0, 0, 0, 0, 0},
    kpds_get_region, kpds_put_region},


   {KPDS_TIME_POINT, 0, 0, KDMS_SEGMENT_TIME,
    {0, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_TIME_ALL, 0, 0, KDMS_SEGMENT_TIME,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_TIME_REGION, 0, 0, KDMS_SEGMENT_TIME,
    {0, 0, 0, 0, 0},
    kpds_get_region, kpds_put_region},


   {KPDS_MAP_POINT, 0, 0, KDMS_SEGMENT_MAP,
    {0, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_MAP_LINE, 0, 0, KDMS_SEGMENT_MAP,
    {1, 0, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_MAP_PLANE, 0, 0, KDMS_SEGMENT_MAP,
    {1, 1, 0, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_MAP_VOLUME, 0, 0, KDMS_SEGMENT_MAP,
    {1, 1, 1, 0, 0},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_MAP_ALL, 0, 0, KDMS_SEGMENT_MAP,
    {1, 1, 1, 1, 1},
    kaps_get_primitive, kaps_put_primitive},
   {KPDS_MAP_REGION, 0, 0, KDMS_SEGMENT_MAP,
    {0, 0, 0, 0, 0},
    kpds_get_region, kpds_put_region}
};

int kpds_num_primitives = sizeof(kpds_primitives) / sizeof(kpds_primitives[0]);

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kpds_lookup_primitive
|
|       Purpose: This routine returns the index into
|                the primitive table for a given primitive
|                name.  If the primitive is not found, 
|                then an errno is set.
|
|         Input: prim - the name of the primitive to look up
|
|        Output: none
|
|       Returns: the index into the primitive table on success, 
|		 -1 otherwise.
|
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 30, 1993 14:09
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_kpds_lookup_primitive(char *prim)
{
   int i;
   int token = kstring_to_token(prim);

   for (i = 0; i < kpds_num_primitives; i++)
      if (kpds_primitives[i].token == token)
	 return i;

   /* errno = INVALID PRIMITIVE  // SK */

   return -1;
}

/*
 *    ================================================================== 
 *    KPDS Quasi Attribute Methods
 *    ==================================================================
 */

/*-----------------------------------------------------------
|  Routine Name: (static) _nop_set
|       Purpose: This routine is used in lieu of a set function when
|		 the function is considered to be a read only attribute.
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_nop_set(kobject obj, int assoc, int attr, kaddr clientData, kva_list * list)
{
   return FALSE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _nop_copy
|       Purpose: This routine is used in lieu of a copy function when
|		 the function is considered to be a read only attribute.
|    Written By: Steve Kubica
|          Date: Apr 04, 1994 10:49
------------------------------------------------------------*/
/* ARGSUSED */
static int
_nop_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	  kaddr clientData1, kaddr clientData2)
{
   return FALSE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_LINE_INFO
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _line_info_get
|       Purpose: This routine retrieves the "line information".
|                This information consists of the size in pixels of data
|                points of a line, and the total number of lines.
|    Written By: Jeremy Worley
|          Date: Apr 01, 1994 19:44
------------------------------------------------------------*/
/* ARGSUSED */
static int
_line_info_get(kobject obj, int assoc, int attr, kaddr clientData,
	       kva_list * list)
{
   int *size = NULL;
   int *tmp = NULL;
   int  dim;
   int  i;


   /* get the model-based size of this segment */
   if ((size = kaps_get_segment_size(obj, (char *) clientData)) == NULL)
      return FALSE;

   /* get the line length */
   if ((tmp = kva_arg(*list, int *)) != NULL)
         (*tmp) = size[0];	/* a line is always along width */

   /* get the dimensionality of the segment */
   kdms_get_attribute(obj, (char *) clientData, KDMS_DIMENSION, &dim);

   /* compute the total number of lines */
   if ((tmp = kva_arg(*list, int *)) != NULL)
      for (i = 1, (*tmp) = 1; i < dim; i++)
	 (*tmp) *= size[i];

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _line_info_match
|       Purpose: match the line info attribute
|    Written By: Steve Kubica
|          Date: May 02, 1994 16:31
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_line_info_match(kobject obj1, kobject obj2, int assoc, int attrib,
		 kaddr clientData1, kaddr clientData2)
{
   int *size1 = NULL;
   int *size2 = NULL;
   int  tot1 = 1;
   int  tot2 = 1;
   int  dim;
   int  i;


   /* get the model-based size of the first segment */
   if ((size1 = kaps_get_segment_size(obj1, (char *) clientData1)) == NULL)
      return FALSE;

   /* get the model-based size of the second segment */
   if ((size2 = kaps_get_segment_size(obj2, (char *) clientData2)) == NULL)
      return FALSE;

   /* line length is always along width */
   if (size1[0] != size2[0])
      return FALSE;

   /* get the dimensionality of the segment */
   kdms_get_attribute(obj1, (char *) clientData1, KDMS_DIMENSION, &dim);

   /* compute the total number of lines */
   for (i = 1, tot1 = 1, tot2 = 1; i < dim; i++)
   {
      tot1 *= size1[i];
      tot2 *= size2[i];
   }

   return (tot1 == tot2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _line_info_query
|       Purpose: query the line info attribute
|    Written By: 
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_line_info_query(kobject obj, int assoc, int attrib, kaddr clientData,
		 int *num_args, int *arg_size, int *data_type,
		 int *permanent)
{
   /* if the segment does not exist, then it has no line info */
   if (!kdms_query_segment(obj, (char *) clientData))
      return FALSE;		/* not an error */

   if (num_args)
      *num_args = 2;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _line_info_print
|       Purpose: print the line info attribute
|    Written By: Steve Kubica
|          Date: May 02, 1994 16:32
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_line_info_print(kobject obj, int assoc, int attrib, kaddr clientData,
		 kfile *outfile)
{
   int *size = NULL;
   int  tot = 1;
   int  dim;
   int i;


   /* get the model-based size of the first segment */
   if ((size = kaps_get_segment_size(obj, (char *) clientData)) == NULL)
      return FALSE;

   /* get the dimensionality of the segment */
   kdms_get_attribute(obj, (char *) clientData, KDMS_DIMENSION, &dim);

   /* compute the total number of lines */
   for (i = 1, tot = 1; i < dim; i++)
      tot *= size[i];

   if (outfile)
      kfprintf(outfile, "%d\t%d", size[0], tot);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_PLANE_INFO
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _plane_info_get
|       Purpose: This routine retrieves the "plane information".
|                This information consists of the size in pixels of data
|                points of a plane, and the total number of planes.
|    Written By: Jeremy Worley
|          Date: Apr 01, 1994 20:04
------------------------------------------------------------*/
/* ARGSUSED */
static int
_plane_info_get(kobject obj, int assoc, int attr, kaddr clientData,
		kva_list *list)
{
   int *size = NULL;
   int *tmp = NULL;
   int  dim;
   int  i;


   /* get the model-based size of this segment */
   if ((size = kaps_get_segment_size(obj, (char *) clientData)) == NULL)
      return FALSE;

   /* get the plane size */
   if ((tmp = kva_arg(*list, int *)) != NULL)
         (*tmp) = size[0];	/* a plane is always along width ...  */

   if ((tmp = kva_arg(*list, int *)) != NULL)
         (*tmp) = size[1];	/* ... and height */

   /* get the dimensionality of the segment */
   kdms_get_attribute(obj, (char *) clientData, KDMS_DIMENSION, &dim);

   /* compute the total number of planes */
   if ((tmp = kva_arg(*list, int *)) != NULL)
      for (i = 2, (*tmp) = 1; i < dim; i++)
	 (*tmp) *= size[i];

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _plane_info_match
|       Purpose: match the plane info attribute
|    Written By: Steve Kubica
|          Date: May 02, 1994 16:32
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_plane_info_match(kobject obj1, kobject obj2, int assoc, int attrib,
		  kaddr clientData1, kaddr clientData2)
{
   int *size1 = NULL;
   int *size2 = NULL;
   int  tot1 = 1;
   int  tot2 = 1;
   int  dim;
   int  i;


   /* get the model-based size of the first segment */
   if ((size1 = kaps_get_segment_size(obj1, (char *) clientData1)) == NULL)
      return FALSE;

   /* get the model-based size of the first segment */
   if ((size2 = kaps_get_segment_size(obj1, (char *) clientData1)) == NULL)
      return FALSE;

   /* plane size is always along width and height */
   if ((size1[0] != size2[0]) || (size1[1] != size2[1]))
      return FALSE;

   /* get the dimensionality of the segment */
   kdms_get_attribute(obj1, (char *) clientData1, KDMS_DIMENSION, &dim);

   /* compute the total number of lines */
   for (i = 2, tot1 = 1, tot2 = 1; i < dim; i++)
   {
      tot1 *= size1[i];
      tot2 *= size2[i];
   }

   return (tot1 == tot2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _plane_info_query
|       Purpose: query the plane info attribute
|    Written By: Steve Kubica
|          Date: May 02, 1994 16:32
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_plane_info_query(kobject obj, int assoc, int attrib, kaddr clientData,
		  int *num_args, int *arg_size, int *data_type,
		  int *permanent)
{
   /* if the segment does not exist, then it has no plane info */
   if (!kdms_query_segment(obj, (char *) clientData))
      return FALSE;		/* not an error */

   if (num_args)
      *num_args = 3;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _plane_info_print
|       Purpose: print the plane info attribute
|    Written By: Steve Kubica
|          Date: May 02, 1994 16:32
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_plane_info_print(kobject obj, int assoc, int attrib, kaddr clientData,
		  kfile *outfile)
{
   int *size = NULL;
   int  tot = 1;
   int  dim;
   int  i;


   /* get the model-based size of the first segment */
   if ((size = kaps_get_segment_size(obj, (char *) clientData)) == NULL)
      return FALSE;

   /* get the dimensionality of the segment */
   kdms_get_attribute(obj, (char *) clientData, KDMS_DIMENSION, &dim);

   /* compute the total number of planes */
   for (i = 2, tot = 1; i < dim; i++)
      tot *= size[i];

   if (outfile)
      kfprintf(outfile, "%d x %d\t%d",
	       size[0], size[1], tot);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_VOLUME_INFO
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _volume_info_get
|       Purpose: This routine retrieves the "volume information".
|                This information consists of the size in pixels
|                or data points of a volume, and the total number
|                of volumes.
|    Written By: Jeremy Worley
|          Date: Apr 01, 1994 20:06
------------------------------------------------------------*/
/* ARGSUSED */
static int
_volume_info_get(kobject obj, int assoc, int attr, kaddr clientData,
		 kva_list *list)
{
   int *size = NULL;
   int *tmp = NULL;
   int  dim;
   int  i;


   /* get the model-based size of this segment */
   if ((size = kaps_get_segment_size(obj, (char *) clientData)) == NULL)
      return FALSE;

   /* get the volume size */
   if ((tmp = kva_arg(*list, int *)) != NULL)
         (*tmp) = size[0];	/* a volume is always along width ...  */

   if ((tmp = kva_arg(*list, int *)) != NULL)
         (*tmp) = size[1];	/* ... and height ... */

   if ((tmp = kva_arg(*list, int *)) != NULL)
         (*tmp) = size[2];	/* ... and depth */

   /* get the dimensionality of the segment */
   kdms_get_attribute(obj, (char *) clientData, KDMS_DIMENSION, &dim);

   /* compute the total number of planes */
   if ((tmp = kva_arg(*list, int *)) != NULL)
      for (i = 3, (*tmp) = 1; i < dim; i++)
	 (*tmp) *= size[i];

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _volume_info_match
|       Purpose: match the volume info attribute
|    Written By: 
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_volume_info_match(kobject obj1, kobject obj2, int assoc, int attrib,
		   kaddr clientData1, kaddr clientData2)
{
   int *size1 = NULL;
   int *size2 = NULL;
   int  tot1 = 1;
   int  tot2 = 1;
   int  dim;
   int  i;


   /* get the model-based size of the first segment */
   if ((size1 = kaps_get_segment_size(obj1, (char *) clientData1)) == NULL)
      return FALSE;

   /* get the model-based size of the first segment */
   if ((size2 = kaps_get_segment_size(obj1, (char *) clientData1)) == NULL)
      return FALSE;

   /* volume size is always along width, height, and depth */
   if ((size1[0] != size2[0]) || 
       (size1[1] != size2[1]) || 
       (size1[2] != size2[2]))
      return FALSE;

   /* get the dimensionality of the segment */
   kdms_get_attribute(obj1, (char *) clientData1, KDMS_DIMENSION, &dim);

   /* compute the total number of lines */
   for (i = 3, tot1 = 1, tot2 = 1; i < dim; i++)
   {
      tot1 *= size1[i];
      tot2 *= size2[i];
   }

   return (tot1 == tot2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _volume_info_query
|       Purpose: query the volume info attribute
|    Written By: 
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_volume_info_query(kobject obj, int assoc, int attrib, kaddr clientData,
		   int *num_args, int *arg_size, int *data_type,
		   int *permanent)
{
   /* if the segment does not exist, then it has no volume info */
   if (!kdms_query_segment(obj, (char *) clientData))
      return FALSE;		/* not an error */

   if (num_args)
      *num_args = 4;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _volume_info_print
|       Purpose: print the volume info attribute
|    Written By: 
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_volume_info_print(kobject obj, int assoc, int attrib, kaddr clientData,
		   kfile *outfile)
{
   int *size = NULL;
   int  tot = 1;
   int  dim;
   int  i;


   /* get the model-based size of the first segment */
   if ((size = kaps_get_segment_size(obj, (char *) clientData)) == NULL)
      return FALSE;

   /* get the dimensionality of the segment */
   kdms_get_attribute(obj, (char *) clientData, KDMS_DIMENSION, &dim);

   /* compute the total number of volumes */
   for (i = 3, tot = 1; i < dim; i++)
      tot *= size[i];

   if (outfile)
      kfprintf(outfile, "%d x %d x %d\t%d",
	       size[0], size[1], size[2], tot);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_VECTOR_INFO
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _vector_info_get
|       Purpose: This routine retrieves the "vector information".
|                This information consists of the size in pixels
|                or data points of a vector, and the total number
|                of vectors.
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_vector_info_get(kobject obj, int assoc, int attr, kaddr clientData,
		 kva_list *list)
{
   int *size = NULL;
   int *tmp = NULL;
   int  i;
   

   /* get the model-based size of this segment */
   if ((size = kaps_get_segment_size(obj, (char *) clientData)) == NULL)
      return FALSE;

   /* the vector primitive only exists for the value, mask, and location */
   if ((kstrcmp((char *)clientData, KDMS_SEGMENT_VALUE) == 0) ||
       (kstrcmp((char *)clientData, KDMS_SEGMENT_MASK) == 0))
   {

      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = size[KELM];	/* a vector is along elements */

      /* compute the total number of vectors */
      if ((tmp = kva_arg(*list, int *)) != NULL)
	 for (i = 0, (*tmp) = 1; i < KELM; i++)
	    (*tmp) *= size[i];

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_LOCATION) == 0)
   {

      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = size[KDIM];	/* a vector is along dimension */

      /* compute the total number of vectors */
      if ((tmp = kva_arg(*list, int *)) != NULL)
	 for (i = 0, (*tmp) = 1; i < KDIM; i++)
	    (*tmp) *= size[i];
   }
   else
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _vector_info_match
|       Purpose: match the vector info attribute
|    Written By: 
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_vector_info_match(kobject obj1, kobject obj2, int assoc, int attrib,
		   kaddr clientData1, kaddr clientData2)
{
   int *size1 = NULL;
   int *size2 = NULL;
   int  tot1 = 1;
   int  tot2 = 1;
   int  i;


   /* get the model-based size of the first segment */
   if ((size1 = kaps_get_segment_size(obj1, (char *) clientData1)) == NULL)
      return FALSE;

   /* get the model-based size of the second segment */
   if ((size2 = kaps_get_segment_size(obj2, (char *) clientData2)) == NULL)
      return FALSE;

   /* the vector primitive only exists for the value, mask, and location */
   if ((kstrcmp((char *)clientData1, KDMS_SEGMENT_VALUE) == 0) ||
       (kstrcmp((char *)clientData1, KDMS_SEGMENT_MASK) == 0))
   {

      if (size1[KELM] != size2[KELM])
	 return FALSE;

      /* compute the total number of vectors */
      for (i = 0, tot1 = 1, tot2 = 1; i < KELM; i++)
      {
	 tot1 *= size1[i];
	 tot2 *= size2[i];
      }

   }
   else if (kstrcmp((char *)clientData1, KDMS_SEGMENT_LOCATION) == 0)
   {

      if (size1[KDIM] != size2[KDIM])
	 return FALSE;

      /* compute the total number of vectors */
      for (i = 0, tot1 = 1, tot2 = 1; i < KDIM; i++)
      {
	 tot1 *= size1[i];
	 tot2 *= size2[i];
      }

   }
   else
      return FALSE;

   return (tot1 == tot2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _vector_info_query
|       Purpose: query the vector info attribute
|    Written By: 
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_vector_info_query(kobject obj, int assoc, int attrib, kaddr clientData,
		   int *num_args, int *arg_size, int *data_type,
		   int *permanent)
{
   /* if the segment does not exist, then it has no vector info */
   if (!kdms_query_segment(obj, (char *) clientData))
      return FALSE;		/* not an error */

   /* the vector primitive only exists for the value, mask, and location */
   if ((kstrcmp((char *)clientData, KDMS_SEGMENT_VALUE) != 0) &&
       (kstrcmp((char *)clientData, KDMS_SEGMENT_MASK) != 0) &&
       (kstrcmp((char *)clientData, KDMS_SEGMENT_LOCATION) != 0))
      return FALSE;		/* not really an error either */

   if (num_args)
      *num_args = 2;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _vector_info_print
|       Purpose: print the vector info attribute
|    Written By: 
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_vector_info_print(kobject obj, int assoc, int attrib, kaddr clientData,
		   kfile *outfile)
{
   int *size = NULL;
   int  tot = 1;
   int  i;


   /* get the model-based size of this segment */
   if ((size = kaps_get_segment_size(obj, (char *) clientData)) == NULL)
      return FALSE;

   /* the vector primitive only exists for the value, mask, and location */
   if ((kstrcmp((char *)clientData, KDMS_SEGMENT_VALUE) == 0) ||
       (kstrcmp((char *)clientData, KDMS_SEGMENT_MASK) == 0))
   {

      /* compute the total number of vectors */
      for (i = 0, tot = 1; i < KELM; i++)
	 tot *= size[i];

      if (outfile)
	 kfprintf(outfile, "%d\t%d",
		  size[KELM], tot);


   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_LOCATION) == 0)
   {

      /* compute the total number of vectors */
      for (i = 0, tot = 1; i < KDIM; i++)
	 tot *= size[i];

      if (outfile)
	 kfprintf(outfile, "Vector Size = %d \tNumber of Vectors = %d",
		  size[KDIM], tot);

   }
   else
      return FALSE;

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_REGION_INFO
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _region_info_get
|       Purpose: This routine retrieves the "region information".
|                This information consists of the size in pixels
|                or data points of a region, and the total number
|                of vectors.
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_region_info_get(kobject obj, int assoc, int attr, kaddr clientData,
		 kva_list *list)
{
   int *size = NULL;
   int *tmp = NULL;
   int  tot;
   int  rw;
   int  rh;
   int  rd;
   int  rt;
   int  re;


   /* get the model-based size of this segment */
   if ((size = kaps_get_segment_size(obj, (char *) clientData)) == NULL)
      return FALSE;

   /* the region primitive varies for every segment */
   if (kstrcmp((char *)clientData, KDMS_SEGMENT_VALUE) == 0)
   {

      kpds_get_attribute(obj, KPDS_VALUE_REGION_SIZE, &rw, &rh, &rd, &rt, &re);

      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rw;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rh;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rd;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rt;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = re;

      /* calculate and return the number of regions */
      tot = (int)(kceil((float)size[0]/(float)rw) * 
		  kceil((float)size[1]/(float)rh) *
		  kceil((float)size[2]/(float)rd) *
		  kceil((float)size[3]/(float)rt) *
		  kceil((float)size[4]/(float)re));
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = tot;

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MASK) == 0)
   {

      kpds_get_attribute(obj, KPDS_MASK_REGION_SIZE, &rw, &rh, &rd, &rt, &re);

      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rw;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rh;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rd;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rt;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = re;

      /* calculate and return the number of regions */
      tot = (int)(kceil((float)size[0]/(float)rw) * 
		  kceil((float)size[1]/(float)rh) *
		  kceil((float)size[2]/(float)rd) *
		  kceil((float)size[3]/(float)rt) *
		  kceil((float)size[4]/(float)re));
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = tot;

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_LOCATION) == 0)
   {

      kpds_get_attribute(obj, KPDS_LOCATION_REGION_SIZE, &rw, &rh, &rd, &re);

      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rw;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rh;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rd;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = re;

      /* calculate and return the number of regions */
      tot = (int)(kceil((float)size[0]/(float)rw) * 
		  kceil((float)size[1]/(float)rh) *
		  kceil((float)size[2]/(float)rd) *
		  kceil((float)size[3]/(float)re));
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = tot;

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_TIME) == 0)
   {

      kpds_get_attribute(obj, KPDS_TIME_REGION_SIZE, &rt);

      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rt;

      /* calculate and return the number of regions */
      tot = kceil((float)size[0] / (float)rt);
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = tot;

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MAP) == 0)
   {

      kpds_get_attribute(obj, KPDS_MAP_REGION_SIZE, &rw, &rh, &rd, &rt, &re);

      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rw;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rh;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rd;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = rt;
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = re;

      /* calculate and return the number of regions */
      tot = (int)(kceil((float)size[0]/(float)rw) * 
		  kceil((float)size[1]/(float)rh) *
		  kceil((float)size[2]/(float)rd) *
		  kceil((float)size[3]/(float)rt) *
		  kceil((float)size[4]/(float)re));
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = tot;

   }
   else
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _region_info_match
|       Purpose: match the region info attribute
|    Written By: 
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_region_info_match(kobject obj1, kobject obj2, int assoc, int attrib,
		   kaddr clientData1, kaddr clientData2)
{
   int *size1 = NULL;
   int *size2 = NULL;
   int  dim;
   int  status;
   int  i;


   /* get the model-based size of the first segment */
   if ((size1 = kaps_get_segment_size(obj1, (char *) clientData1)) == NULL)
      return FALSE;

   /* get the model-based size of the first segment */
   if ((size2 = kaps_get_segment_size(obj2, (char *) clientData2)) == NULL)
      return FALSE;

   /* the region primitive varies for every segment */
   if (kstrcmp((char *)clientData1, KDMS_SEGMENT_VALUE) == 0)
      status = kpds_match_attribute(obj1, obj2, KPDS_VALUE_REGION_SIZE);

   else if (kstrcmp((char *)clientData1, KDMS_SEGMENT_MASK) == 0)
      status = kpds_match_attribute(obj1, obj2, KPDS_MASK_REGION_SIZE);

   else if (kstrcmp((char *)clientData1, KDMS_SEGMENT_LOCATION) == 0)
      status = kpds_match_attribute(obj1, obj2, KPDS_LOCATION_REGION_SIZE);

   else if (kstrcmp((char *)clientData1, KDMS_SEGMENT_TIME) == 0)
      status = kpds_match_attribute(obj1, obj2, KPDS_TIME_REGION_SIZE);

   else if (kstrcmp((char *)clientData1, KDMS_SEGMENT_MAP) == 0)
      status = kpds_match_attribute(obj1, obj2, KPDS_MAP_REGION_SIZE);

   else
      return FALSE;

   /* get the dimensionality of the segment */
   kdms_get_attribute(obj1, (char *) clientData1, KDMS_DIMENSION, &dim);

   if (status)	/* regions are same size - so we're cool if the sizes match */
      for (i = 0; i < dim; i++)
	 status &= (size1[i] == size2[i]);

   return status;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _region_info_query
|       Purpose: query the region info attribute
|    Written By: 
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_region_info_query(kobject obj, int assoc, int attrib, kaddr clientData,
		   int *num_args, int *arg_size, int *data_type,
		   int *permanent)
{
   /* if the segment does not exist, then it has no region info */
   if (!kdms_query_segment(obj, (char *) clientData))
      return FALSE;		/* not an error */

   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = FALSE;

   /* the num_args of the region info primitive varies for every segment */
   if (kstrcmp((char *)clientData, KDMS_SEGMENT_VALUE) == 0)
   {
      if (num_args)
	 *num_args = 6;
   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MASK) == 0)
   {
      if (num_args)
	 *num_args = 6;
   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_LOCATION) == 0)
   {
      if (num_args)
	 *num_args = 5;
   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_TIME) == 0)
   {
      if (num_args)
	 *num_args = 2;
   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MAP) == 0)
   {
      if (num_args)
	 *num_args = 6;
   }
   else
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _region_info_print
|       Purpose: print the region info attribute
|    Written By: 
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_region_info_print(kobject obj, int assoc, int attrib, kaddr clientData,
		   kfile *outfile)
{
   int rw;
   int rh;
   int rd;
   int rt;
   int re;
   int num;


   /* if the segment does not exist, then it has no region info */
   if (!kdms_query_segment(obj, (char *) clientData))
      return FALSE;		/* not an error */

   /* the region primitive varies for every segment */
   if (kstrcmp((char *)clientData, KDMS_SEGMENT_VALUE) == 0)
   {

      kpds_get_attribute(obj, KPDS_VALUE_REGION_INFO, 
			 &rw, &rh, &rd, &rt, &re, &num);
      if (outfile)
	 kfprintf(outfile, "%d x %d x %d x %d x %d\t"
		  "%d", rw, rh, rd, rt, re, num);

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MASK) == 0)
   {

      kpds_get_attribute(obj, KPDS_MASK_REGION_INFO, 
			 &rw, &rh, &rd, &rt, &re, &num);
      if (outfile)
	 kfprintf(outfile, "%d x %d x %d x %d x %d\t"
		  "%d", rw, rh, rd, rt, re, num);

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_LOCATION) == 0)
   {

      kpds_get_attribute(obj, KPDS_LOCATION_REGION_INFO,
			 &rw, &rh, &re, &re, &num);
      if (outfile)
	 kfprintf(outfile, "%d x %d x %d x %d\t"
		  "%d", rw, rh, rd, rt, re, num);

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_TIME) == 0)
   {

      kpds_get_attribute(obj, KPDS_TIME_REGION_INFO, &rt, &num);
      if (outfile)
	 kfprintf(outfile, "%d\t"
		  "%d", rt, num);

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MAP) == 0)
   {

      kpds_get_attribute(obj, KPDS_MAP_REGION_INFO, 
			 &rw, &rh, &re, &rd, &rt, &num);
      if (outfile)
	 kfprintf(outfile, "%d x %d x %d x %d x %d\t"
		  "%d", rw, rh, re, rt, rd, num);
   }
   else
      return FALSE;

   return TRUE;
}


/*********** ---------------------------------------------------------------
 ***********  KPDS_OPTIMAL_REGION_CONSTRAINT
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_region_constraint_get
|       Purpose: This routine retrieves the "region constraint information".
|    Written By: Jeremy Worley
|          Date: Thu Dec  8 14:37:51 1994
------------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_constraint_get(kobject obj, int assoc, int attr, kaddr clientData,
		 kva_list *list)
{
   int *constraint = NULL;
   int dim;
   int i;
   int *tmp;
   
   kdms_get_attributes(obj, (char *)clientData, 
		       _INTERNAL_REGION_CONSTRAINT, &constraint,
		       KDMS_DIMENSION, &dim, NULL);
   
   for (i = 0; i < dim; i++)
      if ((tmp = kva_arg(*list, int *)) != NULL)
	 (*tmp) = constraint[i];

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_region_constraint_set
|       Purpose: This routine stores the "region constraint information".
|    Written By: Jeremy Worley
|          Date: Thu Dec  8 14:37:51 1994
------------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_constraint_set(kobject obj, int assoc, int attr, kaddr clientData,
		 kva_list *list)
{
   int *constraint = NULL;
   int dim;
   int i;
   
   if (!kdms_get_attributes(obj, (char *)clientData, 
		       _INTERNAL_REGION_CONSTRAINT, &constraint,
		       KDMS_DIMENSION, &dim, NULL))
      return FALSE;
   
   for (i = 0; i < dim; i++)
      constraint[i] = kva_arg(*list, int);
   
   if (!kdms_set_attribute(obj, (char *)clientData, 
		       _INTERNAL_REGION_CONSTRAINT, constraint))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_region_constraint_match
|       Purpose: match the region info attribute
|    Written By: Jeremy Worley 
|          Date: Thu Dec  8 14:37:51 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_constraint_match(kobject obj1, kobject obj2, int assoc, int attrib,
		   kaddr clientData1, kaddr clientData2)
{
   int *constraint1 = NULL;
   int dim1;
   int *constraint2 = NULL;
   int dim2;
   int i;
   
   kdms_get_attributes(obj1, (char *)clientData1, 
		       _INTERNAL_REGION_CONSTRAINT, &constraint1,
		       KDMS_DIMENSION, &dim1, NULL);

   kdms_get_attributes(obj2, (char *)clientData2, 
		       _INTERNAL_REGION_CONSTRAINT, &constraint2,
		       KDMS_DIMENSION, &dim2, NULL);
 
   if (dim1 != dim2)
      return (FALSE);
   
   for (i = 0; i < dim1; i++)
      if (constraint1[i] != constraint2[i])
	 return(FALSE);
   
   return(TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_region_constraint_query
|       Purpose: query the region info attribute
|    Written By: Jeremy Worley 
|          Date: Thu Dec  8 14:37:51 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_constraint_query(kobject obj, int assoc, int attrib, kaddr clientData,
		   int *num_args, int *arg_size, int *data_type,
		   int *permanent)
{
   /* if the segment does not exist, then it has no region info */
   if (!kdms_query_segment(obj, (char *) clientData))
      return FALSE;		/* not an error */

   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = FALSE;

   /* the num_args of the region info primitive varies for every segment */
   if (kstrcmp((char *)clientData, KDMS_SEGMENT_VALUE) == 0)
   {
      if (num_args)
	 *num_args = 5;
   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MASK) == 0)
   {
      if (num_args)
	 *num_args = 5;
   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_LOCATION) == 0)
   {
      if (num_args)
	 *num_args = 4;
   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_TIME) == 0)
   {
      if (num_args)
	 *num_args = 1;
   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MAP) == 0)
   {
      if (num_args)
	 *num_args = 5;
   }
   else
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_region_constraint_print
|       Purpose: print the region info attribute
|    Written By: Jeremy Worley 
|          Date: Thu Dec  8 14:37:51 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_constraint_print(kobject obj, int assoc, int attrib, kaddr clientData,
		   kfile *outfile)
{
   int rw;
   int rh;
   int rd;
   int rt;
   int re;

   /* if the segment does not exist, then it has no region info */
   if (!kdms_query_segment(obj, (char *) clientData))
      return FALSE;		/* not an error */

   /* the region primitive varies for every segment */
   if (kstrcmp((char *)clientData, KDMS_SEGMENT_VALUE) == 0)
   {

      kpds_get_attribute(obj, KPDS_VALUE_OPTIMAL_REGION_CONSTRAINT, 
			 &rw, &rh, &rd, &rt, &re);
      if (outfile)
	 kfprintf(outfile, "%d x %d x %d x %d x %d\t",
		  rw, rh, rd, rt, re);

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MASK) == 0)
   {

      kpds_get_attribute(obj, KPDS_MASK_OPTIMAL_REGION_CONSTRAINT, 
			 &rw, &rh, &rd, &rt, &re);
      if (outfile)
	 kfprintf(outfile, "%d x %d x %d x %d x %d\t",
		  rw, rh, rd, rt, re);

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_LOCATION) == 0)
   {

      kpds_get_attribute(obj, KPDS_LOCATION_OPTIMAL_REGION_CONSTRAINT,
			 &rw, &rh, &re, &re);
      if (outfile)
	 kfprintf(outfile, "%d x %d x %d x %d\t",
		  rw, rh, rd, rt, re);

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_TIME) == 0)
   {

      kpds_get_attribute(obj, KPDS_TIME_OPTIMAL_REGION_CONSTRAINT, &rt);
      if (outfile)
	 kfprintf(outfile, "%d\t", rt);

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MAP) == 0)
   {

      kpds_get_attribute(obj, KPDS_MAP_OPTIMAL_REGION_CONSTRAINT, 
			 &rw, &rh, &re, &rd, &rt);
      if (outfile)
	 kfprintf(outfile, "%d x %d x %d x %d x %d\t"
		  "%d", rw, rh, re, rt, rd);
   }
   else
      return FALSE;

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_OPTIMAL_REGION_FLAGS
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_region_flags_get
|       Purpose: This routine retrieves the "region flags information".
|    Written By: Jeremy Worley
|          Date: Thu Dec  8 14:37:51 1994
------------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_flags_get(kobject obj, int assoc, int attr, kaddr clientData,
		 kva_list *list)
{
   int *flags = NULL;
   int dim;
   int i;
   int *tmp;
   
   kdms_get_attributes(obj, (char *)clientData, 
		       _INTERNAL_REGION_FLAGS, &flags,
		       KDMS_DIMENSION, &dim, NULL);
   
   for (i = 0; i < dim; i++)
      if ((tmp = kva_arg(*list, int *)) != NULL)
	 (*tmp) = flags[i];

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_region_flags_set
|       Purpose: This routine stores the "region flags information".
|    Written By: Jeremy Worley
|          Date: Thu Dec  8 14:37:51 1994
------------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_flags_set(kobject obj, int assoc, int attr, kaddr clientData,
		 kva_list *list)
{
   int *flags = NULL;
   int dim;
   int i;
   
   if (!kdms_get_attributes(obj, (char *)clientData, 
		       _INTERNAL_REGION_FLAGS, &flags,
		       KDMS_DIMENSION, &dim, NULL))
      return FALSE;
   
   for (i = 0; i < dim; i++)
      flags[i] = kva_arg(*list, int);
   
   if (!kdms_set_attribute(obj, (char *)clientData, 
		       _INTERNAL_REGION_FLAGS, flags))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_region_flags_match
|       Purpose: match the region info attribute
|    Written By: Jeremy Worley 
|          Date: Thu Dec  8 14:37:51 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_flags_match(kobject obj1, kobject obj2, int assoc, int attrib,
		   kaddr clientData1, kaddr clientData2)
{
   int *flags1 = NULL;
   int dim1;
   int *flags2 = NULL;
   int dim2;
   int i;
   
   kdms_get_attributes(obj1, (char *)clientData1, 
		       _INTERNAL_REGION_FLAGS, &flags1,
		       KDMS_DIMENSION, &dim1, NULL);

   kdms_get_attributes(obj2, (char *)clientData2, 
		       _INTERNAL_REGION_FLAGS, &flags2,
		       KDMS_DIMENSION, &dim2, NULL);
 
   if (dim1 != dim2)
      return (FALSE);
   
   for (i = 0; i < dim1; i++)
      if (flags1[i] != flags2[i])
	 return(FALSE);
   
   return(TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_region_flags_query
|       Purpose: query the region info attribute
|    Written By: Jeremy Worley 
|          Date: Thu Dec  8 14:37:51 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_flags_query(kobject obj, int assoc, int attrib, kaddr clientData,
		   int *num_args, int *arg_size, int *data_type,
		   int *permanent)
{
   /* if the segment does not exist, then it has no region info */
   if (!kdms_query_segment(obj, (char *) clientData))
      return FALSE;		/* not an error */

   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = FALSE;

   /* the num_args of the region info primitive varies for every segment */
   if (kstrcmp((char *)clientData, KDMS_SEGMENT_VALUE) == 0)
   {
      if (num_args)
	 *num_args = 5;
   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MASK) == 0)
   {
      if (num_args)
	 *num_args = 5;
   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_LOCATION) == 0)
   {
      if (num_args)
	 *num_args = 4;
   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_TIME) == 0)
   {
      if (num_args)
	 *num_args = 1;
   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MAP) == 0)
   {
      if (num_args)
	 *num_args = 5;
   }
   else
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_region_flags_print
|       Purpose: print the region info attribute
|    Written By: Jeremy Worley 
|          Date: Thu Dec  8 14:37:51 1994 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_flags_print(kobject obj, int assoc, int attrib, kaddr clientData,
		   kfile *outfile)
{
   int rw;
   int rh;
   int rd;
   int rt;
   int re;

   /* if the segment does not exist, then it has no region info */
   if (!kdms_query_segment(obj, (char *) clientData))
      return FALSE;		/* not an error */

   /* the region primitive varies for every segment */
   if (kstrcmp((char *)clientData, KDMS_SEGMENT_VALUE) == 0)
   {

      kpds_get_attribute(obj, KPDS_VALUE_OPTIMAL_REGION_FLAGS, 
			 &rw, &rh, &rd, &rt, &re);
      if (outfile)
	 kfprintf(outfile, "%d, %d, %d, %d, %d\t",
		  rw, rh, rd, rt, re);

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MASK) == 0)
   {

      kpds_get_attribute(obj, KPDS_MASK_OPTIMAL_REGION_FLAGS, 
			 &rw, &rh, &rd, &rt, &re);
      if (outfile)
	 kfprintf(outfile, "%d, %d, %d, %d, %d\t",
		  rw, rh, rd, rt, re);

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_LOCATION) == 0)
   {

      kpds_get_attribute(obj, KPDS_LOCATION_OPTIMAL_REGION_FLAGS,
			 &rw, &rh, &re, &re);
      if (outfile)
	 kfprintf(outfile, "%d, %d, %d, %d\t",
		  rw, rh, rd, rt, re);

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_TIME) == 0)
   {

      kpds_get_attribute(obj, KPDS_TIME_OPTIMAL_REGION_FLAGS, &rt);
      if (outfile)
	 kfprintf(outfile, "%d\t", rt);

   }
   else if (kstrcmp((char *)clientData, KDMS_SEGMENT_MAP) == 0)
   {

      kpds_get_attribute(obj, KPDS_MAP_OPTIMAL_REGION_FLAGS, 
			 &rw, &rh, &re, &rd, &rt);
      if (outfile)
	 kfprintf(outfile, "%d, %d, %d, %d, %d\t"
		  "%d", rw, rh, re, rt, rd);
   }
   else
      return FALSE;

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_OPTIMAL_REGION_SIZE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_region_size_get
|       Purpose: get the optimal region size
|    Written By: Steve Kubica & Jeremy Worley
|          Date: Apr 06, 1994 12:29
------------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_size_get(kobject obj, int assoc, int attr, kaddr clientData,
			 kva_list *list)
{
   int *size = NULL;
   int *opt_size = NULL;
   int *constraint = NULL;
   int *flags = NULL;
   int *tmp = NULL;
   int  num_regions;
   int  dim;
   int  i;
   int tmp_size[KDMS_MAX_DIM];

   /* if the segment does not exist, then it has no optimum region size */
   if (!kdms_query_segment(obj, (char *) clientData))
      return FALSE;		/* not an error */

   /* 
    * get the kdms optimal region size for this region.  note that 
    * this attribute is not mapping aware, but that's ok for right now
    * because it doesn't need to be.   the last element in this array
    * is conveniently going to be the number of optimum regions
    */
   if (!kdms_get_attributes(obj, (char *) clientData,
			    KDMS_OPTIMAL_REGION_SIZE, &opt_size,
			    KDMS_SIZE, &size,
			    _INTERNAL_REGION_CONSTRAINT, &constraint,
			    _INTERNAL_REGION_FLAGS, &flags,
			    KDMS_DIMENSION, &dim, NULL))
      return FALSE;

   /*
    * copy the size because otherwise it'll modify the presentation
    * size if mapping mode is set.
    */
   kmemcpy(&(tmp_size[0]),size, dim * sizeof(int));
   size = &(tmp_size[0]);
   
   /*
    * if the mapping mode is on, then we want to multiply the number of
    * "optimal regions" by the number size of the map_width dimension in
    * the map segment.
    */
   num_regions = opt_size[dim];

   if (kaps_mapping_mode(obj))
   {
      if ( (kstrcmp((char *) clientData, KDMS_SEGMENT_VALUE) == 0) ||
	   (kstrcmp((char *) clientData, KDMS_SEGMENT_MASK)  == 0))
      {
	 int *msize;
	 

	 kdms_get_attribute(obj, KDMS_SEGMENT_MAP, KDMS_SIZE, &msize);
	 num_regions *= msize[KMAP_WID];
	 size[KELM] *= msize[KMAP_WID];
      }
   }

   /*
    * ok.  now lets mess with the optimal region size based on the
    * constraints and constraint flags.
    */
   for (i = 0; i < dim; i++)
   {
      if (flags[i] == 0) /* factor */
      {
         /* subtract off what doesn't fit */
	 opt_size[i] -= (opt_size[i] % constraint[i]); 
	 opt_size[i] = kmax(opt_size[i], constraint[i]);
      }
      else /* fixed value */
      {
	 if (i < dim - 1 && constraint[i] < size[i])
	 {
	    opt_size[i+1] *= kmax(1, opt_size[i] / constraint[i]);
	    opt_size[i+1] = kmin(size[i+1], opt_size[i+1]);
	 }
	 opt_size[i] = constraint[i];
      }
   }

   /*
    * recompute the number of optimal regions because its been
    * hopelessly munged by the last round of twiddles.
    */
   for (i = 0, num_regions = 1; i < dim; i++)
      num_regions *= (int)(kceil((float)size[i]/(float)opt_size[i]));
   
	       
   /* first return the region size */
   for (i = 0; i < dim; i++)
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = opt_size[i];

   /* now get the number of those regions */
   if ((tmp = kva_arg(*list, int *)) != NULL)
         (*tmp) = num_regions;

   /*
    * After everything is done, check opt_size to make sure that none
    * of the values are zero.  Do it here so that if plural
    * get_attributes is called, later attributes will get set
    * properly.  Issue a warning message if necessary.
    */
   for (i = 0; i < dim; i++)
      if (opt_size[i] <= 0)
      {
	 kinfo(KSTANDARD,"Computed optimal region is invalid.  This is "
	       "probably due to an uninitialized size attribute or "
	       "illegal optimal region constraints.  Please make sure that "
	       "the size attribute has been properly initialized.\n");
	 return FALSE;
      }
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_regon_size_match
|       Purpose: match the optimal_region_size attribute
|    Written By: 
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_size_match(kobject obj1, kobject obj2, int assoc, int attrib,
			   kaddr clientData1, kaddr clientData2)
{
   return (kdms_match_attribute(obj1, obj2, (char *) clientData1, 
				KDMS_OPTIMAL_REGION_SIZE));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_region_size_query
|       Purpose: query the optimal_region_size attribute
|    Written By: 
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_size_query(kobject obj, int assoc, int attrib,
			   kaddr clientData, int *num_args, int *arg_size,
			   int *data_type, int *permanent)
{
   return (kdms_query_attribute(obj, (char *) clientData, 
				KDMS_OPTIMAL_REGION_SIZE,
				num_args, arg_size, data_type, permanent));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _optimal_region_size_print
|       Purpose: print the optimal_region_size attribute
|    Written By: 
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_optimal_region_size_print(kobject obj, int assoc, int attrib,
			   kaddr clientData, kfile *outfile)
{
   return (kdms_print_attribute(obj, (char *) clientData,
				KDMS_OPTIMAL_REGION_SIZE, outfile));
}


/*
 *    ==================================================================
 *    Initialization Routine
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kpds_init
|
|       Purpose: 
|
|		 We don't need:	KPDS_NAME, KPDS_ARCHITECTURE
|		 because they are #defined to KDMS_*,
|		 have exactly the same arguments, and are
|		 object level attributes.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 08:10
| Modifications:
|
------------------------------------------------------------*/
int
kpds_init(void)
{
   int status = TRUE;
   int i;

   if (pds_initialized)
      return TRUE;
   else
      pds_initialized = TRUE;

   /* -- initialize the data segments -- */
   status &= kpds_init_value_segment();
   status &= kpds_init_mask_segment();
   status &= kpds_init_map_segment();
   status &= kpds_init_location_segment();
   status &= kpds_init_time_segment();
   
   /*
    *   tokenize the primitive table
    */
   for (i = 0; i < kpds_num_primitives; i++)
   {
      kpds_primitives[i].token = kstring_to_token(kpds_primitives[i].name);
      kpds_primitives[i].seg_token =
	 kstring_to_token(kpds_primitives[i].segment);
   }

   /* 
    *   region size attributes
    */
   kdms_define_attribute(KDMS_OBJECT, KPDS_MASK_REGION_SIZE, 5, 1, KINT,
			 FALSE, FALSE, 0, 0, 0, 0, 0);

   kdms_define_attribute(KDMS_OBJECT, KPDS_VALUE_REGION_SIZE, 5, 1, KINT,
			 FALSE, FALSE, 0, 0, 0, 0, 0);

   kdms_define_attribute(KDMS_OBJECT, KPDS_MAP_REGION_SIZE, 5, 1, KINT,
			 FALSE, FALSE, 0, 0, 0, 0, 0);

   kdms_define_attribute(KDMS_OBJECT, KPDS_LOCATION_REGION_SIZE, 4, 1, KINT,
			 FALSE, FALSE, 0, 0, 0, 0);

   kdms_define_attribute(KDMS_OBJECT, KPDS_TIME_REGION_SIZE, 1, 1, KINT,
			 FALSE, FALSE, 0);

   /* 
    *  line info attributes
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MASK_LINE_INFO,
			       _pds_seg_mask,
			       _line_info_get, _nop_set, _line_info_match,
			       _nop_copy, _line_info_query, _line_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_VALUE_LINE_INFO,
			       _pds_seg_value,
			       _line_info_get, _nop_set, _line_info_match,
			       _nop_copy, _line_info_query, _line_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MAP_LINE_INFO,
			       _pds_seg_map,
			       _line_info_get, _nop_set, _line_info_match,
			       _nop_copy, _line_info_query, _line_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_LOCATION_LINE_INFO,
			       _pds_seg_location,
			       _line_info_get, _nop_set, _line_info_match,
			       _nop_copy, _line_info_query, _line_info_print);


   /* 
    *   plane info attributes
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MASK_PLANE_INFO,
			       _pds_seg_mask,
			       _plane_info_get, _nop_set, _plane_info_match,
			       _nop_copy, _plane_info_query, 
			       _plane_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_VALUE_PLANE_INFO,
			       _pds_seg_value,
			       _plane_info_get, _nop_set, _plane_info_match,
			       _nop_copy, _plane_info_query, 
			       _plane_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MAP_PLANE_INFO,
			       _pds_seg_map,
			       _plane_info_get, _nop_set, _plane_info_match,
			       _nop_copy, _plane_info_query, 
			       _plane_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_LOCATION_PLANE_INFO,
			       _pds_seg_location,
			       _plane_info_get, _nop_set, _plane_info_match,
			       _nop_copy, _plane_info_query, 
			       _plane_info_print);

   /* 
    *   volume info attributes
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MASK_VOLUME_INFO,
			       _pds_seg_mask,
			       _volume_info_get, _nop_set, _volume_info_match,
			       _nop_copy, _volume_info_query, 
			       _volume_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_VALUE_VOLUME_INFO,
			       _pds_seg_value,
			       _volume_info_get, _nop_set, _volume_info_match,
			       _nop_copy, _volume_info_query, 
			       _volume_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MAP_VOLUME_INFO,
			       _pds_seg_map,
			       _volume_info_get, _nop_set, _volume_info_match,
			       _nop_copy, _volume_info_query, 
			       _volume_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_LOCATION_VOLUME_INFO,
			       _pds_seg_location,
			       _volume_info_get, _nop_set, _volume_info_match,
			       _nop_copy, _volume_info_query, 
			       _volume_info_print);

   /* 
    *   region info attributes
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MASK_REGION_INFO,
			       _pds_seg_mask,
			       _region_info_get, _nop_set, _region_info_match,
			       _nop_copy, _region_info_query, 
			       _region_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_VALUE_REGION_INFO,
			       _pds_seg_value,
			       _region_info_get, _nop_set, _region_info_match,
			       _nop_copy, _region_info_query,
			       _region_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MAP_REGION_INFO,
			       _pds_seg_map,
			       _region_info_get, _nop_set, _region_info_match,
			       _nop_copy, _region_info_query, 
			       _region_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_LOCATION_REGION_INFO,
			       _pds_seg_location,
			       _region_info_get, _nop_set, _region_info_match,
			       _nop_copy, _region_info_query, 
			       _region_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_TIME_REGION_INFO,
			       _pds_seg_time,
			       _region_info_get, _nop_set, _region_info_match,
			       _nop_copy, _region_info_query, 
			       _region_info_print);

   /* 
    *   vector info attributes
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MASK_VECTOR_INFO,
			       _pds_seg_mask,
			       _vector_info_get, _nop_set, _vector_info_match,
			       _nop_copy, _vector_info_query, 
			       _vector_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_VALUE_VECTOR_INFO,
			       _pds_seg_value,
			       _vector_info_get, _nop_set, _vector_info_match,
			       _nop_copy, _vector_info_query,
			       _vector_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MAP_VECTOR_INFO,
			       _pds_seg_map,
			       _vector_info_get, _nop_set, _vector_info_match,
			       _nop_copy, _vector_info_query, 
			       _vector_info_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_LOCATION_VECTOR_INFO,
			       _pds_seg_location,
			       _vector_info_get, _nop_set, _vector_info_match,
			       _nop_copy, _vector_info_query, 
			       _vector_info_print);

   /*
    * optimal_region attributes
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_LOCATION_OPTIMAL_REGION_SIZE,
			       _pds_seg_location,
			       _optimal_region_size_get, _nop_set,
			       _optimal_region_size_match, _nop_copy,
			       _optimal_region_size_query, 
			       _optimal_region_size_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MAP_OPTIMAL_REGION_SIZE,
			       _pds_seg_map,
			       _optimal_region_size_get, _nop_set,
			       _optimal_region_size_match, _nop_copy,
			       _optimal_region_size_query, 
			       _optimal_region_size_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MASK_OPTIMAL_REGION_SIZE,
			       _pds_seg_mask,
			       _optimal_region_size_get, _nop_set,
			       _optimal_region_size_match, _nop_copy,
			       _optimal_region_size_query, 
			       _optimal_region_size_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_TIME_OPTIMAL_REGION_SIZE,
			       _pds_seg_time,
			       _optimal_region_size_get, _nop_set,
			       _optimal_region_size_match, _nop_copy,
			       _optimal_region_size_query, 
			       _optimal_region_size_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_VALUE_OPTIMAL_REGION_SIZE,
			       _pds_seg_value,
			       _optimal_region_size_get, _nop_set,
			       _optimal_region_size_match, _nop_copy,
			       _optimal_region_size_query, 
			       _optimal_region_size_print);

   /*
    * optimal_region_constraint attributes
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_LOCATION_OPTIMAL_REGION_CONSTRAINT,
			       _pds_seg_location,
			       _optimal_region_constraint_get,
			       _optimal_region_constraint_set,
			       _optimal_region_constraint_match, _nop_copy,
			       _optimal_region_constraint_query, 
			       _optimal_region_constraint_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MAP_OPTIMAL_REGION_CONSTRAINT,
			       _pds_seg_map,
			       _optimal_region_constraint_get,
			       _optimal_region_constraint_set,
			       _optimal_region_constraint_match, _nop_copy,
			       _optimal_region_constraint_query, 
			       _optimal_region_constraint_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MASK_OPTIMAL_REGION_CONSTRAINT,
			       _pds_seg_mask,
			       _optimal_region_constraint_get,
			       _optimal_region_constraint_set,
			       _optimal_region_constraint_match, _nop_copy,
			       _optimal_region_constraint_query, 
			       _optimal_region_constraint_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_TIME_OPTIMAL_REGION_CONSTRAINT,
			       _pds_seg_time,
			       _optimal_region_constraint_get,
			       _optimal_region_constraint_set,
			       _optimal_region_constraint_match, _nop_copy,
			       _optimal_region_constraint_query, 
			       _optimal_region_constraint_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_VALUE_OPTIMAL_REGION_CONSTRAINT,
			       _pds_seg_value,
			       _optimal_region_constraint_get,
			       _optimal_region_constraint_set,
			       _optimal_region_constraint_match, _nop_copy,
			       _optimal_region_constraint_query, 
			       _optimal_region_constraint_print);

   /*
    * optimal_region_flags attributes
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_LOCATION_OPTIMAL_REGION_FLAGS,
			       _pds_seg_location,
			       _optimal_region_flags_get,
			       _optimal_region_flags_set,
			       _optimal_region_flags_match, _nop_copy,
			       _optimal_region_flags_query, 
			       _optimal_region_flags_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MAP_OPTIMAL_REGION_FLAGS,
			       _pds_seg_map,
			       _optimal_region_flags_get,
			       _optimal_region_flags_set,
			       _optimal_region_flags_match, _nop_copy,
			       _optimal_region_flags_query, 
			       _optimal_region_flags_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_MASK_OPTIMAL_REGION_FLAGS,
			       _pds_seg_mask,
			       _optimal_region_flags_get,
			       _optimal_region_flags_set,
			       _optimal_region_flags_match, _nop_copy,
			       _optimal_region_flags_query, 
			       _optimal_region_flags_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_TIME_OPTIMAL_REGION_FLAGS,
			       _pds_seg_time,
			       _optimal_region_flags_get,
			       _optimal_region_flags_set,
			       _optimal_region_flags_match, _nop_copy,
			       _optimal_region_flags_query, 
			       _optimal_region_flags_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, KPDS_VALUE_OPTIMAL_REGION_FLAGS,
			       _pds_seg_value,
			       _optimal_region_flags_get,
			       _optimal_region_flags_set,
			       _optimal_region_flags_match, _nop_copy,
			       _optimal_region_flags_query, 
			       _optimal_region_flags_print);

   /*
    * region constraint internal variables
    */
   kdms_define_attribute(KDMS_ALL_SEGMENTS, _INTERNAL_REGION_CONSTRAINT, 
					    1, 5, KINT, FALSE, FALSE, ones);

   kdms_define_attribute(KDMS_ALL_SEGMENTS, _INTERNAL_REGION_FLAGS, 
					    1, 5, KINT, FALSE, FALSE, zeros);

   /*
    *   histogram attributes 
    */
   kdms_define_attribute(KDMS_OBJECT, KPDS_VALUE_HIST_REGION_SIZE, 5, 1, KINT,
			 FALSE, FALSE, 1, 1, 1, 1, 1);
   kdms_define_attribute(KDMS_OBJECT, KPDS_VALUE_HIST_POSITION, 5, 1, KINT,
			 FALSE, FALSE, 0, 0, 0, 0, 0);
   kdms_define_attribute(KDMS_OBJECT, KPDS_VALUE_HIST_NUMBINS, 1, 1, KINT,
			 FALSE, FALSE, 256);
   kdms_define_attribute(KDMS_OBJECT, KPDS_VALUE_HIST_RANGE, 2, 1, KDOUBLE,
			 FALSE, FALSE, 0.0, 1.0);

   /*
    * initialize the common ground
    */
   kaps_init();

   /*
    * It is possible to generate errnos with duplicate attribute
    * definitions.  To prevent the user from seeing this, we'll
    * re-zero the errno.
    */
   errno = 0;

   return TRUE;
}


/*
 *    ==================================================================
 *    Data Retrieval Routines
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kpds_get_histogram
|
|       Purpose: This is the data retrieval routine for the
|		 KPDS_VALUE_HISTOGRAM primitive.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 30, 1993 14:27
| Modifications:
|
------------------------------------------------------------*/
kaddr
kpds_get_histogram(
   kobject object,
   kaps_primitive *prim,
   kaddr data)
{
   int    i;
   int    begin[5];
   int    end[5];
   int    p[5];
   int    r[5];
   int   *size;
   int    dim;
   int    type;
   int    numbins;
   int    numpts;
   double rmin;
   double rmax;
   kaddr  tmp;
   unsigned int *idata;


   kdms_get_attributes(object, prim->segment,
		       KDMS_DIMENSION, &dim,
		       KDMS_DATA_TYPE, &type,
		       KDMS_SIZE, &size, NULL);

   kdms_get_attributes(object, KDMS_OBJECT,
		       KPDS_VALUE_HIST_POSITION, 
		          &p[0], &p[1], &p[2], &p[3], &p[4],
		       KPDS_VALUE_HIST_REGION_SIZE, 
		          &r[0], &r[1], &r[2], &r[3], &r[4],
		       KPDS_VALUE_HIST_NUMBINS, &numbins,
		       KPDS_VALUE_HIST_RANGE, &rmin, &rmax,
		       NULL);

   r[0] += p[0] - 1;
   r[1] += p[1] - 1;
   r[2] += p[2] - 1;
   r[3] += p[3] - 1;
   r[4] += p[4] - 1;

   for (i = 0, numpts = 1; i < dim; i++)
   {
      begin[i] = p[i];
      end[i] = r[i];
      numpts *= r[i] + 1;
   }

   tmp = kdms_get_data(object, prim->segment, begin, end, NULL);

   if (data == NULL)
      data = (unsigned int *)kcalloc((unsigned)numbins, (signed)
				     sizeof(unsigned int));
   else
      kmemset(data, (signed)(numbins * sizeof(unsigned int)), 0);

   idata = (unsigned int *)data;
   switch (type)
   {
      case KBYTE:
	 {
	    char *dta = (char *)tmp;
	    char dmin = (char)rmin;
	    double binwidth = kceil(((rmax - rmin) / (double)numbins));
	    for (i = 0; i < numpts; i++)
	       idata[(int)(kmin(numbins - 1, 
			  kmax(0, ((dta[i] - dmin) * binwidth))))] += 1;
	 }
	 break;
      case KUBYTE:
	 {
	    unsigned char *dta = (unsigned char *)tmp;
	    unsigned char dmin = (unsigned char)rmin;
	    double binwidth = kceil(((rmax - rmin) / (double)numbins));
	    for (i = 0; i < numpts; i++)
	       idata[(int)(kmin((int)(numbins - 1),
			  (int)kmax(0, (int)((signed)(dta[i] - dmin) * binwidth))))] += 1;
	 }
	 break;
      case KSHORT:
	 {
	    short *dta = (short *)tmp;
	    short dmin = (short)rmin;
	    double binwidth = kceil(((rmax - rmin) / (double)numbins));
	    for (i = 0; i < numpts; i++)
	       idata[(int)(kmin(numbins - 1,
			  kmax(0, ((dta[i] - dmin) * binwidth))))] += 1;
	 }
	 break;
      case KUSHORT:
	 {
	    unsigned short *dta = (unsigned short *)tmp;
	    unsigned short dmin = (unsigned short)rmin;
	    double binwidth = kceil(((rmax - rmin) / (double)numbins));
	    for (i = 0; i < numpts; i++)
	       idata[(int)(kmin((int)(numbins - 1),
			  (int)kmax(0, (int)((signed)(dta[i] - dmin) * binwidth))))] += 1;
	 }
	 break;
      case KINT:
	 {
	    int *dta = (int *)tmp;
	    int dmin = (int)rmin;
	    double binwidth = kceil(((rmax - rmin) / (double)numbins));
	    for (i = 0; i < numpts; i++)
	       idata[(int)(kmin(numbins - 1,
			  kmax(0, ((dta[i] - dmin) * binwidth))))] += 1;
	 }
	 break;
      case KUINT:
	 {
	    unsigned int *dta = (unsigned int *)tmp;
	    unsigned int dmin = (unsigned int)rmin;
	    double binwidth = kceil(((rmax - rmin) / (double)numbins));
	    for (i = 0; i < numpts; i++)
	       idata[(int)(kmin(numbins - 1,
			  ((dta[i] - dmin) * binwidth)))] += 1;
	 }
	 break;
      case KLONG:
	 {
	    long *dta = (long *)tmp;
	    long dmin = (long)rmin;
	    double binwidth = kceil(((rmax - rmin) / (double)numbins));
	    for (i = 0; i < numpts; i++)
	       idata[(int)(kmin(numbins - 1, 
			  kmax(0, ((dta[i] - dmin) * binwidth))))] += 1;
	 }
	 break;
      case KULONG:
	 {
	    unsigned long *dta = (unsigned long *)tmp;
	    unsigned long dmin = (unsigned long)rmin;
	    double binwidth = kceil(((rmax - rmin) / (double)numbins));
	    for (i = 0; i < numpts; i++)
	       idata[(int)(kmin(numbins - 1, 
			  ((dta[i] - dmin) * binwidth)))] += 1;
	 }
	 break;
      case KFLOAT:
	 {
	    float *dta = (float *)tmp;
	    float dmin = (float)rmin;
	    double binwidth = kceil(((rmax - rmin) / (double)numbins));
	    for (i = 0; i < numpts; i++)
	       idata[(int)(kmin(numbins - 1,
				kmax(0, ((dta[i] - dmin) * binwidth))))] += 1;
	 }
	 break;
      case KDOUBLE:
	 {
	    double *dta = (double *)tmp;
	    double dmin = (double)rmin;
	    double binwidth = kceil(((rmax - rmin) / (double)numbins));
	    for (i = 0; i < numpts; i++)
	       idata[(int)(kmin(numbins - 1,
				kmax(0, ((dta[i] - dmin) * binwidth))))] += 1;
	 }
	 break;
   }
   kfree(tmp);
   return data;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kpds_region_corners
|
|       Purpose: 
|
|         Input: 
|
|		  begin - preallocated array to put the 
|			  region begin point into
|		  end   - preallocated array to put the 
|			  region end point into
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: May 02, 1994 18:10
| Modifications:
|
------------------------------------------------------------*/
static int
_kpds_region_corners(
   kobject obj,
   int     seg_token,
   int    *begin,
   int    *end)
{
   int w;
   int h;
   int d;
   int t;
   int e;
   int i;


   /* the begin point is always at zero */
   for (i = 0; i < KDMS_MAX_DIM; i++)
      begin[i] = 0;

   /* the end point is dependent on the region size for the given segment */
   if (seg_token == value_token)
   {
      kpds_get_attribute(obj, KPDS_VALUE_REGION_SIZE, &w, &h, &d, &t, &e);
      end[KWID] = w - 1;
      end[KHGT] = h - 1;
      end[KDEP] = d - 1;
      end[KTIM] = t - 1;
      end[KELM] = e - 1;

   }
   else if (seg_token == mask_token)
   {
      kpds_get_attribute(obj, KPDS_MASK_REGION_SIZE, &w, &h, &d, &t, &e);
      end[KWID] = w - 1;
      end[KHGT] = h - 1;
      end[KDEP] = d - 1;
      end[KTIM] = t - 1;
      end[KELM] = e - 1;

   }
   else if (seg_token == location_token)
   {
      kpds_get_attribute(obj, KPDS_LOCATION_REGION_SIZE, &w, &h, &d, &e);

      end[KWID] = w - 1;
      end[KHGT] = h - 1;
      end[KDEP] = d - 1;
      end[KDIM] = e - 1;

   }
   else if (seg_token == time_token)
   {
      kpds_get_attribute(obj, KPDS_TIME_REGION_SIZE, &t);
      end[0] = t - 1;

   }
   else if (seg_token == map_token)
   {
      kpds_get_attribute(obj, KPDS_MAP_REGION_SIZE, &w, &h, &d, &t, &e);
      end[KMAP_WID] = w - 1;
      end[KMAP_HGT] = h - 1;
      end[KMAP_DEP] = d - 1;
      end[KMAP_TIM] = t - 1;
      end[KMAP_ELM] = e - 1;

   }
   else if (seg_token == width_token)
   {
      kpds_get_attribute(obj, KPDS_LOCATION_REGION_SIZE, &w, NULL, NULL, NULL);
      end[0] = w - 1;
   }
   else if (seg_token == height_token)
   {
      kpds_get_attribute(obj, KPDS_LOCATION_REGION_SIZE, NULL, &h, NULL, NULL);
      end[0] = h - 1;
   }
   else if (seg_token == depth_token)
   {
      kpds_get_attribute(obj, KPDS_LOCATION_REGION_SIZE, NULL, NULL, &d, NULL);
      end[0] = d - 1;
   }
   else
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kpds_get_region
|
|       Purpose: This is the data retrieval routine for the
|		 region primitives.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: May 02, 1994 18:10
| Modifications:
|
------------------------------------------------------------*/
kaddr
kpds_get_region(
   kobject         object,
   kaps_primitive *prim,
   kaddr           data)
{
   /* this code depends on these being initialized to zero! */
   static int  begin[KDMS_MAX_DIM];
   static int *b = begin;
   static int  end[KDMS_MAX_DIM];
   static int *e = end;
   int *size;


   /* verify that the segment is present and get its size */
   if ((size = kaps_get_segment_size(object, prim->segment)) == NULL)
      return NULL;

   /* get the begin and end corners marking the region size */
   _kpds_region_corners(object, prim->seg_token, b, e);

   /* increment the position for this segment */
   kaps_advance_segment_position(object, prim->segment, begin, end, size, 
				 TRUE);

   /* move the begin and end to the appropriate position and offset */
   kaps_offset_corners(object, prim->segment, prim->usage, b, e);

   return (kdms_get_data(object, prim->segment, begin, end, data));
}

/*-----------------------------------------------------------
|
|  Routine Name: kpds_put_region
|
|       Purpose: This is the data storage routine for the
|		 region primitives.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: May 02, 1994 18:10
| Modifications:
|
------------------------------------------------------------*/
int
kpds_put_region(
   kobject         object,
   kaps_primitive *prim,
   kaddr           data)
{
   /* this code depends on these being initialized to zero! */
   static int  begin[KDMS_MAX_DIM];
   static int *b = begin;
   static int  end[KDMS_MAX_DIM];
   static int *e = end;
   int *size;

   /* verify that the segment is present and get its size */
   if ((size = kaps_get_segment_size(object, prim->segment)) == NULL)
      return FALSE;

   /* get the begin and end corners marking the region size */
   _kpds_region_corners(object, prim->seg_token, b, e);

   /* increment the position for this segment */
   kaps_advance_segment_position(object, prim->segment, b, e, size, FALSE);

   /* move the begin and end to the appropriate position and offset */
   kaps_offset_corners(object, prim->segment, prim->usage, b, e);

   return (kdms_put_data(object, prim->segment, begin, end, data));
}


/*
 *    ==================================================================
 *    Public API Routines
 *    ==================================================================
 */

/************************************************************
*
*  Routine Name: kpds_get_attribute - get the value of an attribute
*  		                      from a data object
*
*       Purpose: This routine is used to get the value of
*		 an attribute from a data object.
*
*		 Data Services manages two versions of some of the
*		 attributes associated with each object.  These
*		 attributes are the size and data type.  Internally,
*		 the two versions of these attributes are referred to
*		 as the physical attribute and the presentation
*		 attribute.  Typically, the programmer has access
*		 to only the presentation versions of these attributes.
*		 The physical attributes are set indirectly depending
*		 on the setting of KPDS_COUPLING.  See kpds_get_data
*		 for a description of how the presentation and physical
*		 attributes affect interaction with the data object.
*
*	         Other attributes are classified as either shared or
*	         unshared.  Shared attributes are stored at the
*	         physical layer of the attribute, and thus can be
*	         shared by multiple references of the data object (see
*	         kpds_reference_object for more information about
*	         references).  Unshared attributes, on the other hand,
*	         can only be used by the local object (see
*	         kpds_query_attribute for information on how to
*	         determine whether an attribute is shared or
*	         unshared).
*		
*		 This function accepts an object, the name of the
*		 attribute that is desired and the address of a
*		 location in which to return the value of the
*		 attribute.  Certain KPDS attributes require more than
*		 one argument in order to return the entire attribute.
*		 For example, KPDS_VALUE_SIZE requires five (5)
*		 arguments.  Getting the KPDS_VALUE_DATA_TYPE
*		 attribute might look like this:
*
*		!	kpds_get_attribute(object, KPDS_VALUE_DATA_TYPE, &typ);
*
*		 Getting the KPDS_VALUE_SIZE attribute, is a little
*		 more complex:
*
*		!	kpds_get_attribute(object, KPDS_VALUE_SIZE,
*		!		&w, &h, &d, &t, &e);
*
*		 A complete list of the polymorphic attributes can be
*		 found in Chapter 2 of Programming Services Volume II.
*
*         Input: object    - the object from which to retrieve the
*			     specified attribute.  This must be a
*			     legal kobject that has been opened or
*			     instantiated by an appropriate kpds
*			     function call, such as kpds_open_object
*			     or kpds_reference.
*		 attribute - the name of the attribute to retrieve.
*			     this is a character string that is
*			     the name of an existing attribute in
*			     the object.  There are a large number
*			     of predefined KPDS attributes.  Users
*			     can also create attributes via the
*			     kpds_create_attribute function call.
*
*        Output: va_alist   - a C variable argument list that contains
*			     addresses to locations in which 
*			     KPDS can return the value of the 
*			     specified attribute.
*
*       Returns: TRUE (1) if the attribute was successfully retrieved
*	 	 and FALSE (0) otherwise.
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 30, 1993 13:44
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration:
*
*************************************************************/
int
kpds_get_attribute(
   kobject object, 
   char   *attribute, 
   kvalist)
{
   kva_list list;
   int status;


   kpds_init();

   kva_start(list, attribute);

   status = kdms_vget_attribute(object, KDMS_OBJECT, attribute, &list);

   kva_end(list);

   return status;
}

/************************************************************
*
*  Routine Name: kpds_get_attributes - get the values of multiple 
*				       attributes from a data object
*
*       Purpose: This function is used to retrieve the values of an 
*		 arbitrary number of attributes from a data object.
*
*		 Data Services manages two versions of some of the
*		 attributes associated with each object.  These
*		 attributes are the size and data type.  Internally,
*		 the two versions of these attributes are referred to
*		 as the physical attribute and the presentation
*		 attribute.  Typically, the programmer has access
*		 to only the presentation versions of these attributes.
*		 The physical attributes are set indirectly depending
*		 on the setting of KPDS_COUPLING.  See kpds_get_data
*		 for a description of how the presentation and physical
*		 attributes affect interaction with the data object.
*
*	         Other attributes are classified as either shared or
*	         unshared.  Shared attributes are stored at the
*	         physical layer of the attribute, and thus can be
*	         shared by multiple references of the data object (see
*	         kpds_reference_object for more information about
*	         references).  Unshared attributes, on the other hand,
*	         can only be used by the local object (see
*	         kpds_query_attribute for information on how to
*	         determine whether an attribute is shared or
*	         unshared).
*		
*		 This function accepts an object followed by a list of
*		 of arguments that alternate between an attribute
*		 name and an address into which the attribute's value
*		 will be stored.  The last argument on this list is
*		 a NULL which serves as a flag that indicates that no
*		 more attributes are present on the list.  
*		 Certain KPDS attributes require more than one address 
*		 in order to return the entire attribute.  For example, 
*	         KPDS_VALUE_SIZE requires five (5) arguments.  For example,
*
*		!	kpds_get_attributes(object,
*		!		KPDS_VALUE_DATA_TYPE, &typ,
*		!		KPDS_VALUE_SIZE, &w, &h, &d, &t, &e,
*		!		NULL);
*
*		 The last argument to this function must be NULL.  If
*		 it is not NULL, then the behavior of this function
*		 will be unpredictable (the NULL argument is used as a
*		 sentinal to indicate the end of the variable argument
*		 list.  If the sentinal is not present, then this
*		 function will continue to attempt to pull arguments
*		 off of the stack, until it finds a NULL).
*
*		 A complete list of the polymorphic attributes can be
*		 found in Chapter 2 of Programming Services Volume II.
*
*         Input: object    - the object from which to retrieve the
*			     specified attribute.  This must be a
*			     legal kobject that has been opened or
*			     instantiated by an appropriate kpds
*			     function call, such as kpds_open_object
*			     or kpds_reference.
*
*        Output: va_alist   - a C variable argument list that contains
*			     a set of attributes, each
*			     followed by addresses to variables
*			     that will be assigned with values
*			     associated with that attribute.
*			     The variable argument list takes the
*			     form:
*
*			     ATTRIBUTE_NAME1, &value1 [, &value2, ...],
*			     ATTRIBUTE_NAME2, &value1,[, &value2, ...],
*			     ..., NULL.
*
*			     The number of value arguments in
*			     the variable argument list for each
*			     attribute depends on the specific
*			     attribute.  For
*			     example, KPDS_VALUE_DATA_TYPE takes only
*			     one value, but KPDS_VALUE_SIZE takes
*			     multiple values.  The NULL at the
*			     end of the variable argument
*			     list serves as a flag indicating
*			     the end of the list to kdms_get_attributes.
*
*       Returns: TRUE (1) if the attribute was successfully retrieved
*	 	 and FALSE (0) otherwise.
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 30, 1993 13:44
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration:
*
*************************************************************/
int
kpds_get_attributes(
   kobject object,
   kvalist)
{
   kva_list list;
   int status;


   kpds_init();

   kva_start(list, object);

   status = kdms_vget_attributes(object, KDMS_OBJECT, &list);

   kva_end(list);

   return status;
}

/************************************************************
*
*  Routine Name: kpds_set_attribute - set the values of an 
*                                     attribute in a data object
*
*       Purpose: This function is used to assign the value of a
*	 	 attribute to a data object.
*
*		 Data Services manages two versions of some of the
*		 attributes associated with each object.  These
*		 attributes are the size and data type.  Internally,
*		 the two versions of these attributes are referred to
*		 as the physical attribute and the presentation
*		 attribute.  Typically, the programmer has access
*		 to only the presentation versions of these attributes.
*		 The physical attributes are set indirectly depending
*		 on the setting of KPDS_COUPLING.  See kpds_get_data
*		 for a description of how the presentation and physical
*		 attributes affect interaction with the data object.
*
*	         Other attributes are classified as either shared or
*	         unshared.  Shared attributes are stored at the
*	         physical layer of the attribute, and thus can be
*	         shared by multiple references of the data object (see
*	         kpds_reference_object for more information about
*	         references).  Unshared attributes, on the other hand,
*	         can only be used by the local object (see
*	         kpds_query_attribute for information on how to
*	         determine whether an attribute is shared or
*	         unshared).
*		
*		 This function accepts an object, the name of the
*		 attribute that is desired and the value of the
*		 attribute.  Certain KPDS attributes require more than
*		 one value in order to set the entire attribute.  For
*		 example, KPDS_VALUE_SIZE requires five (5) arguments.
*		 For example, setting the KPDS_VALUE_DATA_TYPE
*		 attribute might look like this:
*
*		!	kpds_set_attribute(object, KPDS_VALUE_DATA_TYPE, KFLOAT);
*
*		 Setting the KPDS_VALUE_SIZE attribute, is a little
*		 more complex:
*
*		!	kpds_set_attribute(object, KPDS_VALUE_SIZE,
*		!		100, 100, 1, 1, 1);
*
*		 A complete list of the polymorphic attributes can be
*		 found in Chapter 2 of Programming Services Volume II.
*
*         Input: object    - the object in which to set the
*			     specified attribute.  This must be a
*			     legal kobject that has been opened or
*			     instantiated by an appropriate kpds
*			     function call, such as kpds_open_object
*			     or kpds_reference.
*		 attribute - the name of the attribute to set.
*			     this is a character string that is
*			     the name of an existing attribute in
*			     the object.  There are a large number
*			     of predefined KPDS attributes.  Users
*			     can also create attributes via the
*			     kpds_create_attribute function call.
*		 va_alist   - a C variable argument list that contains
*			     value or values of the specified attribute.
*
*        Output:
*
*       Returns: TRUE (1) if the attribute was successfully set
*	 	 and FALSE (0) otherwise.
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 30, 1993 13:44
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration:
*
*************************************************************/
int
kpds_set_attribute(
   kobject object, 
   char   *attribute, 
   kvalist)
{
   kva_list list;
   int status;


   kpds_init();

   kva_start(list, attribute);

   status = kdms_vset_attribute(object, KDMS_OBJECT, attribute, &list);

   kva_end(list);

   return status;
}

/************************************************************
*
*  Routine Name: kpds_set_attributes - set the values of multiple
*                                      attributes in a data object.
*
*       Purpose: This function is used to assign the values of an 
*		 arbitrary number of attributes to a data object.
*
*		 Data Services manages two versions of some of the
*		 attributes associated with each object.  These
*		 attributes are the size and data type.  Internally,
*		 the two versions of these attributes are referred to
*		 as the physical attribute and the presentation
*		 attribute.  Typically, the programmer has access
*		 to only the presentation versions of these attributes.
*		 The physical attributes are set indirectly depending
*		 on the setting of KPDS_COUPLING.  See kpds_get_data
*		 for a description of how the presentation and physical
*		 attributes affect interaction with the data object.
*
*	         Other attributes are classified as either shared or
*	         unshared.  Shared attributes are stored at the
*	         physical layer of the attribute, and thus can be
*	         shared by multiple references of the data object (see
*	         kpds_reference_object for more information about
*	         references).  Unshared attributes, on the other hand,
*	         can only be used by the local object (see
*	         kpds_query_attribute for information on how to
*	         determine whether an attribute is shared or
*	         unshared).
*		
*		 This function accepts an object followed by a list of
*		 of arguments that alternate between an attribute
*		 name and values that the specified attribute is to
*		 be set to.  The last argument on this list is
*		 a NULL which serves as a flag that indicates that no
*		 more attributes are present on the list.  
*		 Certain KPDS attributes require more than one value
*		 in order to return the entire attribute.  For example, 
*	         KPDS_VALUE_SIZE requires five (5) arguments.  For example,
*
*		!	kpds_set_attributes(object,
*		!		KPDS_VALUE_DATA_TYPE, KFLOAT,
*		!		KPDS_VALUE_SIZE, 100, 100, 1, 1, 1,
*		!		NULL);
*
*		 The last argument to this function must be NULL.  If
*		 it is not NULL, then the behavior of this function
*		 will be unpredictable (the NULL argument is used as a
*		 sentinal to indicate the end of the variable argument
*		 list.  If the sentinal is not present, then this
*		 function will continue to attempt to pull arguments
*		 off of the stack, until it finds a NULL).
*
*		 A complete list of the polymorphic attributes can be
*		 found in Chapter 2 of Programming Services Volume II.
*
*         Input: object    - the object in which to set the
*			     specified attribute.  This must be a
*			     legal kobject that has been opened or
*			     instantiated by an appropriate kpds
*			     function call, such as kpds_open_object
*			     or kpds_reference.
*
*        Output: va_alist  - a C variable argument list that contains
*			     a set of attributes, each
*			     followed by values will be assigned to
*			     the specified attribute.
*			     The variable argument list takes the
*			     form:
*
*			     ATTRIBUTE_NAME1, value1 [, value2, ...],
*			     ATTRIBUTE_NAME2, value1,[, value2, ...],
*			     ..., NULL.
*
*			     The number of value arguments in
*			     the variable argument list for each
*			     attribute depends on the specific
*			     attribute.  For
*			     example, KPDS_VALUE_DATA_TYPE takes only
*			     one value, but KPDS_VALUE_SIZE takes
*			     multiple values.  The NULL at the
*			     end of the variable argument
*			     list serves as a flag indicating
*			     the end of the list to kdms_set_attributes.
*
*       Returns: TRUE (1) if the attribute was successfully retrieved
*	 	 and FALSE (0) otherwise.
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 30, 1993 13:44
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration:
*
*************************************************************/
int
kpds_set_attributes(
   kobject object,
   kvalist)
{
   kva_list list;
   int status;


   kpds_init();

   kva_start(list, object);

   status = kdms_vset_attributes(object, KDMS_OBJECT, &list);

   kva_end(list);

   return status;
}

/************************************************************
*
*  Routine Name: kpds_match_attribute - returns TRUE if the
*		 same attribute in two objects match.
*
*       Purpose: 
*		 The purpose of this routine is to allow
*		 the programmer to compare a single in two data
*		 objects.
*
*		 Data Services manages two versions of some of the
*		 attributes associated with each object.  These
*		 attributes are the size and data type.  Internally,
*		 the two versions of these attributes are referred to
*		 as the physical attribute and the presentation
*		 attribute.  Typically, the programmer has access
*		 to only the presentation versions of these attributes.
*		 The physical attributes are set indirectly depending
*		 on the setting of KPDS_COUPLING.  See kpds_get_data
*		 for a description of how the presentation and physical
*		 attributes affect interaction with the data object.
*
*	         Other attributes are classified as either shared or
*	         unshared.  Shared attributes are stored at the
*	         physical layer of the attribute, and thus can be
*	         shared by multiple references of the data object (see
*	         kpds_reference_object for more information about
*	         references).  Unshared attributes, on the other hand,
*	         can only be used by the local object (see
*	         kpds_query_attribute for information on how to
*	         determine whether an attribute is shared or
*	         unshared).
*		
*		 This routine will return TRUE if the specified
*		 attribute has the same value in both of the objects.
*		 This routine will return FALSE if the attribute does
*		 not have the same value in both of of the
*		 objects. kpds_match_attribute will also return FALSE
*		 if the attribute does not exist in either or both of
*		 the objects.
*
*		 A complete list of the polymorphic attributes can be
*		 found in Chapter 2 of Programming Services Volume II.
*
*        Input:   object1 - the first object on
*			    which to match the specified attribute
*		  object2 - the second  object on
*			    which to match the specified attribute
*		  attribute - the attribute that will be
*			      compared in the two objects.
*
*        Output: none
*       Returns: There are three ways for this routine to
*		 return a FALSE:  (1) if the attribute
*		 in the two objects does not match; (2) if
*		 either object does not contain the specified 
*		 attribute; (3) an error condition
*		 resulting from an invalid object or segment.
*		 If none of these three conditions exist, then
*		 this function will return TRUE.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley & John Salas
*          Date: Oct 18, 1993 23:23
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kpds_match_attribute(
   kobject object1, 
   kobject object2, 
   char *attr)
{
   kpds_init();
   return kdms_match_attribute(object1, object2, KDMS_OBJECT, attr);
}

/************************************************************
*
*  Routine Name: kpds_match_attributes - returns true if the
*		 list of segment attributes in two
*		 objects match.
*
*       Purpose: 
*		 The purpose of this routine is to allow
*		 the programmer to compare multiple attributes in two
*		 object.
*
*		 Data Services manages two versions of some of the
*		 attributes associated with each object.  These
*		 attributes are the size and data type.  Internally,
*		 the two versions of these attributes are referred to
*		 as the physical attribute and the presentation
*		 attribute.  Typically, the programmer has access
*		 to only the presentation versions of these attributes.
*		 The physical attributes are set indirectly depending
*		 on the setting of KPDS_COUPLING.  See kpds_get_data
*		 for a description of how the presentation and physical
*		 attributes affect interaction with the data object.
*
*	         Other attributes are classified as either shared or
*	         unshared.  Shared attributes are stored at the
*	         physical layer of the attribute, and thus can be
*	         shared by multiple references of the data object (see
*	         kpds_reference_object for more information about
*	         references).  Unshared attributes, on the other hand,
*	         can only be used by the local object (see
*	         kpds_query_attribute for information on how to
*	         determine whether an attribute is shared or
*	         unshared).
*		
*		 This routine will return TRUE if all of the specified
*		 attributes have the same value in the objects.  This
*		 routine will return FALSE if any of the attributes do
*		 not match kpds_match_attributes will also return
*		 FALSE if any of the attributes do not exist in either
*		 or both of the two objects.
*
*		 The last argument to this function must be NULL.  If
*		 it is not NULL, then the behavior of this function
*		 will be unpredictable (the NULL argument is used as a
*		 sentinal to indicate the end of the variable argument
*		 list.  If the sentinal is not present, then this
*		 function will continue to attempt to pull arguments
*		 off of the stack, until it finds a NULL).
*
*		 A complete list of the polymorphic attributes can be
*		 found in Chapter 2 of Programming Services Volume II.
*
*        Input:   object1 - the first  object on
*			    which to match the specified attributes
*		  object2 - the second  object on
*			    which to match the specified attributes
*		  va_alist - variable argument list, that
*			     contains an arbitrarily long
*			     list of attributes followed
*			     a NULL.  It takes the form:
*
*			     ATTRIBUTE_NAME1, ATTRIBUTE_NAME2, ..., NULL
*
*        Output: none
*       Returns: There are three ways for this routine to
*		 return a FALSE:  (1) if any of the attributes
*		 between the two objects do not match; (2) if
*		 either object does not contain one or more of
*		 the specified attributes; (3) an error condition
*		 resulting from an invalid object or segment.
*		 If none of these three conditions exist, then
*		 this function will return TRUE.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley & John Salas
*          Date: Oct 18, 1993 23:30
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kpds_match_attributes(
   kobject object1, 
   kobject object2, 
   kvalist)
{
   kva_list list;
   int status;


   kpds_init();

   kva_start(list, object2);

   status = kdms_vmatch_attributes(object1, object2, KDMS_OBJECT, &list);

   kva_end(list);

   return status;
}

/************************************************************
*
*  Routine Name: kpds_copy_attribute - copy an attribute from one
*		 object to another
*
*       Purpose: 
*		 This function is used to copy a single 
*		 attribute from one object to another object.
*
*		 Data Services manages two versions of some of the
*		 attributes associated with each object.  These
*		 attributes are the size and data type.  Internally,
*		 the two versions of these attributes are referred to
*		 as the physical attribute and the presentation
*		 attribute.  Typically, the programmer has access
*		 to only the presentation versions of these attributes.
*		 The physical attributes are set indirectly depending
*		 on the setting of KPDS_COUPLING.  See kpds_get_data
*		 for a description of how the presentation and physical
*		 attributes affect interaction with the data object.
*
*	         Other attributes are classified as either shared or
*	         unshared.  Shared attributes are stored at the
*	         physical layer of the attribute, and thus can be
*	         shared by multiple references of the data object (see
*	         kpds_reference_object for more information about
*	         references).  Unshared attributes, on the other hand,
*	         can only be used by the local object (see
*	         kpds_query_attribute for information on how to
*	         determine whether an attribute is shared or
*	         unshared).
*
*		 This function accepts a source object, a
*		 destination object, and an attribute name.
*		 If the attribute exists in the source object,
*		 then it will be copied to the destination object.
*		 If the attribute does not exist in the source
*		 object, then an error condition is returned.
*		
*		 A complete list of the polymorphic attributes can be
*		 found in Chapter 2 of Programming Services Volume II.
*
*         Input: object1     - the object to copy from
*		 object2     - the object to copy to
*		 attribute   - the attribute to copy
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*
*    Written By: Steve Kubica
*          Date: May 22, 1994 15:11
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kpds_copy_attribute(
   kobject object1, 
   kobject object2, 
   char   *attribute)
{
   kpds_init();
   return (kdms_copy_attribute(object1, object2, KDMS_OBJECT, attribute));
}

/************************************************************
*
*  Routine Name: kpds_copy_attributes - copy multiple attributes
*		 from one object to another.
*
*       Purpose: 
*		 This function is used to copy multiple
*		 attributes from one object to another.  The
*		 attributes should be provided in a NULL terminated
*		 variable argument list.
*
*		 Data Services manages two versions of some of the
*		 attributes associated with each object.  These
*		 attributes are the size and data type.  Internally,
*		 the two versions of these attributes are referred to
*		 as the physical attribute and the presentation
*		 attribute.  Typically, the programmer has access
*		 to only the presentation versions of these attributes.
*		 The physical attributes are set indirectly depending
*		 on the setting of KPDS_COUPLING.  See kpds_get_data
*		 for a description of how the presentation and physical
*		 attributes affect interaction with the data object.
*
*	         Other attributes are classified as either shared or
*	         unshared.  Shared attributes are stored at the
*	         physical layer of the attribute, and thus can be
*	         shared by multiple references of the data object (see
*	         kpds_reference_object for more information about
*	         references).  Unshared attributes, on the other hand,
*	         can only be used by the local object (see
*	         kpds_query_attribute for information on how to
*	         determine whether an attribute is shared or
*	         unshared).
*
*		 This function accepts a source object, a destination
*		 object, and a list of attribute names.  The last
*		 argument to this function must be NULL.  If it is not
*		 NULL, then the behavior of this function will be
*		 unpredictable (the NULL argument is used as a
*		 sentinal to indicate the end of the variable argument
*		 list.  If the sentinal is not present, then this
*		 function will continue to attempt to pull arguments
*		 off of the stack, until it finds a NULL).  If each of
*		 the attributes exist in the source object, then it
*		 will be copied to the destination object.  If an
*		 attribute does not exist in the source object, then
*		 an error condition is returned.  In the event that an
*		 attribute does not exist, then the remainder of the
*		 attributes on the list will not be copied.
*
*		 A complete list of the polymorphic attributes can be
*		 found in Chapter 2 of Programming Services Volume II.
*
*         Input: object1     - the object to copy from
*		 object2     - the object to copy to
*		 va_alist    - NULL terminated list of attribute names to
*			       copy.
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*
*    Written By: Steve Kubica
*          Date:
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kpds_copy_attributes(
   kobject object1, 
   kobject object2, 
   kvalist)
{
   kva_list list;
   int status;


   kpds_init();

   kva_start(list, object2);

   status = kdms_vcopy_attributes(object1, object2, KDMS_OBJECT, &list);

   kva_end(list);

   return status;
}

/************************************************************
*
*  Routine Name: kpds_query_attribute - get information about an attribute
*
*       Purpose: 
*		 This function is used for two purposes: (1) to
*		 determine the existence of an attribute; and (2) to
*		 obtain the characteristics of the attribute.
*
*		 Data Services manages two versions of some of the
*		 attributes associated with each object.  These
*		 attributes are the size and data type.  Internally,
*		 the two versions of these attributes are referred to
*		 as the physical attribute and the presentation
*		 attribute.  Typically, the programmer has access
*		 to only the presentation versions of these attributes.
*		 The physical attributes are set indirectly depending
*		 on the setting of KPDS_COUPLING.  See kpds_get_data
*		 for a description of how the presentation and physical
*		 attributes affect interaction with the data object.
*
*	         Other attributes are classified as either shared or
*	         unshared.  Shared attributes are stored at the
*	         physical layer of the attribute, and thus can be
*	         shared by multiple references of the data object (see
*	         kpds_reference_object for more information about
*	         references).  Unshared attributes, on the other hand,
*	         can only be used by the local object.
*
*		 The difference between shared and unshared attributes
*		 is abstracted from the user at the PDS level.  The
*		 permanent attributes are generally shared, and the
*		 non-permanent attributes are generally non-shared.
*		 Permanent attributes are attributes that will be stored
*		 as part of an output object when it is written.  Any
*		 attributes that are retrieved when an object is opened
*		 are also permanent attributes.  Non-permanent attributes
*		 exist only while the program that is operating on the
*		 object is executing.
*
*		 The datatype argument indicates what kind of
*		 information is stored in the attribute. Attributes
*		 can be one of the following data types: KBYTE,
*		 KUBYTE, KSHORT, KUSHORT, KINT, KUINT, KLONG, KULONG,
*		 KFLOAT, KDOUBLE, KCOMPLEX, or KDCOMPLEX.  
*
*		 The num_args argument indicates how many arguments
*		 must be passed in an argument list to one of the
*		 attribute functions.
*
*		 The size arguments indicates the number of
*		 units of the data type there are in each argument.
*		 This argument allows arrays of information to be
*		 stored as attributes.
*
*         Input: object      - the object with the attribute
*		 segment     - the segment that the attribute
*			       is stored in.  If this argument
*			       is NULL, then the attribute is
*			       global to the object.
*		 name        - name of the attribute to be
*			       queried.
*		 datatype    - datatype of the attribute
*		 num_args    - number of arguments in this attribute
*		 size        - size of each argument in this
*		 	       attribute.
*		 permanent   - is the attribute stored or
*			       transient?  The return value
*			       will be either TRUE or FALSE
*
*       Returns: TRUE (1) if attribute exists, FALSE (0) otherwise
*
*  Restrictions: 
*
*    Written By: Steve Kubica
*          Date: May 22, 1994 15:10
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kpds_query_attribute(
   kobject object,
   char   *attribute,
   int    *num_args,
   int    *arg_size,
   int    *data_type,
   int    *permanent)
{
   kpds_init();
   return (kdms_query_attribute(object, NULL, attribute, num_args, arg_size,
				data_type, permanent));
}


/************************************************************
*
*  Routine Name: kpds_print_attribute - print the value of an
*					attribute from a
*					data object.
*
*       Purpose: This function is used to print the value of
*		 an  attribute from a data object to an output
*		 file.
*
*		 This function is typically used by such programs as
*		 kprdata to print out the values of attributes in
*		 an object.
*
*         Input: object     - the object containing the attribute
*		 attribute  - the attribute to print
*		 printfile  - the open kfile to print to
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*
*    Written By: Steve Kubica
*          Date: May 22, 1994 15:15
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kpds_print_attribute(
   kobject object,
   char   *attribute,
   kfile  *printfile)
{
   kpds_init();
   return (kdms_print_attribute(object, NULL, attribute, printfile));
}


/************************************************************
*
*  Routine Name: kpds_get_data - retrieve data from
*		 a segment in a data object.
*
*       Purpose: 
*		 This function is used to retrieve a unit of 
*		 data, which is referred to as a "primitive", 
*		 from a data object.
*
*		 The first argument is the data object that
*		 will be accessed in order to retrieve the
*		 data.  
*
*		 The second argument is the data primitive that is 
*		 desired.  Data services understands a number of
*		 primitives that allow access to different parts of
*		 a data set.  Except for the vector primitives
*		 (KPDS_VALUE_VECTOR, KPDS_MAP_VECTOR, KPDS_TIME_VECTOR,
*		 KPDS_LOCATION_VECTOR), the 
*		 interpretation of all of these primitives depends on 
*		 the attribute index order primitives (KPDS_VALUE_INDEX_ORDER,
*		 KPDS_MAP_INDEX_ORDER, etc).
*		 The index order attributes determine how data is
*		 retrieved and stored.  Successive calls to kpds_get_data
*		 cause an automatic increment of the position of each
*		 part of the data set.  How the position is incremented depends
*		 on the primitive that is being retrieved and the index
*		 order.  For example, if a KPDS_VALUE_LINE is being retrieved,
*		 and the index order is KWIDTH, KHEIGHT, KDEPTH, ...,
*		 then the line will be defined in the width direction
*		 and successive lines can be indexed by incrementing
*		 the height dimension until it wraps around, then
*		 incrementing the depth dimension, and so on.  Below
*		 is a list of the types of primitives that are available:
*		 
*		.RS
*	 	.IP "point" 8
*			  specifies that a single value will be returned.
*	 		  Successive calls to kpds_get_data will result
*	 		  in adjacent points being returned, as 
*	 		  described above.  An example of a point primitive
*	 		  is the KPDS_VALUE_POINT primitive.
*	 		  
*	 	.IP "line" 8
*			  specifies that a one-dimensional unit of data 
*	 		  will be returned.  The direction in which
*	 		  a line is defined is given by the first (or
*	 		  lowest order) value of the index order.  In
*	 		  the above example, KWIDTH was the first value
*	 		  in the index order attribute.  An example of a 
*	 		  line primitive is the KPDS_VALUE_LINE primitive.
*	 	
*	 	.IP "plane" 8
*			  specifies that a two-dimensional unit of data
*	 		  will be returned.  The plane is defined along
*	 		  the two lowest order dimensions in the index
*	 		  order for the segment specified.  An example of a 
*	 		  plane primitive is the KPDS_VALUE_PLANE primitive.
*	 
*	 	.IP "volume" 8
*			  specifies that a three-dimensional unit of data
*	 		  will be returned.  The volume is defined along
*	 		  the three lowest order dimensions in the
*	 		  index order for the segment specified.   An example 
*	 		  of a volume primitive is the KPDS_VALUE_VOLUME 
*	 		  primitive.
*	 
*	 	.IP "region" 8
*			  specifies that a n-dimensional unit of data
*	 		  will be returned.  The n varies from segment
*	 		  to segment and is based on the dimensionality
*	 		  of the segment being accessed.  For example,
*	 		  Data Services considers the Value segment
*	 		  to be a five-dimensional segment, so n is five
*	 		  for the value segment.  A region front, upper
*	 		  corner is specified by the current position
*	 		  (for the Value segment, this is a five-tuple).
*	 		  The size of the region is given by the 
*	 		  attribute KPDS_REGION_SIZE, which must be
*	 		  set prior to using this primitive.  An example of a 
*	 		  region primitive is the KPDS_VALUE_REGION primitive.
*	 		  
*	 	.IP "all" 8
*			  specifies that all data for the specified segment
*	 		  should be retrieved and returned.  An example of an
*	 		  all primitive is the KPDS_VALUE_ALL primitive.
*	 
*	 	.IP "vector" 8
*			 specifies that a one-dimensional unit of data
*	 		  will be returned.  The dimension in which
*	 		  this unit of data is defined is fixed for each
*	 		  segment.  Thus, regardless of the index order,
*	 		  the same dimension will be used to obtain
*	 		  the vector of data.  The only thing that is
*	 		  impacted by index order when retrieving vectors
*	 		  is the actual order in which the vectors 
*	 		  in a data set are retrieved due to the
*	 		  auto-incrementing of position.  An example of a 
*	 		  vector primitive is the KPDS_VALUE_VECTOR primitive.
*	 		  For the five predefined part of a data set, each has
*	 		  a different dimension that specifies a vector.
*	 		  Below is a list of the data components and their
*	 		  associated vector definition:
*		
*	 	.TS
*		center;
*		c c.
*				\fBSegment\fP	\fBVector Definition\fP
*				_
*	 			VALUE	KELEMENTS
*	 			MASK	KELEMENTS
*	 			MAP	KWIDTH
*	 			LOCATION	KDIMENSION
*	 			TIME	KTIME
*		.TE
*		.RE
*
*                The third argument, "data", serves as both
*                an input and an output argument.  As input,
*                it dictates whether kpds_get_data must
*                allocate space sufficient for the operation.
*                If the argument is NULL, then memory will
*                be allocated to store the data primitive
*                requested.  A pointer to that memory will
*                be returned.  If this argument is not NULL,
*                then kpds_get_data assumes that the "data"
*                argument is a pointer to a sufficient
*                amount of memory with the correct
*                dimensionality for the primitive (no memory
*                allocation occurs).  In this case, if this
*                routine returns successfully, then the
*                return value is the pointer "data".
*
*         Input: object    - the  object that
*			     will serve as a source for the
*			     data requested.
*		 primitive - a description of the unit of
*                            data desired.
*                data      - a pointer to the region of
*                            memory that will serve as a
*                            destination for the data. If
*                            this value is NULL, then
*                            sufficient space for this
*                            operation will be allocated
*                            for this operation.  The data
*                            type kaddr is used because
*                            it indicates a generic data
*                            pointer.
*
*        Output: If "data" is not initially NULL, then the memory
*		 that it points to will be filled with the data
*		 that is being requested.
*       Returns: If "data" is not initially NULL, then the data
*                space pointed to by "data" will be returned on
*                success.  If the "data" argument is NULL, then
*                a new pointer to the requested data will be
*                returned.  Unsuccessful calls to this routine
*                are indicated by a return value of NULL.
*
*  Restrictions: This routine assumes that if the argument
*                "data" is not NULL, then it contains the
*                appropriate amount of memory with the
*                appropriate dimensionality for the
*                requested primitive.
*    Written By: Jeremy Worley & John Salas
*          Date: Oct 18, 1993 23:32
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
kaddr
kpds_get_data(
   kobject object,
   char   *primitive,
   kaddr   data)
{
   int prim = _kpds_lookup_primitive(primitive);

   /* initialize polymorphic data services if necessary */
   kpds_init();

   /* verify that we have a valid primitive */
   if (prim < 0)
      return FALSE;		/* errno already set */

   /* get the data */
   if (kdms_query_segment(object, kpds_primitives[prim].segment))
      return (kpds_primitives[prim].get(object, &kpds_primitives[prim], data));
   else
      return NULL;
}

/************************************************************
*
*  Routine Name: kpds_put_data - store a unit of data
*		 to a segment in a data object.
*
*       Purpose: 
*		 This function is used to store data in a data 
*		 object.
*
*		 The first argument is the data object that
*		 will be accessed in order to put the data.  
*
*		 The second argument is the data primitive that is 
*		 to be written.  Data services understands a number of
*		 primitives that allow access to different parts of
*		 a data set.  Except for the vector primitives
*		 (KPDS_VALUE_VECTOR, KPDS_MAP_VECTOR, KPDS_TIME_VECTOR,
*		 KPDS_LOCATION_VECTOR), the 
*		 interpretation of all of these primitives depends on 
*		 the attribute index order primitives (KPDS_VALUE_INDEX_ORDER,
*		 KPDS_MAP_INDEX_ORDER, etc).
*		 The index order attributes determine how data is
*		 retrieved and stored.  Successive calls to kpds_get_data
*		 cause an automatic increment of the position of each
*		 part of the data set.  How the position is incremented depends
*		 on the primitive that is being retrieved and the index
*		 order.  For example, if a KPDS_VALUE_LINE is being stored,
*		 and the index order is KWIDTH, KHEIGHT, KDEPTH, ...,
*		 then the line will be defined in the width direction
*		 and successive lines can be indexed by incrementing
*		 the height dimension until it wraps around, then
*		 incrementing the depth dimension, and so on.  Below
*		 is a list of the types of primitives that are available:
*		
*		.RS
*	 	.IP "point" 8
*			  specifies that a single value will be stored.
*	 		  Successive calls to kpds_put_data will result
*	 		  in adjacent points being stored, as 
*	 		  described above.  An example of a point primitive
*	 		  is the KPDS_VALUE_POINT primitive.
*	 		  
*	 	.IP "line" 8
*			  specifies that a one-dimensional unit of data 
*	 		  will be returned.  The direction in which
*	 		  a line is defined is given by the first (or
*	 		  lowest order) value of the index order.  In
*	 		  the above example, KWIDTH was the first value
*	 		  in the index order attribute.  An example of a 
*	 		  line primitive is the KPDS_VALUE_LINE primitive.
*	 	
*	 	.IP "plane" 8
*			  specifies that a two-dimensional unit of data
*	 		  will be returned.  The plane is defined along
*	 		  the two lowest order dimensions in the index
*	 		  order for the segment specified.  An example of a 
*	 		  plane primitive is the KPDS_VALUE_PLANE primitive.
*	 
*	 	.IP "volume" 8
*			  specifies that a three-dimensional unit of data
*	 		  will be returned.  The volume is defined along
*	 		  the three lowest order dimensions in the
*	 		  index order for the segment specified.   An example 
*	 		  of a volume primitive is the KPDS_VALUE_VOLUME 
*	 		  primitive.
*	 
*	 	.IP "region" 8
*			  specifies that a n-dimensional unit of data
*	 		  will be returned.  The n varies from segment
*	 		  to segment and is based on the dimensionality
*	 		  of the segment being accessed.  For example,
*	 		  Data Services considers the Value segment
*	 		  to be a five-dimensional segment, so n is five
*	 		  for the value segment.  A region front, upper
*	 		  corner is specified by the current position
*	 		  (for the Value segment, this is a five-tuple).
*	 		  The size of the region is given by the 
*	 		  attribute KPDS_REGION_SIZE, which must be
*	 		  set prior to using this primitive.  An example of a 
*	 		  region primitive is the KPDS_VALUE_REGION primitive.
*	 		  
*	 	.IP "all" 8
*			  specifies that all data for the specified segment
*	 		  should be retrieved and returned.  An example of an
*	 		  all primitive is the KPDS_VALUE_ALL primitive.
*	 
*	 	.IP "vector" 8
*			  specifies that a one-dimensional unit of data
*	 		  will be returned.  The dimension in which
*	 		  this unit of data is defined is fixed for each
*	 		  segment.  Thus, regardless of the index order,
*	 		  the same dimension will be used to obtain
*	 		  the vector of data.  The only thing that is
*	 		  impacted by index order when retrieving vectors
*	 		  is the actual order in which the vectors 
*	 		  in a data set are retrieved due to the
*	 		  auto-incrementing of position.  An example of a 
*	 		  vector primitive is the KPDS_VALUE_VECTOR primitive.
*	 		  For the five predefined part of a data set, each has
*	 		  a different dimension that specifies a vector.
*	 		  Below is a list of the data components and their
*	 		  associated vector definition:
*	 
*	 	.TS
*		center;
*		c c.
*				\fBSegment\fP	\fBVector Definition\fP
*				_
*	 			VALUE	KELEMENTS
*	 			MASK	KELEMENTS
*	 			MAP	KWIDTH
*	 			LOCATION	KDIMENSION
*	 			TIME	KTIME
*		.TE
*		.RE
*
*		The fourth argument is the data to be written.
*		This must be a non-NULL pointer to valid data of
*		the appropriate size (as defined, for example, 
*		by the KPDS_VALUE_SIZE attribute)
*		and data type (as defined, for example,  by the 
*		KPDS_VALUE_DATA_TYPE attribute).
*
*         Input: object    - the data object that
*			     will serve as a destination for
*			     the data.
*		 primitive - a description of the unit of
*                            data in the argument "data".
*                data      - a pointer to the region of
*                            memory that will serve as a
*                            source for the data.
*        Output:
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley & John Salas
*          Date: Nov 23, 1992 22:42
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kpds_put_data(
   kobject object,
   char   *primitive,
   kaddr   data)
{
   int prim = _kpds_lookup_primitive(primitive);

   /* initialize polymorphic data services if necessary */
   kpds_init();

   /* verify that we have a valid primitive */
   if (prim < 0)
      return FALSE;		/* errno already set */

   /* write the data */
   if (kdms_query_segment(object, kpds_primitives[prim].segment))
      return (kpds_primitives[prim].put(object, &kpds_primitives[prim], data));
   else
      return FALSE;
}

/************************************************************
*
*  Routine Name: kpds_open_object - create an object associated
*                with an input or output transport.
*
*       Purpose: This function is used to
*                instantiate a data object (kobject) that
*                is associated with a permanent file or
*                transport.  If a permanent file is not
*                desired (i.e. the object is going to be
*                used as temporary storage, and will not
*                be used by any other process) then the
*                kpds_create_object function call should be used
*                instead.
*
*                The first argument to this function is
*                the transport or file name.  This argument
*                indicates the name of the transport that
*                is associated with the object.  The transport
*                name can be any legal khoros transport
*                description.  While this will usually be a
*                regular UNIX file name, it is also possible
*                to specify such things as shared memory pointers,
*                memory map pointers, sockets, streams, and
*                even transports on other machines.  For more
*                information on the syntax of a Khoros transport
*                name, refer to the online man page for the
*                Khoros function kopen.
*
*                The second argument to the kpds_open_object function
*                call, flags, is used to provide polymorphic services with
*                specific information about how the object
*                is going to be manipulated.  The flags argument
*                is analogous to kopen's flags argument.
*                The flags argument is constructed by bitwise OR'ing
*                predefined values from the following list:
*
*		.RS
*       	.IP "KOBJ_READ" 10
*				Open an existing file or transport
*                               for reading (input).  By using this flag,
*                               you are indicating that the file or transport
*                               exists and that it contains valid data.
*                               If it does not exist, or the data is not
*                               recognized, then an error message will
*                               be generated and this function will
*                               return KOBJECT_INVALID.
*        
*        	.IP "KOBJ_WRITE" 10
*				Open a file or transport for writing
*                               (output).  By using this flag, you are
*                               indicating that any data that you write
*                               to the object will be stored in the
*                               file or transport specified.
*        
*        	.IP "KOBJ_STREAM" 10
*				If the transport is a stream then polymorphic
*	                        services does not attempt to buffer the
*	                        data.  By setting this value, you are
*	                        indicating that that process reads and/or
*	                        writes the data strictly sequentially from the
*                               object in its native index order, and
*                               it is not necessary to buffer the data
*                               in a temporary transport that can be
*                               accessed randomly.
*        
*        	.IP "KOBJ_RAW" 10
*				When an object is opened,  data
*	 			services attempts to recognize the file
*	 			format by examining the first part of the
*	 			file (typically there are "magic number"
*	 			or identifiers that make this sort of
*	 			recognition trivial).  If it cannot
*	 			determine the file format, then the open
*	 			operation will fail, unless KOBJ_RAW
*	 			is set.  In other words, if KOBJ_RAW 
*	 			is set, then data services will assume that 
*	 			if it cannot recognize the format, then the
*	 			file contains raw unformatted data.
*		.RE
*
* 		 These flags can be combined arbitrarily to determine
*		 how the file or transport is opened and subsequently
*		 manipulated.  This is done by bitwise OR'ing any combination
*		 of these options.  For example KOBJ_READ | KOBJ_WRITE
*                will result in a read/write file object.
*                This implies that the file already exists and will be
*                read from using kpds_get_data and written to using
*                kpds_put_data.  When kpds_close is called, the changes
*                that are a result of calls to kpds_put_data will be
*                stored to the file or transport.
*
*                However, if you intend to open an output object, but
*                you need to occasionally read data from it that you
*                have already written, it is not necessary to specify
*                KOBJ_READ (in fact, doing so may result result in an
*                error if the file or transport does not already exist).
*
*                Likewise, it is possible to call kpds_put_data on
*                an input object (one which was opened without the
*                KOBJ_WRITE flag).  If this is done, then subsequent
*                calls to kpds_get_data on a region that has been
*                written to will contain the new data.  However,
* 		 the file or transport that is associated with this input 
* 		 object will not be changed.  Thus, the KOBJ_READ and 
*		 KOBJ_WRITE flags only indicate what operations are allowed 
*		 on the permanent file or transport that is associated with 
*		 the object, not what operations are allowable on the object 
*		 itself.
*
*                If KOBJ_READ is specified, then the Data Services
*                will attempt to recognize the file format automatically.
*                If it fails, then this function will return KOBJECT_INVALID,
*                indicating that it was unable to open the object,
*                unless the KOBJ_RAW flag was also specified,
*                in which case, it will assume that the input file
*                is simply raw data.  The structured file formats
*                that are currently recognized are VIFF (The Khoros
*                2.0 standard file format), Xvimage (The Khoros 1.0
*                standard file format, which was referred to as VIFF
*                in Khoros 1.0), Pnm (Portable Any Map, which includes
*                PBM, PGM, and PNM), and Sun Raster.
*
*                The default index order will be set to
*                KWIDTH, KHEIGHT, KDEPTH, KTIME and KELEMENTS for
*                the value and mask data,  to KMAP_WIDTH, KMAP_HEIGHT,
*                KMAP_ELEMENTS, KMAP_DEPTH and KMAP_TIME for the
*                map data and KWIDTH, KHEIGHT, KDEPTH and KDIMENSION
*                for the location data.  The only way to get the
*		 index order to reflect the stored index order of
*		 the data is to call kpds_sync_object.  See the
*		 man page for kpds_sync_object for more information.
*
*         Input: name - a string that contains the path name
*                       of a file or transport that will be
*                       associated with the object.
*
*                flags - how the object is to be opened.
*                        A bitwise OR of KOBJ_READ, KOBJ_WRITE,
*                        KOBJ_STREAM and KOBJ_RAW as
*                        described above.
*
*        Output: 
*       Returns: kobject on success, KOBJECT_INVALID upon failure
*
*
*  Restrictions: The KOBJ_RAW flag will have unpredictable
*                results if it is combined with the KOBJ_WRITE flag.
*                This limitation will be removed in a later release of
*                the Khoros 2.0 system.
*
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 16, 1993 17:22
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: kobject kpds_open_object(
*		!   char *name,
*		!   int flags)
*
*************************************************************/
kobject
kpds_open_object(
   char *name, 
   int   flags)
{
   kobject tmp;
   char buf[KLENGTH];
   int vorder[5] = {KWIDTH, KHEIGHT, KDEPTH, KTIME, KELEMENTS};
   int morder[5] = {KMAP_WIDTH, KMAP_HEIGHT, KMAP_DEPTH, KMAP_TIME,
		    KMAP_ELEMENTS};
   int lorder[4] = {KWIDTH, KHEIGHT, KDEPTH, KDIMENSION};

   kpds_init();

   tmp = kdms_open(name, flags);

   if (tmp == NULL)
      return KOBJECT_INVALID;

   /*
    * set the date of the object, if it is an output object
    */
   if (flags & KOBJ_WRITE)
   {
      kaps_current_date(buf);
      kdms_set_attribute(tmp, KDMS_OBJECT, KDMS_DATE, buf);
   }

   /*
    * initialize the mask segment's pad value to 1 instead of zero.
    * Also set the index order...see below.
    */
   if (kdms_query_segment(tmp, KDMS_SEGMENT_MASK))
      kdms_set_attributes(tmp, KDMS_SEGMENT_MASK, 
			       KDMS_PAD_VALUE, 1.0, 0.0,
			       KDMS_INDEX_ORDER, vorder, NULL);

   /*
    * This little bit is here to ensure that the proper index order is
    * initialized when reading in a data set.
    */
   if (kpds_query_value(tmp))
      kdms_set_attribute(tmp, KDMS_SEGMENT_VALUE, KDMS_INDEX_ORDER, vorder);

   if (kpds_query_map(tmp))
      kdms_set_attribute(tmp, KDMS_SEGMENT_MAP, KDMS_INDEX_ORDER, morder);

   if (kpds_query_location(tmp))
      kdms_set_attribute(tmp, KDMS_SEGMENT_LOCATION, KDMS_INDEX_ORDER, 
			 lorder);

   return tmp;
}

/************************************************************
*
*  Routine Name: kpds_reference_object - create a reference of a
*		 data object.
*
*       Purpose: 
*		 This function is used to create a reference
*		 object for a data object that can be treated as a second
*		 independent data object under most circumstances.
*		 A referenced object is similar conceptually to a 
*		 symbolic link in a UNIX file system in most 
*		 respects.  For example, getting data from an
*		 input object and a reference of the object will
*		 result in the same data.  Data that is put on an
*		 output object can then be accessed from any of that 
*		 object's references.
*
*		 The similarity ends there.  Once an object is
*		 referenced, the two resulting objects are 
*		 equivalent--there is no way to distinguish which was
*		 the original.  In fact, closing the original does
*		 not in any way affect the reference, and visa-versa.
*
*		 kpds_reference_object creates a new object that has
*		 presentation attributes that are independent
*		 of the original object's presentation attributes.
*                The presentation attributes are UNCOUPLED from
*                the physical attributes, see the description
*                found in Table 4 in Chapter 6 of the the Khoros Programmer's
*                Manual on the KPDS_COUPLING attribute for more
*                information.  The two objects (or more if there
*		 are several calls to kpds_reference_object) share all
*		 physical data.
*
*                The default index order will be set to
*                KWIDTH, KHEIGHT, KDEPTH, KTIME and KELEMENTS for
*                the value and mask data,  to KMAP_WIDTH, KMAP_HEIGHT,
*                KMAP_ELEMENTS, KMAP_DEPTH and KMAP_TIME for the
*                map data and KWIDTH, KHEIGHT, KDEPTH and KDIMENSION
*                for the location data.
*
*         Input: object - the object to be
*		          referenced.
*
*        Output: 
*       Returns: a kobject that is a reference of the input
*		 object on success, KOBJECT_INVALID upon failure
*
*  Restrictions: 
*
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 16, 1993 17:23
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: kobject kpds_reference_object(
*		!   kobject object)
*
*************************************************************/
kobject
kpds_reference_object(kobject object)
{
   kpds_init();
   
   return kdms_reference(object);
}

/************************************************************
*
*  Routine Name: kpds_copy_global_attr - copy all global
*		 attributes from one data object to another.
*
*       Purpose: This function copies all the
*		 object level attributes from the source_object
*		 to the destination_object.  None of the segment
*		 related attributes will be copied over.
*
*         Input: source - the object that serves as the
* 				 source for the attributes.
*
*        Output: destination - the object that serves as
*				      the destination for the
*				      operation.
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Steve Kubica
*          Date: Aug 19, 1994 12:02
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kpds_copy_global_attr(
   kobject source,
   kobject destination)
{
   kpds_init();

   /* copy over all of the attributes */
   kaps_copy_global_attributes(source, destination);
   return TRUE;
}

/************************************************************
*
*  Routine Name: kpds_copy_object_attr - copy all presentation
*		 attributes from one data object to another.
*
*       Purpose: This function copies all
*                presentation attributes from the source_object
*		 to the destination_object.  This means that all
*		 the attributes of the object will be copied (not
* 		 just parts of the polymorphic data model) as well.  
*		 For example, the source object may contain
*		 attributes that one of the other services (geometry,
*		 numerical, etc.) uses.  These will also be copied.
*		 Segments present in the source object will be created
*                in the destination object if they are not already present.
*
*		 There are three attributes for each data component 
*		 (i.e. Value, Mask, etc.) that are affected by this 
*		 function call in a special way:  KPDS_*_SIZE, 
*		 KPDS_*_DATA_TYPE, and KPDS_*_INDEX_ORDER.  These 
*		 attributes are used to define how the data is stored.
*		 When this function is called, these attributes will
*		 appear to change to the user, but the storage of the
*		 data will only be affected if the KPDS_COUPLING
*		 attribute is set to KCOUPLED.
*
*		 For more information on the behavior of attributes,
*		 please refer to kpds_sync_object, kpds_get_attribute
*		 and kpds_set_attribute.
*
*         Input: source - the object that serves as the
* 				 source for the attributes.
*
*        Output: destination - the object that serves as
*				      the destination for the
*				      operation.
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*
*          Date: Sep 19, 1993 15:43
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int
kpds_copy_object_attr(
   kobject source,
   kobject destination)
{
   kpds_init();

   /* copy over all of the attributes */
   kaps_copy_object(source, destination, TRUE, FALSE);
   return TRUE;
}


